import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Checkbox } from "@/components/ui/checkbox";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { useToast } from "@/hooks/use-toast";
import {
  GitMerge,
  Plus,
  Trash2,
  Play,
  CheckCircle,
  AlertTriangle,
  Loader2,
  Settings,
  Sparkles,
} from "lucide-react";
import type { ComponentFamily } from "@shared/schema";

interface MergePoint {
  id: number;
  plan_id: number;
  merge_stage: string;
  required_families: string[];
  completion_policy: string;
  manual_release: boolean;
  arrived_families: string[] | null;
  family_progress: Record<string, { arrived: number; total: number }> | null;
  status: string;
  operation_code?: string;
  operation_name?: string;
  notes?: string;
}

interface MergePointsSuggestion {
  families: string[];
  suggestions: Array<{
    mergeStage: string;
    requiredFamilies: string[];
    completionPolicy: string;
    manualRelease: boolean;
    description: string;
  }>;
}

const familyLabels: Record<string, string> = {
  formatki_hdf: "HDF",
  formatki_biale: "Białe",
  formatki_kolor: "Kolorowe",
  tapicerowane: "Tapicerowane",
  siedziska: "Siedziska",
  akcesoria: "Akcesoria",
  mixed: "Mieszane",
};

const familyColors: Record<string, string> = {
  formatki_hdf: "bg-amber-500 text-white",
  formatki_biale: "bg-slate-200 text-slate-900 dark:bg-slate-600 dark:text-white",
  formatki_kolor: "bg-purple-500 text-white",
  tapicerowane: "bg-rose-500 text-white",
  siedziska: "bg-cyan-500 text-white",
  akcesoria: "bg-emerald-500 text-white",
  mixed: "bg-gray-500 text-white",
};

const stageLabels: Record<string, string> = {
  kompletowanie: "Kompletowanie",
  pakowanie: "Pakowanie",
  bufor_wysylkowy: "Bufor wysyłkowy",
};

interface MergePointsPanelProps {
  planId: number;
}

export function MergePointsPanel({ planId }: MergePointsPanelProps) {
  const { toast } = useToast();
  const [showAddDialog, setShowAddDialog] = useState(false);
  const [newMergePoint, setNewMergePoint] = useState({
    mergeStage: "kompletowanie",
    requiredFamilies: [] as string[],
    completionPolicy: "all_required",
    manualRelease: false,
    notes: "",
  });

  const { data: mergePoints, isLoading } = useQuery<MergePoint[]>({
    queryKey: ["/api/production/planning/plans", planId, "merge-points"],
    queryFn: async () => {
      const res = await fetch(`/api/production/planning/plans/${planId}/merge-points`);
      if (!res.ok) throw new Error("Failed to fetch merge points");
      return res.json();
    },
  });

  const { data: suggestions } = useQuery<MergePointsSuggestion>({
    queryKey: ["/api/production/planning/plans", planId, "merge-points/suggestions"],
    queryFn: async () => {
      const res = await fetch(`/api/production/planning/plans/${planId}/merge-points/suggestions`);
      if (!res.ok) throw new Error("Failed to fetch suggestions");
      return res.json();
    },
  });

  const createMutation = useMutation({
    mutationFn: async (data: typeof newMergePoint) => {
      return apiRequest("POST", `/api/production/planning/plans/${planId}/merge-points`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/planning/plans", planId, "merge-points"] });
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/flow`] });
      setShowAddDialog(false);
      setNewMergePoint({
        mergeStage: "kompletowanie",
        requiredFamilies: [],
        completionPolicy: "all_required",
        manualRelease: false,
        notes: "",
      });
      toast({ title: "Punkt zbieżności utworzony" });
    },
    onError: () => {
      toast({ title: "Błąd tworzenia punktu zbieżności", variant: "destructive" });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("DELETE", `/api/production/planning/merge-points/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/planning/plans", planId, "merge-points"] });
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/flow`] });
      toast({ title: "Punkt zbieżności usunięty" });
    },
    onError: () => {
      toast({ title: "Błąd usuwania punktu zbieżności", variant: "destructive" });
    },
  });

  const releaseMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("POST", `/api/production/planning/merge-points/${id}/release`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/planning/plans", planId, "merge-points"] });
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/flow`] });
      toast({ title: "Punkt zbieżności zwolniony" });
    },
    onError: () => {
      toast({ title: "Błąd zwalniania punktu zbieżności", variant: "destructive" });
    },
  });

  const applySuggestion = (suggestion: MergePointsSuggestion["suggestions"][0]) => {
    setNewMergePoint({
      mergeStage: suggestion.mergeStage,
      requiredFamilies: suggestion.requiredFamilies,
      completionPolicy: suggestion.completionPolicy,
      manualRelease: suggestion.manualRelease,
      notes: "",
    });
    setShowAddDialog(true);
  };

  const toggleFamily = (family: string) => {
    setNewMergePoint((prev) => ({
      ...prev,
      requiredFamilies: prev.requiredFamilies.includes(family)
        ? prev.requiredFamilies.filter((f) => f !== family)
        : [...prev.requiredFamilies, family],
    }));
  };

  const getStatusBadge = (status: string) => {
    switch (status) {
      case "released":
        return <Badge variant="default" className="bg-green-500">Zwolniono</Badge>;
      case "ready":
        return <Badge variant="default" className="bg-blue-500">Gotowy</Badge>;
      case "partial":
        return <Badge variant="secondary">Częściowo</Badge>;
      default:
        return <Badge variant="outline">Oczekuje</Badge>;
    }
  };

  if (isLoading) {
    return (
      <Card>
        <CardContent className="flex items-center justify-center py-8">
          <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
        </CardContent>
      </Card>
    );
  }

  return (
    <>
      <Card data-testid="merge-points-panel">
        <CardHeader className="pb-3">
          <div className="flex items-center justify-between gap-2">
            <CardTitle className="flex items-center gap-2 text-base">
              <GitMerge className="h-4 w-4" />
              Punkty zbieżności
            </CardTitle>
            <Button
              size="sm"
              onClick={() => setShowAddDialog(true)}
              data-testid="button-add-merge-point"
            >
              <Plus className="h-4 w-4 mr-1" />
              Dodaj
            </Button>
          </div>
        </CardHeader>
        <CardContent className="space-y-3">
          {suggestions && suggestions.suggestions.length > 0 && mergePoints?.length === 0 && (
            <div className="p-3 rounded-md border border-dashed border-blue-300 dark:border-blue-700 bg-blue-50 dark:bg-blue-950/30">
              <div className="flex items-center gap-2 text-sm font-medium text-blue-700 dark:text-blue-300 mb-2">
                <Sparkles className="h-4 w-4" />
                Sugestie punktów zbieżności
              </div>
              <div className="space-y-2">
                {suggestions.suggestions.map((suggestion, idx) => (
                  <div
                    key={idx}
                    className="flex items-center justify-between p-2 rounded bg-white dark:bg-gray-800 border"
                  >
                    <div>
                      <div className="text-sm font-medium">
                        {stageLabels[suggestion.mergeStage] || suggestion.mergeStage}
                      </div>
                      <div className="text-xs text-muted-foreground">
                        {suggestion.description}
                      </div>
                    </div>
                    <Button
                      size="sm"
                      variant="outline"
                      onClick={() => applySuggestion(suggestion)}
                      data-testid={`button-apply-suggestion-${idx}`}
                    >
                      Zastosuj
                    </Button>
                  </div>
                ))}
              </div>
            </div>
          )}

          {mergePoints && mergePoints.length === 0 && !suggestions?.suggestions.length && (
            <div className="text-center py-6 text-muted-foreground text-sm">
              Brak skonfigurowanych punktów zbieżności.
              <br />
              Dodaj punkt, aby śledzić gdzie ścieżki produkcyjne się łączą.
            </div>
          )}

          {mergePoints && mergePoints.map((mp) => {
            const arrivedCount = mp.arrived_families?.length || 0;
            const requiredCount = mp.required_families?.length || 0;
            const progressPercent = requiredCount > 0 ? (arrivedCount / requiredCount) * 100 : 0;

            return (
              <div
                key={mp.id}
                className="p-3 rounded-md border bg-card"
                data-testid={`merge-point-${mp.id}`}
              >
                <div className="flex items-start justify-between gap-2 mb-2">
                  <div>
                    <div className="flex items-center gap-2">
                      <span className="font-medium text-sm">
                        {stageLabels[mp.merge_stage] || mp.merge_stage}
                      </span>
                      {getStatusBadge(mp.status)}
                    </div>
                    {mp.operation_name && (
                      <div className="text-xs text-muted-foreground mt-0.5">
                        Operacja: {mp.operation_name}
                      </div>
                    )}
                  </div>
                  <div className="flex items-center gap-1">
                    {mp.status === "ready" && mp.manual_release && (
                      <Button
                        size="icon"
                        variant="ghost"
                        className="h-7 w-7 text-green-600"
                        onClick={() => releaseMutation.mutate(mp.id)}
                        disabled={releaseMutation.isPending}
                        data-testid={`button-release-${mp.id}`}
                      >
                        <Play className="h-4 w-4" />
                      </Button>
                    )}
                    <Button
                      size="icon"
                      variant="ghost"
                      className="h-7 w-7 text-destructive"
                      onClick={() => deleteMutation.mutate(mp.id)}
                      disabled={deleteMutation.isPending}
                      data-testid={`button-delete-${mp.id}`}
                    >
                      <Trash2 className="h-4 w-4" />
                    </Button>
                  </div>
                </div>

                <div className="flex items-center gap-2 mb-2">
                  <Progress value={progressPercent} className="h-1.5 flex-1" />
                  <span className="text-xs font-medium text-muted-foreground">
                    {arrivedCount}/{requiredCount}
                  </span>
                </div>

                <div className="flex flex-wrap gap-1">
                  {mp.required_families.map((family) => {
                    const arrived = mp.arrived_families?.includes(family);
                    return (
                      <Badge
                        key={family}
                        variant="outline"
                        className={`text-[10px] ${arrived ? familyColors[family] || "bg-gray-500 text-white" : "opacity-50"}`}
                      >
                        {arrived && <CheckCircle className="h-2.5 w-2.5 mr-0.5" />}
                        {familyLabels[family] || family}
                      </Badge>
                    );
                  })}
                </div>

                {mp.manual_release && mp.status !== "released" && (
                  <div className="mt-2 text-[10px] text-amber-600 dark:text-amber-400 flex items-center gap-1">
                    <AlertTriangle className="h-3 w-3" />
                    Wymaga ręcznego zatwierdzenia
                  </div>
                )}

                {mp.notes && (
                  <div className="mt-2 text-xs text-muted-foreground">
                    {mp.notes}
                  </div>
                )}
              </div>
            );
          })}
        </CardContent>
      </Card>

      <Dialog open={showAddDialog} onOpenChange={setShowAddDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Dodaj punkt zbieżności</DialogTitle>
            <DialogDescription>
              Skonfiguruj gdzie ścieżki produkcyjne mają się łączyć
            </DialogDescription>
          </DialogHeader>

          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label>Etap łączenia</Label>
              <Select
                value={newMergePoint.mergeStage}
                onValueChange={(value) =>
                  setNewMergePoint((prev) => ({ ...prev, mergeStage: value }))
                }
              >
                <SelectTrigger data-testid="select-merge-stage">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="kompletowanie">Kompletowanie</SelectItem>
                  <SelectItem value="pakowanie">Pakowanie</SelectItem>
                  <SelectItem value="bufor_wysylkowy">Bufor wysyłkowy</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label>Wymagane rodziny komponentów</Label>
              <div className="flex flex-wrap gap-2 p-3 border rounded-md">
                {(suggestions?.families || []).map((family) => (
                  <Badge
                    key={family}
                    variant={newMergePoint.requiredFamilies.includes(family) ? "default" : "outline"}
                    className={`cursor-pointer ${newMergePoint.requiredFamilies.includes(family) ? familyColors[family] : ""}`}
                    onClick={() => toggleFamily(family)}
                    data-testid={`toggle-family-${family}`}
                  >
                    {familyLabels[family] || family}
                  </Badge>
                ))}
                {(!suggestions?.families || suggestions.families.length === 0) && (
                  <span className="text-sm text-muted-foreground">
                    Brak wykrytych rodzin w planie
                  </span>
                )}
              </div>
            </div>

            <div className="space-y-2">
              <Label>Polityka ukończenia</Label>
              <Select
                value={newMergePoint.completionPolicy}
                onValueChange={(value) =>
                  setNewMergePoint((prev) => ({ ...prev, completionPolicy: value }))
                }
              >
                <SelectTrigger data-testid="select-completion-policy">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all_required">Wszystkie wymagane</SelectItem>
                  <SelectItem value="any">Dowolna rodzina</SelectItem>
                  <SelectItem value="manual">Ręczne zatwierdzenie</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="flex items-center space-x-2">
              <Checkbox
                id="manualRelease"
                checked={newMergePoint.manualRelease}
                onCheckedChange={(checked) =>
                  setNewMergePoint((prev) => ({ ...prev, manualRelease: !!checked }))
                }
                data-testid="checkbox-manual-release"
              />
              <Label htmlFor="manualRelease" className="text-sm font-normal">
                Wymagaj ręcznego zatwierdzenia do kontynuacji
              </Label>
            </div>

            <div className="space-y-2">
              <Label>Notatki (opcjonalnie)</Label>
              <Textarea
                value={newMergePoint.notes}
                onChange={(e) =>
                  setNewMergePoint((prev) => ({ ...prev, notes: e.target.value }))
                }
                placeholder="Dodatkowe informacje..."
                rows={2}
                data-testid="input-notes"
              />
            </div>
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={() => setShowAddDialog(false)}>
              Anuluj
            </Button>
            <Button
              onClick={() => createMutation.mutate(newMergePoint)}
              disabled={createMutation.isPending || newMergePoint.requiredFamilies.length === 0}
              data-testid="button-create-merge-point"
            >
              {createMutation.isPending ? (
                <Loader2 className="h-4 w-4 mr-1 animate-spin" />
              ) : (
                <Plus className="h-4 w-4 mr-1" />
              )}
              Utwórz
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </>
  );
}
