import { useState } from 'react';
import { Plus, Trash2, Settings, GripVertical } from 'lucide-react';
import {
  DndContext,
  closestCenter,
  KeyboardSensor,
  PointerSensor,
  useSensor,
  useSensors,
  DragEndEvent,
} from '@dnd-kit/core';
import {
  arrayMove,
  SortableContext,
  sortableKeyboardCoordinates,
  useSortable,
  verticalListSortingStrategy,
} from '@dnd-kit/sortable';
import { CSS } from '@dnd-kit/utilities';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from '@/components/ui/dialog';
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from '@/components/ui/alert-dialog';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Badge } from '@/components/ui/badge';
import { FilterTab } from './use-filter-tabs';

interface ManageFiltersDialogProps {
  tabs: FilterTab[];
  onAddTab: (label: string) => void;
  onRemoveTab: (tabId: string) => void;
  onToggleTab: (tabId: string) => void;
  onReorderTabs: (newOrder: FilterTab[]) => void;
}

interface SortableFilterItemProps {
  tab: FilterTab;
  index: number;
  onToggle: () => void;
  onDelete: () => void;
}

function SortableFilterItem({ tab, index, onToggle, onDelete }: SortableFilterItemProps) {
  const {
    attributes,
    listeners,
    setNodeRef,
    transform,
    transition,
    isDragging,
  } = useSortable({ id: tab.id });

  const style = {
    transform: CSS.Transform.toString(transform),
    transition,
    opacity: isDragging ? 0.5 : 1,
  };

  return (
    <div
      ref={setNodeRef}
      style={style}
      className="flex items-center gap-2 p-2 rounded-md border hover-elevate bg-card"
      data-testid={`filter-row-${tab.token.toLowerCase()}`}
    >
      <div
        {...attributes}
        {...listeners}
        className="cursor-grab active:cursor-grabbing text-muted-foreground hover:text-foreground"
        data-testid={`drag-handle-${tab.token.toLowerCase()}`}
      >
        <GripVertical className="h-4 w-4" />
      </div>
      
      <div className="text-sm font-mono text-muted-foreground w-6 text-center">
        {index + 1}
      </div>
      
      <div className="flex items-center gap-2 flex-1">
        <Badge
          variant={tab.isActive ? 'default' : 'outline'}
          className="cursor-pointer"
          onClick={onToggle}
          data-testid={`badge-persistent-${tab.token.toLowerCase()}`}
        >
          {tab.label}
        </Badge>
        <span className="text-sm text-muted-foreground">
          {tab.isActive ? 'Aktywny' : 'Nieaktywny'}
        </span>
      </div>
      
      <Button
        size="icon"
        variant="ghost"
        onClick={onDelete}
        className="h-8 w-8 text-destructive hover:text-destructive"
        data-testid={`button-delete-${tab.token.toLowerCase()}`}
      >
        <Trash2 className="h-4 w-4" />
      </Button>
    </div>
  );
}

export function ManageFiltersDialog({ 
  tabs, 
  onAddTab, 
  onRemoveTab,
  onToggleTab,
  onReorderTabs
}: ManageFiltersDialogProps) {
  const [open, setOpen] = useState(false);
  const [inputValue, setInputValue] = useState('');
  const [deleteConfirmTab, setDeleteConfirmTab] = useState<FilterTab | null>(null);

  const sensors = useSensors(
    useSensor(PointerSensor),
    useSensor(KeyboardSensor, {
      coordinateGetter: sortableKeyboardCoordinates,
    })
  );

  const handleDragEnd = (event: DragEndEvent) => {
    const { active, over } = event;

    if (over && active.id !== over.id) {
      const oldIndex = tabs.findIndex((tab) => tab.id === active.id);
      const newIndex = tabs.findIndex((tab) => tab.id === over.id);

      const newOrder = arrayMove(tabs, oldIndex, newIndex);
      onReorderTabs(newOrder);
    }
  };

  const handleAddTab = () => {
    if (!inputValue.trim()) return;
    
    onAddTab(inputValue.trim());
    setInputValue('');
  };

  const handleKeyDown = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter') {
      e.preventDefault();
      handleAddTab();
    }
  };

  const handleDeleteClick = (tab: FilterTab) => {
    setDeleteConfirmTab(tab);
  };

  const handleConfirmDelete = () => {
    if (deleteConfirmTab) {
      onRemoveTab(deleteConfirmTab.id);
      setDeleteConfirmTab(null);
    }
  };

  return (
    <>
      <Dialog open={open} onOpenChange={setOpen}>
        <DialogTrigger asChild>
          <Button
            size="icon"
            variant="ghost"
            className="h-[25px] w-[25px]"
            data-testid="button-manage-filters"
            title="Zarządzaj filtrami stałymi"
          >
            <Settings className="h-3 w-3" />
          </Button>
        </DialogTrigger>
        <DialogContent className="sm:max-w-[500px]">
          <DialogHeader>
            <DialogTitle>Zarządzaj filtrami stałymi</DialogTitle>
            <DialogDescription>
              Dodaj lub usuń filtry stałe. Filtry stałe są zapisywane i dostępne po odświeżeniu strony.
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4">
            {/* Add new filter */}
            <div className="flex gap-2">
              <Input
                value={inputValue}
                onChange={(e) => setInputValue(e.target.value)}
                onKeyDown={handleKeyDown}
                placeholder="Nazwa filtra (np. BIA, WOTAN)"
                className="flex-1"
                data-testid="input-add-persistent-filter"
              />
              <Button
                onClick={handleAddTab}
                disabled={!inputValue.trim()}
                data-testid="button-add-persistent-filter"
              >
                <Plus className="h-4 w-4 mr-1" />
                Dodaj
              </Button>
            </div>

            {/* List of existing filters with drag & drop */}
            <div className="space-y-2">
              <div className="text-sm font-medium text-muted-foreground flex items-center gap-2">
                <span>Filtry stałe ({tabs.length})</span>
                {tabs.length > 0 && (
                  <span className="text-xs text-muted-foreground/70">
                    Przeciągnij aby zmienić kolejność
                  </span>
                )}
              </div>
              
              {tabs.length === 0 ? (
                <div className="text-sm text-muted-foreground text-center py-8">
                  Brak zapisanych filtrów stałych
                </div>
              ) : (
                <DndContext
                  sensors={sensors}
                  collisionDetection={closestCenter}
                  onDragEnd={handleDragEnd}
                >
                  <SortableContext
                    items={tabs.map(t => t.id)}
                    strategy={verticalListSortingStrategy}
                  >
                    <div className="space-y-2 max-h-[400px] overflow-y-auto">
                      {tabs.map((tab, index) => (
                        <SortableFilterItem
                          key={tab.id}
                          tab={tab}
                          index={index}
                          onToggle={() => onToggleTab(tab.id)}
                          onDelete={() => handleDeleteClick(tab)}
                        />
                      ))}
                    </div>
                  </SortableContext>
                </DndContext>
              )}
            </div>
          </div>
        </DialogContent>
      </Dialog>

      {/* Delete confirmation dialog */}
      <AlertDialog open={!!deleteConfirmTab} onOpenChange={(open) => !open && setDeleteConfirmTab(null)}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Usunąć filtr stały?</AlertDialogTitle>
            <AlertDialogDescription>
              Czy na pewno chcesz usunąć filtr <strong>"{deleteConfirmTab?.label}"</strong>? 
              Tej operacji nie można cofnąć.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel data-testid="button-cancel-delete">Anuluj</AlertDialogCancel>
            <AlertDialogAction 
              onClick={handleConfirmDelete}
              className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
              data-testid="button-confirm-delete"
            >
              Usuń
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </>
  );
}
