import { useMemo, useState, useEffect } from "react";
import { ChevronRight, ChevronDown, Package, Layers } from "lucide-react";
import { Badge } from "@/components/ui/badge";
import { Checkbox } from "@/components/ui/checkbox";
import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { DraggableProductItem, type MarketplaceOrderItem } from "./DraggableProductItem";

interface DictionaryItem {
  id: number;
  code: string;
  name: string | null;
  shortName: string | null;
  color: string | null;
  isActive: boolean;
}

function getTextColorForBackground(hexColor: string | null | undefined): string {
  if (!hexColor || !hexColor.startsWith('#')) return 'black';
  
  const r = parseInt(hexColor.slice(1, 3), 16);
  const g = parseInt(hexColor.slice(3, 5), 16);
  const b = parseInt(hexColor.slice(5, 7), 16);
  
  const luminance = 0.299 * r + 0.587 * g + 0.114 * b;
  
  return luminance > 128 ? 'black' : 'white';
}

interface MarketplaceOrder {
  order_id: number;
  order_number: string;
  marketplace: string;
  buyer_first_name: string;
  buyer_last_name: string;
  buyer_email: string;
  order_date: string;
  payment_status: string;
  total_to_pay_amount: string;
  currency: string;
  items: MarketplaceOrderItem[];
}

interface ItemWithContext extends MarketplaceOrderItem {
  orderNumber: string;
  buyerName: string;
  marketplace: string;
  orderDate: string;
  orderTone: 0 | 1;
}

// Helper: Create deterministic order tone map (0 or 1) for alternating backgrounds
function getOrderToneMap(orders: MarketplaceOrder[]): Map<number, 0 | 1> {
  const uniqueOrderIds = Array.from(new Set(orders.map(o => o.order_id))).sort((a, b) => a - b);
  const toneMap = new Map<number, 0 | 1>();
  
  uniqueOrderIds.forEach((orderId, index) => {
    toneMap.set(orderId, (index % 2) as 0 | 1);
  });
  
  return toneMap;
}

interface GroupedOrdersListProps {
  orders: MarketplaceOrder[];
  colors?: DictionaryItem[];
  productTypes?: DictionaryItem[];
  dimensionLengths?: DictionaryItem[];
  dimensionWidths?: DictionaryItem[];
  onAddItem?: (item: MarketplaceOrderItem, orderNumber: string, buyerName: string, marketplace: string, setId?: number, componentId?: number) => void;
  isAddingItem?: boolean;
  recentlyAddedItems?: Set<number>;
  selectedItemIds?: Set<string>;
  onToggleItemSelection?: (itemId: string) => void;
  expandedColorGroups?: Set<string>;
  setExpandedColorGroups?: (value: Set<string> | ((prev: Set<string>) => Set<string>)) => void;
  expandedTypeGroups?: Set<string>;
  setExpandedTypeGroups?: (value: Set<string> | ((prev: Set<string>) => Set<string>)) => void;
  expandedSets?: Set<number>;
  setExpandedSets?: (value: Set<number> | ((prev: Set<number>) => Set<number>)) => void;
  onHighlightInPlan?: (productId: number, orderNumber: string) => void;
  isLocked?: boolean;
}

export function GroupedOrdersList({ 
  orders, 
  colors,
  productTypes = [],
  dimensionLengths = [],
  dimensionWidths = [],
  onAddItem, 
  isAddingItem, 
  recentlyAddedItems = new Set(), 
  selectedItemIds = new Set(), 
  onToggleItemSelection,
  expandedColorGroups: propExpandedColorGroups,
  setExpandedColorGroups: propSetExpandedColorGroups,
  expandedTypeGroups: propExpandedTypeGroups,
  setExpandedTypeGroups: propSetExpandedTypeGroups,
  expandedSets: propExpandedSets,
  setExpandedSets: propSetExpandedSets,
  onHighlightInPlan,
  isLocked = false
}: GroupedOrdersListProps) {
  // Use props if provided, otherwise use local state (for backwards compatibility)
  const [localExpandedColorGroups, setLocalExpandedColorGroups] = useState<Set<string>>(new Set());
  const [localExpandedTypeGroups, setLocalExpandedTypeGroups] = useState<Set<string>>(new Set());
  
  // Initialize with all sets expanded by default (same as CompactOrdersList)
  const [localExpandedSets, setLocalExpandedSets] = useState<Set<number>>(() => {
    const allSetItemIds = new Set<number>();
    orders.forEach(order => {
      order.items.forEach(item => {
        if (item.set_components && item.set_components.length > 0) {
          allSetItemIds.add(item.item_id);
        }
      });
    });
    return allSetItemIds;
  });
  
  const expandedColorGroups = propExpandedColorGroups ?? localExpandedColorGroups;
  const setExpandedColorGroups = propSetExpandedColorGroups ?? setLocalExpandedColorGroups;
  const expandedTypeGroups = propExpandedTypeGroups ?? localExpandedTypeGroups;
  const setExpandedTypeGroups = propSetExpandedTypeGroups ?? setLocalExpandedTypeGroups;
  const expandedSets = propExpandedSets ?? localExpandedSets;
  const setExpandedSets = propSetExpandedSets ?? setLocalExpandedSets;

  const toggleSetExpansion = (itemId: number) => {
    setExpandedSets(prev => {
      const newSet = new Set(prev);
      if (newSet.has(itemId)) {
        newSet.delete(itemId);
      } else {
        newSet.add(itemId);
      }
      return newSet;
    });
  };

  // Create order tone mapping for alternating backgrounds
  const orderToneMap = useMemo(() => getOrderToneMap(orders), [orders]);
  
  // Group items by Color → Product Type with order tone
  const groupedData = useMemo(() => {
    const groups = new Map<string, Map<string, ItemWithContext[]>>();

    orders.forEach(order => {
      const buyerName = `${order.buyer_first_name} ${order.buyer_last_name}`;
      const tone = orderToneMap.get(order.order_id) ?? 0;
      
      order.items.forEach(item => {
        const colorKey = item.product_color || 'Bez koloru';
        const typeKey = item.product_type || 'Bez typu';
        
        if (!groups.has(colorKey)) {
          groups.set(colorKey, new Map());
        }
        
        const colorGroup = groups.get(colorKey)!;
        if (!colorGroup.has(typeKey)) {
          colorGroup.set(typeKey, []);
        }
        
        colorGroup.get(typeKey)!.push({
          ...item,
          orderNumber: order.order_number,
          buyerName,
          marketplace: order.marketplace,
          orderDate: order.order_date,
          orderTone: tone
        });
      });
    });

    return groups;
  }, [orders, orderToneMap]);

  // Helper: Get all item IDs in a color group
  const getColorGroupItemIds = (colorKey: string): string[] => {
    const typeGroups = groupedData.get(colorKey);
    if (!typeGroups) return [];
    const ids: string[] = [];
    typeGroups.forEach(items => {
      items.forEach(item => ids.push(`item-${item.item_id}`));
    });
    return ids;
  };

  // Helper: Get all item IDs in a type group
  const getTypeGroupItemIds = (colorKey: string, typeKey: string): string[] => {
    const typeGroups = groupedData.get(colorKey);
    if (!typeGroups) return [];
    const items = typeGroups.get(typeKey);
    return items ? items.map(item => `item-${item.item_id}`) : [];
  };

  // Helper: Check if all items in group are selected
  const areAllColorItemsSelected = (colorKey: string): boolean => {
    const ids = getColorGroupItemIds(colorKey);
    return ids.length > 0 && ids.every(id => selectedItemIds.has(id));
  };

  const areAllTypeItemsSelected = (colorKey: string, typeKey: string): boolean => {
    const ids = getTypeGroupItemIds(colorKey, typeKey);
    return ids.length > 0 && ids.every(id => selectedItemIds.has(id));
  };

  // Helper: Check if some (but not all) items in group are selected
  const areSomeColorItemsSelected = (colorKey: string): boolean => {
    const ids = getColorGroupItemIds(colorKey);
    const selectedCount = ids.filter(id => selectedItemIds.has(id)).length;
    return selectedCount > 0 && selectedCount < ids.length;
  };

  const areSomeTypeItemsSelected = (colorKey: string, typeKey: string): boolean => {
    const ids = getTypeGroupItemIds(colorKey, typeKey);
    const selectedCount = ids.filter(id => selectedItemIds.has(id)).length;
    return selectedCount > 0 && selectedCount < ids.length;
  };

  // Toggle all items in color group
  const toggleColorGroupSelection = (colorKey: string) => {
    if (!onToggleItemSelection) return;
    const ids = getColorGroupItemIds(colorKey);
    const allSelected = areAllColorItemsSelected(colorKey);
    
    ids.forEach(id => {
      // If all are selected, deselect all; otherwise select all
      if (allSelected) {
        if (selectedItemIds.has(id)) {
          onToggleItemSelection(id);
        }
      } else {
        if (!selectedItemIds.has(id)) {
          onToggleItemSelection(id);
        }
      }
    });
  };

  // Toggle all items in type group
  const toggleTypeGroupSelection = (colorKey: string, typeKey: string) => {
    if (!onToggleItemSelection) return;
    const ids = getTypeGroupItemIds(colorKey, typeKey);
    const allSelected = areAllTypeItemsSelected(colorKey, typeKey);
    
    ids.forEach(id => {
      if (allSelected) {
        if (selectedItemIds.has(id)) {
          onToggleItemSelection(id);
        }
      } else {
        if (!selectedItemIds.has(id)) {
          onToggleItemSelection(id);
        }
      }
    });
  };

  const toggleColorGroup = (colorKey: string) => {
    setExpandedColorGroups(prev => {
      const newSet = new Set(prev);
      const isExpanding = !newSet.has(colorKey);
      
      if (newSet.has(colorKey)) {
        newSet.delete(colorKey);
        // Also collapse all type groups within this color
        setExpandedTypeGroups(typeSet => {
          const newTypeSet = new Set(typeSet);
          const typeGroups = groupedData.get(colorKey);
          if (typeGroups) {
            typeGroups.forEach((_, typeKey) => {
              newTypeSet.delete(`${colorKey}-${typeKey}`);
            });
          }
          return newTypeSet;
        });
      } else {
        newSet.add(colorKey);
        // Auto-expand all type groups within this color
        setExpandedTypeGroups(typeSet => {
          const newTypeSet = new Set(typeSet);
          const typeGroups = groupedData.get(colorKey);
          if (typeGroups) {
            typeGroups.forEach((_, typeKey) => {
              newTypeSet.add(`${colorKey}-${typeKey}`);
            });
          }
          return newTypeSet;
        });
      }
      return newSet;
    });
  };

  const toggleTypeGroup = (typeKey: string) => {
    setExpandedTypeGroups(prev => {
      const newSet = new Set(prev);
      if (newSet.has(typeKey)) {
        newSet.delete(typeKey);
      } else {
        newSet.add(typeKey);
      }
      return newSet;
    });
  };



  if (groupedData.size === 0) {
    return (
      <div className="flex flex-col items-center justify-center p-8 text-center text-muted-foreground">
        <Package className="h-12 w-12 mb-4 opacity-50" />
        <p>Brak zamówień spełniających kryteria</p>
        <p className="text-sm">Zmień filtry aby zobaczyć więcej zamówień</p>
      </div>
    );
  }

  return (
    <div className="space-y-0" data-testid="grouped-orders-list">
        {Array.from(groupedData.entries()).map(([colorKey, typeGroups]) => {
          const isColorExpanded = expandedColorGroups.has(colorKey);
          const colorDict = colors?.find(c => c.code === colorKey);
          const bgColor = colorDict?.color || '#94A3B8';
          const textColor = getTextColorForBackground(bgColor);
          
          // Count total items in this color group
          let totalItems = 0;
          typeGroups.forEach(items => { totalItems += items.length; });

          return (
            <div key={colorKey} className="border border-border rounded-none mt-4 first:mt-0">
              {/* Color Group Header - 50px - light background */}
              <div
                className="flex items-center gap-2 h-[50px] px-3 hover-elevate font-medium bg-background"
                data-testid={`color-group-${colorKey}`}
                style={{ borderLeft: `4px solid ${bgColor}` }}
              >
                {onToggleItemSelection && (
                  <Checkbox
                    checked={areAllColorItemsSelected(colorKey)}
                    onCheckedChange={() => toggleColorGroupSelection(colorKey)}
                    onClick={(e) => e.stopPropagation()}
                    className="flex-shrink-0"
                    data-testid={`checkbox-color-${colorKey}`}
                  />
                )}
                
                <div 
                  className="flex items-center gap-2 flex-1 cursor-pointer"
                  onClick={() => toggleColorGroup(colorKey)}
                >
                  {isColorExpanded ? (
                    <ChevronDown className="h-3 w-3 flex-shrink-0" />
                  ) : (
                    <ChevronRight className="h-3 w-3 flex-shrink-0" />
                  )}
                  
                  <Badge 
                    variant="outline" 
                    className="h-4 px-2 text-[10px] border-transparent"
                    style={{ backgroundColor: bgColor, color: textColor }}
                  >
                    {colorKey}
                  </Badge>
                  
                  <span className="text-xs text-muted-foreground">
                    {totalItems} {totalItems === 1 ? 'produkt' : 'produktów'}
                  </span>
                </div>
              </div>

              {/* Type Groups within Color */}
              {isColorExpanded && (
                <div>
                  {Array.from(typeGroups.entries()).map(([typeKey, items], index) => {
                    const isTypeExpanded = expandedTypeGroups.has(`${colorKey}-${typeKey}`);
                    const isFirstType = index === 0;
                    
                    return (
                      <div key={`${colorKey}-${typeKey}`} className={isFirstType ? '' : 'border-t border-border/50'}>
                        {/* Product Type Header - 40px with indentation - light background */}
                        <div
                          className="flex items-center gap-2 h-[40px] pl-6 pr-2 hover-elevate bg-background/80"
                          data-testid={`type-group-${colorKey}-${typeKey}`}
                        >
                          {onToggleItemSelection && (
                            <Checkbox
                              checked={areAllTypeItemsSelected(colorKey, typeKey)}
                              onCheckedChange={() => toggleTypeGroupSelection(colorKey, typeKey)}
                              onClick={(e) => e.stopPropagation()}
                              className="flex-shrink-0"
                              data-testid={`checkbox-type-${colorKey}-${typeKey}`}
                            />
                          )}
                          
                          <div 
                            className="flex items-center gap-2 flex-1 cursor-pointer"
                            onClick={() => toggleTypeGroup(`${colorKey}-${typeKey}`)}
                          >
                            {isTypeExpanded ? (
                              <ChevronDown className="h-3 w-3 flex-shrink-0" />
                            ) : (
                              <ChevronRight className="h-3 w-3 flex-shrink-0" />
                            )}
                            
                            <Badge variant="secondary" className="h-4 px-1 text-[9px]">
                              {typeKey}
                            </Badge>
                            
                            <span className="text-[10px] text-muted-foreground">
                              {items.length} {items.length === 1 ? 'szt.' : 'szt.'}
                            </span>
                          </div>
                        </div>

                        {/* Items within Type Group - dark background container */}
                        {isTypeExpanded && (
                          <div className="pl-3 space-y-0 pb-2 bg-muted/20">
                            {items.map((item) => {
                              const hasSetComponents = item.set_components && item.set_components.length > 0;
                              const isExpanded = expandedSets.has(item.item_id);
                              
                              return (
                                <div key={`${item.orderNumber}-${item.item_id}`}>
                                  {/* Parent Item (Set) */}
                                  <div className="relative flex items-stretch">
                                    {hasSetComponents && (
                                      <button
                                        onClick={() => toggleSetExpansion(item.item_id)}
                                        className="flex-shrink-0 w-6 flex items-center justify-center hover:bg-muted/50 transition-colors border-r border-border/50"
                                        title={isExpanded ? "Zwiń komponenty" : "Rozwiń komponenty"}
                                        data-testid={`toggle-set-${item.item_id}`}
                                      >
                                        {isExpanded ? <ChevronDown className="w-3 h-3" /> : <ChevronRight className="w-3 h-3" />}
                                      </button>
                                    )}
                                    {!hasSetComponents && <div className="flex-shrink-0 w-6" />}
                                    <div className="flex-1">
                                      <DraggableProductItem
                                        item={item}
                                        orderNumber={item.orderNumber}
                                        buyerName={item.buyerName}
                                        marketplace={item.marketplace}
                                        orderDate={item.orderDate}
                                        colors={colors}
                                        productTypes={productTypes}
                                        dimensionLengths={dimensionLengths}
                                        dimensionWidths={dimensionWidths}
                                        onAddClick={hasSetComponents ? undefined : onAddItem}
                                        isAdding={isAddingItem}
                                        isRecentlyAdded={recentlyAddedItems.has(item.item_id)}
                                        rowHeight="48px"
                                        showOrderInfo={true}
                                        isSelected={selectedItemIds?.has(`item-${item.item_id}`) ?? false}
                                        onToggleSelect={onToggleItemSelection ? () => onToggleItemSelection(`item-${item.item_id}`) : undefined}
                                        orderTone={item.orderTone}
                                        onAddSetComponents={hasSetComponents && item.set_components?.some(c => !c.is_in_any_plan) ? () => {
                                          const componentsToAdd = item.set_components?.filter(c => !c.is_in_any_plan) || [];
                                          componentsToAdd.forEach(component => {
                                            if (onAddItem) {
                                              const componentItem: typeof item = {
                                                ...item,
                                                item_id: component.component_id,
                                                name: component.component_title,
                                                product_sku: component.component_sku,
                                                catalog_product_id: component.component_id,
                                                quantity: component.quantity,
                                                image_url: component.primary_image_url,
                                              };
                                              onAddItem(componentItem, item.orderNumber, item.buyerName, item.marketplace, item.catalog_set_id ?? undefined, component.component_id);
                                            }
                                          });
                                        } : undefined}
                                        allSetComponentsInPlan={Boolean(hasSetComponents && item.set_components?.every(c => c.is_in_any_plan))}
                                        isSetParent={Boolean(hasSetComponents)}
                                        onHighlightInPlan={onHighlightInPlan}
                                        isLocked={isLocked}
                                      />
                                    </div>
                                  </div>
                                  
                                  {/* Set Components (expandable) - larger indent to align with parent image right edge */}
                                  {hasSetComponents && isExpanded && (
                                    <div className="border-l-2 border-primary/30 ml-[88px]">
                                      {item.set_components!.map((component) => {
                                        return (
                                          <div key={`${item.catalog_set_id}-${component.component_id}-${item.orderNumber}`} className="relative flex items-stretch min-h-[40px]">
                                            {/* Component status indicator */}
                                            <div className="w-1 flex-shrink-0 h-full">
                                              {component.is_in_any_plan && (
                                                <div className="w-full h-full bg-green-500/30" title={`W planie: ${component.in_plan_number}`} />
                                              )}
                                            </div>
                                            
                                            {/* Component item */}
                                            <div className="flex-1">
                                              <DraggableProductItem
                                                item={{
                                                  ...item,
                                                  item_id: component.component_id,
                                                  name: component.component_title,
                                                  product_sku: component.component_sku,
                                                  product_length: component.component_length,
                                                  product_width: component.component_width,
                                                  product_height: component.component_height,
                                                  product_color: component.component_color,
                                                  product_type: component.component_product_type,
                                                  product_doors: component.component_doors,
                                                  product_legs: component.component_legs,
                                                  image_url: component.primary_image_url,
                                                  catalog_product_id: component.component_id,
                                                  catalog_set_id: item.catalog_set_id,
                                                  bom_component_count: null,
                                                  is_in_plan: component.is_in_any_plan,
                                                  in_plan_number: component.in_plan_number,
                                                  in_plan_id: component.in_plan_id,
                                                  in_current_plan: component.is_in_current_plan,
                                                  quantity: component.quantity,
                                                  set_components: null,
                                                }}
                                                orderNumber={item.orderNumber}
                                                buyerName={item.buyerName}
                                                marketplace={item.marketplace}
                                                colors={colors}
                                                productTypes={productTypes}
                                                dimensionLengths={dimensionLengths}
                                                dimensionWidths={dimensionWidths}
                                                onAddClick={onAddItem}
                                                setId={item.catalog_set_id ?? undefined}
                                                componentId={component.component_id}
                                                isAdding={isAddingItem}
                                                isRecentlyAdded={false}
                                                rowHeight="40px"
                                                isSelected={selectedItemIds?.has(`component-${item.orderNumber}-${component.component_id}`) ?? false}
                                                onToggleSelect={onToggleItemSelection ? () => onToggleItemSelection(`component-${item.orderNumber}-${component.component_id}`) : undefined}
                                                showOrderInfo={false}
                                                isSetChild={true}
                                                onHighlightInPlan={onHighlightInPlan}
                                                isLocked={isLocked}
                                              />
                                            </div>
                                            
                                            {/* Component metadata badge */}
                                            <div className="flex items-center px-2 gap-1 h-[40px]">
                                              <Badge variant="outline" className="h-5 text-[9px] px-1.5 whitespace-nowrap">
                                                <Layers className="w-2.5 h-2.5 mr-0.5" />
                                                {component.quantity}x
                                              </Badge>
                                              {component.is_in_any_plan && (
                                                <Badge variant="outline" className="h-5 text-[9px] px-1.5 bg-green-500/10 border-green-500/30 whitespace-nowrap">
                                                  {component.in_plan_number}
                                                </Badge>
                                              )}
                                            </div>
                                          </div>
                                        );
                                      })}
                                      
                                      {/* Summary footer */}
                                      <div className="px-3 py-1 bg-muted/20 text-xs text-muted-foreground flex items-center justify-between">
                                        <span>
                                          Komponentów: {item.set_components!.length}
                                        </span>
                                        {item.set_components!.every(c => c.is_in_any_plan) && (
                                          <Badge variant="outline" className="h-4 text-[9px] bg-green-500/10 border-green-500/30">
                                            ✓ Wszystkie w planach
                                          </Badge>
                                        )}
                                      </div>
                                    </div>
                                  )}
                                </div>
                              );
                            })}
                          </div>
                        )}
                      </div>
                    );
                  })}
                </div>
              )}
            </div>
          );
        })}
      </div>
  );
}
