import { useMemo } from "react";
import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Plus, Scissors, AlertCircle, Clock, Play, CheckCircle2 } from "lucide-react";
import { cn } from "@/lib/utils";

interface CuttingPattern {
  id: number;
  code: string;
  name: string;
  description: string | null;
  status: string;
  board_length: string;
  board_width: string;
  board_thickness: string;
  kerf: string;
  total_quantity: number;
  items_count: number;
  created_at: string;
  is_in_plan: boolean;
  in_plan_number: string | null;
  in_current_plan: boolean;
}

interface CuttingPatternsListProps {
  patterns: CuttingPattern[];
  onAddPattern: (pattern: CuttingPattern) => void;
  isAddingPattern?: boolean;
  recentlyAddedPatterns?: Set<number>;
  isLocked?: boolean;
}

const getStatusBadge = (status: string) => {
  const statusMap: Record<string, { label: string; icon: React.ElementType; variant: "default" | "secondary" | "outline" }> = {
    planowany: {
      label: "Planowany",
      icon: Clock,
      variant: "outline",
    },
    w_produkcji: {
      label: "W produkcji",
      icon: Play,
      variant: "default",
    },
    wykonany: {
      label: "Wykonany",
      icon: CheckCircle2,
      variant: "secondary",
    },
  };

  const config = statusMap[status] || { label: status, icon: Clock, variant: "outline" as const };
  const Icon = config.icon;

  return (
    <Badge variant={config.variant} className="h-5 text-[10px]">
      <Icon className="h-2.5 w-2.5 mr-1" />
      {config.label}
    </Badge>
  );
};

export function CuttingPatternsList({ 
  patterns, 
  onAddPattern, 
  isAddingPattern = false,
  recentlyAddedPatterns = new Set(),
  isLocked = false
}: CuttingPatternsListProps) {

  if (patterns.length === 0) {
    return (
      <div className="flex flex-col items-center justify-center py-12 text-center" data-testid="empty-cutting-patterns">
        <Scissors className="h-12 w-12 text-muted-foreground mb-3" />
        <p className="text-sm text-muted-foreground">
          Brak szablonów rozkrojów
        </p>
      </div>
    );
  }

  return (
    <div className="space-y-1" data-testid="cutting-patterns-list">
      {patterns.map((pattern) => {
        const isRecentlyAdded = recentlyAddedPatterns.has(pattern.id);
        
        return (
          <div
            key={pattern.id}
            className={cn(
              "flex items-center gap-2 px-3 py-2 rounded-md hover:bg-muted/30 transition-colors",
              isRecentlyAdded && "bg-green-500/10",
              pattern.in_current_plan && "opacity-50"
            )}
            data-testid={`cutting-pattern-${pattern.id}`}
          >
            {/* Pattern info */}
            <div className="flex-1 min-w-0">
              <div className="flex items-center gap-2">
                <Badge variant="outline" className="h-5 px-1.5 text-[10px] flex-shrink-0 bg-orange-500/10 border-orange-500/30">
                  {pattern.code}
                </Badge>
                <span className="text-sm font-medium truncate">
                  {pattern.name}
                </span>
              </div>
              <div className="flex items-center gap-2 text-xs text-muted-foreground mt-0.5">
                <span>{pattern.board_length}×{pattern.board_width}×{pattern.board_thickness}</span>
                <span>•</span>
                <span>{pattern.items_count} formatek</span>
                <span>•</span>
                <span>{pattern.total_quantity} szt.</span>
                <span>•</span>
                {getStatusBadge(pattern.status)}
              </div>
            </div>

            {/* Action */}
            <div className="flex-shrink-0">
              {pattern.in_current_plan ? (
                <Badge variant="secondary" className="h-6 text-[10px]">
                  W planie
                </Badge>
              ) : pattern.is_in_plan ? (
                <div className="flex flex-col items-end gap-0.5">
                  <Badge variant="outline" className="h-5 text-[10px] bg-yellow-500/10 border-yellow-500/30">
                    <AlertCircle className="h-2.5 w-2.5 mr-0.5" />
                    {pattern.in_plan_number}
                  </Badge>
                  <Button
                    size="sm"
                    variant="ghost"
                    onClick={() => onAddPattern(pattern)}
                    disabled={isAddingPattern || isLocked}
                    className="h-6 px-2 text-xs"
                    data-testid={`button-add-pattern-${pattern.id}`}
                  >
                    <Plus className="h-3 w-3 mr-1" />
                    Dodaj
                  </Button>
                </div>
              ) : (
                <Button
                  size="sm"
                  variant="ghost"
                  onClick={() => onAddPattern(pattern)}
                  disabled={isAddingPattern || isLocked}
                  className="h-6 px-2 text-xs"
                  data-testid={`button-add-pattern-${pattern.id}`}
                >
                  <Plus className="h-3 w-3 mr-1" />
                  Dodaj
                </Button>
              )}
            </div>
          </div>
        );
      })}
    </div>
  );
}
