import { useState, useEffect } from "react";
import { useQuery } from "@tanstack/react-query";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { Factory, Package, Hammer, CheckCircle2, AlertCircle, ExternalLink, PackageCheck, Clock, XCircle, History } from "lucide-react";
import { Button } from "@/components/ui/button";
import { useLocation } from "wouter";
import { Tooltip, TooltipContent, TooltipTrigger } from "@/components/ui/tooltip";
import { format } from "date-fns";
import { pl } from "date-fns/locale";

interface LineReservationHistoryItem {
  id: number;
  sku: string;
  status: 'ACTIVE' | 'CONSUMED' | 'CANCELLED';
  quantityReserved: number;
  quantityConsumed: number | null;
  reservedAt: string;
  consumedAt: string | null;
  cancelledAt: string | null;
  notes: string | null;
}

interface BomFormatka {
  componentId: number;
  generatedName: string;
  componentType: string;
  length: number;
  width: number;
  thickness: number;
  color: string;
  colorHex: string | null;
  quantity: number;
  edgingPattern: string | null;
  routingVariant: string;
  status: 'produced' | 'reserved' | 'to_production';
  formatkiReservation: {
    id: number;
    sku: string;
    quantityReserved: number;
    materialId: number | null;
    materialName: string | null;
    reservedAt: string | null;
  } | null;
}

interface BomFormatkiResponse {
  lineId: number;
  productSku: string;
  productTitle: string;
  bomId: number | null;
  isPackedProductReserved: boolean;
  formatki: BomFormatka[];
  lineReservationHistory: LineReservationHistoryItem[] | null;
  summary: {
    total: number;
    produced: number;
    toProduction: number;
  };
}

export function BomFormatkiModal() {
  const [isOpen, setIsOpen] = useState(false);
  const [lineId, setLineId] = useState<number | null>(null);
  const [productSku, setProductSku] = useState<string>("");
  const [, setLocation] = useLocation();

  useEffect(() => {
    const handleOpen = (e: CustomEvent<{ lineId: number; productSku: string }>) => {
      setLineId(e.detail.lineId);
      setProductSku(e.detail.productSku);
      setIsOpen(true);
    };

    window.addEventListener('openBomFormatkiModal', handleOpen as EventListener);
    return () => {
      window.removeEventListener('openBomFormatkiModal', handleOpen as EventListener);
    };
  }, []);

  const { data, isLoading, error } = useQuery<BomFormatkiResponse>({
    queryKey: [`/api/production/planning/lines/${lineId}/bom-formatki`],
    enabled: !!lineId && isOpen,
  });

  const handleClose = () => {
    setIsOpen(false);
    setLineId(null);
  };

  const goToWarehouseMaterial = (materialId: number) => {
    setLocation(`/warehouse/formatki/${materialId}`);
    handleClose();
  };

  return (
    <Dialog open={isOpen} onOpenChange={(open) => !open && handleClose()}>
      <DialogContent className="max-w-4xl max-h-[85vh] overflow-hidden flex flex-col">
        <DialogHeader className="pb-2 border-b">
          <div className="flex items-center justify-between gap-4">
            <div className="flex items-center gap-3">
              <div className="h-8 w-8 rounded bg-blue-100 dark:bg-blue-900/30 flex items-center justify-center">
                <Factory className="h-4 w-4 text-blue-600 dark:text-blue-400" />
              </div>
              <div>
                <DialogTitle className="text-base">Formatki BOM</DialogTitle>
                <p className="text-xs text-muted-foreground font-mono">{productSku}</p>
              </div>
            </div>
            {data?.summary && (
              <div className="flex gap-3 flex-wrap">
                <div className="flex items-center gap-1.5 px-2 py-1 rounded bg-muted/50">
                  <Package className="h-3.5 w-3.5 text-muted-foreground" />
                  <span className="text-xs font-medium">{data.summary.total} formatek</span>
                </div>
                {data.isPackedProductReserved && (
                  <div className="flex items-center gap-1.5 px-2 py-1 rounded bg-blue-100 dark:bg-blue-900/30">
                    <PackageCheck className="h-3.5 w-3.5 text-blue-600 dark:text-blue-400" />
                    <span className="text-xs font-medium text-blue-700 dark:text-blue-400">Produkt z magazynu</span>
                  </div>
                )}
                <div className="flex items-center gap-1.5 px-2 py-1 rounded bg-green-100 dark:bg-green-900/30">
                  <CheckCircle2 className="h-3.5 w-3.5 text-green-600 dark:text-green-400" />
                  <span className="text-xs font-medium text-green-700 dark:text-green-400">{data.summary.produced} gotowych</span>
                </div>
                <div className="flex items-center gap-1.5 px-2 py-1 rounded bg-orange-100 dark:bg-orange-900/30">
                  <Hammer className="h-3.5 w-3.5 text-orange-600 dark:text-orange-400" />
                  <span className="text-xs font-medium text-orange-700 dark:text-orange-400">{data.summary.toProduction} do produkcji</span>
                </div>
              </div>
            )}
          </div>
        </DialogHeader>

        <div className="flex-1 overflow-auto py-2">
          {isLoading ? (
            <div className="space-y-2">
              {[...Array(6)].map((_, i) => (
                <Skeleton key={i} className="h-10 w-full" />
              ))}
            </div>
          ) : error ? (
            <div className="flex flex-col items-center justify-center py-8 text-muted-foreground">
              <AlertCircle className="h-10 w-10 mb-2 text-destructive" />
              <p className="text-sm">Nie udało się pobrać danych formatek</p>
            </div>
          ) : !data?.bomId ? (
            <div className="flex flex-col items-center justify-center py-8 text-muted-foreground">
              <Package className="h-10 w-10 mb-2 opacity-50" />
              <p className="text-sm">Produkt nie ma przypisanego BOM</p>
            </div>
          ) : data.formatki.length === 0 ? (
            <div className="flex flex-col items-center justify-center py-8 text-muted-foreground">
              <Factory className="h-10 w-10 mb-2 opacity-50" />
              <p className="text-sm">Brak formatek w strukturze BOM</p>
            </div>
          ) : (
            <Table>
              <TableHeader className="bg-muted/30 sticky top-0 z-10">
                <TableRow className="h-8">
                  <TableHead className="text-[10px] font-semibold py-1 w-[30px]">#</TableHead>
                  <TableHead className="text-[10px] font-semibold py-1">Typ</TableHead>
                  <TableHead className="text-[10px] font-semibold py-1">Nazwa</TableHead>
                  <TableHead className="text-[10px] font-semibold py-1 text-center">DxS</TableHead>
                  <TableHead className="text-[10px] font-semibold py-1 text-center">Gr.</TableHead>
                  <TableHead className="text-[10px] font-semibold py-1">Kolor</TableHead>
                  <TableHead className="text-[10px] font-semibold py-1 text-center">Ilość</TableHead>
                  <TableHead className="text-[10px] font-semibold py-1">Marszruta</TableHead>
                  <TableHead className="text-[10px] font-semibold py-1 text-center">Status</TableHead>
                  <TableHead className="text-[10px] font-semibold py-1 w-[60px]">Magazyn</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {data.formatki.map((formatka, index) => {
                  const rowBgClass = formatka.status === 'produced' 
                    ? 'bg-blue-50/50 dark:bg-blue-900/10' 
                    : formatka.status === 'reserved'
                      ? 'bg-green-50/50 dark:bg-green-900/10'
                      : 'bg-orange-50/50 dark:bg-orange-900/10';
                  
                  return (
                    <TableRow 
                      key={formatka.componentId} 
                      className={`h-9 ${rowBgClass}`}
                    >
                      <TableCell className="py-1 text-[10px] text-muted-foreground font-mono">
                        {index + 1}
                      </TableCell>
                      <TableCell className="py-1">
                        <Badge 
                          variant="outline" 
                          className="h-5 px-1.5 text-[10px] bg-muted/50"
                        >
                          {formatka.componentType}
                        </Badge>
                      </TableCell>
                      <TableCell className="py-1 text-xs font-mono max-w-[160px] truncate" title={formatka.generatedName}>
                        {formatka.generatedName}
                      </TableCell>
                      <TableCell className="py-1 text-xs text-center font-mono">
                        {formatka.length}×{formatka.width}
                      </TableCell>
                      <TableCell className="py-1 text-xs text-center">
                        {formatka.thickness}
                      </TableCell>
                      <TableCell className="py-1">
                        <Badge 
                          variant="outline" 
                          className="h-5 px-1.5 text-[10px]"
                          style={{
                            backgroundColor: formatka.colorHex || '#e5e5e5',
                            borderColor: formatka.colorHex ? adjustColorBrightness(formatka.colorHex, -20) : '#a3a3a3',
                            color: formatka.colorHex ? getContrastColor(formatka.colorHex) : '#374151'
                          }}
                        >
                          {formatka.color}
                        </Badge>
                      </TableCell>
                      <TableCell className="py-1 text-xs text-center font-semibold">
                        {formatka.quantity}
                      </TableCell>
                      <TableCell className="py-1">
                        <Badge 
                          variant="outline" 
                          className={`h-5 px-1.5 text-[10px] ${
                            formatka.routingVariant === 'OKLEINOWANA' 
                              ? 'bg-purple-100 dark:bg-purple-900/30 border-purple-500 text-purple-700 dark:text-purple-400'
                              : 'bg-gray-100 dark:bg-gray-800/50 border-gray-400 text-gray-600 dark:text-gray-400'
                          }`}
                        >
                          {formatka.routingVariant}
                        </Badge>
                      </TableCell>
                      <TableCell className="py-1 text-center">
                        {formatka.status === 'produced' ? (
                          <Badge 
                            className="h-5 px-1.5 text-[10px] bg-blue-600 hover:bg-blue-600 text-white border-transparent"
                          >
                            <PackageCheck className="h-3 w-3 mr-0.5" />
                            Wyprodukowana
                          </Badge>
                        ) : formatka.status === 'reserved' ? (
                          <Badge 
                            className="h-5 px-1.5 text-[10px] bg-green-600 hover:bg-green-600 text-white border-transparent"
                          >
                            <CheckCircle2 className="h-3 w-3 mr-0.5" />
                            Zarezerwowana
                          </Badge>
                        ) : (
                          <Badge 
                            variant="outline"
                            className="h-5 px-1.5 text-[10px] bg-orange-100 dark:bg-orange-900/30 border-orange-500 text-orange-700 dark:text-orange-400"
                          >
                            <Hammer className="h-3 w-3 mr-0.5" />
                            Do produkcji
                          </Badge>
                        )}
                      </TableCell>
                      <TableCell className="py-1">
                        {formatka.formatkiReservation?.materialId ? (
                          <Button
                            variant="ghost"
                            size="sm"
                            className="h-6 w-6 p-0"
                            onClick={() => goToWarehouseMaterial(formatka.formatkiReservation!.materialId!)}
                            title={`Przejdź do: ${formatka.formatkiReservation.materialName || formatka.formatkiReservation.sku}`}
                          >
                            <ExternalLink className="h-3.5 w-3.5" />
                          </Button>
                        ) : (
                          <span className="text-muted-foreground text-[10px]">-</span>
                        )}
                      </TableCell>
                    </TableRow>
                  );
                })}
              </TableBody>
            </Table>
          )}
          
          {data?.lineReservationHistory && data.lineReservationHistory.length > 0 && (
            <div className="mt-4 border-t pt-3" data-testid="section-reservation-history">
              <div className="flex items-center gap-2 mb-2">
                <History className="h-4 w-4 text-muted-foreground" />
                <span className="text-xs font-semibold">Historia rezerwacji formatek ({data.lineReservationHistory.length})</span>
              </div>
              <div className="space-y-1.5 max-h-[150px] overflow-y-auto">
                {data.lineReservationHistory.map((item) => (
                  <div 
                    key={item.id} 
                    className="flex flex-col gap-1 p-2 rounded bg-muted/30 text-[11px]"
                    data-testid={`row-history-${item.id}`}
                  >
                    <div className="flex items-center justify-between gap-2">
                      <div className="flex items-center gap-2">
                        {item.status === 'ACTIVE' ? (
                          <Clock className="h-3.5 w-3.5 text-yellow-500 flex-shrink-0" />
                        ) : item.status === 'CANCELLED' ? (
                          <XCircle className="h-3.5 w-3.5 text-red-500 flex-shrink-0" />
                        ) : (
                          <CheckCircle2 className="h-3.5 w-3.5 text-green-500 flex-shrink-0" />
                        )}
                        <Badge 
                          variant="outline" 
                          className={`h-4 px-1.5 text-[9px] ${
                            item.status === 'ACTIVE' 
                              ? 'bg-yellow-100 dark:bg-yellow-900/30 text-yellow-700 dark:text-yellow-400 border-yellow-500' 
                              : item.status === 'CANCELLED'
                                ? 'bg-red-100 dark:bg-red-900/30 text-red-700 dark:text-red-400 border-red-500'
                                : 'bg-green-100 dark:bg-green-900/30 text-green-700 dark:text-green-400 border-green-500'
                          }`}
                        >
                          {item.status === 'ACTIVE' ? 'W toku' : item.status === 'CANCELLED' ? 'Anulowana' : 'Zużyta'}
                        </Badge>
                        <span className="font-mono text-muted-foreground">{item.sku}</span>
                        <span className="font-medium">{item.quantityReserved} szt.</span>
                      </div>
                      <div className="flex items-center gap-3 text-muted-foreground">
                        {item.reservedAt && (
                          <span className="flex items-center gap-1">
                            <span className="text-yellow-600 dark:text-yellow-400">+</span>
                            {format(new Date(item.reservedAt), 'dd.MM.yy HH:mm', { locale: pl })}
                          </span>
                        )}
                        {item.cancelledAt && (
                          <span className="flex items-center gap-1">
                            <span className="text-red-600 dark:text-red-400">-</span>
                            {format(new Date(item.cancelledAt), 'dd.MM.yy HH:mm', { locale: pl })}
                          </span>
                        )}
                        {item.consumedAt && (
                          <span className="flex items-center gap-1">
                            <span className="text-green-600 dark:text-green-400">✓</span>
                            {format(new Date(item.consumedAt), 'dd.MM.yy HH:mm', { locale: pl })}
                          </span>
                        )}
                      </div>
                    </div>
                    {item.notes && (
                      <div className="text-muted-foreground text-[10px] italic pl-5" data-testid={`text-notes-${item.id}`}>
                        {item.notes}
                      </div>
                    )}
                  </div>
                ))}
              </div>
            </div>
          )}
        </div>
      </DialogContent>
    </Dialog>
  );
}

function adjustColorBrightness(hex: string, percent: number): string {
  const num = parseInt(hex.replace('#', ''), 16);
  const r = Math.min(255, Math.max(0, (num >> 16) + percent));
  const g = Math.min(255, Math.max(0, ((num >> 8) & 0x00FF) + percent));
  const b = Math.min(255, Math.max(0, (num & 0x0000FF) + percent));
  return `#${((r << 16) | (g << 8) | b).toString(16).padStart(6, '0')}`;
}

function getContrastColor(hex: string): string {
  const num = parseInt(hex.replace('#', ''), 16);
  const r = (num >> 16) & 0xFF;
  const g = (num >> 8) & 0xFF;
  const b = num & 0xFF;
  const luminance = (0.299 * r + 0.587 * g + 0.114 * b) / 255;
  return luminance > 0.5 ? '#000000' : '#ffffff';
}
