import { useState, useEffect } from "react";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Loader2 } from "lucide-react";

interface CatalogProduct {
  id: number;
  sku: string;
  title: string;
  color: string | null;
  length: string | null;
  width: string | null;
  height: string | null;
}

interface AddCatalogProductDialogProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  product: CatalogProduct | null;
  onConfirm: (quantity: number) => void;
  isLoading?: boolean;
}

export function AddCatalogProductDialog({
  open,
  onOpenChange,
  product,
  onConfirm,
  isLoading = false
}: AddCatalogProductDialogProps) {
  const [quantity, setQuantity] = useState("1");
  
  // Reset quantity when dialog opens or product changes
  useEffect(() => {
    if (open && product) {
      setQuantity("1");
    }
  }, [open, product]);

  const handleConfirm = () => {
    const qty = parseInt(quantity);
    if (isNaN(qty) || qty < 1) {
      return;
    }
    onConfirm(qty);
  };

  const handleKeyDown = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter' && !isLoading) {
      handleConfirm();
    }
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-md" data-testid="dialog-add-catalog-product">
        <DialogHeader>
          <DialogTitle>Dodaj produkt do ZLP</DialogTitle>
          <DialogDescription>
            Określ ilość sztuk produktu do wyprodukowania
          </DialogDescription>
        </DialogHeader>

        {product && (
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <div className="text-sm font-medium text-muted-foreground">Produkt</div>
              <div className="space-y-1">
                <div className="font-medium">{product.title}</div>
                <div className="text-sm text-muted-foreground">SKU: {product.sku}</div>
                {product.color && (
                  <div className="text-sm text-muted-foreground">Kolor: {product.color}</div>
                )}
                {(product.length || product.width) && (
                  <div className="text-sm text-muted-foreground">
                    Wymiary: {product.length || '?'} × {product.width || '?'} 
                    {product.height && ` × ${product.height}`} mm
                  </div>
                )}
              </div>
            </div>

            <div className="space-y-2">
              <Label htmlFor="quantity">Ilość sztuk</Label>
              <Input
                id="quantity"
                type="number"
                min="1"
                value={quantity}
                onChange={(e) => setQuantity(e.target.value)}
                onKeyDown={handleKeyDown}
                autoFocus
                disabled={isLoading}
                data-testid="input-quantity"
              />
            </div>
          </div>
        )}

        <DialogFooter>
          <Button
            variant="outline"
            onClick={() => onOpenChange(false)}
            disabled={isLoading}
            data-testid="button-cancel"
          >
            Anuluj
          </Button>
          <Button
            onClick={handleConfirm}
            disabled={isLoading || !quantity || parseInt(quantity) < 1}
            data-testid="button-confirm"
          >
            {isLoading && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
            Dodaj do planu
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}
