import { useState } from "react";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Tooltip, TooltipContent, TooltipTrigger } from "@/components/ui/tooltip";
import { 
  Dialog, 
  DialogContent, 
  DialogHeader, 
  DialogTitle, 
  DialogDescription,
  DialogFooter 
} from "@/components/ui/dialog";
import { useMutation } from "@tanstack/react-query";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import { 
  Play, 
  CheckCircle, 
  Clock, 
  Pause, 
  ArrowRight, 
  Factory, 
  Timer,
  User,
  Loader2,
  AlertTriangle,
  XCircle
} from "lucide-react";
import { format } from "date-fns";
import { pl } from "date-fns/locale";

interface WorkOrder {
  id: number;
  workOrderNumber: string;
  sequence: number;
  status: string;
  quantityPlanned: string | number;
  quantityProduced: string | number;
  quantityScrap?: string | number;
  qualityCheckPassed?: boolean;
  qualityCheckNotes?: string;
  notes?: string;
  routingOperationId: number | null;
  operationCode: string | null;
  operationName: string | null;
  routingId?: number | null;
  routingCode?: string | null;
  routingName?: string | null;
  workCenterId: number | null;
  workCenterCode: string | null;
  workCenterName: string | null;
  scheduledStartTime: string | null;
  scheduledEndTime: string | null;
  actualStartTime: string | null;
  actualEndTime: string | null;
}

interface RoutingPath {
  routingKey: string;
  routingId: number;
  routingCode: string;
  routingName: string;
  flowCode: string;
  color: string;
  textColor: string;
  borderColor: string;
  workOrders: WorkOrder[];
}

interface DamagedFormatka {
  id: number;
  generatedName: string;
  damageType?: string;
  damageNotes?: string;
}

interface WorkOrderFlowProps {
  workOrders: WorkOrder[];
  productionOrderId: number;
  orderStatus: string;
  damagedFormatki?: DamagedFormatka[];
}

const statusConfig: Record<string, { color: string; bgColor: string; label: string; icon: any }> = {
  pending: { 
    color: "text-gray-500", 
    bgColor: "bg-gray-100 dark:bg-gray-800", 
    label: "Oczekuje",
    icon: Clock 
  },
  ready: { 
    color: "text-blue-500", 
    bgColor: "bg-blue-100 dark:bg-blue-900/30", 
    label: "Gotowe",
    icon: Play 
  },
  in_progress: { 
    color: "text-yellow-500", 
    bgColor: "bg-yellow-100 dark:bg-yellow-900/30", 
    label: "W trakcie",
    icon: Loader2 
  },
  paused: { 
    color: "text-orange-500", 
    bgColor: "bg-orange-100 dark:bg-orange-900/30", 
    label: "Wstrzymane",
    icon: Pause 
  },
  done: { 
    color: "text-green-500", 
    bgColor: "bg-green-100 dark:bg-green-900/30", 
    label: "Zakończone",
    icon: CheckCircle 
  },
  cancelled: { 
    color: "text-red-500", 
    bgColor: "bg-red-100 dark:bg-red-900/30", 
    label: "Anulowane",
    icon: Clock 
  },
};

const operationIcons: Record<string, string> = {
  cutting: "✂️",
  edging: "📏",
  drilling: "🔩",
  drilling_holes: "🔩",
  drilling_mount: "🔧",
  upholstering: "🪑",
  assembly: "🔨",
  packing: "📦",
  strapping: "🎁",
  magazynowanie: "🏪",
  transport_samochodowy: "🚚",
};

const getFlowCodeFromRouting = (routingCode: string | null | undefined): string => {
  if (!routingCode) return 'DEFAULT';
  const code = routingCode.toLowerCase();
  if (code.includes('cow')) return 'COW';
  if (code.includes('_co')) return 'CO';
  if (code.includes('_cw')) return 'CW';
  if (code.includes('_c_')) return 'C';
  return 'DEFAULT';
};

const flowCodeColors: Record<string, { bg: string; text: string; border: string }> = {
  CO: { bg: 'bg-blue-500 dark:bg-blue-600', text: 'text-white', border: 'border-l-blue-500 dark:border-l-blue-600' },
  COW: { bg: 'bg-purple-500 dark:bg-purple-600', text: 'text-white', border: 'border-l-purple-500 dark:border-l-purple-600' },
  CW: { bg: 'bg-orange-500 dark:bg-orange-600', text: 'text-white', border: 'border-l-orange-500 dark:border-l-orange-600' },
  C: { bg: 'bg-teal-500 dark:bg-teal-600', text: 'text-white', border: 'border-l-teal-500 dark:border-l-teal-600' },
  DEFAULT: { bg: 'bg-gray-500 dark:bg-gray-600', text: 'text-white', border: 'border-l-gray-500 dark:border-l-gray-600' },
};

const extractRoutingIdFromWorkOrderNumber = (woNumber: string): number | null => {
  const match = woNumber.match(/WO-\d+-R(\d+)-/);
  if (match) {
    return parseInt(match[1], 10);
  }
  return null;
};

export function WorkOrderFlow({ workOrders, productionOrderId, orderStatus, damagedFormatki = [] }: WorkOrderFlowProps) {
  const { toast } = useToast();
  const [completeDialogOpen, setCompleteDialogOpen] = useState(false);
  const [selectedWorkOrder, setSelectedWorkOrder] = useState<WorkOrder | null>(null);
  
  const startWorkOrderMutation = useMutation({
    mutationFn: async (workOrderId: number) => {
      const res = await apiRequest("POST", `/api/production/work-orders/${workOrderId}/start`);
      return await res.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const firstKey = query.queryKey[0];
          return firstKey === "/api/production/orders";
        },
      });
      toast({
        title: "Sukces",
        description: "Operacja rozpoczęta",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się rozpocząć operacji",
      });
    },
  });

  const completeWorkOrderMutation = useMutation({
    mutationFn: async (workOrderId: number) => {
      const res = await apiRequest("POST", `/api/production/work-orders/${workOrderId}/complete`, {});
      return await res.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const firstKey = query.queryKey[0];
          return firstKey === "/api/production/orders";
        },
      });
      setCompleteDialogOpen(false);
      setSelectedWorkOrder(null);
      toast({
        title: "Sukces",
        description: "Operacja zakończona",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się zakończyć operacji",
      });
    },
  });

  const handleCompleteClick = (wo: WorkOrder) => {
    setSelectedWorkOrder(wo);
    setCompleteDialogOpen(true);
  };

  const handleConfirmComplete = () => {
    if (!selectedWorkOrder) return;
    completeWorkOrderMutation.mutate(selectedWorkOrder.id);
  };

  const formatTime = (date: string | null) => {
    if (!date) return null;
    return format(new Date(date), "HH:mm", { locale: pl });
  };

  const formatDateTime = (date: string | null) => {
    if (!date) return "-";
    return format(new Date(date), "dd.MM HH:mm", { locale: pl });
  };

  const getCompletedCount = () => workOrders.filter(wo => wo.status === 'done').length;
  const getInProgressCount = () => workOrders.filter(wo => wo.status === 'in_progress').length;
  const getProgressPercentage = () => Math.round((getCompletedCount() / workOrders.length) * 100);

  const routingPaths: RoutingPath[] = (() => {
    const pathMap = new Map<string, RoutingPath>();
    
    workOrders.forEach((wo, idx) => {
      let routingId = wo.routingId;
      if (!routingId && wo.workOrderNumber) {
        routingId = extractRoutingIdFromWorkOrderNumber(wo.workOrderNumber);
      }
      
      const routingKey = routingId 
        ? `routing-${routingId}` 
        : (wo.routingCode ? `code-${wo.routingCode}` : `wo-${idx}`);
      
      if (!pathMap.has(routingKey)) {
        const flowCode = getFlowCodeFromRouting(wo.routingCode);
        const colors = flowCodeColors[flowCode] || flowCodeColors.DEFAULT;
        pathMap.set(routingKey, {
          routingKey,
          routingId: routingId || 0,
          routingCode: wo.routingCode || '',
          routingName: wo.routingName || '',
          flowCode,
          color: colors.bg,
          textColor: colors.text,
          borderColor: colors.border,
          workOrders: [],
        });
      }
      pathMap.get(routingKey)!.workOrders.push(wo);
    });
    
    return Array.from(pathMap.values()).sort((a, b) => {
      const firstSeqA = a.workOrders[0]?.sequence || 0;
      const firstSeqB = b.workOrders[0]?.sequence || 0;
      return firstSeqA - firstSeqB;
    });
  })();

  const hasMultiplePaths = routingPaths.length > 1;

  const canStartWorkOrderInPath = (wo: WorkOrder, indexInPath: number, pathWorkOrders: WorkOrder[]) => {
    if (orderStatus !== 'in_progress' && orderStatus !== 'confirmed') return false;
    if (wo.status !== 'pending' && wo.status !== 'ready') return false;
    if (indexInPath === 0) return true;
    return pathWorkOrders[indexInPath - 1]?.status === 'done';
  };

  const canCompleteWorkOrder = (wo: WorkOrder) => {
    return wo.status === 'in_progress';
  };

  if (!workOrders || workOrders.length === 0) {
    return null;
  }

  return (
    <Card className="lg:col-span-2">
      <CardHeader className="pb-3">
        <div className="flex items-center justify-between">
          <CardTitle className="flex items-center gap-2">
            <Factory className="h-5 w-5" />
            Przepływ Operacji
          </CardTitle>
          <div className="flex items-center gap-3 text-sm">
            <div className="flex items-center gap-1.5">
              <div className="h-2 w-2 rounded-full bg-green-500" />
              <span className="text-muted-foreground">
                {getCompletedCount()}/{workOrders.length} zakończone
              </span>
            </div>
            {damagedFormatki.length > 0 && (
              <>
                <div className="h-4 w-px bg-border" />
                <div className="flex items-center gap-1.5">
                  <AlertTriangle className="h-3.5 w-3.5 text-red-500" />
                  <span className="text-red-500 font-medium">{damagedFormatki.length} uszkodz.</span>
                </div>
              </>
            )}
            <div className="h-4 w-px bg-border" />
            <div className="flex items-center gap-1.5">
              <Timer className="h-3.5 w-3.5 text-muted-foreground" />
              <span className="text-muted-foreground">{getProgressPercentage()}%</span>
            </div>
          </div>
        </div>
        
        <div className="w-full h-2 bg-muted rounded-full overflow-hidden mt-2">
          <div 
            className="h-full bg-gradient-to-r from-green-500 to-green-400 transition-all duration-500"
            style={{ width: `${getProgressPercentage()}%` }}
          />
        </div>
      </CardHeader>
      
      <CardContent>
        {hasMultiplePaths && (
          <div className="flex flex-wrap items-center gap-2 mb-4">
            <span className="text-xs text-muted-foreground">Ścieżki:</span>
            {routingPaths.map((path) => {
              const completedInPath = path.workOrders.filter(wo => wo.status === 'done').length;
              return (
                <Badge 
                  key={path.routingKey} 
                  className={`${path.color} ${path.textColor} text-xs`}
                  data-testid={`badge-routing-path-${path.routingKey}`}
                >
                  {path.flowCode} ({completedInPath}/{path.workOrders.length})
                </Badge>
              );
            })}
          </div>
        )}
        
        <div className="space-y-4">
          {routingPaths.map((path, pathIndex) => (
            <div key={path.routingKey} className="space-y-2" data-testid={`routing-path-${path.routingKey}`}>
              {hasMultiplePaths && (
                <div className="flex items-center gap-2 mb-2">
                  <Badge className={`${path.color} ${path.textColor} text-[10px]`}>
                    {path.flowCode}
                  </Badge>
                  <span className="text-xs text-muted-foreground">
                    {path.routingName || path.routingCode}
                  </span>
                  <span className="text-xs text-muted-foreground">
                    ({path.workOrders.length} operacji)
                  </span>
                </div>
              )}
              
              <div className="flex items-start gap-0 overflow-x-auto pb-2">
                {path.workOrders.map((wo, index) => {
                  const config = statusConfig[wo.status] || statusConfig.pending;
                  const StatusIcon = config.icon;
                  const operationIcon = wo.operationCode ? operationIcons[wo.operationCode] : "⚙️";
                  const isActive = wo.status === 'in_progress';
                  const isCompleted = wo.status === 'done';
                  
                  return (
                    <div key={wo.id} className="flex items-start" data-testid={`work-order-flow-item-${wo.id}`}>
                      <div 
                        className={`
                          relative flex flex-col min-w-[160px] p-3 rounded-lg border transition-all
                          ${isActive ? 'ring-2 ring-yellow-500 shadow-lg shadow-yellow-500/20' : ''}
                          ${isCompleted ? 'opacity-80' : ''}
                          ${config.bgColor}
                          ${hasMultiplePaths ? `border-l-4 ${path.borderColor} border-border` : 'border-border'}
                        `}
                      >
                        <div className="flex items-center justify-between mb-2">
                          <span className="text-lg">{operationIcon}</span>
                          <Badge 
                            variant="outline" 
                            className={`text-[10px] px-1.5 py-0 ${config.color}`}
                          >
                            {config.label}
                          </Badge>
                        </div>
                        
                        <div className="flex-1">
                          <div className="font-medium text-sm leading-tight">
                            {wo.operationName || `Operacja ${wo.sequence}`}
                          </div>
                          {wo.operationCode && (
                            <div className="text-[10px] text-muted-foreground font-mono mt-0.5">
                              {wo.operationCode}
                            </div>
                          )}
                        </div>
                        
                        {wo.workCenterName && (
                          <div className="flex items-center gap-1 mt-2 text-[10px] text-muted-foreground">
                            <Factory className="h-3 w-3" />
                            <span className="truncate">{wo.workCenterName}</span>
                          </div>
                        )}
                        
                        <div className="flex items-center justify-between mt-2 pt-2 border-t border-border/50">
                          <div className="text-[10px] text-muted-foreground">
                            {wo.actualStartTime ? (
                              <span className="text-green-600 dark:text-green-400">
                                {formatTime(wo.actualStartTime)}
                              </span>
                            ) : wo.scheduledStartTime ? (
                              <span>{formatTime(wo.scheduledStartTime)}</span>
                            ) : (
                              <span>-</span>
                            )}
                            {" → "}
                            {wo.actualEndTime ? (
                              <span className="text-green-600 dark:text-green-400">
                                {formatTime(wo.actualEndTime)}
                              </span>
                            ) : wo.scheduledEndTime ? (
                              <span>{formatTime(wo.scheduledEndTime)}</span>
                            ) : (
                              <span>-</span>
                            )}
                          </div>
                          <div className="font-mono text-[10px]">
                            {wo.quantityProduced ?? 0}/{wo.quantityPlanned}
                          </div>
                        </div>
                        
                        <div className="flex gap-1 mt-2">
                          {canStartWorkOrderInPath(wo, index, path.workOrders) && (
                            <Tooltip>
                              <TooltipTrigger asChild>
                                <Button
                                  size="sm"
                                  variant="default"
                                  className="flex-1 h-7 text-xs"
                                  onClick={() => startWorkOrderMutation.mutate(wo.id)}
                                  disabled={startWorkOrderMutation.isPending}
                                  data-testid={`button-start-work-order-${wo.id}`}
                                >
                                  {startWorkOrderMutation.isPending ? (
                                    <Loader2 className="h-3 w-3 animate-spin" />
                                  ) : (
                                    <>
                                      <Play className="h-3 w-3 mr-1" />
                                      Start
                                    </>
                                  )}
                                </Button>
                              </TooltipTrigger>
                              <TooltipContent>Rozpocznij operację</TooltipContent>
                            </Tooltip>
                          )}
                          
                          {canCompleteWorkOrder(wo) && (
                            <Tooltip>
                              <TooltipTrigger asChild>
                                <Button
                                  size="sm"
                                  variant="default"
                                  className="flex-1 h-7 text-xs bg-green-600 hover:bg-green-700"
                                  onClick={() => handleCompleteClick(wo)}
                                  disabled={completeWorkOrderMutation.isPending}
                                  data-testid={`button-complete-work-order-${wo.id}`}
                                >
                                  {completeWorkOrderMutation.isPending ? (
                                    <Loader2 className="h-3 w-3 animate-spin" />
                                  ) : (
                                    <>
                                      <CheckCircle className="h-3 w-3 mr-1" />
                                      Zakończ
                                    </>
                                  )}
                                </Button>
                              </TooltipTrigger>
                              <TooltipContent>Zakończ operację</TooltipContent>
                            </Tooltip>
                          )}
                        </div>
                        
                        {isActive && (
                          <div className="absolute -top-1 -right-1 h-3 w-3 rounded-full bg-yellow-500 animate-pulse" />
                        )}
                      </div>
                      
                      {index < path.workOrders.length - 1 && (
                        <div className="flex items-center justify-center w-8 h-[100px] flex-shrink-0">
                          <ArrowRight 
                            className={`h-5 w-5 ${
                              wo.status === 'done' 
                                ? 'text-green-500' 
                                : 'text-muted-foreground/40'
                            }`} 
                          />
                        </div>
                      )}
                    </div>
                  );
                })}
              </div>
            </div>
          ))}
        </div>
      </CardContent>
      
      <Dialog open={completeDialogOpen} onOpenChange={setCompleteDialogOpen}>
        <DialogContent className="sm:max-w-md">
          <DialogHeader>
            <DialogTitle>Zakończ operację</DialogTitle>
            <DialogDescription>
              {selectedWorkOrder?.operationName || selectedWorkOrder?.workOrderNumber}
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4 py-4">
            <div className="grid grid-cols-2 gap-4 text-sm">
              <div>
                <span className="text-muted-foreground">Ilość zaplanowana:</span>
                <span className="ml-2 font-medium">{selectedWorkOrder?.quantityPlanned} szt.</span>
              </div>
            </div>
            
            {damagedFormatki.length > 0 && (
              <div className="p-3 bg-yellow-100 dark:bg-yellow-900/30 rounded-md space-y-2">
                <div className="flex items-center gap-2">
                  <AlertTriangle className="h-4 w-4 text-yellow-600" />
                  <span className="text-sm font-medium text-yellow-700 dark:text-yellow-400">
                    {damagedFormatki.length} szt. oznaczonych jako uszkodzone:
                  </span>
                </div>
                <div className="max-h-32 overflow-y-auto space-y-1 pl-6">
                  {damagedFormatki.map((f) => (
                    <div key={f.id} className="text-xs text-yellow-700 dark:text-yellow-400 flex items-center gap-2">
                      <span className="font-medium">{f.generatedName}</span>
                      {f.damageType && (
                        <span className="text-yellow-600 dark:text-yellow-500">({f.damageType})</span>
                      )}
                    </div>
                  ))}
                </div>
              </div>
            )}
            
            <p className="text-sm text-muted-foreground">
              Czy na pewno chcesz zakończyć tę operację?
            </p>
          </div>
          
          <DialogFooter className="gap-2">
            <Button
              variant="outline"
              onClick={() => setCompleteDialogOpen(false)}
              data-testid="button-cancel-complete"
            >
              Anuluj
            </Button>
            <Button
              onClick={handleConfirmComplete}
              disabled={completeWorkOrderMutation.isPending}
              data-testid="button-confirm-complete"
            >
              {completeWorkOrderMutation.isPending ? (
                <Loader2 className="h-4 w-4 animate-spin mr-2" />
              ) : (
                <CheckCircle className="h-4 w-4 mr-2" />
              )}
              Zakończ operację
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </Card>
  );
}

export function WorkOrderTable({ workOrders, productionOrderId, orderStatus, damagedFormatki = [] }: WorkOrderFlowProps) {
  const { toast } = useToast();
  const [completeDialogOpen, setCompleteDialogOpen] = useState(false);
  const [selectedWorkOrder, setSelectedWorkOrder] = useState<WorkOrder | null>(null);

  const startWorkOrderMutation = useMutation({
    mutationFn: async (workOrderId: number) => {
      const res = await apiRequest("POST", `/api/production/work-orders/${workOrderId}/start`);
      return await res.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const firstKey = query.queryKey[0];
          return firstKey === "/api/production/orders";
        },
      });
      toast({
        title: "Sukces",
        description: "Operacja rozpoczęta",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się rozpocząć operacji",
      });
    },
  });

  const completeWorkOrderMutation = useMutation({
    mutationFn: async (workOrderId: number) => {
      const res = await apiRequest("POST", `/api/production/work-orders/${workOrderId}/complete`, {});
      return await res.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const firstKey = query.queryKey[0];
          return firstKey === "/api/production/orders";
        },
      });
      setCompleteDialogOpen(false);
      setSelectedWorkOrder(null);
      toast({
        title: "Sukces",
        description: "Operacja zakończona",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się zakończyć operacji",
      });
    },
  });

  const handleCompleteClick = (wo: WorkOrder) => {
    setSelectedWorkOrder(wo);
    setCompleteDialogOpen(true);
  };

  const handleConfirmComplete = () => {
    if (!selectedWorkOrder) return;
    completeWorkOrderMutation.mutate(selectedWorkOrder.id);
  };

  const formatDateTime = (date: string | null) => {
    if (!date) return "-";
    return format(new Date(date), "dd.MM HH:mm", { locale: pl });
  };

  const canStartWorkOrder = (wo: WorkOrder, index: number) => {
    if (orderStatus !== 'in_progress' && orderStatus !== 'confirmed') return false;
    if (wo.status !== 'pending' && wo.status !== 'ready') return false;
    if (index === 0) return true;
    return workOrders[index - 1]?.status === 'done';
  };

  const canCompleteWorkOrder = (wo: WorkOrder) => {
    return wo.status === 'in_progress';
  };

  if (!workOrders || workOrders.length === 0) {
    return null;
  }

  return (
    <Card className="lg:col-span-2">
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <Factory className="h-5 w-5" />
          Operacje - Szczegóły ({workOrders.length})
        </CardTitle>
      </CardHeader>
      <CardContent>
        <div className="overflow-x-auto">
          <table className="w-full text-sm">
            <thead>
              <tr className="border-b">
                <th className="text-left py-2 px-3 font-medium text-muted-foreground w-10">#</th>
                <th className="text-left py-2 px-3 font-medium text-muted-foreground">Operacja</th>
                <th className="text-left py-2 px-3 font-medium text-muted-foreground">Stanowisko</th>
                <th className="text-left py-2 px-3 font-medium text-muted-foreground">Status</th>
                <th className="text-center py-2 px-3 font-medium text-muted-foreground">Rzecz. Start</th>
                <th className="text-center py-2 px-3 font-medium text-muted-foreground">Rzecz. Koniec</th>
                <th className="text-right py-2 px-3 font-medium text-muted-foreground">Ilość</th>
                <th className="text-right py-2 px-3 font-medium text-muted-foreground">Uszkodz.</th>
                <th className="text-right py-2 px-3 font-medium text-muted-foreground w-24">Akcje</th>
              </tr>
            </thead>
            <tbody>
              {workOrders.map((wo, index) => {
                const config = statusConfig[wo.status] || statusConfig.pending;
                
                return (
                  <tr 
                    key={wo.id} 
                    className={`border-b hover:bg-muted/50 ${
                      wo.status === 'in_progress' ? 'bg-yellow-500/10' : ''
                    }`}
                    data-testid={`work-order-row-${wo.id}`}
                  >
                    <td className="py-2 px-3 font-mono text-muted-foreground">{wo.sequence}</td>
                    <td className="py-2 px-3">
                      <div className="font-medium">{wo.operationName || wo.workOrderNumber}</div>
                      {wo.operationCode && (
                        <div className="text-xs text-muted-foreground font-mono">{wo.operationCode}</div>
                      )}
                    </td>
                    <td className="py-2 px-3">
                      {wo.workCenterName ? (
                        <div>
                          <div className="font-medium">{wo.workCenterName}</div>
                          {wo.workCenterCode && (
                            <div className="text-xs text-muted-foreground font-mono">{wo.workCenterCode}</div>
                          )}
                        </div>
                      ) : (
                        <span className="text-muted-foreground">-</span>
                      )}
                    </td>
                    <td className="py-2 px-3">
                      <Badge 
                        variant="outline" 
                        className={`${config.bgColor} ${config.color} border-current`}
                      >
                        {config.label}
                      </Badge>
                    </td>
                    <td className="py-2 px-3 text-center text-xs">
                      {wo.actualStartTime ? (
                        <span className="text-green-600 dark:text-green-400 font-medium">
                          {formatDateTime(wo.actualStartTime)}
                        </span>
                      ) : (
                        <span className="text-muted-foreground">-</span>
                      )}
                    </td>
                    <td className="py-2 px-3 text-center text-xs">
                      {wo.actualEndTime ? (
                        <span className="text-green-600 dark:text-green-400 font-medium">
                          {formatDateTime(wo.actualEndTime)}
                        </span>
                      ) : (
                        <span className="text-muted-foreground">-</span>
                      )}
                    </td>
                    <td className="py-2 px-3 text-right font-mono">
                      <span className={wo.quantityProduced && parseFloat(String(wo.quantityProduced)) > 0 ? 'text-green-600 dark:text-green-400' : ''}>
                        {wo.quantityProduced ?? 0}
                      </span>
                      <span className="text-muted-foreground">/{wo.quantityPlanned}</span>
                    </td>
                    <td className="py-2 px-3 text-right">
                      {wo.quantityScrap && parseFloat(String(wo.quantityScrap)) > 0 ? (
                        <Tooltip>
                          <TooltipTrigger asChild>
                            <div className="flex items-center justify-end gap-1 cursor-help">
                              <XCircle className="h-3 w-3 text-red-500" />
                              <span className="text-red-500 font-medium">{wo.quantityScrap}</span>
                            </div>
                          </TooltipTrigger>
                          <TooltipContent>
                            <p className="font-medium">Uszkodzone elementy</p>
                            {wo.qualityCheckNotes && (
                              <p className="text-xs text-muted-foreground mt-1">{wo.qualityCheckNotes}</p>
                            )}
                          </TooltipContent>
                        </Tooltip>
                      ) : (
                        <span className="text-muted-foreground">-</span>
                      )}
                    </td>
                    <td className="py-2 px-3 text-right">
                      <div className="flex justify-end gap-1">
                        {canStartWorkOrder(wo, index) && (
                          <Button
                            size="sm"
                            variant="default"
                            className="h-7 px-2"
                            onClick={() => startWorkOrderMutation.mutate(wo.id)}
                            disabled={startWorkOrderMutation.isPending}
                            data-testid={`button-start-wo-table-${wo.id}`}
                          >
                            {startWorkOrderMutation.isPending ? (
                              <Loader2 className="h-3 w-3 animate-spin" />
                            ) : (
                              <Play className="h-3 w-3" />
                            )}
                          </Button>
                        )}
                        {canCompleteWorkOrder(wo) && (
                          <Button
                            size="sm"
                            variant="default"
                            className="h-7 px-2 bg-green-600 hover:bg-green-700"
                            onClick={() => handleCompleteClick(wo)}
                            disabled={completeWorkOrderMutation.isPending}
                            data-testid={`button-complete-wo-table-${wo.id}`}
                          >
                            {completeWorkOrderMutation.isPending ? (
                              <Loader2 className="h-3 w-3 animate-spin" />
                            ) : (
                              <CheckCircle className="h-3 w-3" />
                            )}
                          </Button>
                        )}
                        {wo.status === 'done' && (
                          <CheckCircle className="h-4 w-4 text-green-500" />
                        )}
                      </div>
                    </td>
                  </tr>
                );
              })}
            </tbody>
          </table>
        </div>
      </CardContent>
      
      <Dialog open={completeDialogOpen} onOpenChange={setCompleteDialogOpen}>
        <DialogContent className="sm:max-w-md">
          <DialogHeader>
            <DialogTitle>Zakończ operację</DialogTitle>
            <DialogDescription>
              {selectedWorkOrder?.operationName || selectedWorkOrder?.workOrderNumber}
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4 py-4">
            <div className="grid grid-cols-2 gap-4 text-sm">
              <div>
                <span className="text-muted-foreground">Ilość zaplanowana:</span>
                <span className="ml-2 font-medium">{selectedWorkOrder?.quantityPlanned} szt.</span>
              </div>
            </div>
            
            {damagedFormatki.length > 0 && (
              <div className="p-3 bg-yellow-100 dark:bg-yellow-900/30 rounded-md space-y-2">
                <div className="flex items-center gap-2">
                  <AlertTriangle className="h-4 w-4 text-yellow-600" />
                  <span className="text-sm font-medium text-yellow-700 dark:text-yellow-400">
                    {damagedFormatki.length} szt. oznaczonych jako uszkodzone:
                  </span>
                </div>
                <div className="max-h-32 overflow-y-auto space-y-1 pl-6">
                  {damagedFormatki.map((f) => (
                    <div key={f.id} className="text-xs text-yellow-700 dark:text-yellow-400 flex items-center gap-2">
                      <span className="font-medium">{f.generatedName}</span>
                      {f.damageType && (
                        <span className="text-yellow-600 dark:text-yellow-500">({f.damageType})</span>
                      )}
                    </div>
                  ))}
                </div>
              </div>
            )}
            
            <p className="text-sm text-muted-foreground">
              Czy na pewno chcesz zakończyć tę operację?
            </p>
          </div>
          
          <DialogFooter className="gap-2">
            <Button
              variant="outline"
              onClick={() => setCompleteDialogOpen(false)}
              data-testid="button-cancel-complete-table"
            >
              Anuluj
            </Button>
            <Button
              onClick={handleConfirmComplete}
              disabled={completeWorkOrderMutation.isPending}
              data-testid="button-confirm-complete-table"
            >
              {completeWorkOrderMutation.isPending ? (
                <Loader2 className="h-4 w-4 animate-spin mr-2" />
              ) : (
                <CheckCircle className="h-4 w-4 mr-2" />
              )}
              Zakończ operację
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </Card>
  );
}
