import { FilterState } from '@/hooks/useWarehouseTableState';

interface TriStateFilterButtonProps {
  code: string;
  label?: string;
  color?: string;
  filterState: FilterState | undefined;
  count?: number;
  onCycle: () => void;
  testId?: string;
}

function getTextColorForBackground(bgColor: string): string {
  if (!bgColor || bgColor === 'transparent') return '#ffffff';
  const hex = bgColor.replace('#', '');
  const r = parseInt(hex.substr(0, 2), 16);
  const g = parseInt(hex.substr(2, 2), 16);
  const b = parseInt(hex.substr(4, 2), 16);
  const luminance = (0.299 * r + 0.587 * g + 0.114 * b) / 255;
  return luminance > 0.5 ? '#000000' : '#ffffff';
}

export function TriStateFilterButton({
  code,
  label,
  color,
  filterState,
  count,
  onCycle,
  testId,
}: TriStateFilterButtonProps) {
  const isInclude = filterState === 'include';
  const isExclude = filterState === 'exclude';
  
  const bgColor = isInclude ? '#3b82f6' : isExclude ? '#ef4444' : (color || '#71717a');
  const textColor = (isInclude || isExclude) ? '#ffffff' : getTextColorForBackground(color || '#71717a');
  
  return (
    <button
      onClick={onCycle}
      className={`h-5 text-[10px] font-medium inline-flex items-center justify-center ${
        count === 0 ? 'opacity-30' : ''
      }`}
      style={{
        backgroundColor: bgColor,
        color: textColor,
        padding: '0 6px',
        borderRadius: '2px',
        textDecoration: isExclude ? 'line-through' : 'none',
        minWidth: '24px',
      }}
      data-testid={testId}
    >
      {label || code}
      {count !== undefined && (
        <span className="ml-1 text-[9px] opacity-70">({count})</span>
      )}
    </button>
  );
}

interface TriStateFilterRowProps {
  label: string;
  items: Array<{
    code: string;
    label?: string;
    color?: string;
    count?: number;
  }>;
  filterStates: Record<string, FilterState | undefined>;
  onCycle: (code: string) => void;
  searchValue?: string;
  onSearchChange?: (value: string) => void;
  testIdPrefix?: string;
}

export function TriStateFilterRow({
  label,
  items,
  filterStates,
  onCycle,
  searchValue = '',
  onSearchChange,
  testIdPrefix = 'filter',
}: TriStateFilterRowProps) {
  const filteredItems = items.filter(item => 
    !searchValue || 
    item.code.toLowerCase().includes(searchValue.toLowerCase()) ||
    (item.label && item.label.toLowerCase().includes(searchValue.toLowerCase()))
  );
  
  return (
    <div className="flex flex-wrap items-center gap-1">
      <span className="text-xs text-muted-foreground mr-1">{label}:</span>
      {onSearchChange && (
        <input
          type="text"
          placeholder="Szukaj..."
          value={searchValue}
          onChange={(e) => onSearchChange(e.target.value)}
          className="h-5 w-16 px-1 text-[10px] bg-zinc-800 border border-zinc-600 rounded text-zinc-300 placeholder:text-zinc-500"
          data-testid={`input-${testIdPrefix}-search`}
        />
      )}
      {filteredItems.map((item) => (
        <TriStateFilterButton
          key={item.code}
          code={item.code}
          label={item.label}
          color={item.color}
          filterState={filterStates[item.code]}
          count={item.count}
          onCycle={() => onCycle(item.code)}
          testId={`button-${testIdPrefix}-${item.code}`}
        />
      ))}
    </div>
  );
}

interface StockFilterRowProps {
  value: 'all' | 'inStock' | 'outOfStock';
  onChange: (value: 'all' | 'inStock' | 'outOfStock') => void;
  counts?: { inStock?: number; outOfStock?: number };
}

export function StockFilterRow({ value, onChange, counts }: StockFilterRowProps) {
  return (
    <div className="flex items-center gap-1">
      <span className="text-xs text-muted-foreground mr-1">Stan:</span>
      <button
        onClick={() => onChange('all')}
        className={`h-5 px-2 text-[10px] font-medium inline-flex items-center justify-center ${
          value === 'all' 
            ? 'bg-zinc-400 text-zinc-800' 
            : 'bg-zinc-700/50 text-zinc-400 hover:bg-zinc-600/50'
        }`}
        data-testid="button-stock-all"
      >
        Wszystkie
      </button>
      <button
        onClick={() => onChange('inStock')}
        className={`h-5 px-2 text-[10px] font-medium inline-flex items-center justify-center gap-1 ${
          value === 'inStock' 
            ? 'bg-zinc-400 text-zinc-800' 
            : 'bg-zinc-700/50 text-zinc-400 hover:bg-zinc-600/50'
        }`}
        data-testid="button-stock-in"
      >
        &gt;0
        {counts?.inStock !== undefined && (
          <span className="text-[9px] opacity-70">({counts.inStock})</span>
        )}
      </button>
      <button
        onClick={() => onChange('outOfStock')}
        className={`h-5 px-2 text-[10px] font-medium inline-flex items-center justify-center gap-1 ${
          value === 'outOfStock' 
            ? 'bg-zinc-400 text-zinc-800' 
            : 'bg-zinc-700/50 text-zinc-400 hover:bg-zinc-600/50'
        }`}
        data-testid="button-stock-out"
      >
        =0
        {counts?.outOfStock !== undefined && (
          <span className="text-[9px] opacity-70">({counts.outOfStock})</span>
        )}
      </button>
    </div>
  );
}
