import { Pool } from "pg";
import type { ProductionOrder } from "@shared/schema";

interface InsertProductionOrder {
  orderNumber: string;
  productId: number;
  bomId?: number | null;
  routingId?: number | null;
  status?: string;
  priority?: string;
  quantityPlanned: number;
  unitOfMeasure?: string;
  plannedStartDate?: Date | null;
  plannedEndDate?: Date | null;
  responsibleUserId?: number | null;
  sourceOrderNumber?: string | null;
  locationId?: number | null;
  notes?: string | null;
}

function mapRowToOrder(row: any): ProductionOrder {
  return {
    id: row.id,
    orderNumber: row.order_number,
    productId: row.product_id,
    bomId: row.bom_id,
    routingId: row.routing_id,
    status: row.status,
    workflowStage: row.workflow_stage,
    priority: row.priority,
    quantityPlanned: row.quantity_planned,
    quantityProduced: row.quantity_produced,
    quantityScrap: row.quantity_scrap,
    unitOfMeasure: row.unit_of_measure,
    plannedStartDate: row.planned_start_date,
    plannedEndDate: row.planned_end_date,
    actualStartDate: row.actual_start_date,
    actualEndDate: row.actual_end_date,
    workflowStageUpdatedAt: row.workflow_stage_updated_at,
    workflowStageUpdatedBy: row.workflow_stage_updated_by,
    responsibleUserId: row.responsible_user_id,
    sourceOrderNumber: row.source_order_number,
    carrierRequirements: row.carrier_requirements,
    locationId: row.location_id,
    parameters: row.parameters,
    notes: row.notes,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

interface OrderFilters {
  status?: string;
  workflowStage?: string;
  priority?: string;
  productId?: number;
  responsibleUserId?: number;
  startDate?: Date;
  endDate?: Date;
  search?: string; // order number or product name
  limit?: number;
  offset?: number;
}

export async function getOrders(pool: Pool, filters?: OrderFilters): Promise<ProductionOrder[]> {
  let query = `SELECT * FROM production.production_orders WHERE 1=1`;
  const params: any[] = [];
  let paramIndex = 1;

  if (filters?.status) {
    query += ` AND status = $${paramIndex++}`;
    params.push(filters.status);
  }

  if (filters?.workflowStage) {
    query += ` AND workflow_stage = $${paramIndex++}`;
    params.push(filters.workflowStage);
  }

  if (filters?.priority) {
    query += ` AND priority = $${paramIndex++}`;
    params.push(filters.priority);
  }

  if (filters?.productId) {
    query += ` AND product_id = $${paramIndex++}`;
    params.push(filters.productId);
  }

  if (filters?.responsibleUserId) {
    query += ` AND responsible_user_id = $${paramIndex++}`;
    params.push(filters.responsibleUserId);
  }

  if (filters?.startDate) {
    query += ` AND planned_start_date >= $${paramIndex++}`;
    params.push(filters.startDate);
  }

  if (filters?.endDate) {
    query += ` AND planned_end_date <= $${paramIndex++}`;
    params.push(filters.endDate);
  }

  if (filters?.search) {
    query += ` AND order_number ILIKE $${paramIndex++}`;
    params.push(`%${filters.search}%`);
  }

  query += ` ORDER BY created_at DESC`;

  if (filters?.limit) {
    query += ` LIMIT $${paramIndex++}`;
    params.push(filters.limit);
  }

  if (filters?.offset) {
    query += ` OFFSET $${paramIndex++}`;
    params.push(filters.offset);
  }

  const result = await pool.query(query, params);
  
  return result.rows.map(mapRowToOrder);
}

export async function getOrderById(pool: Pool, id: number): Promise<any | null> {
  const result = await pool.query(`
    SELECT * FROM production.production_orders WHERE id = $1
  `, [id]);
  
  if (result.rows.length === 0) return null;
  
  const order = mapRowToOrder(result.rows[0]);
  
  // Fetch BOM components - prioritize order's specific bom_id, fallback to product_id
  if (order.bomId || order.productId) {
    let componentsResult;
    
    if (order.bomId) {
      // Use specific BOM ID from order
      componentsResult = await pool.query(`
        SELECT 
          pc.id,
          pc.generated_name,
          pc.component_type,
          pc.calculated_length,
          pc.calculated_width,
          pc.thickness,
          pc.color,
          pc.board_type,
          pc.drilling_pattern,
          pc.edging_pattern,
          pc.edging_material,
          pc.quantity,
          pc.unit_of_measure,
          pc.visualization_url,
          pc.is_damaged,
          pc.damage_notes,
          pc.damaged_at,
          pc.damaged_by,
          pc.production_notes
        FROM bom.product_components pc
        WHERE pc.product_bom_id = $1
        ORDER BY pc.position_in_bom, pc.id
      `, [order.bomId]);
    } else {
      // Fallback: find BOM that existed at order creation time
      // This ensures we show components from historical BOM, even if it's now inactive
      const bomResult = await pool.query(`
        SELECT id FROM bom.product_boms 
        WHERE product_id = $1 
          AND created_at <= $2
        ORDER BY created_at DESC 
        LIMIT 1
      `, [order.productId, order.createdAt]);
      
      if (bomResult.rows.length === 0) {
        // No BOM found for this product at order creation time
        return { ...order, components: [] };
      }
      
      componentsResult = await pool.query(`
        SELECT 
          pc.id,
          pc.generated_name,
          pc.component_type,
          pc.calculated_length,
          pc.calculated_width,
          pc.thickness,
          pc.color,
          pc.board_type,
          pc.drilling_pattern,
          pc.edging_pattern,
          pc.edging_material,
          pc.quantity,
          pc.unit_of_measure,
          pc.visualization_url,
          pc.is_damaged,
          pc.damage_notes,
          pc.damaged_at,
          pc.damaged_by,
          pc.production_notes
        FROM bom.product_components pc
        WHERE pc.product_bom_id = $1
        ORDER BY pc.position_in_bom, pc.id
      `, [bomResult.rows[0].id]);
    }
    
    return {
      ...order,
      components: componentsResult.rows.map((row: any) => ({
        id: row.id,
        generatedName: row.generated_name,
        componentType: row.component_type,
        calculatedLength: row.calculated_length,
        calculatedWidth: row.calculated_width,
        thickness: row.thickness,
        color: row.color,
        boardType: row.board_type,
        drillingPattern: row.drilling_pattern,
        edgingPattern: row.edging_pattern,
        edgingMaterial: row.edging_material,
        quantity: row.quantity,
        unitOfMeasure: row.unit_of_measure,
        visualizationUrl: row.visualization_url,
        isDamaged: row.is_damaged,
        damageNotes: row.damage_notes,
        damagedAt: row.damaged_at,
        damagedBy: row.damaged_by,
        productionNotes: row.production_notes,
      })),
    };
  }
  
  return { ...order, components: [] };
}

export async function createOrder(pool: Pool, data: InsertProductionOrder): Promise<ProductionOrder> {
  const result = await pool.query(`
    INSERT INTO production.production_orders 
    (order_number, product_id, bom_id, routing_id, status, priority, 
     quantity_planned, unit_of_measure, planned_start_date, planned_end_date,
     responsible_user_id, source_order_number, location_id, notes)
    VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14)
    RETURNING *
  `, [
    data.orderNumber,
    data.productId,
    data.bomId ?? null,
    data.routingId ?? null,
    data.status ?? 'draft',
    data.priority ?? 'normal',
    data.quantityPlanned,
    data.unitOfMeasure ?? 'szt',
    data.plannedStartDate ?? null,
    data.plannedEndDate ?? null,
    data.responsibleUserId ?? null,
    data.sourceOrderNumber ?? null,
    data.locationId ?? null,
    data.notes ?? null,
  ]);
  
  return mapRowToOrder(result.rows[0]);
}

export async function updateOrder(pool: Pool, id: number, data: Partial<InsertProductionOrder>): Promise<ProductionOrder | null> {
  const updates: string[] = [];
  const values: any[] = [];
  let paramIndex = 1;

  if (data.orderNumber !== undefined) {
    updates.push(`order_number = $${paramIndex++}`);
    values.push(data.orderNumber);
  }
  if (data.productId !== undefined) {
    updates.push(`product_id = $${paramIndex++}`);
    values.push(data.productId);
  }
  if (data.bomId !== undefined) {
    updates.push(`bom_id = $${paramIndex++}`);
    values.push(data.bomId);
  }
  if (data.routingId !== undefined) {
    updates.push(`routing_id = $${paramIndex++}`);
    values.push(data.routingId);
  }
  if (data.status !== undefined) {
    updates.push(`status = $${paramIndex++}`);
    values.push(data.status);
  }
  if (data.priority !== undefined) {
    updates.push(`priority = $${paramIndex++}`);
    values.push(data.priority);
  }
  if (data.quantityPlanned !== undefined) {
    updates.push(`quantity_planned = $${paramIndex++}`);
    values.push(data.quantityPlanned);
  }
  if (data.unitOfMeasure !== undefined) {
    updates.push(`unit_of_measure = $${paramIndex++}`);
    values.push(data.unitOfMeasure);
  }
  if (data.plannedStartDate !== undefined) {
    updates.push(`planned_start_date = $${paramIndex++}`);
    values.push(data.plannedStartDate);
  }
  if (data.plannedEndDate !== undefined) {
    updates.push(`planned_end_date = $${paramIndex++}`);
    values.push(data.plannedEndDate);
  }
  if (data.responsibleUserId !== undefined) {
    updates.push(`responsible_user_id = $${paramIndex++}`);
    values.push(data.responsibleUserId);
  }
  if (data.sourceOrderNumber !== undefined) {
    updates.push(`source_order_number = $${paramIndex++}`);
    values.push(data.sourceOrderNumber);
  }
  if (data.locationId !== undefined) {
    updates.push(`location_id = $${paramIndex++}`);
    values.push(data.locationId);
  }
  if (data.notes !== undefined) {
    updates.push(`notes = $${paramIndex++}`);
    values.push(data.notes);
  }

  if (updates.length === 0) return null;

  updates.push(`updated_at = CURRENT_TIMESTAMP`);
  values.push(id);

  const result = await pool.query(`
    UPDATE production.production_orders 
    SET ${updates.join(', ')}
    WHERE id = $${paramIndex}
    RETURNING *
  `, values);

  if (result.rows.length === 0) return null;
  
  return mapRowToOrder(result.rows[0]);
}

export async function deleteOrder(pool: Pool, id: number): Promise<boolean> {
  const result = await pool.query(`
    DELETE FROM production.production_orders WHERE id = $1
  `, [id]);
  
  return result.rowCount ? result.rowCount > 0 : false;
}

// Status transition methods
export async function startOrder(pool: Pool, id: number): Promise<ProductionOrder | null> {
  const result = await pool.query(`
    UPDATE production.production_orders 
    SET status = 'in_progress',
        actual_start_date = CURRENT_TIMESTAMP,
        updated_at = CURRENT_TIMESTAMP
    WHERE id = $1 AND status IN ('draft', 'confirmed', 'planned')
    RETURNING *
  `, [id]);

  if (result.rows.length === 0) return null;
  
  return mapRowToOrder(result.rows[0]);
}

export async function completeOrder(pool: Pool, id: number): Promise<ProductionOrder | null> {
  const result = await pool.query(`
    UPDATE production.production_orders 
    SET status = 'done',
        actual_end_date = CURRENT_TIMESTAMP,
        updated_at = CURRENT_TIMESTAMP
    WHERE id = $1 AND status = 'in_progress'
    RETURNING *
  `, [id]);

  if (result.rows.length === 0) return null;
  
  return mapRowToOrder(result.rows[0]);
}

export async function cancelOrder(pool: Pool, id: number): Promise<ProductionOrder | null> {
  const result = await pool.query(`
    UPDATE production.production_orders 
    SET status = 'cancelled',
        updated_at = CURRENT_TIMESTAMP
    WHERE id = $1 AND status NOT IN ('done', 'cancelled')
    RETURNING *
  `, [id]);

  if (result.rows.length === 0) return null;
  
  return mapRowToOrder(result.rows[0]);
}

// Workflow stage transitions
const WORKFLOW_STAGES = [
  'magazine',      // Material in warehouse
  'cutting',       // Cutting operation
  'edging',        // Edging operation
  'drilling',      // Drilling operation
  'upholstering',  // Upholstering operation
  'picking',       // Picking components
  'packing',       // Packing finished product
  'strapping',     // Strapping for shipping
  'ready',         // Ready for dispatch
  'shipped'        // Shipped to customer
] as const;

export async function updateWorkflowStage(
  pool: Pool,
  orderId: number,
  stage: string,
  userId?: number
): Promise<ProductionOrder | null> {
  // Validate stage
  if (!WORKFLOW_STAGES.includes(stage as any)) {
    throw new Error(`Invalid workflow stage: ${stage}`);
  }

  // Update workflow_stage with audit trail
  const result = await pool.query(`
    UPDATE production.production_orders 
    SET workflow_stage = $1,
        workflow_stage_updated_at = CURRENT_TIMESTAMP,
        workflow_stage_updated_by = $2,
        updated_at = CURRENT_TIMESTAMP
    WHERE id = $3
    RETURNING *
  `, [stage, userId || null, orderId]);

  if (result.rows.length === 0) return null;
  
  return mapRowToOrder(result.rows[0]);
}

// Auto-generate order number
export async function generateOrderNumber(pool: Pool): Promise<string> {
  const year = new Date().getFullYear();
  const result = await pool.query(`
    SELECT order_number FROM production.production_orders 
    WHERE order_number LIKE 'ZLP-${year}-%'
    ORDER BY order_number DESC
    LIMIT 1
  `);

  if (result.rows.length === 0) {
    return `ZLP-${year}-001`;
  }

  const lastNumber = result.rows[0].order_number;
  const match = lastNumber.match(/ZLP-\d{4}-(\d+)$/);
  
  if (match) {
    const nextNum = (parseInt(match[1]) + 1).toString().padStart(3, '0');
    return `ZLP-${year}-${nextNum}`;
  }

  return `ZLP-${year}-001`;
}
