import fs from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';
import { pool } from './postgres';
import Papa from 'papaparse';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

interface CSVRow {
  nazwa: string;
  Attachments: string;
  rozmiarPlyty: string;
  nazwGen: string;
  autonumer: string;
  'ROZKROJE FORMATKI': string;
  'Formatki Agregowane do ZKO': string;
  ZKO: string;
  'ZKO (Wybrany rozkrój)': string;
}

function parseDimensions(sizeStr: string): { length: number | null; width: number | null } {
  // Format: "462x279" lub "500x300" etc
  const match = sizeStr.match(/(\d+)\s*x\s*(\d+)/i);
  if (match) {
    return {
      width: parseInt(match[1]),  // Pierwsza liczba to szerokość
      length: parseInt(match[2])  // Druga liczba to długość
    };
  }
  return { length: null, width: null };
}

function parseBoardDimensions(boardStr: string): { length: number; width: number } {
  // Format: "2800x2070" lub "2800x2100"
  const match = boardStr.match(/(\d+)\s*x\s*(\d+)/i);
  if (match) {
    return {
      length: parseInt(match[1]),
      width: parseInt(match[2])
    };
  }
  // Default
  return { length: 2800, width: 2070 };
}

function extractImageUrl(attachmentsStr: string): string | null {
  // Extract URL from format: "rozk-xxx.png (https://....)"
  const match = attachmentsStr.match(/\((https?:\/\/[^)]+)\)/);
  return match ? match[1] : null;
}

async function importRozkrojeAgregowane() {
  console.log('🚀 Starting agregowane rozkroje import...');
  
  // Read CSV file with proper parsing
  const csvPath = path.join(__dirname, '..', 'attached_assets', 'ROZKROJE-Grid view (3)_1762945071998.csv');
  const csvContent = fs.readFileSync(csvPath, 'utf-8');
  
  // Use Papa Parse for better CSV handling
  const parseResult = Papa.parse<CSVRow>(csvContent, {
    header: true,
    skipEmptyLines: true,
    transformHeader: (header: string) => header.trim().replace(/^\uFEFF/, ''),
  });
  
  const rows = parseResult.data.filter(row => row.nazwGen && row['ROZKROJE FORMATKI']);
  console.log(`📊 Parsed ${rows.length} valid rozkroje from CSV`);
  
  let patternsUpdated = 0;
  let patternsCreated = 0;
  
  for (const row of rows) {
    const rozkrojCode = row.nazwGen;
    const formatkiStr = row['ROZKROJE FORMATKI'];
    const imageUrl = extractImageUrl(row.Attachments);
    
    // Parse board dimensions
    let boardLength = 2800;
    let boardWidth = 2070;
    
    if (row.rozmiarPlyty) {
      const dimensions = parseBoardDimensions(row.rozmiarPlyty);
      boardLength = dimensions.length;
      boardWidth = dimensions.width;
    }
    
    // Check if pattern already exists
    const existingPattern = await pool.query(
      `SELECT id FROM production.cut_patterns WHERE code = $1`,
      [rozkrojCode]
    );
    
    let patternId: number;
    
    if (existingPattern.rows.length > 0) {
      // Update existing pattern with image_url
      patternId = existingPattern.rows[0].id;
      await pool.query(
        `UPDATE production.cut_patterns 
         SET image_url = $1, 
             board_length = $2, 
             board_width = $3,
             updated_at = NOW()
         WHERE id = $4`,
        [imageUrl, boardLength, boardWidth, patternId]
      );
      patternsUpdated++;
      console.log(`✓ Updated pattern ${rozkrojCode} with image`);
    } else {
      // Create new pattern
      const result = await pool.query(
        `INSERT INTO production.cut_patterns (
          code, board_length, board_width, board_thickness, kerf, 
          image_url, status, notes, is_active
        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9)
        RETURNING id`,
        [
          rozkrojCode,
          boardLength,
          boardWidth,
          18,
          4,
          imageUrl,
          'planowany',
          `Agregowane formatki: ${formatkiStr}`,
          true
        ]
      );
      patternId = result.rows[0].id;
      patternsCreated++;
      console.log(`✓ Created pattern ${rozkrojCode} (ID: ${patternId})`);
    }
    
    // Parse formatki list and create/update items if they don't exist from previous import
    const formatkiList = formatkiStr.split(',').map(f => f.trim()).filter(f => f);
    
    for (const formatkaSize of formatkiList) {
      const { width, length } = parseDimensions(formatkaSize);
      
      if (!width || !length) {
        console.log(`  ⚠️  Skipping formatka "${formatkaSize}" - invalid format`);
        continue;
      }
      
      // Check if item with these dimensions already exists for this pattern
      const existingItem = await pool.query(
        `SELECT id FROM production.cut_pattern_items 
         WHERE pattern_id = $1 AND length = $2 AND width = $3`,
        [patternId, length, width]
      );
      
      if (existingItem.rows.length === 0) {
        // Create new item
        await pool.query(
          `INSERT INTO production.cut_pattern_items (
            pattern_id, item_name, length, width, thickness, 
            quantity_requested, status, is_active
          ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8)`,
          [
            patternId,
            formatkaSize,
            length,
            width,
            18,
            1, // Default quantity
            'zaplanowana',
            true
          ]
        );
        console.log(`    + Added formatka ${formatkaSize} (${width}x${length})`);
      }
    }
  }
  
  console.log(`\n✨ Import completed successfully!`);
  console.log(`📦 Created ${patternsCreated} new cut patterns`);
  console.log(`🔄 Updated ${patternsUpdated} existing cut patterns with images`);
}

// Run import
importRozkrojeAgregowane()
  .then(() => {
    console.log('✅ Import finished');
    process.exit(0);
  })
  .catch((error) => {
    console.error('❌ Import failed:', error);
    process.exit(1);
  });
