import { useEffect, useState } from "react";
import { useRoute, useLocation } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Switch } from "@/components/ui/switch";
import {
  Form,
  FormControl,
  FormDescription,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { RichTextEditor } from "@/components/rich-text-editor";
import { AvailableTagsPanel } from "@/components/template/available-tags-panel";
import { ExamplesModal } from "@/components/template/examples-modal";
import { ArrowLeft, Loader2, Eye } from "lucide-react";

interface TemplateCategory {
  id: number;
  name: string;
  description: string | null;
}

interface DescriptionTemplate {
  id: number;
  name: string;
  categoryId: number | null;
  templateType: string;
  contentDoc?: any;
  htmlContent: string;
  variables: any;
  isGlobal: boolean;
  isActive: boolean;
}

const templateFormSchema = z.object({
  name: z.string().min(1, "Nazwa szablonu jest wymagana"),
  categoryId: z.number().nullable(),
  templateType: z.string().min(1, "Typ szablonu jest wymagany"),
  htmlContent: z.string().min(1, "Treść szablonu jest wymagana"),
  variables: z.string().optional(),
  isGlobal: z.boolean().default(false),
  isActive: z.boolean().default(true),
});

type TemplateFormData = z.infer<typeof templateFormSchema>;

export default function TemplateEditor() {
  const [, navigate] = useLocation();
  const [, params] = useRoute("/template-editor/:id");
  const { toast } = useToast();
  const [contentDocJson, setContentDocJson] = useState<any>({});
  const [editorInstance, setEditorInstance] = useState<any>(null);
  const [isExamplesModalOpen, setIsExamplesModalOpen] = useState(false);
  
  const isNewTemplate = params?.id === "new";
  const templateId = isNewTemplate ? null : Number(params?.id);

  // Fetch categories
  const { data: categories } = useQuery<TemplateCategory[]>({
    queryKey: ["/api/template-categories"],
  });

  // Fetch template if editing
  const { data: template, isLoading: templateLoading } = useQuery<DescriptionTemplate>({
    queryKey: ["/api/description-templates", templateId],
    queryFn: async () => {
      const res = await fetch(`/api/description-templates/${templateId}`, {
        credentials: "include",
      });
      if (!res.ok) {
        const contentType = res.headers.get("content-type");
        let errorMessage = `Failed to fetch template: ${res.statusText}`;
        
        if (contentType?.includes("application/json")) {
          try {
            const errorData = await res.json();
            errorMessage = errorData.error || errorData.message || errorMessage;
          } catch (e) {
            // Ignore JSON parse errors and use default message
          }
        }
        
        throw new Error(errorMessage);
      }
      return res.json();
    },
    enabled: !isNewTemplate && templateId !== null,
  });

  const form = useForm<TemplateFormData>({
    resolver: zodResolver(templateFormSchema),
    defaultValues: {
      name: "",
      categoryId: null,
      templateType: "product_description",
      htmlContent: "",
      variables: "{}",
      isGlobal: false,
      isActive: true,
    },
  });

  // Update form when template loads
  useEffect(() => {
    if (template && !isNewTemplate) {
      form.reset({
        name: template.name,
        categoryId: template.categoryId,
        templateType: template.templateType,
        htmlContent: template.htmlContent,
        variables: typeof template.variables === "string" 
          ? template.variables 
          : JSON.stringify(template.variables || {}, null, 2),
        isGlobal: template.isGlobal,
        isActive: template.isActive,
      });
      
      // Set contentDoc if available for the editor
      if (template.contentDoc) {
        setContentDocJson(template.contentDoc);
      }
    }
  }, [template, isNewTemplate, form]);

  const createMutation = useMutation({
    mutationFn: async (data: TemplateFormData) => {
      const response = await apiRequest("POST", "/api/description-templates", {
        ...data,
        contentDoc: contentDocJson,
        variables: data.variables ? JSON.parse(data.variables) : {},
      });
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/description-templates"] });
      toast({
        title: "Sukces",
        description: "Szablon został utworzony",
      });
      navigate("/templates-manager");
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się utworzyć szablonu",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: async (data: TemplateFormData) => {
      const response = await apiRequest("PUT", `/api/description-templates/${templateId}`, {
        ...data,
        contentDoc: contentDocJson,
        variables: data.variables ? JSON.parse(data.variables) : {},
      });
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/description-templates"] });
      queryClient.invalidateQueries({ queryKey: ["/api/description-templates", templateId] });
      toast({
        title: "Sukces",
        description: "Szablon został zaktualizowany",
      });
      navigate("/templates-manager");
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zaktualizować szablonu",
        variant: "destructive",
      });
    },
  });

  const onSubmit = (data: TemplateFormData) => {
    if (isNewTemplate) {
      createMutation.mutate(data);
    } else {
      updateMutation.mutate(data);
    }
  };

  if (templateLoading && !isNewTemplate) {
    return (
      <div className="flex items-center justify-center h-full">
        <Loader2 className="h-8 w-8 animate-spin" />
      </div>
    );
  }

  return (
    <div className="h-full overflow-hidden flex flex-col">
      {/* Header */}
      <div className="flex-shrink-0 border-b bg-background p-4">
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-4">
            <Button
              variant="ghost"
              size="icon"
              onClick={() => window.history.back()}
              data-testid="button-back"
            >
              <ArrowLeft className="h-5 w-5" />
            </Button>
            <div>
              <h1 className="text-2xl font-bold">
                {isNewTemplate ? "Nowy szablon" : "Edytuj szablon"}
              </h1>
              <p className="text-sm text-muted-foreground">
                {isNewTemplate 
                  ? "Utwórz nowy szablon opisu produktu. Użyj tagów z panelu po prawej." 
                  : "Zaktualizuj szablon opisu produktu"}
              </p>
            </div>
          </div>
          <div className="flex gap-2">
            <Button
              variant="outline"
              onClick={() => navigate("/templates-manager")}
              data-testid="button-cancel"
            >
              Anuluj
            </Button>
            <Button
              onClick={form.handleSubmit(onSubmit)}
              disabled={createMutation.isPending || updateMutation.isPending}
              data-testid="button-save"
            >
              {createMutation.isPending || updateMutation.isPending ? (
                <>
                  <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                  Zapisywanie...
                </>
              ) : (
                "Zapisz szablon"
              )}
            </Button>
          </div>
        </div>
      </div>

      {/* Content */}
      <div className="flex flex-1 min-h-0">
        {/* Left: Form */}
        <div className="flex-1 overflow-y-auto p-6">
          <Form {...form}>
            <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6 max-w-4xl">
              <FormField
                control={form.control}
                name="name"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Nazwa szablonu</FormLabel>
                    <FormControl>
                      <Input {...field} placeholder="np. Opis Podstawowy" data-testid="input-template-name" />
                    </FormControl>
                    <FormDescription>
                      Unikalna nazwa szablonu do identyfikacji
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="categoryId"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Kategoria</FormLabel>
                      <Select
                        value={field.value?.toString() || ""}
                        onValueChange={(value) => field.onChange(value ? Number(value) : null)}
                      >
                        <FormControl>
                          <SelectTrigger data-testid="select-category">
                            <SelectValue placeholder="Wybierz kategorię" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          {categories?.map((category) => (
                            <SelectItem key={category.id} value={category.id.toString()}>
                              {category.name}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <FormDescription className="text-xs">
                        Opcjonalna kategoria szablonu
                      </FormDescription>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="templateType"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Typ szablonu</FormLabel>
                      <Select value={field.value} onValueChange={field.onChange}>
                        <FormControl>
                          <SelectTrigger data-testid="select-type">
                            <SelectValue placeholder="Wybierz typ" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          <SelectItem value="product_description">Opis produktu</SelectItem>
                          <SelectItem value="email">Email</SelectItem>
                          <SelectItem value="announcement">Ogłoszenie</SelectItem>
                        </SelectContent>
                      </Select>
                      <FormDescription className="text-xs">
                        np. product_description, email, announcement
                      </FormDescription>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={form.control}
                name="htmlContent"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Treść szablonu</FormLabel>
                    <FormControl>
                      <RichTextEditor
                        content={template?.contentDoc || field.value}
                        onContentChange={(html, json) => {
                          field.onChange(html);
                          setContentDocJson(json);
                        }}
                        placeholder="Zacznij pisać opis produktu..."
                        enableGalleryPicker={true}
                        enableAddonPicker={true}
                        onEditorReady={(editor) => setEditorInstance(editor)}
                      />
                    </FormControl>
                    <FormDescription className="text-xs">
                      Użyj edytora WYSIWYG do formatowania tekstu. Zmienne w formacie {`{{zmienna}}`} będą renderowane dynamicznie.
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="variables"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Zmienne (JSON)</FormLabel>
                    <FormControl>
                      <Textarea 
                        {...field} 
                        placeholder='{"nazwa_produktu": "string", "cena": "number"}'
                        rows={4}
                        data-testid="input-template-variables"
                      />
                    </FormControl>
                    <FormDescription className="text-xs">
                      Definicje zmiennych w formacie JSON
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="isGlobal"
                  render={({ field }) => (
                    <FormItem className="flex items-center justify-between rounded-lg border p-3">
                      <div className="space-y-0.5">
                        <FormLabel>Szablon globalny</FormLabel>
                        <FormDescription className="text-xs">
                          Dostępny dla wszystkich użytkowników
                        </FormDescription>
                      </div>
                      <FormControl>
                        <Switch
                          checked={field.value}
                          onCheckedChange={field.onChange}
                          data-testid="switch-template-global"
                        />
                      </FormControl>
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="isActive"
                  render={({ field }) => (
                    <FormItem className="flex items-center justify-between rounded-lg border p-3">
                      <div className="space-y-0.5">
                        <FormLabel>Aktywny</FormLabel>
                        <FormDescription className="text-xs">
                          Szablon jest aktywny i widoczny
                        </FormDescription>
                      </div>
                      <FormControl>
                        <Switch
                          checked={field.value}
                          onCheckedChange={field.onChange}
                          data-testid="switch-template-active"
                        />
                      </FormControl>
                    </FormItem>
                  )}
                />
              </div>
            </form>
          </Form>
        </div>

        {/* Right: Available Tags Panel */}
        <div className="w-96 flex-shrink-0 border-l bg-muted/30 overflow-y-auto p-4">
          <AvailableTagsPanel
            onTagInsert={(tag) => {
              if (editorInstance) {
                editorInstance.commands.insertContent(tag);
                editorInstance.commands.focus();
              }
            }}
            onShowExamples={() => setIsExamplesModalOpen(true)}
          />
        </div>
      </div>

      {/* Examples Modal */}
      <ExamplesModal
        open={isExamplesModalOpen}
        onOpenChange={setIsExamplesModalOpen}
      />
    </div>
  );
}
