import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { useToast } from "@/hooks/use-toast";
import { Plus, Trash2, Edit2, Settings as SettingsIcon } from "lucide-react";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Badge } from "@/components/ui/badge";
import { Switch } from "@/components/ui/switch";
import type { ProductionSetting, ProductionNormGroup, ProductionNorm, ProductionMaterialConsumptionNorm } from "@shared/schema";

export default function ProductionSettings() {
  const { toast } = useToast();
  
  // Settings state
  const [isAddSettingDialogOpen, setIsAddSettingDialogOpen] = useState(false);
  const [isEditSettingDialogOpen, setIsEditSettingDialogOpen] = useState(false);
  const [editingSetting, setEditingSetting] = useState<ProductionSetting | null>(null);
  const [settingFormData, setSettingFormData] = useState<{
    key: string;
    value: string;
    valueType: "string" | "number" | "boolean" | "json";
    description: string;
    category: string;
    isActive: boolean;
  }>({
    key: "",
    value: "",
    valueType: "string",
    description: "",
    category: "general",
    isActive: true,
  });

  // Norm Groups state
  const [isAddNormGroupDialogOpen, setIsAddNormGroupDialogOpen] = useState(false);
  const [isEditNormGroupDialogOpen, setIsEditNormGroupDialogOpen] = useState(false);
  const [editingNormGroup, setEditingNormGroup] = useState<ProductionNormGroup | null>(null);
  const [normGroupFormData, setNormGroupFormData] = useState({
    code: "",
    name: "",
    description: "",
    unitOfMeasure: "",
    isActive: true,
    sortOrder: 0,
  });

  // Norms state
  const [isAddNormDialogOpen, setIsAddNormDialogOpen] = useState(false);
  const [isEditNormDialogOpen, setIsEditNormDialogOpen] = useState(false);
  const [editingNorm, setEditingNorm] = useState<ProductionNorm | null>(null);
  const [normFormData, setNormFormData] = useState({
    normGroupId: 0,
    code: "",
    name: "",
    description: "",
    defaultValue: "",
    minValue: "",
    maxValue: "",
    tolerance: "",
    unitOfMeasure: "",
    applicableTo: [] as string[],
    isActive: true,
  });

  // Material Consumption Norms state
  const [isAddMaterialNormDialogOpen, setIsAddMaterialNormDialogOpen] = useState(false);
  const [isEditMaterialNormDialogOpen, setIsEditMaterialNormDialogOpen] = useState(false);
  const [editingMaterialNorm, setEditingMaterialNorm] = useState<ProductionMaterialConsumptionNorm | null>(null);
  const [materialTypeFilter, setMaterialTypeFilter] = useState<string>("all");
  const [materialNormFormData, setMaterialNormFormData] = useState({
    materialType: "board",
    materialCategory: "",
    wastePercentage: "",
    edgeOverageCm: "",
    consumptionPerUnit: "",
    inputUnit: "m2",
    outputUnit: "m2",
    isActive: true,
    description: "",
    notes: "",
  });

  // Queries
  const { data: settings = [], isLoading: settingsLoading } = useQuery<ProductionSetting[]>({
    queryKey: ["/api/production/settings"],
  });

  const { data: normGroups = [], isLoading: normGroupsLoading } = useQuery<ProductionNormGroup[]>({
    queryKey: ["/api/production/norms/groups"],
  });

  const { data: norms = [], isLoading: normsLoading } = useQuery<ProductionNorm[]>({
    queryKey: ["/api/production/norms"],
  });

  const { data: materialNorms = [], isLoading: materialNormsLoading } = useQuery<ProductionMaterialConsumptionNorm[]>({
    queryKey: ["/api/production/norms/materials"],
  });

  // === Settings Mutations ===
  const addSettingMutation = useMutation({
    mutationFn: async () => {
      return apiRequest("POST", "/api/production/settings", settingFormData);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/settings"] });
      toast({ title: "Ustawienie dodane", description: "Nowe ustawienie zostało pomyślnie dodane." });
      setIsAddSettingDialogOpen(false);
      resetSettingForm();
    },
    onError: (error: any) => {
      toast({
        title: "Błąd dodawania",
        description: error.message || "Nie udało się dodać ustawienia.",
        variant: "destructive",
      });
    },
  });

  const updateSettingMutation = useMutation({
    mutationFn: async () => {
      if (!editingSetting) return;
      return apiRequest("PATCH", `/api/production/settings/${editingSetting.id}`, settingFormData);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/settings"] });
      toast({ title: "Ustawienie zaktualizowane", description: "Zmiany zostały zapisane." });
      setIsEditSettingDialogOpen(false);
      setEditingSetting(null);
      resetSettingForm();
    },
    onError: (error: any) => {
      toast({
        title: "Błąd aktualizacji",
        description: error.message || "Nie udało się zaktualizować ustawienia.",
        variant: "destructive",
      });
    },
  });

  const deleteSettingMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("DELETE", `/api/production/settings/${id}`, {});
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/settings"] });
      toast({ title: "Ustawienie usunięte", description: "Ustawienie zostało pomyślnie usunięte." });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd usuwania",
        description: error.message || "Nie udało się usunąć ustawienia.",
        variant: "destructive",
      });
    },
  });

  // === Norm Groups Mutations ===
  const addNormGroupMutation = useMutation({
    mutationFn: async () => {
      return apiRequest("POST", "/api/production/norms/groups", normGroupFormData);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/norms/groups"] });
      toast({ title: "Grupa normatywów dodana", description: "Nowa grupa została pomyślnie dodana." });
      setIsAddNormGroupDialogOpen(false);
      resetNormGroupForm();
    },
    onError: (error: any) => {
      toast({
        title: "Błąd dodawania",
        description: error.message || "Nie udało się dodać grupy normatywów.",
        variant: "destructive",
      });
    },
  });

  const updateNormGroupMutation = useMutation({
    mutationFn: async () => {
      if (!editingNormGroup) return;
      return apiRequest("PATCH", `/api/production/norms/groups/${editingNormGroup.id}`, normGroupFormData);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/norms/groups"] });
      toast({ title: "Grupa zaktualizowana", description: "Zmiany zostały zapisane." });
      setIsEditNormGroupDialogOpen(false);
      setEditingNormGroup(null);
      resetNormGroupForm();
    },
    onError: (error: any) => {
      toast({
        title: "Błąd aktualizacji",
        description: error.message || "Nie udało się zaktualizować grupy.",
        variant: "destructive",
      });
    },
  });

  const deleteNormGroupMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("DELETE", `/api/production/norms/groups/${id}`, {});
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/norms/groups"] });
      toast({ title: "Grupa usunięta", description: "Grupa normatywów została pomyślnie usunięta." });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd usuwania",
        description: error.message || "Nie udało się usunąć grupy.",
        variant: "destructive",
      });
    },
  });

  // === Norms Mutations ===
  const addNormMutation = useMutation({
    mutationFn: async () => {
      return apiRequest("POST", "/api/production/norms", {
        ...normFormData,
        defaultValue: parseFloat(normFormData.defaultValue) || 0,
        minValue: normFormData.minValue ? parseFloat(normFormData.minValue) : null,
        maxValue: normFormData.maxValue ? parseFloat(normFormData.maxValue) : null,
        tolerance: normFormData.tolerance ? parseFloat(normFormData.tolerance) : null,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/norms"] });
      toast({ title: "Normatyw dodany", description: "Nowy normatyw został pomyślnie dodany." });
      setIsAddNormDialogOpen(false);
      resetNormForm();
    },
    onError: (error: any) => {
      toast({
        title: "Błąd dodawania",
        description: error.message || "Nie udało się dodać normatywu.",
        variant: "destructive",
      });
    },
  });

  const updateNormMutation = useMutation({
    mutationFn: async () => {
      if (!editingNorm) return;
      return apiRequest("PATCH", `/api/production/norms/${editingNorm.id}`, {
        ...normFormData,
        defaultValue: normFormData.defaultValue ? parseFloat(normFormData.defaultValue) : undefined,
        minValue: normFormData.minValue ? parseFloat(normFormData.minValue) : undefined,
        maxValue: normFormData.maxValue ? parseFloat(normFormData.maxValue) : undefined,
        tolerance: normFormData.tolerance ? parseFloat(normFormData.tolerance) : undefined,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/norms"] });
      toast({ title: "Normatyw zaktualizowany", description: "Zmiany zostały zapisane." });
      setIsEditNormDialogOpen(false);
      setEditingNorm(null);
      resetNormForm();
    },
    onError: (error: any) => {
      toast({
        title: "Błąd aktualizacji",
        description: error.message || "Nie udało się zaktualizować normatywu.",
        variant: "destructive",
      });
    },
  });

  const deleteNormMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("DELETE", `/api/production/norms/${id}`, {});
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/norms"] });
      toast({ title: "Normatyw usunięty", description: "Normatyw został pomyślnie usunięty." });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd usuwania",
        description: error.message || "Nie udało się usunąć normatywu.",
        variant: "destructive",
      });
    },
  });

  // === Material Consumption Norms Mutations ===
  const addMaterialNormMutation = useMutation({
    mutationFn: async () => {
      // Normalize payload: send only fields relevant to materialType
      const payload: any = {
        materialType: materialNormFormData.materialType,
        materialCategory: materialNormFormData.materialCategory,
        inputUnit: materialNormFormData.inputUnit,
        outputUnit: materialNormFormData.outputUnit,
        isActive: materialNormFormData.isActive,
        description: materialNormFormData.description,
        notes: materialNormFormData.notes,
      };

      // Type-specific fields: send as strings for decimal columns
      if (materialNormFormData.materialType === "board") {
        payload.wastePercentage = materialNormFormData.wastePercentage || null;
        payload.edgeOverageCm = null;
        payload.consumptionPerUnit = null;
      } else if (materialNormFormData.materialType === "edging") {
        payload.wastePercentage = null;
        payload.edgeOverageCm = materialNormFormData.edgeOverageCm || null;
        payload.consumptionPerUnit = null;
      } else if (materialNormFormData.materialType === "adhesive" || materialNormFormData.materialType === "screw") {
        payload.wastePercentage = null;
        payload.edgeOverageCm = null;
        payload.consumptionPerUnit = materialNormFormData.consumptionPerUnit || null;
      }

      return apiRequest("POST", "/api/production/norms/materials", payload);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/norms/materials"] });
      toast({ title: "Norma materiałowa dodana", description: "Nowa norma materiałowa została pomyślnie dodana." });
      setIsAddMaterialNormDialogOpen(false);
      resetMaterialNormForm();
    },
    onError: (error: any) => {
      toast({
        title: "Błąd dodawania",
        description: error.message || "Nie udało się dodać normy materiałowej.",
        variant: "destructive",
      });
    },
  });

  const updateMaterialNormMutation = useMutation({
    mutationFn: async () => {
      if (!editingMaterialNorm) return;
      
      // Normalize payload: send only fields relevant to materialType
      const payload: any = {
        materialType: materialNormFormData.materialType,
        materialCategory: materialNormFormData.materialCategory,
        inputUnit: materialNormFormData.inputUnit,
        outputUnit: materialNormFormData.outputUnit,
        isActive: materialNormFormData.isActive,
        description: materialNormFormData.description,
        notes: materialNormFormData.notes,
      };

      // Type-specific fields: send as strings for decimal columns
      if (materialNormFormData.materialType === "board") {
        payload.wastePercentage = materialNormFormData.wastePercentage || null;
        payload.edgeOverageCm = null;
        payload.consumptionPerUnit = null;
      } else if (materialNormFormData.materialType === "edging") {
        payload.wastePercentage = null;
        payload.edgeOverageCm = materialNormFormData.edgeOverageCm || null;
        payload.consumptionPerUnit = null;
      } else if (materialNormFormData.materialType === "adhesive" || materialNormFormData.materialType === "screw") {
        payload.wastePercentage = null;
        payload.edgeOverageCm = null;
        payload.consumptionPerUnit = materialNormFormData.consumptionPerUnit || null;
      }

      return apiRequest("PATCH", `/api/production/norms/materials/${editingMaterialNorm.id}`, payload);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/norms/materials"] });
      toast({ title: "Norma materiałowa zaktualizowana", description: "Zmiany zostały zapisane." });
      setIsEditMaterialNormDialogOpen(false);
      setEditingMaterialNorm(null);
      resetMaterialNormForm();
    },
    onError: (error: any) => {
      toast({
        title: "Błąd aktualizacji",
        description: error.message || "Nie udało się zaktualizować normy materiałowej.",
        variant: "destructive",
      });
    },
  });

  const deleteMaterialNormMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("DELETE", `/api/production/norms/materials/${id}`, {});
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/norms/materials"] });
      toast({ title: "Norma materiałowa usunięta", description: "Norma materiałowa została pomyślnie usunięta." });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd usuwania",
        description: error.message || "Nie udało się usunąć normy materiałowej.",
        variant: "destructive",
      });
    },
  });

  // Reset forms
  const resetSettingForm = () => {
    setSettingFormData({
      key: "",
      value: "",
      valueType: "string",
      description: "",
      category: "general",
      isActive: true,
    });
  };

  const resetNormGroupForm = () => {
    setNormGroupFormData({
      code: "",
      name: "",
      description: "",
      unitOfMeasure: "",
      isActive: true,
      sortOrder: 0,
    });
  };

  const resetNormForm = () => {
    setNormFormData({
      normGroupId: 0,
      code: "",
      name: "",
      description: "",
      defaultValue: "",
      minValue: "",
      maxValue: "",
      tolerance: "",
      unitOfMeasure: "",
      applicableTo: [],
      isActive: true,
    });
  };

  const resetMaterialNormForm = () => {
    setMaterialNormFormData({
      materialType: "board",
      materialCategory: "",
      wastePercentage: "",
      edgeOverageCm: "",
      consumptionPerUnit: "",
      inputUnit: "m2",
      outputUnit: "m2",
      isActive: true,
      description: "",
      notes: "",
    });
  };

  // Edit handlers
  const handleEditSetting = (setting: ProductionSetting) => {
    setEditingSetting(setting);
    setSettingFormData({
      key: setting.key,
      value: setting.value,
      valueType: (setting.valueType || "string") as "string" | "number" | "boolean" | "json",
      description: setting.description || "",
      category: setting.category,
      isActive: setting.isActive ?? true,
    });
    setIsEditSettingDialogOpen(true);
  };

  const handleEditNormGroup = (group: ProductionNormGroup) => {
    setEditingNormGroup(group);
    setNormGroupFormData({
      code: group.code,
      name: group.name,
      description: group.description || "",
      unitOfMeasure: group.unitOfMeasure || "",
      isActive: group.isActive ?? true,
      sortOrder: group.sortOrder ?? 0,
    });
    setIsEditNormGroupDialogOpen(true);
  };

  const handleEditNorm = (norm: ProductionNorm) => {
    setEditingNorm(norm);
    setNormFormData({
      normGroupId: norm.normGroupId,
      code: norm.code,
      name: norm.name,
      description: norm.description || "",
      defaultValue: norm.defaultValue?.toString() || "",
      minValue: norm.minValue?.toString() || "",
      maxValue: norm.maxValue?.toString() || "",
      tolerance: norm.tolerance?.toString() || "",
      unitOfMeasure: norm.unitOfMeasure || "",
      applicableTo: (norm.applicableTo as string[] | null) || [],
      isActive: norm.isActive ?? true,
    });
    setIsEditNormDialogOpen(true);
  };

  const handleEditMaterialNorm = (norm: ProductionMaterialConsumptionNorm) => {
    setEditingMaterialNorm(norm);
    setMaterialNormFormData({
      materialType: norm.materialType,
      materialCategory: norm.materialCategory || "",
      wastePercentage: norm.wastePercentage?.toString() || "",
      edgeOverageCm: norm.edgeOverageCm?.toString() || "",
      consumptionPerUnit: norm.consumptionPerUnit?.toString() || "",
      inputUnit: norm.inputUnit || "m2",
      outputUnit: norm.outputUnit || "m2",
      isActive: norm.isActive ?? true,
      description: norm.description || "",
      notes: norm.notes || "",
    });
    setIsEditMaterialNormDialogOpen(true);
  };

  return (
    <div className="p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold" data-testid="text-page-title">Ustawienia Produkcji</h1>
          <p className="text-muted-foreground" data-testid="text-page-description">
            Zarządzaj ustawieniami produkcji i normatywami
          </p>
        </div>
        <SettingsIcon className="h-8 w-8 text-muted-foreground" />
      </div>

      <Tabs defaultValue="settings" className="space-y-4">
        <TabsList data-testid="tabs-main">
          <TabsTrigger value="settings" data-testid="tab-settings">Ustawienia</TabsTrigger>
          <TabsTrigger value="norm-groups" data-testid="tab-norm-groups">Grupy Normatywów</TabsTrigger>
          <TabsTrigger value="norms" data-testid="tab-norms">Normatywy</TabsTrigger>
          <TabsTrigger value="material-norms" data-testid="tab-material-norms">Normy Materiałowe</TabsTrigger>
        </TabsList>

        <TabsContent value="settings" className="space-y-4">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between gap-1 space-y-0 pb-2">
              <div>
                <CardTitle>Ustawienia Produkcji</CardTitle>
                <CardDescription>Ogólne ustawienia systemu produkcji</CardDescription>
              </div>
              <Button 
                onClick={() => setIsAddSettingDialogOpen(true)}
                data-testid="button-add-setting"
              >
                <Plus className="h-4 w-4 mr-2" />
                Dodaj ustawienie
              </Button>
            </CardHeader>
            <CardContent>
              {settingsLoading ? (
                <div data-testid="loading-settings">Ładowanie...</div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Klucz</TableHead>
                      <TableHead>Wartość</TableHead>
                      <TableHead>Typ</TableHead>
                      <TableHead>Kategoria</TableHead>
                      <TableHead>Status</TableHead>
                      <TableHead className="text-right">Akcje</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {settings.length === 0 ? (
                      <TableRow>
                        <TableCell colSpan={6} className="text-center text-muted-foreground" data-testid="text-no-settings">
                          Brak ustawień produkcji
                        </TableCell>
                      </TableRow>
                    ) : (
                      settings.map((setting) => (
                        <TableRow key={setting.id} data-testid={`row-setting-${setting.id}`}>
                          <TableCell className="font-medium" data-testid={`text-setting-key-${setting.id}`}>
                            {setting.key}
                          </TableCell>
                          <TableCell data-testid={`text-setting-value-${setting.id}`}>{setting.value}</TableCell>
                          <TableCell data-testid={`text-setting-type-${setting.id}`}>
                            <Badge variant="outline">{setting.valueType}</Badge>
                          </TableCell>
                          <TableCell data-testid={`text-setting-category-${setting.id}`}>{setting.category}</TableCell>
                          <TableCell data-testid={`text-setting-status-${setting.id}`}>
                            <Badge variant={setting.isActive ? "default" : "secondary"}>
                              {setting.isActive ? "Aktywne" : "Nieaktywne"}
                            </Badge>
                          </TableCell>
                          <TableCell className="text-right">
                            <Button
                              size="icon"
                              variant="ghost"
                              onClick={() => handleEditSetting(setting)}
                              data-testid={`button-edit-setting-${setting.id}`}
                            >
                              <Edit2 className="h-4 w-4" />
                            </Button>
                            <Button
                              size="icon"
                              variant="ghost"
                              onClick={() => deleteSettingMutation.mutate(setting.id)}
                              data-testid={`button-delete-setting-${setting.id}`}
                            >
                              <Trash2 className="h-4 w-4" />
                            </Button>
                          </TableCell>
                        </TableRow>
                      ))
                    )}
                  </TableBody>
                </Table>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="norm-groups" className="space-y-4">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between gap-1 space-y-0 pb-2">
              <div>
                <CardTitle>Grupy Normatywów</CardTitle>
                <CardDescription>Kategorie dla normatywów produkcyjnych</CardDescription>
              </div>
              <Button 
                onClick={() => setIsAddNormGroupDialogOpen(true)}
                data-testid="button-add-norm-group"
              >
                <Plus className="h-4 w-4 mr-2" />
                Dodaj grupę
              </Button>
            </CardHeader>
            <CardContent>
              {normGroupsLoading ? (
                <div data-testid="loading-norm-groups">Ładowanie...</div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Kod</TableHead>
                      <TableHead>Nazwa</TableHead>
                      <TableHead>Jednostka</TableHead>
                      <TableHead>Status</TableHead>
                      <TableHead className="text-right">Akcje</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {normGroups.length === 0 ? (
                      <TableRow>
                        <TableCell colSpan={5} className="text-center text-muted-foreground" data-testid="text-no-norm-groups">
                          Brak grup normatywów
                        </TableCell>
                      </TableRow>
                    ) : (
                      normGroups.map((group) => (
                        <TableRow key={group.id} data-testid={`row-norm-group-${group.id}`}>
                          <TableCell className="font-medium" data-testid={`text-norm-group-code-${group.id}`}>
                            {group.code}
                          </TableCell>
                          <TableCell data-testid={`text-norm-group-name-${group.id}`}>{group.name}</TableCell>
                          <TableCell data-testid={`text-norm-group-unit-${group.id}`}>
                            {group.unitOfMeasure || "-"}
                          </TableCell>
                          <TableCell data-testid={`text-norm-group-status-${group.id}`}>
                            <Badge variant={group.isActive ? "default" : "secondary"}>
                              {group.isActive ? "Aktywna" : "Nieaktywna"}
                            </Badge>
                          </TableCell>
                          <TableCell className="text-right">
                            <Button
                              size="icon"
                              variant="ghost"
                              onClick={() => handleEditNormGroup(group)}
                              data-testid={`button-edit-norm-group-${group.id}`}
                            >
                              <Edit2 className="h-4 w-4" />
                            </Button>
                            <Button
                              size="icon"
                              variant="ghost"
                              onClick={() => deleteNormGroupMutation.mutate(group.id)}
                              data-testid={`button-delete-norm-group-${group.id}`}
                            >
                              <Trash2 className="h-4 w-4" />
                            </Button>
                          </TableCell>
                        </TableRow>
                      ))
                    )}
                  </TableBody>
                </Table>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="norms" className="space-y-4">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between gap-1 space-y-0 pb-2">
              <div>
                <CardTitle>Normatywy</CardTitle>
                <CardDescription>Standardy i normy dla procesów produkcyjnych</CardDescription>
              </div>
              <Button 
                onClick={() => setIsAddNormDialogOpen(true)}
                disabled={normGroups.length === 0}
                data-testid="button-add-norm"
              >
                <Plus className="h-4 w-4 mr-2" />
                Dodaj normatyw
              </Button>
            </CardHeader>
            <CardContent>
              {normsLoading ? (
                <div data-testid="loading-norms">Ładowanie...</div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Kod</TableHead>
                      <TableHead>Nazwa</TableHead>
                      <TableHead>Wartość domyślna</TableHead>
                      <TableHead>Min-Max</TableHead>
                      <TableHead>Jednostka</TableHead>
                      <TableHead>Status</TableHead>
                      <TableHead className="text-right">Akcje</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {norms.length === 0 ? (
                      <TableRow>
                        <TableCell colSpan={7} className="text-center text-muted-foreground" data-testid="text-no-norms">
                          {normGroups.length === 0 
                            ? "Najpierw dodaj grupę normatywów" 
                            : "Brak normatywów"}
                        </TableCell>
                      </TableRow>
                    ) : (
                      norms.map((norm) => {
                        const group = normGroups.find(g => g.id === norm.normGroupId);
                        return (
                          <TableRow key={norm.id} data-testid={`row-norm-${norm.id}`}>
                            <TableCell className="font-medium" data-testid={`text-norm-code-${norm.id}`}>
                              {norm.code}
                            </TableCell>
                            <TableCell data-testid={`text-norm-name-${norm.id}`}>
                              <div>{norm.name}</div>
                              <div className="text-sm text-muted-foreground">
                                {group?.name}
                              </div>
                            </TableCell>
                            <TableCell data-testid={`text-norm-default-${norm.id}`}>
                              {norm.defaultValue}
                            </TableCell>
                            <TableCell data-testid={`text-norm-range-${norm.id}`}>
                              {norm.minValue || "-"} - {norm.maxValue || "-"}
                            </TableCell>
                            <TableCell data-testid={`text-norm-unit-${norm.id}`}>
                              {norm.unitOfMeasure || group?.unitOfMeasure || "-"}
                            </TableCell>
                            <TableCell data-testid={`text-norm-status-${norm.id}`}>
                              <Badge variant={norm.isActive ? "default" : "secondary"}>
                                {norm.isActive ? "Aktywny" : "Nieaktywny"}
                              </Badge>
                            </TableCell>
                            <TableCell className="text-right">
                              <Button
                                size="icon"
                                variant="ghost"
                                onClick={() => handleEditNorm(norm)}
                                data-testid={`button-edit-norm-${norm.id}`}
                              >
                                <Edit2 className="h-4 w-4" />
                              </Button>
                              <Button
                                size="icon"
                                variant="ghost"
                                onClick={() => deleteNormMutation.mutate(norm.id)}
                                data-testid={`button-delete-norm-${norm.id}`}
                              >
                                <Trash2 className="h-4 w-4" />
                              </Button>
                            </TableCell>
                          </TableRow>
                        );
                      })
                    )}
                  </TableBody>
                </Table>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="material-norms" className="space-y-4">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between gap-1 space-y-0 pb-2">
              <div>
                <CardTitle>Normy Materiałowe</CardTitle>
                <CardDescription>Normy zużycia materiałów w procesie produkcji</CardDescription>
              </div>
              <Button 
                onClick={() => setIsAddMaterialNormDialogOpen(true)}
                data-testid="button-add-material-norm"
              >
                <Plus className="h-4 w-4 mr-2" />
                Dodaj normę
              </Button>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex items-center gap-4">
                <Label htmlFor="material-type-filter">Filtruj po typie:</Label>
                <Select
                  value={materialTypeFilter}
                  onValueChange={setMaterialTypeFilter}
                >
                  <SelectTrigger id="material-type-filter" className="w-[200px]" data-testid="select-material-type-filter">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Wszystkie</SelectItem>
                    <SelectItem value="board">Płyty</SelectItem>
                    <SelectItem value="edging">Obrzeża</SelectItem>
                    <SelectItem value="adhesive">Kleje</SelectItem>
                    <SelectItem value="screw">Śruby</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              {materialNormsLoading ? (
                <div data-testid="loading-material-norms">Ładowanie...</div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Typ</TableHead>
                      <TableHead>Kategoria</TableHead>
                      <TableHead>Odpad %</TableHead>
                      <TableHead>Naddatek cm</TableHead>
                      <TableHead>Zużycie/jedn.</TableHead>
                      <TableHead>Jednostki</TableHead>
                      <TableHead>Status</TableHead>
                      <TableHead className="text-right">Akcje</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {materialNorms
                      .filter(norm => materialTypeFilter === "all" || norm.materialType === materialTypeFilter)
                      .length === 0 ? (
                      <TableRow>
                        <TableCell colSpan={8} className="text-center text-muted-foreground" data-testid="text-no-material-norms">
                          Brak norm materiałowych
                        </TableCell>
                      </TableRow>
                    ) : (
                      materialNorms
                        .filter(norm => materialTypeFilter === "all" || norm.materialType === materialTypeFilter)
                        .map((norm) => {
                          const typeLabels: Record<string, string> = {
                            board: "Płyta",
                            edging: "Obrzeże",
                            adhesive: "Klej",
                            screw: "Śruba",
                          };
                          return (
                            <TableRow key={norm.id} data-testid={`row-material-norm-${norm.id}`}>
                              <TableCell>
                                <Badge variant="outline" data-testid={`badge-material-type-${norm.id}`}>
                                  {typeLabels[norm.materialType] || norm.materialType}
                                </Badge>
                              </TableCell>
                              <TableCell data-testid={`text-material-category-${norm.id}`}>
                                {norm.materialCategory || "-"}
                              </TableCell>
                              <TableCell data-testid={`text-waste-${norm.id}`}>
                                {norm.wastePercentage !== null && norm.wastePercentage !== undefined 
                                  ? `${norm.wastePercentage}%` 
                                  : "-"}
                              </TableCell>
                              <TableCell data-testid={`text-overage-${norm.id}`}>
                                {norm.edgeOverageCm !== null && norm.edgeOverageCm !== undefined 
                                  ? `${norm.edgeOverageCm} cm` 
                                  : "-"}
                              </TableCell>
                              <TableCell data-testid={`text-consumption-${norm.id}`}>
                                {norm.consumptionPerUnit !== null && norm.consumptionPerUnit !== undefined 
                                  ? norm.consumptionPerUnit 
                                  : "-"}
                              </TableCell>
                              <TableCell data-testid={`text-units-${norm.id}`}>
                                {norm.inputUnit} → {norm.outputUnit}
                              </TableCell>
                              <TableCell>
                                <Badge variant={norm.isActive ? "default" : "secondary"} data-testid={`badge-active-${norm.id}`}>
                                  {norm.isActive ? "Aktywna" : "Nieaktywna"}
                                </Badge>
                              </TableCell>
                              <TableCell className="text-right">
                                <Button
                                  size="icon"
                                  variant="ghost"
                                  onClick={() => handleEditMaterialNorm(norm)}
                                  data-testid={`button-edit-material-norm-${norm.id}`}
                                >
                                  <Edit2 className="h-4 w-4" />
                                </Button>
                                <Button
                                  size="icon"
                                  variant="ghost"
                                  onClick={() => deleteMaterialNormMutation.mutate(norm.id)}
                                  data-testid={`button-delete-material-norm-${norm.id}`}
                                >
                                  <Trash2 className="h-4 w-4" />
                                </Button>
                              </TableCell>
                            </TableRow>
                          );
                        })
                    )}
                  </TableBody>
                </Table>
              )}
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>

      {/* Add Setting Dialog */}
      <Dialog open={isAddSettingDialogOpen} onOpenChange={setIsAddSettingDialogOpen}>
        <DialogContent data-testid="dialog-add-setting">
          <DialogHeader>
            <DialogTitle>Dodaj Ustawienie Produkcji</DialogTitle>
            <DialogDescription>
              Wprowadź dane nowego ustawienia produkcyjnego.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div className="space-y-2">
              <Label htmlFor="setting-key">Klucz</Label>
              <Input
                id="setting-key"
                value={settingFormData.key}
                onChange={(e) => setSettingFormData({ ...settingFormData, key: e.target.value })}
                placeholder="np. max_production_batch_size"
                data-testid="input-setting-key"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="setting-value">Wartość</Label>
              <Input
                id="setting-value"
                value={settingFormData.value}
                onChange={(e) => setSettingFormData({ ...settingFormData, value: e.target.value })}
                placeholder="np. 100"
                data-testid="input-setting-value"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="setting-value-type">Typ wartości</Label>
              <Select
                value={settingFormData.valueType}
                onValueChange={(value) => setSettingFormData({ ...settingFormData, valueType: value as any })}
              >
                <SelectTrigger id="setting-value-type" data-testid="select-setting-value-type">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="string">Tekst</SelectItem>
                  <SelectItem value="number">Liczba</SelectItem>
                  <SelectItem value="boolean">Boolean</SelectItem>
                  <SelectItem value="json">JSON</SelectItem>
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label htmlFor="setting-category">Kategoria</Label>
              <Input
                id="setting-category"
                value={settingFormData.category}
                onChange={(e) => setSettingFormData({ ...settingFormData, category: e.target.value })}
                placeholder="np. production, quality, etc."
                data-testid="input-setting-category"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="setting-description">Opis</Label>
              <Textarea
                id="setting-description"
                value={settingFormData.description}
                onChange={(e) => setSettingFormData({ ...settingFormData, description: e.target.value })}
                placeholder="Opcjonalny opis ustawienia"
                data-testid="textarea-setting-description"
              />
            </div>
            <div className="flex items-center space-x-2">
              <Switch
                id="setting-active"
                checked={settingFormData.isActive}
                onCheckedChange={(checked) => setSettingFormData({ ...settingFormData, isActive: checked })}
                data-testid="switch-setting-active"
              />
              <Label htmlFor="setting-active">Aktywne</Label>
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setIsAddSettingDialogOpen(false)} data-testid="button-cancel-add-setting">
              Anuluj
            </Button>
            <Button onClick={() => addSettingMutation.mutate()} data-testid="button-save-setting">
              Dodaj
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Edit Setting Dialog - Similar structure */}
      <Dialog open={isEditSettingDialogOpen} onOpenChange={setIsEditSettingDialogOpen}>
        <DialogContent data-testid="dialog-edit-setting">
          <DialogHeader>
            <DialogTitle>Edytuj Ustawienie</DialogTitle>
          </DialogHeader>
          <div className="space-y-4">
            <div className="space-y-2">
              <Label htmlFor="edit-setting-key">Klucz</Label>
              <Input
                id="edit-setting-key"
                value={settingFormData.key}
                onChange={(e) => setSettingFormData({ ...settingFormData, key: e.target.value })}
                data-testid="input-edit-setting-key"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="edit-setting-value">Wartość</Label>
              <Input
                id="edit-setting-value"
                value={settingFormData.value}
                onChange={(e) => setSettingFormData({ ...settingFormData, value: e.target.value })}
                data-testid="input-edit-setting-value"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="edit-setting-value-type">Typ wartości</Label>
              <Select
                value={settingFormData.valueType}
                onValueChange={(value) => setSettingFormData({ ...settingFormData, valueType: value as any })}
              >
                <SelectTrigger id="edit-setting-value-type" data-testid="select-edit-setting-value-type">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="string">Tekst</SelectItem>
                  <SelectItem value="number">Liczba</SelectItem>
                  <SelectItem value="boolean">Boolean</SelectItem>
                  <SelectItem value="json">JSON</SelectItem>
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label htmlFor="edit-setting-category">Kategoria</Label>
              <Input
                id="edit-setting-category"
                value={settingFormData.category}
                onChange={(e) => setSettingFormData({ ...settingFormData, category: e.target.value })}
                data-testid="input-edit-setting-category"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="edit-setting-description">Opis</Label>
              <Textarea
                id="edit-setting-description"
                value={settingFormData.description}
                onChange={(e) => setSettingFormData({ ...settingFormData, description: e.target.value })}
                data-testid="textarea-edit-setting-description"
              />
            </div>
            <div className="flex items-center space-x-2">
              <Switch
                id="edit-setting-active"
                checked={settingFormData.isActive}
                onCheckedChange={(checked) => setSettingFormData({ ...settingFormData, isActive: checked })}
                data-testid="switch-edit-setting-active"
              />
              <Label htmlFor="edit-setting-active">Aktywne</Label>
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setIsEditSettingDialogOpen(false)} data-testid="button-cancel-edit-setting">
              Anuluj
            </Button>
            <Button onClick={() => updateSettingMutation.mutate()} data-testid="button-update-setting">
              Zapisz
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Add Material Norm Dialog */}
      <Dialog open={isAddMaterialNormDialogOpen} onOpenChange={setIsAddMaterialNormDialogOpen}>
        <DialogContent data-testid="dialog-add-material-norm" className="max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>Dodaj Normę Materiałową</DialogTitle>
            <DialogDescription>
              Wprowadź dane nowej normy zużycia materiału. Pola zmieniają się w zależności od typu materiału.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div className="space-y-2">
              <Label htmlFor="material-type">Typ materiału *</Label>
              <Select
                value={materialNormFormData.materialType}
                onValueChange={(value) => setMaterialNormFormData({ ...materialNormFormData, materialType: value })}
              >
                <SelectTrigger id="material-type" data-testid="select-material-type">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="board">Płyta</SelectItem>
                  <SelectItem value="edging">Obrzeże</SelectItem>
                  <SelectItem value="adhesive">Klej</SelectItem>
                  <SelectItem value="screw">Śruba</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label htmlFor="material-category">Kategoria materiału</Label>
              <Input
                id="material-category"
                value={materialNormFormData.materialCategory}
                onChange={(e) => setMaterialNormFormData({ ...materialNormFormData, materialCategory: e.target.value })}
                placeholder="np. PAL, MDF, HDF, ABS, PCV"
                data-testid="input-material-category"
              />
            </div>

            {/* Board-specific: Waste Percentage */}
            <div className="space-y-2">
              <Label htmlFor="waste-percentage">
                Odpad % {materialNormFormData.materialType === "board" && <span className="text-destructive">*</span>}
              </Label>
              <Input
                id="waste-percentage"
                type="number"
                step="0.01"
                value={materialNormFormData.wastePercentage}
                onChange={(e) => setMaterialNormFormData({ ...materialNormFormData, wastePercentage: e.target.value })}
                placeholder="np. 20"
                disabled={materialNormFormData.materialType !== "board"}
                data-testid="input-waste-percentage"
              />
              {materialNormFormData.materialType === "board" && (
                <p className="text-sm text-muted-foreground">Procent odpadu przy krojeniu płyt (np. 20%)</p>
              )}
            </div>

            {/* Edging-specific: Edge Overage */}
            <div className="space-y-2">
              <Label htmlFor="edge-overage">
                Naddatek cm {materialNormFormData.materialType === "edging" && <span className="text-destructive">*</span>}
              </Label>
              <Input
                id="edge-overage"
                type="number"
                step="0.1"
                value={materialNormFormData.edgeOverageCm}
                onChange={(e) => setMaterialNormFormData({ ...materialNormFormData, edgeOverageCm: e.target.value })}
                placeholder="np. 2"
                disabled={materialNormFormData.materialType !== "edging"}
                data-testid="input-edge-overage"
              />
              {materialNormFormData.materialType === "edging" && (
                <p className="text-sm text-muted-foreground">Naddatek obrzeży w centymetrach (np. 2 cm)</p>
              )}
            </div>

            {/* Adhesive/Screw-specific: Consumption Per Unit */}
            <div className="space-y-2">
              <Label htmlFor="consumption-per-unit">
                Zużycie na jednostkę {(materialNormFormData.materialType === "adhesive" || materialNormFormData.materialType === "screw") && <span className="text-destructive">*</span>}
              </Label>
              <Input
                id="consumption-per-unit"
                type="number"
                step="0.01"
                value={materialNormFormData.consumptionPerUnit}
                onChange={(e) => setMaterialNormFormData({ ...materialNormFormData, consumptionPerUnit: e.target.value })}
                placeholder="np. 50"
                disabled={materialNormFormData.materialType !== "adhesive" && materialNormFormData.materialType !== "screw"}
                data-testid="input-consumption-per-unit"
              />
              {(materialNormFormData.materialType === "adhesive" || materialNormFormData.materialType === "screw") && (
                <p className="text-sm text-muted-foreground">
                  {materialNormFormData.materialType === "adhesive" ? "Gramów kleju na metr bieżący (np. 50g/mb)" : "Sztuk śrub na jednostkę (np. 4 szt/produkt)"}
                </p>
              )}
            </div>

            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="input-unit">Jednostka wejściowa *</Label>
                <Select
                  value={materialNormFormData.inputUnit}
                  onValueChange={(value) => setMaterialNormFormData({ ...materialNormFormData, inputUnit: value })}
                >
                  <SelectTrigger id="input-unit" data-testid="select-input-unit">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="m2">m² (metry kwadratowe)</SelectItem>
                    <SelectItem value="mb">mb (metry bieżące)</SelectItem>
                    <SelectItem value="szt">szt (sztuki)</SelectItem>
                    <SelectItem value="kg">kg (kilogramy)</SelectItem>
                    <SelectItem value="g">g (gramy)</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <Label htmlFor="output-unit">Jednostka wyjściowa *</Label>
                <Select
                  value={materialNormFormData.outputUnit}
                  onValueChange={(value) => setMaterialNormFormData({ ...materialNormFormData, outputUnit: value })}
                >
                  <SelectTrigger id="output-unit" data-testid="select-output-unit">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="m2">m² (metry kwadratowe)</SelectItem>
                    <SelectItem value="mb">mb (metry bieżące)</SelectItem>
                    <SelectItem value="szt">szt (sztuki)</SelectItem>
                    <SelectItem value="kg">kg (kilogramy)</SelectItem>
                    <SelectItem value="g">g (gramy)</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>

            <div className="space-y-2">
              <Label htmlFor="material-description">Opis</Label>
              <Textarea
                id="material-description"
                value={materialNormFormData.description}
                onChange={(e) => setMaterialNormFormData({ ...materialNormFormData, description: e.target.value })}
                placeholder="Dodatkowe informacje o normie"
                data-testid="textarea-material-description"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="material-notes">Notatki</Label>
              <Textarea
                id="material-notes"
                value={materialNormFormData.notes}
                onChange={(e) => setMaterialNormFormData({ ...materialNormFormData, notes: e.target.value })}
                placeholder="Notatki techniczne, uwagi"
                data-testid="textarea-material-notes"
              />
            </div>

            <div className="flex items-center space-x-2">
              <Switch
                id="material-active"
                checked={materialNormFormData.isActive}
                onCheckedChange={(checked) => setMaterialNormFormData({ ...materialNormFormData, isActive: checked })}
                data-testid="switch-material-active"
              />
              <Label htmlFor="material-active">Aktywna</Label>
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setIsAddMaterialNormDialogOpen(false)} data-testid="button-cancel-add-material-norm">
              Anuluj
            </Button>
            <Button onClick={() => addMaterialNormMutation.mutate()} disabled={addMaterialNormMutation.isPending} data-testid="button-save-material-norm">
              {addMaterialNormMutation.isPending ? "Zapisywanie..." : "Dodaj"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Edit Material Norm Dialog */}
      <Dialog open={isEditMaterialNormDialogOpen} onOpenChange={setIsEditMaterialNormDialogOpen}>
        <DialogContent data-testid="dialog-edit-material-norm" className="max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>Edytuj Normę Materiałową</DialogTitle>
            <DialogDescription>
              Zaktualizuj dane normy zużycia materiału. Pola zmieniają się w zależności od typu materiału.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div className="space-y-2">
              <Label htmlFor="edit-material-type">Typ materiału *</Label>
              <Select
                value={materialNormFormData.materialType}
                onValueChange={(value) => setMaterialNormFormData({ ...materialNormFormData, materialType: value })}
              >
                <SelectTrigger id="edit-material-type" data-testid="select-edit-material-type">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="board">Płyta</SelectItem>
                  <SelectItem value="edging">Obrzeże</SelectItem>
                  <SelectItem value="adhesive">Klej</SelectItem>
                  <SelectItem value="screw">Śruba</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label htmlFor="edit-material-category">Kategoria materiału</Label>
              <Input
                id="edit-material-category"
                value={materialNormFormData.materialCategory}
                onChange={(e) => setMaterialNormFormData({ ...materialNormFormData, materialCategory: e.target.value })}
                placeholder="np. PAL, MDF, HDF, ABS, PCV"
                data-testid="input-edit-material-category"
              />
            </div>

            {/* Board-specific: Waste Percentage */}
            <div className="space-y-2">
              <Label htmlFor="edit-waste-percentage">
                Odpad % {materialNormFormData.materialType === "board" && <span className="text-destructive">*</span>}
              </Label>
              <Input
                id="edit-waste-percentage"
                type="number"
                step="0.01"
                value={materialNormFormData.wastePercentage}
                onChange={(e) => setMaterialNormFormData({ ...materialNormFormData, wastePercentage: e.target.value })}
                placeholder="np. 20"
                disabled={materialNormFormData.materialType !== "board"}
                data-testid="input-edit-waste-percentage"
              />
              {materialNormFormData.materialType === "board" && (
                <p className="text-sm text-muted-foreground">Procent odpadu przy krojeniu płyt (np. 20%)</p>
              )}
            </div>

            {/* Edging-specific: Edge Overage */}
            <div className="space-y-2">
              <Label htmlFor="edit-edge-overage">
                Naddatek cm {materialNormFormData.materialType === "edging" && <span className="text-destructive">*</span>}
              </Label>
              <Input
                id="edit-edge-overage"
                type="number"
                step="0.1"
                value={materialNormFormData.edgeOverageCm}
                onChange={(e) => setMaterialNormFormData({ ...materialNormFormData, edgeOverageCm: e.target.value })}
                placeholder="np. 2"
                disabled={materialNormFormData.materialType !== "edging"}
                data-testid="input-edit-edge-overage"
              />
              {materialNormFormData.materialType === "edging" && (
                <p className="text-sm text-muted-foreground">Naddatek obrzeży w centymetrach (np. 2 cm)</p>
              )}
            </div>

            {/* Adhesive/Screw-specific: Consumption Per Unit */}
            <div className="space-y-2">
              <Label htmlFor="edit-consumption-per-unit">
                Zużycie na jednostkę {(materialNormFormData.materialType === "adhesive" || materialNormFormData.materialType === "screw") && <span className="text-destructive">*</span>}
              </Label>
              <Input
                id="edit-consumption-per-unit"
                type="number"
                step="0.01"
                value={materialNormFormData.consumptionPerUnit}
                onChange={(e) => setMaterialNormFormData({ ...materialNormFormData, consumptionPerUnit: e.target.value })}
                placeholder="np. 50"
                disabled={materialNormFormData.materialType !== "adhesive" && materialNormFormData.materialType !== "screw"}
                data-testid="input-edit-consumption-per-unit"
              />
              {(materialNormFormData.materialType === "adhesive" || materialNormFormData.materialType === "screw") && (
                <p className="text-sm text-muted-foreground">
                  {materialNormFormData.materialType === "adhesive" ? "Gramów kleju na metr bieżący (np. 50g/mb)" : "Sztuk śrub na jednostkę (np. 4 szt/produkt)"}
                </p>
              )}
            </div>

            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="edit-input-unit">Jednostka wejściowa *</Label>
                <Select
                  value={materialNormFormData.inputUnit}
                  onValueChange={(value) => setMaterialNormFormData({ ...materialNormFormData, inputUnit: value })}
                >
                  <SelectTrigger id="edit-input-unit" data-testid="select-edit-input-unit">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="m2">m² (metry kwadratowe)</SelectItem>
                    <SelectItem value="mb">mb (metry bieżące)</SelectItem>
                    <SelectItem value="szt">szt (sztuki)</SelectItem>
                    <SelectItem value="kg">kg (kilogramy)</SelectItem>
                    <SelectItem value="g">g (gramy)</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <Label htmlFor="edit-output-unit">Jednostka wyjściowa *</Label>
                <Select
                  value={materialNormFormData.outputUnit}
                  onValueChange={(value) => setMaterialNormFormData({ ...materialNormFormData, outputUnit: value })}
                >
                  <SelectTrigger id="edit-output-unit" data-testid="select-edit-output-unit">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="m2">m² (metry kwadratowe)</SelectItem>
                    <SelectItem value="mb">mb (metry bieżące)</SelectItem>
                    <SelectItem value="szt">szt (sztuki)</SelectItem>
                    <SelectItem value="kg">kg (kilogramy)</SelectItem>
                    <SelectItem value="g">g (gramy)</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>

            <div className="space-y-2">
              <Label htmlFor="edit-material-description">Opis</Label>
              <Textarea
                id="edit-material-description"
                value={materialNormFormData.description}
                onChange={(e) => setMaterialNormFormData({ ...materialNormFormData, description: e.target.value })}
                placeholder="Dodatkowe informacje o normie"
                data-testid="textarea-edit-material-description"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="edit-material-notes">Notatki</Label>
              <Textarea
                id="edit-material-notes"
                value={materialNormFormData.notes}
                onChange={(e) => setMaterialNormFormData({ ...materialNormFormData, notes: e.target.value })}
                placeholder="Notatki techniczne, uwagi"
                data-testid="textarea-edit-material-notes"
              />
            </div>

            <div className="flex items-center space-x-2">
              <Switch
                id="edit-material-active"
                checked={materialNormFormData.isActive}
                onCheckedChange={(checked) => setMaterialNormFormData({ ...materialNormFormData, isActive: checked })}
                data-testid="switch-edit-material-active"
              />
              <Label htmlFor="edit-material-active">Aktywna</Label>
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setIsEditMaterialNormDialogOpen(false)} data-testid="button-cancel-edit-material-norm">
              Anuluj
            </Button>
            <Button onClick={() => updateMaterialNormMutation.mutate()} disabled={updateMaterialNormMutation.isPending} data-testid="button-update-material-norm">
              {updateMaterialNormMutation.isPending ? "Zapisywanie..." : "Zapisz"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
