import { useState } from "react";
import { useMutation } from "@tanstack/react-query";
import { AccessoriesSubmenu } from "@/components/accessories-submenu";
import { Sheet, SheetContent, SheetTrigger } from "@/components/ui/sheet";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogFooter } from "@/components/ui/dialog";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { AlertTriangle, Menu } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { apiRequest, queryClient } from "@/lib/queryClient";

interface AccessoriesLayoutProps {
  children: React.ReactNode;
}

type AccessoryGroup = {
  id: number;
  name: string;
  code: string;
  category: string | null;
  description: string | null;
  displayOrder: number;
  isActive: boolean;
};

const groupFormSchema = z.object({
  name: z.string().min(1, "Nazwa jest wymagana"),
  code: z.string().min(1, "Kod jest wymagany"),
  category: z.string().optional(),
  description: z.string().optional(),
  displayOrder: z.number().int().min(0).default(0),
});

type GroupFormData = z.infer<typeof groupFormSchema>;

export default function AccessoriesLayout({ children }: AccessoriesLayoutProps) {
  const { toast } = useToast();
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);
  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false);
  const [isEditDialogOpen, setIsEditDialogOpen] = useState(false);
  const [editingGroup, setEditingGroup] = useState<AccessoryGroup | null>(null);
  const [isDuplicateDialogOpen, setIsDuplicateDialogOpen] = useState(false);
  const [duplicatingGroup, setDuplicatingGroup] = useState<AccessoryGroup | null>(null);
  const [isDeleteDialogOpen, setIsDeleteDialogOpen] = useState(false);
  const [deletingGroup, setDeletingGroup] = useState<AccessoryGroup | null>(null);

  const form = useForm<GroupFormData>({
    resolver: zodResolver(groupFormSchema),
    defaultValues: {
      name: "",
      code: "",
      category: "",
      description: "",
      displayOrder: 0,
    },
  });

  const createGroupMutation = useMutation({
    mutationFn: async (data: GroupFormData) => {
      return apiRequest("POST", "/api/accessory-groups", data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/accessory-groups"] });
      setIsAddDialogOpen(false);
      form.reset();
      toast({
        title: "Sukces",
        description: "Grupa akcesoriów została utworzona",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się utworzyć grupy",
        variant: "destructive",
      });
    },
  });

  const handleAddNew = () => {
    form.reset({
      name: "",
      code: "",
      category: "",
      description: "",
      displayOrder: 0,
    });
    setIsAddDialogOpen(true);
  };

  const handleEditGroup = (group: AccessoryGroup) => {
    setEditingGroup(group);
    form.reset({
      name: group.name,
      code: group.code,
      category: group.category || "",
      description: group.description || "",
      displayOrder: group.displayOrder,
    });
    setIsEditDialogOpen(true);
  };

  const handleDuplicateGroup = (group: AccessoryGroup) => {
    setDuplicatingGroup(group);
    form.reset({
      name: `${group.name} (kopia)`,
      code: `${group.code}_copy`,
      category: group.category || "",
      description: group.description || "",
      displayOrder: group.displayOrder,
    });
    setIsDuplicateDialogOpen(true);
  };

  const handleSubmit = (data: GroupFormData) => {
    createGroupMutation.mutate(data);
  };

  const updateGroupMutation = useMutation({
    mutationFn: async (data: GroupFormData) => {
      if (!editingGroup) throw new Error("Brak grupy do edycji");
      return apiRequest("PATCH", `/api/accessory-groups/${editingGroup.id}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/accessory-groups"] });
      setIsEditDialogOpen(false);
      setEditingGroup(null);
      form.reset();
      toast({
        title: "Sukces",
        description: "Grupa akcesoriów została zaktualizowana",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zaktualizować grupy",
        variant: "destructive",
      });
    },
  });

  const duplicateGroupMutation = useMutation({
    mutationFn: async (data: GroupFormData) => {
      if (!duplicatingGroup) throw new Error("Brak grupy do duplikacji");
      return apiRequest("POST", `/api/accessory-groups/${duplicatingGroup.id}/duplicate`, {
        name: data.name,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/accessory-groups"] });
      queryClient.invalidateQueries({ queryKey: ["/api/accessories"] });
      setIsDuplicateDialogOpen(false);
      setDuplicatingGroup(null);
      form.reset();
      toast({
        title: "Sukces",
        description: "Grupa akcesoriów została zduplikowana",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zduplikować grupy",
        variant: "destructive",
      });
    },
  });

  const handleUpdateSubmit = (data: GroupFormData) => {
    updateGroupMutation.mutate(data);
  };

  const handleDuplicateSubmit = (data: GroupFormData) => {
    duplicateGroupMutation.mutate(data);
  };

  const handleDeleteGroup = (group: AccessoryGroup) => {
    setDeletingGroup(group);
    setIsDeleteDialogOpen(true);
  };

  const deleteGroupMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("DELETE", `/api/accessory-groups/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/accessory-groups"] });
      queryClient.invalidateQueries({ queryKey: ["/api/accessories"] });
      setIsDeleteDialogOpen(false);
      setDeletingGroup(null);
      toast({
        title: "Sukces",
        description: "Grupa akcesoriów została usunięta wraz z zawartością",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się usunąć grupy",
        variant: "destructive",
      });
    },
  });

  const confirmDelete = () => {
    if (deletingGroup) {
      deleteGroupMutation.mutate(deletingGroup.id);
    }
  };

  return (
    <div className="flex h-full w-full overflow-hidden">
      {/* Left submenu - hidden on mobile, visible on md+ */}
      <div className="hidden md:flex w-64 border-r bg-card flex-shrink-0 overflow-y-auto">
        <AccessoriesSubmenu 
          onAddNew={handleAddNew}
          onEditGroup={handleEditGroup}
          onDuplicateGroup={handleDuplicateGroup}
          onDeleteGroup={handleDeleteGroup}
        />
      </div>

      {/* Main content area */}
      <div className="flex-1 overflow-auto bg-background">
        {/* Mobile menu button */}
        <div className="md:hidden sticky top-0 z-10 bg-background border-b p-2">
          <Sheet open={mobileMenuOpen} onOpenChange={setMobileMenuOpen}>
            <SheetTrigger asChild>
              <Button variant="outline" size="sm" data-testid="button-mobile-accessories-menu">
                <Menu className="w-4 h-4 mr-2" />
                Grupy akcesoriów
              </Button>
            </SheetTrigger>
            <SheetContent side="left" className="w-64 p-0">
              <AccessoriesSubmenu 
                onAddNew={handleAddNew}
                onEditGroup={handleEditGroup}
                onDuplicateGroup={handleDuplicateGroup}
                onDeleteGroup={handleDeleteGroup}
              />
            </SheetContent>
          </Sheet>
        </div>

        {children}
      </div>

      {/* Add Group Dialog */}
      <Dialog open={isAddDialogOpen} onOpenChange={setIsAddDialogOpen}>
        <DialogContent className="max-w-[95vw] sm:max-w-lg">
          <DialogHeader>
            <DialogTitle>Dodaj grupę akcesoriów</DialogTitle>
            <DialogDescription>
              Utwórz nową grupę akcesoriów
            </DialogDescription>
          </DialogHeader>

          <Form {...form}>
            <form onSubmit={form.handleSubmit(handleSubmit)} className="space-y-4">
              <FormField
                control={form.control}
                name="name"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Nazwa *</FormLabel>
                    <FormControl>
                      <Input {...field} placeholder="np. Tkanina welur" data-testid="input-name" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="code"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Kod *</FormLabel>
                    <FormControl>
                      <Input {...field} placeholder="np. tkanina_welur" data-testid="input-code" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="category"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Kategoria</FormLabel>
                    <Select onValueChange={field.onChange} value={field.value}>
                      <FormControl>
                        <SelectTrigger data-testid="select-category">
                          <SelectValue placeholder="Wybierz kategorię" />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent>
                        <SelectItem value="Okucia">Okucia</SelectItem>
                        <SelectItem value="Płyty">Płyty meblowe</SelectItem>
                        <SelectItem value="Tkaniny">Tkaniny</SelectItem>
                      </SelectContent>
                    </Select>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="description"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Opis</FormLabel>
                    <FormControl>
                      <Textarea {...field} rows={3} data-testid="input-description" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="displayOrder"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Kolejność wyświetlania</FormLabel>
                    <FormControl>
                      <Input
                        type="number"
                        {...field}
                        onChange={(e) => field.onChange(parseInt(e.target.value) || 0)}
                        data-testid="input-display-order"
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="flex justify-end gap-2">
                <Button
                  type="button"
                  variant="outline"
                  onClick={() => setIsAddDialogOpen(false)}
                  data-testid="button-cancel"
                >
                  Anuluj
                </Button>
                <Button type="submit" disabled={createGroupMutation.isPending} data-testid="button-submit">
                  {createGroupMutation.isPending ? "Tworzenie..." : "Utwórz grupę"}
                </Button>
              </div>
            </form>
          </Form>
        </DialogContent>
      </Dialog>

      {/* Edit Group Dialog */}
      <Dialog open={isEditDialogOpen} onOpenChange={setIsEditDialogOpen}>
        <DialogContent className="max-w-[95vw] sm:max-w-lg">
          <DialogHeader>
            <DialogTitle>Edytuj grupę akcesoriów</DialogTitle>
            <DialogDescription>
              Zaktualizuj informacje o grupie akcesoriów
            </DialogDescription>
          </DialogHeader>

          <Form {...form}>
            <form onSubmit={form.handleSubmit(handleUpdateSubmit)} className="space-y-4">
              <FormField
                control={form.control}
                name="name"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Nazwa *</FormLabel>
                    <FormControl>
                      <Input {...field} placeholder="np. Tkanina welur" data-testid="input-edit-name" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="code"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Kod *</FormLabel>
                    <FormControl>
                      <Input {...field} placeholder="np. tkanina_welur" data-testid="input-edit-code" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="category"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Kategoria</FormLabel>
                    <Select onValueChange={field.onChange} value={field.value}>
                      <FormControl>
                        <SelectTrigger data-testid="select-edit-category">
                          <SelectValue placeholder="Wybierz kategorię" />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent>
                        <SelectItem value="Okucia">Okucia</SelectItem>
                        <SelectItem value="Płyty">Płyty meblowe</SelectItem>
                        <SelectItem value="Tkaniny">Tkaniny</SelectItem>
                      </SelectContent>
                    </Select>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="description"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Opis</FormLabel>
                    <FormControl>
                      <Textarea {...field} rows={3} data-testid="input-edit-description" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="displayOrder"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Kolejność wyświetlania</FormLabel>
                    <FormControl>
                      <Input
                        type="number"
                        {...field}
                        onChange={(e) => field.onChange(parseInt(e.target.value) || 0)}
                        data-testid="input-edit-display-order"
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="flex justify-end gap-2">
                <Button
                  type="button"
                  variant="outline"
                  onClick={() => setIsEditDialogOpen(false)}
                  data-testid="button-edit-cancel"
                >
                  Anuluj
                </Button>
                <Button type="submit" disabled={updateGroupMutation.isPending} data-testid="button-edit-submit">
                  {updateGroupMutation.isPending ? "Zapisywanie..." : "Zapisz zmiany"}
                </Button>
              </div>
            </form>
          </Form>
        </DialogContent>
      </Dialog>

      {/* Duplicate Group Dialog */}
      <Dialog open={isDuplicateDialogOpen} onOpenChange={setIsDuplicateDialogOpen}>
        <DialogContent className="max-w-[95vw] sm:max-w-lg">
          <DialogHeader>
            <DialogTitle>Duplikuj grupę akcesoriów</DialogTitle>
            <DialogDescription>
              Utwórz kopię grupy wraz z wszystkimi akcesoriami
            </DialogDescription>
          </DialogHeader>

          <Form {...form}>
            <form onSubmit={form.handleSubmit(handleDuplicateSubmit)} className="space-y-4">
              <FormField
                control={form.control}
                name="name"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Nazwa *</FormLabel>
                    <FormControl>
                      <Input {...field} placeholder="np. Tkanina welur" data-testid="input-duplicate-name" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="code"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Kod *</FormLabel>
                    <FormControl>
                      <Input {...field} placeholder="np. tkanina_welur" data-testid="input-duplicate-code" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="category"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Kategoria</FormLabel>
                    <Select onValueChange={field.onChange} value={field.value}>
                      <FormControl>
                        <SelectTrigger data-testid="select-duplicate-category">
                          <SelectValue placeholder="Wybierz kategorię" />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent>
                        <SelectItem value="Okucia">Okucia</SelectItem>
                        <SelectItem value="Płyty">Płyty meblowe</SelectItem>
                        <SelectItem value="Tkaniny">Tkaniny</SelectItem>
                      </SelectContent>
                    </Select>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="description"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Opis</FormLabel>
                    <FormControl>
                      <Textarea {...field} rows={3} data-testid="input-duplicate-description" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="displayOrder"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Kolejność wyświetlania</FormLabel>
                    <FormControl>
                      <Input
                        type="number"
                        {...field}
                        onChange={(e) => field.onChange(parseInt(e.target.value) || 0)}
                        data-testid="input-duplicate-display-order"
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="flex justify-end gap-2">
                <Button
                  type="button"
                  variant="outline"
                  onClick={() => setIsDuplicateDialogOpen(false)}
                  data-testid="button-duplicate-cancel"
                >
                  Anuluj
                </Button>
                <Button type="submit" disabled={duplicateGroupMutation.isPending} data-testid="button-duplicate-submit">
                  {duplicateGroupMutation.isPending ? "Duplikowanie..." : "Duplikuj z zawartością"}
                </Button>
              </div>
            </form>
          </Form>
        </DialogContent>
      </Dialog>

      {/* Delete Group Confirmation Dialog */}
      <Dialog open={isDeleteDialogOpen} onOpenChange={setIsDeleteDialogOpen}>
        <DialogContent className="max-w-[95vw] sm:max-w-md">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <AlertTriangle className="h-5 w-5 text-destructive" />
              Usuń grupę akcesoriów
            </DialogTitle>
            <DialogDescription>
              Czy na pewno chcesz usunąć grupę <strong>{deletingGroup?.name}</strong>?
              <br />
              <br />
              <span className="text-destructive font-medium">
                Ta akcja spowoduje również usunięcie wszystkich akcesoriów z tej grupy i nie może być cofnięta.
              </span>
            </DialogDescription>
          </DialogHeader>
          <DialogFooter>
            <Button
              type="button"
              variant="outline"
              onClick={() => setIsDeleteDialogOpen(false)}
              disabled={deleteGroupMutation.isPending}
              data-testid="button-delete-cancel"
            >
              Anuluj
            </Button>
            <Button
              type="button"
              variant="destructive"
              onClick={confirmDelete}
              disabled={deleteGroupMutation.isPending}
              data-testid="button-delete-confirm"
            >
              {deleteGroupMutation.isPending ? "Usuwanie..." : "Usuń z zawartością"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
