import { Link, useRoute } from "wouter";
import { useQuery } from "@tanstack/react-query";
import { Loader2, Package, Pencil, Plus } from "lucide-react";
import { cn } from "@/lib/utils";
import { Button } from "@/components/ui/button";

interface MaterialGroup {
  id: number;
  name: string;
  code: string;
  category: string | null;
  description: string | null;
  displayOrder: number;
  isActive: boolean;
}

interface WarehouseSubmenuProps {
  category: string;
  onEditGroup?: (group: MaterialGroup) => void;
  onAddGroup?: () => void;
}

interface StatsResponse {
  products: {
    catalogProducts: number;
    catalogSets: number;
    productMatrices: number;
    setMatrices: number;
    accessories: number;
  };
  warehouse: {
    categories: Record<string, number>;
    groups: Array<{
      id: number;
      code: string;
      name: string;
      category: string;
      count: number;
    }>;
  };
}

export function WarehouseSubmenu({ category, onEditGroup, onAddGroup }: WarehouseSubmenuProps) {
  const [, params] = useRoute("/warehouse/:category/:groupCode?");
  const activeGroupCode = params?.groupCode;

  const { data: allGroups = [], isLoading } = useQuery<MaterialGroup[]>({
    queryKey: ["/api/warehouse/material-groups"],
  });

  // Fetch stats for counts
  const { data: stats } = useQuery<StatsResponse>({
    queryKey: ['/api/stats/counts'],
    refetchInterval: 30000,
  });

  // Filter groups by category
  const categoryGroups = allGroups
    .filter(g => g.category === category && g.isActive)
    .sort((a, b) => {
      if (a.displayOrder !== b.displayOrder) {
        return a.displayOrder - b.displayOrder;
      }
      return a.name.localeCompare(b.name);
    });

  // Get count for specific group
  const getGroupCount = (groupCode: string): number | null => {
    if (!stats?.warehouse.groups) return null;
    const group = stats.warehouse.groups.find(g => g.code === groupCode);
    return group?.count ?? null;
  };

  // Get total count for category
  const getCategoryCount = (): number | null => {
    if (!stats?.warehouse.categories) return null;
    const categoryKey = category === 'plyty' ? 'plyty_meblowe' : category;
    return stats.warehouse.categories[categoryKey] ?? null;
  };

  if (isLoading) {
    return (
      <div className="flex items-center justify-center p-8">
        <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
      </div>
    );
  }

  return (
    <div className="flex flex-col h-full max-w-64">
      {/* Header */}
      <div className="border-b px-4 py-4 flex items-center justify-between gap-2">
        <h2 className="text-lg font-semibold flex-1">Grupy materiałów</h2>
        {onAddGroup && (
          <Button
            variant="outline"
            size="icon"
            className="h-8 w-8 flex-shrink-0"
            onClick={onAddGroup}
            data-testid="button-add-group-sidebar"
          >
            <Plus className="h-4 w-4" />
          </Button>
        )}
      </div>

      {/* Menu items */}
      <div className="flex-1 overflow-y-auto">
        <div className="p-2">
          <nav className="space-y-1">
            {/* All materials link */}
            <Link 
              href={`/warehouse/${category}`}
              data-testid={`link-warehouse-${category}-all`}
            >
              <div
                className={cn(
                  "flex items-center gap-3 px-3 py-2 rounded-md text-sm transition-colors hover-elevate active-elevate-2",
                  !activeGroupCode
                    ? "bg-accent text-accent-foreground font-medium" 
                    : "text-foreground"
                )}
              >
                <Package className="h-4 w-4 flex-shrink-0" />
                <span className="truncate flex items-center gap-1.5">
                  Wszystkie materiały
                  {getCategoryCount() !== null && (
                    <span className="text-xs text-muted-foreground">
                      ({getCategoryCount()})
                    </span>
                  )}
                </span>
              </div>
            </Link>

            {/* Group links */}
            {categoryGroups.map((group) => {
              const isActive = activeGroupCode === group.code;
              
              return (
                <Link 
                  key={group.id}
                  href={`/warehouse/${category}/${group.code}`}
                  data-testid={`link-warehouse-${category}-${group.code}`}
                  className="min-w-0"
                >
                  <div
                    className={cn(
                      "flex items-center gap-3 px-3 py-2 rounded-md text-sm transition-colors hover-elevate active-elevate-2 min-w-0",
                      isActive 
                        ? "bg-accent text-accent-foreground font-medium" 
                        : "text-foreground"
                    )}
                  >
                    <div className="h-2 w-2 rounded-full bg-current flex-shrink-0" />
                    <span className="truncate flex-1 flex items-center gap-1.5 min-w-0">
                      {group.name}
                      {getGroupCount(group.code) !== null && (
                        <span className="text-xs text-muted-foreground">
                          ({getGroupCount(group.code)})
                        </span>
                      )}
                    </span>
                    {onEditGroup && (
                      <Button
                        variant="ghost"
                        size="icon"
                        className="h-6 w-6 flex-shrink-0"
                        onClick={(e) => {
                          e.preventDefault();
                          e.stopPropagation();
                          onEditGroup(group);
                        }}
                        data-testid={`button-edit-group-${group.id}`}
                      >
                        <Pencil className="h-3 w-3" />
                      </Button>
                    )}
                  </div>
                </Link>
              );
            })}

            {categoryGroups.length === 0 && (
              <div className="text-sm text-muted-foreground px-3 py-2">
                Brak grup w tej kategorii
              </div>
            )}
          </nav>
        </div>
      </div>
    </div>
  );
}
