import { ArrowLeft, Calendar, Hash } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import { format } from "date-fns";
import { pl } from "date-fns/locale";

interface ProductionPlan {
  id: number;
  planNumber: string;
  name: string;
  description?: string;
  plannedStartDate?: string;
  plannedEndDate?: string;
  status: string;
  priority: string;
  notes?: string;
  createdAt: string;
}

interface PlanHeaderProps {
  plan: ProductionPlan;
  onBack: () => void;
}

const statusColors: Record<string, string> = {
  draft: "bg-gray-500",
  approved: "bg-blue-500",
  in_progress: "bg-yellow-500",
  completed: "bg-green-500",
  cancelled: "bg-red-500",
};

const statusLabels: Record<string, string> = {
  draft: "Projekt",
  approved: "Zatwierdzony",
  in_progress: "W realizacji",
  completed: "Zakończony",
  cancelled: "Anulowany",
};

const priorityColors: Record<string, string> = {
  low: "bg-gray-400",
  normal: "bg-blue-400",
  high: "bg-orange-400",
  urgent: "bg-red-500",
};

const priorityLabels: Record<string, string> = {
  low: "Niski",
  normal: "Normalny",
  high: "Wysoki",
  urgent: "Pilny",
};

export function PlanHeader({ plan, onBack }: PlanHeaderProps) {
  return (
    <div 
      className="min-h-[60px] py-2 border-b bg-card flex items-center px-4"
    >
      <div className="flex items-center gap-2 w-full flex-wrap">
        {/* Back Button */}
        <Button
          variant="ghost"
          size="icon"
          onClick={onBack}
          data-testid="button-back"
          className="flex-shrink-0 h-9 w-9"
        >
          <ArrowLeft className="h-5 w-5" />
        </Button>

        <div>
          <h1 className="text-base font-bold leading-tight" data-testid="title-plan-name">
            {plan.name}
          </h1>
        </div>

        <Separator orientation="vertical" className="hidden lg:block h-8" />

        {/* Plan Number with Icon */}
        <div className="flex items-center gap-1.5">
          <Hash className="text-primary h-4 w-4" />
          <span className="font-mono text-primary font-semibold text-sm" data-testid="text-plan-number">
            {plan.planNumber}
          </span>
        </div>

        {/* Start Date */}
        {plan.plannedStartDate && (
          <>
            <Separator orientation="vertical" className="hidden lg:block h-8" />
            <div className="flex items-center gap-1.5">
              <Calendar className="text-muted-foreground h-4 w-4" />
              <span className="text-muted-foreground text-xs hidden sm:inline">Start:</span>
              <span className="font-medium text-sm" data-testid="text-planned-start">
                {format(new Date(plan.plannedStartDate), "dd.MM.yyyy", { locale: pl })}
              </span>
            </div>
          </>
        )}

        {/* End Date */}
        {plan.plannedEndDate && (
          <>
            <Separator orientation="vertical" className="hidden lg:block h-8" />
            <div className="flex items-center gap-1.5">
              <Calendar className="text-muted-foreground h-4 w-4" />
              <span className="text-muted-foreground text-xs hidden sm:inline">Koniec:</span>
              <span className="font-medium text-sm" data-testid="text-planned-end">
                {format(new Date(plan.plannedEndDate), "dd.MM.yyyy", { locale: pl })}
              </span>
            </div>
          </>
        )}

        {/* Spacer */}
        <div className="flex-1 min-w-[20px]" />

        {/* Status Badge */}
        <Badge
          className={`${statusColors[plan.status]} text-white h-6 px-2 text-xs`}
          data-testid={`badge-status-${plan.status}`}
        >
          {statusLabels[plan.status]}
        </Badge>

        {/* Priority Badge */}
        <Badge
          className={`${priorityColors[plan.priority]} text-white h-6 px-2 text-xs`}
          data-testid={`badge-priority-${plan.priority}`}
        >
          {priorityLabels[plan.priority]}
        </Badge>
      </div>
    </div>
  );
}
