import { Pool } from "pg";
import type { ProductionWorkCenter, InsertProductionWorkCenter } from "@shared/schema";

export async function getWorkCenters(pool: Pool): Promise<ProductionWorkCenter[]> {
  const result = await pool.query(`
    SELECT * FROM production.production_work_centers 
    ORDER BY code ASC
  `);
  
  return result.rows.map(row => ({
    id: row.id,
    code: row.code,
    name: row.name,
    description: row.description,
    locationId: row.location_id,
    capabilities: row.capabilities || [],
    status: row.status,
    throughputPerHour: row.throughput_per_hour,
    throughputUnit: row.throughput_unit,
    operatingCostPerHour: row.operating_cost_per_hour,
    shiftCalendar: row.shift_calendar,
    oeeTarget: row.oee_target,
    currentOee: row.current_oee,
    isActive: row.is_active,
    notes: row.notes,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  }));
}

export async function getWorkCenterById(pool: Pool, id: number): Promise<ProductionWorkCenter | null> {
  const result = await pool.query(`
    SELECT * FROM production.production_work_centers WHERE id = $1
  `, [id]);
  
  if (result.rows.length === 0) return null;
  
  const row = result.rows[0];
  return {
    id: row.id,
    code: row.code,
    name: row.name,
    description: row.description,
    locationId: row.location_id,
    capabilities: row.capabilities || [],
    status: row.status,
    throughputPerHour: row.throughput_per_hour,
    throughputUnit: row.throughput_unit,
    operatingCostPerHour: row.operating_cost_per_hour,
    shiftCalendar: row.shift_calendar,
    oeeTarget: row.oee_target,
    currentOee: row.current_oee,
    isActive: row.is_active,
    notes: row.notes,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function createWorkCenter(pool: Pool, data: InsertProductionWorkCenter): Promise<ProductionWorkCenter> {
  const result = await pool.query(`
    INSERT INTO production.production_work_centers 
    (code, name, description, location_id, capabilities, status, throughput_per_hour, throughput_unit,
     operating_cost_per_hour, shift_calendar, oee_target, current_oee, is_active, notes)
    VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14)
    RETURNING *
  `, [
    data.code,
    data.name,
    data.description ?? null,
    data.locationId ?? null,
    data.capabilities ?? [],
    data.status ?? 'available',
    data.throughputPerHour ?? null,
    data.throughputUnit ?? null,
    data.operatingCostPerHour ?? null,
    data.shiftCalendar ?? null,
    data.oeeTarget ?? null,
    data.currentOee ?? null,
    data.isActive ?? true,
    data.notes ?? null,
  ]);
  
  const row = result.rows[0];
  return {
    id: row.id,
    code: row.code,
    name: row.name,
    description: row.description,
    locationId: row.location_id,
    capabilities: row.capabilities || [],
    status: row.status,
    throughputPerHour: row.throughput_per_hour,
    throughputUnit: row.throughput_unit,
    operatingCostPerHour: row.operating_cost_per_hour,
    shiftCalendar: row.shift_calendar,
    oeeTarget: row.oee_target,
    currentOee: row.current_oee,
    isActive: row.is_active,
    notes: row.notes,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function updateWorkCenter(pool: Pool, id: number, data: Partial<InsertProductionWorkCenter>): Promise<ProductionWorkCenter | null> {
  const updates: string[] = [];
  const values: any[] = [];
  let paramIndex = 1;

  if (data.code !== undefined) {
    updates.push(`code = $${paramIndex++}`);
    values.push(data.code);
  }
  if (data.name !== undefined) {
    updates.push(`name = $${paramIndex++}`);
    values.push(data.name);
  }
  if (data.description !== undefined) {
    updates.push(`description = $${paramIndex++}`);
    values.push(data.description);
  }
  if (data.locationId !== undefined) {
    updates.push(`location_id = $${paramIndex++}`);
    values.push(data.locationId);
  }
  if (data.capabilities !== undefined) {
    updates.push(`capabilities = $${paramIndex++}`);
    values.push(data.capabilities);
  }
  if (data.status !== undefined) {
    updates.push(`status = $${paramIndex++}`);
    values.push(data.status);
  }
  if (data.throughputPerHour !== undefined) {
    updates.push(`throughput_per_hour = $${paramIndex++}`);
    values.push(data.throughputPerHour);
  }
  if (data.throughputUnit !== undefined) {
    updates.push(`throughput_unit = $${paramIndex++}`);
    values.push(data.throughputUnit);
  }
  if (data.operatingCostPerHour !== undefined) {
    updates.push(`operating_cost_per_hour = $${paramIndex++}`);
    values.push(data.operatingCostPerHour);
  }
  if (data.shiftCalendar !== undefined) {
    updates.push(`shift_calendar = $${paramIndex++}`);
    values.push(data.shiftCalendar);
  }
  if (data.oeeTarget !== undefined) {
    updates.push(`oee_target = $${paramIndex++}`);
    values.push(data.oeeTarget);
  }
  if (data.currentOee !== undefined) {
    updates.push(`current_oee = $${paramIndex++}`);
    values.push(data.currentOee);
  }
  if (data.isActive !== undefined) {
    updates.push(`is_active = $${paramIndex++}`);
    values.push(data.isActive);
  }
  if (data.notes !== undefined) {
    updates.push(`notes = $${paramIndex++}`);
    values.push(data.notes);
  }

  if (updates.length === 0) {
    return getWorkCenterById(pool, id);
  }

  updates.push(`updated_at = NOW()`);
  values.push(id);

  const result = await pool.query(`
    UPDATE production.production_work_centers 
    SET ${updates.join(", ")}
    WHERE id = $${paramIndex}
    RETURNING *
  `, values);

  if (result.rows.length === 0) return null;

  const row = result.rows[0];
  return {
    id: row.id,
    code: row.code,
    name: row.name,
    description: row.description,
    locationId: row.location_id,
    capabilities: row.capabilities || [],
    status: row.status,
    throughputPerHour: row.throughput_per_hour,
    throughputUnit: row.throughput_unit,
    operatingCostPerHour: row.operating_cost_per_hour,
    shiftCalendar: row.shift_calendar,
    oeeTarget: row.oee_target,
    currentOee: row.current_oee,
    isActive: row.is_active,
    notes: row.notes,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function deleteWorkCenter(pool: Pool, id: number): Promise<boolean> {
  const result = await pool.query(`
    DELETE FROM production.production_work_centers WHERE id = $1
  `, [id]);
  
  return result.rowCount !== null && result.rowCount > 0;
}

export async function seedDefaultWorkCenters(pool: Pool): Promise<{ inserted: number; skipped: number }> {
  const defaultWorkCenters = [
    {
      code: 'PILA-MW',
      name: 'Piła panelowa Masterwood',
      description: 'Piła panelowa do cięcia płyt meblowych',
      status: 'available',
      throughputPerHour: 120.00,
      throughputUnit: 'szt',
      operatingCostPerHour: 150.00,
      oeeTarget: 85.00,
      isActive: true,
    },
    {
      code: 'OKL-FALIT',
      name: 'Okleiniarka Falit',
      description: 'Okleiniarka krawędziowa Falit',
      status: 'available',
      throughputPerHour: 80.00,
      throughputUnit: 'szt',
      operatingCostPerHour: 100.00,
      oeeTarget: 80.00,
      isActive: true,
    },
    {
      code: 'OKL-MW',
      name: 'Okleiniarka Masterwood',
      description: 'Okleiniarka krawędziowa Masterwood',
      status: 'available',
      throughputPerHour: 90.00,
      throughputUnit: 'szt',
      operatingCostPerHour: 120.00,
      oeeTarget: 82.00,
      isActive: true,
    },
    {
      code: 'WIERT-M1',
      name: 'Wiertarka Master 1',
      description: 'Wiertarka CNC Master 1',
      status: 'available',
      throughputPerHour: 60.00,
      throughputUnit: 'szt',
      operatingCostPerHour: 80.00,
      oeeTarget: 85.00,
      isActive: true,
    },
    {
      code: 'WIERT-M2',
      name: 'Wiertarka Master 2',
      description: 'Wiertarka CNC Master 2',
      status: 'available',
      throughputPerHour: 60.00,
      throughputUnit: 'szt',
      operatingCostPerHour: 80.00,
      oeeTarget: 85.00,
      isActive: true,
    },
    {
      code: 'WIERT-NEXT',
      name: 'Wiertarka Next',
      description: 'Wiertarka CNC Next',
      status: 'available',
      throughputPerHour: 70.00,
      throughputUnit: 'szt',
      operatingCostPerHour: 90.00,
      oeeTarget: 88.00,
      isActive: true,
    },
    {
      code: 'KOMPL',
      name: 'Kompletowanie',
      description: 'Stanowisko kompletowania elementów',
      status: 'available',
      throughputPerHour: 50.00,
      throughputUnit: 'szt',
      operatingCostPerHour: 40.00,
      oeeTarget: 75.00,
      isActive: true,
    },
    {
      code: 'PAKOW',
      name: 'Pakowanie',
      description: 'Stanowisko pakowania gotowych produktów',
      status: 'available',
      throughputPerHour: 40.00,
      throughputUnit: 'szt',
      operatingCostPerHour: 30.00,
      oeeTarget: 70.00,
      isActive: true,
    },
  ];

  const result = await pool.query(`
    INSERT INTO production.production_work_centers 
    (code, name, description, status, throughput_per_hour, throughput_unit, operating_cost_per_hour, oee_target, is_active)
    VALUES 
      ('PILA-MW', 'Piła panelowa Masterwood', 'Piła panelowa do cięcia płyt meblowych', 'available', 120.00, 'szt', 150.00, 85.00, true),
      ('OKL-FALIT', 'Okleiniarka Falit', 'Okleiniarka krawędziowa Falit', 'available', 80.00, 'szt', 100.00, 80.00, true),
      ('OKL-MW', 'Okleiniarka Masterwood', 'Okleiniarka krawędziowa Masterwood', 'available', 90.00, 'szt', 120.00, 82.00, true),
      ('WIERT-M1', 'Wiertarka Master 1', 'Wiertarka CNC Master 1', 'available', 60.00, 'szt', 80.00, 85.00, true),
      ('WIERT-M2', 'Wiertarka Master 2', 'Wiertarka CNC Master 2', 'available', 60.00, 'szt', 80.00, 85.00, true),
      ('WIERT-NEXT', 'Wiertarka Next', 'Wiertarka CNC Next', 'available', 70.00, 'szt', 90.00, 88.00, true),
      ('KOMPL', 'Kompletowanie', 'Stanowisko kompletowania elementów', 'available', 50.00, 'szt', 40.00, 75.00, true),
      ('PAKOW', 'Pakowanie', 'Stanowisko pakowania gotowych produktów', 'available', 40.00, 'szt', 30.00, 70.00, true)
    ON CONFLICT (code) DO NOTHING
  `);

  const inserted = result.rowCount || 0;
  const skipped = defaultWorkCenters.length - inserted;

  return { inserted, skipped };
}
