import puppeteer from 'puppeteer';
import crypto from 'crypto';
import { getFileStorageAdapter } from './file-storage-adapter';

interface AccessoryItem {
  code: string;
  name: string;
  imageUrl: string | null;
  description?: string | null;
}

interface GridLayout {
  cols: number;
  rows: number;
  infinite?: boolean; // True if rows should auto-adjust to item count
}

/**
 * Parse grid parameter to determine layout
 * Examples:
 *   "grid3" -> { cols: 3, rows: 1 }
 *   "grid4" -> { cols: 4, rows: 1 }
 *   "grid2-2" -> { cols: 2, rows: 2 }
 *   "grid3-" -> { cols: 3, rows: Infinity, infinite: true } (auto-adjusts to item count)
 *   "grid4-" -> { cols: 4, rows: Infinity, infinite: true } (auto-adjusts to item count)
 */
function parseGridLayout(gridParam: string, itemCount?: number): GridLayout {
  if (gridParam.endsWith('-')) {
    // Format: "grid3-" or "grid4-" (infinite rows)
    const cols = parseInt(gridParam.replace('grid', '').replace('-', ''));
    const rows = itemCount ? Math.ceil(itemCount / cols) : 1;
    return { cols, rows, infinite: true };
  } else if (gridParam.includes('-')) {
    // Format: "grid2-2" (fixed rows-cols)
    const [rows, cols] = gridParam.replace('grid', '').split('-').map(Number);
    return { cols, rows };
  } else {
    // Format: "grid3" or "grid4" (fixed single row)
    const cols = parseInt(gridParam.replace('grid', ''));
    return { cols, rows: 1 };
  }
}

/**
 * Convert relative image URLs to absolute URLs
 */
function normalizeImageUrl(imageUrl: string | null): string | null {
  if (!imageUrl) return null;
  
  // If already absolute URL, return as-is
  if (imageUrl.startsWith('http://') || imageUrl.startsWith('https://')) {
    return imageUrl;
  }
  
  // Convert relative /uploads/accessories/* to full SFTP URL
  if (imageUrl.startsWith('/uploads/accessories/')) {
    const filename = imageUrl.split('/').pop();
    return `https://files.alpsys.pl/OMS/accessories/images/${filename}`;
  }
  
  // Fallback: prepend base URL for any other relative path
  if (imageUrl.startsWith('/')) {
    return `https://files.alpsys.pl${imageUrl}`;
  }
  
  return imageUrl;
}

/**
 * Generate HTML for accessory grid with white background
 */
function generateGridHTML(accessories: AccessoryItem[], layout: GridLayout): string {
  const { cols, rows, infinite } = layout;
  const totalTiles = infinite ? accessories.length : cols * rows;
  
  // Take only the number of accessories we need for the grid
  const displayAccessories = accessories.slice(0, totalTiles).map(acc => ({
    ...acc,
    imageUrl: normalizeImageUrl(acc.imageUrl)
  }));
  
  // Calculate tile dimensions (responsive)
  const tileWidth = Math.floor(100 / cols);
  
  return `
    <!DOCTYPE html>
    <html>
    <head>
      <meta charset="UTF-8">
      <style>
        * {
          margin: 0;
          padding: 0;
          box-sizing: border-box;
        }
        
        body {
          font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
          background: #ffffff;
          padding: 40px;
        }
        
        .grid-container {
          display: grid;
          grid-template-columns: repeat(${cols}, 1fr);
          grid-template-rows: repeat(${rows}, 1fr);
          gap: 30px;
          width: 100%;
          max-width: ${cols * 280}px;
          margin: 0 auto;
        }
        
        .tile {
          background: #ffffff;
          border: 2px solid #e5e7eb;
          border-radius: 12px;
          padding: 20px;
          display: flex;
          flex-direction: column;
          align-items: center;
          justify-content: flex-start;
          min-height: 300px;
        }
        
        .tile-name {
          font-size: 18px;
          font-weight: 700;
          color: #111827;
          text-align: center;
          margin-bottom: 20px;
          line-height: 1.3;
          min-height: 48px;
          display: flex;
          align-items: center;
          justify-content: center;
        }
        
        .tile-image {
          width: 100%;
          max-width: 200px;
          height: auto;
          object-fit: contain;
          flex-grow: 1;
          display: flex;
          align-items: center;
          justify-content: center;
        }
        
        .tile-image img {
          max-width: 100%;
          max-height: 200px;
          width: auto;
          height: auto;
          object-fit: contain;
        }
        
        .no-image {
          width: 200px;
          height: 200px;
          background: #f3f4f6;
          border-radius: 8px;
          display: flex;
          align-items: center;
          justify-content: center;
          color: #9ca3af;
          font-size: 14px;
        }
      </style>
    </head>
    <body>
      <div class="grid-container">
        ${displayAccessories.map(acc => `
          <div class="tile">
            <div class="tile-name">${acc.name}</div>
            <div class="tile-image">
              ${acc.imageUrl 
                ? `<img src="${acc.imageUrl}" alt="${acc.name}" />` 
                : `<div class="no-image">Brak zdjęcia</div>`
              }
            </div>
          </div>
        `).join('')}
      </div>
    </body>
    </html>
  `;
}

/**
 * Generate a unique hash for cache key based on accessory codes and layout
 */
function generateCacheKey(accessories: AccessoryItem[], gridParam: string): string {
  const codes = accessories.map(a => a.code).sort().join(',');
  const key = `${codes}_${gridParam}`;
  return crypto.createHash('md5').update(key).digest('hex');
}

/**
 * Render accessory grid as JPG and upload to SFTP
 * @returns Public URL of the generated JPG image
 */
export async function renderAccessoryGridAsJPG(
  accessories: AccessoryItem[],
  gridParam: string // "grid3", "grid4", "grid2-2", "grid3-", "grid4-"
): Promise<string> {
  const layout = parseGridLayout(gridParam, accessories.length);
  const cacheKey = generateCacheKey(accessories, gridParam);
  const filename = `accessory-grid-${cacheKey}.jpg`;
  const storagePath = `ai-cache/accessory-grids/${filename}`;
  
  // Check if already exists in cache
  const fileStorage = await getFileStorageAdapter();
  
  // Build URL first to check if file exists
  const cachedUrl = fileStorage.getUrl(storagePath);
  const cached = await fileStorage.exists(cachedUrl);
  
  if (cached) {
    console.log(`✅ Cache hit for accessory grid: ${filename}`);
    console.log(`   Returning cached URL: ${cachedUrl}`);
    return cachedUrl;
  }
  
  console.log(`🎨 Rendering new accessory grid: ${filename} (${layout.cols}×${layout.rows})`);
  
  // Generate HTML
  const html = generateGridHTML(accessories, layout);
  
  // Launch Puppeteer and render to JPG
  // Try to find Chromium executable in common locations
  const { execSync } = await import('child_process');
  let chromiumPath: string | undefined;
  
  try {
    // Try to find chromium-browser using 'which' command
    chromiumPath = execSync('which chromium-browser 2>/dev/null || which chromium 2>/dev/null || which google-chrome 2>/dev/null', { encoding: 'utf-8' }).trim();
  } catch (error) {
    // If 'which' fails, try common Nix store paths
    const nixStorePaths = [
      '/nix/store/*/bin/chromium-browser',
      '/nix/store/*/bin/chromium'
    ];
    
    for (const pattern of nixStorePaths) {
      try {
        const paths = execSync(`ls ${pattern} 2>/dev/null | head -1`, { encoding: 'utf-8' }).trim();
        if (paths) {
          chromiumPath = paths;
          break;
        }
      } catch {}
    }
  }
  
  console.log(`🌐 Using Chromium executable: ${chromiumPath || 'bundled'}`);
  
  // Create unique temp dir for this browser instance
  const tempDir = `/tmp/chrome-${Date.now()}-${Math.random().toString(36).slice(2, 8)}`;
  
  const browser = await puppeteer.launch({
    headless: true,
    ...(chromiumPath ? { executablePath: chromiumPath } : {}),
    args: [
      '--no-sandbox',
      '--disable-setuid-sandbox',
      '--disable-dev-shm-usage',
      '--disable-gpu',
      '--disable-web-security',
      '--disable-software-rasterizer',
      '--disable-extensions',
      '--no-first-run',
      '--no-zygote',
      '--single-process',
      '--disable-background-networking',
      '--disable-background-timer-throttling',
      '--disable-backgrounding-occluded-windows',
      '--disable-breakpad',
      '--disable-component-extensions-with-background-pages',
      '--disable-features=TranslateUI,AudioServiceOutOfProcess',
      '--disable-ipc-flooding-protection',
      '--disable-renderer-backgrounding',
      '--disable-dbus',
      '--disable-features=IsolateOrigins,site-per-process',
      '--disable-blink-features=AutomationControlled',
      '--disable-hang-monitor',
      '--disable-prompt-on-repost',
      '--disable-sync',
      '--force-color-profile=srgb',
      '--metrics-recording-only',
      '--safebrowsing-disable-auto-update',
      '--password-store=basic',
      '--use-mock-keychain',
      `--user-data-dir=${tempDir}`,
      `--disk-cache-dir=${tempDir}/cache`
    ],
    // Reduce memory usage
    dumpio: false,
    timeout: 60000,
    protocolTimeout: 180000 // 3 minutes for protocol operations
  });
  
  try {
    const page = await browser.newPage();
    
    // Set viewport based on layout
    const width = layout.cols * 280 + 80; // 280px per tile + padding
    const height = layout.rows * 350 + 80; // 350px per tile + padding
    await page.setViewport({ width, height, deviceScaleFactor: 2 });
    
    // Calculate actual number of items being rendered
    const totalTiles = layout.infinite ? accessories.length : layout.cols * layout.rows;
    const itemsToRender = Math.min(accessories.length, totalTiles);
    
    // Adaptive timeout based on grid size
    const baseTimeout = 60000; // 1 minute base
    const timeoutPerItem = 2000; // 2 seconds per item
    const adaptiveTimeout = Math.min(baseTimeout + (itemsToRender * timeoutPerItem), 180000); // Max 3 minutes
    
    page.setDefaultNavigationTimeout(adaptiveTimeout);
    page.setDefaultTimeout(adaptiveTimeout);
    
    console.log(`⏱️  Setting ${Math.round(adaptiveTimeout/1000)}s timeout for grid with ${itemsToRender} items (${layout.cols}×${layout.rows}) out of ${accessories.length} available`);
    
    // Load HTML with domcontentloaded (faster than networkidle0)
    await page.setContent(html, { 
      waitUntil: 'domcontentloaded', // Changed from networkidle0 for faster loading
      timeout: 180000 
    });
    
    // Wait for images to load with individual timeout per image
    console.log(`🖼️  Waiting for ${itemsToRender} images to load (max 8s per image)...`);
    
    let imagesLoaded: boolean[] = [];
    try {
      // Set a reasonable timeout for the entire evaluate call
      const evaluateTimeout = Math.min(60000, itemsToRender * 8000); // Max 60s or 8s per image
      
      imagesLoaded = await Promise.race([
        page.evaluate(() => {
          const images = Array.from(document.images);
          const imagePromises = images.map((img, index) => {
            return new Promise<boolean>(resolve => {
              if (img.complete) {
                console.log(`Image ${index + 1} already loaded`);
                resolve(true);
              } else {
                // 8 second timeout per image (reduced from 10s)
                const timeout = setTimeout(() => {
                  console.warn(`Image ${index + 1} timed out after 8s: ${img.src}`);
                  resolve(false);
                }, 8000);
                
                img.onload = () => {
                  clearTimeout(timeout);
                  console.log(`Image ${index + 1} loaded successfully`);
                  resolve(true);
                };
                
                img.onerror = () => {
                  clearTimeout(timeout);
                  console.error(`Image ${index + 1} failed to load: ${img.src}`);
                  resolve(false);
                };
              }
            });
          });
          
          return Promise.all(imagePromises);
        }),
        new Promise<boolean[]>((_, reject) => 
          setTimeout(() => reject(new Error('Image loading timed out')), evaluateTimeout)
        )
      ]);
    } catch (error) {
      console.warn(`⚠️  Some images failed to load or timed out, continuing anyway...`);
      // Continue with screenshot even if images didn't fully load
      imagesLoaded = [];
    }
    
    const loadedCount = imagesLoaded.filter(Boolean).length;
    const totalCount = imagesLoaded.length;
    if (totalCount > 0) {
      console.log(`✅ Images loaded: ${loadedCount}/${totalCount} (${Math.round(loadedCount/totalCount*100)}%)`);
    } else {
      console.log(`⚠️  Proceeding to screenshot without waiting for all images`);
    }
    
    console.log(`📸 Taking screenshot of ${layout.cols}×${layout.rows} grid...`);
    
    // Screenshot as JPG
    const screenshot = await page.screenshot({
      type: 'jpeg',
      quality: 90,
      fullPage: true
    });
    
    // Upload to SFTP
    console.log(`📤 Uploading JPG to SFTP: ${filename}`);
    const uploadedUrl = await fileStorage.upload({
      filename,
      buffer: Buffer.from(screenshot),
      mimetype: 'image/jpeg',
      subfolder: 'ai-cache/accessory-grids',
      metadata: {
        gridParam,
        accessoryCodes: accessories.map(a => a.code).join(','),
        timestamp: new Date().toISOString()
      }
    });
    
    console.log(`✅ Accessory grid rendered and uploaded: ${uploadedUrl}`);
    
    // Close browser after successful upload
    console.log(`🚀 Closing browser...`);
    await browser.close();
    
    // Cleanup temp directory
    try {
      const fs = await import('fs/promises');
      await fs.rm(tempDir, { recursive: true, force: true });
      console.log(`🧹 Cleaned up temp dir: ${tempDir}`);
    } catch (cleanupError) {
      console.warn(`⚠️  Failed to cleanup temp dir:`, cleanupError);
    }
    
    return uploadedUrl;
    
  } catch (error) {
    console.error(`❌ Error rendering grid JPG:`, error);
    // Close browser if it's still open
    if (browser && browser.process()) {
      try {
        console.log(`🚀 Closing browser after error...`);
        await browser.close();
      } catch (closeError) {
        console.error(`⚠️  Failed to close browser:`, closeError);
      }
    }
    
    // Cleanup temp directory on error
    try {
      const fs = await import('fs/promises');
      await fs.rm(tempDir, { recursive: true, force: true });
      console.log(`🧹 Cleaned up temp dir after error: ${tempDir}`);
    } catch (cleanupError) {
      console.warn(`⚠️  Failed to cleanup temp dir:`, cleanupError);
    }
    
    // Provide more detailed error message
    const errorMessage = error instanceof Error ? error.message : String(error);
    throw new Error(`Failed to render accessory grid JPG for ${gridParam}: ${errorMessage}`);
  }
}
