import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Link } from "wouter";
import { Plus, Edit2, Trash2, Loader2, FileText, FolderOpen, ChevronRight, Eye, Copy } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import { Form, FormControl, FormDescription, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Switch } from "@/components/ui/switch";
import { useToast } from "@/hooks/use-toast";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { RichTextEditor } from "@/components/rich-text-editor";
import { renderTemplate, createSampleData } from "@/lib/template-renderer";
import { TemplateVariablesHelp } from "@/components/template-variables-help";
import { AvailableTagsPanel } from "@/components/template/available-tags-panel";
import { ExamplesModal } from "@/components/template/examples-modal";

// Types dla kategorii
interface TemplateCategory {
  id: number;
  name: string;
  description: string | null;
  sortOrder: number;
  createdAt: string;
}

// Types dla szablonów
interface DescriptionTemplate {
  id: number;
  categoryId: number | null;
  categoryName: string | null;
  name: string;
  templateType: string;
  contentDoc: Record<string, any> | null;
  htmlContent: string;
  variables: Record<string, any> | null;
  isGlobal: boolean;
  isActive: boolean;
  createdBy: number | null;
  createdAt: string;
  updatedAt: string;
}

// Schema dla kategorii
const categoryFormSchema = z.object({
  name: z.string().min(1, "Nazwa jest wymagana"),
  description: z.string().optional(),
  sortOrder: z.number().min(0).default(0),
});

type CategoryFormData = z.infer<typeof categoryFormSchema>;

// Schema dla szablonu
const templateFormSchema = z.object({
  name: z.string().min(1, "Nazwa jest wymagana"),
  categoryId: z.number().optional().nullable(),
  templateType: z.string().min(1, "Typ szablonu jest wymagany"),
  htmlContent: z.string().min(1, "Treść HTML jest wymagana"),
  variables: z.string().optional().refine((val) => {
    if (!val || val.trim() === "") return true; // Empty is OK
    try {
      JSON.parse(val);
      return true;
    } catch {
      return false;
    }
  }, {
    message: "Niepoprawny format JSON. Wprowadź poprawny obiekt JSON, np. {\"klucz\": \"wartość\"}",
  }),
  isGlobal: z.boolean().default(false),
  isActive: z.boolean().default(true),
});

type TemplateFormData = z.infer<typeof templateFormSchema>;

export default function TemplatesManager() {
  const { toast } = useToast();
  const [selectedCategory, setSelectedCategory] = useState<number | null>(null);
  const [isCategoryDialogOpen, setIsCategoryDialogOpen] = useState(false);
  const [isTemplateDialogOpen, setIsTemplateDialogOpen] = useState(false);
  const [isPreviewDialogOpen, setIsPreviewDialogOpen] = useState(false);
  const [isExamplesModalOpen, setIsExamplesModalOpen] = useState(false);
  const [editingCategory, setEditingCategory] = useState<TemplateCategory | null>(null);
  const [editingTemplate, setEditingTemplate] = useState<DescriptionTemplate | null>(null);
  const [previewTemplate, setPreviewTemplate] = useState<DescriptionTemplate | null>(null);
  const [contentDocJson, setContentDocJson] = useState<any>({});
  const [editorInstance, setEditorInstance] = useState<any>(null);

  // Fetch categories
  const { data: categories, isLoading: categoriesLoading } = useQuery<TemplateCategory[]>({
    queryKey: ["/api/template-categories"],
  });

  // Fetch templates
  const { data: templates, isLoading: templatesLoading } = useQuery<DescriptionTemplate[]>({
    queryKey: selectedCategory 
      ? ["/api/description-templates", selectedCategory] 
      : ["/api/description-templates"],
  });

  // Category CRUD
  const createCategoryMutation = useMutation({
    mutationFn: async (data: CategoryFormData) => {
      return await apiRequest("POST", "/api/template-categories", data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/template-categories"] });
      toast({ title: "Kategoria utworzona pomyślnie" });
      setIsCategoryDialogOpen(false);
      categoryForm.reset();
    },
    onError: () => {
      toast({ title: "Błąd tworzenia kategorii", variant: "destructive" });
    },
  });

  const updateCategoryMutation = useMutation({
    mutationFn: async ({ id, data }: { id: number; data: CategoryFormData }) => {
      return await apiRequest("PUT", `/api/template-categories/${id}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/template-categories"] });
      toast({ title: "Kategoria zaktualizowana" });
      setIsCategoryDialogOpen(false);
      setEditingCategory(null);
      categoryForm.reset();
    },
    onError: () => {
      toast({ title: "Błąd aktualizacji kategorii", variant: "destructive" });
    },
  });

  const deleteCategoryMutation = useMutation({
    mutationFn: async (id: number) => {
      return await apiRequest("DELETE", `/api/template-categories/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/template-categories"] });
      toast({ title: "Kategoria usunięta" });
    },
    onError: () => {
      toast({ title: "Błąd usuwania kategorii", variant: "destructive" });
    },
  });

  // Template CRUD
  const createTemplateMutation = useMutation({
    mutationFn: async (data: TemplateFormData) => {
      let parsedVariables = {};
      try {
        parsedVariables = data.variables && data.variables.trim() !== "" 
          ? JSON.parse(data.variables) 
          : {};
      } catch (error) {
        throw new Error("Niepoprawny format JSON w zmiennych");
      }
      
      const payload = {
        name: data.name,
        categoryId: data.categoryId,
        templateType: data.templateType,
        htmlContent: data.htmlContent,
        contentDoc: contentDocJson || {}, // Tiptap JSON output
        variables: parsedVariables,
        isGlobal: data.isGlobal,
        isActive: data.isActive,
      };
      return await apiRequest("POST", "/api/description-templates", payload);
    },
    onSuccess: () => {
      // Invalidate all templates queries (with and without category filter)
      queryClient.invalidateQueries({ 
        predicate: (query) => query.queryKey[0] === "/api/description-templates"
      });
      toast({ title: "Szablon utworzony pomyślnie" });
      setIsTemplateDialogOpen(false);
      setContentDocJson({});
      templateForm.reset();
    },
    onError: (error: any) => {
      const message = error?.message || "Błąd tworzenia szablonu";
      toast({ title: message, variant: "destructive" });
    },
  });

  const updateTemplateMutation = useMutation({
    mutationFn: async ({ id, data }: { id: number; data: TemplateFormData }) => {
      let parsedVariables = {};
      try {
        parsedVariables = data.variables && data.variables.trim() !== "" 
          ? JSON.parse(data.variables) 
          : {};
      } catch (error) {
        throw new Error("Niepoprawny format JSON w zmiennych");
      }
      
      const payload = {
        name: data.name,
        categoryId: data.categoryId,
        templateType: data.templateType,
        htmlContent: data.htmlContent,
        contentDoc: contentDocJson || {}, // Tiptap JSON output
        variables: parsedVariables,
        isGlobal: data.isGlobal,
        isActive: data.isActive,
      };
      return await apiRequest("PUT", `/api/description-templates/${id}`, payload);
    },
    onSuccess: () => {
      // Invalidate all templates queries (with and without category filter)
      queryClient.invalidateQueries({ 
        predicate: (query) => query.queryKey[0] === "/api/description-templates"
      });
      toast({ title: "Szablon zaktualizowany" });
      setIsTemplateDialogOpen(false);
      setEditingTemplate(null);
      setContentDocJson({});
      templateForm.reset();
    },
    onError: (error: any) => {
      const message = error?.message || "Błąd aktualizacji szablonu";
      toast({ title: message, variant: "destructive" });
    },
  });

  const deleteTemplateMutation = useMutation({
    mutationFn: async (id: number) => {
      return await apiRequest("DELETE", `/api/description-templates/${id}`);
    },
    onSuccess: () => {
      // Invalidate all templates queries (with and without category filter)
      queryClient.invalidateQueries({ 
        predicate: (query) => query.queryKey[0] === "/api/description-templates"
      });
      toast({ title: "Szablon usunięty" });
    },
    onError: () => {
      toast({ title: "Błąd usuwania szablonu", variant: "destructive" });
    },
  });

  const duplicateTemplateMutation = useMutation({
    mutationFn: async (id: number) => {
      return await apiRequest("POST", `/api/description-templates/${id}/duplicate`);
    },
    onSuccess: () => {
      // Invalidate all templates queries (with and without category filter)
      queryClient.invalidateQueries({ 
        predicate: (query) => query.queryKey[0] === "/api/description-templates"
      });
      toast({ title: "Szablon zduplikowany pomyślnie" });
    },
    onError: () => {
      toast({ title: "Błąd duplikowania szablonu", variant: "destructive" });
    },
  });

  // Forms
  const categoryForm = useForm<CategoryFormData>({
    resolver: zodResolver(categoryFormSchema),
    defaultValues: {
      name: "",
      description: "",
      sortOrder: 0,
    },
  });

  const templateForm = useForm<TemplateFormData>({
    resolver: zodResolver(templateFormSchema),
    defaultValues: {
      name: "",
      categoryId: null,
      templateType: "product_description",
      htmlContent: "",
      variables: "{}",
      isGlobal: false,
      isActive: true,
    },
  });

  const onCategorySubmit = (data: CategoryFormData) => {
    if (editingCategory) {
      updateCategoryMutation.mutate({ id: editingCategory.id, data });
    } else {
      createCategoryMutation.mutate(data);
    }
  };

  const onTemplateSubmit = (data: TemplateFormData) => {
    if (editingTemplate) {
      updateTemplateMutation.mutate({ id: editingTemplate.id, data });
    } else {
      createTemplateMutation.mutate(data);
    }
  };

  const handleEditCategory = (category: TemplateCategory) => {
    setEditingCategory(category);
    categoryForm.reset({
      name: category.name,
      description: category.description || "",
      sortOrder: category.sortOrder,
    });
    setIsCategoryDialogOpen(true);
  };

  const handleEditTemplate = (template: DescriptionTemplate) => {
    setEditingTemplate(template);
    setContentDocJson(template.contentDoc || {});
    templateForm.reset({
      name: template.name,
      categoryId: template.categoryId,
      templateType: template.templateType,
      htmlContent: template.htmlContent,
      variables: template.variables ? JSON.stringify(template.variables, null, 2) : "{}",
      isGlobal: template.isGlobal,
      isActive: template.isActive,
    });
    setIsTemplateDialogOpen(true);
  };

  const handleNewCategory = () => {
    setEditingCategory(null);
    categoryForm.reset();
    setIsCategoryDialogOpen(true);
  };

  const handleNewTemplate = () => {
    setEditingTemplate(null);
    setContentDocJson({});
    templateForm.reset({
      name: "",
      categoryId: selectedCategory || undefined,
      templateType: "product_description",
      htmlContent: "",
      variables: "{}",
      isGlobal: false,
      isActive: true,
    });
    setIsTemplateDialogOpen(true);
  };

  const handlePreviewTemplate = (template: DescriptionTemplate) => {
    setPreviewTemplate(template);
    setIsPreviewDialogOpen(true);
  };

  return (
    <div className="flex h-full min-h-0">
      {/* Left Panel - Categories */}
      <div className="w-80 border-r flex flex-col min-h-0">
        <div className="p-6 border-b flex-shrink-0">
          <div className="flex items-center justify-between mb-4">
            <div className="flex items-center gap-2">
              <FolderOpen className="h-5 w-5" />
              <h2 className="text-lg font-semibold">Kategorie</h2>
            </div>
            <Dialog open={isCategoryDialogOpen} onOpenChange={setIsCategoryDialogOpen}>
              <DialogTrigger asChild>
                <Button size="sm" onClick={handleNewCategory} data-testid="button-add-category">
                  <Plus className="h-4 w-4 mr-2" />
                  Dodaj
                </Button>
              </DialogTrigger>
              <DialogContent data-testid="dialog-category-form">
                <DialogHeader>
                  <DialogTitle>
                    {editingCategory ? "Edytuj kategorię" : "Nowa kategoria"}
                  </DialogTitle>
                  <DialogDescription>
                    {editingCategory
                      ? "Zaktualizuj informacje o kategorii"
                      : "Stwórz nową kategorię szablonów"}
                  </DialogDescription>
                </DialogHeader>
                <Form {...categoryForm}>
                  <form onSubmit={categoryForm.handleSubmit(onCategorySubmit)} className="space-y-4">
                    <FormField
                      control={categoryForm.control}
                      name="name"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Nazwa</FormLabel>
                          <FormControl>
                            <Input {...field} placeholder="Nazwa kategorii" data-testid="input-category-name" />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                    <FormField
                      control={categoryForm.control}
                      name="description"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Opis</FormLabel>
                          <FormControl>
                            <Textarea {...field} placeholder="Opis kategorii" data-testid="input-category-description" />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                    <FormField
                      control={categoryForm.control}
                      name="sortOrder"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Kolejność sortowania</FormLabel>
                          <FormControl>
                            <Input 
                              type="number" 
                              {...field} 
                              onChange={(e) => field.onChange(parseInt(e.target.value) || 0)}
                              data-testid="input-category-sort-order"
                            />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                    <div className="flex justify-end gap-2">
                      <Button
                        type="button"
                        variant="outline"
                        onClick={() => {
                          setIsCategoryDialogOpen(false);
                          setEditingCategory(null);
                          categoryForm.reset();
                        }}
                        data-testid="button-cancel-category"
                      >
                        Anuluj
                      </Button>
                      <Button
                        type="submit"
                        disabled={createCategoryMutation.isPending || updateCategoryMutation.isPending}
                        data-testid="button-save-category"
                      >
                        {createCategoryMutation.isPending || updateCategoryMutation.isPending ? (
                          <>
                            <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                            Zapisywanie...
                          </>
                        ) : (
                          "Zapisz"
                        )}
                      </Button>
                    </div>
                  </form>
                </Form>
              </DialogContent>
            </Dialog>
          </div>
        </div>

        {/* Categories List */}
        <div className="flex-1 overflow-auto">
          {categoriesLoading ? (
            <div className="flex items-center justify-center p-8">
              <Loader2 className="h-6 w-6 animate-spin" />
            </div>
          ) : (
            <div className="p-4 space-y-2">
              <Button
                variant={selectedCategory === null ? "default" : "ghost"}
                className="w-full justify-start"
                onClick={() => setSelectedCategory(null)}
                data-testid="button-category-all"
              >
                <FileText className="h-4 w-4 mr-2" />
                Wszystkie szablony
                <Badge variant="secondary" className="ml-auto">
                  {templates?.length || 0}
                </Badge>
              </Button>
              {categories?.map((category) => (
                <div key={category.id} className="group relative">
                  <Button
                    variant={selectedCategory === category.id ? "default" : "ghost"}
                    className="w-full justify-start pr-16"
                    onClick={() => setSelectedCategory(category.id)}
                    data-testid={`button-category-${category.id}`}
                  >
                    <ChevronRight className="h-4 w-4 mr-2" />
                    {category.name}
                  </Button>
                  <div className="absolute right-2 top-1/2 -translate-y-1/2 flex gap-1 opacity-0 group-hover:opacity-100 transition-opacity">
                    <Button
                      size="icon"
                      variant="ghost"
                      className="h-7 w-7"
                      onClick={(e) => {
                        e.stopPropagation();
                        handleEditCategory(category);
                      }}
                      data-testid={`button-edit-category-${category.id}`}
                    >
                      <Edit2 className="h-3 w-3" />
                    </Button>
                    <Button
                      size="icon"
                      variant="ghost"
                      className="h-7 w-7"
                      onClick={(e) => {
                        e.stopPropagation();
                        if (confirm("Czy na pewno chcesz usunąć tę kategorię?")) {
                          deleteCategoryMutation.mutate(category.id);
                        }
                      }}
                      data-testid={`button-delete-category-${category.id}`}
                    >
                      <Trash2 className="h-3 w-3" />
                    </Button>
                  </div>
                </div>
              ))}
            </div>
          )}
        </div>
      </div>

      {/* Right Panel - Templates */}
      <div className="flex-1 flex flex-col min-h-0">
        <div className="p-6 border-b flex-shrink-0">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-2xl font-bold">Szablony opisów produktów</h1>
              <p className="text-muted-foreground">
                Zarządzaj szablonami opisów z zmiennymi i kategoriami
              </p>
            </div>
            <Link href="/template-editor/new">
              <Button data-testid="button-add-template">
                <Plus className="h-4 w-4 mr-2" />
                Nowy szablon
              </Button>
            </Link>
          </div>
        </div>

        {/* Templates List */}
        <div className="flex-1 overflow-auto p-6">
          {templatesLoading ? (
            <div className="flex items-center justify-center p-12">
              <Loader2 className="h-8 w-8 animate-spin" />
            </div>
          ) : templates && templates.length > 0 ? (
            <Card>
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Nazwa</TableHead>
                    <TableHead>Typ</TableHead>
                    <TableHead>Kategoria</TableHead>
                    <TableHead>Status</TableHead>
                    <TableHead className="text-right">Akcje</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {templates.map((template) => (
                    <TableRow key={template.id} data-testid={`row-template-${template.id}`}>
                      <TableCell className="font-medium">{template.name}</TableCell>
                      <TableCell>
                        <Badge variant="outline">{template.templateType}</Badge>
                      </TableCell>
                      <TableCell>
                        {template.categoryName ? (
                          <Badge variant="secondary">{template.categoryName}</Badge>
                        ) : (
                          <span className="text-muted-foreground text-sm">Brak kategorii</span>
                        )}
                      </TableCell>
                      <TableCell>
                        <div className="flex gap-2">
                          {template.isActive ? (
                            <Badge variant="default">Aktywny</Badge>
                          ) : (
                            <Badge variant="secondary">Nieaktywny</Badge>
                          )}
                          {template.isGlobal && <Badge variant="outline">Globalny</Badge>}
                        </div>
                      </TableCell>
                      <TableCell className="text-right">
                        <div className="flex justify-end gap-2">
                          <Button
                            size="sm"
                            variant="ghost"
                            onClick={() => handlePreviewTemplate(template)}
                            data-testid={`button-preview-template-${template.id}`}
                          >
                            <Eye className="h-4 w-4" />
                          </Button>
                          <Link href={`/template-editor/${template.id}`}>
                            <Button
                              size="sm"
                              variant="ghost"
                              data-testid={`button-edit-template-${template.id}`}
                            >
                              <Edit2 className="h-4 w-4" />
                            </Button>
                          </Link>
                          <Button
                            size="sm"
                            variant="ghost"
                            onClick={() => duplicateTemplateMutation.mutate(template.id)}
                            disabled={duplicateTemplateMutation.isPending}
                            data-testid={`button-duplicate-template-${template.id}`}
                          >
                            <Copy className="h-4 w-4" />
                          </Button>
                          <Button
                            size="sm"
                            variant="ghost"
                            onClick={() => {
                              if (confirm("Czy na pewno chcesz usunąć ten szablon?")) {
                                deleteTemplateMutation.mutate(template.id);
                              }
                            }}
                            data-testid={`button-delete-template-${template.id}`}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </Card>
          ) : (
            <Card>
              <CardContent className="flex flex-col items-center justify-center p-12">
                <FileText className="h-12 w-12 text-muted-foreground mb-4" />
                <p className="text-lg font-medium mb-2">Brak szablonów</p>
                <p className="text-muted-foreground text-center mb-4">
                  {selectedCategory
                    ? "Nie znaleziono szablonów w tej kategorii. Stwórz nowy szablon."
                    : "Nie znaleziono żadnych szablonów. Stwórz pierwszy szablon."}
                </p>
                <Link href="/template-editor/new">
                  <Button data-testid="button-add-first-template">
                    <Plus className="h-4 w-4 mr-2" />
                    Dodaj szablon
                  </Button>
                </Link>
              </CardContent>
            </Card>
          )}
        </div>
      </div>

      {/* Preview Dialog */}
      <Dialog open={isPreviewDialogOpen} onOpenChange={setIsPreviewDialogOpen}>
        <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto" data-testid="dialog-template-preview">
          <DialogHeader>
            <DialogTitle>Podgląd szablonu: {previewTemplate?.name}</DialogTitle>
            <DialogDescription>
              Podgląd renderowanego szablonu z przykładowymi danymi
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            {/* Rendered Preview */}
            <div className="border rounded-lg p-6 bg-background">
              <div 
                className="prose prose-sm dark:prose-invert max-w-none"
                dangerouslySetInnerHTML={{
                  __html: previewTemplate 
                    ? renderTemplate(
                        previewTemplate.htmlContent, 
                        previewTemplate.variables || createSampleData()
                      )
                    : ""
                }}
                data-testid="preview-content"
              />
            </div>

            {/* Sample Data Used */}
            <div className="border rounded-lg p-4 bg-muted/50">
              <p className="text-sm font-medium mb-2">Użyte dane (variables):</p>
              <pre className="text-xs overflow-auto max-h-40 bg-background p-3 rounded">
                {JSON.stringify(
                  previewTemplate?.variables || createSampleData(), 
                  null, 
                  2
                )}
              </pre>
            </div>
          </div>
          <div className="flex justify-end">
            <Button
              variant="outline"
              onClick={() => setIsPreviewDialogOpen(false)}
              data-testid="button-close-preview"
            >
              Zamknij
            </Button>
          </div>
        </DialogContent>
      </Dialog>
    </div>
  );
}
