import { useRoute, useLocation, Link } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { ArrowLeft, ArrowRight, Package, User, MapPin, CreditCard, Truck, FileText, Calendar, Copy, Check, X, Banknote, Receipt, ExternalLink, ChevronLeft, ChevronRight, ShoppingCart, Box, RefreshCw, PackageX, MessageSquare, Upload } from "lucide-react";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Separator } from "@/components/ui/separator";
import { StatusBadge } from "@/components/status-badge";
import { useToast } from "@/hooks/use-toast";
import {
  Tooltip,
  TooltipContent,
  TooltipProvider,
  TooltipTrigger,
} from "@/components/ui/tooltip";
import {
  Tabs,
  TabsContent,
  TabsList,
  TabsTrigger,
} from "@/components/ui/tabs";
import type { AllegroOrder, OrderStatus, FulfillmentStatus } from "@shared/schema";
import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { useState, useEffect } from "react";

export default function OrderDetails() {
  const [, params] = useRoute("/order/:orderNumber");
  const [, navigate] = useLocation();
  const orderNumber = params?.orderNumber;
  const [copied, setCopied] = useState(false);
  const { toast } = useToast();

  // Scroll na górę strony po załadowaniu szczegółów zamówienia
  useEffect(() => {
    window.scrollTo({ top: 0, behavior: "smooth" });
  }, [orderNumber]);

  const { data: rawData, isLoading, error } = useQuery<any>({
    queryKey: [`/api/orders/by-number/${orderNumber}`],
    enabled: !!orderNumber,
  });

  const { data: allOrders = [], isLoading: isLoadingOrders } = useQuery<any[]>({
    queryKey: ["/api/orders", { sortBy: "order_number", sortOrder: "ASC", limit: 10000 }],
  });

  const { data: orderChanges = [], isLoading: changesLoading } = useQuery<Array<{
    id: string;
    order_id: string;
    order_number: string;
    source: string;
    change_type: string;
    field_changed: string;
    old_value: string;
    new_value: string;
    detected_at: string;
  }>>({
    queryKey: rawData?.orderNumber ? [`/api/order-changes?orderNumber=${rawData.orderNumber}`] : [],
    enabled: !!rawData?.orderNumber,
  });

  const { data: odooConfig } = useQuery<{
    configured: boolean;
    url?: string;
    user_url?: string;
  }>({
    queryKey: ["/api/odoo/config"],
  });

  const fieldLabels: Record<string, string> = {
    status: "Status zamówienia",
    payment_status: "Status płatności",
    payment_amount: "Kwota płatności",
    refund_amount: "Kwota zwrotu",
    has_returns: "Zwroty produktów",
    tracking_numbers: "Numery przesyłek",
    buyer_address: "Adres kupującego",
    buyer_city: "Miasto",
    buyer_zip: "Kod pocztowy",
    buyer_first_name: "Imię",
    buyer_last_name: "Nazwisko",
    buyer_phone: "Telefon",
    buyer_email: "Email",
  };

  // Mapowanie statusów płatności na polskie nazwy
  const getPaymentStatusLabel = (status: string | undefined): string => {
    if (!status) return 'Nieznany';
    const statusMap: Record<string, string> = {
      'PAID': 'Zapłacone',
      'PENDING': 'Oczekuje na płatność',
      'UNPAID': 'Nieopłacone',
      'CANCELLED': 'Anulowane',
      'REFUNDED': 'Zwrócone'
    };
    return statusMap[status] || status;
  };

  const copyOrderNumber = async () => {
    if (!rawData?.orderNumber) return;
    
    try {
      await navigator.clipboard.writeText(String(rawData.orderNumber));
      setCopied(true);
      toast({
        title: "Skopiowano!",
        description: `Numer zamówienia #${rawData.orderNumber} został skopiowany do schowka.`,
      });
      setTimeout(() => setCopied(false), 2000);
    } catch (err) {
      toast({
        title: "Błąd",
        description: "Nie udało się skopiować numeru zamówienia.",
        variant: "destructive",
      });
    }
  };

  const getOdooOrderUrl = (odooOrderId: number): string | null => {
    // Użyj user_url dla linków klikanych przez użytkowników, fallback na url
    const odooUserUrl = odooConfig?.user_url || odooConfig?.url;
    if (!odooUserUrl || !odooOrderId) return null;
    // Remove trailing slash if exists
    const baseUrl = odooUserUrl.replace(/\/$/, '');
    return `${baseUrl}/sales/${odooOrderId}`;
  };

  const translateAllegroStatus = (status: string): string => {
    const statusMap: { [key: string]: string } = {
      'NEW': 'Nowe',
      'PROCESSING': 'W trakcie przetwarzania',
      'READY_FOR_PROCESSING': 'W realizacji',
      'SENT': 'Wysłane',
      'CANCELLED': 'Anulowane',
      'RETURNED': 'Zwrócone',
      'READY_FOR_SHIPMENT': 'Gotowe do wysyłki',
      'FINISHED': 'Zakończone'
    };
    return statusMap[status] || status;
  };

  const translateShoperStatus = (statusId: string | number): string => {
    const statusMap: { [key: string]: string } = {
      '1': 'Złożone',
      '2': 'Przyjęte do realizacji',
      '3': 'Oczekiwanie na dostawę',
      '4': 'W trakcie kompletowania',
      '5': 'Oczekiwanie na płatność',
      '6': 'Gotowe do wysłania',
      '7': 'Przesyłka wysłana',
      '8': 'Anulowane',
      '9': 'Odrzucone',
      '11': 'Zwrócone'
    };
    return statusMap[statusId?.toString()] || `Status ${statusId}`;
  };

  const getShoperStatusVariant = (statusId: string | number): "default" | "secondary" | "destructive" | "outline" => {
    const id = statusId?.toString();
    if (id === '7') return 'default'; // Wysłane - zielony
    if (id === '8' || id === '9') return 'destructive'; // Anulowane/Odrzucone - czerwony
    if (id === '11') return 'destructive'; // Zwrócone - czerwony
    if (id === '6') return 'default'; // Gotowe do wysłania - zielony
    return 'secondary'; // Pozostałe - szary
  };

  const refreshOrderMutation = useMutation({
    mutationFn: async () => {
      if (!rawData?.id) throw new Error("No order ID");
      const res = await apiRequest("POST", `/api/orders/${rawData.id}/refresh`);
      return await res.json();
    },
    onSuccess: (data: { success: boolean; platform: string }) => {
      queryClient.invalidateQueries({ queryKey: [`/api/orders/by-number/${orderNumber}`] });
      queryClient.invalidateQueries({ queryKey: ["/api/orders"] });
      queryClient.invalidateQueries({ queryKey: ["/api/orders/statistics"] });
      queryClient.invalidateQueries({ queryKey: ["/api/orders/chart"] });
      toast({
        title: "Sukces",
        description: `Zamówienie odświeżone z platformy ${data.platform}`,
      });
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się odświeżyć zamówienia",
        variant: "destructive",
      });
    }
  });

  const syncToOdooMutation = useMutation({
    mutationFn: async () => {
      if (!rawData?.orderNumber) throw new Error("No order number");
      const res = await apiRequest("POST", `/api/orders/${rawData.orderNumber}/sync-to-odoo`);
      return await res.json();
    },
    onSuccess: (data: { success: boolean; orderNumber: number; operation: string; message: string }) => {
      queryClient.invalidateQueries({ queryKey: [`/api/orders/by-number/${orderNumber}`] });
      queryClient.invalidateQueries({ queryKey: ["/api/orders"] });
      toast({
        title: "Dodano do kolejki",
        description: data.message,
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się dodać do kolejki Odoo",
        variant: "destructive",
      });
    }
  });

  const currentOrderNumber = rawData?.orderNumber ? String(rawData.orderNumber) : "0";
  const currentOrderIndex = allOrders.findIndex((o: any) => {
    return String(o.orderNumber) === currentOrderNumber;
  });
  
  const hasPrevious = currentOrderIndex > 0 && allOrders.length > 0;
  const hasNext = currentOrderIndex >= 0 && currentOrderIndex < allOrders.length - 1 && allOrders.length > 0;

  const navigateToPrevious = () => {
    if (hasPrevious && currentOrderIndex > 0) {
      const prevOrder = allOrders[currentOrderIndex - 1];
      navigate(`/order/${prevOrder.orderNumber || prevOrder.id}`);
    }
  };

  const navigateToNext = () => {
    if (hasNext && currentOrderIndex < allOrders.length - 1) {
      const nextOrder = allOrders[currentOrderIndex + 1];
      navigate(`/order/${nextOrder.orderNumber || nextOrder.id}`);
    }
  };

  const { data: shoperConfig } = useQuery<{
    orders_url?: string;
    products_url?: string;
  }>({
    queryKey: ["/api/shoper/config"],
  });

  const getOrderLink = (source: string, sourceOrderId: string) => {
    if (source === 'ALLEGRO') {
      return `https://allegro.pl/moje-allegro/sprzedaz/zamowienia/${sourceOrderId}`;
    } else {
      const baseUrl = (shoperConfig?.orders_url || 'https://alpmeb.pl/admin/orders/').replace(/\/$/, '');
      return `${baseUrl}/${sourceOrderId}`;
    }
  };

  const getProductLink = (source: string, offerId: string) => {
    if (source === 'ALLEGRO') {
      return `https://allegro.pl/oferta/${offerId}`;
    } else {
      const baseUrl = (shoperConfig?.products_url || 'https://sklep378098.shoparena.pl/admin/products/edit/id/').replace(/\/$/, '');
      return `${baseUrl}/${offerId}`;
    }
  };

  if (isLoading) {
    return (
      <div className="flex-1 overflow-auto">
        <div className="max-w-7xl mx-auto px-6 py-8">
          <div className="h-8 w-48 bg-muted rounded animate-pulse mb-8" />
          <div className="space-y-4">
            {[1, 2, 3, 4].map((i) => (
              <div key={i} className="h-32 bg-muted rounded animate-pulse" />
            ))}
          </div>
        </div>
      </div>
    );
  }

  if (!rawData || error) {
    return (
      <div className="flex-1 overflow-auto">
        <div className="max-w-7xl mx-auto px-6 py-8">
          <Button
            variant="ghost"
            onClick={() => navigate("/")}
            className="mb-6"
            data-testid="button-back"
          >
            <ArrowLeft className="h-4 w-4" />
            Powrót do listy zamówień
          </Button>
          <div className="text-center py-12">
            <h2 className="text-2xl font-semibold mb-2">Nie znaleziono zamówienia</h2>
            <p className="text-muted-foreground">
              Zamówienie #{orderNumber} nie zostało znalezione.
            </p>
          </div>
        </div>
      </div>
    );
  }

  const buyer = rawData?.buyer || {};
  const deliveryRaw = rawData?.delivery || {};
  const delivery = {
    ...deliveryRaw,
    shipments: typeof deliveryRaw.shipments === 'string' 
      ? (deliveryRaw.shipments ? JSON.parse(deliveryRaw.shipments) : [])
      : (deliveryRaw.shipments || [])
  };
  const payment = rawData?.payment || {};
  const invoice = rawData?.invoice || {};
  const billingAddress = rawData?.billingAddress || {};
  const lineItems = rawData?.lineItems || [];
  const source = rawData?.source || 'ALLEGRO';
  const isCashOnDelivery = payment?.type === 'CASH_ON_DELIVERY';
  const totalAmount = parseFloat(rawData.summary?.totalToPay?.amount || "0");
  const paymentType = payment?.type || '—';
  const paymentProvider = payment?.provider || '—';
  const isPaid = payment?.paidAmount?.amount && parseFloat(payment.paidAmount.amount) > 0;
  const paidAmountValue = isPaid ? parseFloat(payment.paidAmount.amount) : 0;
  const hasReturns = rawData?.hasReturns || false;
  const refundAmount = parseFloat(payment?.refundAmount || "0");
  const hasRefund = refundAmount > 0;
  const netPaidAmount = paidAmountValue - refundAmount;
  
  const getPaymentColorClass = () => {
    if (hasRefund && netPaidAmount === 0) return 'text-red-600 dark:text-red-400';
    if (isPaid) return 'text-green-600 dark:text-green-400';
    if (isCashOnDelivery) return 'text-orange-600 dark:text-orange-400';
    return 'text-red-600 dark:text-red-400';
  };

  return (
    <div className="flex-1 overflow-y-auto">
      <div className="w-full px-6 py-8">
        <Button
          variant="ghost"
          onClick={() => navigate("/")}
          className="mb-6"
          data-testid="button-back"
        >
          <ArrowLeft className="h-4 w-4" />
          Powrót do listy zamówień
        </Button>

        {/* Nagłówek zamówienia - kompaktowy */}
        <div className="mb-6">
          <TooltipProvider>
            <div className="flex items-center justify-between gap-4 flex-wrap">
              {/* Lewa strona - Numer i ikonki */}
              <div className="flex items-center gap-3">
                {/* Numer zamówienia */}
                <Tooltip>
                  <TooltipTrigger asChild>
                    <Badge
                      className="cursor-pointer hover-elevate px-4 py-2 inline-flex items-center gap-2 bg-blue-500 hover:bg-blue-600 text-white text-2xl font-bold border-0"
                      onClick={copyOrderNumber}
                      data-testid="text-order-number"
                    >
                      <span>{rawData.orderNumber}</span>
                      {copied ? (
                        <Check className="h-5 w-5" />
                      ) : (
                        <Copy className="h-5 w-5" />
                      )}
                    </Badge>
                  </TooltipTrigger>
                  <TooltipContent>Kliknij aby skopiować numer zamówienia</TooltipContent>
                </Tooltip>

                {/* Imię i nazwisko kupującego */}
                {buyer.firstName && buyer.lastName && (
                  <Tooltip>
                    <TooltipTrigger asChild>
                      <Badge 
                        variant="outline"
                        className="gap-1"
                        data-testid="badge-buyer-name"
                      >
                        <User className="h-4 w-4" />
                        {buyer.firstName} {buyer.lastName}
                      </Badge>
                    </TooltipTrigger>
                    <TooltipContent>Kupujący</TooltipContent>
                  </Tooltip>
                )}

                {/* Data zamówienia */}
                {rawData.boughtAt && (
                  <Tooltip>
                    <TooltipTrigger asChild>
                      <Badge 
                        variant="outline"
                        className="gap-1"
                        data-testid="badge-order-date"
                      >
                        <Calendar className="h-4 w-4" />
                        {rawData.boughtAt 
                          ? format(new Date(rawData.boughtAt), "dd.MM.yyyy", { locale: pl })
                          : "—"
                        }
                      </Badge>
                    </TooltipTrigger>
                    <TooltipContent>
                      Data zamówienia: {rawData.boughtAt 
                        ? format(new Date(rawData.boughtAt), "dd MMMM yyyy, HH:mm", { locale: pl })
                        : "—"
                      }
                    </TooltipContent>
                  </Tooltip>
                )}

                {/* Źródło */}
                <Tooltip>
                  <TooltipTrigger asChild>
                    <a 
                      href={getOrderLink(source, source === 'ALLEGRO' ? rawData.allegroOrderId : rawData.shoperOrderId)}
                      target="_blank"
                      rel="noopener noreferrer"
                      className="inline-flex"
                      data-testid="badge-source"
                    >
                      <Badge 
                        variant="outline"
                        className="gap-1 cursor-pointer hover:bg-accent"
                      >
                        {source === 'ALLEGRO' ? (
                          <>
                            <span className="text-orange-500 font-bold text-base">A</span>
                            <span>llegro</span>
                          </>
                        ) : (
                          <>
                            <span className="text-blue-500 font-bold text-base">S</span>
                            <span>hoper</span>
                          </>
                        )}
                        <ExternalLink className="h-3 w-3" />
                      </Badge>
                    </a>
                  </TooltipTrigger>
                  <TooltipContent>Kliknij aby otworzyć zamówienie w {source === 'ALLEGRO' ? 'Allegro' : 'Shoper'}</TooltipContent>
                </Tooltip>

                {/* Numer Odoo */}
                {rawData?.odooOrderId && (
                  <Tooltip>
                    <TooltipTrigger asChild>
                      {getOdooOrderUrl(rawData.odooOrderId) ? (
                        <a 
                          href={getOdooOrderUrl(rawData.odooOrderId)!}
                          target="_blank"
                          rel="noopener noreferrer"
                          className="inline-flex"
                          data-testid="badge-odoo-order"
                        >
                          <Badge 
                            variant="outline"
                            className="gap-1 bg-purple-500/10 text-purple-600 dark:text-purple-400 border-purple-500/20 cursor-pointer hover:bg-purple-500/20"
                          >
                            <span className="font-semibold inline-flex items-center gap-1">
                              Odoo: #{rawData.odooOrderId}
                              <ExternalLink className="h-3 w-3" />
                            </span>
                          </Badge>
                        </a>
                      ) : (
                        <span className="inline-flex" data-testid="badge-odoo-order">
                          <Badge 
                            variant="outline"
                            className="gap-1 bg-purple-500/10 text-purple-600 dark:text-purple-400 border-purple-500/20"
                          >
                            <span className="font-semibold">Odoo: #{rawData.odooOrderId}</span>
                          </Badge>
                        </span>
                      )}
                    </TooltipTrigger>
                    <TooltipContent>
                      {getOdooOrderUrl(rawData.odooOrderId) 
                        ? "Kliknij aby otworzyć zamówienie w Odoo" 
                        : "Numer zamówienia w Odoo"}
                    </TooltipContent>
                  </Tooltip>
                )}

                {/* Status zamówienia */}
                {source === 'SHOPER' && rawData.status && (
                  <Tooltip>
                    <TooltipTrigger asChild>
                      <Badge 
                        variant="outline"
                        className={`gap-1 ${
                          rawData.status?.toString() === '7' 
                            ? 'bg-purple-500/10 text-purple-600 dark:text-purple-400 border-purple-500/20' 
                            : rawData.status?.toString() === '8' || rawData.status?.toString() === '9'
                              ? 'bg-red-500/10 text-red-600 dark:text-red-400 border-red-500/20'
                              : ''
                        }`}
                        data-testid="badge-order-status"
                      >
                        {translateShoperStatus(rawData.status)}
                      </Badge>
                    </TooltipTrigger>
                    <TooltipContent>Status zamówienia w Shoper</TooltipContent>
                  </Tooltip>
                )}
                {source === 'ALLEGRO' && rawData.status && (
                  <Tooltip>
                    <TooltipTrigger asChild>
                      <Badge 
                        variant="outline"
                        className={`gap-1 ${
                          rawData.status === 'Wysłane' || rawData.status === 'SENT' 
                            ? 'bg-purple-500/10 text-purple-600 dark:text-purple-400 border-purple-500/20' 
                            : rawData.status === 'Anulowane' || rawData.status === 'CANCELLED'
                              ? 'bg-red-500/10 text-red-600 dark:text-red-400 border-red-500/20'
                              : ''
                        }`}
                        data-testid="badge-order-status"
                      >
                        {translateAllegroStatus(rawData.status)}
                      </Badge>
                    </TooltipTrigger>
                    <TooltipContent>Status zamówienia w Allegro</TooltipContent>
                  </Tooltip>
                )}

                {/* Kupujący zgłosił zwrot */}
                {hasReturns && (
                  <Badge 
                    variant="destructive"
                    className="gap-1"
                    data-testid="badge-customer-return"
                  >
                    <PackageX className="h-3 w-3" />
                    Kupujący zgłosił zwrot
                  </Badge>
                )}

                {/* Typ płatności */}
                <Tooltip>
                  <TooltipTrigger asChild>
                    <Badge 
                      variant="outline"
                      className={`gap-1 ${
                        hasRefund && netPaidAmount === 0
                          ? 'bg-red-500/10 text-red-600 dark:text-red-400 border-red-500/20'
                          : isCashOnDelivery 
                            ? 'bg-orange-500/10 text-orange-600 dark:text-orange-400 border-orange-500/20'
                            : isPaid
                            ? 'bg-green-500/10 text-green-600 dark:text-green-400 border-green-500/20'
                            : 'bg-red-500/10 text-red-600 dark:text-red-400 border-red-500/20'
                      }`}
                      data-testid="badge-payment-type"
                    >
                      {isCashOnDelivery ? (
                        <>
                          <Banknote className="h-4 w-4" />
                          Pobranie
                        </>
                      ) : (
                        <>
                          <CreditCard className="h-4 w-4" />
                          {paymentType === 'ONLINE' ? 'Online' : paymentType}
                        </>
                      )}
                    </Badge>
                  </TooltipTrigger>
                  <TooltipContent>
                    Typ płatności: {isCashOnDelivery ? 'Za pobraniem' : 'Płatność online'}
                  </TooltipContent>
                </Tooltip>

                {/* Payment Provider */}
                <Tooltip>
                  <TooltipTrigger asChild>
                    <Badge 
                      variant="outline"
                      data-testid="badge-payment-provider"
                    >
                      {paymentProvider}
                    </Badge>
                  </TooltipTrigger>
                  <TooltipContent>Dostawca płatności</TooltipContent>
                </Tooltip>

                {/* Faktura */}
                <Tooltip>
                  <TooltipTrigger asChild>
                    <div className="inline-flex">
                      {invoice.required ? (
                        <Receipt className="h-5 w-5 text-green-500" data-testid="icon-invoice-yes" />
                      ) : (
                        <X className="h-5 w-5 text-muted-foreground" data-testid="icon-invoice-no" />
                      )}
                    </div>
                  </TooltipTrigger>
                  <TooltipContent>
                    {invoice.required ? 'Wymaga faktury VAT' : 'Nie wymaga faktury'}
                  </TooltipContent>
                </Tooltip>

                {/* Zwroty */}
                {hasReturns && (
                  <Tooltip>
                    <TooltipTrigger asChild>
                      <div className="inline-flex">
                        <PackageX className="h-5 w-5 text-red-500" data-testid="icon-returns" />
                      </div>
                    </TooltipTrigger>
                    <TooltipContent>
                      Produkt zgłoszony do zwrotu
                    </TooltipContent>
                  </Tooltip>
                )}

                {/* Kwota */}
                <Tooltip>
                  <TooltipTrigger asChild>
                    <Badge 
                      className={`font-semibold ${
                        hasRefund
                          ? 'bg-red-500 hover:bg-red-600 text-white'
                          : isPaid
                          ? 'bg-green-500 hover:bg-green-600 text-white'
                          : isCashOnDelivery 
                            ? 'bg-orange-500 hover:bg-orange-600 text-white' 
                            : 'bg-red-500 hover:bg-red-600 text-white'
                      }`}
                      data-testid="badge-total-amount"
                    >
                      {totalAmount.toFixed(2)} {rawData.summary?.totalToPay?.currency || "PLN"}
                    </Badge>
                  </TooltipTrigger>
                  <TooltipContent>
                    {hasRefund && refundAmount > 0 ? `Zwrot: ${refundAmount.toFixed(2)} ${payment.paidAmount?.currency || 'PLN'}` : 'Łączna kwota do zapłaty'}
                  </TooltipContent>
                </Tooltip>
              </div>

              {/* Prawa strona - Nawigacja i Linki */}
              <div className="flex items-center gap-2">
                {/* Nawigacja - Poprzednie zamówienie */}
                <Tooltip>
                  <TooltipTrigger asChild>
                    <Button
                      variant="ghost"
                      size="icon"
                      onClick={navigateToPrevious}
                      disabled={!hasPrevious}
                      data-testid="button-prev-order"
                    >
                      <ChevronLeft className="h-5 w-5" />
                    </Button>
                  </TooltipTrigger>
                  <TooltipContent>
                    {hasPrevious ? 'Poprzednie zamówienie' : `Brak poprzedniego (${currentOrderIndex + 1}/${allOrders.length})`}
                  </TooltipContent>
                </Tooltip>

                {/* Nawigacja - Następne zamówienie */}
                <Tooltip>
                  <TooltipTrigger asChild>
                    <Button
                      variant="ghost"
                      size="icon"
                      onClick={navigateToNext}
                      disabled={!hasNext}
                      data-testid="button-next-order"
                    >
                      <ChevronRight className="h-5 w-5" />
                    </Button>
                  </TooltipTrigger>
                  <TooltipContent>
                    {hasNext ? 'Następne zamówienie' : `Brak następnego (${currentOrderIndex + 1}/${allOrders.length})`}
                  </TooltipContent>
                </Tooltip>

                {/* Przycisk odświeżania zamówienia */}
                <Tooltip>
                  <TooltipTrigger asChild>
                    <Button
                      variant="ghost"
                      size="icon"
                      onClick={() => refreshOrderMutation.mutate()}
                      disabled={refreshOrderMutation.isPending}
                      data-testid="button-refresh-order"
                    >
                      <RefreshCw className={`h-5 w-5 ${refreshOrderMutation.isPending ? 'animate-spin' : ''}`} />
                    </Button>
                  </TooltipTrigger>
                  <TooltipContent>Odśwież zamówienie z platformy {source === 'ALLEGRO' ? 'Allegro' : 'Shoper'}</TooltipContent>
                </Tooltip>

                {/* Przycisk synchronizacji z Odoo */}
                <Tooltip>
                  <TooltipTrigger asChild>
                    <Button
                      variant="ghost"
                      size="icon"
                      onClick={() => syncToOdooMutation.mutate()}
                      disabled={syncToOdooMutation.isPending}
                      data-testid="button-sync-odoo"
                    >
                      <Upload className={`h-5 w-5 text-purple-600 dark:text-purple-400 ${syncToOdooMutation.isPending ? 'animate-pulse' : ''}`} />
                    </Button>
                  </TooltipTrigger>
                  <TooltipContent>Synchronizuj zamówienie z Odoo</TooltipContent>
                </Tooltip>

                {/* Link do zamówienia */}
                <Tooltip>
                  <TooltipTrigger asChild>
                    <Button
                      variant="ghost"
                      size="icon"
                      asChild
                      data-testid="link-order-external"
                    >
                      <a 
                        href={getOrderLink(source, source === 'ALLEGRO' ? rawData.allegroOrderId : rawData.shoperOrderId)} 
                        target="_blank" 
                        rel="noopener noreferrer"
                      >
                        <ShoppingCart className="h-5 w-5" />
                      </a>
                    </Button>
                  </TooltipTrigger>
                  <TooltipContent>Otwórz zamówienie w {source === 'ALLEGRO' ? 'Allegro' : 'Shoper'}</TooltipContent>
                </Tooltip>

                {/* Link do produktu */}
                {lineItems.length > 0 && lineItems[0].offer?.id && (
                  <Tooltip>
                    <TooltipTrigger asChild>
                      <Button
                        variant="ghost"
                        size="icon"
                        asChild
                        data-testid="link-product-external"
                      >
                        <a 
                          href={getProductLink(source, lineItems[0].offer.id) || '#'} 
                          target="_blank" 
                          rel="noopener noreferrer"
                        >
                          <Box className="h-5 w-5" />
                        </a>
                      </Button>
                    </TooltipTrigger>
                    <TooltipContent>Otwórz produkt w {source === 'ALLEGRO' ? 'Allegro' : 'Shoper'}</TooltipContent>
                  </Tooltip>
                )}
              </div>
            </div>
          </TooltipProvider>

          {/* Daty i email - kompaktowo */}
          <div className="mt-3 flex items-center gap-6 flex-wrap text-xs text-muted-foreground">
            {/* Data zamówienia */}
            {rawData.boughtAt && (
              <div className="inline-flex items-center gap-1.5">
                <span>Data zamówienia:</span>
                <span className="font-medium" data-testid="text-order-date-small">
                  {rawData.boughtAt 
                    ? format(new Date(rawData.boughtAt), "dd MMM yyyy, HH:mm", { locale: pl })
                    : "—"
                  }
                </span>
              </div>
            )}
            
            {/* Ostatnia aktualizacja */}
            {rawData.updatedAt && (
              <div className="inline-flex items-center gap-1.5">
                <span>Ostatnia aktualizacja:</span>
                <span className="font-medium" data-testid="text-updated-at-small">
                  {rawData.updatedAt 
                    ? format(new Date(rawData.updatedAt), "dd MMM yyyy, HH:mm", { locale: pl })
                    : "—"
                  }
                </span>
              </div>
            )}

            {/* Email klienta */}
            {buyer.email && (
              <a 
                href={`mailto:${buyer.email}`}
                className="hover:text-foreground transition-colors inline-flex items-center gap-1.5"
                data-testid="link-buyer-email"
              >
                <User className="h-3.5 w-3.5" />
                {buyer.email}
              </a>
            )}
          </div>
        </div>

        {/* Produkty i Płatności - obok siebie */}
        <div className="grid gap-6 md:grid-cols-2 mb-6">
          {/* Produkty - lewa kolumna */}
          <Card>
            <CardHeader>
              <div className="flex items-center gap-2">
                <Package className="h-5 w-5 text-muted-foreground" />
                <CardTitle>Produkty ({lineItems.length})</CardTitle>
              </div>
              <CardDescription>
                Lista produktów zawartych w zamówieniu
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {lineItems.map((item: any, index: number) => (
                  <div key={item.id || index}>
                    {index > 0 && <Separator className="my-4" />}
                    <div className="flex gap-4">
                      <div className="w-24 h-24 flex-shrink-0 rounded-md overflow-hidden bg-muted flex items-center justify-center">
                        {item.imageUrl ? (
                          <img
                            src={item.imageUrl}
                            alt={item.offer?.name || "Produkt"}
                            className="w-full h-full object-cover"
                            onError={(e) => {
                              (e.target as HTMLImageElement).src = 'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iOTYiIGhlaWdodD0iOTYiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+PHJlY3Qgd2lkdGg9Ijk2IiBoZWlnaHQ9Ijk2IiBmaWxsPSIjZjNmNGY2Ii8+PC9zdmc+';
                            }}
                            data-testid={`img-product-${item.offer?.external?.id || item.id}`}
                          />
                        ) : (
                          <Package className="h-8 w-8 text-muted-foreground" />
                        )}
                      </div>
                      <div className="flex-1">
                        <div className="font-medium mb-2">
                          {item.offer?.name || "Produkt"}
                        </div>
                        <div className="grid gap-2 text-sm">
                          {item.offer?.external?.id && (
                            <div className="flex justify-between">
                              <span className="text-muted-foreground">SKU:</span>
                              <Link 
                                href={`/product/${encodeURIComponent(item.offer.external.id)}`}
                                className="font-mono text-xs bg-muted px-2 py-0.5 rounded hover:bg-accent transition-colors cursor-pointer inline-flex items-center gap-1"
                                data-testid={`link-sku-${item.offer.external.id}`}
                              >
                                {item.offer.external.id}
                                <ExternalLink className="h-3 w-3" />
                              </Link>
                            </div>
                          )}
                          <div className="flex justify-between">
                            <span className="text-muted-foreground">ID oferty:</span>
                            {item.offer?.id ? (
                              <a 
                                href={getProductLink(source, item.offer.id)} 
                                target="_blank" 
                                rel="noopener noreferrer"
                                className="font-mono text-xs bg-muted px-2 py-0.5 rounded hover:bg-accent transition-colors cursor-pointer"
                              >
                                {item.offer.id}
                              </a>
                            ) : (
                              <code className="font-mono text-xs bg-muted px-2 py-0.5 rounded">—</code>
                            )}
                          </div>
                          <div className="flex justify-between">
                            <span className="text-muted-foreground">Ilość:</span>
                            <span className="font-medium">{item.quantity || 1} szt.</span>
                          </div>
                          {((item.returnsQuantity || 0) > 0) && (
                            <div className="flex justify-between">
                              <span className="text-red-600 dark:text-red-400 flex items-center gap-1">
                                <PackageX className="h-4 w-4" />
                                Zgłoszono do zwrotu:
                              </span>
                              <span className="font-medium text-red-600 dark:text-red-400">
                                {item.returnsQuantity} szt.
                              </span>
                            </div>
                          )}
                          <div className="flex justify-between">
                            <span className="text-muted-foreground">Cena jednostkowa:</span>
                            <span className="font-medium">
                              {item.price?.amount || "0.00"} {item.price?.currency || "PLN"}
                            </span>
                          </div>
                          {item.originalPrice &&
                            item.originalPrice.amount !== item.price?.amount && (
                              <div className="flex justify-between">
                                <span className="text-muted-foreground">
                                  Cena oryginalna:
                                </span>
                                <span className="line-through text-muted-foreground">
                                  {item.originalPrice.amount} {item.originalPrice.currency}
                                </span>
                              </div>
                            )}
                          <div className="flex justify-between text-base font-semibold pt-2 border-t">
                            <span>Razem:</span>
                            <span>
                              {(
                                parseFloat(item.price?.amount || "0") *
                                (item.quantity || 1)
                              ).toFixed(2)}{" "}
                              {item.price?.currency || "PLN"}
                            </span>
                          </div>
                        </div>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>

          {/* Płatność - prawa kolumna */}
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle>Płatności</CardTitle>
              </div>
              <CardDescription>
                {isPaid ? (
                  <div>
                    <span className={`text-base ${hasRefund && netPaidAmount === 0 ? 'text-red-600 dark:text-red-400' : getPaymentColorClass()}`}>
                      Zapłacono <span className="font-semibold">{netPaidAmount.toFixed(2)} {payment.paidAmount.currency}</span> z <span className="font-semibold">{totalAmount.toFixed(2)} {rawData.summary?.totalToPay?.currency || "PLN"}</span>
                    </span>
                    {hasRefund && (
                      <div className="mt-1 flex items-center gap-2">
                        <span className="text-sm text-red-600 dark:text-red-400 uppercase font-medium tracking-wider">
                          ZWRÓĆ WPŁATĘ
                        </span>
                        {source === 'ALLEGRO' && rawData.allegroOrderId && (
                          <a 
                            href={`https://allegro.pl/moje-allegro/sprzedaz/zamowienia/${rawData.allegroOrderId}`}
                            target="_blank"
                            rel="noopener noreferrer"
                            className="inline-flex items-center gap-1 text-xs text-primary hover:underline"
                            data-testid="link-allegro-order"
                          >
                            <ExternalLink className="h-3 w-3" />
                            Przejdź do Allegro
                          </a>
                        )}
                        {source === 'SHOPER' && rawData.shoperOrderId && (
                          <a 
                            href={getOrderLink(source, rawData.shoperOrderId)}
                            target="_blank"
                            rel="noopener noreferrer"
                            className="inline-flex items-center gap-1 text-xs text-primary hover:underline"
                            data-testid="link-shoper-order"
                          >
                            <ExternalLink className="h-3 w-3" />
                            Przejdź do Shoper
                          </a>
                        )}
                      </div>
                    )}
                  </div>
                ) : (
                  <span className={`text-base ${getPaymentColorClass()}`}>
                    {isCashOnDelivery ? 'Do zapłaty (za pobraniem)' : 'Nie zapłacono'}: <span className="font-semibold">{totalAmount.toFixed(2)} {rawData.summary?.totalToPay?.currency || "PLN"}</span>
                  </span>
                )}
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="rounded-lg border">
                <table className="w-full">
                  <thead>
                    <tr className="border-b bg-muted/50">
                      <th className="p-3 text-left text-sm font-medium text-muted-foreground">kwota</th>
                      <th className="p-3 text-left text-sm font-medium text-muted-foreground">operacja</th>
                      <th className="p-3 text-left text-sm font-medium text-muted-foreground">operator płatności</th>
                      <th className="p-3 text-left text-sm font-medium text-muted-foreground">identyfikator płatności</th>
                      <th className="p-3 text-right text-sm font-medium text-muted-foreground"></th>
                    </tr>
                  </thead>
                  <tbody>
                    <tr className="border-b last:border-0">
                      <td className="p-3">
                        <div className={`font-semibold ${getPaymentColorClass()}`}>{paidAmountValue.toFixed(2)} {rawData.summary?.totalToPay?.currency || "PLN"}</div>
                        {payment.finishedAt && (
                          <div className="text-xs text-muted-foreground">
                            {format(new Date(payment.finishedAt), "d MMM yyyy, HH:mm", { locale: pl })}
                          </div>
                        )}
                      </td>
                      <td className="p-3 text-sm">
                        płatność
                      </td>
                      <td className="p-3 text-sm font-medium">
                        {payment.provider || "—"}
                      </td>
                      <td className="p-3">
                        <div className="text-xs text-primary break-all font-mono">
                          {payment.id || rawData.id?.substring(0, 32) || "—"}
                        </div>
                      </td>
                      <td className="p-3 text-right">
                        <Badge variant="default" className="bg-green-600 hover:bg-green-700">
                          OPŁACONE
                        </Badge>
                      </td>
                    </tr>
                    {hasRefund && (
                      <tr className="border-b last:border-0">
                        <td className="p-3">
                          <div className="font-semibold text-red-600 dark:text-red-400">-{refundAmount.toFixed(2)} {payment.paidAmount.currency}</div>
                          {payment.refundDate && (
                            <div className="text-xs text-muted-foreground">
                              {format(new Date(payment.refundDate), "d MMM yyyy, HH:mm", { locale: pl })}
                            </div>
                          )}
                        </td>
                        <td className="p-3 text-sm">
                          zwrot
                        </td>
                        <td className="p-3 text-sm font-medium">
                          {payment.provider || "—"}
                        </td>
                        <td className="p-3">
                          <div className="text-xs text-muted-foreground">
                            —
                          </div>
                        </td>
                        <td className="p-3 text-right">
                          <Badge variant="default" className="bg-red-600 hover:bg-red-700">
                            ZWRÓCONE
                          </Badge>
                        </td>
                      </tr>
                    )}
                  </tbody>
                  {hasRefund && (
                    <tfoot>
                      <tr className="border-t-2 bg-muted/30">
                        <td className="p-3">
                          <div className={`font-bold text-lg ${netPaidAmount === 0 ? 'text-red-600 dark:text-red-400' : getPaymentColorClass()}`}>
                            {netPaidAmount.toFixed(2)} {payment.paidAmount.currency}
                          </div>
                        </td>
                        <td className="p-3 text-sm font-semibold" colSpan={3}>
                          Netto do zapłaty
                        </td>
                        <td className="p-3"></td>
                      </tr>
                    </tfoot>
                  )}
                </table>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Pozostałe dane */}
        <div className="grid gap-6 md:grid-cols-2">
          {/* Kupujący */}
          <Card>
            <CardHeader>
              <div className="flex items-center gap-2">
                <User className="h-5 w-5 text-muted-foreground" />
                <CardTitle>Dane kupującego</CardTitle>
              </div>
            </CardHeader>
            <CardContent className="space-y-4">
              {/* Imię i nazwisko - wyróżnione */}
              {buyer.firstName && buyer.lastName && (
                <div>
                  <div className="text-lg font-bold text-foreground">
                    {buyer.firstName} {buyer.lastName}
                  </div>
                  <div className="text-sm text-muted-foreground mt-0.5">Imię i nazwisko</div>
                </div>
              )}
              
              {/* Firma jeśli jest */}
              {buyer.companyName && (
                <div>
                  <div className="font-semibold text-base">{buyer.companyName}</div>
                  <div className="text-sm text-muted-foreground mt-0.5">Firma</div>
                </div>
              )}

              <Separator />

              {/* Kontakt - grid 2 kolumny na desktop */}
              <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                <div>
                  <div className="text-sm text-muted-foreground mb-1">Email</div>
                  <div className="font-medium">{buyer.email || "—"}</div>
                </div>
                <div>
                  <div className="text-sm text-muted-foreground mb-1">Telefon</div>
                  <div className="font-medium">{buyer.phoneNumber || "—"}</div>
                </div>
                {buyer.login && (
                  <div className="md:col-span-2">
                    <div className="text-sm text-muted-foreground mb-1">Login</div>
                    <div className="font-medium text-sm">{buyer.login}</div>
                  </div>
                )}
              </div>
            </CardContent>
          </Card>

          {/* Dostawa */}
          <Card>
            <CardHeader>
              <div className="flex items-center gap-2">
                <Truck className="h-5 w-5 text-muted-foreground" />
                <CardTitle>Informacje o dostawie</CardTitle>
              </div>
            </CardHeader>
            <CardContent className="space-y-4">
              {/* Adres dostawy - wyróżniony */}
              {delivery.address && (
                <div>
                  {/* Nazwa/Firma - duża */}
                  {delivery.address.companyName ? (
                    <div className="font-semibold text-base mb-1">
                      {delivery.address.companyName}
                    </div>
                  ) : (delivery.address.firstName && delivery.address.lastName) ? (
                    <div className="text-lg font-bold text-foreground mb-1">
                      {delivery.address.firstName} {delivery.address.lastName}
                    </div>
                  ) : null}
                  
                  {/* Adres - wyraźny */}
                  <div className="font-medium text-base">
                    {delivery.address.street}
                    <br />
                    {delivery.address.zipCode} {delivery.address.city}
                  </div>
                  
                  {delivery.address.phoneNumber && (
                    <div className="text-sm text-muted-foreground mt-1">
                      Tel: {delivery.address.phoneNumber}
                    </div>
                  )}
                  
                  <div className="text-sm text-muted-foreground mt-2">Adres dostawy</div>
                </div>
              )}

              <Separator />

              {/* Szczegóły dostawy - grid 2 kolumny na desktop */}
              <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                {delivery.method?.name && (
                  <div>
                    <div className="text-sm text-muted-foreground mb-1">Metoda dostawy</div>
                    <div className="font-medium">{delivery.method.name}</div>
                  </div>
                )}
                {delivery.cost && (
                  <div>
                    <div className="text-sm text-muted-foreground mb-1">Koszt dostawy</div>
                    <div className="font-medium">
                      {delivery.cost.amount} {delivery.cost.currency}
                    </div>
                  </div>
                )}
                {delivery.smart && (
                  <div>
                    <div className="text-sm text-muted-foreground mb-1">Allegro Smart</div>
                    <div className="font-medium">Tak</div>
                  </div>
                )}
                {delivery.time && (
                  <div className="md:col-span-2">
                    <div className="text-sm text-muted-foreground mb-1">Termin dostawy</div>
                    <div className="font-medium">
                      {format(new Date(delivery.time.from), "dd MMM yyyy", {
                        locale: pl,
                      })}{" "}
                      -{" "}
                      {format(new Date(delivery.time.to), "dd MMM yyyy", {
                        locale: pl,
                      })}
                    </div>
                  </div>
                )}
              </div>
            </CardContent>
          </Card>

          {/* Przesyłki */}
          {delivery.shipments && delivery.shipments.length > 0 && (
            <Card className="md:col-span-2">
              <CardHeader>
                <div className="flex items-center gap-2">
                  <Package className="h-5 w-5 text-muted-foreground" />
                  <CardTitle>Przesyłki ({delivery.shipments.length})</CardTitle>
                </div>
                <CardDescription>
                  Numery śledzenia przesyłek dla tego zamówienia
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="rounded-lg border">
                  <table className="w-full">
                    <thead>
                      <tr className="border-b bg-muted/50">
                        <th className="p-3 text-left text-sm font-medium text-muted-foreground">Przewoźnik</th>
                        <th className="p-3 text-left text-sm font-medium text-muted-foreground">Numer przesyłki</th>
                        <th className="p-3 text-left text-sm font-medium text-muted-foreground">Data utworzenia</th>
                        <th className="p-3 text-right text-sm font-medium text-muted-foreground">Śledzenie</th>
                      </tr>
                    </thead>
                    <tbody>
                      {delivery.shipments.map((shipment: any, index: number) => {
                        const trackingNumber = shipment.waybill || shipment.waybill_number || shipment.tracking_number || shipment.number;
                        const carrierId = shipment.carrierId || shipment.carrier_id || shipment.carrier || shipment.delivery_id;
                        const createdAt = shipment.createdAt || shipment.created_at || shipment.date_add;
                        
                        const getTrackingUrl = (carrier: string, number: string) => {
                          if (!number) return null;
                          
                          const carrierLower = (carrier || '').toLowerCase();
                          
                          if (carrierLower.includes('dpd')) {
                            return `https://tracktrace.dpd.com.pl/parcelDetails?typ=1&p1=${encodeURIComponent(number)}`;
                          } else if (carrierLower.includes('inpost')) {
                            return `https://inpost.pl/sledzenie-przesylek?number=${encodeURIComponent(number)}`;
                          } else if (carrierLower.includes('dhl')) {
                            return `https://www.dhl.com/pl-pl/home/tracking/tracking-parcel.html?submit=1&tracking-id=${encodeURIComponent(number)}`;
                          } else if (carrierLower.includes('ups')) {
                            return `https://www.ups.com/track?tracknum=${encodeURIComponent(number)}`;
                          } else if (carrierLower.includes('fedex')) {
                            return `https://www.fedex.com/fedextrack/?trknbr=${encodeURIComponent(number)}`;
                          } else if (carrierLower.includes('poczta') || carrierLower.includes('pocztex')) {
                            return `https://emonitoring.poczta-polska.pl/?numer=${encodeURIComponent(number)}`;
                          } else if (carrierLower.includes('gls')) {
                            return `https://gls-group.eu/PL/pl/sledzenie-paczek?match=${encodeURIComponent(number)}`;
                          } else {
                            return `https://www.google.com/search?q=${encodeURIComponent(number)}`;
                          }
                        };
                        
                        const trackingUrl = getTrackingUrl(carrierId, trackingNumber);
                        
                        return (
                          <tr key={index} className="border-b last:border-0">
                            <td className="p-3">
                              <div className="font-medium">{carrierId || "—"}</div>
                            </td>
                            <td className="p-3">
                              <code className="text-sm font-mono bg-muted px-2 py-1 rounded">
                                {trackingNumber || "—"}
                              </code>
                            </td>
                            <td className="p-3 text-sm">
                              {createdAt ? format(new Date(createdAt), "dd MMM yyyy, HH:mm", { locale: pl }) : "—"}
                            </td>
                            <td className="p-3 text-right">
                              {trackingUrl && (
                                <Button
                                  variant="ghost"
                                  size="sm"
                                  asChild
                                  data-testid={`button-track-shipment-${index}`}
                                >
                                  <a href={trackingUrl} target="_blank" rel="noopener noreferrer">
                                    <ExternalLink className="h-4 w-4 mr-1" />
                                    Śledź
                                  </a>
                                </Button>
                              )}
                            </td>
                          </tr>
                        );
                      })}
                    </tbody>
                  </table>
                </div>
              </CardContent>
            </Card>
          )}

          {/* Faktura - wyróżniona (lewa kolumna) */}
          {invoice.required && (
            <Card className="border-l-4 border-l-blue-500 bg-blue-500/5 dark:bg-blue-500/10">
              <CardHeader>
                <div className="flex items-center gap-2">
                  <FileText className="h-5 w-5 text-blue-600 dark:text-blue-400" />
                  <CardTitle className="text-blue-600 dark:text-blue-400">Dane do faktury</CardTitle>
                </div>
              </CardHeader>
              <CardContent className="space-y-4">
                {/* NIP / Tax ID - wyróżniony */}
                {billingAddress.taxId && (
                  <div>
                    <div className="text-sm text-muted-foreground mb-1">NIP</div>
                    <div className="font-bold text-xl font-mono" data-testid="text-invoice-nip">
                      {billingAddress.taxId}
                    </div>
                  </div>
                )}

                <Separator />

                {/* Nazwa firmy */}
                {billingAddress.companyName && (
                  <div>
                    <div className="font-semibold text-base mb-1" data-testid="text-invoice-company">
                      {billingAddress.companyName}
                    </div>
                    <div className="text-sm text-muted-foreground">Nazwa firmy</div>
                  </div>
                )}

                {/* Pełny adres */}
                <div>
                  <div className="font-medium space-y-0.5 mb-1" data-testid="text-invoice-address">
                    {billingAddress.firstName && billingAddress.lastName && (
                      <div className="font-semibold">{billingAddress.firstName} {billingAddress.lastName}</div>
                    )}
                    {billingAddress.street && (
                      <div>{billingAddress.street}</div>
                    )}
                    {(billingAddress.zipCode || billingAddress.city) && (
                      <div>
                        {billingAddress.zipCode && `${billingAddress.zipCode} `}
                        {billingAddress.city}
                      </div>
                    )}
                    {billingAddress.countryCode && billingAddress.countryCode !== 'PL' && (
                      <div className="text-sm">
                        {billingAddress.countryCode}
                      </div>
                    )}
                  </div>
                  <div className="text-sm text-muted-foreground">Adres faktury</div>
                </div>

                {/* Email i telefon - grid 2 kolumny na desktop */}
                {(billingAddress.email || billingAddress.phoneNumber) && (
                  <>
                    <Separator />
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                      {billingAddress.email && (
                        <div>
                          <div className="text-sm text-muted-foreground">Email</div>
                          <div className="font-medium text-sm">{billingAddress.email}</div>
                        </div>
                      )}
                      {billingAddress.phoneNumber && (
                        <div>
                          <div className="text-sm text-muted-foreground">Telefon</div>
                          <div className="font-medium text-sm">{billingAddress.phoneNumber}</div>
                        </div>
                      )}
                    </div>
                  </>
                )}
              </CardContent>
            </Card>
          )}
          
          {/* Faktura niewymagana - dyskretna informacja (lewa kolumna) */}
          {!invoice.required && (
            <Card>
              <CardHeader>
                <div className="flex items-center gap-2">
                  <FileText className="h-5 w-5 text-muted-foreground" />
                  <CardTitle>Faktura</CardTitle>
                </div>
              </CardHeader>
              <CardContent>
                <div className="text-sm text-muted-foreground">
                  Faktura nie jest wymagana dla tego zamówienia
                </div>
              </CardContent>
            </Card>
          )}

          {/* Komentarz klienta (prawa kolumna) */}
          {rawData.messageToSeller ? (
            <Card>
              <CardHeader>
                <div className="flex items-center gap-2">
                  <MessageSquare className="h-5 w-5 text-muted-foreground" />
                  <CardTitle>Komentarz klienta</CardTitle>
                </div>
              </CardHeader>
              <CardContent>
                <div className="p-4 bg-muted rounded-md">
                  <p className="text-base">{rawData.messageToSeller}</p>
                </div>
              </CardContent>
            </Card>
          ) : (
            <Card>
              <CardHeader>
                <div className="flex items-center gap-2">
                  <MessageSquare className="h-5 w-5 text-muted-foreground" />
                  <CardTitle>Komentarz klienta</CardTitle>
                </div>
              </CardHeader>
              <CardContent>
                <div className="text-sm text-muted-foreground">
                  Brak komentarza do zamówienia
                </div>
              </CardContent>
            </Card>
          )}

          {/* Informacje dodatkowe */}
          <Card className="md:col-span-2">
            <CardHeader>
              <div className="flex items-center gap-2">
                <Calendar className="h-5 w-5 text-muted-foreground" />
                <CardTitle>Informacje dodatkowe</CardTitle>
              </div>
            </CardHeader>
            <CardContent className="space-y-3">
              <div className="grid gap-3 md:grid-cols-3">
                <div>
                  <div className="text-sm text-muted-foreground">Data zamówienia</div>
                  <div className="font-medium">
                    {rawData.boughtAt 
                      ? format(new Date(rawData.boughtAt), "dd MMM yyyy, HH:mm", {
                          locale: pl,
                        })
                      : "—"
                    }
                  </div>
                </div>
                <div>
                  <div className="text-sm text-muted-foreground">Ostatnia aktualizacja</div>
                  <div className="font-medium">
                    {rawData.updatedAt 
                      ? format(new Date(rawData.updatedAt), "dd MMM yyyy, HH:mm", { locale: pl })
                      : "—"
                    }
                  </div>
                </div>
                <div>
                  <div className="text-sm text-muted-foreground">Status zamówienia</div>
                  <div className="font-medium">
                    {source === 'ALLEGRO' && rawData.status 
                      ? translateAllegroStatus(rawData.status) 
                      : rawData.status || "—"}
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Historia zmian zamówienia */}
          {orderChanges.length > 0 && (
            <Card className="md:col-span-2">
              <CardHeader>
                <div className="flex items-center gap-2">
                  <MessageSquare className="h-5 w-5 text-muted-foreground" />
                  <CardTitle>Historia zmian</CardTitle>
                </div>
                <CardDescription>
                  Automatycznie wykryte zmiany w tym zamówieniu
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {orderChanges.map((change) => (
                    <div 
                      key={change.id} 
                      className="flex items-start gap-4 p-4 rounded-lg border bg-card hover-elevate transition-all"
                      data-testid={`change-item-${change.id}`}
                    >
                      <div className="flex-1 space-y-1">
                        <div className="flex items-center gap-2">
                          <span className="font-semibold">
                            {fieldLabels[change.field_changed] || change.field_changed}
                          </span>
                          <span className="text-muted-foreground text-sm">
                            •
                          </span>
                          <span className="text-sm text-muted-foreground">
                            {format(new Date(change.detected_at), "dd.MM.yyyy HH:mm", { locale: pl })}
                          </span>
                        </div>
                        <div className="flex items-center gap-2 text-sm">
                          <code className="px-2 py-1 bg-muted rounded text-xs">
                            {change.old_value || '—'}
                          </code>
                          <ArrowRight className="h-3 w-3 text-muted-foreground" />
                          <code className="px-2 py-1 bg-primary/10 rounded text-xs font-semibold">
                            {change.new_value || '—'}
                          </code>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          )}

          {/* Raw Data z API - zakładki z różnymi sekcjami */}
          <Card className="md:col-span-2">
            <CardHeader>
              <CardTitle>Surowe dane z API ({source})</CardTitle>
              <CardDescription>
                Oryginalne dane pobrane z platformy {source === 'ALLEGRO' ? 'Allegro' : 'Shoper'} - porównaj z danymi przetworzonymi
              </CardDescription>
            </CardHeader>
            <CardContent>
              <Tabs defaultValue="order" className="w-full">
                <TabsList className="grid w-full grid-cols-5">
                  <TabsTrigger value="order" data-testid="tab-api-order">Zamówienie</TabsTrigger>
                  <TabsTrigger value="products" data-testid="tab-api-products">Produkty</TabsTrigger>
                  <TabsTrigger value="payment" data-testid="tab-api-payment">Płatność</TabsTrigger>
                  <TabsTrigger value="delivery" data-testid="tab-api-delivery">Dostawa</TabsTrigger>
                  <TabsTrigger value="all" data-testid="tab-api-all">Wszystko</TabsTrigger>
                </TabsList>
                
                <TabsContent value="order" className="mt-4">
                  <div className="space-y-2">
                    <h4 className="text-sm font-semibold text-muted-foreground">
                      API Order - Główne dane zamówienia z {source === 'ALLEGRO' ? 'Allegro' : 'Shoper'}
                    </h4>
                    <pre className="text-xs bg-muted p-4 rounded-md overflow-auto max-h-96 font-mono">
                      {rawData.rawPayload ? JSON.stringify({
                        order_id: rawData.rawPayload.order_id || rawData.rawPayload.id,
                        status: rawData.rawPayload.status || rawData.rawPayload.status_id,
                        date: rawData.rawPayload.date || rawData.rawPayload.boughtAt,
                        sum: rawData.rawPayload.sum,
                        currency: rawData.rawPayload.currency || rawData.rawPayload.currency_id,
                        payment_id: rawData.rawPayload.payment_id,
                        shipping_id: rawData.rawPayload.shipping_id,
                        email: rawData.rawPayload.email,
                        notes: rawData.rawPayload.notes,
                        is_paid: rawData.rawPayload.is_paid,
                        paid: rawData.rawPayload.paid,
                        confirm: rawData.rawPayload.confirm,
                        origin: rawData.rawPayload.origin
                      }, null, 2) : 'Brak danych'}
                    </pre>
                  </div>
                </TabsContent>

                <TabsContent value="products" className="mt-4">
                  <div className="space-y-2">
                    <h4 className="text-sm font-semibold text-muted-foreground">
                      API Products - Produkty z {source === 'ALLEGRO' ? 'Allegro' : 'Shoper'}
                    </h4>
                    <pre className="text-xs bg-muted p-4 rounded-md overflow-auto max-h-96 font-mono">
                      {rawData.rawPayload?.products_data || rawData.rawPayload?.lineItems 
                        ? JSON.stringify(rawData.rawPayload.products_data || rawData.rawPayload.lineItems, null, 2)
                        : 'Brak danych produktów'}
                    </pre>
                  </div>
                </TabsContent>

                <TabsContent value="payment" className="mt-4">
                  <div className="space-y-2">
                    <h4 className="text-sm font-semibold text-muted-foreground">
                      API Payment - Dane płatności z {source === 'ALLEGRO' ? 'Allegro' : 'Shoper'}
                    </h4>
                    <pre className="text-xs bg-muted p-4 rounded-md overflow-auto max-h-96 font-mono">
                      {rawData.rawPayload ? JSON.stringify({
                        payment_id: rawData.rawPayload.payment_id,
                        payment_method_name: rawData.rawPayload.payment_method_name,
                        is_paid: rawData.rawPayload.is_paid,
                        paid: rawData.rawPayload.paid,
                        sum: rawData.rawPayload.sum,
                        currency: rawData.rawPayload.currency || rawData.rawPayload.currency_id,
                        payment: rawData.rawPayload.payment,
                        cod: rawData.rawPayload.cod
                      }, null, 2) : 'Brak danych płatności'}
                    </pre>
                  </div>
                </TabsContent>

                <TabsContent value="delivery" className="mt-4">
                  <div className="space-y-2">
                    <h4 className="text-sm font-semibold text-muted-foreground">
                      API Delivery - Dane dostawy z {source === 'ALLEGRO' ? 'Allegro' : 'Shoper'}
                    </h4>
                    <pre className="text-xs bg-muted p-4 rounded-md overflow-auto max-h-96 font-mono">
                      {rawData.rawPayload ? JSON.stringify({
                        shipping_id: rawData.rawPayload.shipping_id,
                        delivery_method_name: rawData.rawPayload.delivery_method_name,
                        shipping_cost: rawData.rawPayload.shipping_cost,
                        delivery_price: rawData.rawPayload.delivery_price,
                        delivery_date: rawData.rawPayload.delivery_date,
                        delivery_address: rawData.rawPayload.delivery_address || rawData.rawPayload.shipping_address,
                        billing_address: rawData.rawPayload.billing_address,
                        delivery: rawData.rawPayload.delivery,
                        shipments: rawData.rawPayload.shipments
                      }, null, 2) : 'Brak danych dostawy'}
                    </pre>
                  </div>
                </TabsContent>

                <TabsContent value="all" className="mt-4">
                  <div className="space-y-2">
                    <h4 className="text-sm font-semibold text-muted-foreground">
                      Wszystkie surowe dane z API {source === 'ALLEGRO' ? 'Allegro' : 'Shoper'}
                    </h4>
                    <pre className="text-xs bg-muted p-4 rounded-md overflow-auto max-h-96 font-mono">
                      {rawData.rawPayload ? JSON.stringify(rawData.rawPayload, null, 2) : 'Brak surowych danych'}
                    </pre>
                  </div>
                </TabsContent>
              </Tabs>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}
