import { useQuery } from "@tanstack/react-query";
import { useAuth } from "@/hooks/use-auth";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { ArrowLeft, FileText } from "lucide-react";
import { useLocation } from "wouter";
import { format } from "date-fns";
import { pl } from "date-fns/locale";

interface ApiRequestLog {
  id: number;
  token_id: number;
  token_description: string;
  method: string;
  path: string;
  status_code: number;
  response_time_ms: number;
  ip_address: string;
  user_agent: string;
  created_at: string;
}

export default function ApiTokenLogsPage() {
  const { user } = useAuth();
  const [, setLocation] = useLocation();

  const { data: logs, isLoading } = useQuery<ApiRequestLog[]>({
    queryKey: ["/api/api-tokens/logs"],
    enabled: user?.role === 'admin',
  });

  if (user?.role !== 'admin') {
    return (
      <div className="flex items-center justify-center h-full">
        <Card>
          <CardHeader>
            <CardTitle>Brak dostępu</CardTitle>
            <CardDescription>
              Tylko administratorzy mają dostęp do logów API.
            </CardDescription>
          </CardHeader>
        </Card>
      </div>
    );
  }

  const getStatusBadge = (statusCode: number) => {
    if (statusCode >= 200 && statusCode < 300) {
      return <Badge variant="default" data-testid={`badge-status-${statusCode}`}>{statusCode}</Badge>;
    } else if (statusCode >= 400 && statusCode < 500) {
      return <Badge variant="secondary" data-testid={`badge-status-${statusCode}`}>{statusCode}</Badge>;
    } else if (statusCode >= 500) {
      return <Badge variant="destructive" data-testid={`badge-status-${statusCode}`}>{statusCode}</Badge>;
    }
    return <Badge variant="outline" data-testid={`badge-status-${statusCode}`}>{statusCode}</Badge>;
  };

  const getMethodBadge = (method: string) => {
    const variants: Record<string, any> = {
      GET: "outline",
      POST: "default",
      PUT: "secondary",
      DELETE: "destructive",
    };
    return <Badge variant={variants[method] || "outline"} data-testid={`badge-method-${method}`}>{method}</Badge>;
  };

  return (
    <div className="p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold">Logi API Requestów</h1>
          <p className="text-muted-foreground">Historia zapytań do zewnętrznego API</p>
        </div>
        <Button
          variant="outline"
          onClick={() => setLocation("/api-tokens")}
          data-testid="button-back"
        >
          <ArrowLeft className="w-4 h-4 mr-2" />
          Powrót do tokenów
        </Button>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Ostatnie zapytania</CardTitle>
          <CardDescription>
            Audyt wszystkich zapytań do API z informacjami o tokenach, statusach i czasach odpowiedzi
          </CardDescription>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="text-center py-8 text-muted-foreground">Ładowanie...</div>
          ) : logs && logs.length > 0 ? (
            <div className="overflow-x-auto">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Data</TableHead>
                    <TableHead>Token</TableHead>
                    <TableHead>Metoda</TableHead>
                    <TableHead>Ścieżka</TableHead>
                    <TableHead>Status</TableHead>
                    <TableHead>Czas</TableHead>
                    <TableHead>IP</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {logs.map((log) => (
                    <TableRow key={log.id} data-testid={`row-log-${log.id}`}>
                      <TableCell className="text-muted-foreground whitespace-nowrap">
                        {format(new Date(log.created_at), "dd.MM.yyyy HH:mm:ss", { locale: pl })}
                      </TableCell>
                      <TableCell>
                        <span className="text-sm" data-testid={`text-token-${log.id}`}>
                          {log.token_description}
                        </span>
                      </TableCell>
                      <TableCell>{getMethodBadge(log.method)}</TableCell>
                      <TableCell>
                        <code className="text-sm" data-testid={`text-path-${log.id}`}>{log.path}</code>
                      </TableCell>
                      <TableCell>{getStatusBadge(log.status_code)}</TableCell>
                      <TableCell>
                        <span className="text-sm text-muted-foreground" data-testid={`text-time-${log.id}`}>
                          {log.response_time_ms}ms
                        </span>
                      </TableCell>
                      <TableCell>
                        <span className="text-sm text-muted-foreground font-mono" data-testid={`text-ip-${log.id}`}>
                          {log.ip_address}
                        </span>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          ) : (
            <div className="text-center py-8 text-muted-foreground">
              <FileText className="w-12 h-12 mx-auto mb-4 opacity-50" />
              <p>Brak logów zapytań</p>
              <p className="text-sm">Logi pojawią się po pierwszym zapytaniu do API</p>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
