import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Button } from "@/components/ui/button";
import { Copy } from "lucide-react";
import { useToast } from "@/hooks/use-toast";

interface ExamplesModalProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
}

export function ExamplesModal({ open, onOpenChange }: ExamplesModalProps) {
  const { toast } = useToast();

  const copyExample = (code: string) => {
    navigator.clipboard.writeText(code);
    toast({
      title: "Skopiowano",
      description: "Przykład został skopiowany do schowka",
    });
  };

  const examples = {
    basic: [
      {
        title: "Prosty opis z wymiarami",
        description: "Podstawowy szablon z danymi produktu",
        code: `Przedstawiamy {{title}} w kolorze {{color}}.

Wymiary: {{dimensions}}
Cena: {{price}} PLN`,
      },
      {
        title: "Opis z ceną bazową",
        description: "Pokazanie ceny i ceny bazowej",
        code: `{{title}}

Wymiary: {{length}}×{{width}}×{{height}} cm
Kolor: {{color}}
Cena: {{price}} PLN
Cena bazowa: {{basePrice}} PLN`,
      },
    ],
    accessories: [
      {
        title: "Akcesorium jako HTML lista",
        description: "Tag bez parametru :grid generuje tradycyjną HTML listę",
        code: `{{title}} - elegancka komoda

Wykonana z najwyższej jakości materiałów:
{{akcesorium-tkanina_welur}}

Dostępne kolory: {{color}}`,
      },
      {
        title: "Grid JPG - 3 kolumny",
        description: "Parametr :grid3 generuje obrazek JPG z 3 kolumnami (lepsze dla Allegro/Shoper)",
        code: `{{title}}

Dostępne tkaniny (wybierz spośród):
{{akcesorium-tkanina_welur:grid3}}

Wymiary: {{dimensions}}`,
      },
      {
        title: "Grid JPG - 4 kolumny",
        description: "Parametr :grid4 generuje obrazek JPG z 4 kolumnami (kompaktowy układ)",
        code: `Komoda {{dimensions}}

Tkanina (4 kolumny w siatce):
{{akcesorium-tkanina_welur:grid4}}

Wszystkie materiały są łatwe w czyszczeniu i odporne na ścieranie.`,
      },
      {
        title: "Grid JPG - 2×2 siatka",
        description: "Parametr :grid2-2 tworzy siatkę 2 wiersze × 2 kolumny",
        code: `Dostępne zawiasy (2×2):

{{akcesorium-zawiasy_meblowe:grid2-2}}

Wymiary: {{length}}×{{width}}×{{height}} cm`,
      },
      {
        title: "Mieszane - HTML i gridy JPG",
        description: "Możesz łączyć różne formaty w jednym szablonie",
        code: `Komoda {{title}}

Tkaniny (grid JPG 4 kolumny):
{{akcesorium-tkanina_welur:grid4}}

Prowadnice (HTML lista):
{{akcesorium-prowadnice_szuflad}}

Wymiary: {{dimensions}}
Cena: {{price}} PLN`,
      },
    ],
    loops: [
      {
        title: "Lista wszystkich akcesoriów",
        description: "Pętla przez wszystkie przypisane akcesoria",
        code: `{{title}}

Dostępne akcesoria ({{accessories.length}}):
{{#each accessories}}
• {{name}} ({{code}})
  {{description}}
{{/each}}`,
      },
      {
        title: "Akcesoria z obrazkami HTML",
        description: "Pętla z generowaniem HTML dla zdjęć",
        code: `<h2>{{title}}</h2>
<p>Wymiary: {{dimensions}}</p>

<h3>Dostępne akcesoria:</h3>
{{#each accessories}}
<div class="accessory">
  <img src="{{imageUrl}}" alt="{{name}}" />
  <h4>{{name}}</h4>
  <p>{{description}}</p>
  <p>Grupa: {{groupName}}</p>
</div>
{{/each}}`,
      },
      {
        title: "Grupowanie akcesoriów po grupach",
        description: "Wyświetlanie akcesoriów z nazwami grup",
        code: `Komoda {{title}}

Materiały i akcesoria:
{{#each accessories}}
{{#if @first}}
<strong>{{groupName}}:</strong>
{{/if}}
• {{name}} - {{description}}
{{/each}}`,
      },
    ],
    conditionals: [
      {
        title: "Warunek na podstawie koloru",
        description: "Różny tekst dla różnych kolorów",
        code: `{{title}}

{{#if (eq color "BIAŁY")}}
Ten produkt w kolorze białym idealnie pasuje do jasnych wnętrz.
{{else if (eq color "CZARNY")}}
Elegancki czarny kolor dodaje wnętrzu charakteru.
{{else}}
Dostępny w kolorze {{color}}.
{{/if}}`,
      },
      {
        title: "Warunek na liczbę akcesoriów",
        description: "Różny tekst gdy są/nie ma akcesoriów",
        code: `{{title}}

{{#if accessories.length}}
W zestawie znajdziesz {{accessories.length}} akcesoriów:
{{#each accessories}}
• {{name}}
{{/each}}
{{else}}
Produkt bez dodatkowych akcesoriów.
{{/if}}`,
      },
    ],
    sizeTables: [
      {
        title: "Tabela wymiarów HTML",
        description: "Profesjonalna tabela z wymiarami",
        code: `<h2>{{title}}</h2>

{{#size_table}}
<table class="size-table">
  <thead>
    <tr>
      <th>SKU</th>
      <th>Długość</th>
      <th>Szerokość</th>
      <th>Wysokość</th>
      <th>Cena</th>
    </tr>
  </thead>
  <tbody>
    {{#each combinations}}
    <tr>
      <td>{{sku}}</td>
      <td>{{length}} cm</td>
      <td>{{width}} cm</td>
      <td>{{height}} cm</td>
      <td>{{price}} PLN</td>
    </tr>
    {{/each}}
  </tbody>
</table>
{{/size_table}}`,
      },
      {
        title: "Tabela wymiarów prosta",
        description: "Prosta lista wariantów",
        code: `Dostępne rozmiary dla {{title}}:

{{#size_table}}
{{#each combinations}}
• {{dimensions}} - {{price}} PLN (SKU: {{sku}})
{{/each}}
{{/size_table}}`,
      },
    ],
    advanced: [
      {
        title: "Pełny szablon e-commerce",
        description: "Kompletny opis produktu z wszystkimi elementami",
        code: `<h1>{{title}}</h1>

<div class="product-intro">
  <p>Elegancka komoda w kolorze {{color}}, idealna do każdego wnętrza.</p>
  <p><strong>Wymiary:</strong> {{dimensions}}</p>
  <p><strong>Cena:</strong> {{price}} PLN</p>
</div>

<h2>Materiały</h2>
{{akcesorium-tkaniny_wotan}}

<h2>Dostępne akcesoria</h2>
{{#if accessories.length}}
<ul>
{{#each accessories}}
  <li>
    <strong>{{name}}</strong> ({{groupName}})
    <br>{{description}}
    {{#if imageUrl}}
    <br><img src="{{imageUrl}}" alt="{{name}}" style="max-width: 200px;">
    {{/if}}
  </li>
{{/each}}
</ul>
{{/if}}

<h2>Dostępne rozmiary</h2>
{{#size_table}}
<table>
  <tr><th>Wymiary</th><th>SKU</th><th>Cena</th></tr>
  {{#each combinations}}
  <tr>
    <td>{{dimensions}}</td>
    <td>{{sku}}</td>
    <td>{{price}} PLN</td>
  </tr>
  {{/each}}
</table>
{{/size_table}}`,
      },
    ],
  };

  const ExampleCard = ({ title, description, code }: { title: string; description: string; code: string }) => (
    <Card>
      <CardHeader className="pb-3">
        <div className="flex items-start justify-between">
          <div>
            <CardTitle className="text-sm">{title}</CardTitle>
            <CardDescription className="text-xs mt-1">{description}</CardDescription>
          </div>
          <Button
            variant="ghost"
            size="sm"
            onClick={() => copyExample(code)}
            data-testid={`button-copy-example-${title.toLowerCase().replace(/\s+/g, '-')}`}
          >
            <Copy className="h-4 w-4" />
          </Button>
        </div>
      </CardHeader>
      <CardContent>
        <pre className="text-xs bg-muted p-3 rounded-md overflow-x-auto">
          <code>{code}</code>
        </pre>
      </CardContent>
    </Card>
  );

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-4xl max-h-[85vh]" data-testid="dialog-examples">
        <DialogHeader>
          <DialogTitle>Przykłady użycia tagów w szablonach</DialogTitle>
          <DialogDescription>
            Kliknij przycisk kopiowania aby skopiować przykład do schowka
          </DialogDescription>
        </DialogHeader>

        <Tabs defaultValue="basic" className="flex-1">
          <TabsList className="grid w-full grid-cols-6">
            <TabsTrigger value="basic" data-testid="tab-basic">Podstawowe</TabsTrigger>
            <TabsTrigger value="accessories" data-testid="tab-accessories">Akcesoria</TabsTrigger>
            <TabsTrigger value="loops" data-testid="tab-loops">Pętle</TabsTrigger>
            <TabsTrigger value="conditionals" data-testid="tab-conditionals">Warunki</TabsTrigger>
            <TabsTrigger value="sizeTables" data-testid="tab-size-tables">Tabele</TabsTrigger>
            <TabsTrigger value="advanced" data-testid="tab-advanced">Zaawansowane</TabsTrigger>
          </TabsList>

          <ScrollArea className="h-[calc(85vh-12rem)] mt-4">
            <TabsContent value="basic" className="space-y-4 mt-0">
              {examples.basic.map((ex, idx) => (
                <ExampleCard key={idx} {...ex} />
              ))}
            </TabsContent>

            <TabsContent value="accessories" className="space-y-4 mt-0">
              {examples.accessories.map((ex, idx) => (
                <ExampleCard key={idx} {...ex} />
              ))}
            </TabsContent>

            <TabsContent value="loops" className="space-y-4 mt-0">
              {examples.loops.map((ex, idx) => (
                <ExampleCard key={idx} {...ex} />
              ))}
            </TabsContent>

            <TabsContent value="conditionals" className="space-y-4 mt-0">
              {examples.conditionals.map((ex, idx) => (
                <ExampleCard key={idx} {...ex} />
              ))}
            </TabsContent>

            <TabsContent value="sizeTables" className="space-y-4 mt-0">
              {examples.sizeTables.map((ex, idx) => (
                <ExampleCard key={idx} {...ex} />
              ))}
            </TabsContent>

            <TabsContent value="advanced" className="space-y-4 mt-0">
              {examples.advanced.map((ex, idx) => (
                <ExampleCard key={idx} {...ex} />
              ))}
            </TabsContent>
          </ScrollArea>
        </Tabs>
      </DialogContent>
    </Dialog>
  );
}
