import { pool } from './server/postgres';
import { generateProductionOrdersFromPlan } from './server/services/production/production-order-generator';

interface WorkOrder {
  id: number;
  work_order_number: string;
  production_order_id: number;
  order_number: string;
  sequence: number;
  quantity_planned: number;
  status: string;
  routing_operation_id: number | null;
  creates_buffer: boolean;
  buffer_location_id: number | null;
  location_name: string | null;
}

async function simulateProductionFlow() {
  console.log('\n🏭 ═══════════════════════════════════════════════════════════');
  console.log('   SYMULACJA KOMPLETNEGO PROCESU PRODUKCYJNEGO');
  console.log('═══════════════════════════════════════════════════════════\n');

  const planId = 14; // PLAN-0010
  
  // ═══════════════════════════════════════════════════════════
  // KROK 1: GENEROWANIE ZLP Z PLANU
  // ═══════════════════════════════════════════════════════════
  console.log('┌─────────────────────────────────────────────────────────┐');
  console.log('│ KROK 1: GENEROWANIE ZLP (Production Orders)            │');
  console.log('└─────────────────────────────────────────────────────────┘\n');
  
  // Sprawdź czy ZLP już wygenerowane (przez production_plan_lines)
  const existingZLP = await pool.query(`
    SELECT COUNT(DISTINCT po.id) as count
    FROM production.production_orders po
    JOIN production.production_plan_lines ppl ON ppl.production_order_id = po.id
    WHERE ppl.plan_id = $1
  `, [planId]);
  
  if (existingZLP.rows[0].count === '0') {
    console.log('⚠️  Brak ZLP dla planu - generuję...\n');
    const result = await generateProductionOrdersFromPlan(planId);
    
    if (!result.success) {
      console.error('❌ Błąd generowania:', result.errors);
      process.exit(1);
    }
    
    console.log('✅ Wygenerowano ZLP:');
    result.generatedOrders.forEach(order => {
      console.log(`   - ${order.orderNumber}: ${order.componentCount} komponentów`);
    });
  } else {
    console.log('✅ ZLP już wygenerowane (wykorzystuję istniejące)');
  }
  
  // Pokaż stan po generacji
  const zlpSummary = await pool.query(`
    SELECT 
      po.id,
      po.order_number,
      po.color_code,
      COUNT(DISTINCT pobi.id) as bom_items,
      COUNT(DISTINCT pwo.id) as work_orders
    FROM production.production_orders po
    JOIN production.production_plan_lines ppl ON ppl.production_order_id = po.id
    LEFT JOIN production.production_order_boms pob ON pob.production_order_id = po.id
    LEFT JOIN production.production_order_bom_items pobi ON pobi.production_order_bom_id = pob.id
    LEFT JOIN production.production_work_orders pwo ON pwo.production_order_id = po.id
    WHERE ppl.plan_id = $1
    GROUP BY po.id, po.order_number, po.color_code
    ORDER BY po.order_number
  `, [planId]);
  
  console.log('\n📊 Podsumowanie wygenerowanych ZLP:\n');
  zlpSummary.rows.forEach(row => {
    console.log(`   ${row.order_number}:`);
    console.log(`   ├─ BOM Items: ${row.bom_items} formatek (każda quantity=1)`);
    console.log(`   └─ Work Orders: ${row.work_orders} operacji do wykonania\n`);
  });
  
  // ═══════════════════════════════════════════════════════════
  // KROK 2: SPRAWDZENIE ROUTING RESOLUTION
  // ═══════════════════════════════════════════════════════════
  console.log('\n┌─────────────────────────────────────────────────────────┐');
  console.log('│ KROK 2: WERYFIKACJA ROUTING RESOLUTION                 │');
  console.log('└─────────────────────────────────────────────────────────┘\n');
  
  const routingCheck = await pool.query(`
    SELECT 
      po.order_number,
      rv.variant_code,
      rv.variant_name,
      pobi.required_operations,
      COUNT(*) as ile_komponentow
    FROM production.production_order_bom_items pobi
    JOIN production.production_order_boms pob ON pob.id = pobi.production_order_bom_id
    JOIN production.production_orders po ON po.id = pob.production_order_id
    JOIN production.production_plan_lines ppl ON ppl.production_order_id = po.id
    LEFT JOIN production.production_routing_variants rv ON rv.id = pobi.routing_variant_id
    WHERE ppl.plan_id = $1
    GROUP BY po.order_number, rv.variant_code, rv.variant_name, pobi.required_operations
    ORDER BY po.order_number, rv.variant_code
  `, [planId]);
  
  let currentOrder = '';
  routingCheck.rows.forEach(row => {
    if (row.order_number !== currentOrder) {
      console.log(`\n   ${row.order_number}:`);
      currentOrder = row.order_number;
    }
    const ops = JSON.parse(row.required_operations).join(' → ');
    console.log(`   ├─ ${row.variant_code}: ${row.ile_komponentow} szt`);
    console.log(`   │  └─ Operacje: ${ops}`);
  });
  
  // ═══════════════════════════════════════════════════════════
  // KROK 3: POBIERZ WORK ORDERS DO WYKONANIA
  // ═══════════════════════════════════════════════════════════
  console.log('\n\n┌─────────────────────────────────────────────────────────┐');
  console.log('│ KROK 3: ROZPOCZĘCIE PRODUKCJI - Work Orders             │');
  console.log('└─────────────────────────────────────────────────────────┘\n');
  
  const workOrders = await pool.query<WorkOrder>(`
    SELECT 
      pwo.id,
      pwo.work_order_number,
      pwo.production_order_id,
      po.order_number,
      pwo.sequence,
      pwo.quantity_planned,
      pwo.status,
      pwo.routing_operation_id,
      COALESCE(pro.creates_buffer, false) as creates_buffer,
      pro.buffer_location_id,
      pl.name as location_name
    FROM production.production_work_orders pwo
    JOIN production.production_orders po ON po.id = pwo.production_order_id
    JOIN production.production_plan_lines ppl ON ppl.production_order_id = po.id
    LEFT JOIN production.production_routing_operations pro ON pro.id = pwo.routing_operation_id
    LEFT JOIN production.production_locations pl ON pl.id = pro.buffer_location_id
    WHERE ppl.plan_id = $1
    ORDER BY po.order_number, pwo.sequence
  `, [planId]);
  
  console.log(`📋 Znaleziono ${workOrders.rows.length} Work Orders:\n`);
  
  currentOrder = '';
  workOrders.rows.forEach(wo => {
    if (wo.order_number !== currentOrder) {
      console.log(`\n   ${wo.order_number}:`);
      currentOrder = wo.order_number;
    }
    const bufferIndicator = wo.creates_buffer ? '📦→ BUFFER' : '';
    console.log(`   ${wo.sequence}. ${wo.work_order_number}: ${wo.quantity_planned} szt [${wo.status}] ${bufferIndicator}`);
    if (wo.creates_buffer && wo.location_name) {
      console.log(`      └─ Składowanie → ${wo.location_name}`);
    }
  });
  
  // ═══════════════════════════════════════════════════════════
  // KROK 4: SYMULACJA WYKONANIA PIERWSZEGO WORK ORDER
  // ═══════════════════════════════════════════════════════════
  console.log('\n\n┌─────────────────────────────────────────────────────────┐');
  console.log('│ KROK 4: SYMULACJA WYKONANIA WORK ORDER                 │');
  console.log('└─────────────────────────────────────────────────────────┘\n');
  
  // Wybierz pierwszy WO z buforem (jeśli istnieje)
  const woToExecute = workOrders.rows.find(wo => wo.creates_buffer) || workOrders.rows[0];
  
  if (!woToExecute) {
    console.log('⚠️  Brak Work Orders do wykonania');
  } else {
    console.log(`🎯 Wykonuję: ${woToExecute.work_order_number} (${woToExecute.order_number})\n`);
    console.log(`   Operacja: Seq ${woToExecute.sequence}`);
    console.log(`   Planowana ilość: ${woToExecute.quantity_planned} szt`);
    console.log(`   Tworzy bufor: ${woToExecute.creates_buffer ? 'TAK ✓' : 'NIE'}`);
    
    // Rozpocznij work order
    await pool.query(`
      UPDATE production.production_work_orders
      SET 
        status = 'in_progress',
        actual_start_time = NOW(),
        updated_at = NOW()
      WHERE id = $1
    `, [woToExecute.id]);
    
    console.log(`\n   ⏱️  Status: pending → in_progress`);
    console.log(`   🔧 Pracownik rozpoczął operację...`);
    
    // Symuluj wyprodukowanie (np. 100% planowanej ilości)
    const producedQty = parseFloat(woToExecute.quantity_planned.toString());
    
    await pool.query(`
      UPDATE production.production_work_orders
      SET 
        quantity_produced = $1,
        updated_at = NOW()
      WHERE id = $2
    `, [producedQty, woToExecute.id]);
    
    console.log(`   ✅ Wyprodukowano: ${producedQty} szt`);
    
    // ═══════════════════════════════════════════════════════════
    // KROK 5: ZAKOŃCZENIE WORK ORDER + BUFFER MOVEMENT
    // ═══════════════════════════════════════════════════════════
    if (woToExecute.creates_buffer && woToExecute.buffer_location_id) {
      console.log('\n   📦 AKTYWACJA BUFFER SYSTEM:');
      console.log(`   └─ Operacja tworzy półprodukt → składowanie w magazynie\n`);
      
      // Pobierz SKU półproduktu z BOM
      const bomInfo = await pool.query(`
        SELECT 
          pobi.component_name as sku,
          po.order_number
        FROM production.production_order_bom_items pobi
        JOIN production.production_order_boms pob ON pob.id = pobi.production_order_bom_id
        JOIN production.production_orders po ON po.id = pob.production_order_id
        WHERE po.id = $1
        LIMIT 1
      `, [woToExecute.production_order_id]);
      
      const productSku = bomInfo.rows[0]?.sku || `PANEL-${woToExecute.order_number}`;
      
      console.log(`   🏷️  Produkt SKU: ${productSku}`);
      console.log(`   📍 Lokalizacja: ${woToExecute.location_name}`);
      console.log(`   📊 Ilość: ${producedQty} szt\n`);
      
      // Utwórz ruch magazynowy IN
      const movementResult = await pool.query(`
        INSERT INTO production.production_buffer_movements (
          movement_type,
          product_sku,
          quantity,
          unit_of_measure,
          location_id,
          source_type,
          source_id,
          notes,
          created_at
        ) VALUES (
          'IN',
          $1,
          $2,
          'szt',
          $3,
          'WORK_ORDER',
          $4,
          'Automatyczne składowanie po zakończeniu operacji',
          NOW()
        )
        RETURNING id, movement_type, quantity
      `, [productSku, producedQty, woToExecute.buffer_location_id, woToExecute.id]);
      
      console.log(`   ✅ Utworzono ruch magazynowy:`);
      console.log(`      - ID: ${movementResult.rows[0].id}`);
      console.log(`      - Typ: ${movementResult.rows[0].movement_type} (wejście do magazynu)`);
      console.log(`      - Ilość: +${movementResult.rows[0].quantity} szt`);
      
      // Zaktualizuj stan magazynowy
      await pool.query(`
        INSERT INTO production.production_buffer_stock (
          product_sku,
          product_name,
          location_id,
          quantity_available,
          quantity_reserved,
          quantity_total,
          unit_of_measure,
          created_at,
          updated_at
        ) VALUES (
          $1,
          $1,
          $2,
          $3,
          0,
          $3,
          'szt',
          NOW(),
          NOW()
        )
        ON CONFLICT (product_sku, location_id) 
        DO UPDATE SET
          quantity_available = production_buffer_stock.quantity_available + $3,
          quantity_total = production_buffer_stock.quantity_total + $3,
          updated_at = NOW()
        RETURNING quantity_available, quantity_total
      `, [productSku, woToExecute.buffer_location_id, producedQty]);
      
      const stockState = await pool.query(`
        SELECT 
          quantity_available,
          quantity_reserved,
          quantity_total
        FROM production.production_buffer_stock
        WHERE product_sku = $1 AND location_id = $2
      `, [productSku, woToExecute.buffer_location_id]);
      
      console.log(`\n   📦 STAN MAGAZYNU po operacji:`);
      console.log(`      - Dostępne: ${stockState.rows[0].quantity_available} szt`);
      console.log(`      - Zarezerwowane: ${stockState.rows[0].quantity_reserved} szt`);
      console.log(`      - RAZEM: ${stockState.rows[0].quantity_total} szt`);
    }
    
    // Zakończ work order
    await pool.query(`
      UPDATE production.production_work_orders
      SET 
        status = 'completed',
        actual_end_time = NOW(),
        updated_at = NOW()
      WHERE id = $1
    `, [woToExecute.id]);
    
    console.log(`\n   ✅ Work Order zakończony: in_progress → completed`);
  }
  
  // ═══════════════════════════════════════════════════════════
  // KROK 6: SPRAWDZENIE STANU BUFORA
  // ═══════════════════════════════════════════════════════════
  console.log('\n\n┌─────────────────────────────────────────────────────────┐');
  console.log('│ KROK 6: STAN MAGAZYNU BUFOROWEGO                       │');
  console.log('└─────────────────────────────────────────────────────────┘\n');
  
  const bufferStock = await pool.query(`
    SELECT 
      bs.product_sku,
      bs.product_name,
      pl.name as location_name,
      bs.quantity_available,
      bs.quantity_reserved,
      bs.quantity_total,
      bs.unit_of_measure,
      bs.updated_at
    FROM production.production_buffer_stock bs
    LEFT JOIN production.production_locations pl ON pl.id = bs.location_id
    ORDER BY bs.updated_at DESC
    LIMIT 10
  `);
  
  if (bufferStock.rows.length === 0) {
    console.log('   📦 Magazyn buforowy pusty (brak półproduktów)\n');
  } else {
    console.log('   📦 Półprodukty w magazynie:\n');
    bufferStock.rows.forEach(stock => {
      console.log(`   ┌─ ${stock.product_sku}`);
      console.log(`   │  Lokalizacja: ${stock.location_name || 'BRAK'}`);
      console.log(`   │  Dostępne: ${stock.quantity_available} ${stock.unit_of_measure}`);
      console.log(`   │  Zarezerwowane: ${stock.quantity_reserved} ${stock.unit_of_measure}`);
      console.log(`   │  RAZEM: ${stock.quantity_total} ${stock.unit_of_measure}`);
      console.log(`   └─ Ostatnia aktualizacja: ${new Date(stock.updated_at).toLocaleString('pl-PL')}\n`);
    });
  }
  
  // Historia ruchów
  const movements = await pool.query(`
    SELECT 
      bm.id,
      bm.movement_type,
      bm.product_sku,
      bm.quantity,
      bm.unit_of_measure,
      pl.name as location_name,
      bm.source_type,
      bm.notes,
      bm.created_at
    FROM production.production_buffer_movements bm
    LEFT JOIN production.production_locations pl ON pl.id = bm.location_id
    ORDER BY bm.created_at DESC
    LIMIT 5
  `);
  
  if (movements.rows.length > 0) {
    console.log('\n   📜 Ostatnie ruchy magazynowe:\n');
    movements.rows.forEach(mov => {
      const icon = mov.movement_type === 'IN' ? '📥' : 
                   mov.movement_type === 'OUT' ? '📤' : 
                   mov.movement_type === 'RESERVE' ? '🔒' : '🔓';
      console.log(`   ${icon} ${mov.movement_type}: ${mov.quantity > 0 ? '+' : ''}${mov.quantity} ${mov.unit_of_measure} ${mov.product_sku}`);
      console.log(`      └─ ${mov.notes || 'Brak notatek'}`);
    });
  }
  
  // ═══════════════════════════════════════════════════════════
  // PODSUMOWANIE FINALNE
  // ═══════════════════════════════════════════════════════════
  console.log('\n\n┌═════════════════════════════════════════════════════════┐');
  console.log('│ 🎯 PODSUMOWANIE SYMULACJI                              │');
  console.log('└═════════════════════════════════════════════════════════┘\n');
  
  const finalSummary = await pool.query(`
    SELECT 
      COUNT(DISTINCT po.id) as zlp_count,
      COUNT(DISTINCT pwo.id) as total_work_orders,
      COUNT(DISTINCT CASE WHEN pwo.status = 'completed' THEN pwo.id END) as completed_wo,
      COUNT(DISTINCT CASE WHEN pwo.status = 'in_progress' THEN pwo.id END) as in_progress_wo,
      COUNT(DISTINCT CASE WHEN pwo.status = 'pending' THEN pwo.id END) as pending_wo,
      COUNT(DISTINCT pobi.id) as total_components
    FROM production.production_orders po
    JOIN production.production_plan_lines ppl ON ppl.production_order_id = po.id
    LEFT JOIN production.production_work_orders pwo ON pwo.production_order_id = po.id
    LEFT JOIN production.production_order_boms pob ON pob.production_order_id = po.id
    LEFT JOIN production.production_order_bom_items pobi ON pobi.production_order_bom_id = pob.id
    WHERE ppl.plan_id = $1
  `, [planId]);
  
  const summary = finalSummary.rows[0];
  
  console.log('   ✅ PROCES PRODUKCYJNY:');
  console.log(`      - Wygenerowano ZLP: ${summary.zlp_count}`);
  console.log(`      - Formatek do wyprodukowania: ${summary.total_components} szt (quantity=1 każda)`);
  console.log(`      - Work Orders łącznie: ${summary.total_work_orders}`);
  console.log(`      - Zakończone: ${summary.completed_wo}`);
  console.log(`      - W trakcie: ${summary.in_progress_wo}`);
  console.log(`      - Oczekujące: ${summary.pending_wo}`);
  
  const bufferCount = await pool.query(`
    SELECT 
      COUNT(DISTINCT bs.product_sku) as unique_products,
      SUM(bs.quantity_total) as total_in_buffer
    FROM production.production_buffer_stock bs
  `);
  
  console.log(`\n   📦 MAGAZYN BUFOROWY:`);
  console.log(`      - Rodzajów produktów: ${bufferCount.rows[0].unique_products}`);
  console.log(`      - Łączna ilość: ${bufferCount.rows[0].total_in_buffer || 0} szt`);
  
  console.log('\n\n═══════════════════════════════════════════════════════════');
  console.log('✨ SYMULACJA ZAKOŃCZONA POMYŚLNIE!');
  console.log('═══════════════════════════════════════════════════════════\n');
  
  await pool.end();
}

simulateProductionFlow().catch(err => {
  console.error('\n❌ BŁĄD SYMULACJI:', err);
  process.exit(1);
});
