import { sql } from "drizzle-orm";
import { pgTable, pgSchema, text, varchar, timestamp, decimal, jsonb, boolean, serial, index, integer, uniqueIndex, unique, type AnyPgColumn } from "drizzle-orm/pg-core";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";

export const allegroConnections = pgTable("allegro_connections", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  clientId: text("client_id").notNull(),
  clientSecret: text("client_secret").notNull(),
  accessToken: text("access_token"),
  refreshToken: text("refresh_token"),
  tokenExpiresAt: timestamp("token_expires_at"),
  isActive: boolean("is_active").default(false),
  connectionError: text("connection_error"),
  lastErrorAt: timestamp("last_error_at"),
  requiresReauth: boolean("requires_reauth").default(false),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// System Error Logs - centralized error tracking
export const errorLogs = pgTable("error_logs", {
  id: serial("id").primaryKey(),
  type: varchar("type", { length: 50 }).notNull(), // allegro, shoper, odoo, database, api, webhook, sync, etc.
  message: text("message").notNull(),
  stackTrace: text("stack_trace"),
  context: jsonb("context"), // {orderId, userId, endpoint, etc.}
  severity: varchar("severity", { length: 20 }).notNull().default("error"), // error, warning, info
  timestamp: timestamp("timestamp").notNull().defaultNow(),
  resolvedAt: timestamp("resolved_at"),
  resolvedBy: integer("resolved_by"), // user id who marked as resolved
  createdAt: timestamp("created_at").defaultNow(),
}, (table) => [
  index("idx_error_logs_type").on(table.type),
  index("idx_error_logs_severity").on(table.severity),
  index("idx_error_logs_timestamp").on(table.timestamp),
  index("idx_error_logs_resolved").on(table.resolvedAt),
]);

export const allegroOrders = pgTable("allegro_orders", {
  id: varchar("id").primaryKey(),
  allegroOrderId: text("allegro_order_id").notNull().unique(),
  buyerLogin: text("buyer_login").notNull(),
  buyerEmail: text("buyer_email"),
  totalAmount: decimal("total_amount", { precision: 10, scale: 2 }).notNull(),
  currency: text("currency").default("PLN"),
  paymentStatus: text("payment_status").notNull(),
  fulfillmentStatus: text("fulfillment_status"),
  itemsCount: text("items_count").notNull(),
  orderDate: timestamp("order_date").notNull(),
  paymentDate: timestamp("payment_date"),
  rawData: jsonb("raw_data"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const syncSettings = pgTable("sync_settings", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  autoRefreshEnabled: boolean("auto_refresh_enabled").default(true),
  refreshIntervalMinutes: text("refresh_interval_minutes").default("3"),
  lastSyncAt: timestamp("last_sync_at"),
  fullSyncEnabled: boolean("full_sync_enabled").default(true),
  fullSyncWindowHours: text("full_sync_window_hours").default("2"),
  shoperAutoRefreshEnabled: boolean("shoper_auto_refresh_enabled").default(true),
  shoperRefreshIntervalMinutes: text("shoper_refresh_interval_minutes").default("5"),
  lastShoperSyncAt: timestamp("last_shoper_sync_at"),
  shoperFullSyncEnabled: boolean("shoper_full_sync_enabled").default(true),
  shoperFullSyncWindowHours: text("shoper_full_sync_window_hours").default("2"),
  nightlySyncEnabled: boolean("nightly_sync_enabled").default(true),
  nightlySyncHour: integer("nightly_sync_hour").default(2),
  nightlySyncWindowDays: integer("nightly_sync_window_days").default(7),
  lastNightlySyncAt: timestamp("last_nightly_sync_at"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const orderChanges = pgTable("order_changes", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  orderId: varchar("order_id").notNull(),
  orderNumber: text("order_number"),
  source: text("source").notNull(),
  changeType: text("change_type").notNull(),
  fieldChanged: text("field_changed"),
  oldValue: text("old_value"),
  newValue: text("new_value"),
  detectedAt: timestamp("detected_at").defaultNow(),
  createdAt: timestamp("created_at").defaultNow(),
});

// Allegro Fees tables
export const allegroFees = pgTable("allegro_fees", {
  id: serial("id").primaryKey(),
  entryId: text("entry_id").unique(), // Unikalny ID z Allegro API
  typeId: varchar("type_id", { length: 50 }).notNull(), // SUC, LIS, NSP, DTR, etc.
  typeName: text("type_name").notNull(),
  amount: decimal("amount", { precision: 10, scale: 2 }).notNull(),
  currency: varchar("currency", { length: 10 }).default("PLN"),
  occurredAt: timestamp("occurred_at").notNull(),
  orderId: text("order_id"), // ID zamówienia jeśli dotyczy
  offerId: text("offer_id"), // ID oferty jeśli dotyczy
  rawData: jsonb("raw_data"), // Pełne dane z API
  category: varchar("category", { length: 50 }), // commission, ads, delivery, refund, other
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_allegro_fees_occurred_at").on(table.occurredAt),
  index("idx_allegro_fees_type_id").on(table.typeId),
  index("idx_allegro_fees_category").on(table.category),
]);

export const allegroFeeSummaries = pgTable("allegro_fee_summaries", {
  id: serial("id").primaryKey(),
  summaryDate: timestamp("summary_date").notNull(), // Data podsumowania (dzień)
  commissionTotal: decimal("commission_total", { precision: 10, scale: 2 }).default("0"),
  commissionCount: integer("commission_count").default(0),
  listingTotal: decimal("listing_total", { precision: 10, scale: 2 }).default("0"),
  listingCount: integer("listing_count").default(0),
  adsTotal: decimal("ads_total", { precision: 10, scale: 2 }).default("0"),
  adsCount: integer("ads_count").default(0),
  deliveryTotal: decimal("delivery_total", { precision: 10, scale: 2 }).default("0"),
  deliveryCount: integer("delivery_count").default(0),
  refundsTotal: decimal("refunds_total", { precision: 10, scale: 2 }).default("0"),
  refundsCount: integer("refunds_count").default(0),
  otherTotal: decimal("other_total", { precision: 10, scale: 2 }).default("0"),
  otherCount: integer("other_count").default(0),
  grandTotal: decimal("grand_total", { precision: 10, scale: 2 }).default("0"),
  currency: varchar("currency", { length: 10 }).default("PLN"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  uniqueIndex("idx_allegro_fee_summaries_date").on(table.summaryDate),
]);

export const insertAllegroConnectionSchema = createInsertSchema(allegroConnections).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertErrorLogSchema = createInsertSchema(errorLogs).omit({
  id: true,
  createdAt: true,
});

export const insertAllegroOrderSchema = createInsertSchema(allegroOrders).omit({
  createdAt: true,
  updatedAt: true,
});

export const insertSyncSettingsSchema = createInsertSchema(syncSettings).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertOrderChangeSchema = createInsertSchema(orderChanges).omit({
  id: true,
  createdAt: true,
});

export const insertAllegroFeeSchema = createInsertSchema(allegroFees).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertAllegroFeeSummarySchema = createInsertSchema(allegroFeeSummaries).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type AllegroConnection = typeof allegroConnections.$inferSelect;
export type InsertAllegroConnection = z.infer<typeof insertAllegroConnectionSchema>;

export type ErrorLog = typeof errorLogs.$inferSelect;
export type InsertErrorLog = z.infer<typeof insertErrorLogSchema>;

export type AllegroOrder = typeof allegroOrders.$inferSelect & {
  orderNumber?: number;
  source?: string;
};
export type InsertAllegroOrder = z.infer<typeof insertAllegroOrderSchema>;

export type SyncSettings = typeof syncSettings.$inferSelect;
export type InsertSyncSettings = z.infer<typeof insertSyncSettingsSchema>;

export type OrderChange = typeof orderChanges.$inferSelect;
export type InsertOrderChange = z.infer<typeof insertOrderChangeSchema>;

export type AllegroFee = typeof allegroFees.$inferSelect;
export type InsertAllegroFee = z.infer<typeof insertAllegroFeeSchema>;

export type AllegroFeeSummary = typeof allegroFeeSummaries.$inferSelect;
export type InsertAllegroFeeSummary = z.infer<typeof insertAllegroFeeSummarySchema>;

export const orderStatusSchema = z.enum([
  "PAID",
  "UNPAID",
  "PENDING",
  "CANCELLED",
]);

export const fulfillmentStatusSchema = z.enum([
  "NEW",
  "PROCESSING",
  "READY_FOR_SHIPMENT",
  "SENT",
  "DELIVERED",
  "CANCELLED",
]);

export type OrderStatus = z.infer<typeof orderStatusSchema>;
export type FulfillmentStatus = z.infer<typeof fulfillmentStatusSchema>;

// Auth tables
export const sessions = pgTable(
  "sessions",
  {
    sid: varchar("sid").primaryKey(),
    sess: jsonb("sess").notNull(),
    expire: timestamp("expire").notNull(),
  },
  (table) => [index("IDX_session_expire").on(table.expire)],
);

export const users = pgTable("users", {
  id: serial("id").primaryKey(),
  username: varchar("username", { length: 255 }).notNull().unique(),
  email: varchar("email", { length: 255 }).notNull().unique(),
  password: text("password").notNull(),
  firstName: varchar("first_name", { length: 255 }),
  lastName: varchar("last_name", { length: 255 }),
  role: varchar("role", { length: 50 }).notNull().default("user"),
  permissions: jsonb("permissions").default(sql`'[]'::jsonb`),
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const passwordResetTokens = pgTable("password_reset_tokens", {
  id: serial("id").primaryKey(),
  userId: integer("user_id").notNull().references(() => users.id),
  token: varchar("token", { length: 255 }).notNull().unique(),
  expiresAt: timestamp("expires_at").notNull(),
  used: boolean("used").default(false),
  createdAt: timestamp("created_at").defaultNow(),
});

// User Saved Filters - persistent filter presets per user
export const userSavedFilters = pgTable("user_saved_filters", {
  id: serial("id").primaryKey(),
  userId: integer("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  context: varchar("context", { length: 50 }).notNull(), // 'catalog_products', 'cutting_patterns', 'orders', etc.
  name: varchar("name", { length: 255 }).notNull(), // User-defined name
  filterData: jsonb("filter_data").notNull(), // JSON with filter state
  isLastUsed: boolean("is_last_used").default(false), // Auto-load on next visit
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_user_saved_filters_user").on(table.userId),
  index("idx_user_saved_filters_context").on(table.context),
  index("idx_user_saved_filters_last_used").on(table.userId, table.context, table.isLastUsed),
]);

export const insertUserSchema = createInsertSchema(users, {
  username: z.string().min(3, "Username musi mieć minimum 3 znaki").max(255),
  email: z.string().email("Nieprawidłowy adres email"),
  password: z.string().min(6, "Hasło musi mieć minimum 6 znaków"),
}).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertPasswordResetTokenSchema = createInsertSchema(passwordResetTokens).omit({
  id: true,
  createdAt: true,
});

// User Saved Filters schemas
export const filterContextSchema = z.enum(["catalog_products", "cutting_patterns", "orders", "marketplace_orders"]);
export type FilterContext = z.infer<typeof filterContextSchema>;

export const insertUserSavedFilterSchema = createInsertSchema(userSavedFilters, {
  name: z.string().min(1, "Nazwa filtru jest wymagana").max(255),
  context: filterContextSchema,
  filterData: z.record(z.any()), // Flexible JSON structure
}).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type InsertUserSavedFilter = z.infer<typeof insertUserSavedFilterSchema>;
export type UserSavedFilter = typeof userSavedFilters.$inferSelect;

// User Page Settings - persistent page-specific settings per user (filters, column visibility, column order)
export const userPageSettings = pgTable("user_page_settings", {
  id: serial("id").primaryKey(),
  userId: integer("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  pageSlug: varchar("page_slug", { length: 100 }).notNull(), // 'warehouse-formatki', 'warehouse-plyty', etc.
  filters: jsonb("filters").default(sql`'{}'::jsonb`), // {selectedColors: [], selectedLengths: [], stockFilter: 'all', etc.}
  columnVisibility: jsonb("column_visibility").default(sql`'{}'::jsonb`), // {name: true, quantity: true, color: false, etc.}
  columnOrder: jsonb("column_order").default(sql`'[]'::jsonb`), // ['name', 'quantity', 'color', ...]
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_user_page_settings_user").on(table.userId),
  index("idx_user_page_settings_page").on(table.pageSlug),
  uniqueIndex("idx_user_page_settings_user_page").on(table.userId, table.pageSlug),
]);

export const insertUserPageSettingsSchema = createInsertSchema(userPageSettings, {
  pageSlug: z.string().min(1).max(100),
  filters: z.record(z.any()).optional(),
  columnVisibility: z.record(z.boolean()).optional(),
  columnOrder: z.array(z.string()).optional(),
}).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type InsertUserPageSettings = z.infer<typeof insertUserPageSettingsSchema>;
export type UserPageSettings = typeof userPageSettings.$inferSelect;

export const userRoleSchema = z.enum(["admin", "manager", "user"]);
export type UserRole = z.infer<typeof userRoleSchema>;

export const permissionSchema = z.enum([
  "orders.view",
  "orders.edit",
  "orders.delete",
  "settings.view",
  "settings.edit",
  "users.view",
  "users.edit",
  "sync.run",
]);
export type Permission = z.infer<typeof permissionSchema>;

export type User = typeof users.$inferSelect & {
  role: UserRole;
  permissions: Permission[];
};
export type InsertUser = z.infer<typeof insertUserSchema>;
export type PasswordResetToken = typeof passwordResetTokens.$inferSelect;
export type InsertPasswordResetToken = z.infer<typeof insertPasswordResetTokenSchema>;

// API Tokens for external integrations (e.g., Odoo)
export const apiTokens = pgTable("api_tokens", {
  id: serial("id").primaryKey(),
  name: varchar("name", { length: 255 }).notNull(),
  description: text("description"),
  token: text("token").notNull().unique(),
  tokenPrefix: varchar("token_prefix", { length: 16 }),
  permissions: jsonb("permissions").default(sql`'[]'::jsonb`),
  isActive: boolean("is_active").default(true),
  lastUsedAt: timestamp("last_used_at"),
  expiresAt: timestamp("expires_at"),
  createdByUserId: integer("created_by_user_id").references(() => users.id),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Webhook configurations for pushing updates to external systems
export const webhookConfigs = pgTable("webhook_configs", {
  id: serial("id").primaryKey(),
  name: varchar("name", { length: 255 }).notNull(),
  url: text("url").notNull(),
  secret: text("secret"),
  events: jsonb("events").default(sql`'[]'::jsonb`),
  isActive: boolean("is_active").default(true),
  retryAttempts: integer("retry_attempts").default(3),
  lastTriggeredAt: timestamp("last_triggered_at"),
  createdBy: integer("created_by").references(() => users.id),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// API request logs for audit trail
export const apiRequestLogs = pgTable("api_request_logs", {
  id: serial("id").primaryKey(),
  tokenId: integer("token_id").references(() => apiTokens.id),
  method: varchar("method", { length: 10 }).notNull(),
  path: text("path").notNull(),
  statusCode: integer("status_code"),
  responseTime: integer("response_time"),
  ipAddress: varchar("ip_address", { length: 45 }),
  userAgent: text("user_agent"),
  createdAt: timestamp("created_at").defaultNow(),
});

// Webhook delivery logs
export const webhookLogs = pgTable("webhook_logs", {
  id: serial("id").primaryKey(),
  webhookId: integer("webhook_id").references(() => webhookConfigs.id),
  event: varchar("event", { length: 100 }).notNull(),
  payload: jsonb("payload"),
  statusCode: integer("status_code"),
  responseBody: text("response_body"),
  attemptNumber: integer("attempt_number").default(1),
  success: boolean("success").default(false),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertApiTokenSchema = createInsertSchema(apiTokens).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertWebhookConfigSchema = createInsertSchema(webhookConfigs).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type ApiToken = typeof apiTokens.$inferSelect;
export type InsertApiToken = z.infer<typeof insertApiTokenSchema>;
export type WebhookConfig = typeof webhookConfigs.$inferSelect;
export type InsertWebhookConfig = z.infer<typeof insertWebhookConfigSchema>;
export type ApiRequestLog = typeof apiRequestLogs.$inferSelect;
export type WebhookLog = typeof webhookLogs.$inferSelect;

// Shoper Configuration
export const shoperConfig = pgTable("shoper_config", {
  id: serial("id").primaryKey(),
  ordersUrl: text("orders_url").notNull().default("https://alpmeb.pl/admin/orders/"),
  productsUrl: text("products_url").notNull().default("https://sklep378098.shoparena.pl/admin/products/edit/id/"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Odoo Integration tables
export const odooConfig = pgTable("odoo_config", {
  id: serial("id").primaryKey(),
  url: text("url").notNull(), // URL dla synchronizacji (API) - np. https://alpma.app/odoo
  userUrl: text("user_url"), // URL dla linków klikanych przez użytkowników - np. http://100.99.76.111:8070/odoo
  omsDomain: text("oms_domain").default("alp-oms.replit.app"), // Domena OMS do generowania linków zwrotnych
  database: varchar("database", { length: 255 }).notNull(),
  username: varchar("username", { length: 255 }).notNull(),
  isActive: boolean("is_active").default(true),
  autoConfirmOrders: boolean("auto_confirm_orders").default(false),
  syncIntervalMinutes: integer("sync_interval_minutes").default(3),
  lastSyncAt: timestamp("last_sync_at"),
  lastSyncStatus: varchar("last_sync_status", { length: 50 }),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const odooSyncQueue = pgTable("odoo_sync_queue", {
  id: serial("id").primaryKey(),
  orderNumber: integer("order_number").notNull(), // UNIQUE constraint already exists in DB
  source: varchar("source", { length: 20 }).notNull(),
  operation: varchar("operation", { length: 20 }).notNull().default("create"),
  status: varchar("status", { length: 50 }).notNull().default("pending"),
  attempts: integer("attempts").default(0),
  maxAttempts: integer("max_attempts").default(3),
  lastError: text("last_error"),
  odooOrderId: integer("odoo_order_id"),
  scheduledAt: timestamp("scheduled_at").defaultNow(),
  processedAt: timestamp("processed_at"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const odooSyncLogs = pgTable("odoo_sync_logs", {
  id: serial("id").primaryKey(),
  queueId: integer("queue_id").references(() => odooSyncQueue.id),
  orderNumber: integer("order_number"),
  operation: varchar("operation", { length: 20 }).notNull(),
  status: varchar("status", { length: 50 }).notNull(),
  message: text("message"),
  errorDetails: jsonb("error_details"),
  requestPayload: jsonb("request_payload"),
  responseData: jsonb("response_data"),
  duration: integer("duration"),
  odooOrderId: integer("odoo_order_id"),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertShoperConfigSchema = createInsertSchema(shoperConfig).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertOdooConfigSchema = createInsertSchema(odooConfig).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertOdooSyncQueueSchema = createInsertSchema(odooSyncQueue).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertOdooSyncLogSchema = createInsertSchema(odooSyncLogs).omit({
  id: true,
  createdAt: true,
});

export type ShoperConfig = typeof shoperConfig.$inferSelect;
export type InsertShoperConfig = z.infer<typeof insertShoperConfigSchema>;
export type OdooConfig = typeof odooConfig.$inferSelect;
export type InsertOdooConfig = z.infer<typeof insertOdooConfigSchema>;
export type OdooSyncQueue = typeof odooSyncQueue.$inferSelect;
export type InsertOdooSyncQueue = z.infer<typeof insertOdooSyncQueueSchema>;
export type OdooSyncLog = typeof odooSyncLogs.$inferSelect;
export type InsertOdooSyncLog = z.infer<typeof insertOdooSyncLogSchema>;

// Background Job Queue
export const jobs = pgTable("jobs", {
  id: serial("id").primaryKey(),
  jobType: varchar("job_type", { length: 50 }).notNull(), // 'generate_bom', 'render_formatka'
  payload: jsonb("payload").notNull(), // { productId, skipFormatkaRendering, etc }
  status: varchar("status", { length: 20 }).notNull().default("pending"), // 'pending', 'processing', 'completed', 'failed'
  priority: integer("priority").default(0), // Higher number = higher priority
  attempts: integer("attempts").default(0),
  maxAttempts: integer("max_attempts").default(3),
  error: text("error"),
  createdAt: timestamp("created_at").defaultNow(),
  startedAt: timestamp("started_at"),
  completedAt: timestamp("completed_at"),
});

export const insertJobSchema = createInsertSchema(jobs).omit({
  id: true,
  createdAt: true,
  startedAt: true,
  completedAt: true,
});

export type Job = typeof jobs.$inferSelect;
export type InsertJob = z.infer<typeof insertJobSchema>;

export const jobTypeSchema = z.enum(["generate_bom", "render_formatka"]);
export const jobStatusSchema = z.enum(["pending", "processing", "completed", "failed"]);

// ==================== Catalog Schema ====================
// Dedykowany schemat PostgreSQL dla katalogu produktów
export const catalogSchema = pgSchema("catalog");

// Product Catalog Tables
export const products = catalogSchema.table("products", {
  id: serial("id").primaryKey(),
  sku: varchar("sku", { length: 100 }).notNull().unique(),
  skuAirtable: varchar("sku_airtable", { length: 50 }), // Tymczasowe SKU z Airtable przed migracją
  
  // Wspólne pola (synchronizowane między platformami)
  title: text("title").notNull(),
  shortDescription: text("short_description"),
  longDescriptionHtml: text("long_description_html"),
  longDescriptionDoc: jsonb("long_description_doc"),
  gallery: jsonb("gallery"), // Array of image URLs
  
  // Wymiary i waga
  length: decimal("length", { precision: 10, scale: 2 }),
  width: decimal("width", { precision: 10, scale: 2 }),
  height: decimal("height", { precision: 10, scale: 2 }),
  weight: decimal("weight", { precision: 10, scale: 2 }),
  
  // Kolory i materiały
  color: text("color"),
  colorOptions: text("color_options").array().notNull().default(sql`ARRAY[]::text[]`), // Opcje kolorystyczne: ["LISTWA-P-WOTAN", "DRZWI-CZARNY"]
  material: text("material"),
  
  // Rodzaj mebla (ze słownika product_type)
  productType: text("product_type"),
  
  // Grupa produktu (ze słownika product_group: A1, A2, A3, A4)
  productGroup: text("product_group"),
  
  // Drzwi (ze słownika door: D1, D2, D3, D4)
  doors: text("doors"),
  
  // Nóżki (ze słownika leg: N1, N2, N3, N4)
  legs: text("legs"),
  
  // Cena bazowa
  basePrice: decimal("base_price", { precision: 10, scale: 2 }),
  currency: varchar("currency", { length: 3 }).default("PLN"),
  
  // Status
  isActive: boolean("is_active").default(true),
  
  // Matrix generation
  matrixId: integer("matrix_id").references(() => productMatrices.id, { onDelete: "set null" }),
  combinationKey: text("combination_key"), // Unique key: matrixId:lengthIdx:widthIdx:heightIdx:colorIdx
  generatedFromMatrix: boolean("generated_from_matrix").default(false),
  matrixVariantId: text("matrix_variant_id"), // Stable variant identifier (survives SKU/dimension changes): "matrixId-length-width-height-color"
  
  // Odoo integration
  odooProductId: integer("odoo_product_id"), // ID produktu w Odoo
  odooExportedAt: timestamp("odoo_exported_at"), // Ostatni eksport do Odoo
  odooExportError: text("odoo_export_error"), // Błąd ostatniego eksportu (null = sukces)
  
  // Metadata
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => ({
  uniqueMatrixCombination: uniqueIndex("unique_matrix_combination").on(table.matrixId, table.combinationKey),
}));

export const productPlatformData = catalogSchema.table("product_platform_data", {
  id: serial("id").primaryKey(),
  productId: integer("product_id").references(() => products.id, { onDelete: "cascade" }),
  platform: varchar("platform", { length: 20 }).notNull(), // 'allegro', 'shoper', 'odoo'
  
  // Link type discriminator - marketplace can link to either product or set
  linkType: varchar("link_type", { length: 10 }).default("product"), // 'product' | 'set'
  setId: integer("set_id"), // Reference to product_creator.product_sets (will be added later with cross-schema reference)
  
  // Tytuł specyficzny dla platformy (może być edytowany)
  customTitle: text("custom_title"),
  
  // Kategoria specyficzna dla platformy
  categoryId: text("category_id"),
  categoryName: text("category_name"),
  
  // Opis specyficzny dla platformy
  description: text("description"),
  inheritsSharedDescription: boolean("inherits_shared_description").default(true),
  
  // Cena specyficzna dla platformy
  customPrice: decimal("custom_price", { precision: 10, scale: 2 }),
  
  // Parametry specyficzne dla platformy (JSON)
  customAttributes: jsonb("custom_attributes"),
  
  // Status publikacji
  isPublished: boolean("is_published").default(false),
  publishedAt: timestamp("published_at"),
  
  // ID w systemie zewnętrznym
  externalId: text("external_id"),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const productImages = catalogSchema.table("product_images", {
  id: serial("id").primaryKey(),
  productId: integer("product_id").references(() => products.id, { onDelete: "cascade" }).notNull(),
  
  // Ścieżka do pliku
  url: text("url").notNull(),
  filename: text("filename").notNull(),
  
  // Optimized image versions (generated automatically)
  thumbnailUrl: text("thumbnail_url"),
  mediumUrl: text("medium_url"),
  
  // Typ zdjęcia
  imageType: varchar("image_type", { length: 30 }).notNull(), // 'packshot', 'render', 'visualization'
  
  // Atrybuty SEO
  altText: text("alt_text"),
  title: text("title"),
  
  // Tagi (wielokrotny wybór)
  tags: text("tags").array().default(sql`ARRAY[]::text[]`),
  
  // Kolejność wyświetlania
  sortOrder: integer("sort_order").default(0),
  
  // Czy to główne zdjęcie (packshot na listę)
  isPrimary: boolean("is_primary").default(false),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Set Images - obrazy dla zestawów (catalog sets)
export const setImages = catalogSchema.table("set_images", {
  id: serial("id").primaryKey(),
  setId: integer("set_id").notNull(), // FK do product_creator.product_sets (cross-schema)
  
  // Ścieżka do pliku
  url: text("url").notNull(),
  filename: text("filename").notNull(),
  
  // Optimized image versions (generated automatically)
  thumbnailUrl: text("thumbnail_url"),
  mediumUrl: text("medium_url"),
  
  // Typ zdjęcia
  imageType: varchar("image_type", { length: 30 }).notNull(), // 'packshot', 'render', 'visualization'
  
  // Atrybuty SEO
  altText: text("alt_text"),
  title: text("title"),
  
  // Tagi (wielokrotny wybór)
  tags: text("tags").array().default(sql`ARRAY[]::text[]`),
  
  // Kolejność wyświetlania
  sortOrder: integer("sort_order").default(0),
  
  // Czy to główne zdjęcie (packshot na listę)
  isPrimary: boolean("is_primary").default(false),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const productAddons = catalogSchema.table("product_addons", {
  id: serial("id").primaryKey(),
  
  // Typ dodatku
  addonType: varchar("addon_type", { length: 50 }).notNull(), // 'fabric', 'board', 'certificate', 'accessory', 'component'
  
  // Nazwa dodatku
  name: text("name").notNull(),
  
  // Kod/SKU dodatku
  code: varchar("code", { length: 100 }),
  
  // Opis dodatku
  description: text("description"),
  
  // Szablon HTML do wstawienia w opis produktu
  htmlTemplate: text("html_template"),
  
  // Parametry dodatku (JSON)
  attributes: jsonb("attributes"),
  
  // Status
  isActive: boolean("is_active").default(true),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const addonImages = catalogSchema.table("addon_images", {
  id: serial("id").primaryKey(),
  addonId: integer("addon_id").references(() => productAddons.id, { onDelete: "cascade" }).notNull(),
  
  // Ścieżka do pliku
  url: text("url").notNull(),
  filename: text("filename").notNull(),
  
  // Atrybuty SEO
  altText: text("alt_text"),
  title: text("title"),
  
  // Kolejność wyświetlania
  sortOrder: integer("sort_order").default(0),
  
  createdAt: timestamp("created_at").defaultNow(),
});

export const productAddonAssignments = catalogSchema.table("product_addon_assignments", {
  id: serial("id").primaryKey(),
  productId: integer("product_id").references(() => products.id, { onDelete: "cascade" }).notNull(),
  addonId: integer("addon_id").references(() => productAddons.id, { onDelete: "cascade" }).notNull(),
  
  // Kolejność w opisie produktu
  sortOrder: integer("sort_order").default(0),
  
  // Parametry nadpisane dla tego przypisania
  customAttributes: jsonb("custom_attributes"),
  
  createdAt: timestamp("created_at").defaultNow(),
});

// Template Categories for organizing description templates
export const descriptionTemplateCategories = catalogSchema.table("description_template_categories", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(),
  description: text("description"),
  sortOrder: integer("sort_order").default(0),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Description Templates with Tiptap support and variables
export const descriptionTemplates = catalogSchema.table("description_templates", {
  id: serial("id").primaryKey(),
  
  // Nazwa szablonu
  name: text("name").notNull(),
  
  // Kategoria szablonu (FK)
  categoryId: integer("category_id").references(() => descriptionTemplateCategories.id, { onDelete: "set null" }),
  
  // Typ szablonu (deprecated - użyj categoryId zamiast tego)
  templateType: varchar("template_type", { length: 50 }).notNull(), // 'banner', 'certificate', 'footer', 'custom'
  
  // Treść szablonu jako dokument Tiptap (JSON)
  contentDoc: jsonb("content_doc"),
  
  // Treść szablonu HTML (generowany z contentDoc)
  htmlContent: text("html_content").notNull(),
  
  // Metadata o zmiennych używanych w szablonie
  variables: jsonb("variables"), // { "color": { type: "string", required: true }, "addons": { type: "array", required: false } }
  
  // Twórca szablonu
  createdBy: integer("created_by").references(() => users.id, { onDelete: "set null" }),
  
  // Czy szablon jest globalny (stosowany automatycznie)
  isGlobal: boolean("is_global").default(false),
  
  // Status
  isActive: boolean("is_active").default(true),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Junction table: Products ↔ Templates
export const productDescriptionTemplates = catalogSchema.table("product_description_templates", {
  id: serial("id").primaryKey(),
  productId: integer("product_id").references(() => products.id, { onDelete: "cascade" }).notNull(),
  templateId: integer("template_id").references(() => descriptionTemplates.id, { onDelete: "cascade" }).notNull(),
  
  // Konfiguracja renderowania (np. mapowanie zmiennych)
  renderConfig: jsonb("render_config"),
  
  createdAt: timestamp("created_at").defaultNow(),
});

// AI Generation Requests tracking
export const aiGenerationRequests = catalogSchema.table("ai_generation_requests", {
  id: serial("id").primaryKey(),
  productId: integer("product_id").references(() => products.id, { onDelete: "cascade" }).notNull(),
  templateId: integer("template_id").references(() => descriptionTemplates.id, { onDelete: "set null" }),
  
  // Status: 'pending', 'processing', 'completed', 'failed'
  status: varchar("status", { length: 20 }).notNull().default("pending"),
  
  // Prompt wysłany do AI
  prompt: text("prompt"),
  
  // Odpowiedź od AI
  response: text("response"),
  
  // Metadata o kosztach (tokens, cost, model)
  costMetadata: jsonb("cost_metadata"),
  
  // Twórca requestu
  createdBy: integer("created_by").references(() => users.id, { onDelete: "set null" }),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Insert schemas
export const insertProductSchema = createInsertSchema(products).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductPlatformDataSchema = createInsertSchema(productPlatformData).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductImageSchema = createInsertSchema(productImages).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductAddonSchema = createInsertSchema(productAddons).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertAddonImageSchema = createInsertSchema(addonImages).omit({
  id: true,
  createdAt: true,
});

export const insertProductAddonAssignmentSchema = createInsertSchema(productAddonAssignments).omit({
  id: true,
  createdAt: true,
});

export const insertDescriptionTemplateSchema = createInsertSchema(descriptionTemplates).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertDescriptionTemplateCategorySchema = createInsertSchema(descriptionTemplateCategories).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductDescriptionTemplateSchema = createInsertSchema(productDescriptionTemplates).omit({
  id: true,
  createdAt: true,
});

export const insertAiGenerationRequestSchema = createInsertSchema(aiGenerationRequests).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

// AI Prompt Settings (global configuration for AI description generation)
export const aiPromptSettings = pgTable("ai_prompt_settings", {
  id: serial("id").primaryKey(),
  systemInstructions: text("system_instructions").default(""),
  productMaterialInfo: text("product_material_info").default("płyta meblowa 18mm renomowanych polskich firm"),
  edgeBandInfo: text("edge_band_info").default("obrzeże ABS 0.8mm"),
  includeProductTable: boolean("include_product_table").default(true),
  productTableHeader: text("product_table_header").default("Wymiary produktu"),
  topBannerImageId: integer("top_banner_image_id"),
  bottomBannerImageId: integer("bottom_banner_image_id"),
  customInstructions: text("custom_instructions").default(""),
  
  // AI Section Prompts - base prompts for different description sections
  promptIntro: text("prompt_intro"),
  promptFeatures: text("prompt_features"),
  promptSafety: text("prompt_safety"),
  promptCare: text("prompt_care"),
  promptWarranty: text("prompt_warranty"),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const insertAiPromptSettingsSchema = createInsertSchema(aiPromptSettings).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

// Types
export type Product = typeof products.$inferSelect;
export type InsertProduct = z.infer<typeof insertProductSchema>;

export type ProductPlatformData = typeof productPlatformData.$inferSelect;
export type InsertProductPlatformData = z.infer<typeof insertProductPlatformDataSchema>;

// ProductPlatformData enriched with marketplace link fields from JOIN (for GET /api/catalog-products/:id)
export type ProductPlatformDataWithMarketplace = ProductPlatformData & {
  marketplaceProductId?: number | null;
  marketplaceName?: string | null;
  marketplaceSku?: string | null;
};

export type ProductImage = typeof productImages.$inferSelect;
export type InsertProductImage = z.infer<typeof insertProductImageSchema>;

export type ProductAddon = typeof productAddons.$inferSelect;
export type InsertProductAddon = z.infer<typeof insertProductAddonSchema>;

export type AddonImage = typeof addonImages.$inferSelect;
export type InsertAddonImage = z.infer<typeof insertAddonImageSchema>;

export type ProductAddonAssignment = typeof productAddonAssignments.$inferSelect;
export type InsertProductAddonAssignment = z.infer<typeof insertProductAddonAssignmentSchema>;

export type DescriptionTemplate = typeof descriptionTemplates.$inferSelect;
export type InsertDescriptionTemplate = z.infer<typeof insertDescriptionTemplateSchema>;

export type DescriptionTemplateCategory = typeof descriptionTemplateCategories.$inferSelect;
export type InsertDescriptionTemplateCategory = z.infer<typeof insertDescriptionTemplateCategorySchema>;

export type ProductDescriptionTemplate = typeof productDescriptionTemplates.$inferSelect;
export type InsertProductDescriptionTemplate = z.infer<typeof insertProductDescriptionTemplateSchema>;

export type AiGenerationRequest = typeof aiGenerationRequests.$inferSelect;
export type InsertAiGenerationRequest = z.infer<typeof insertAiGenerationRequestSchema>;

export type AiPromptSettings = typeof aiPromptSettings.$inferSelect;
export type InsertAiPromptSettings = z.infer<typeof insertAiPromptSettingsSchema>;

// ==================== Product Creator Schema ====================
// Dedykowany schemat PostgreSQL dla modułu Kreator Produktu
export const productCreatorSchema = pgSchema("product_creator");

// Product Creator - Dictionary Tables
export const productCreatorDictionaries = productCreatorSchema.table("dictionaries", {
  id: serial("id").primaryKey(),
  
  // Typ słownika: 'product_type', 'unit', 'color', 'product_group', 'door', 'leg', 'fabric'
  dictionaryType: varchar("dictionary_type", { length: 50 }).notNull(),
  
  // Kod techniczny (wielkie litery, bez polskich znaków) - np. SUPRA, BOCZKI, BIALY
  code: varchar("code", { length: 100 }).notNull(),
  
  // Nazwa (do wyświetlenia)
  name: text("name").notNull(),
  
  // Nazwa czytelna/opisowa - np. "Dąb Lancelot", "Dąb Wotan"
  readableName: text("readable_name"),
  
  // Nazwa krótka (do budowania nazw produktów) - np. "SUPRA", "D1", "N2"
  shortName: text("short_name"),
  
  // Opis
  description: text("description"),
  
  // Kolejność sortowania
  sortOrder: integer("sort_order").default(0),
  
  // Kategoria (opcjonalna)
  category: text("category"),
  
  // Ścieżka do zdjęcia
  imageUrl: text("image_url"),
  
  // Kolor HEX dla etykiet (np. #6366F1)
  color: varchar("color", { length: 7 }),
  
  // Status
  isActive: boolean("is_active").default(true),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const insertProductCreatorDictionarySchema = createInsertSchema(productCreatorDictionaries).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type ProductCreatorDictionary = typeof productCreatorDictionaries.$inferSelect;
export type InsertProductCreatorDictionary = z.infer<typeof insertProductCreatorDictionarySchema>;

// Enum dla typów słowników
export const dictionaryTypeSchema = z.enum([
  "product_type",     // Rodzaj produktu: SUPRA, TRES, VB, BINI
  "element_type",     // Rodzaj elementu: BOCZKI, BOK-L, BOK-P, POLKA, DRZWI, PLECY, WG, WD
  "component_cz1",    // Komponenty cz1: BOK-L, BOK-P, POLKA, DRZWI (główny typ elementu)
  "component_cz2",    // Komponenty cz2: ALTUS, FORM, SUPRA (podtyp/wariant elementu)
  "unit",            // Jednostki miary: szt, m, m², m³, kg, l, mb, t, opak
  "color",           // Kolor: BIALY, KASZMIR, WOTAN, CZARNY, etc.
  "product_group",   // Grupa produktów: A1, A2, A3, A4
  "door",            // Drzwi: D1, D2, D3, D4
  "leg",             // Nóżki: N1, N2, N3, N4
  "fabric",          // Tkaniny: ES1, ES2
  "material",        // Materiały: płyta meblowa, sklejka, itp.
  "dimension_length", // Długości: 500, 600, 700, 800, 900, 1000 mm
  "dimension_width",  // Szerokości: 300, 320, 340, 360 mm
  "dimension_height", // Wysokości: 450, 820, 2050 mm
  "scrap_reason_product",  // Powody złomowania produktów: uszkodzony transport, błąd produkcji, wada materiału
  "scrap_reason_cutting",  // Powody złomowania formatek: błąd cięcia, uszkodzenie, wada płyty
]);

export type DictionaryType = z.infer<typeof dictionaryTypeSchema>;

// Formula Presets - Predefiniowane wzory do wariantów wymiarów
export const formulaPresets = productCreatorSchema.table("formula_presets", {
  id: serial("id").primaryKey(),
  
  // Wzór (np. "dlMeb - 38", "szMeb * 10")
  formula: text("formula").notNull(),
  
  // Etykieta (opcjonalna, do wyświetlania)
  label: text("label"),
  
  // Typ: 'length' lub 'width'
  type: varchar("type", { length: 10 }).notNull(), // 'length' or 'width'
  
  // Kolejność sortowania
  sortOrder: integer("sort_order").default(0),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const insertFormulaPresetSchema = createInsertSchema(formulaPresets).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type FormulaPreset = typeof formulaPresets.$inferSelect;
export type InsertFormulaPreset = z.infer<typeof insertFormulaPresetSchema>;

// Product Matrices - Matryce do generowania produktów
export const productMatrices = productCreatorSchema.table("product_matrices", {
  id: serial("id").primaryKey(),
  
  // Nazwa matryce
  name: text("name").notNull(),
  
  // Prefix i suffix dla nazwy produktu
  namePrefix: text("name_prefix"),
  nameSuffix: text("name_suffix"),
  
  // ===== PODSTAWOWE =====
  // Typ produktu (pojedyncza wartość ze słownika lub własna)
  productType: text("product_type"),
  
  // Drzwi (opcjonalne)
  doors: text("doors"),
  
  // Nóżki (opcjonalne)
  legs: text("legs"),
  
  // Materiał (opcjonalne)
  material: text("material"),
  
  // ===== ROZMIARY =====
  // Osobne pola dla wymiarów (arrays of numbers)
  lengths: decimal("lengths", { precision: 10, scale: 2 }).array().notNull().default(sql`ARRAY[]::numeric[]`),
  widths: decimal("widths", { precision: 10, scale: 2 }).array().notNull().default(sql`ARRAY[]::numeric[]`),
  heights: decimal("heights", { precision: 10, scale: 2 }).array().notNull().default(sql`ARRAY[]::numeric[]`),
  
  // Grupy produktu dla każdego rozmiaru (array - każdy indeks odpowiada rozmiarowi)
  productGroups: text("product_groups").array().notNull().default(sql`ARRAY[]::text[]`),
  
  // Modyfikatory ceny dla każdego rozmiaru (array - każdy indeks odpowiada rozmiarowi)
  // Wartość może być procentem lub kwotą stałą w zależności od priceModifierTypes
  priceModifiers: decimal("price_modifiers", { precision: 10, scale: 2 }).array().notNull().default(sql`ARRAY[]::numeric[]`),
  
  // Typy modyfikatorów ceny ("percent" lub "fixed") dla każdego rozmiaru
  // np. ["percent", "fixed", "percent"] oznacza: pierwszy rozmiar +x%, drugi +x PLN, trzeci +x%
  priceModifierTypes: text("price_modifier_types").array().notNull().default(sql`ARRAY[]::text[]`),
  
  // ===== KOLORY =====
  // Lista kolorów (array of strings)
  colors: text("colors").array().notNull().default(sql`ARRAY[]::text[]`),
  
  // Zdjęcia przypisane do kolorów jako JSON: {colorName: ["url1", "url2"]}
  colorImages: jsonb("color_images").notNull().default(sql`'{}'::jsonb`),
  
  // Kolory zaznaczone do generowania (array of color names - jeśli puste, generuje wszystkie kolory)
  selectedColors: text("selected_colors").array().notNull().default(sql`ARRAY[]::text[]`),
  
  // Daty ostatniej generacji dla każdego koloru jako JSON: {colorName: "2025-10-28T14:30:00.000Z"}
  colorLastGenerated: jsonb("color_last_generated").notNull().default(sql`'{}'::jsonb`),
  
  // Opcje kolorystyczne dla każdego koloru jako JSON: {colorName: ["LISTWA-P-WOTAN", "DRZWI-CZARNY"]}
  colorOptions: jsonb("color_options").notNull().default(sql`'{}'::jsonb`),
  
  // ===== OPIS =====
  // Opis jako dokument Tiptap (JSON)
  descriptionDoc: jsonb("description_doc"),
  
  // Opis jako HTML (generowany z descriptionDoc)
  descriptionHtml: text("description_html"),
  
  // Prompt dla AI do generowania opisów
  aiPrompt: text("ai_prompt"),
  
  // Czy używać AI do generowania opisów
  useAiGeneration: boolean("use_ai_generation").default(false),
  
  // ===== SZABLON OPISU =====
  // Szablon opisu produktu (FK do description_templates)
  templateId: integer("template_id").references(() => descriptionTemplates.id, { onDelete: "set null" }),
  
  // Konfiguracja AI dla różnych sekcji opisu
  // Format JSON: { "intro": { "prompt": "...", "enabled": true }, "features": { "prompt": "...", "enabled": true } }
  aiConfig: jsonb("ai_config").default(sql`'{}'::jsonb`),
  
  // Czy produkt ma gwarancję (wpływa na generowanie sekcji {{ai.warranty}})
  hasWarranty: boolean("has_warranty").default(true),
  
  // ===== WYMIARY I CENA =====
  // Domyślna cena
  defaultPrice: decimal("default_price", { precision: 10, scale: 2 }),
  
  // Twórca matryce
  createdBy: integer("created_by").references(() => users.id, { onDelete: "set null" }),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const insertProductMatrixSchema = createInsertSchema(productMatrices).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type ProductMatrix = typeof productMatrices.$inferSelect;
export type InsertProductMatrix = z.infer<typeof insertProductMatrixSchema>;

// ===== PRODUCT ACCESSORIES SYSTEM =====
// System dodatków/akcesoriów do produktów (okucia, płyty, nóżki, uchwyty itp.)

// Grupy dodatków (np. "Okucia", "Płyty meblowe", "Nóżki")
export const accessoryGroups = catalogSchema.table("accessory_groups", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(), // np. "Okucia", "Płyty meblowe"
  code: text("code").notNull().unique(), // np. "hardware", "boards", "legs"
  category: text("category"), // Kategoria grupy (np. "Tkaniny", "Okucia", "Płyty meblowe")
  description: text("description"), // Krótki opis grupy
  displayOrder: integer("display_order").default(0).notNull(), // Kolejność wyświetlania
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_accessory_groups_code").on(table.code),
  index("idx_accessory_groups_active").on(table.isActive),
  index("idx_accessory_groups_category").on(table.category),
]);

export const insertAccessoryGroupSchema = createInsertSchema(accessoryGroups).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type AccessoryGroup = typeof accessoryGroups.$inferSelect;
export type InsertAccessoryGroup = z.infer<typeof insertAccessoryGroupSchema>;

// Dodatki (konkretne elementy w grupach)
export const accessories = catalogSchema.table("accessories", {
  id: serial("id").primaryKey(),
  groupId: integer("group_id").references(() => accessoryGroups.id, { onDelete: "cascade" }),
  warehouseMaterialId: integer("warehouse_material_id"), // Link to warehouse.materials
  name: text("name").notNull(), // np. "Zawiasy Silent System"
  code: text("code").notNull().unique(), // np. "silent_hinges"
  alpmaCode: text("alpma_code"), // Własny kod Alpma
  description: text("description"), // Szczegółowy opis dodatku
  price: decimal("price", { precision: 10, scale: 2 }), // Cena from warehouse or manual
  imageUrl: text("image_url"), // URL do zdjęcia dodatku
  displayOrder: integer("display_order").default(0).notNull(),
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_accessories_group").on(table.groupId),
  index("idx_accessories_code").on(table.code),
  index("idx_accessories_active").on(table.isActive),
  index("idx_accessories_warehouse_material").on(table.warehouseMaterialId),
]);

export const insertAccessorySchema = createInsertSchema(accessories).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type Accessory = typeof accessories.$inferSelect;
export type InsertAccessory = z.infer<typeof insertAccessorySchema>;

// Przypisanie GRUP dodatków do matryc produktowych (relacja many-to-many)
// Teraz przypisujemy całe grupy zamiast pojedynczych akcesoriów
export const matrixAccessories = catalogSchema.table("matrix_accessories", {
  id: serial("id").primaryKey(),
  matrixId: integer("matrix_id").references(() => productMatrices.id, { onDelete: "cascade" }).notNull(),
  groupId: integer("group_id").references(() => accessoryGroups.id, { onDelete: "cascade" }).notNull(), // Zmienione: groupId zamiast accessoryId
  displayOrder: integer("display_order").default(0).notNull(), // Kolejność wyświetlania w opisie
  createdAt: timestamp("created_at").defaultNow(),
}, (table) => [
  index("idx_matrix_accessories_matrix").on(table.matrixId),
  index("idx_matrix_accessories_group").on(table.groupId), // Zmienione: indeks na groupId
  uniqueIndex("idx_matrix_accessories_unique").on(table.matrixId, table.groupId), // Zmienione: unique constraint na matrixId + groupId
]);

export const insertMatrixAccessorySchema = createInsertSchema(matrixAccessories).omit({
  id: true,
  createdAt: true,
});

export type MatrixAccessory = typeof matrixAccessories.$inferSelect;
export type InsertMatrixAccessory = z.infer<typeof insertMatrixAccessorySchema>;

// Sync Health Stats - Monitorowanie stanu synchronizacji
export const syncHealthStats = pgTable("sync_health_stats", {
  id: serial("id").primaryKey(),
  
  // Typ synchronizacji
  syncType: varchar("sync_type", { length: 50 }).notNull(), // allegro, shoper, odoo_queue
  
  // Ostatnia synchronizacja
  lastSyncAt: timestamp("last_sync_at"),
  lastSuccessAt: timestamp("last_success_at"),
  lastFailureAt: timestamp("last_failure_at"),
  
  // Statystyki (24h)
  successCount24h: integer("success_count_24h").default(0),
  failureCount24h: integer("failure_count_24h").default(0),
  totalSynced24h: integer("total_synced_24h").default(0),
  
  // Aktualny status
  healthStatus: varchar("health_status", { length: 20 }).notNull().default("unknown"), // healthy, degraded, critical, unknown
  consecutiveFailures: integer("consecutive_failures").default(0),
  
  // Ostatni błąd
  lastError: text("last_error"),
  lastErrorContext: jsonb("last_error_context"),
  
  // Kolejka (dla Odoo)
  queuePending: integer("queue_pending").default(0),
  queueFailed: integer("queue_failed").default(0),
  
  // Auto-healing
  lastAutoHealAt: timestamp("last_auto_heal_at"),
  autoHealCount: integer("auto_heal_count").default(0),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_sync_health_type").on(table.syncType),
  index("idx_sync_health_status").on(table.healthStatus),
  uniqueIndex("idx_sync_health_unique_type").on(table.syncType),
]);

export const insertSyncHealthStatsSchema = createInsertSchema(syncHealthStats).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type SyncHealthStats = typeof syncHealthStats.$inferSelect;
export type InsertSyncHealthStats = z.infer<typeof insertSyncHealthStatsSchema>;

// ===== FILE STORAGE SETTINGS =====
// External file server configuration (SFTP, S3, etc.)
export const fileStorageSettings = pgTable("file_storage_settings", {
  id: serial("id").primaryKey(),
  
  // Provider type
  provider: varchar("provider", { length: 20 }).notNull().default("local"), // local, sftp, s3
  
  // SFTP/Server settings
  host: text("host"), // np. files.alpsys.pl
  port: integer("port").default(22), // SFTP port
  username: text("username"), // np. oms-sftp
  
  // Base URL for accessing files
  baseUrl: text("base_url"), // np. https://files.alpsys.pl
  
  // Path prefix on server
  pathPrefix: text("path_prefix").default("/OMS"), // np. /OMS
  
  // Is this configuration active
  isActive: boolean("is_active").default(false),
  
  // Connection test results
  lastConnectionTest: timestamp("last_connection_test"),
  connectionStatus: varchar("connection_status", { length: 20 }).default("unknown"), // connected, failed, unknown
  lastError: text("last_error"),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const insertFileStorageSettingsSchema = createInsertSchema(fileStorageSettings).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type FileStorageSettings = typeof fileStorageSettings.$inferSelect;
export type InsertFileStorageSettings = z.infer<typeof insertFileStorageSettingsSchema>;

// ===== PRODUCT SETS (ZESTAWY) =====

// Set Matrices - Szablony/matrycy zestawów
export const setMatrices = productCreatorSchema.table("set_matrices", {
  id: serial("id").primaryKey(),
  
  // Nazwa matrycy
  name: text("name").notNull(),
  namePrefix: text("name_prefix"), // "Zestaw"
  nameSuffix: text("name_suffix"),
  
  // Grupa produktów dla zestawu (ze słownika product_group)
  productGroup: text("product_group"),
  
  // Wymiary zestawu (ze słowników dimension_length, dimension_width, dimension_height)
  length: text("length"), // Długość - wartość ze słownika (np. "900")
  width: text("width"), // Szerokość - wartość ze słownika (np. "300")
  height: text("height"), // Wysokość - wartość ze słownika (np. "2050")
  
  // Drzwi i nogi (ze słownika door, leg)
  doors: text("doors"), // np. "D1", "D2" ze słownika door
  legs: text("legs"), // np. "N1", "N2" ze słownika leg
  
  // Zdjęcie poglądowe
  imageUrl: text("image_url"),
  
  // Ustawienia zestawu
  defaultDepth: decimal("default_depth", { precision: 10, scale: 2 }), // Głębokość zestawu (mm)
  hookLength: decimal("hook_length", { precision: 10, scale: 2 }), // Długość wieszaka (mm)
  
  // Komponenty zestawu (JSON array)
  // [{componentType: "SZAFKA", length: 60, width: 30, quantity: 1, name: "Szafka", ignoreColorOptions: false, relevantOptionPrefixes: ["DRZWI", "FRONT"]},
  //  {componentType: "PANEL", length: 50, width: 30, quantity: 2, name: "Panel", ignoreColorOptions: true},
  //  {componentType: "VB", length: 60, width: 30, quantity: 1, name: "Szafka bez frontu", ignoreColorOptions: false, relevantOptionPrefixes: ["DRZWI"]}]
  // relevantOptionPrefixes - opcjonalne, określa które prefiksy opcji kolorystycznych są relevantne dla tego komponentu (np. VB ma tylko DRZWI, nie ma FRONT)
  components: jsonb("components").notNull().default(sql`'[]'::jsonb`),
  
  // Modyfikator ceny - zastosowany do calculated_price
  modifierType: text("modifier_type").default("amount"), // "amount" lub "percentage"
  modifierOperation: text("modifier_operation").default("add"), // "add" lub "subtract"
  modifierValue: decimal("modifier_value", { precision: 10, scale: 2 }).default("0"),
  
  // ===== KOLORY =====
  // Lista kolorów (nazwy)
  colors: text("colors").array().notNull().default(sql`ARRAY[]::text[]`),
  
  // Zdjęcia przypisane do kolorów jako JSON: {colorName: ["url1", "url2"]}
  colorImages: jsonb("color_images").notNull().default(sql`'{}'::jsonb`),
  
  // Kolory zaznaczone do generowania (jeśli puste, generuje wszystkie kolory)
  selectedColors: text("selected_colors").array().notNull().default(sql`ARRAY[]::text[]`),
  
  // Opcje kolorystyczne dla każdego koloru jako JSON: {colorName: ["LISTWA-P-WOTAN", "DRZWI-CZARNY"]}
  colorOptions: jsonb("color_options").notNull().default(sql`'{}'::jsonb`),
  
  // Nadpisania produktów dla komponentów w kolorach
  // Struktura: { "colorIndex": { "componentType": { "productId": number } } }
  // Przykład: { "0": { "WIESZAK-Z-POLKA": { "productId": 497 } } }
  // Podczas generowania: jeśli jest nadpisanie, używa tego produktu zamiast szukać po parametrach
  componentProductOverrides: jsonb("component_product_overrides").notNull().default(sql`'{}'::jsonb`),
  
  // Generowanie
  useAiGeneration: boolean("use_ai_generation").default(false),
  aiPrompt: text("ai_prompt"),
  descriptionTemplateId: integer("description_template_id").references(() => descriptionTemplates.id, { onDelete: "set null" }),
  
  // Enhanced descriptions - build descriptions from component products
  useEnhancedDescription: boolean("use_enhanced_description").default(false),
  
  // Status
  isActive: boolean("is_active").default(true),
  
  // Generation tracking
  lastGenerationStatus: varchar("last_generation_status", { length: 20 }), // 'success', 'partial', 'failed', 'never', null
  lastGenerationAt: timestamp("last_generation_at"),
  lastMissingComponents: jsonb("last_missing_components"), // Array of {componentType, length?, width?}
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Product Sets - Wygenerowane zestawy
export const productSets = productCreatorSchema.table("product_sets", {
  id: serial("id").primaryKey(),
  setMatrixId: integer("set_matrix_id").references(() => setMatrices.id, { onDelete: "cascade" }),
  
  // Identyfikatory
  sku: text("sku").notNull().unique(),
  title: text("title").notNull(),
  shortDescription: text("short_description"),
  fullDescription: text("full_description"),
  
  // Parametry tego zestawu
  color: text("color"),
  colorOptions: text("color_options").array().default(sql`ARRAY[]::text[]`),
  depth: decimal("depth", { precision: 10, scale: 2 }),
  panelCount: integer("panel_count"),
  hookLength: decimal("hook_length", { precision: 10, scale: 2 }),
  
  // Cena
  basePrice: decimal("base_price", { precision: 10, scale: 2 }), // Ręczna cena bazowa (deprecated - używaj modyfikatorów)
  calculatedPrice: decimal("calculated_price", { precision: 10, scale: 2 }), // Suma kosztów komponentów
  finalPrice: decimal("final_price", { precision: 10, scale: 2 }), // Cena końcowa po zastosowaniu modyfikatora
  
  // Zdjęcia (URLs)
  images: text("images").array().default(sql`ARRAY[]::text[]`),
  
  // Generowanie
  generatedFromMatrix: boolean("generated_from_matrix").default(false),
  combinationKey: text("combination_key"), // Unikalny klucz: matrixId:color
  
  // Status
  isActive: boolean("is_active").default(true),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => ({
  uniqueMatrixCombination: uniqueIndex("unique_set_matrix_combination").on(table.setMatrixId, table.combinationKey),
}));

// Set Product Links - Powiązania produktów z zestawami
export const setProductLinks = productCreatorSchema.table("set_product_links", {
  id: serial("id").primaryKey(),
  setId: integer("set_id").notNull().references(() => productSets.id, { onDelete: "cascade" }),
  productId: integer("product_id").references(() => products.id, { onDelete: "cascade" }), // Nullable - komponenty mogą nie mieć przypisanych produktów
  
  componentType: text("component_type").notNull(), // "cabinet", "panel", "hook", etc.
  quantity: integer("quantity").default(1),
  position: integer("position").default(0), // Kolejność w zestawie
  
  createdAt: timestamp("created_at").defaultNow(),
}, (table) => [
  index("idx_set_product_links_set").on(table.setId),
  index("idx_set_product_links_product").on(table.productId),
]);

// Insert schemas
export const insertSetMatrixSchema = createInsertSchema(setMatrices).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductSetSchema = createInsertSchema(productSets).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertSetProductLinkSchema = createInsertSchema(setProductLinks).omit({
  id: true,
  createdAt: true,
});

// Types
export type SetMatrix = typeof setMatrices.$inferSelect;
export type InsertSetMatrix = z.infer<typeof insertSetMatrixSchema>;

export type ProductSet = typeof productSets.$inferSelect;
export type InsertProductSet = z.infer<typeof insertProductSetSchema>;

export type SetProductLink = typeof setProductLinks.$inferSelect;
export type InsertSetProductLink = z.infer<typeof insertSetProductLinkSchema>;

// ==================== Warehouse Schema ====================
// Dedykowany schemat PostgreSQL dla magazynu surowców
export const warehouseSchema = pgSchema("warehouse");

// Warehouse Categories - Konfigurowalne kategorie magazynu (np. "Okucia", "Płyty meblowe", "Śruby")
export const warehouseCategories = warehouseSchema.table("categories", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(), // Nazwa wyświetlana: "Okucia", "Płyty meblowe"
  code: text("code").notNull().unique(), // Kod URL: "okucia", "plyty_meblowe"
  icon: text("icon").default("Box"), // Nazwa ikony z lucide-react
  iconColor: text("icon_color").default("text-gray-600/70"), // Kolor ikony
  description: text("description"), // Opcjonalny opis kategorii
  displayOrder: integer("display_order").default(0).notNull(),
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_warehouse_categories_code").on(table.code),
  index("idx_warehouse_categories_active").on(table.isActive),
  index("idx_warehouse_categories_order").on(table.displayOrder),
]);

// Material Groups - Grupy materiałów (np. "Zawiasy VB", "Haki wieszaka", "Płyta biała")
export const materialGroups = warehouseSchema.table("material_groups", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(), // np. "Zawiasy VB", "Haki wieszaka"
  code: text("code").notNull().unique(), // np. "hinges_vb", "hooks_hanger"
  category: text("category"), // Kod kategorii: "okucia", "plyty_meblowe", "sruby", "tkaniny", "pianki"
  description: text("description"), // Opis grupy
  
  // Kolor grupy (hex format, np. "#FF5733")
  color: varchar("color", { length: 7 }), // Kolor grupy do wyświetlania ikon, pasków itp.
  
  // Galeria zdjęć dla grupy
  gallery: text("gallery").array().notNull().default(sql`ARRAY[]::text[]`),
  primaryImage: text("primary_image"), // Główne zdjęcie grupy
  
  displayOrder: integer("display_order").default(0).notNull(),
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_material_groups_code").on(table.code),
  index("idx_material_groups_category").on(table.category),
  index("idx_material_groups_active").on(table.isActive),
]);

// Materials - Konkretne materiały/surowce
export const materials = warehouseSchema.table("materials", {
  id: serial("id").primaryKey(),
  groupId: integer("group_id").references(() => materialGroups.id, { onDelete: "cascade" }),
  
  // Powiązanie z produktem katalogowym (dla produktów-spakowanych)
  catalogProductId: integer("catalog_product_id").references(() => products.id, { onDelete: "set null" }),
  
  // Podstawowe informacje
  name: text("name").notNull(), // np. "Zawias Silent System 110°"
  internalCode: text("internal_code").notNull().unique(), // Kod wewnętrzny Alpma
  supplierCode: text("supplier_code"), // Kod dostawcy
  
  // Opis i specyfikacja
  description: text("description"),
  specifications: jsonb("specifications"), // Dodatkowe parametry jako JSON
  
  // Jednostka miary
  unitOfMeasure: varchar("unit_of_measure", { length: 20 }).default("szt"), // szt, m, m², kg, l
  
  // Cena
  price: decimal("price", { precision: 10, scale: 2 }), // Cena materiału
  
  // Stan magazynowy
  quantity: decimal("quantity", { precision: 12, scale: 3 }).default("0").notNull(), // Aktualny stan magazynowy
  
  // Lokalizacja magazynowa (hierarchia: Hala -> Nośnik -> Miejsce)
  locationId: integer("location_id").references(() => productionLocations.id, { onDelete: "set null" }), // FK do production.locations
  carrierId: integer("carrier_id").references(() => productionCarriers.id, { onDelete: "set null" }), // FK do production.carriers (Regał/Wózek/Paleta)
  
  // Galeria (ścieżki do zdjęć jako tablica)
  gallery: text("gallery").array().notNull().default(sql`ARRAY[]::text[]`),
  primaryImageUrl: text("primary_image_url"), // Główne zdjęcie
  
  // Pola specyficzne dla formatek (nullable dla innych kategorii)
  parentId: integer("parent_id").references((): AnyPgColumn => materials.id, { onDelete: "set null" }), // Rodzic formatki
  length: decimal("length", { precision: 10, scale: 2 }), // Długość formatki (cm)
  width: decimal("width", { precision: 10, scale: 2 }), // Szerokość formatki (cm)
  materialThickness: decimal("material_thickness", { precision: 10, scale: 2 }), // Grubość formatki (mm)
  cz1: varchar("cz1", { length: 100 }), // Część nazwy 1 (np. "BOCZKI", "DNO", "DRZWI")
  cz2: varchar("cz2", { length: 100 }), // Część nazwy 2 (np. "SZUFLADA", "TRES", "NADST")
  furnitureType: varchar("furniture_type", { length: 100 }), // Typ mebla (np. "SUPRA", "TRES-30", "VB")
  color: varchar("color", { length: 100 }), // Kolor (np. "WOTAN", "BIALY", "CZARNY")
  plateType: varchar("plate_type", { length: 100 }), // Typ płyty (np. "18_BIALY", "HDF_BIALY")
  edgingMaterial: varchar("edging_material", { length: 100 }), // Materiał obrzeża (np. "0.8_BIALY")
  edge1: boolean("edge1").default(false), // Obrzeże 1 (góra)
  edge2: boolean("edge2").default(false), // Obrzeże 2 (prawo)
  edge3: boolean("edge3").default(false), // Obrzeże 3 (dół)
  edge4: boolean("edge4").default(false), // Obrzeże 4 (lewo)
  edge5: boolean("edge5").default(false), // Obrzeże 5 (dodatkowe)
  status: varchar("status", { length: 50 }), // Status formatki (np. "Zapas", "W produkcji")
  source: varchar("source", { length: 100 }), // Źródło (np. "Airtable", "Ręcznie")
  referenceCode: varchar("reference_code", { length: 100 }), // Kod referencyjny
  
  // Status
  displayOrder: integer("display_order").default(0).notNull(),
  isActive: boolean("is_active").default(true).notNull(),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_materials_group").on(table.groupId),
  index("idx_materials_catalog_product").on(table.catalogProductId),
  index("idx_materials_internal_code").on(table.internalCode),
  index("idx_materials_supplier_code").on(table.supplierCode),
  index("idx_materials_active").on(table.isActive),
  index("idx_materials_location").on(table.locationId),
  index("idx_materials_carrier").on(table.carrierId),
  index("idx_materials_parent").on(table.parentId),
  index("idx_materials_cz1").on(table.cz1),
  index("idx_materials_furniture_type").on(table.furnitureType),
]);

// Material Media - Galeria zdjęć dla materiałów
export const materialMedia = warehouseSchema.table("material_media", {
  id: serial("id").primaryKey(),
  materialId: integer("material_id").references(() => materials.id, { onDelete: "cascade" }).notNull(),
  
  url: text("url").notNull(), // Ścieżka do pliku
  filename: text("filename").notNull(),
  
  // Metadane
  altText: text("alt_text"),
  caption: text("caption"),
  
  // Kolejność i typ
  displayOrder: integer("display_order").default(0).notNull(),
  isPrimary: boolean("is_primary").default(false),
  
  createdAt: timestamp("created_at").defaultNow(),
}, (table) => [
  index("idx_material_media_material").on(table.materialId),
  index("idx_material_media_order").on(table.displayOrder),
]);

// Material Accessory Links - Powiązania materiałów z akcesoriami
export const materialAccessoryLinks = warehouseSchema.table("material_accessory_links", {
  id: serial("id").primaryKey(),
  materialId: integer("material_id").references(() => materials.id, { onDelete: "cascade" }).notNull(),
  accessoryId: integer("accessory_id").notNull(), // Reference to catalog.accessories
  
  quantity: integer("quantity").default(1), // Ilość materiału na jedno akcesorium
  notes: text("notes"), // Notatki (np. "na 1 szafę 180cm")
  
  createdAt: timestamp("created_at").defaultNow(),
}, (table) => [
  index("idx_material_accessory_material").on(table.materialId),
  index("idx_material_accessory_accessory").on(table.accessoryId),
  uniqueIndex("idx_material_accessory_unique").on(table.materialId, table.accessoryId),
]);

// Inventory History - Historia zmian stanów magazynowych materiałów
export const inventoryHistory = warehouseSchema.table("inventory_history", {
  id: serial("id").primaryKey(),
  materialId: integer("material_id").references(() => materials.id, { onDelete: "cascade" }).notNull(),
  
  // Typ operacji
  operationType: varchar("operation_type", { length: 50 }).notNull(), // 'inventory_count', 'usage', 'complaint', 'purchase', 'adjustment'
  
  // Zmiana ilości (może być dodatnia lub ujemna)
  quantityChange: decimal("quantity_change", { precision: 12, scale: 3 }).notNull(),
  
  // Stan przed i po operacji
  quantityBefore: decimal("quantity_before", { precision: 12, scale: 3 }).notNull(),
  quantityAfter: decimal("quantity_after", { precision: 12, scale: 3 }).notNull(),
  
  // Powiązania z innymi encjami
  productionOrderId: integer("production_order_id"), // Jeśli związane z zamówieniem produkcyjnym
  productionPlanId: integer("production_plan_id"), // Jeśli związane z planem produkcji (ZLP)
  
  // Szczegóły operacji
  notes: text("notes"), // Notatki, np. "Spis inwentaryzacyjny 2025-01", "Zużycie do ZLP-123"
  documentNumber: varchar("document_number", { length: 100 }), // Numer dokumentu (WZ, PZ, etc.)
  
  // Kto wykonał operację
  performedBy: varchar("performed_by", { length: 255 }), // Użytkownik który wykonał operację
  
  createdAt: timestamp("created_at").defaultNow().notNull(),
}, (table) => [
  index("idx_inventory_history_material").on(table.materialId),
  index("idx_inventory_history_operation_type").on(table.operationType),
  index("idx_inventory_history_production_order").on(table.productionOrderId),
  index("idx_inventory_history_production_plan").on(table.productionPlanId),
  index("idx_inventory_history_created_at").on(table.createdAt),
]);

// Inventory Counts - Spisy inwentaryzacyjne
export const inventoryCounts = warehouseSchema.table("inventory_counts", {
  id: serial("id").primaryKey(),
  name: varchar("name", { length: 255 }).notNull(), // Nazwa spisu, np. "Spis 2025-11"
  status: varchar("status", { length: 50 }).notNull().default("draft"), // 'draft', 'finalized'
  
  // Metadata
  notes: text("notes"),
  
  // Audit trail
  createdBy: varchar("created_by", { length: 255 }).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  finalizedBy: varchar("finalized_by", { length: 255 }),
  finalizedAt: timestamp("finalized_at"),
}, (table) => [
  index("idx_inventory_counts_status").on(table.status),
  index("idx_inventory_counts_created_at").on(table.createdAt),
]);

// Inventory Count Items - Pozycje w spisie inwentaryzacyjnym
export const inventoryCountItems = warehouseSchema.table("inventory_count_items", {
  id: serial("id").primaryKey(),
  inventoryCountId: integer("inventory_count_id").references(() => inventoryCounts.id, { onDelete: "cascade" }).notNull(),
  
  // Either materialId OR stockPanelId OR packagingMaterialId OR packedProductId must be set (not multiple)
  materialId: integer("material_id").references(() => materials.id, { onDelete: "restrict" }),
  stockPanelId: integer("stock_panel_id").references(() => stockPanels.id, { onDelete: "restrict" }),
  packagingMaterialId: integer("packaging_material_id").references(() => packagingMaterials.id, { onDelete: "restrict" }),
  packedProductId: integer("packed_product_id").references(() => packedProducts.id, { onDelete: "restrict" }),
  
  // Quantities
  systemQuantity: decimal("system_quantity", { precision: 12, scale: 3 }).notNull(), // Stan systemowy przy tworzeniu spisu
  countedQuantity: decimal("counted_quantity", { precision: 12, scale: 3 }), // Ilość spisana (nullable - może być jeszcze niesprawdzona)
  difference: decimal("difference", { precision: 12, scale: 3 }), // Różnica (countedQuantity - systemQuantity)
  
  // Notes per item
  notes: text("notes"),
  
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
}, (table) => [
  index("idx_inventory_count_items_count").on(table.inventoryCountId),
  index("idx_inventory_count_items_material").on(table.materialId),
  index("idx_inventory_count_items_stock_panel").on(table.stockPanelId),
  index("idx_inventory_count_items_packaging_material").on(table.packagingMaterialId),
  index("idx_inventory_count_items_packed_product").on(table.packedProductId),
]);

// Insert schemas
export const insertWarehouseCategorySchema = createInsertSchema(warehouseCategories).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertMaterialGroupSchema = createInsertSchema(materialGroups).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertMaterialSchema = createInsertSchema(materials).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertMaterialMediaSchema = createInsertSchema(materialMedia).omit({
  id: true,
  createdAt: true,
});

export const insertMaterialAccessoryLinkSchema = createInsertSchema(materialAccessoryLinks).omit({
  id: true,
  createdAt: true,
});

export const insertInventoryHistorySchema = createInsertSchema(inventoryHistory).omit({
  id: true,
  createdAt: true,
});

export const insertInventoryCountSchema = createInsertSchema(inventoryCounts).omit({
  id: true,
  createdAt: true,
});

export const insertInventoryCountItemSchema = createInsertSchema(inventoryCountItems).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

// Types
export type WarehouseCategory = typeof warehouseCategories.$inferSelect;
export type InsertWarehouseCategory = z.infer<typeof insertWarehouseCategorySchema>;

export type MaterialGroup = typeof materialGroups.$inferSelect;
export type InsertMaterialGroup = z.infer<typeof insertMaterialGroupSchema>;

export type Material = typeof materials.$inferSelect;
export type InsertMaterial = z.infer<typeof insertMaterialSchema>;

export type MaterialMedia = typeof materialMedia.$inferSelect;
export type InsertMaterialMedia = z.infer<typeof insertMaterialMediaSchema>;

export type MaterialAccessoryLink = typeof materialAccessoryLinks.$inferSelect;
export type InsertMaterialAccessoryLink = z.infer<typeof insertMaterialAccessoryLinkSchema>;

export type InventoryHistory = typeof inventoryHistory.$inferSelect;
export type InsertInventoryHistory = z.infer<typeof insertInventoryHistorySchema>;

export type InventoryCount = typeof inventoryCounts.$inferSelect;
export type InsertInventoryCount = z.infer<typeof insertInventoryCountSchema>;

export type InventoryCountItem = typeof inventoryCountItems.$inferSelect;
export type InsertInventoryCountItem = z.infer<typeof insertInventoryCountItemSchema>;

// Formatki - Stock Panels - magazyn gotowych komponentów meblowych
export const stockPanels = warehouseSchema.table("stock_panels", {
  id: serial("id").primaryKey(),
  
  // Nazwa generowana (auto lub manual)
  generatedName: varchar("generated_name", { length: 255 }).notNull(), // "500x300-BIALY" lub "LISTWA-T-VB-462x70-KASZMIR"
  
  // Części nazwy (opcjonalne dla wierconych)
  cz1: varchar("cz1", { length: 100 }), // "BOCZKI", "BOK-L", "DNO", "DRZWI", "LISTWA-T", itp.
  cz2: varchar("cz2", { length: 100 }), // "SZUFLADA", "TRES", "NADST", "VB", itp.
  furnitureType: varchar("furniture_type", { length: 100 }), // "SUPRA", "TRES-30", "VB", "ALTUS" - dla wierconych
  
  // Wymiary formatki
  length: decimal("length", { precision: 10, scale: 2 }).notNull(), // np. 500, 462, 760
  width: decimal("width", { precision: 10, scale: 2 }).notNull(), // np. 300, 279, 280
  thickness: decimal("thickness", { precision: 10, scale: 2 }).notNull().default("18"), // 18mm, 3mm
  
  // Materiały - referencje do słowników
  boardCode: varchar("board_code", { length: 50 }), // Kod płyty z dictionary_material
  edgingCode: varchar("edging_code", { length: 50 }), // Kod obrzeża z dictionary_material  
  colorCode: varchar("color_code", { length: 50 }), // Kod koloru z dictionary_color
  
  // Obrzeża - które krawędzie oklejone
  edge1: boolean("edge1").default(false), // Krawędź 1
  edge2: boolean("edge2").default(false), // Krawędź 2
  edge3: boolean("edge3").default(false), // Krawędź 3
  edge4: boolean("edge4").default(false), // Krawędź 4
  
  // Status obróbki
  isDrilled: boolean("is_drilled").default(false).notNull(), // Czy wiercone
  isEdged: boolean("is_edged").default(false).notNull(), // Czy oklejone
  
  // Pochodzenie formatki
  source: varchar("source", { length: 50 }).notNull().default("stock"), // 'stock', 'production_surplus', 'returns'
  sourceReference: varchar("source_reference", { length: 255 }), // Numer zamówienia zwrotu, batch produkcyjny, itp.
  
  // Magazyn
  quantity: integer("quantity").notNull().default(0), // Ilość sztuk
  unit: varchar("unit", { length: 20 }).notNull().default("szt"), // Jednostka miary: 'szt', 'kg', 'm', 'rolka', 'm2'
  location: varchar("location", { length: 100 }), // Lokalizacja w magazynie (np. "A-12-3") - deprecated, używaj location_id
  locationId: integer("location_id"), // Referencja do production.production_locations
  carrierId: integer("carrier_id"), // Referencja do production.production_carriers
  
  // Grupa materiałowa
  groupId: integer("group_id").references(() => materialGroups.id, { onDelete: "set null" }),
  
  // Uwagi
  notes: text("notes"),
  
  // Zdjęcie (opcjonalne)
  imageUrl: text("image_url"),
  
  // Koszt materiału (wyliczany: cena płyty za m² × powierzchnia formatki)
  materialCost: decimal("material_cost", { precision: 10, scale: 4 }),
  
  // Status
  isActive: boolean("is_active").default(true).notNull(),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_stock_panels_board").on(table.boardCode),
  index("idx_stock_panels_edging").on(table.edgingCode),
  index("idx_stock_panels_color").on(table.colorCode),
  index("idx_stock_panels_source").on(table.source),
  index("idx_stock_panels_drilled").on(table.isDrilled),
  index("idx_stock_panels_edged").on(table.isEdged),
  index("idx_stock_panels_active").on(table.isActive),
  index("idx_stock_panels_dimensions").on(table.length, table.width),
]);

export const insertStockPanelSchema = createInsertSchema(stockPanels).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type StockPanel = typeof stockPanels.$inferSelect;
export type InsertStockPanel = z.infer<typeof insertStockPanelSchema>;

// Stock Panel Events - audit log zmian magazynowych
export const stockPanelEvents = warehouseSchema.table("stock_panel_events", {
  id: serial("id").primaryKey(),
  stockPanelId: integer("stock_panel_id").references(() => stockPanels.id),
  sourceType: varchar("source_type", { length: 50 }).notNull(),
  sourceReference: varchar("source_reference", { length: 255 }),
  deltaQuantity: integer("delta_quantity").notNull(),
  notes: text("notes"),
  createdBy: integer("created_by"),
  createdAt: timestamp("created_at").defaultNow(),
}, (table) => [
  index("idx_stock_panel_events_panel").on(table.stockPanelId),
  index("idx_stock_panel_events_source").on(table.sourceType, table.sourceReference),
]);

export const insertStockPanelEventSchema = createInsertSchema(stockPanelEvents).omit({
  id: true,
  createdAt: true,
});

export type StockPanelEvent = typeof stockPanelEvents.$inferSelect;
export type InsertStockPanelEvent = z.infer<typeof insertStockPanelEventSchema>;

// Packaging Materials - Opakowania (kartony, wypełniacz, taśmy, stretch)
export const packagingMaterials = warehouseSchema.table("packaging_materials", {
  id: serial("id").primaryKey(),
  
  // Kategoria opakowania
  category: varchar("category", { length: 50 }).notNull(), // 'karton', 'wypełniacz', 'taśma', 'stretch', 'inne'
  
  // Podstawowe informacje
  name: varchar("name", { length: 255 }).notNull(), // Nazwa opakowania
  description: text("description"), // Opis szczegółowy
  
  // Wymiary (opcjonalne - głównie dla kartonów)
  length: decimal("length", { precision: 10, scale: 2 }), // Długość w mm
  width: decimal("width", { precision: 10, scale: 2 }), // Szerokość w mm
  height: decimal("height", { precision: 10, scale: 2 }), // Wysokość w mm
  
  // Ilość i jednostka
  quantity: decimal("quantity", { precision: 12, scale: 3 }).notNull().default("0"), // Ilość w magazynie
  unit: varchar("unit", { length: 20 }).notNull().default("szt"), // 'szt', 'kg', 'm', 'rolka', 'm2'
  
  // Lokalizacja i nośnik
  locationId: integer("location_id"), // Referencja do production.production_locations
  carrierId: integer("carrier_id"), // Referencja do production.production_carriers
  
  // Grupa materiałowa
  groupId: integer("group_id").references(() => materialGroups.id, { onDelete: "set null" }),
  
  // Dodatkowe informacje
  supplierCode: varchar("supplier_code", { length: 100 }), // Kod dostawcy/producenta
  notes: text("notes"), // Uwagi
  
  // Zdjęcie
  imageUrl: text("image_url"),
  
  // Status
  isActive: boolean("is_active").default(true).notNull(),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_packaging_category").on(table.category),
  index("idx_packaging_active").on(table.isActive),
  index("idx_packaging_location").on(table.locationId),
  index("idx_packaging_carrier").on(table.carrierId),
  unique("packaging_unique_category_name").on(table.category, table.name),
]);

export const insertPackagingMaterialSchema = createInsertSchema(packagingMaterials).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type PackagingMaterial = typeof packagingMaterials.$inferSelect;
export type InsertPackagingMaterial = z.infer<typeof insertPackagingMaterialSchema>;

// Packed Products - Gotowe produkty spakowane z katalogu
export const packedProducts = warehouseSchema.table("packed_products", {
  id: serial("id").primaryKey(),
  
  // Reference to catalog product OR product set (one must be set, but not both)
  catalogProductId: integer("catalog_product_id"), // FK to catalog.products
  catalogSetId: integer("catalog_set_id"), // FK to product_creator.product_sets
  
  // Product identification (copied from catalog for faster queries)
  productSku: varchar("product_sku", { length: 100 }).notNull(),
  productName: text("product_name").notNull(),
  productType: varchar("product_type", { length: 50 }), // 'product' | 'set'
  
  // Warehouse data
  quantity: integer("quantity").notNull().default(0), // Total quantity in stock
  reservedQuantity: integer("reserved_quantity").notNull().default(0), // Reserved for production
  locationId: integer("location_id"), // FK to production.production_locations
  carrierId: integer("carrier_id"), // FK to production.production_carriers
  
  // Grupa materiałowa
  groupId: integer("group_id").references(() => materialGroups.id, { onDelete: "set null" }),
  
  // Product images
  imageUrl: text("image_url"),
  
  // Notes
  notes: text("notes"),
  
  // External symbol (for Subiekt Nexo integration)
  externalSymbol: varchar("external_symbol", { length: 255 }),
  
  // Status
  isActive: boolean("is_active").default(true).notNull(),
  isArchived: boolean("is_archived").default(false).notNull(),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_packed_products_catalog_product").on(table.catalogProductId),
  index("idx_packed_products_catalog_set").on(table.catalogSetId),
  index("idx_packed_products_sku").on(table.productSku),
  index("idx_packed_products_location").on(table.locationId),
  index("idx_packed_products_carrier").on(table.carrierId),
  index("idx_packed_products_active").on(table.isActive),
  index("idx_packed_products_archived").on(table.isArchived),
]);

// Packed Product Packages - Information about packages/cartons for each packed product
export const packedProductPackages = warehouseSchema.table("packed_product_packages", {
  id: serial("id").primaryKey(),
  packedProductId: integer("packed_product_id").references(() => packedProducts.id, { onDelete: "cascade" }).notNull(),
  
  // Package identification
  packageNumber: integer("package_number").notNull(), // Package number (1, 2, 3, ...) for multi-package products
  packageName: varchar("package_name", { length: 255 }), // Optional name like "Karton główny", "Karton z półkami"
  
  // Package dimensions
  length: decimal("length", { precision: 10, scale: 2 }), // cm
  width: decimal("width", { precision: 10, scale: 2 }), // cm
  height: decimal("height", { precision: 10, scale: 2 }), // cm
  weight: decimal("weight", { precision: 10, scale: 2 }), // kg
  
  // Package location (can differ from main product location)
  locationId: integer("location_id"), // FK to production.production_locations
  carrierId: integer("carrier_id"), // FK to production.production_carriers
  
  // Notes specific to this package
  notes: text("notes"),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_packed_product_packages_product").on(table.packedProductId),
  index("idx_packed_product_packages_location").on(table.locationId),
  index("idx_packed_product_packages_carrier").on(table.carrierId),
  uniqueIndex("idx_packed_product_packages_unique").on(table.packedProductId, table.packageNumber),
]);

// Packed Product Items - Individual serialized items (each physical product is one record)
// This enables FIFO tracking, individual defect marking, and product transformations
export const packedProductItems = warehouseSchema.table("packed_product_items", {
  id: serial("id").primaryKey(),
  
  // Reference to aggregate product type (for grouping)
  packedProductId: integer("packed_product_id").references(() => packedProducts.id, { onDelete: "restrict" }).notNull(),
  
  // Product identification (denormalized for faster queries)
  catalogProductId: integer("catalog_product_id"), // FK to catalog.products
  catalogSetId: integer("catalog_set_id"), // FK to product_creator.product_sets
  productSku: varchar("product_sku", { length: 100 }).notNull(),
  
  // Unique serial number for this physical item (format: {catalogProductId}-{YYYYMMDD}-{seq})
  serialNumber: varchar("serial_number", { length: 50 }).notNull().unique(),
  
  // Status tracking
  status: varchar("status", { length: 20 }).notNull().default("available"), // available, reserved, consumed, defective, transformed
  
  // Production provenance
  bomId: integer("bom_id"), // FK to bom.product_boms - which BOM was used
  productionPlanLineId: integer("production_plan_line_id"), // FK to production.production_plan_lines - from which plan line
  productionOrderId: integer("production_order_id"), // FK to production.production_orders
  
  // Reservation tracking
  reservationId: integer("reservation_id"), // FK to production.production_buffer_reservations
  reservedForPlanLineId: integer("reserved_for_plan_line_id"), // Which plan line reserved this item
  reservedAt: timestamp("reserved_at"),
  
  // Dates for FIFO
  producedAt: timestamp("produced_at"), // When was this item produced
  packedAt: timestamp("packed_at").defaultNow(), // When was it packed (creation date)
  consumedAt: timestamp("consumed_at"), // When was it consumed (shipped, used, etc.)
  
  // Warehouse location
  locationId: integer("location_id"), // FK to production.production_locations
  carrierId: integer("carrier_id"), // FK to production.production_carriers
  
  // Defect/transformation tracking
  defectDescription: text("defect_description"), // If status=defective, what's wrong
  transformedFromItemId: integer("transformed_from_item_id"), // If created by transforming another item
  transformedToProductId: integer("transformed_to_product_id"), // If this item was transformed into something else
  
  // Metadata for additional info
  metadata: jsonb("metadata").$type<Record<string, any>>(),
  
  notes: text("notes"),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_packed_product_items_packed_product").on(table.packedProductId),
  index("idx_packed_product_items_catalog_product").on(table.catalogProductId),
  index("idx_packed_product_items_sku").on(table.productSku),
  index("idx_packed_product_items_serial").on(table.serialNumber),
  index("idx_packed_product_items_status").on(table.status),
  index("idx_packed_product_items_production_plan").on(table.productionPlanLineId),
  index("idx_packed_product_items_reserved_for").on(table.reservedForPlanLineId),
  index("idx_packed_product_items_location").on(table.locationId),
  // Composite index for FIFO queries (available items, ordered by packed date)
  index("idx_packed_product_items_fifo").on(table.packedProductId, table.status, table.packedAt),
]);

export const insertPackedProductSchema = createInsertSchema(packedProducts).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertPackedProductPackageSchema = createInsertSchema(packedProductPackages).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertPackedProductItemSchema = createInsertSchema(packedProductItems).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type PackedProduct = typeof packedProducts.$inferSelect;
export type InsertPackedProduct = z.infer<typeof insertPackedProductSchema>;
export type PackedProductPackage = typeof packedProductPackages.$inferSelect;
export type InsertPackedProductPackage = z.infer<typeof insertPackedProductPackageSchema>;
export type PackedProductItem = typeof packedProductItems.$inferSelect;
export type InsertPackedProductItem = z.infer<typeof insertPackedProductItemSchema>;

// ==================== BOM Schema ====================
// Schemat dla Bill of Materials - zarządzanie komponentami i produkcją
export const bomSchema = pgSchema("bom");

// Component Templates - Szablony formatek z matrycy Airtable
export const componentTemplates = bomSchema.table("component_templates", {
  id: serial("id").primaryKey(),
  
  // Części nazwy (z Airtable: cz1, cz2)
  cz1: varchar("cz1", { length: 100 }).notNull(), // "BOCZKI", "BOK-L", "DNO", "DRZWI", itp.
  cz2: varchar("cz2", { length: 100 }), // "SZUFLADA", "TRES", "NADST", "VB", itp.
  
  // Rodzaj mebla (z Airtable: Rodzaj)
  furnitureType: varchar("furniture_type", { length: 100 }).notNull(), // "SUPRA", "TRES-30", "VB", "ALTUS"
  
  // Wymiary bazowe formatki (z Airtable: Długość 50, Szerokość 30)
  baseLength: decimal("base_length", { precision: 10, scale: 2 }), // np. 100, 462, 760
  baseWidth: decimal("base_width", { precision: 10, scale: 2 }), // np. 260, 279, 280
  thickness: decimal("thickness", { precision: 10, scale: 2 }), // 18mm, 3mm
  
  // Wymiary mebla dla których ma zastosowanie (z Airtable: Długość Mebla, Szerokość Mebla)
  furnitureLengthCondition: integer("furniture_length_condition"), // 50, 60, 80, 100, 120
  furnitureWidthCondition: integer("furniture_width_condition"), // 30, 36, 55, 65
  
  // Alternatywne wymiary z wartościami i wzorami
  // Format: [{ dimension: 30, value: 262, formula: "dlSzaf - 38" }, { dimension: 50, value: 462, formula: "dlSzaf - 38" }]
  alternativeLengths: jsonb("alternative_lengths").default([]).$type<Array<{
    dimension: number;  // 30, 50, 60, 80, 100, 120
    value: number | null;  // Obliczona wartość
    formula: string | null;  // Wzór np. "dlSzaf - 38", "dlSzaf / 2", null
  }>>(),
  alternativeWidths: jsonb("alternative_widths").default([]).$type<Array<{
    dimension: number;  // 30, 36, 46, 55, 65
    value: number | null;  // Obliczona wartość
    formula: string | null;  // Wzór np. "szSzaf - 20", null
  }>>(),
  
  // Płyta i kolor (z Airtable: plyty, Kolor)
  plateType: varchar("plate_type", { length: 100 }).notNull(), // "18_BIALY", "HDF_BIALY", "18_CZARNY"
  color: varchar("color", { length: 100 }).notNull(), // "BIALY", "CZARNY", "SUROWA", "HDF-BIALY"
  
  // Obrzeża - 4 pozycje (góra po długości, prawo po szerokości, dół po długości, lewo po szerokości)
  edge1: boolean("edge1").default(false), // Obrzeże 1 - po długości góra (top edge along length)
  edge2: boolean("edge2").default(false), // Obrzeże 2 - po szerokości prawo (right edge along width)
  edge3: boolean("edge3").default(false), // Obrzeże 3 - po długości dół (bottom edge along length)
  edge4: boolean("edge4").default(false), // Obrzeże 4 - po szerokości lewo (left edge along width)
  edgingMaterial: varchar("edging_material", { length: 100 }), // "0.8_BIALY", "0.8_CZARNY"
  
  // Deprecated - kept for backward compatibility (use edge1-4 instead)
  edgingPattern: varchar("edging_pattern", { length: 10 }), // DEPRECATED: "TTTT", "TNNN", "NNNN"
  
  // Opcje (z Airtable: różne flagi)
  drillingRequired: boolean("drilling_required").default(false), // Wiercenie
  noColorChange: boolean("no_color_change").default(false), // Brak zmiany koloru
  excludeFromCutting: boolean("exclude_from_cutting").default(false), // Wyklucz z cięcia
  applyByLength: boolean("apply_by_length").default(false), // Stosuj po Długości
  applyByWidth: boolean("apply_by_width").default(false), // Stosuj po Szerokości
  
  // Deprecated - kept for backward compatibility
  halfPlate: boolean("half_plate").default(false), // DEPRECATED: Płyta na pół
  
  // Kategoria produkcyjna (z Airtable: rodzajText)
  productionCategory: varchar("production_category", { length: 200 }), // "Bok Supra", "Drzwi VB", "Siedzisko"
  
  // Cena (z Airtable: Cena wiercenia)
  unitPrice: decimal("unit_price", { precision: 10, scale: 2 }),
  
  // Link do materiału w magazynie
  warehouseMaterialId: integer("warehouse_material_id"), // FK do warehouse.materials
  
  // Warianty drzwi i nóżek (dla wariantów CZ2)
  door: varchar("door", { length: 50 }), // "D1", "D2", "D3" ze słownika door
  leg: varchar("leg", { length: 50 }), // "N1", "N2", "N3", "N4" ze słownika leg
  
  // Hierarchia parent-branch (automatyczna przez duplikowanie)
  parentId: integer("parent_id").references((): AnyPgColumn => componentTemplates.id), // FK do parenta, NULL = parent
  
  // Metadata
  airtableRecordId: varchar("airtable_record_id", { length: 100 }), // Oryginalny ID z Airtable
  displayOrder: integer("display_order").default(0),
  isActive: boolean("is_active").default(true),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_component_templates_furniture_type").on(table.furnitureType),
  index("idx_component_templates_cz1").on(table.cz1),
  index("idx_component_templates_active").on(table.isActive),
  index("idx_component_templates_warehouse").on(table.warehouseMaterialId),
  index("idx_component_templates_parent").on(table.parentId),
]);

// Product Components - Wygenerowane komponenty dla konkretnego produktu
export const productComponents = bomSchema.table("product_components", {
  id: serial("id").primaryKey(),
  productId: integer("product_id").notNull(), // FK do catalog.products
  templateId: integer("template_id").references(() => componentTemplates.id, { onDelete: "set null" }),
  
  // Nazwa wygenerowana według formuły
  generatedName: varchar("generated_name", { length: 255 }).notNull(), // "BOCZKI-30-SZUFLADA-100x260-CZARNY"
  
  // Typ komponentu
  componentType: varchar("component_type", { length: 50 }).notNull(), // "board", "edge", "glue", "fitting", "fabric", "packaging"
  
  // Obliczone wymiary
  calculatedLength: decimal("calculated_length", { precision: 10, scale: 2 }),
  calculatedWidth: decimal("calculated_width", { precision: 10, scale: 2 }),
  thickness: decimal("thickness", { precision: 10, scale: 2 }),
  
  // Kolor formatki
  color: varchar("color", { length: 50 }), // Kod koloru z dictionaries (np. "WOTAN", "LANCELOT")
  
  // Typ płyty
  boardType: varchar("board_type", { length: 50 }), // "MDF", "płyta wiórowa", "sklejka"
  
  // Wzór wiercenia
  drillingPattern: varchar("drilling_pattern", { length: 50 }), // Wzór otworów wiertarskich
  
  // Obrzeża
  edgingPattern: varchar("edging_pattern", { length: 10 }), // "TTTT"
  edgingMaterial: varchar("edging_material", { length: 100 }),
  
  // Które krawędzie obrzeżone (kopiowane z component_templates)
  edge1: boolean("edge1").default(false), // Krawędź 1 - po długości góra
  edge2: boolean("edge2").default(false), // Krawędź 2 - po szerokości prawo
  edge3: boolean("edge3").default(false), // Krawędź 3 - po długości dół
  edge4: boolean("edge4").default(false), // Krawędź 4 - po szerokości lewo
  
  // Link do materiału
  warehouseMaterialId: integer("warehouse_material_id"), // FK do warehouse.materials
  
  // Ilość i jednostka
  quantity: decimal("quantity", { precision: 10, scale: 3 }).notNull().default("1"),
  unitOfMeasure: varchar("unit_of_measure", { length: 20 }).default("pcs"), // pcs, sqm, m, kg
  
  // Uwagi produkcyjne
  productionNotes: text("production_notes"),
  supplierNotes: text("supplier_notes"),
  
  // Kolejność w BOM
  positionInBom: integer("position_in_bom").default(0),
  
  // Czy obowiązkowy
  isMandatory: boolean("is_mandatory").default(true),
  
  // Wizualizacja formatki
  visualizationUrl: text("visualization_url"), // URL do wygenerowanego obrazu formatki
  
  // Śledzenie uszkodzeń komponentów
  isDamaged: boolean("is_damaged").default(false),
  damageNotes: text("damage_notes"),
  damagedAt: timestamp("damaged_at"),
  damagedBy: integer("damaged_by"), // FK do users.id
  
  // Odoo integration
  odooProductId: integer("odoo_product_id"), // ID produktu w Odoo (formatka jako produkt)
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_product_components_product").on(table.productId),
  index("idx_product_components_template").on(table.templateId),
  index("idx_product_components_warehouse").on(table.warehouseMaterialId),
  index("idx_product_components_type").on(table.componentType),
]);

// Product Component Metrics - Przeliczone zużycie i koszty materiałów (osobna tabela)
export const productComponentMetrics = bomSchema.table("product_component_metrics", {
  id: serial("id").primaryKey(),
  componentId: integer("component_id").notNull().references(() => productComponents.id, { onDelete: "cascade" }).unique(),
  
  // Przeliczone zużycie materiałów
  boardAreaM2: decimal("board_area_m2", { precision: 10, scale: 6 }), // Powierzchnia płyty w m²
  edgingLengthM: decimal("edging_length_m", { precision: 10, scale: 3 }), // Długość obrzeża w metrach
  glueAmountG: decimal("glue_amount_g", { precision: 10, scale: 2 }), // Ilość kleju w gramach
  
  // Linki do materiałów z magazynu
  boardMaterialId: integer("board_material_id"), // FK do warehouse.materials (płyta)
  edgingMaterialId: integer("edging_material_id"), // FK do warehouse.materials (obrzeże)
  glueMaterialId: integer("glue_material_id"), // FK do warehouse.materials (klej)
  
  // Przeliczone koszty
  boardCost: decimal("board_cost", { precision: 10, scale: 2 }), // Koszt płyty w PLN
  edgingCost: decimal("edging_cost", { precision: 10, scale: 2 }), // Koszt obrzeża w PLN
  glueCost: decimal("glue_cost", { precision: 10, scale: 2 }), // Koszt kleju w PLN
  totalMaterialCost: decimal("total_material_cost", { precision: 10, scale: 2 }), // Suma kosztów
  
  // Timestamp ostatniego przeliczenia (do wykrywania nieaktualnych danych)
  calculatedAt: timestamp("calculated_at").defaultNow(),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_component_metrics_component").on(table.componentId),
  index("idx_component_metrics_board_material").on(table.boardMaterialId),
  index("idx_component_metrics_edging_material").on(table.edgingMaterialId),
  index("idx_component_metrics_calculated_at").on(table.calculatedAt),
]);

// Product BOMs - Główna tabela Bill of Materials
export const productBoms = bomSchema.table("product_boms", {
  id: serial("id").primaryKey(),
  productId: integer("product_id").notNull().unique(), // FK do catalog.products - jeden BOM na produkt
  
  // Nazwa i wersja BOM
  name: varchar("name", { length: 255 }).notNull(),
  version: varchar("version", { length: 50 }).default("1.0"),
  
  // Typ BOM
  bomType: varchar("bom_type", { length: 50 }).default("final_product"), // final_product, semi_finished, component
  
  // Lokalizacja produkcji
  productionLocation: varchar("production_location", { length: 100 }).default("internal"), // internal, external_upholstery, external_fittings
  
  // Szacowany czas produkcji (w minutach)
  estimatedProductionTimeMinutes: integer("estimated_production_time_minutes"),
  
  // Uwagi
  notes: text("notes"),
  
  // Status
  isActive: boolean("is_active").default(true),
  isApproved: boolean("is_approved").default(false), // Czy zatwierdzony do produkcji
  approvedBy: integer("approved_by"), // FK do users.id
  approvedAt: timestamp("approved_at"),
  
  // Synchronizacja z Odoo
  odooBomId: integer("odoo_bom_id"), // ID BOM w Odoo
  odooSyncedAt: timestamp("odoo_synced_at"),
  odooSyncStatus: varchar("odoo_sync_status", { length: 50 }), // pending, synced, failed
  odooSyncError: text("odoo_sync_error"),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_product_boms_product").on(table.productId),
  index("idx_product_boms_active").on(table.isActive),
  index("idx_product_boms_approved").on(table.isApproved),
  index("idx_product_boms_odoo_id").on(table.odooBomId),
  index("idx_product_boms_odoo_status").on(table.odooSyncStatus),
]);

// BOM Operations - Operacje produkcyjne dla BOM
export const bomOperations = bomSchema.table("bom_operations", {
  id: serial("id").primaryKey(),
  bomId: integer("bom_id").references(() => productBoms.id, { onDelete: "cascade" }).notNull(),
  
  // Typ operacji
  operationType: varchar("operation_type", { length: 50 }).notNull(), // cutting, edging, drilling, assembly, upholstery, packing
  
  // Nazwa operacji
  operationName: varchar("operation_name", { length: 255 }).notNull(),
  
  // Kolejność wykonania
  sequence: integer("sequence").default(0).notNull(),
  
  // Stanowisko pracy
  workstation: varchar("workstation", { length: 100 }), // "Piła formatowa", "Okleiniarka", "Wiertarka CNC"
  
  // Szacowany czas (w minutach)
  estimatedTimeMinutes: integer("estimated_time_minutes"),
  
  // Instrukcje
  instructions: text("instructions"),
  
  // Synchronizacja z Odoo
  odooOperationId: integer("odoo_operation_id"),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_bom_operations_bom").on(table.bomId),
  index("idx_bom_operations_type").on(table.operationType),
  index("idx_bom_operations_sequence").on(table.sequence),
]);

// Insert schemas
export const insertComponentTemplateSchema = createInsertSchema(componentTemplates).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductComponentSchema = createInsertSchema(productComponents).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductBomSchema = createInsertSchema(productBoms).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertBomOperationSchema = createInsertSchema(bomOperations).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductComponentMetricsSchema = createInsertSchema(productComponentMetrics).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  calculatedAt: true,
});

// Types
export type ComponentTemplate = typeof componentTemplates.$inferSelect;
export type InsertComponentTemplate = z.infer<typeof insertComponentTemplateSchema>;

export type ProductComponent = typeof productComponents.$inferSelect;
export type InsertProductComponent = z.infer<typeof insertProductComponentSchema>;

export type ProductComponentMetrics = typeof productComponentMetrics.$inferSelect;
export type InsertProductComponentMetrics = z.infer<typeof insertProductComponentMetricsSchema>;

export type ProductBom = typeof productBoms.$inferSelect;
export type InsertProductBom = z.infer<typeof insertProductBomSchema>;

export type BomOperation = typeof bomOperations.$inferSelect;
export type InsertBomOperation = z.infer<typeof insertBomOperationSchema>;

// Composite types for BOM API responses
export interface ProductComponentWithMetrics extends ProductComponent {
  boardAreaM2?: number | null;
  edgingLengthM?: number | null;
  glueAmountG?: number | null;
  boardMaterialId?: number | null;
  boardMaterialName?: string | null;
  edgingMaterialId?: number | null;
  edgingMaterialName?: string | null;
  glueMaterialId?: number | null;
  glueMaterialName?: string | null;
  boardCost?: number | null;
  edgingCost?: number | null;
  glueCost?: number | null;
  totalMaterialCost?: number | null;
  // Template fields (from component_templates LEFT JOIN)
  templateCz1?: string | null;
  templateCz2?: string | null;
}

export interface ProductBOMResponse {
  bom: ProductBom;
  product: {
    id: number;
    sku: string;
    title: string;
    imageUrl: string | null;
  };
  components: ProductComponentWithMetrics[];
}

// ==================== Production Schema ====================
// Dedykowany schemat PostgreSQL dla modułu produkcji
export const productionSchema = pgSchema("production");

// Production Settings - globalne ustawienia produkcji
export const productionSettings = productionSchema.table("production_settings", {
  id: serial("id").primaryKey(),
  key: varchar("key", { length: 100 }).notNull().unique(),
  value: text("value").notNull(),
  valueType: varchar("value_type", { length: 20 }).notNull().default("string"), // string, number, boolean, json
  description: text("description"),
  category: varchar("category", { length: 50 }).notNull().default("general"), // general, norms, carriers, locations
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_production_settings_category").on(table.category),
  index("idx_production_settings_key").on(table.key),
]);

// Production Norm Groups - grupy normatywów (np. "Czasy obróbki", "Zużycie materiałów")
export const productionNormGroups = productionSchema.table("production_norm_groups", {
  id: serial("id").primaryKey(),
  code: varchar("code", { length: 50 }).notNull().unique(),
  name: varchar("name", { length: 255 }).notNull(),
  description: text("description"),
  unitOfMeasure: varchar("unit_of_measure", { length: 50 }), // min, szt, kg, m, etc.
  isActive: boolean("is_active").default(true),
  sortOrder: integer("sort_order").default(0),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Production Norms - normatywy produkcyjne (czasy, materiały, koszty)
export const productionNorms = productionSchema.table("production_norms", {
  id: serial("id").primaryKey(),
  normGroupId: integer("norm_group_id").references(() => productionNormGroups.id, { onDelete: "cascade" }).notNull(),
  code: varchar("code", { length: 100 }).notNull(),
  name: varchar("name", { length: 255 }).notNull(),
  description: text("description"),
  defaultValue: decimal("default_value", { precision: 10, scale: 2 }).notNull(),
  minValue: decimal("min_value", { precision: 10, scale: 2 }),
  maxValue: decimal("max_value", { precision: 10, scale: 2 }),
  tolerance: decimal("tolerance", { precision: 5, scale: 2 }), // % odchylenia
  unitOfMeasure: varchar("unit_of_measure", { length: 50 }),
  applicableTo: jsonb("applicable_to"), // { carriers: [], locations: [], workCenters: [] }
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  uniqueIndex("idx_production_norms_group_code").on(table.normGroupId, table.code),
  index("idx_production_norms_group").on(table.normGroupId),
  index("idx_production_norms_active").on(table.isActive),
]);

// Production Carrier Groups - grupy nośników (np. "Palety", "Wózki", "Kontenery")
export const productionCarrierGroups = productionSchema.table("production_carrier_groups", {
  id: serial("id").primaryKey(),
  code: varchar("code", { length: 50 }).notNull().unique(),
  name: varchar("name", { length: 255 }).notNull(),
  description: text("description"),
  defaultCapacity: decimal("default_capacity", { precision: 10, scale: 2 }),
  capacityUnit: varchar("capacity_unit", { length: 50 }), // szt, kg, m3
  isActive: boolean("is_active").default(true),
  sortOrder: integer("sort_order").default(0),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Production Carriers - nośniki produkcyjne (palety, wózki, kontenery)
export const productionCarriers = productionSchema.table("production_carriers", {
  id: serial("id").primaryKey(),
  carrierGroupId: integer("carrier_group_id").references(() => productionCarrierGroups.id, { onDelete: "cascade" }).notNull(),
  code: varchar("code", { length: 100 }).notNull(),
  name: varchar("name", { length: 255 }).notNull(),
  barcode: varchar("barcode", { length: 100 }),
  status: varchar("status", { length: 20 }).notNull().default("available"), // available, in_use, maintenance, retired
  capacity: decimal("capacity", { precision: 10, scale: 2 }),
  capacityUnit: varchar("capacity_unit", { length: 50 }),
  currentLoad: decimal("current_load", { precision: 10, scale: 2 }).default("0"),
  dimensions: jsonb("dimensions"), // { length, width, height, unit }
  weight: decimal("weight", { precision: 10, scale: 2 }),
  currentLocationId: integer("current_location_id"),
  defaultLocationId: integer("default_location_id"),
  lastMaintenanceDate: timestamp("last_maintenance_date"),
  nextMaintenanceDate: timestamp("next_maintenance_date"),
  notes: text("notes"),
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  uniqueIndex("idx_production_carriers_group_code").on(table.carrierGroupId, table.code),
  index("idx_production_carriers_status_group").on(table.status, table.carrierGroupId),
  index("idx_production_carriers_barcode").on(table.barcode),
  index("idx_production_carriers_location").on(table.currentLocationId),
]);

// Production Location Groups - grupy lokalizacji (np. "Hale", "Magazyny", "Strefy")
export const productionLocationGroups = productionSchema.table("production_location_groups", {
  id: serial("id").primaryKey(),
  code: varchar("code", { length: 50 }).notNull().unique(),
  name: varchar("name", { length: 255 }).notNull(),
  description: text("description"),
  locationType: varchar("location_type", { length: 50 }).notNull(), // hall, warehouse, zone, rack, shelf
  allowsSublocations: boolean("allows_sublocations").default(true),
  isActive: boolean("is_active").default(true),
  sortOrder: integer("sort_order").default(0),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Production Locations - lokalizacje hierarchiczne (hala -> stanowisko -> regał -> półka)
export const productionLocations = productionSchema.table("production_locations", {
  id: serial("id").primaryKey(),
  locationGroupId: integer("location_group_id").references(() => productionLocationGroups.id, { onDelete: "cascade" }).notNull(),
  parentId: integer("parent_id").references((): AnyPgColumn => productionLocations.id, { onDelete: "cascade" }),
  code: varchar("code", { length: 100 }).notNull().unique(),
  name: varchar("name", { length: 255 }).notNull(),
  path: text("path"), // Hierarchical path: /hala1/stanowisko1/regal1
  level: integer("level").default(0), // 0 = root, 1 = child, 2 = grandchild, etc.
  barcode: varchar("barcode", { length: 100 }),
  capacity: decimal("capacity", { precision: 10, scale: 2 }),
  capacityUnit: varchar("capacity_unit", { length: 50 }),
  currentLoad: decimal("current_load", { precision: 10, scale: 2 }).default("0"),
  dimensions: jsonb("dimensions"), // { length, width, height, unit }
  status: varchar("status", { length: 20 }).notNull().default("active"), // active, inactive, maintenance, full
  allowsStorage: boolean("allows_storage").default(true),
  isActive: boolean("is_active").default(true),
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_production_locations_parent").on(table.parentId),
  index("idx_production_locations_group_level").on(table.locationGroupId, table.level),
  index("idx_production_locations_path").on(table.path),
  index("idx_production_locations_status").on(table.status),
  index("idx_production_locations_barcode").on(table.barcode),
]);

// Add foreign key constraints for carrier locations (delayed to avoid circular dependency)
// This will be handled in migrations

// Production Work Centers - gniazda produkcyjne (stanowiska pracy)
export const productionWorkCenters = productionSchema.table("production_work_centers", {
  id: serial("id").primaryKey(),
  code: varchar("code", { length: 100 }).notNull().unique(),
  name: varchar("name", { length: 255 }).notNull(),
  description: text("description"),
  locationId: integer("location_id").references(() => productionLocations.id, { onDelete: "set null" }),
  capabilities: text("capabilities").array().default(sql`ARRAY[]::text[]`), // ["CNC", "Laser", "Montaż"]
  status: varchar("status", { length: 20 }).notNull().default("available"), // available, busy, maintenance, offline
  throughputPerHour: decimal("throughput_per_hour", { precision: 10, scale: 2 }),
  throughputUnit: varchar("throughput_unit", { length: 50 }),
  operatingCostPerHour: decimal("operating_cost_per_hour", { precision: 10, scale: 2 }),
  shiftCalendar: jsonb("shift_calendar"), // { shifts: [{ start, end, days }] }
  oeeTarget: decimal("oee_target", { precision: 5, scale: 2 }), // Overall Equipment Effectiveness %
  currentOee: decimal("current_oee", { precision: 5, scale: 2 }),
  isActive: boolean("is_active").default(true),
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_production_work_centers_status").on(table.status),
  index("idx_production_work_centers_location").on(table.locationId),
  index("idx_production_work_centers_active").on(table.isActive),
]);

// Production Routings - marszruty produkcyjne (ścieżki produkcyjne)
export const productionRoutings = productionSchema.table("production_routings", {
  id: serial("id").primaryKey(),
  code: varchar("code", { length: 100 }).notNull().unique(),
  name: varchar("name", { length: 255 }).notNull(),
  description: text("description"),
  productId: integer("product_id"), // Optional: routing for specific product (from catalog.products)
  productType: varchar("product_type", { length: 100 }), // Or routing for product type
  defaultWorkCenterId: integer("default_work_center_id").references(() => productionWorkCenters.id, { onDelete: "set null" }),
  totalEstimatedTime: decimal("total_estimated_time", { precision: 10, scale: 2 }), // In minutes
  isActive: boolean("is_active").default(true),
  version: integer("version").default(1),
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_production_routings_product").on(table.productId),
  index("idx_production_routings_type").on(table.productType),
  index("idx_production_routings_active").on(table.isActive),
]);

// Production Routing Operations - operacje w marszrutach
export const productionRoutingOperations = productionSchema.table("production_routing_operations", {
  id: serial("id").primaryKey(),
  routingId: integer("routing_id").references(() => productionRoutings.id, { onDelete: "cascade" }).notNull(),
  sequence: integer("sequence").notNull(),
  code: varchar("code", { length: 100 }).notNull(),
  name: varchar("name", { length: 255 }).notNull(),
  description: text("description"),
  workCenterId: integer("work_center_id").references(() => productionWorkCenters.id, { onDelete: "set null" }),
  normId: integer("norm_id").references(() => productionNorms.id, { onDelete: "set null" }),
  estimatedTime: decimal("estimated_time", { precision: 10, scale: 2 }), // In minutes
  setupTime: decimal("setup_time", { precision: 10, scale: 2 }), // Setup time in minutes
  allowParallel: boolean("allow_parallel").default(false), // Can run in parallel with others
  prerequisiteOperationIds: jsonb("prerequisite_operation_ids"), // [id1, id2] - operations that must complete first
  instructions: text("instructions"),
  qualityCheckRequired: boolean("quality_check_required").default(false),
  isActive: boolean("is_active").default(true),
  // Buffer/Warehouse integration - półprodukty do magazynu
  createsBuffer: boolean("creates_buffer").default(false),
  bufferLocationId: integer("buffer_location_id").references(() => productionLocations.id, { onDelete: "set null" }),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  uniqueIndex("idx_production_routing_ops_routing_seq").on(table.routingId, table.sequence),
  index("idx_production_routing_ops_routing").on(table.routingId),
  index("idx_production_routing_ops_workcenter").on(table.workCenterId),
]);

// Production Buffer Stock - stany magazynowe półproduktów
export const productionBufferStock = productionSchema.table("production_buffer_stock", {
  id: serial("id").primaryKey(),
  productSku: varchar("product_sku", { length: 255 }).notNull(),
  productName: varchar("product_name", { length: 500 }),
  locationId: integer("location_id").references(() => productionLocations.id, { onDelete: "set null" }),
  quantityAvailable: decimal("quantity_available", { precision: 10, scale: 2 }).notNull().default("0"),
  quantityReserved: decimal("quantity_reserved", { precision: 10, scale: 2 }).notNull().default("0"),
  quantityTotal: decimal("quantity_total", { precision: 10, scale: 2 }).notNull().default("0"),
  unitOfMeasure: varchar("unit_of_measure", { length: 20 }).default("szt"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  uniqueIndex("idx_buffer_stock_sku_location").on(table.productSku, table.locationId),
  index("idx_buffer_stock_sku").on(table.productSku),
  index("idx_buffer_stock_location").on(table.locationId),
]);

// Production Buffer Movements - ruchy magazynowe półproduktów
export const productionBufferMovements = productionSchema.table("production_buffer_movements", {
  id: serial("id").primaryKey(),
  movementType: varchar("movement_type", { length: 50 }).notNull(), // IN, RESERVE, RELEASE, OUT
  productSku: varchar("product_sku", { length: 255 }).notNull(),
  quantity: decimal("quantity", { precision: 10, scale: 2 }).notNull(),
  unitOfMeasure: varchar("unit_of_measure", { length: 20 }).default("szt"),
  locationId: integer("location_id").references(() => productionLocations.id, { onDelete: "set null" }),
  sourceType: varchar("source_type", { length: 50 }), // WORK_ORDER, ZLP, MANUAL
  sourceId: integer("source_id"),
  zlpId: integer("zlp_id").references((): AnyPgColumn => productionPlans.id, { onDelete: "set null" }),
  reservationId: integer("reservation_id").references((): AnyPgColumn => productionBufferReservations.id, { onDelete: "set null" }),
  createdBy: integer("created_by").references(() => users.id, { onDelete: "set null" }),
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow(),
}, (table) => [
  index("idx_buffer_movements_type").on(table.movementType),
  index("idx_buffer_movements_sku").on(table.productSku),
  index("idx_buffer_movements_location").on(table.locationId),
  index("idx_buffer_movements_source").on(table.sourceType, table.sourceId),
  index("idx_buffer_movements_zlp").on(table.zlpId),
  index("idx_buffer_movements_created_at").on(table.createdAt),
]);

// Production Buffer Reservations - rezerwacje półproduktów dla ZLP
export const productionBufferReservations = productionSchema.table("production_buffer_reservations", {
  id: serial("id").primaryKey(),
  zlpId: integer("zlp_id").notNull().references((): AnyPgColumn => productionPlans.id, { onDelete: "cascade" }),
  zlpItemId: integer("zlp_item_id"),
  productSku: varchar("product_sku", { length: 255 }).notNull(),
  quantityReserved: decimal("quantity_reserved", { precision: 10, scale: 2 }).notNull(),
  quantityConsumed: decimal("quantity_consumed", { precision: 10, scale: 2 }).default("0"),
  unitOfMeasure: varchar("unit_of_measure", { length: 20 }).default("szt"),
  locationId: integer("location_id").references(() => productionLocations.id, { onDelete: "set null" }),
  status: varchar("status", { length: 50 }).default("ACTIVE"), // ACTIVE, CONSUMED, CANCELLED
  reservedAt: timestamp("reserved_at").defaultNow(),
  reservedBy: integer("reserved_by").references(() => users.id, { onDelete: "set null" }),
  consumedAt: timestamp("consumed_at"),
  cancelledAt: timestamp("cancelled_at"),
  notes: text("notes"),
}, (table) => [
  index("idx_buffer_reservations_zlp").on(table.zlpId),
  index("idx_buffer_reservations_sku").on(table.productSku),
  index("idx_buffer_reservations_location").on(table.locationId),
  index("idx_buffer_reservations_status").on(table.status),
]);

// Production Orders - zlecenia produkcyjne (ZLP)
export const productionOrders = productionSchema.table("production_orders", {
  id: serial("id").primaryKey(),
  orderNumber: varchar("order_number", { length: 100 }).notNull().unique(),
  productId: integer("product_id").notNull(), // From catalog.products
  bomId: integer("bom_id"), // From bom.product_boms
  routingId: integer("routing_id").references(() => productionRoutings.id, { onDelete: "set null" }),
  status: varchar("status", { length: 20 }).notNull().default("draft"), // draft, confirmed, planned, in_progress, paused, done, cancelled
  workflowStage: varchar("workflow_stage", { length: 30 }), // cutting, edging, drilling, upholstering, picking, packing, strapping, ready, shipped
  priority: varchar("priority", { length: 20 }).notNull().default("normal"), // low, normal, high, urgent
  quantityPlanned: decimal("quantity_planned", { precision: 10, scale: 2 }).notNull(),
  quantityProduced: decimal("quantity_produced", { precision: 10, scale: 2 }).default("0"),
  quantityScrap: decimal("quantity_scrap", { precision: 10, scale: 2 }).default("0"),
  unitOfMeasure: varchar("unit_of_measure", { length: 50 }).notNull().default("szt"),
  plannedStartDate: timestamp("planned_start_date"),
  plannedEndDate: timestamp("planned_end_date"),
  actualStartDate: timestamp("actual_start_date"),
  actualEndDate: timestamp("actual_end_date"),
  workflowStageUpdatedAt: timestamp("workflow_stage_updated_at"),
  workflowStageUpdatedBy: integer("workflow_stage_updated_by"),
  responsibleUserId: integer("responsible_user_id"),
  sourceOrderNumber: varchar("source_order_number", { length: 100 }), // Link to sales order
  carrierRequirements: jsonb("carrier_requirements"), // Required carriers for this order
  locationId: integer("location_id").references(() => productionLocations.id, { onDelete: "set null" }),
  parameters: jsonb("parameters"), // Custom parameters for production
  colorCode: varchar("color_code", { length: 50 }), // Kolor do agregacji ZLP (WOTAN, BIALY, CZARNY, SUROWA)
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_production_orders_status_date").on(table.status, table.plannedStartDate),
  index("idx_production_orders_workflow_stage").on(table.workflowStage),
  index("idx_production_orders_product").on(table.productId),
  index("idx_production_orders_bom").on(table.bomId),
  index("idx_production_orders_routing").on(table.routingId),
  index("idx_production_orders_priority").on(table.priority),
  index("idx_production_orders_location").on(table.locationId),
  index("idx_production_orders_color").on(table.colorCode),
]);

// Production Work Orders - zlecenia robocze (generowane z ZLP dla każdej operacji)
export const productionWorkOrders = productionSchema.table("production_work_orders", {
  id: serial("id").primaryKey(),
  workOrderNumber: varchar("work_order_number", { length: 100 }).notNull().unique(),
  productionOrderId: integer("production_order_id").references(() => productionOrders.id, { onDelete: "cascade" }).notNull(),
  routingOperationId: integer("routing_operation_id").references(() => productionRoutingOperations.id, { onDelete: "set null" }),
  workCenterId: integer("work_center_id").references(() => productionWorkCenters.id, { onDelete: "set null" }),
  sequence: integer("sequence").notNull(),
  status: varchar("status", { length: 20 }).notNull().default("pending"), // pending, ready, in_progress, paused, done, cancelled
  operatorUserId: integer("operator_user_id"),
  quantityPlanned: decimal("quantity_planned", { precision: 10, scale: 2 }).notNull(),
  quantityProduced: decimal("quantity_produced", { precision: 10, scale: 2 }).default("0"),
  quantityScrap: decimal("quantity_scrap", { precision: 10, scale: 2 }).default("0"),
  estimatedDuration: decimal("estimated_duration", { precision: 10, scale: 2 }), // Minutes
  actualDuration: decimal("actual_duration", { precision: 10, scale: 2 }), // Minutes
  scheduledStartTime: timestamp("scheduled_start_time"),
  scheduledEndTime: timestamp("scheduled_end_time"),
  actualStartTime: timestamp("actual_start_time"),
  actualEndTime: timestamp("actual_end_time"),
  carrierId: integer("carrier_id").references(() => productionCarriers.id, { onDelete: "set null" }),
  locationId: integer("location_id").references(() => productionLocations.id, { onDelete: "set null" }),
  qualityCheckPassed: boolean("quality_check_passed"),
  qualityCheckNotes: text("quality_check_notes"),
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  uniqueIndex("idx_production_work_orders_po_seq").on(table.productionOrderId, table.sequence),
  index("idx_production_work_orders_po_status").on(table.productionOrderId, table.status),
  index("idx_production_work_orders_workcenter").on(table.workCenterId),
  index("idx_production_work_orders_operator").on(table.operatorUserId),
  index("idx_production_work_orders_status").on(table.status),
]);

// Production Order Materials - materiały zaplanowane/użyte w zleceniu (snapshot z BOM)
export const productionOrderMaterials = productionSchema.table("production_order_materials", {
  id: serial("id").primaryKey(),
  productionOrderId: integer("production_order_id").references(() => productionOrders.id, { onDelete: "cascade" }).notNull(),
  materialId: integer("material_id"), // From warehouse.materials
  productId: integer("product_id"), // Or from catalog.products
  code: varchar("code", { length: 100 }).notNull(),
  name: varchar("name", { length: 255 }).notNull(),
  quantityPlanned: decimal("quantity_planned", { precision: 10, scale: 2 }).notNull(),
  quantityReserved: decimal("quantity_reserved", { precision: 10, scale: 2 }).default("0"),
  quantityIssued: decimal("quantity_issued", { precision: 10, scale: 2 }).default("0"),
  quantityConsumed: decimal("quantity_consumed", { precision: 10, scale: 2 }).default("0"),
  quantityReturned: decimal("quantity_returned", { precision: 10, scale: 2 }).default("0"),
  unitOfMeasure: varchar("unit_of_measure", { length: 50 }).notNull(),
  workOrderId: integer("work_order_id").references(() => productionWorkOrders.id, { onDelete: "set null" }), // Optional: consumed in specific work order
  locationId: integer("location_id").references(() => productionLocations.id, { onDelete: "set null" }),
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_production_order_materials_po").on(table.productionOrderId),
  index("idx_production_order_materials_wo").on(table.workOrderId),
  index("idx_production_order_materials_material").on(table.materialId),
]);

// Production Order Logs - historia zmian statusów i wydarzeń w zleceniach
export const productionOrderLogs = productionSchema.table("production_order_logs", {
  id: serial("id").primaryKey(),
  productionOrderId: integer("production_order_id").references(() => productionOrders.id, { onDelete: "cascade" }),
  workOrderId: integer("work_order_id").references(() => productionWorkOrders.id, { onDelete: "cascade" }),
  eventType: varchar("event_type", { length: 50 }).notNull(), // status_change, quantity_update, assignment, note_added, etc.
  eventData: jsonb("event_data"), // { oldStatus, newStatus, quantity, etc. }
  message: text("message"),
  userId: integer("user_id"),
  createdAt: timestamp("created_at").defaultNow(),
}, (table) => [
  index("idx_production_order_logs_po").on(table.productionOrderId),
  index("idx_production_order_logs_wo").on(table.workOrderId),
  index("idx_production_order_logs_type").on(table.eventType),
  index("idx_production_order_logs_created").on(table.createdAt),
]);

// Production Plan Name Templates - szablony nazw dla planów produkcji
export const productionPlanNameTemplates = productionSchema.table("plan_name_templates", {
  id: serial("id").primaryKey(),
  templateName: varchar("template_name", { length: 255 }).notNull().unique(),
  pattern: varchar("pattern", { length: 500 }).notNull(), // np. "Zapas-{date|YYYY-MM-DD}-{seq|3}", "Zamówienia-{date|DD.MM}-{seq|2}"
  description: text("description"),
  isActive: boolean("is_active").default(true),
  sortOrder: integer("sort_order").default(0),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_plan_name_templates_active").on(table.isActive),
]);

// Production Plan Name Counters - liczniki sekwencyjne dla szablonów (per template per day)
export const productionPlanNameCounters = productionSchema.table("plan_name_counters", {
  id: serial("id").primaryKey(),
  templateId: integer("template_id").references(() => productionPlanNameTemplates.id, { onDelete: "cascade" }).notNull(),
  counterDate: varchar("counter_date", { length: 10 }).notNull(), // YYYY-MM-DD
  nextValue: integer("next_value").notNull().default(1),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_plan_name_counters_template_date").on(table.templateId, table.counterDate),
]);

// Production Plans - plany produkcji (nagłówek planu łączący zamówienia z produkcją)
export const productionPlans = productionSchema.table("production_plans", {
  id: serial("id").primaryKey(),
  planNumber: varchar("plan_number", { length: 100 }).notNull().unique(),
  shortName: varchar("short_name", { length: 50 }), // PLAN-00019 format
  name: varchar("name", { length: 255 }).notNull(),
  description: text("description"),
  
  // Name generation metadata (for auditability and debugging)
  nameTemplateId: integer("name_template_id").references(() => productionPlanNameTemplates.id, { onDelete: "set null" }),
  nameSequence: integer("name_sequence"), // Sequence number used in generated name
  nameGeneratedAt: timestamp("name_generated_at"),
  nameGenerationMeta: jsonb("name_generation_meta"), // { resolvedPattern, placeholders, etc. }
  
  // Scheduling
  plannedStartDate: timestamp("planned_start_date"),
  plannedEndDate: timestamp("planned_end_date"),
  actualStartDate: timestamp("actual_start_date"),
  actualEndDate: timestamp("actual_end_date"),
  
  // Status
  status: varchar("status", { length: 30 }).notNull().default("draft"), // draft, approved, in_progress, completed, cancelled
  priority: varchar("priority", { length: 20 }).default("normal"), // low, normal, high, urgent
  
  // Metadata
  notes: text("notes"),
  metadata: jsonb("metadata"),
  
  createdBy: integer("created_by"),
  approvedBy: integer("approved_by"),
  approvedAt: timestamp("approved_at"),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_production_plans_status").on(table.status),
  index("idx_production_plans_dates").on(table.plannedStartDate, table.plannedEndDate),
  index("idx_production_plans_created").on(table.createdAt),
]);

// Production Plan Lines - linie planu (każda linia = produkt + ilość + źródło popytu)
export const productionPlanLines = productionSchema.table("production_plan_lines", {
  id: serial("id").primaryKey(),
  planId: integer("plan_id").references(() => productionPlans.id, { onDelete: "cascade" }).notNull(),
  
  // Product reference
  productId: integer("product_id").notNull(), // From catalog.products
  quantity: integer("quantity").notNull(),
  
  // Warehouse reservation tracking (per-line)
  reservedQuantity: integer("reserved_quantity").notNull().default(0), // Ilość zarezerwowana przez tę linię planu
  
  // Color aggregation - klucz do grupowania produktów po kolorach w ZLP
  colorCode: varchar("color_code", { length: 50 }), // Kod koloru z dictionary (np. "BIALY", "WOTAN", "KASZMIR")
  
  // Demand source (skąd się wzięło zapotrzebowanie)
  sourceType: varchar("source_type", { length: 30 }), // sales_order, forecast, buffer_stock, manual
  sourceId: integer("source_id"), // ID zamówienia Allegro/Shoper jeśli source_type = sales_order
  sourceReference: varchar("source_reference", { length: 255 }), // np. numer zamówienia klienta
  
  // Production order link (po utworzeniu zlecenia)
  productionOrderId: integer("production_order_id").references(() => productionOrders.id, { onDelete: "set null" }),
  
  // Routing override (jeśli ta linia ma specjalny routing)
  routingId: integer("routing_id").references(() => productionRoutings.id, { onDelete: "set null" }),
  routingOverride: jsonb("routing_override"), // { stop_before: "packing", skip_operations: ["drilling"] }
  
  // BOM snapshot reference (opcjonalnie dla auditability)
  bomId: integer("bom_id"), // From bom.product_boms - snapshot w momencie planowania
  
  // Scheduling
  plannedStartDate: timestamp("planned_start_date"),
  plannedEndDate: timestamp("planned_end_date"),
  
  // Status
  status: varchar("status", { length: 30 }).notNull().default("pending"), // pending, scheduled, in_progress, completed, cancelled
  sequence: integer("sequence"), // Kolejność w planie
  
  notes: text("notes"),
  metadata: jsonb("metadata"),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_plan_lines_plan").on(table.planId),
  index("idx_plan_lines_product").on(table.productId),
  index("idx_plan_lines_po").on(table.productionOrderId),
  index("idx_plan_lines_source").on(table.sourceType, table.sourceId),
  index("idx_plan_lines_status").on(table.status),
  index("idx_plan_lines_color").on(table.planId, table.colorCode), // Dla agregacji po kolorach w ZLP
]);

// Production Product Groups - grupy produktowe dla organizacji planowania
export const productionProductGroups = productionSchema.table("production_product_groups", {
  id: serial("id").primaryKey(),
  name: varchar("name", { length: 255 }).notNull(),
  colorHex: varchar("color_hex", { length: 7 }), // #RRGGBB format for UI
  description: text("description"),
  isActive: boolean("is_active").default(true),
  sortOrder: integer("sort_order").default(0),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_product_groups_active").on(table.isActive),
  index("idx_product_groups_sort").on(table.sortOrder),
]);

// Production Product Group Items - many-to-many: groups ↔ products
export const productionProductGroupItems = productionSchema.table("production_product_group_items", {
  id: serial("id").primaryKey(),
  groupId: integer("group_id").references(() => productionProductGroups.id, { onDelete: "cascade" }).notNull(),
  productId: integer("product_id").references(() => products.id, { onDelete: "restrict" }).notNull(), // FK to catalog.products (cross-schema)
  createdAt: timestamp("created_at").defaultNow(),
}, (table) => [
  uniqueIndex("idx_product_group_items_unique").on(table.groupId, table.productId),
  index("idx_product_group_items_group").on(table.groupId),
  index("idx_product_group_items_product").on(table.productId),
]);

// Production Component Batches - batche komponentów pogrupowanych według właściwości (kolor, wiercenie, oklejanie)
export const productionComponentBatches = productionSchema.table("production_component_batches", {
  id: serial("id").primaryKey(),
  batchNumber: varchar("batch_number", { length: 100 }).notNull().unique(),
  
  // Plan line reference (batch należy do linii planu)
  planLineId: integer("plan_line_id").references(() => productionPlanLines.id, { onDelete: "cascade" }).notNull(),
  
  // Optional production context (podczas wykonania)
  productionOrderId: integer("production_order_id").references(() => productionOrders.id, { onDelete: "set null" }),
  workOrderId: integer("work_order_id").references(() => productionWorkOrders.id, { onDelete: "set null" }),
  
  // Batch grouping attributes (po czym grupujemy)
  color: varchar("color", { length: 100 }),
  edgingPattern: varchar("edging_pattern", { length: 100 }),
  drillingPattern: varchar("drilling_pattern", { length: 100 }),
  boardType: varchar("board_type", { length: 50 }),
  componentType: varchar("component_type", { length: 100 }), // np. SUROWA, FRONT, BOK
  
  // Quantities
  quantity: integer("quantity").notNull(), // Ile formatek w tym batchu
  quantityCompleted: integer("quantity_completed").default(0),
  quantityRejected: integer("quantity_rejected").default(0),
  
  // Current station and status
  currentStationId: integer("current_station_id").references(() => productionWorkCenters.id, { onDelete: "set null" }),
  status: varchar("status", { length: 30 }).notNull().default("waiting"), // waiting, in_progress, completed, rejected
  
  // Routing
  routingId: integer("routing_id").references(() => productionRoutings.id, { onDelete: "set null" }),
  currentOperationSequence: integer("current_operation_sequence"), // Która operacja teraz
  
  // Material consumption summary (calculated)
  materialConsumption: jsonb("material_consumption"), // { board_m2: 10.5, edging_mb: 25.3, adhesive_g: 1265 }
  
  notes: text("notes"),
  metadata: jsonb("metadata"),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_batches_plan_line").on(table.planLineId),
  index("idx_batches_po").on(table.productionOrderId),
  index("idx_batches_attributes").on(table.color, table.edgingPattern, table.drillingPattern),
  index("idx_batches_station").on(table.currentStationId),
  index("idx_batches_status").on(table.status),
]);

// Production Material Consumption Norms - normatywy zużycia materiałów
export const productionMaterialConsumptionNorms = productionSchema.table("production_material_consumption_norms", {
  id: serial("id").primaryKey(),
  
  // Material type
  materialType: varchar("material_type", { length: 50 }).notNull(), // board, edging, adhesive, screw, etc.
  materialCategory: varchar("material_category", { length: 50 }), // PAL, MDF, HDF, ABS, PCV, etc.
  
  // Consumption rules
  wastePercentage: decimal("waste_percentage", { precision: 5, scale: 2 }), // np. 20% dla płyty
  edgeOverageCm: decimal("edge_overage_cm", { precision: 5, scale: 2 }), // np. 2cm dla obrzeża
  consumptionPerUnit: decimal("consumption_per_unit", { precision: 10, scale: 3 }), // np. 50g kleju na mb obrzeża
  
  // Units
  inputUnit: varchar("input_unit", { length: 20 }).notNull(), // m2, mb, szt
  outputUnit: varchar("output_unit", { length: 20 }).notNull(), // m2, mb, g, kg
  
  // Scope
  isActive: boolean("is_active").default(true),
  description: text("description"),
  
  notes: text("notes"),
  metadata: jsonb("metadata"),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_norms_material_type").on(table.materialType),
  index("idx_norms_category").on(table.materialCategory),
  index("idx_norms_active").on(table.isActive),
]);

// Component Material Breakdown - rozk\u0142ad materia\u0142owy formatki (p\u0142yta, obrze\u017ce, klej)
export const componentMaterialBreakdown = bomSchema.table("component_material_breakdown", {
  id: serial("id").primaryKey(),
  componentId: integer("component_id").references(() => productComponents.id, { onDelete: "cascade" }).notNull(),
  
  // P\u0142yta meblowa
  boardAreaM2: decimal("board_area_m2", { precision: 10, scale: 4 }), // Powierzchnia p\u0142yty w m2
  boardWastePercentage: decimal("board_waste_percentage", { precision: 5, scale: 2 }), // Odpad % (z normatyw\u00f3w)
  boardTotalM2: decimal("board_total_m2", { precision: 10, scale: 4 }), // Ca\u0142kowite zu\u017cycie z odpadem
  
  // Obrze\u017ce
  edgingLengthMb: decimal("edging_length_mb", { precision: 10, scale: 2 }), // D\u0142ugo\u015b\u0107 obrzeży w mb
  edgingOverageCm: decimal("edging_overage_cm", { precision: 5, scale: 2 }), // Naddatek cm (z normatyw\u00f3w)
  edgingTotalMb: decimal("edging_total_mb", { precision: 10, scale: 2 }), // Ca\u0142kowite zu\u017cycie z naddatkiem
  
  // Klej
  adhesiveGramsPerMb: decimal("adhesive_grams_per_mb", { precision: 10, scale: 3 }), // Zu\u017cycie kleju g/mb (z normatyw\u00f3w)
  adhesiveTotalG: decimal("adhesive_total_g", { precision: 10, scale: 2 }), // Ca\u0142kowite zu\u017cycie kleju
  
  // Koszty (pobierane z warehouse.materials)
  boardUnitPrice: decimal("board_unit_price", { precision: 10, scale: 2 }), // Cena p\u0142yty za m2
  edgingUnitPrice: decimal("edging_unit_price", { precision: 10, scale: 2 }), // Cena obrzeża za mb
  adhesiveUnitPrice: decimal("adhesive_unit_price", { precision: 10, scale: 2 }), // Cena kleju za gram
  
  boardCost: decimal("board_cost", { precision: 10, scale: 2 }), // Koszt p\u0142yty
  edgingCost: decimal("edging_cost", { precision: 10, scale: 2 }), // Koszt obrzeży
  adhesiveCost: decimal("adhesive_cost", { precision: 10, scale: 2 }), // Koszt kleju
  totalCost: decimal("total_cost", { precision: 10, scale: 2 }), // Ca\u0142kowity koszt formatki
  
  // Metadata
  calculatedAt: timestamp("calculated_at").defaultNow(),
  notes: text("notes"),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_component_material_breakdown_component").on(table.componentId),
  index("idx_component_material_breakdown_calculated").on(table.calculatedAt),
]);

// Production Routing Branches - definicje rozgałęzień w marszrutach (split routing)
export const productionRoutingBranches = productionSchema.table("production_routing_branches", {
  id: serial("id").primaryKey(),
  
  // Parent operation (od którego rozgałęziamy)
  routingOperationId: integer("routing_operation_id").references(() => productionRoutingOperations.id, { onDelete: "cascade" }).notNull(),
  
  // Branch definition
  branchName: varchar("branch_name", { length: 100 }).notNull(), // np. "Path A: Tapicernia", "Path B: Wiertarka"
  branchCode: varchar("branch_code", { length: 50 }).notNull(), // np. "UPHOLSTERY", "DRILLING"
  allocationPercentage: decimal("allocation_percentage", { precision: 5, scale: 2 }).notNull(), // np. 60.00 dla 60%
  
  // Next operation in this branch
  nextOperationId: integer("next_operation_id").references(() => productionRoutingOperations.id, { onDelete: "set null" }),
  
  // Conditions (opcjonalne - kiedy używać tej gałęzi)
  conditions: jsonb("conditions"), // { component_type: "SUROWA SIEDZISKO", requires_upholstery: true }
  
  sequence: integer("sequence"), // Kolejność wyświetlania
  isActive: boolean("is_active").default(true),
  
  notes: text("notes"),
  metadata: jsonb("metadata"),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_routing_branches_operation").on(table.routingOperationId),
  index("idx_routing_branches_next").on(table.nextOperationId),
  index("idx_routing_branches_active").on(table.isActive),
]);

// Production Material Movements - śledzenie ruchu materiałów między lokalizacjami
export const productionMaterialMovements = productionSchema.table("production_material_movements", {
  id: serial("id").primaryKey(),
  movementDate: timestamp("movement_date").notNull().defaultNow(),
  movementBatchId: varchar("movement_batch_id", { length: 100 }), // Group related scans together
  
  // Material identification
  materialType: varchar("material_type", { length: 30 }).notNull(), // board, component, finished_product, accessory
  materialCode: varchar("material_code", { length: 100 }).notNull(), // SKU or internal code
  materialName: varchar("material_name", { length: 255 }),
  
  // Quantity tracking
  quantity: decimal("quantity", { precision: 10, scale: 2 }).notNull(),
  quantityRejected: decimal("quantity_rejected", { precision: 10, scale: 2 }).default("0"), // For OEE metrics
  unit: varchar("unit", { length: 50 }).notNull().default("szt"),
  
  // Location flow
  sourceLocationId: integer("source_location_id").references(() => productionLocations.id, { onDelete: "set null" }),
  targetLocationId: integer("target_location_id").references(() => productionLocations.id, { onDelete: "set null" }),
  
  // Carrier tracking
  carrierId: integer("carrier_id").references(() => productionCarriers.id, { onDelete: "set null" }),
  
  // Production context
  productionOrderId: integer("production_order_id").references(() => productionOrders.id, { onDelete: "set null" }),
  workOrderId: integer("work_order_id").references(() => productionWorkOrders.id, { onDelete: "set null" }),
  workCenterId: integer("work_center_id").references(() => productionWorkCenters.id, { onDelete: "set null" }),
  routingOperationId: integer("routing_operation_id").references(() => productionRoutingOperations.id, { onDelete: "set null" }),
  
  // Operator and scanning
  operatorId: integer("operator_id"), // User who performed/scanned the movement
  scannedBarcode: varchar("scanned_barcode", { length: 255 }), // QR/Barcode that was scanned
  scanMethod: varchar("scan_method", { length: 20 }), // manual, qr, barcode, rfid
  
  // Status and metadata
  status: varchar("status", { length: 20 }).notNull().default("planned"), // planned, in_transit, completed, cancelled
  movementType: varchar("movement_type", { length: 30 }), // transfer, consumption, production, return, scrap
  
  notes: text("notes"),
  metadata: jsonb("metadata"), // Additional flexible data
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_material_movements_date").on(table.movementDate),
  index("idx_material_movements_po").on(table.productionOrderId),
  index("idx_material_movements_wo").on(table.workOrderId),
  index("idx_material_movements_carrier").on(table.carrierId),
  index("idx_material_movements_source_target").on(table.sourceLocationId, table.targetLocationId),
  index("idx_material_movements_type_status").on(table.materialType, table.status),
  index("idx_material_movements_operator").on(table.operatorId),
  index("idx_material_movements_batch").on(table.movementBatchId),
]);

// Cut Patterns - rozkroje płyt meblowych
export const cutPatterns = productionSchema.table("cut_patterns", {
  id: serial("id").primaryKey(),
  code: varchar("code", { length: 50 }).notNull().unique(),
  boardTypeCode: varchar("board_type_code", { length: 50 }),
  boardLength: decimal("board_length", { precision: 10, scale: 2 }).notNull(),
  boardWidth: decimal("board_width", { precision: 10, scale: 2 }).notNull(),
  boardThickness: decimal("board_thickness", { precision: 10, scale: 2 }).default("18"),
  kerf: decimal("kerf", { precision: 10, scale: 2 }).default("4"),
  imageUrl: text("image_url"),
  opis: text("opis"),
  status: varchar("status", { length: 20 }).notNull().default("planowany"),
  scheduledAt: timestamp("scheduled_at"),
  startedAt: timestamp("started_at"),
  completedAt: timestamp("completed_at"),
  createdBy: integer("created_by"),
  notes: text("notes"),
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_cut_patterns_status").on(table.status),
  index("idx_cut_patterns_code").on(table.code),
]);

// Cut Pattern Items - pozycje (formatki) w rozkrojach
export const cutPatternItems = productionSchema.table("cut_pattern_items", {
  id: serial("id").primaryKey(),
  patternId: integer("pattern_id").references(() => cutPatterns.id, { onDelete: "cascade" }).notNull(),
  sourcePlanLineId: integer("source_plan_line_id").references(() => productionPlanLines.id),
  componentTemplateId: integer("component_template_id"),
  itemName: varchar("item_name", { length: 255 }),
  length: decimal("length", { precision: 10, scale: 2 }).notNull(),
  width: decimal("width", { precision: 10, scale: 2 }).notNull(),
  thickness: decimal("thickness", { precision: 10, scale: 2 }).default("18"),
  boardMaterialId: integer("board_material_id").references(() => materials.id),
  edgingMaterialId: integer("edging_material_id").references(() => materials.id),
  boardCode: varchar("board_code", { length: 50 }),
  edgingCode: varchar("edging_code", { length: 50 }),
  colorCode: varchar("color_code", { length: 50 }),
  edge1: boolean("edge1").default(false),
  edge2: boolean("edge2").default(false),
  edge3: boolean("edge3").default(false),
  edge4: boolean("edge4").default(false),
  quantityRequested: integer("quantity_requested").notNull().default(1),
  quantityCompleted: integer("quantity_completed").default(0),
  positionMetadata: jsonb("position_metadata"),
  status: varchar("status", { length: 20 }).notNull().default("zaplanowana"),
  notes: text("notes"),
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_cut_pattern_items_pattern").on(table.patternId),
  index("idx_cut_pattern_items_status").on(table.status),
  index("idx_cut_pattern_items_board_material").on(table.boardMaterialId),
  index("idx_cut_pattern_items_edging_material").on(table.edgingMaterialId),
]);

// Cut Pattern Templates - szablony rozkrojów (wielokrotnego użytku)
export const cutPatternTemplates = productionSchema.table("cut_pattern_templates", {
  id: serial("id").primaryKey(),
  code: varchar("code", { length: 50 }).notNull().unique(),
  name: varchar("name", { length: 255 }).notNull(),
  description: text("description"),
  boardTypeCode: varchar("board_type_code", { length: 50 }),
  boardLength: decimal("board_length", { precision: 10, scale: 2 }).notNull(),
  boardWidth: decimal("board_width", { precision: 10, scale: 2 }).notNull(),
  boardThickness: decimal("board_thickness", { precision: 10, scale: 2 }).default("18"),
  kerf: decimal("kerf", { precision: 10, scale: 2 }).default("4"),
  imageUrl: text("image_url"),
  opis: text("opis"),
  usageCount: integer("usage_count").default(0), // Ile razy szablon był używany
  lastUsedAt: timestamp("last_used_at"),
  createdBy: integer("created_by"),
  notes: text("notes"),
  metadata: jsonb("metadata"), // Dodatkowe elastyczne dane
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_cut_pattern_templates_code").on(table.code),
  index("idx_cut_pattern_templates_active").on(table.isActive),
  index("idx_cut_pattern_templates_usage").on(table.usageCount),
]);

// Cut Pattern Template Items - formatki w szablonach
export const cutPatternTemplateItems = productionSchema.table("cut_pattern_template_items", {
  id: serial("id").primaryKey(),
  templateId: integer("template_id").references(() => cutPatternTemplates.id, { onDelete: "cascade" }).notNull(),
  itemName: varchar("item_name", { length: 255 }),
  length: decimal("length", { precision: 10, scale: 2 }).notNull(),
  width: decimal("width", { precision: 10, scale: 2 }).notNull(),
  thickness: decimal("thickness", { precision: 10, scale: 2 }).default("18"),
  boardMaterialId: integer("board_material_id").references(() => materials.id),
  edgingMaterialId: integer("edging_material_id").references(() => materials.id),
  boardCode: varchar("board_code", { length: 50 }),
  edgingCode: varchar("edging_code", { length: 50 }),
  colorCode: varchar("color_code", { length: 50 }),
  edge1: boolean("edge1").default(false),
  edge2: boolean("edge2").default(false),
  edge3: boolean("edge3").default(false),
  edge4: boolean("edge4").default(false),
  quantityDefault: integer("quantity_default").notNull().default(1), // Domyślna ilość w szablonie
  positionMetadata: jsonb("position_metadata"),
  notes: text("notes"),
  isActive: boolean("is_active").default(true),
  sortOrder: integer("sort_order").default(0), // Kolejność formatek w szablonie
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_cut_pattern_template_items_template").on(table.templateId),
  index("idx_cut_pattern_template_items_board_material").on(table.boardMaterialId),
  index("idx_cut_pattern_template_items_edging_material").on(table.edgingMaterialId),
]);

// Cut Pattern Runs - instancje produkcyjne rozkrojów (utworzone z szablonów)
export const cutPatternRuns = productionSchema.table("cut_pattern_runs", {
  id: serial("id").primaryKey(),
  runNumber: varchar("run_number", { length: 50 }).notNull().unique(), // Automatycznie generowany numer (np. RUN-2024-0001)
  templateId: integer("template_id").references(() => cutPatternTemplates.id, { onDelete: "set null" }),
  templateCode: varchar("template_code", { length: 50 }), // Snapshot kodu szablonu
  templateName: varchar("template_name", { length: 255 }), // Snapshot nazwy szablonu
  planLineId: integer("plan_line_id").references(() => productionPlanLines.id, { onDelete: "set null" }), // Powiązanie z planem produkcji
  status: varchar("status", { length: 20 }).notNull().default("planowany"), // planowany, w_produkcji, wykonany, anulowany
  scheduledAt: timestamp("scheduled_at"),
  startedAt: timestamp("started_at"),
  completedAt: timestamp("completed_at"),
  quantityMultiplier: decimal("quantity_multiplier", { precision: 10, scale: 2 }).default("1"), // Mnożnik ilości formatek
  createdBy: integer("created_by"),
  notes: text("notes"),
  metadata: jsonb("metadata"),
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_cut_pattern_runs_template").on(table.templateId),
  index("idx_cut_pattern_runs_status").on(table.status),
  index("idx_cut_pattern_runs_plan").on(table.planLineId),
  index("idx_cut_pattern_runs_scheduled").on(table.scheduledAt),
]);

// Cut Pattern Run Items - formatki w instancjach produkcyjnych (snapshot z szablonu + tracking)
export const cutPatternRunItems = productionSchema.table("cut_pattern_run_items", {
  id: serial("id").primaryKey(),
  runId: integer("run_id").references(() => cutPatternRuns.id, { onDelete: "cascade" }).notNull(),
  templateItemId: integer("template_item_id"), // Referencja do template item (może być null jeśli template usunięty)
  itemName: varchar("item_name", { length: 255 }),
  length: decimal("length", { precision: 10, scale: 2 }).notNull(),
  width: decimal("width", { precision: 10, scale: 2 }).notNull(),
  thickness: decimal("thickness", { precision: 10, scale: 2 }).default("18"),
  boardMaterialId: integer("board_material_id").references(() => materials.id),
  edgingMaterialId: integer("edging_material_id").references(() => materials.id),
  boardCode: varchar("board_code", { length: 50 }),
  edgingCode: varchar("edging_code", { length: 50 }),
  colorCode: varchar("color_code", { length: 50 }),
  edge1: boolean("edge1").default(false),
  edge2: boolean("edge2").default(false),
  edge3: boolean("edge3").default(false),
  edge4: boolean("edge4").default(false),
  quantityRequested: integer("quantity_requested").notNull().default(1),
  quantityCompleted: integer("quantity_completed").default(0),
  positionMetadata: jsonb("position_metadata"),
  status: varchar("status", { length: 20 }).notNull().default("zaplanowana"), // zaplanowana, w_produkcji, wykonana
  notes: text("notes"),
  isActive: boolean("is_active").default(true),
  sortOrder: integer("sort_order").default(0),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_cut_pattern_run_items_run").on(table.runId),
  index("idx_cut_pattern_run_items_template_item").on(table.templateItemId),
  index("idx_cut_pattern_run_items_status").on(table.status),
  index("idx_cut_pattern_run_items_board_material").on(table.boardMaterialId),
  index("idx_cut_pattern_run_items_edging_material").on(table.edgingMaterialId),
]);

// Production Tasks - kolejka zadań produkcji formatek
export const productionTasks = productionSchema.table("production_tasks", {
  id: serial("id").primaryKey(),
  formatkaReference: varchar("formatka_reference", { length: 255 }).notNull(),
  length: decimal("length", { precision: 10, scale: 2 }).notNull(),
  width: decimal("width", { precision: 10, scale: 2 }).notNull(),
  thickness: decimal("thickness", { precision: 10, scale: 2 }).default("18"),
  boardCode: varchar("board_code", { length: 50 }),
  colorCode: varchar("color_code", { length: 50 }),
  quantity: integer("quantity").notNull().default(1),
  priority: integer("priority").default(0),
  requiredBy: timestamp("required_by"),
  sourceType: varchar("source_type", { length: 50 }),
  sourceReference: varchar("source_reference", { length: 255 }),
  status: varchar("status", { length: 20 }).notNull().default("oczekuje"),
  assignedPatternId: integer("assigned_pattern_id").references(() => cutPatterns.id),
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_production_tasks_status").on(table.status),
]);

// Production Routing Variants - definicje ścieżek produkcyjnych
export const productionRoutingVariants = productionSchema.table("production_routing_variants", {
  id: serial("id").primaryKey(),
  variantCode: varchar("variant_code", { length: 50 }).notNull().unique(),
  variantName: varchar("variant_name", { length: 255 }).notNull(),
  description: text("description"),
  defaultOperations: jsonb("default_operations").notNull(), // ["cutting", "edging", "drilling"]
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_routing_variants_code").on(table.variantCode),
  index("idx_routing_variants_active").on(table.isActive),
]);

// Production Routing Variant Rules - reguły przypisywania routing variants
export const productionRoutingVariantRules = productionSchema.table("production_routing_variant_rules", {
  id: serial("id").primaryKey(),
  routingVariantId: integer("routing_variant_id").references(() => productionRoutingVariants.id, { onDelete: "cascade" }).notNull(),
  namePattern: varchar("name_pattern", { length: 100 }), // SQL LIKE pattern: "WD-%", "POLKA-%"
  colorPattern: varchar("color_pattern", { length: 50 }), // np. "SUROWA" dla tapicerni
  priority: integer("priority").notNull().default(0), // Wyższy = ważniejszy (dla konfliktów)
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
}, (table) => [
  index("idx_routing_rules_variant").on(table.routingVariantId),
  index("idx_routing_rules_priority").on(table.priority),
  index("idx_routing_rules_active").on(table.isActive),
]);

// Production Order BOMs - BOM snapshot dla Production Order
export const productionOrderBoms = productionSchema.table("production_order_boms", {
  id: serial("id").primaryKey(),
  productionOrderId: integer("production_order_id").references(() => productionOrders.id, { onDelete: "cascade" }).notNull().unique(),
  sourcePlanId: integer("source_plan_id").references(() => productionPlans.id, { onDelete: "set null" }),
  colorCode: varchar("color_code", { length: 50 }), // Kolor tego ZLP
  status: varchar("status", { length: 20 }).notNull().default("active"), // active, locked, modified
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_prod_order_boms_order").on(table.productionOrderId),
  index("idx_prod_order_boms_plan").on(table.sourcePlanId),
  index("idx_prod_order_boms_color").on(table.colorCode),
]);

// Production Order BOM Items - komponenty w BOM dla Production Order
export const productionOrderBomItems = productionSchema.table("production_order_bom_items", {
  id: serial("id").primaryKey(),
  productionOrderBomId: integer("production_order_bom_id").references(() => productionOrderBoms.id, { onDelete: "cascade" }).notNull(),
  
  // Identyfikacja komponentu
  componentName: varchar("component_name", { length: 255 }).notNull(),
  componentType: varchar("component_type", { length: 50 }), // "formatka", "accessory", etc.
  quantity: decimal("quantity", { precision: 10, scale: 2 }).notNull(),
  unitOfMeasure: varchar("unit_of_measure", { length: 20 }).notNull().default("szt"),
  
  // Routing
  routingVariantId: integer("routing_variant_id").references(() => productionRoutingVariants.id, { onDelete: "set null" }),
  requiredOperations: jsonb("required_operations"), // ["cutting", "edging", "drilling"]
  
  // Kolory i wymiary
  colorCode: varchar("color_code", { length: 50 }),
  length: decimal("length", { precision: 10, scale: 2 }),
  width: decimal("width", { precision: 10, scale: 2 }),
  thickness: decimal("thickness", { precision: 10, scale: 2 }),
  
  // Traceability (skąd pochodzi ten komponent)
  sourcePlanLineId: integer("source_plan_line_id").references(() => productionPlanLines.id, { onDelete: "set null" }),
  sourceProductId: integer("source_product_id"),
  sourceFurnitureReference: varchar("source_furniture_reference", { length: 255 }),
  
  // Tapicernia
  externalProcessingStatus: varchar("external_processing_status", { length: 30 }), // null, 'awaiting_external', 'returned'
  externalProcessingNotes: text("external_processing_notes"),
  
  // Uszkodzenia
  isDamaged: boolean("is_damaged").default(false),
  damageType: varchar("damage_type", { length: 50 }), // 'SIZE_ERROR', 'COLOR_ERROR', 'DRILLING_ERROR', etc.
  damageNotes: text("damage_notes"),
  damagedAt: timestamp("damaged_at"),
  damagedByUserId: integer("damaged_by_user_id"),
  
  // Statusy elementu
  itemStatus: varchar("item_status", { length: 30 }).notNull().default("pending"), // 'pending', 'in_production', 'completed', 'damaged', 'scrapped', 'rework', 'awaiting_client'
  
  // Śledzenie ilości
  quantityOrdered: decimal("quantity_ordered", { precision: 10, scale: 2 }),
  quantityProduced: decimal("quantity_produced", { precision: 10, scale: 2 }).default("0"),
  quantityDamaged: decimal("quantity_damaged", { precision: 10, scale: 2 }).default("0"),
  quantityScrapped: decimal("quantity_scrapped", { precision: 10, scale: 2 }).default("0"),
  quantityRework: decimal("quantity_rework", { precision: 10, scale: 2 }).default("0"),
  
  // Metadata
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_prod_order_bom_items_bom").on(table.productionOrderBomId),
  index("idx_prod_order_bom_items_routing").on(table.routingVariantId),
  index("idx_prod_order_bom_items_status").on(table.itemStatus),
  index("idx_prod_order_bom_items_damaged").on(table.isDamaged),
  index("idx_prod_order_bom_items_color").on(table.colorCode),
  index("idx_prod_order_bom_items_plan_line").on(table.sourcePlanLineId),
]);

// Upholstery Component Materials - składowe do tapicerni (tkanina, pianka)
export const upholsteryComponentMaterials = productionSchema.table("upholstery_component_materials", {
  id: serial("id").primaryKey(),
  productionOrderBomItemId: integer("production_order_bom_item_id").references(() => productionOrderBomItems.id, { onDelete: "cascade" }).notNull(),
  materialType: varchar("material_type", { length: 50 }).notNull(), // "fabric", "foam", "other"
  warehouseMaterialId: integer("warehouse_material_id"), // Link do warehouse.materials (opcjonalny)
  materialName: varchar("material_name", { length: 255 }).notNull(),
  materialCode: varchar("material_code", { length: 100 }),
  quantity: decimal("quantity", { precision: 10, scale: 2 }).notNull(),
  unitOfMeasure: varchar("unit_of_measure", { length: 20 }).notNull(), // "mb", "szt", "kg"
  notes: text("notes"),
  addedAt: timestamp("added_at").defaultNow(),
  addedByUserId: integer("added_by_user_id"),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_upholstery_materials_bom_item").on(table.productionOrderBomItemId),
  index("idx_upholstery_materials_type").on(table.materialType),
  index("idx_upholstery_materials_warehouse").on(table.warehouseMaterialId),
]);

// Zod schemas and types for Production module

export const insertProductionSettingSchema = createInsertSchema(productionSettings).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductionNormGroupSchema = createInsertSchema(productionNormGroups).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductionNormSchema = createInsertSchema(productionNorms).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductionCarrierGroupSchema = createInsertSchema(productionCarrierGroups).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductionCarrierSchema = createInsertSchema(productionCarriers).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductionLocationGroupSchema = createInsertSchema(productionLocationGroups).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductionLocationSchema = createInsertSchema(productionLocations).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductionWorkCenterSchema = createInsertSchema(productionWorkCenters).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductionRoutingSchema = createInsertSchema(productionRoutings).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductionRoutingOperationSchema = createInsertSchema(productionRoutingOperations).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  sequence: z.number().optional(), // Backend auto-generates if not provided
});

export const insertProductionBufferStockSchema = createInsertSchema(productionBufferStock).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductionBufferMovementSchema = createInsertSchema(productionBufferMovements).omit({
  id: true,
  createdAt: true,
});

export const insertProductionBufferReservationSchema = createInsertSchema(productionBufferReservations).omit({
  id: true,
  reservedAt: true,
  consumedAt: true,
  cancelledAt: true,
});

export const insertProductionOrderSchema = createInsertSchema(productionOrders).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductionWorkOrderSchema = createInsertSchema(productionWorkOrders).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductionOrderMaterialSchema = createInsertSchema(productionOrderMaterials).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductionOrderLogSchema = createInsertSchema(productionOrderLogs).omit({
  id: true,
  createdAt: true,
});

export const insertProductionMaterialMovementSchema = createInsertSchema(productionMaterialMovements).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductionPlanNameTemplateSchema = createInsertSchema(productionPlanNameTemplates).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductionPlanNameCounterSchema = createInsertSchema(productionPlanNameCounters).omit({
  id: true,
  updatedAt: true,
});

export const insertProductionPlanSchema = createInsertSchema(productionPlans).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductionPlanLineSchema = createInsertSchema(productionPlanLines).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductionProductGroupSchema = createInsertSchema(productionProductGroups).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductionProductGroupItemSchema = createInsertSchema(productionProductGroupItems).omit({
  id: true,
  createdAt: true,
});

export const insertProductionComponentBatchSchema = createInsertSchema(productionComponentBatches).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductionMaterialConsumptionNormSchema = createInsertSchema(productionMaterialConsumptionNorms).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductionRoutingBranchSchema = createInsertSchema(productionRoutingBranches).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertComponentMaterialBreakdownSchema = createInsertSchema(componentMaterialBreakdown).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

// Types
export type ProductionSetting = typeof productionSettings.$inferSelect;
export type InsertProductionSetting = z.infer<typeof insertProductionSettingSchema>;

export type ProductionNormGroup = typeof productionNormGroups.$inferSelect;
export type InsertProductionNormGroup = z.infer<typeof insertProductionNormGroupSchema>;

export type ProductionNorm = typeof productionNorms.$inferSelect;
export type InsertProductionNorm = z.infer<typeof insertProductionNormSchema>;

export type ProductionCarrierGroup = typeof productionCarrierGroups.$inferSelect;
export type InsertProductionCarrierGroup = z.infer<typeof insertProductionCarrierGroupSchema>;

export type ProductionCarrier = typeof productionCarriers.$inferSelect;
export type InsertProductionCarrier = z.infer<typeof insertProductionCarrierSchema>;

export type ProductionLocationGroup = typeof productionLocationGroups.$inferSelect;
export type InsertProductionLocationGroup = z.infer<typeof insertProductionLocationGroupSchema>;

export type ProductionLocation = typeof productionLocations.$inferSelect;
export type InsertProductionLocation = z.infer<typeof insertProductionLocationSchema>;

export type ProductionWorkCenter = typeof productionWorkCenters.$inferSelect;
export type InsertProductionWorkCenter = z.infer<typeof insertProductionWorkCenterSchema>;

export type ProductionRouting = typeof productionRoutings.$inferSelect;
export type InsertProductionRouting = z.infer<typeof insertProductionRoutingSchema>;

export type ProductionRoutingOperation = typeof productionRoutingOperations.$inferSelect;
export type InsertProductionRoutingOperation = z.infer<typeof insertProductionRoutingOperationSchema>;

export type ProductionBufferStock = typeof productionBufferStock.$inferSelect;
export type InsertProductionBufferStock = z.infer<typeof insertProductionBufferStockSchema>;

export type ProductionBufferMovement = typeof productionBufferMovements.$inferSelect;
export type InsertProductionBufferMovement = z.infer<typeof insertProductionBufferMovementSchema>;

export type ProductionBufferReservation = typeof productionBufferReservations.$inferSelect;
export type InsertProductionBufferReservation = z.infer<typeof insertProductionBufferReservationSchema>;

export type ProductionOrder = typeof productionOrders.$inferSelect;
export type InsertProductionOrder = z.infer<typeof insertProductionOrderSchema>;

export type ProductionWorkOrder = typeof productionWorkOrders.$inferSelect;
export type InsertProductionWorkOrder = z.infer<typeof insertProductionWorkOrderSchema>;

export type ProductionOrderMaterial = typeof productionOrderMaterials.$inferSelect;
export type InsertProductionOrderMaterial = z.infer<typeof insertProductionOrderMaterialSchema>;

export type ProductionOrderLog = typeof productionOrderLogs.$inferSelect;
export type InsertProductionOrderLog = z.infer<typeof insertProductionOrderLogSchema>;

export type ProductionMaterialMovement = typeof productionMaterialMovements.$inferSelect;
export type InsertProductionMaterialMovement = z.infer<typeof insertProductionMaterialMovementSchema>;

export type ProductionPlanNameTemplate = typeof productionPlanNameTemplates.$inferSelect;
export type InsertProductionPlanNameTemplate = z.infer<typeof insertProductionPlanNameTemplateSchema>;

export type ProductionPlanNameCounter = typeof productionPlanNameCounters.$inferSelect;
export type InsertProductionPlanNameCounter = z.infer<typeof insertProductionPlanNameCounterSchema>;

export type ProductionPlan = typeof productionPlans.$inferSelect;
export type InsertProductionPlan = z.infer<typeof insertProductionPlanSchema>;

export type ProductionPlanLine = typeof productionPlanLines.$inferSelect;
export type InsertProductionPlanLine = z.infer<typeof insertProductionPlanLineSchema>;

export type ProductionProductGroup = typeof productionProductGroups.$inferSelect;
export type InsertProductionProductGroup = z.infer<typeof insertProductionProductGroupSchema>;

export type ProductionProductGroupItem = typeof productionProductGroupItems.$inferSelect;
export type InsertProductionProductGroupItem = z.infer<typeof insertProductionProductGroupItemSchema>;

export type ProductionComponentBatch = typeof productionComponentBatches.$inferSelect;
export type InsertProductionComponentBatch = z.infer<typeof insertProductionComponentBatchSchema>;

export type ProductionMaterialConsumptionNorm = typeof productionMaterialConsumptionNorms.$inferSelect;
export type InsertProductionMaterialConsumptionNorm = z.infer<typeof insertProductionMaterialConsumptionNormSchema>;

export type ProductionRoutingBranch = typeof productionRoutingBranches.$inferSelect;
export type InsertProductionRoutingBranch = z.infer<typeof insertProductionRoutingBranchSchema>;

export type ComponentMaterialBreakdown = typeof componentMaterialBreakdown.$inferSelect;
export type InsertComponentMaterialBreakdown = z.infer<typeof insertComponentMaterialBreakdownSchema>;

// Cut Pattern schemas (legacy - zostają dla kompatybilności wstecznej)
export const insertCutPatternSchema = createInsertSchema(cutPatterns).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertCutPatternItemSchema = createInsertSchema(cutPatternItems).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type CutPattern = typeof cutPatterns.$inferSelect;
export type InsertCutPattern = z.infer<typeof insertCutPatternSchema>;

export type CutPatternItem = typeof cutPatternItems.$inferSelect;
export type InsertCutPatternItem = z.infer<typeof insertCutPatternItemSchema>;

// Cut Pattern Template schemas (nowy system szablonów)
export const insertCutPatternTemplateSchema = createInsertSchema(cutPatternTemplates).omit({
  id: true,
  usageCount: true,
  lastUsedAt: true,
  createdAt: true,
  updatedAt: true,
});

export const insertCutPatternTemplateItemSchema = createInsertSchema(cutPatternTemplateItems).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

// Form schema dla dodawania/edycji szablonów rozkrojów (z walidacją)
export const cutPatternTemplateFormSchema = insertCutPatternTemplateSchema
  .omit({ code: true, createdBy: true, isActive: true })
  .extend({
    name: z.string().min(1, "Nazwa jest wymagana").max(255, "Nazwa może mieć maksymalnie 255 znaków"),
    boardLength: z.coerce
      .number()
      .positive("Długość musi być większa niż 0")
      .max(5000, "Długość nie może przekraczać 5000 mm"),
    boardWidth: z.coerce
      .number()
      .positive("Szerokość musi być większa niż 0")
      .max(5000, "Szerokość nie może przekraczać 5000 mm"),
    boardThickness: z.coerce
      .number()
      .positive("Grubość musi być większa niż 0")
      .max(100, "Grubość nie może przekraczać 100 mm")
      .optional()
      .default(18),
    kerf: z.coerce
      .number()
      .nonnegative("Kerf nie może być ujemny")
      .max(20, "Kerf nie może przekraczać 20 mm")
      .optional()
      .default(4),
    description: z.string().optional(),
    boardTypeCode: z.string().optional(),
    imageUrl: z.string().optional(),
    opis: z.string().optional(),
    notes: z.string().optional(),
    metadata: z.any().optional(),
  });

export type CutPatternTemplate = typeof cutPatternTemplates.$inferSelect;
export type InsertCutPatternTemplate = z.infer<typeof insertCutPatternTemplateSchema>;
export type CutPatternTemplateFormData = z.infer<typeof cutPatternTemplateFormSchema>;

export type CutPatternTemplateItem = typeof cutPatternTemplateItems.$inferSelect;
export type InsertCutPatternTemplateItem = z.infer<typeof insertCutPatternTemplateItemSchema>;

// Cut Pattern Run schemas (instancje produkcyjne)
export const insertCutPatternRunSchema = createInsertSchema(cutPatternRuns).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertCutPatternRunItemSchema = createInsertSchema(cutPatternRunItems).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type CutPatternRun = typeof cutPatternRuns.$inferSelect;
export type InsertCutPatternRun = z.infer<typeof insertCutPatternRunSchema>;

export type CutPatternRunItem = typeof cutPatternRunItems.$inferSelect;
export type InsertCutPatternRunItem = z.infer<typeof insertCutPatternRunItemSchema>;

// Production Task schemas
export const insertProductionTaskSchema = createInsertSchema(productionTasks).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type ProductionTask = typeof productionTasks.$inferSelect;
export type InsertProductionTask = z.infer<typeof insertProductionTaskSchema>;

// Production Routing Variant schemas
export const insertProductionRoutingVariantSchema = createInsertSchema(productionRoutingVariants).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type ProductionRoutingVariant = typeof productionRoutingVariants.$inferSelect;
export type InsertProductionRoutingVariant = z.infer<typeof insertProductionRoutingVariantSchema>;

// Production Routing Variant Rule schemas
export const insertProductionRoutingVariantRuleSchema = createInsertSchema(productionRoutingVariantRules).omit({
  id: true,
  createdAt: true,
});

export type ProductionRoutingVariantRule = typeof productionRoutingVariantRules.$inferSelect;
export type InsertProductionRoutingVariantRule = z.infer<typeof insertProductionRoutingVariantRuleSchema>;

// Production Order BOM schemas
export const insertProductionOrderBomSchema = createInsertSchema(productionOrderBoms).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type ProductionOrderBom = typeof productionOrderBoms.$inferSelect;
export type InsertProductionOrderBom = z.infer<typeof insertProductionOrderBomSchema>;

// Production Order BOM Item schemas
export const insertProductionOrderBomItemSchema = createInsertSchema(productionOrderBomItems).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type ProductionOrderBomItem = typeof productionOrderBomItems.$inferSelect;
export type InsertProductionOrderBomItem = z.infer<typeof insertProductionOrderBomItemSchema>;

// Upholstery Component Material schemas
export const insertUpholsteryComponentMaterialSchema = createInsertSchema(upholsteryComponentMaterials).omit({
  id: true,
  addedAt: true,
  updatedAt: true,
});

export type UpholsteryComponentMaterial = typeof upholsteryComponentMaterials.$inferSelect;
export type InsertUpholsteryComponentMaterial = z.infer<typeof insertUpholsteryComponentMaterialSchema>;

// ============================================================================
// Formatki Helper Functions
// ============================================================================

/**
 * List of all component name prefixes that identify formatki (cut parts).
 * These components require cutting operations and go through the full manufacturing process.
 * 
 * Based on production-flow-strategy.md routing variant mappings.
 */
export const FORMATKI_PREFIXES = [
  'WD-',       // Wnęka dolna (bottom cavity)
  'WG-',       // Wnęka górna (top cavity)
  'BOK-',      // Bok (side panel)
  'DRZWI-',    // Drzwi (door)
  'PÓŁKA-',    // Półka (shelf)
  'POLKA-',    // Półka alternative spelling
  'TYŁ-',      // Tył (back panel)
  'TYL-',      // Tył alternative spelling
  'FRONT-',    // Front (drawer front)
  'BLAT-',     // Blat (top/countertop)
  'DNA-',      // Dno (bottom)
  'DNO-',      // Dno alternative spelling
  'DENKO-',    // Denko (small bottom)
  'PANEL-',    // Panel (generic panel)
  'HDF-',      // HDF panel
  'LISTWA-',   // Listwa (strip/trim)
  'BOCZKI-',   // Boczki (side pieces - drawer)
  'SIEDZISKO-' // Siedzisko (seat - for upholstery)
] as const;

/**
 * Determines if a component is a formatka (cut part) based on its generated name.
 * 
 * @param generatedName - The component's generated name (e.g., "WD-VB-300x200-80-BIAŁY")
 * @returns true if the component is a formatka, false otherwise
 * 
 * @example
 * isFormatka("WD-VB-300x200-80-BIAŁY") // true
 * isFormatka("ZAWIASY-3MM-SREBRNE") // false
 */
export function isFormatka(generatedName: string | null | undefined): boolean {
  if (!generatedName) return false;
  const upperName = generatedName.toUpperCase();
  return FORMATKI_PREFIXES.some(prefix => upperName.startsWith(prefix));
}
