/**
 * 🧪 PEŁNA SYMULACJA SYSTEMU BUFORA - przez API services
 * 
 * Symulacja end-to-end:
 * 1. Utworzenie ZLP z prawdziwymi zamówieniami klientów
 * 2. Dodanie produktów do ZLP
 * 3. Symulacja produkcji półproduktów (IN do bufora)
 * 4. Rezerwacja półproduktów dla drugiego ZLP
 * 5. Konsumpcja podczas montażu
 * 6. Weryfikacja stanu
 */

import { pool } from "./postgres";
import * as planningService from "./services/production/planning";
import { createBufferMovement } from "./services/production/buffer-movements";
import { 
  createReservation, 
  consumeReservation,
  getReservationsByZlp 
} from "./services/production/buffer-reservations";
import { getBufferStock } from "./services/production/buffer-stock";

async function runSimulation() {
  console.log("\n╔══════════════════════════════════════════════════════════╗");
  console.log("║  🧪 SYMULACJA SYSTEMU BUFORA PRODUKCYJNEGO              ║");
  console.log("║  Zamówienia klientów: 01839, 01830, 01812, 01771, 02025, 02111 ║");
  console.log("╚══════════════════════════════════════════════════════════╝\n");

  try {
    // ═══════════════════════════════════════════════════════════
    // KROK 1: Pobierz zamówienia klientów z bazy
    // ═══════════════════════════════════════════════════════════
    console.log("📋 KROK 1: Pobieranie zamówień klientów...");
    const ordersResult = await pool.query(`
      SELECT 
        o.id as order_id,
        o.order_number,
        o.buyer_first_name || ' ' || o.buyer_last_name as customer_name,
        oi.id as item_id,
        oi.name as product_name,
        oi.quantity,
        oi.product_id as marketplace_product_id
      FROM commerce.orders o
      JOIN commerce.order_items oi ON o.id = oi.order_id
      WHERE o.order_number IN ('01839', '01830', '01812', '01771', '02025', '02111')
      ORDER BY o.order_number
    `);
    
    console.log(`✅ Znaleziono ${ordersResult.rows.length} pozycji zamówień`);
    ordersResult.rows.forEach(row => {
      console.log(`   - Zamówienie ${row.order_number}: ${row.product_name} (${row.quantity} szt)`);
    });

    // ═══════════════════════════════════════════════════════════
    // KROK 2: Dla symulacji - używamy demo produktu katalogowego
    // ═══════════════════════════════════════════════════════════
    console.log("\n🔗 KROK 2: Mapowanie produktów...");
    console.log("   ℹ️  Dla symulacji używamy produktu demo: Szafka VB 50×30×45 WOTAN (ID: 334)");
    console.log("   ℹ️  W produkcji mapowanie odbywa się przez catalog.product_platform_data");

    // ═══════════════════════════════════════════════════════════
    // KROK 3: Utwórz ZLP #1 - produkcja półproduktów
    // ═══════════════════════════════════════════════════════════
    console.log("\n🏭 KROK 3: Tworzenie ZLP #1 - Produkcja półproduktów...");
    const zlp1 = await planningService.createPlan(pool, {
      name: "Produkcja półproduktów - siedziska tapicerowane",
      description: "Zamówienia klientów: 01839, 01830, 01812, 01771, 02025, 02111",
      status: "confirmed",
      priority: "normal",
      createdBy: 1,
    });
    
    console.log(`✅ Utworzono ZLP #${zlp1.id}: ${zlp1.planNumber} - ${zlp1.name}`);

    // ═══════════════════════════════════════════════════════════
    // KROK 4: Dodaj produkty do ZLP #1
    // ═══════════════════════════════════════════════════════════
    console.log("\n📦 KROK 4: Dodawanie produktów do ZLP #1...");
    
    // Dla symulacji dodajemy demo produkt z routingiem
    const demoProduct = {
      planId: zlp1.id,
      productId: 334, // Szafka VB 50×30×45 WOTAN
      quantity: 12, // Suma dla 6 zamówień (po 2 szt każde)
      sourceType: 'manual' as const,
      sourceReference: 'Zamówienia: 01839, 01830, 01812, 01771, 02025, 02111',
      routingId: 2392, // SIEDZISKA TAPICEROWANE
      status: 'pending' as const,
      sequence: 1,
      notes: 'Produkcja siedzisk tapicerowanych - routing z buforem',
    };

    const line1 = await planningService.createPlanLine(pool, demoProduct);
    console.log(`✅ Dodano pozycję planu #${line1.id}: ${demoProduct.quantity} szt (routing: SIEDZISKA TAPICEROWANE)`);

    // ═══════════════════════════════════════════════════════════
    // KROK 5: SYMULACJA PRODUKCJI - Operacja "Cięcie płyt" tworzy bufor
    // ═══════════════════════════════════════════════════════════
    console.log("\n🔧 KROK 5: SYMULACJA PRODUKCJI - Operacja tworzy półprodukty...");
    
    const movement1 = await createBufferMovement(pool, {
      movementType: 'IN',
      productSku: 'SIEDZISKO-TAP-WOTAN-50x30',
      quantity: "12.0",
      unitOfMeasure: 'szt',
      locationId: 15, // Bufor Tapicernia
      sourceType: 'ZLP',
      zlpId: zlp1.id,
      createdBy: 1,
      notes: `Wyprodukowano 12 siedzisk tapicerowanych (półprodukt) dla ZLP ${zlp1.planNumber} - po operacji: Cięcie płyt`,
    });

    console.log(`✅ Ruch bufora #${movement1.id}: +${movement1.quantity} ${movement1.unitOfMeasure} → Bufor Tapicernia`);
    console.log(`   SKU: ${movement1.productSku}`);

    // ═══════════════════════════════════════════════════════════
    // KROK 6: Sprawdź stan bufora
    // ═══════════════════════════════════════════════════════════
    console.log("\n📊 KROK 6: Sprawdzanie stanu bufora...");
    const stock = await getBufferStock(pool);
    const ourStock = stock.find(s => s.productSku === 'SIEDZISKO-TAP-WOTAN-50x30');
    
    if (ourStock) {
      console.log(`✅ Stan bufora:`);
      console.log(`   SKU: ${ourStock.productSku}`);
      console.log(`   Dostępne: ${ourStock.quantityAvailable} szt`);
      console.log(`   Zarezerwowane: ${ourStock.quantityReserved} szt`);
      console.log(`   RAZEM: ${ourStock.quantityTotal} szt`);
    }

    // ═══════════════════════════════════════════════════════════
    // KROK 7: Utwórz ZLP #2 - montaż wykorzystujący półprodukty
    // ═══════════════════════════════════════════════════════════
    console.log("\n🏭 KROK 7: Tworzenie ZLP #2 - Montaż z półproduktów...");
    const zlp2 = await planningService.createPlan(pool, {
      name: "Montaż szafek VB - wykorzystanie bufora",
      description: "Montaż finalny z wykorzystaniem siedzisk z bufora tapicernia",
      status: "confirmed",
      priority: "high",
      createdBy: 1,
    });
    
    console.log(`✅ Utworzono ZLP #${zlp2.id}: ${zlp2.planNumber} - ${zlp2.name}`);

    // Dodaj produkty do ZLP #2
    const line2 = await planningService.createPlanLine(pool, {
      planId: zlp2.id,
      productId: 334,
      quantity: 7, // Potrzebujemy 7 siedzisk do montażu
      sourceType: 'manual' as const,
      sourceReference: 'Zamówienie klienta #DEMO-001',
      routingId: 2392,
      status: 'pending' as const,
      sequence: 1,
      notes: 'Montaż szafek - siedziska z bufora',
    });
    
    console.log(`✅ Dodano pozycję planu #${line2.id}: ${line2.quantity} szt do montażu`);

    // ═══════════════════════════════════════════════════════════
    // KROK 8: REZERWACJA półproduktów z bufora dla ZLP #2
    // ═══════════════════════════════════════════════════════════
    console.log("\n🔒 KROK 8: REZERWACJA półproduktów dla ZLP #2...");
    
    const reservation = await createReservation(pool, {
      zlpId: zlp2.id,
      productSku: 'SIEDZISKO-TAP-WOTAN-50x30',
      quantityReserved: "7.0",
      quantityConsumed: "0.0",
      unitOfMeasure: 'szt',
      locationId: 15,
      status: 'ACTIVE',
      reservedBy: 1,
      notes: `Rezerwacja dla montażu ${zlp2.planNumber}`,
    });

    console.log(`✅ Rezerwacja #${reservation.id}:`);
    console.log(`   ZLP: ${zlp2.planNumber}`);
    console.log(`   Ilość: ${reservation.quantityReserved} szt`);
    console.log(`   Status: ${reservation.status}`);

    // ═══════════════════════════════════════════════════════════
    // KROK 9: Sprawdź stan bufora PO REZERWACJI
    // ═══════════════════════════════════════════════════════════
    console.log("\n📊 KROK 9: Stan bufora PO REZERWACJI...");
    const stock2 = await getBufferStock(pool);
    const ourStock2 = stock2.find(s => s.productSku === 'SIEDZISKO-TAP-WOTAN-50x30');
    
    if (ourStock2) {
      console.log(`✅ Stan bufora:`);
      console.log(`   🟢 Dostępne: ${ourStock2.quantityAvailable} szt (12 - 7 = 5)`);
      console.log(`   🔒 Zarezerwowane: ${ourStock2.quantityReserved} szt`);
      console.log(`   📦 RAZEM: ${ourStock2.quantityTotal} szt`);
    }

    // ═══════════════════════════════════════════════════════════
    // KROK 10: KONSUMPCJA podczas montażu
    // ═══════════════════════════════════════════════════════════
    console.log("\n🔨 KROK 10: KONSUMPCJA półproduktów podczas montażu...");
    
    await consumeReservation(pool, reservation.id, 7.0, 1);

    console.log(`✅ Skonsumowano 7 szt siedzisk z rezerwacji #${reservation.id}`);

    // ═══════════════════════════════════════════════════════════
    // KROK 11: Sprawdź stan bufora PO KONSUMPCJI
    // ═══════════════════════════════════════════════════════════
    console.log("\n📊 KROK 11: Stan bufora PO KONSUMPCJI...");
    const stock3 = await getBufferStock(pool);
    const ourStock3 = stock3.find(s => s.productSku === 'SIEDZISKO-TAP-WOTAN-50x30');
    
    if (ourStock3) {
      console.log(`✅ Stan bufora KOŃCOWY:`);
      console.log(`   🟢 Dostępne: ${ourStock3.quantityAvailable} szt`);
      console.log(`   🔒 Zarezerwowane: ${ourStock3.quantityReserved} szt`);
      console.log(`   📦 RAZEM: ${ourStock3.quantityTotal} szt (12 - 7 = 5)`);
    }

    // ═══════════════════════════════════════════════════════════
    // KROK 12: Podsumowanie przepływu
    // ═══════════════════════════════════════════════════════════
    console.log("\n╔══════════════════════════════════════════════════════════╗");
    console.log("║  ✅ SYMULACJA ZAKOŃCZONA POMYŚLNIE                      ║");
    console.log("╚══════════════════════════════════════════════════════════╝\n");

    console.log("📋 PODSUMOWANIE:");
    console.log(`1. ZLP #${zlp1.id} (${zlp1.planNumber}): Wyprodukowano 12 siedzisk → Bufor`);
    console.log(`2. ZLP #${zlp2.id} (${zlp2.planNumber}): Zarezerwowano 7 siedzisk`);
    console.log(`3. Montaż: Skonsumowano 7 siedzisk z rezerwacji`);
    console.log(`4. Stan końcowy bufora: 5 siedzisk dostępnych\n`);

    console.log("🎯 Przepływ kompletny:");
    console.log("   Zamówienia klientów → ZLP → Produkcja → Bufor → Rezerwacja → Konsumpcja\n");

  } catch (error) {
    console.error("\n❌ BŁĄD PODCZAS SYMULACJI:", error);
    throw error;
  }
}

// Uruchom symulację
runSimulation()
  .then(() => {
    console.log("🏁 Symulacja zakończona");
    process.exit(0);
  })
  .catch((error) => {
    console.error("💥 Symulacja nie powiodła się:", error);
    process.exit(1);
  });
