import { pool } from '../../postgres';

export interface RoutingVariant {
  id: number;
  variantCode: string;
  variantName: string;
  defaultOperations: string[];
}

export interface RoutingResolutionResult {
  variant: RoutingVariant | null;
  matchedRule: {
    ruleId: number;
    namePattern: string | null;
    colorPattern: string | null;
    priority: number;
  } | null;
}

/**
 * Rozwiązuje routing variant dla komponentu na podstawie nazwy i koloru.
 * 
 * Algorytm:
 * 1. Pobiera wszystkie aktywne reguły posortowane według priorytetu (DESC)
 * 2. Dla każdej reguły sprawdza czy:
 *    - name_pattern pasuje do componentName (SQL LIKE)
 *    - color_pattern == null LUB color_pattern == color
 * 3. Zwraca pierwszy matching routing variant
 * 
 * @param componentName - Nazwa komponentu (np. "WD-100x50", "SIEDZISKO 45x45", "POLKA 80x30")
 * @param color - Kod koloru (np. "WOTAN", "SUROWA", "BIALY") lub null
 * @returns RoutingResolutionResult z matched variant i reguła lub null jeśli nie znaleziono
 */
export async function resolveRoutingVariant(
  componentName: string,
  color: string | null = null
): Promise<RoutingResolutionResult> {
  const client = await pool.connect();

  try {
    // Pobierz wszystkie aktywne reguły z routing variants, posortowane według priorytetu
    const result = await client.query<{
      rule_id: number;
      variant_id: number;
      variant_code: string;
      variant_name: string;
      default_operations: any;
      name_pattern: string | null;
      color_pattern: string | null;
      priority: number;
    }>(
      `
      SELECT 
        r.id as rule_id,
        v.id as variant_id,
        v.variant_code,
        v.variant_name,
        v.default_operations,
        r.name_pattern,
        r.color_pattern,
        r.priority
      FROM production.production_routing_variant_rules r
      JOIN production.production_routing_variants v ON v.id = r.routing_variant_id
      WHERE r.is_active = true 
        AND v.is_active = true
      ORDER BY r.priority DESC, r.id ASC
      `
    );

    // Iteruj przez reguły i znajdź pierwszy match
    for (const rule of result.rows) {
      // Sprawdź name_pattern (SQL LIKE)
      const nameMatches = rule.name_pattern 
        ? await checkNamePattern(client, componentName, rule.name_pattern)
        : true; // null pattern = match wszystko

      // Sprawdź color_pattern (exact match lub null)
      const colorMatches = rule.color_pattern
        ? rule.color_pattern === color
        : true; // null pattern = match wszystko

      if (nameMatches && colorMatches) {
        return {
          variant: {
            id: rule.variant_id,
            variantCode: rule.variant_code,
            variantName: rule.variant_name,
            defaultOperations: rule.default_operations,
          },
          matchedRule: {
            ruleId: rule.rule_id,
            namePattern: rule.name_pattern,
            colorPattern: rule.color_pattern,
            priority: rule.priority,
          },
        };
      }
    }

    // Nie znaleziono żadnego matcha
    return {
      variant: null,
      matchedRule: null,
    };
  } finally {
    client.release();
  }
}

/**
 * Sprawdza czy componentName pasuje do SQL LIKE pattern.
 * 
 * @param client - PostgreSQL client
 * @param componentName - Nazwa komponentu
 * @param pattern - SQL LIKE pattern (np. "WD-%", "SIEDZISKO%", "%SUROWA%")
 * @returns true jeśli pasuje, false w przeciwnym wypadku
 */
async function checkNamePattern(
  client: any,
  componentName: string,
  pattern: string
): Promise<boolean> {
  const result = await client.query(
    'SELECT $1 LIKE $2 as matches',
    [componentName, pattern]
  );
  return result.rows[0]?.matches || false;
}

/**
 * Batch resolve routing variants dla wielu komponentów naraz.
 * Efektywniejsze niż pojedyncze wywołania.
 * 
 * @param components - Array obiektów z componentName i color
 * @returns Map<componentKey, RoutingResolutionResult>
 */
export async function resolveRoutingVariantsBatch(
  components: Array<{ componentName: string; color: string | null }>
): Promise<Map<string, RoutingResolutionResult>> {
  const results = new Map<string, RoutingResolutionResult>();

  // Dla simplicity wykonujemy sekwencyjnie, ale można to zoptymalizować
  for (const comp of components) {
    const key = `${comp.componentName}|${comp.color || 'null'}`;
    const result = await resolveRoutingVariant(comp.componentName, comp.color);
    results.set(key, result);
  }

  return results;
}

/**
 * Testowa funkcja do debugowania - zwraca wszystkie aktywne reguły
 */
export async function getAllActiveRoutingRules(): Promise<any[]> {
  const client = await pool.connect();

  try {
    const result = await client.query(
      `
      SELECT 
        r.id as rule_id,
        v.variant_code,
        v.variant_name,
        r.name_pattern,
        r.color_pattern,
        r.priority,
        r.is_active
      FROM production.production_routing_variant_rules r
      JOIN production.production_routing_variants v ON v.id = r.routing_variant_id
      WHERE r.is_active = true AND v.is_active = true
      ORDER BY r.priority DESC, v.variant_code
      `
    );

    return result.rows;
  } finally {
    client.release();
  }
}
