import { pool } from '../../postgres';
import { resolveRoutingVariant, type RoutingVariant } from './routing-variant-resolver';

interface ComponentAggregation {
  color: string;
  components: ComponentForProduction[];
}

interface ComponentForProduction {
  componentId: number;
  productId: number;
  planLineId: number;
  generatedName: string;
  componentType: string;
  color: string;
  length: number | null;
  width: number | null;
  thickness: number | null;
  quantity: number;
  edge1: boolean;
  edge2: boolean;
  edge3: boolean;
  edge4: boolean;
  routingVariant: RoutingVariant | null;
  sourceFurnitureReference: string | null;
}

interface GeneratedProductionOrder {
  orderNumber: string;
  colorCode: string;
  componentCount: number;
  totalQuantity: number;
  bomId: number;
  bomItemIds: number[];
  workOrderIds: number[];
}

export interface ProductionOrderGenerationResult {
  success: boolean;
  generatedOrders: GeneratedProductionOrder[];
  errors: string[];
  summary: {
    planId: number;
    planNumber: string;
    totalOrders: number;
    totalComponents: number;
    colorBreakdown: Record<string, number>;
  };
}

/**
 * Generuje Production Orders (ZLP) z Production Plan z agregacją po kolorze.
 * 
 * Proces:
 * 1. Pobiera wszystkie linie z Production Plan
 * 2. Dla każdej linii pobiera komponenty produktu (formatki z bom.product_components)
 * 3. Resolve routing variant dla każdego komponentu
 * 4. Agreguje komponenty po color_code
 * 5. Dla każdego koloru tworzy Production Order z BOM i Work Orders
 * 
 * @param planId - ID Production Plan
 * @returns ProductionOrderGenerationResult z listą utworzonych zleceń
 */
export async function generateProductionOrdersFromPlan(
  planId: number
): Promise<ProductionOrderGenerationResult> {
  const client = await pool.connect();
  const errors: string[] = [];

  try {
    await client.query('BEGIN');

    // 1. Pobierz plan i sprawdź status
    const planResult = await client.query(`
      SELECT id, plan_number, name, status
      FROM production.production_plans
      WHERE id = $1
    `, [planId]);

    if (planResult.rows.length === 0) {
      throw new Error(`Production Plan ${planId} not found`);
    }

    const plan = planResult.rows[0];
    
    if (plan.status !== 'draft' && plan.status !== 'approved') {
      throw new Error(`Cannot generate orders for plan with status: ${plan.status}`);
    }

    // 2. Pobierz wszystkie linie planu z produktami
    const linesResult = await client.query(`
      SELECT 
        ppl.id as line_id,
        ppl.product_id,
        ppl.quantity as line_quantity,
        ppl.source_reference,
        p.sku,
        p.name as product_name
      FROM production.production_plan_lines ppl
      JOIN catalog.products p ON p.id = ppl.product_id
      WHERE ppl.plan_id = $1
        AND ppl.status = 'pending'
      ORDER BY ppl.sequence, ppl.id
    `, [planId]);

    if (linesResult.rows.length === 0) {
      return {
        success: true,
        generatedOrders: [],
        errors: ['No pending lines found in plan'],
        summary: {
          planId,
          planNumber: plan.plan_number,
          totalOrders: 0,
          totalComponents: 0,
          colorBreakdown: {},
        },
      };
    }

    // 3. Dla każdej linii pobierz komponenty produktu
    const allComponents: ComponentForProduction[] = [];

    for (const line of linesResult.rows) {
      // Najpierw pobierz BOM ID dla produktu
      const bomResult = await client.query(`
        SELECT id
        FROM bom.product_boms
        WHERE product_id = $1
        LIMIT 1
      `, [line.product_id]);

      if (bomResult.rows.length === 0) {
        errors.push(`No BOM found for product ${line.product_sku} (ID: ${line.product_id})`);
        continue;
      }

      const bomId = bomResult.rows[0].id;

      // Teraz pobierz komponenty z BOM
      const componentsResult = await client.query(`
        SELECT 
          pc.id as component_id,
          pc.generated_name,
          pc.component_type,
          pc.color,
          pc.calculated_length,
          pc.calculated_width,
          pc.thickness,
          pc.quantity,
          pc.edging_pattern
        FROM bom.product_components pc
        WHERE pc.product_bom_id = $1
        ORDER BY pc.id
      `, [bomId]);

      // WAŻNE: Dla każdego produktu w linii planu tworzymy OSOBNE rekordy formatek
      // Zamiast mnożyć quantity, tworzymy indywidualne ComponentForProduction
      // Przykład: 20× VB-50, każdy ma 10 formatek → 200 osobnych rekordów w allComponents
      const lineQuantity = parseInt(line.line_quantity);
      
      for (let productIndex = 0; productIndex < lineQuantity; productIndex++) {
        for (const comp of componentsResult.rows) {
          const componentQuantity = parseFloat(comp.quantity || '1');
          
          // Dla każdej formatki w BOM produktu tworzymy osobny rekord
          for (let compIndex = 0; compIndex < componentQuantity; compIndex++) {
            // Pobierz kolor BEZPOŚREDNIO z pola color w bazie
            // WAŻNE: HDF-BIALY i BIALY to RÓŻNE kolory!
            const color = comp.color || 'UNKNOWN';
            
            // Wyciągnij typ komponentu z nazwy (pierwsza część przed myślnikiem)
            // np. "BOK-L-VB-390x280-WOTAN" → "BOK"
            // np. "SIEDZISKO-VB-760x360-WOTAN" → "SIEDZISKO"
            const nameParts = comp.generated_name.split('-');
            const componentType = comp.component_type || (nameParts.length > 0 ? nameParts[0] : 'unknown');

            // Parse edging pattern (np. "TTTT" → edge1=T, edge2=T, edge3=T, edge4=T)
            const edgingPattern = comp.edging_pattern || '';
            const edge1 = edgingPattern.length > 0 && edgingPattern[0] !== 'F';
            const edge2 = edgingPattern.length > 1 && edgingPattern[1] !== 'F';
            const edge3 = edgingPattern.length > 2 && edgingPattern[2] !== 'F';
            const edge4 = edgingPattern.length > 3 && edgingPattern[3] !== 'F';

            allComponents.push({
              componentId: comp.component_id,
              productId: line.product_id,
              planLineId: line.line_id,
              generatedName: comp.generated_name,
              componentType,
              color,
              length: comp.calculated_length ? parseFloat(comp.calculated_length) : null,
              width: comp.calculated_width ? parseFloat(comp.calculated_width) : null,
              thickness: comp.thickness ? parseFloat(comp.thickness) : null,
              quantity: 1, // ZAWSZE 1 - każda formatka to osobny rekord
              edge1,
              edge2,
              edge3,
              edge4,
              routingVariant: null, // Wypełnimy później
              sourceFurnitureReference: line.source_reference,
            });
          }
        }
      }
    }

    // 4. Resolve routing variants dla wszystkich komponentów
    for (const component of allComponents) {
      const resolution = await resolveRoutingVariant(component.generatedName, component.color);
      component.routingVariant = resolution.variant;
      
      if (!resolution.variant) {
        errors.push(`No routing variant found for: ${component.generatedName} (${component.color})`);
      }
    }

    // 5. Agreguj komponenty po kolorze
    const colorGroups = aggregateComponentsByColor(allComponents);

    // 6. Generuj Production Orders dla każdej grupy kolorów
    const generatedOrders: GeneratedProductionOrder[] = [];

    for (const group of colorGroups) {
      const orderResult = await createProductionOrderForColor(
        client,
        planId,
        plan.plan_number,
        group.color,
        group.components
      );

      generatedOrders.push(orderResult);
    }

    // 7. Oznacz linie planu jako scheduled (jeśli wszystko poszło dobrze)
    if (errors.length === 0) {
      await client.query(`
        UPDATE production.production_plan_lines
        SET status = 'scheduled'
        WHERE plan_id = $1 AND status = 'pending'
      `, [planId]);

      await client.query(`
        UPDATE production.production_plans
        SET status = 'in_progress'
        WHERE id = $1 AND status = 'draft'
      `, [planId]);
      
      await client.query('COMMIT');
    } else {
      // Jeśli są błędy, wycofaj wszystkie zmiany
      await client.query('ROLLBACK');
    }

    // Oblicz podsumowanie
    const colorBreakdown: Record<string, number> = {};
    let totalComponents = 0;
    
    for (const order of generatedOrders) {
      colorBreakdown[order.colorCode] = order.componentCount;
      totalComponents += order.componentCount;
    }

    return {
      success: errors.length === 0,
      generatedOrders,
      errors,
      summary: {
        planId,
        planNumber: plan.plan_number,
        totalOrders: generatedOrders.length,
        totalComponents,
        colorBreakdown,
      },
    };

  } catch (error) {
    await client.query('ROLLBACK');
    throw error;
  } finally {
    client.release();
  }
}


/**
 * Agreguje komponenty według koloru.
 */
function aggregateComponentsByColor(
  components: ComponentForProduction[]
): ComponentAggregation[] {
  const groups = new Map<string, ComponentForProduction[]>();

  for (const comp of components) {
    const color = comp.color || 'UNKNOWN';
    
    if (!groups.has(color)) {
      groups.set(color, []);
    }
    
    groups.get(color)!.push(comp);
  }

  return Array.from(groups.entries()).map(([color, comps]) => ({
    color,
    components: comps,
  }));
}

/**
 * Tworzy Production Order dla danej grupy kolorów.
 */
async function createProductionOrderForColor(
  client: any,
  planId: number,
  planNumber: string,
  colorCode: string,
  components: ComponentForProduction[]
): Promise<GeneratedProductionOrder> {
  
  // 1. Generuj numer zlecenia ZLP z kolorem (używamy numeru planu, nie globalnej sekwencji)
  // Usuń prefix "PLAN-" z planNumber jeśli istnieje
  const cleanPlanNumber = planNumber.replace(/^PLAN-/i, '');
  const orderNumber = `ZLP-${cleanPlanNumber}-${colorCode}`;

  // 2. Utwórz Production Order
  const orderResult = await client.query(`
    INSERT INTO production.production_orders (
      order_number,
      product_id,
      status,
      priority,
      quantity_planned,
      unit_of_measure,
      color_code,
      notes,
      source_order_number
    ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9)
    RETURNING id
  `, [
    orderNumber,
    components[0].productId, // Pierwszy produkt jako reprezentant
    'draft',
    'normal',
    components.length,
    'szt',
    colorCode,
    `Generated from Production Plan: ${planNumber}`,
    planNumber,
  ]);

  const productionOrderId = orderResult.rows[0].id;

  // 3. Utwórz Production Order BOM
  const bomResult = await client.query(`
    INSERT INTO production.production_order_boms (
      production_order_id,
      source_plan_id,
      color_code,
      status
    ) VALUES ($1, $2, $3, $4)
    RETURNING id
  `, [productionOrderId, planId, colorCode, 'active']);

  const bomId = bomResult.rows[0].id;

  // 4. Utwórz BOM Items dla wszystkich komponentów
  const bomItemIds: number[] = [];

  for (const comp of components) {
    const itemResult = await client.query(`
      INSERT INTO production.production_order_bom_items (
        production_order_bom_id,
        component_name,
        component_type,
        quantity,
        unit_of_measure,
        routing_variant_id,
        required_operations,
        color_code,
        length,
        width,
        thickness,
        source_plan_line_id,
        source_product_id,
        source_furniture_reference,
        item_status,
        quantity_ordered
      ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16)
      RETURNING id
    `, [
      bomId,
      comp.generatedName,
      comp.componentType,
      comp.quantity,
      'szt',
      comp.routingVariant?.id || null,
      comp.routingVariant?.defaultOperations ? JSON.stringify(comp.routingVariant.defaultOperations) : null,
      comp.color,
      comp.length,
      comp.width,
      comp.thickness,
      comp.planLineId,
      comp.productId,
      comp.sourceFurnitureReference,
      'pending',
      comp.quantity,
    ]);

    bomItemIds.push(itemResult.rows[0].id);
  }

  // 5. Generuj Work Orders na podstawie unique operations
  const workOrderIds = await generateWorkOrdersForBom(
    client,
    productionOrderId,
    bomId,
    components
  );

  return {
    orderNumber,
    colorCode,
    componentCount: components.length,
    totalQuantity: components.reduce((sum, c) => sum + c.quantity, 0),
    bomId,
    bomItemIds,
    workOrderIds,
  };
}

/**
 * Generuje Work Orders dla Production Order na podstawie required_operations z BOM items.
 */
async function generateWorkOrdersForBom(
  client: any,
  productionOrderId: number,
  bomId: number,
  components: ComponentForProduction[]
): Promise<number[]> {
  
  // Zbierz wszystkie unikalne operacje
  const allOperations = new Set<string>();
  
  for (const comp of components) {
    if (comp.routingVariant?.defaultOperations) {
      for (const op of comp.routingVariant.defaultOperations) {
        allOperations.add(op);
      }
    }
  }

  // Sekwencja operacji (preferowane kolejność)
  const operationSequence = [
    'cutting',
    'edging',
    'drilling_holes',
    'upholstering',
    'drilling_mount',
    'drilling',
    'assembly',
    'packing',
  ];

  // Posortuj operacje według preferowanej sekwencji
  const sortedOperations = Array.from(allOperations).sort((a, b) => {
    const indexA = operationSequence.indexOf(a);
    const indexB = operationSequence.indexOf(b);
    
    if (indexA === -1 && indexB === -1) return 0;
    if (indexA === -1) return 1;
    if (indexB === -1) return -1;
    
    return indexA - indexB;
  });

  // Generuj Work Order dla każdej operacji
  const workOrderIds: number[] = [];
  let sequence = 1;

  for (const operation of sortedOperations) {
    const workOrderNumber = `WO-${productionOrderId}-${sequence}`;
    
    const result = await client.query(`
      INSERT INTO production.production_work_orders (
        work_order_number,
        production_order_id,
        sequence,
        status,
        quantity_planned
      ) VALUES ($1, $2, $3, $4, $5)
      RETURNING id
    `, [
      workOrderNumber,
      productionOrderId,
      sequence,
      'pending',
      components.length,
    ]);

    workOrderIds.push(result.rows[0].id);
    sequence++;
  }

  return workOrderIds;
}

/**
 * Pobiera kolejny numer sekwencji dla ZLP (format ZLP-0001 do ZLP-9999).
 */
async function getNextZlpSequence(client: any): Promise<number> {
  const result = await client.query(`
    SELECT COALESCE(MAX(
      CAST(
        SUBSTRING(order_number FROM 'ZLP-([0-9]+)') AS INTEGER
      )
    ), 0) + 1 as next_seq
    FROM production.production_orders
    WHERE order_number LIKE 'ZLP-%'
  `);

  return result.rows[0].next_seq;
}
