import { Pool, PoolClient } from "pg";
import type { ProductionPlan, ProductionPlanLine } from "@shared/schema";
import { findFormatkaInWarehouse } from "../warehouse/formatka-search";
import { createReservation } from "./buffer-reservations";

type PoolOrClient = Pool | PoolClient;

interface InsertProductionPlan {
  planNumber?: string;
  name: string;
  description?: string | null;
  plannedStartDate?: Date | null;
  plannedEndDate?: Date | null;
  status?: string;
  priority?: string;
  notes?: string | null;
  metadata?: any;
  createdBy?: number | null;
}

interface InsertProductionPlanLine {
  planId: number;
  productId: number;
  quantity: number;
  sourceType?: string | null;
  sourceId?: number | null;
  sourceReference?: string | null;
  productionOrderId?: number | null;
  routingId?: number | null;
  routingOverride?: any;
  bomId?: number | null;
  plannedStartDate?: Date | null;
  plannedEndDate?: Date | null;
  status?: string;
  sequence?: number | null;
  notes?: string | null;
  metadata?: any;
}

function mapRowToPlan(row: any): ProductionPlan {
  return {
    id: row.id,
    planNumber: row.plan_number,
    shortName: row.short_name,
    name: row.name,
    description: row.description,
    plannedStartDate: row.planned_start_date,
    plannedEndDate: row.planned_end_date,
    actualStartDate: row.actual_start_date,
    actualEndDate: row.actual_end_date,
    status: row.status,
    priority: row.priority,
    notes: row.notes,
    metadata: row.metadata,
    createdBy: row.created_by,
    approvedBy: row.approved_by,
    approvedAt: row.approved_at,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
    nameTemplateId: row.name_template_id,
    nameSequence: row.name_sequence,
    nameGeneratedAt: row.name_generated_at,
    nameGenerationMeta: row.name_generation_meta,
  };
}

function mapRowToPlanLine(row: any): ProductionPlanLine & {
  reservedQuantity?: number;
  warehouseTotalQty?: number | null;
  warehouseReservedQty?: number | null;
  packedProductId?: number | null;
  packedProductSku?: string | null;
} {
  return {
    id: row.id,
    planId: row.plan_id,
    productId: row.product_id,
    quantity: row.quantity,
    reservedQuantity: row.reserved_quantity ?? 0, // Per-line reservation
    colorCode: row.color_code ?? null,
    sourceType: row.source_type,
    sourceId: row.source_id,
    sourceReference: row.source_reference,
    productionOrderId: row.production_order_id,
    routingId: row.routing_id,
    routingOverride: row.routing_override,
    bomId: row.bom_id,
    plannedStartDate: row.planned_start_date,
    plannedEndDate: row.planned_end_date,
    status: row.status,
    sequence: row.sequence,
    notes: row.notes,
    metadata: row.metadata,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
    // Warehouse data from LEFT JOIN
    warehouseTotalQty: row.warehouse_total_qty ?? null,
    warehouseReservedQty: row.warehouse_reserved_qty ?? null,
    packedProductId: row.packed_product_id ?? null,
    packedProductSku: row.packed_product_sku ?? null,
  };
}

interface PlanFilters {
  status?: string;
  priority?: string;
  createdBy?: number;
  startDate?: Date;
  endDate?: Date;
  search?: string;
  limit?: number;
  offset?: number;
}

export async function getPlans(pool: Pool, filters?: PlanFilters): Promise<ProductionPlan[]> {
  let query = `SELECT * FROM production.production_plans WHERE 1=1`;
  const params: any[] = [];
  let paramIndex = 1;

  if (filters?.status) {
    query += ` AND status = $${paramIndex++}`;
    params.push(filters.status);
  }

  if (filters?.priority) {
    query += ` AND priority = $${paramIndex++}`;
    params.push(filters.priority);
  }

  if (filters?.createdBy) {
    query += ` AND created_by = $${paramIndex++}`;
    params.push(filters.createdBy);
  }

  if (filters?.startDate) {
    query += ` AND planned_start_date >= $${paramIndex++}`;
    params.push(filters.startDate);
  }

  if (filters?.endDate) {
    query += ` AND planned_end_date <= $${paramIndex++}`;
    params.push(filters.endDate);
  }

  if (filters?.search) {
    query += ` AND (plan_number ILIKE $${paramIndex} OR name ILIKE $${paramIndex})`;
    params.push(`%${filters.search}%`);
    paramIndex++;
  }

  query += ` ORDER BY created_at DESC`;

  if (filters?.limit) {
    query += ` LIMIT $${paramIndex++}`;
    params.push(filters.limit);
  }

  if (filters?.offset) {
    query += ` OFFSET $${paramIndex++}`;
    params.push(filters.offset);
  }

  const result = await pool.query(query, params);
  return result.rows.map(mapRowToPlan);
}

export async function getPlanById(pool: Pool, id: number): Promise<ProductionPlan | null> {
  const result = await pool.query(
    'SELECT * FROM production.production_plans WHERE id = $1',
    [id]
  );
  
  if (result.rows.length === 0) {
    return null;
  }
  
  return mapRowToPlan(result.rows[0]);
}

export async function generatePlanNumber(pool: Pool): Promise<string> {
  const result = await pool.query(
    `SELECT plan_number FROM production.production_plans 
     WHERE plan_number ~ '^PLAN-[0-9]{4}$' 
     ORDER BY plan_number DESC LIMIT 1`
  );

  if (result.rows.length === 0) {
    return 'PLAN-0001';
  }

  const lastNumber = parseInt(result.rows[0].plan_number.split('-')[1]);
  return `PLAN-${String(lastNumber + 1).padStart(4, '0')}`;
}

export async function createPlan(pool: Pool, data: InsertProductionPlan): Promise<ProductionPlan> {
  const planNumber = data.planNumber || await generatePlanNumber(pool);

  const result = await pool.query(
    `INSERT INTO production.production_plans 
      (plan_number, name, description, planned_start_date, planned_end_date, 
       status, priority, notes, metadata, created_by, created_at, updated_at)
     VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
     RETURNING *`,
    [
      planNumber,
      data.name,
      data.description,
      data.plannedStartDate,
      data.plannedEndDate,
      data.status || 'draft',
      data.priority || 'normal',
      data.notes,
      data.metadata ? JSON.stringify(data.metadata) : null,
      data.createdBy,
    ]
  );

  const createdPlan = result.rows[0];
  
  // Generate short_name in format PLAN-21 (just the ID number, no padding)
  const shortName = `PLAN-${createdPlan.id}`;
  
  // Update the plan with short_name and append it to the name
  const updatedName = `${createdPlan.name} | ${shortName}`;
  await pool.query(
    `UPDATE production.production_plans SET short_name = $1, name = $2 WHERE id = $3`,
    [shortName, updatedName, createdPlan.id]
  );
  
  createdPlan.short_name = shortName;
  createdPlan.name = updatedName;

  return mapRowToPlan(createdPlan);
}

export async function updatePlan(
  pool: Pool,
  id: number,
  data: Partial<InsertProductionPlan>
): Promise<ProductionPlan | null> {
  const fields: string[] = [];
  const values: any[] = [];
  let paramIndex = 1;

  if (data.name !== undefined) {
    fields.push(`name = $${paramIndex++}`);
    values.push(data.name);
  }

  if (data.description !== undefined) {
    fields.push(`description = $${paramIndex++}`);
    values.push(data.description);
  }

  if (data.plannedStartDate !== undefined) {
    fields.push(`planned_start_date = $${paramIndex++}`);
    values.push(data.plannedStartDate);
  }

  if (data.plannedEndDate !== undefined) {
    fields.push(`planned_end_date = $${paramIndex++}`);
    values.push(data.plannedEndDate);
  }

  if (data.status !== undefined) {
    fields.push(`status = $${paramIndex++}`);
    values.push(data.status);
  }

  if (data.priority !== undefined) {
    fields.push(`priority = $${paramIndex++}`);
    values.push(data.priority);
  }

  if (data.notes !== undefined) {
    fields.push(`notes = $${paramIndex++}`);
    values.push(data.notes);
  }

  if (data.metadata !== undefined) {
    fields.push(`metadata = $${paramIndex++}`);
    values.push(JSON.stringify(data.metadata));
  }

  if (fields.length === 0) {
    return getPlanById(pool, id);
  }

  fields.push(`updated_at = CURRENT_TIMESTAMP`);
  values.push(id);

  const result = await pool.query(
    `UPDATE production.production_plans 
     SET ${fields.join(', ')} 
     WHERE id = $${paramIndex}
     RETURNING *`,
    values
  );

  if (result.rows.length === 0) {
    return null;
  }

  return mapRowToPlan(result.rows[0]);
}

export async function deletePlan(pool: Pool, id: number): Promise<boolean> {
  const result = await pool.query(
    'DELETE FROM production.production_plans WHERE id = $1',
    [id]
  );
  
  return result.rowCount !== null && result.rowCount > 0;
}

export async function getPlanLines(pool: Pool, planId: number): Promise<ProductionPlanLine[]> {
  const result = await pool.query(
    `SELECT 
       ppl.*,
       pp.id as packed_product_id,
       pp.product_sku as packed_product_sku,
       pp.reserved_quantity as warehouse_reserved_qty,
       pp.quantity as warehouse_total_qty
     FROM production.production_plan_lines ppl
     LEFT JOIN warehouse.packed_products pp ON pp.catalog_product_id = ppl.product_id
     WHERE ppl.plan_id = $1 
       AND ppl.deleted_at IS NULL
     ORDER BY ppl.sequence ASC, ppl.created_at ASC`,
    [planId]
  );
  
  return result.rows.map(mapRowToPlanLine);
}

export async function getPlanLineById(pool: Pool, id: number): Promise<ProductionPlanLine | null> {
  const result = await pool.query(
    'SELECT * FROM production.production_plan_lines WHERE id = $1',
    [id]
  );
  
  if (result.rows.length === 0) {
    return null;
  }
  
  return mapRowToPlanLine(result.rows[0]);
}

export interface CreatePlanLineResult {
  planLine: ProductionPlanLine;
  reservationInfo: {
    packedProductReserved: boolean;
    packedProductSku?: string;
    packedQtyReserved?: number;
    formatkaSearched: boolean;
    formatkiFound: string[];
    formatkiReserved: string[];
  };
}

export async function createPlanLine(pool: Pool, data: InsertProductionPlanLine): Promise<CreatePlanLineResult> {
  const client = await pool.connect();
  
  // Tracking reservation info
  const reservationInfo = {
    packedProductReserved: false,
    packedProductSku: undefined as string | undefined,
    packedQtyReserved: undefined as number | undefined,
    formatkaSearched: false,
    formatkiFound: [] as string[],
    formatkiReserved: [] as string[],
  };
  
  try {
    await client.query('BEGIN');
    
    // KROK 0a: Automatycznie pobierz BOM dla produktu (jeśli nie podano)
    let bomIdToUse = data.bomId;
    if (!bomIdToUse && data.productId) {
      const bomResult = await client.query(`
        SELECT id FROM bom.product_boms 
        WHERE product_id = $1 AND is_active = true
        ORDER BY version DESC
        LIMIT 1
      `, [data.productId]);
      
      if (bomResult.rows.length > 0) {
        bomIdToUse = bomResult.rows[0].id;
        console.log(`📋 [BOM] Auto-przypisano BOM #${bomIdToUse} dla produktu #${data.productId}`);
      }
    }
    
    // KROK 0b: Sprawdź czy dokładnie to samo źródło już istnieje W JAKIMKOLWIEK planie (zapobiegaj duplikacji globalnie)
    // NIE pozwalamy na ten sam produkt z tego samego zamówienia w żadnym planie
    // Sprawdzamy po source_reference (numer zamówienia) zamiast source_id
    if (data.sourceType === 'order_demand' && data.sourceReference) {
      const existingCheck = await client.query(
        `SELECT ppl.id, ppl.plan_id, pp.name as plan_name, pp.short_name
         FROM production.production_plan_lines ppl
         JOIN production.production_plans pp ON pp.id = ppl.plan_id
         WHERE ppl.product_id = $1 
           AND ppl.source_type = $2 
           AND ppl.source_reference = $3 
           AND ppl.deleted_at IS NULL
         LIMIT 1`,
        [data.productId, data.sourceType, data.sourceReference]
      );
      
      if (existingCheck.rows.length > 0) {
        const existing = existingCheck.rows[0];
        const planInfo = existing.short_name || `Plan #${existing.plan_id}`;
        await client.query('ROLLBACK');
        throw new Error(`Ten produkt z zamówienia #${data.sourceReference} już istnieje w planie "${planInfo}". Nie można dodać duplikatu.`);
      }
    }
    
    // KROK 1: Utwórz plan line
    const result = await client.query(
      `INSERT INTO production.production_plan_lines 
        (plan_id, product_id, quantity, reserved_quantity, source_type, source_id, source_reference,
         production_order_id, routing_id, routing_override, bom_id,
         planned_start_date, planned_end_date, status, sequence, notes, metadata,
         created_at, updated_at)
       VALUES ($1, $2, $3, 0, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, 
               CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
       RETURNING *`,
      [
        data.planId,
        data.productId,
        data.quantity,
        data.sourceType,
        data.sourceId,
        data.sourceReference,
        data.productionOrderId,
        data.routingId,
        data.routingOverride ? JSON.stringify(data.routingOverride) : null,
        bomIdToUse,
        data.plannedStartDate,
        data.plannedEndDate,
        data.status || 'pending',
        data.sequence,
        data.notes,
        data.metadata ? JSON.stringify(data.metadata) : null,
      ]
    );

    const planLineId = result.rows[0].id;

    // KROK 1.5: Automatycznie zarezerwuj produkt spakowany w magazynie (jeśli istnieje i jest dostępność)
    // Używamy tabeli packed_product_items z FIFO (najstarsze dostępne sztuki pierwsze)
    // Jeśli produkt spakowany zostanie zarezerwowany - NIE rezerwujemy formatek (produkt gotowy)
    let packedProductReserved = false;
    
    const packedProductCheck = await client.query(`
      SELECT id, product_sku, product_name
      FROM warehouse.packed_products
      WHERE catalog_product_id = $1
    `, [data.productId]);

    if (packedProductCheck.rows.length > 0) {
      const pp = packedProductCheck.rows[0];
      
      // Znajdź dostępne sztuki (FIFO - najstarsze pierwsze) i zablokuj je
      const availableItemsResult = await client.query(`
        SELECT id, serial_number
        FROM warehouse.packed_product_items
        WHERE packed_product_id = $1 
          AND status = 'available'
        ORDER BY packed_at ASC
        LIMIT $2
        FOR UPDATE SKIP LOCKED
      `, [pp.id, data.quantity]);
      
      const availableQty = availableItemsResult.rows.length;
      
      if (availableQty > 0) {
        const qtyToReserve = Math.min(data.quantity, availableQty);
        const itemIds = availableItemsResult.rows.slice(0, qtyToReserve).map((r: any) => r.id);
        const serialNumbers = availableItemsResult.rows.slice(0, qtyToReserve).map((r: any) => r.serial_number);
        
        // Zarezerwuj konkretne sztuki
        await client.query(`
          UPDATE warehouse.packed_product_items
          SET status = 'reserved',
              reserved_for_plan_line_id = $1,
              reserved_at = CURRENT_TIMESTAMP,
              updated_at = CURRENT_TIMESTAMP
          WHERE id = ANY($2::int[])
        `, [planLineId, itemIds]);
        
        // Aktualizuj również licznik w tabeli agregacyjnej (dla kompatybilności wstecznej)
        await client.query(`
          UPDATE warehouse.packed_products
          SET reserved_quantity = reserved_quantity + $1,
              updated_at = CURRENT_TIMESTAMP
          WHERE id = $2
        `, [qtyToReserve, pp.id]);
        
        // Update plan line reserved_quantity to match actual reservation
        await client.query(`
          UPDATE production.production_plan_lines
          SET reserved_quantity = $1, updated_at = CURRENT_TIMESTAMP
          WHERE id = $2
        `, [qtyToReserve, planLineId]);
        
        packedProductReserved = true;
        reservationInfo.packedProductReserved = true;
        reservationInfo.packedProductSku = pp.product_sku;
        reservationInfo.packedQtyReserved = qtyToReserve;
        console.log(`✅ [WAREHOUSE] Auto-zarezerwowano ${qtyToReserve}x SPAKOWANY ${pp.product_name} (${pp.product_sku})`);
        console.log(`📦 [WAREHOUSE] Zarezerwowane sztuki (FIFO): ${serialNumbers.join(', ')}`);
        console.log(`📦 [WAREHOUSE] Produkt spakowany - pomijam rezerwację formatek z BOM`);
      } else {
        console.log(`⚠️ [WAREHOUSE] Brak dostępności SPAKOWANEGO dla ${pp.product_name} (${pp.product_sku}). Brak dostępnych sztuk.`);
        console.log(`🔍 [WAREHOUSE] Szukam formatek z BOM do rezerwacji...`);
      }
    }

    // Fetch updated plan line
    const updatedLineResult = await client.query(
      'SELECT * FROM production.production_plan_lines WHERE id = $1',
      [planLineId]
    );
    const planLine = mapRowToPlanLine(updatedLineResult.rows[0]);

    // KROK 2: Jeśli NIE zarezerwowano spakowanego produktu - szukaj formatek z BOM
    // Pomijamy ten krok jeśli produkt spakowany został zarezerwowany (jest gotowy, nie trzeba produkować)
    if (!packedProductReserved) {
      reservationInfo.formatkaSearched = true;
      const bomResult = await client.query(`
        SELECT bom.id as bom_id
        FROM bom.product_boms bom
        WHERE bom.product_id = $1 
          AND bom.is_active = true
        ORDER BY bom.version DESC
        LIMIT 1
      `, [data.productId]);

      if (bomResult.rows.length > 0) {
      const bomId = bomResult.rows[0].bom_id;
      
      // Pobierz komponenty BOM
      const componentsResult = await client.query(`
        SELECT 
          id,
          generated_name,
          quantity,
          calculated_length,
          calculated_width,
          thickness,
          component_type
        FROM bom.product_components
        WHERE product_bom_id = $1
      `, [bomId]);

      const reservationLogs: string[] = [];
      
      for (const component of componentsResult.rows) {
        // Identyfikuj formatki przez component_type lub konwencję nazewniczą
        const isFormatka = component.component_type === 'panel' || 
                          component.generated_name?.includes('-') ||
                          (component.calculated_length && component.calculated_width);

        if (isFormatka && component.generated_name) {
          // Wyodrębnij kolor z nazwy (ostatnia część po ostatnim myślniku)
          const nameParts = component.generated_name.split('-');
          const color = nameParts[nameParts.length - 1]?.toUpperCase() || null;
          
          const length = parseFloat(component.calculated_length) || 0;
          const width = parseFloat(component.calculated_width) || 0;
          const thickness = parseFloat(component.thickness) || null;
          
          if (length > 0 && width > 0) {
            // Wyszukaj formatkę w magazynie
            const formatka = await findFormatkaInWarehouse(
              client,
              component.generated_name,
              length,
              width,
              thickness,
              color
            );

            if (formatka) {
              reservationInfo.formatkiFound.push(formatka.name || formatka.internalCode);
              const quantityNeeded = (component.quantity || 1) * data.quantity;
              
              if (formatka.quantityAvailable >= quantityNeeded) {
                // Utwórz rezerwację
                try {
                  await createReservation(client, {
                    zlpId: null as any, // Plan line nie ma jeszcze ZLP - wypełnimy później
                    zlpItemId: planLine.id,
                    productSku: formatka.internalCode,
                    quantityReserved: quantityNeeded.toString(),
                    unitOfMeasure: formatka.unitOfMeasure,
                    locationId: null,
                    reservedBy: data.metadata?.created_by || null,
                    notes: `Rezerwacja dla planu #${data.planId}, produkt ${data.productId}: ${component.generated_name}`,
                  });
                  
                  reservationInfo.formatkiReserved.push(formatka.name || formatka.internalCode);
                  reservationLogs.push(
                    `✅ Zarezerwowano ${quantityNeeded}x ${formatka.name} (${formatka.internalCode})`
                  );
                } catch (err) {
                  console.error(`❌ Błąd rezerwacji dla ${component.generated_name}:`, err);
                  reservationLogs.push(
                    `❌ Nie udało się zarezerwować ${component.generated_name}: ${err instanceof Error ? err.message : 'Unknown error'}`
                  );
                }
              } else {
                reservationLogs.push(
                  `⚠️ Niewystarczająca ilość ${formatka.name}: potrzeba ${quantityNeeded}, dostępne ${formatka.quantityAvailable}`
                );
              }
            } else {
              reservationLogs.push(
                `⚠️ Nie znaleziono formatki: ${component.generated_name} (${length}x${width}mm, kolor: ${color || 'brak'})`
              );
            }
          }
        }
      }

      if (reservationLogs.length > 0) {
        console.log(`\n📦 [REZERWACJE FORMATEK] Plan Line #${planLine.id}:`);
        reservationLogs.forEach(log => console.log(`  ${log}`));
      }
      }
    } // Koniec bloku if (!packedProductReserved)
    
    await client.query('COMMIT');
    return { planLine, reservationInfo };
  } catch (error) {
    await client.query('ROLLBACK');
    throw error;
  } finally {
    client.release();
  }
}

export async function updatePlanLine(
  pool: Pool,
  id: number,
  data: Partial<InsertProductionPlanLine>
): Promise<ProductionPlanLine | null> {
  const fields: string[] = [];
  const values: any[] = [];
  let paramIndex = 1;

  if (data.productId !== undefined) {
    fields.push(`product_id = $${paramIndex++}`);
    values.push(data.productId);
  }

  if (data.quantity !== undefined) {
    fields.push(`quantity = $${paramIndex++}`);
    values.push(data.quantity);
  }

  if (data.sourceType !== undefined) {
    fields.push(`source_type = $${paramIndex++}`);
    values.push(data.sourceType);
  }

  if (data.sourceId !== undefined) {
    fields.push(`source_id = $${paramIndex++}`);
    values.push(data.sourceId);
  }

  if (data.sourceReference !== undefined) {
    fields.push(`source_reference = $${paramIndex++}`);
    values.push(data.sourceReference);
  }

  if (data.productionOrderId !== undefined) {
    fields.push(`production_order_id = $${paramIndex++}`);
    values.push(data.productionOrderId);
  }

  if (data.routingId !== undefined) {
    fields.push(`routing_id = $${paramIndex++}`);
    values.push(data.routingId);
  }

  if (data.routingOverride !== undefined) {
    fields.push(`routing_override = $${paramIndex++}`);
    values.push(data.routingOverride ? JSON.stringify(data.routingOverride) : null);
  }

  if (data.bomId !== undefined) {
    fields.push(`bom_id = $${paramIndex++}`);
    values.push(data.bomId);
  }

  if (data.plannedStartDate !== undefined) {
    fields.push(`planned_start_date = $${paramIndex++}`);
    values.push(data.plannedStartDate);
  }

  if (data.plannedEndDate !== undefined) {
    fields.push(`planned_end_date = $${paramIndex++}`);
    values.push(data.plannedEndDate);
  }

  if (data.status !== undefined) {
    fields.push(`status = $${paramIndex++}`);
    values.push(data.status);
  }

  if (data.sequence !== undefined) {
    fields.push(`sequence = $${paramIndex++}`);
    values.push(data.sequence);
  }

  if (data.notes !== undefined) {
    fields.push(`notes = $${paramIndex++}`);
    values.push(data.notes);
  }

  if (data.metadata !== undefined) {
    fields.push(`metadata = $${paramIndex++}`);
    values.push(data.metadata ? JSON.stringify(data.metadata) : null);
  }

  if (fields.length === 0) {
    return getPlanLineById(pool, id);
  }

  fields.push(`updated_at = CURRENT_TIMESTAMP`);
  values.push(id);

  const result = await pool.query(
    `UPDATE production.production_plan_lines 
     SET ${fields.join(', ')} 
     WHERE id = $${paramIndex}
     RETURNING *`,
    values
  );

  if (result.rows.length === 0) {
    return null;
  }

  return mapRowToPlanLine(result.rows[0]);
}

export async function deletePlanLine(pool: Pool, id: number): Promise<boolean> {
  const client = await pool.connect();
  try {
    await client.query('BEGIN');
    
    // KROK 1: Pobierz informacje o linii przed usunięciem
    const lineInfo = await client.query(
      `SELECT product_id, quantity FROM production.production_plan_lines WHERE id = $1`,
      [id]
    );
    
    if (lineInfo.rows.length === 0) {
      await client.query('ROLLBACK');
      return false;
    }
    
    const { product_id, quantity } = lineInfo.rows[0];
    
    // KROK 2: Sprawdź czy to packed product i zwolnij rezerwację
    const packedProductCheck = await client.query(
      `SELECT id, product_sku FROM warehouse.packed_products WHERE catalog_product_id = $1 LIMIT 1`,
      [product_id]
    );
    
    if (packedProductCheck.rows.length > 0) {
      const packedProduct = packedProductCheck.rows[0];
      
      // Zwolnij pojedyncze sztuki zarezerwowane dla tej linii planu
      const releasedItemsResult = await client.query(
        `UPDATE warehouse.packed_product_items
         SET status = 'available',
             reserved_for_plan_line_id = NULL,
             reserved_at = NULL,
             updated_at = CURRENT_TIMESTAMP
         WHERE reserved_for_plan_line_id = $1
           AND status = 'reserved'
         RETURNING id, serial_number`,
        [id]
      );
      
      const releasedCount = releasedItemsResult.rows.length;
      
      // Aktualizuj licznik w tabeli agregacyjnej (dla kompatybilności wstecznej)
      await client.query(
        `UPDATE warehouse.packed_products 
         SET reserved_quantity = GREATEST(0, reserved_quantity - $1) 
         WHERE id = $2`,
        [releasedCount, packedProduct.id]
      );
      
      const serialNumbers = releasedItemsResult.rows.map((r: any) => r.serial_number);
      console.log(`📦 [ZWOLNIENIE] Zwolniono ${releasedCount} szt. rezerwacji produktu spakowanego ${packedProduct.product_sku} (ZLP line #${id} usunięta)`);
      if (serialNumbers.length > 0) {
        console.log(`📦 [ZWOLNIONE SZTUKI]: ${serialNumbers.join(', ')}`);
      }
    }
    
    // KROK 2.5: Zwolnij rezerwacje formatek BOM (production_buffer_reservations)
    const formatkiReservationsResult = await client.query(`
      SELECT id, product_sku, quantity_reserved
      FROM production.production_buffer_reservations
      WHERE zlp_item_id = $1 AND status = 'ACTIVE'
    `, [id]);
    
    if (formatkiReservationsResult.rows.length > 0) {
      // Anuluj rezerwacje formatek
      await client.query(`
        UPDATE production.production_buffer_reservations
        SET status = 'CANCELLED',
            cancelled_at = CURRENT_TIMESTAMP,
            notes = COALESCE(notes, '') || ' [Auto-anulowano - linia planu usunięta]'
        WHERE zlp_item_id = $1 AND status = 'ACTIVE'
      `, [id]);
      
      console.log(`📦 [ZWOLNIENIE FORMATEK] Zwolniono ${formatkiReservationsResult.rows.length} rezerwacji formatek (ZLP line #${id} usunięta):`);
      for (const res of formatkiReservationsResult.rows) {
        console.log(`  🔓 ${res.product_sku} (${res.quantity_reserved} szt.)`);
      }
    }
    
    // KROK 3: Soft delete linii planu (zachowaj historię rezerwacji)
    const deleteResult = await client.query(
      `UPDATE production.production_plan_lines 
       SET status = 'cancelled', 
           deleted_at = CURRENT_TIMESTAMP,
           updated_at = CURRENT_TIMESTAMP
       WHERE id = $1 AND deleted_at IS NULL
       RETURNING id`,
      [id]
    );
    
    await client.query('COMMIT');
    return deleteResult.rowCount !== null && deleteResult.rowCount > 0;
  } catch (error) {
    await client.query('ROLLBACK');
    throw error;
  } finally {
    client.release();
  }
}

export interface DemandFilters {
  startDate?: Date;
  endDate?: Date;
  marketplace?: string;
  orderStatus?: string[];
  paymentStatus?: string[];
}

export interface AggregatedDemand {
  productId: number;
  sku: string;
  title: string;
  totalQuantity: number;
  orderCount: number;
  orderReferences: Array<{
    orderId: number;
    orderNumber: string;
    marketplace: string;
    quantity: number;
  }>;
}

export async function aggregateDemand(
  pool: Pool,
  filters?: DemandFilters
): Promise<AggregatedDemand[]> {
  let query = `
    WITH order_product_mapping AS (
      SELECT 
        oi.id as order_item_id,
        oi.order_id,
        oi.quantity,
        oi.name as product_name,
        oi.raw_data->'offer'->'external'->>'id' as external_sku,
        o.source as marketplace,
        o.order_number,
        o.status as order_status,
        o.payment_status,
        o.order_date
      FROM commerce.order_items oi
      JOIN commerce.orders o ON oi.order_id = o.id
      WHERE oi.raw_data->'offer'->'external'->>'id' IS NOT NULL
    ),
    catalog_matched AS (
      SELECT 
        opm.*,
        cp.id as catalog_product_id,
        cp.sku as catalog_sku,
        cp.title as catalog_title
      FROM order_product_mapping opm
      LEFT JOIN catalog.products cp ON cp.sku = opm.external_sku
      WHERE cp.id IS NOT NULL
    )
    SELECT 
      cm.catalog_product_id,
      cm.catalog_sku,
      cm.catalog_title,
      SUM(cm.quantity) as total_quantity,
      COUNT(DISTINCT cm.order_id) as order_count,
      jsonb_agg(
        jsonb_build_object(
          'orderId', cm.order_id,
          'orderNumber', cm.order_number,
          'marketplace', cm.marketplace,
          'quantity', cm.quantity
        ) ORDER BY cm.order_date DESC
      ) as order_references
    FROM catalog_matched cm
    WHERE 1=1
  `;

  const params: any[] = [];
  let paramIndex = 1;

  if (filters?.startDate) {
    query += ` AND cm.order_date >= $${paramIndex++}`;
    params.push(filters.startDate);
  }

  if (filters?.endDate) {
    query += ` AND cm.order_date <= $${paramIndex++}`;
    params.push(filters.endDate);
  }

  if (filters?.marketplace) {
    query += ` AND cm.marketplace = $${paramIndex++}`;
    params.push(filters.marketplace);
  }

  if (filters?.orderStatus && filters.orderStatus.length > 0) {
    query += ` AND cm.order_status = ANY($${paramIndex++})`;
    params.push(filters.orderStatus);
  }

  if (filters?.paymentStatus && filters.paymentStatus.length > 0) {
    query += ` AND cm.payment_status = ANY($${paramIndex++})`;
    params.push(filters.paymentStatus);
  }

  query += `
    GROUP BY cm.catalog_product_id, cm.catalog_sku, cm.catalog_title
    ORDER BY total_quantity DESC
  `;

  const result = await pool.query(query, params);

  return result.rows.map(row => ({
    productId: row.catalog_product_id,
    sku: row.catalog_sku,
    title: row.catalog_title,
    totalQuantity: parseInt(row.total_quantity),
    orderCount: parseInt(row.order_count),
    orderReferences: row.order_references || []
  }));
}

export async function getPlanLinesWithDetails(pool: Pool, planId: number) {
  const result = await pool.query(`
    SELECT 
      ppl.id,
      ppl.plan_id,
      ppl.product_id,
      ppl.quantity,
      ppl.source_type,
      ppl.source_id,
      ppl.source_reference,
      ppl.production_order_id,
      ppl.routing_id,
      ppl.routing_override,
      ppl.bom_id,
      ppl.planned_start_date,
      ppl.planned_end_date,
      ppl.status,
      ppl.sequence,
      ppl.notes,
      ppl.metadata,
      ppl.created_at,
      ppl.updated_at,
      ppl.reserved_quantity as "reservedQuantity",
      
      -- Product data from catalog
      cp.sku as product_sku,
      cp.title as product_title,
      cp.color as product_color,
      cp.color_options as product_color_options,
      cp.length as product_length,
      cp.width as product_width,
      cp.height as product_height,
      cp.product_type,
      cp.product_group,
      cp.doors,
      cp.legs,
      cp.material,
      cp.base_price,
      cp.currency,
      COALESCE(
        ppl.metadata->>'image_url',
        (
          SELECT thumbnail_url
          FROM catalog.product_images
          WHERE product_id = cp.id
          ORDER BY is_primary DESC, sort_order ASC, id ASC
          LIMIT 1
        )
      ) as product_image,
      
      -- Order data from commerce.orders (if source_type = 'order_demand')
      o.order_number,
      o.order_date,
      o.source as marketplace,
      o.buyer_first_name,
      o.buyer_last_name,
      o.buyer_email,
      o.payment_status as order_payment_status,
      o.total_to_pay_amount as order_total_amount,
      
      -- BOM component count
      (SELECT COUNT(*) 
       FROM bom.product_components pc 
       INNER JOIN bom.product_boms pb ON pc.product_bom_id = pb.id 
       WHERE pb.product_id = cp.id) as bom_count,
      
      -- BOM formatki count (only components with dimensions = formatki)
      (SELECT COUNT(*) 
       FROM bom.product_components pc 
       INNER JOIN bom.product_boms pb ON pc.product_bom_id = pb.id 
       WHERE pb.product_id = cp.id 
         AND pc.calculated_length IS NOT NULL 
         AND pc.calculated_width IS NOT NULL) as bom_formatki_count,
      
      -- Reserved formatki count for this plan line from buffer reservations
      (SELECT COUNT(*) 
       FROM production.production_buffer_reservations pbr
       WHERE pbr.zlp_item_id = ppl.id 
         AND pbr.status = 'ACTIVE') as reserved_formatki_count,
      
      -- Order item ID from metadata (for duplicate prevention)
      (ppl.metadata->>'orderItemId')::integer as order_item_id,
      
      -- Warehouse packed products data
      wpp.id as packed_product_id,
      wpp.product_sku as packed_product_sku,
      wpp.quantity as warehouse_total_qty,
      wpp.reserved_quantity as warehouse_reserved_qty
      
    FROM production.production_plan_lines ppl
    INNER JOIN catalog.products cp ON ppl.product_id = cp.id
    LEFT JOIN warehouse.packed_products wpp ON wpp.catalog_product_id = ppl.product_id
    LEFT JOIN commerce.orders o ON (
      ppl.source_type = 'order_demand' 
      AND (
        ppl.source_reference = o.order_number::text
        OR ppl.source_id = o.id
        OR (ppl.metadata->>'order_number') = o.order_number::text
      )
    )
    WHERE ppl.plan_id = $1
      AND ppl.deleted_at IS NULL
    ORDER BY ppl.sequence ASC NULLS LAST, ppl.created_at ASC
  `, [planId]);

  return result.rows;
}

export async function transferPlanLine(
  pool: Pool, 
  lineId: number, 
  targetPlanId: number
): Promise<ProductionPlanLine> {
  const client = await pool.connect();
  try {
    await client.query('BEGIN');
    
    // Sprawdź czy linia istnieje i pobierz jej dane
    const lineResult = await client.query(
      `SELECT * FROM production.production_plan_lines WHERE id = $1 AND deleted_at IS NULL`,
      [lineId]
    );
    
    if (lineResult.rows.length === 0) {
      throw new Error(`Linia planu #${lineId} nie istnieje`);
    }
    
    const line = lineResult.rows[0];
    const sourcePlanId = line.plan_id;
    
    if (sourcePlanId === targetPlanId) {
      throw new Error('Linia jest już w tym planie');
    }
    
    // Sprawdź czy plan docelowy istnieje
    const targetPlanResult = await client.query(
      `SELECT id, short_name, name FROM production.production_plans WHERE id = $1`,
      [targetPlanId]
    );
    
    if (targetPlanResult.rows.length === 0) {
      throw new Error(`Plan docelowy #${targetPlanId} nie istnieje`);
    }
    
    const targetPlan = targetPlanResult.rows[0];
    
    // Sprawdź czy ten sam produkt z tego samego zamówienia już istnieje w planie docelowym
    if (line.source_type === 'order_demand' && line.source_reference) {
      const duplicateCheck = await client.query(
        `SELECT id FROM production.production_plan_lines 
         WHERE plan_id = $1 
           AND product_id = $2 
           AND source_type = $3 
           AND source_reference = $4 
           AND deleted_at IS NULL
           AND id != $5
         LIMIT 1`,
        [targetPlanId, line.product_id, line.source_type, line.source_reference, lineId]
      );
      
      if (duplicateCheck.rows.length > 0) {
        throw new Error(`Ten produkt z zamówienia #${line.source_reference} już istnieje w planie "${targetPlan.short_name || targetPlan.name}"`);
      }
    }
    
    // Pobierz następny numer sekwencji w planie docelowym
    const seqResult = await client.query(
      `SELECT COALESCE(MAX(sequence), 0) + 1 as next_seq 
       FROM production.production_plan_lines 
       WHERE plan_id = $1 AND deleted_at IS NULL`,
      [targetPlanId]
    );
    const nextSequence = seqResult.rows[0].next_seq;
    
    // Przenieś linię do nowego planu (rezerwacje pozostają nietknięte!)
    const updateResult = await client.query(
      `UPDATE production.production_plan_lines 
       SET plan_id = $1, 
           sequence = $2,
           updated_at = CURRENT_TIMESTAMP
       WHERE id = $3
       RETURNING *`,
      [targetPlanId, nextSequence, lineId]
    );
    
    // Pobierz info o rezerwacjach (dla logowania)
    const reservationsResult = await client.query(
      `SELECT COUNT(*) as count, string_agg(serial_number, ', ') as serials
       FROM warehouse.packed_product_items 
       WHERE reserved_for_plan_line_id = $1 AND status = 'reserved'`,
      [lineId]
    );
    
    const reservations = reservationsResult.rows[0];
    
    await client.query('COMMIT');
    
    console.log(`🔄 [TRANSFER] Przeniesiono linię #${lineId} z planu #${sourcePlanId} do planu #${targetPlanId}`);
    if (parseInt(reservations.count) > 0) {
      console.log(`📦 [TRANSFER] Zachowano ${reservations.count} rezerwacji: ${reservations.serials}`);
    }
    
    return updateResult.rows[0];
  } catch (error) {
    await client.query('ROLLBACK');
    throw error;
  } finally {
    client.release();
  }
}

// ==========================================
// AUTOMATYCZNE PRZEŁĄCZANIE REZERWACJI
// ==========================================

/**
 * Zarezerwuj formatki z magazynu dla linii planu (na podstawie BOM)
 * Wywoływane gdy odrezerwowujemy produkt spakowany
 * 
 * @param poolOrClient - Pool lub PoolClient (jeśli przekazany PoolClient, działa w istniejącej transakcji)
 * @throws Error jeśli wystąpi błąd krytyczny (DB error)
 */
export async function reserveFormatkiForPlanLine(
  poolOrClient: PoolOrClient,
  lineId: number,
  userId?: number
): Promise<{ success: boolean; logs: string[] }> {
  const isClient = 'release' in poolOrClient && typeof poolOrClient.release === 'function';
  const client = isClient ? (poolOrClient as PoolClient) : await (poolOrClient as Pool).connect();
  const logs: string[] = [];
  
  try {
    if (!isClient) await client.query('BEGIN');
    
    // Pobierz informacje o linii planu
    const lineResult = await client.query(`
      SELECT ppl.id, ppl.product_id, ppl.quantity, ppl.plan_id, ppl.bom_id
      FROM production.production_plan_lines ppl
      WHERE ppl.id = $1 AND ppl.deleted_at IS NULL
    `, [lineId]);
    
    if (lineResult.rows.length === 0) {
      // Soft error - przy własnej transakcji robimy rollback, przy przekazanym kliencie caller decyduje
      if (!isClient) await client.query('ROLLBACK');
      return { success: false, logs: ['Nie znaleziono linii planu'] };
    }
    
    const line = lineResult.rows[0];
    
    // Jeśli nie ma przypisanego BOM, pobierz aktywny BOM dla produktu
    let bomId = line.bom_id;
    if (!bomId) {
      const bomResult = await client.query(`
        SELECT id FROM bom.product_boms
        WHERE product_id = $1 AND is_active = true
        ORDER BY version DESC LIMIT 1
      `, [line.product_id]);
      
      if (bomResult.rows.length === 0) {
        // Soft success - produkt nie ma BOM, to OK (np. prosty produkt bez formatek)
        if (!isClient) await client.query('COMMIT');
        logs.push('ℹ️ Produkt nie ma przypisanego BOM - pomijam rezerwację formatek');
        return { success: true, logs };
      }
      bomId = bomResult.rows[0].id;
    }
    
    // Pobierz komponenty BOM (formatki)
    const componentsResult = await client.query(`
      SELECT 
        id,
        generated_name,
        quantity,
        calculated_length,
        calculated_width,
        thickness,
        component_type,
        color
      FROM bom.product_components
      WHERE product_bom_id = $1
    `, [bomId]);
    
    for (const component of componentsResult.rows) {
      const isFormatka = component.component_type === 'panel' || 
                        component.generated_name?.includes('-') ||
                        (component.calculated_length && component.calculated_width);

      if (isFormatka && component.generated_name) {
        const nameParts = component.generated_name.split('-');
        const color = component.color || nameParts[nameParts.length - 1]?.toUpperCase() || null;
        
        const length = parseFloat(component.calculated_length) || 0;
        const width = parseFloat(component.calculated_width) || 0;
        const thickness = parseFloat(component.thickness) || null;
        
        if (length > 0 && width > 0) {
          const formatka = await findFormatkaInWarehouse(
            client,
            component.generated_name,
            length,
            width,
            thickness,
            color
          );

          if (formatka) {
            const quantityNeeded = (component.quantity || 1) * line.quantity;
            
            if (formatka.quantityAvailable >= quantityNeeded) {
              // createReservation rzuci wyjątek przy błędzie DB - propagujemy go
              await createReservation(client, {
                zlpId: null as any,
                zlpItemId: lineId,
                productSku: formatka.internalCode,
                quantityReserved: quantityNeeded.toString(),
                unitOfMeasure: formatka.unitOfMeasure,
                locationId: null,
                reservedBy: userId?.toString() || null,
                notes: `Rezerwacja dla planu #${line.plan_id}, linia #${lineId}: ${component.generated_name}`,
              });
              
              logs.push(`✅ Zarezerwowano ${quantityNeeded}x ${formatka.name}`);
            } else {
              logs.push(`⚠️ Niewystarczająca ilość ${formatka.name}: potrzeba ${quantityNeeded}, dostępne ${formatka.quantityAvailable}`);
            }
          } else {
            logs.push(`⚠️ Nie znaleziono formatki: ${component.generated_name}`);
          }
        }
      }
    }
    
    if (!isClient) await client.query('COMMIT');
    console.log(`\n📦 [AUTO-REZERWACJA FORMATEK] Linia #${lineId}:`);
    logs.forEach(log => console.log(`  ${log}`));
    
    return { success: true, logs };
  } catch (error) {
    if (!isClient) await client.query('ROLLBACK');
    console.error('Błąd rezerwacji formatek:', error);
    // Rzucamy wyjątek by główna transakcja też się rollback
    throw error;
  } finally {
    if (!isClient) client.release();
  }
}

/**
 * Zwolnij rezerwacje formatek dla linii planu
 * Wywoływane gdy rezerwujemy produkt spakowany
 * 
 * @param poolOrClient - Pool lub PoolClient (jeśli przekazany PoolClient, działa w istniejącej transakcji)
 * @throws Error jeśli wystąpi błąd krytyczny (DB error)
 */
export async function releaseFormatkiReservationsForPlanLine(
  poolOrClient: PoolOrClient,
  lineId: number
): Promise<{ success: boolean; releasedCount: number; logs: string[] }> {
  const isClient = 'release' in poolOrClient && typeof poolOrClient.release === 'function';
  const client = isClient ? (poolOrClient as PoolClient) : await (poolOrClient as Pool).connect();
  const logs: string[] = [];
  
  try {
    if (!isClient) await client.query('BEGIN');
    
    // Znajdź wszystkie aktywne rezerwacje formatek dla tej linii planu
    const reservationsResult = await client.query(`
      SELECT id, product_sku, quantity_reserved
      FROM production.production_buffer_reservations
      WHERE zlp_item_id = $1 AND status = 'ACTIVE'
    `, [lineId]);
    
    const releasedCount = reservationsResult.rows.length;
    
    if (releasedCount === 0) {
      // Soft success - brak rezerwacji do zwolnienia to OK
      if (!isClient) await client.query('COMMIT');
      logs.push('Brak rezerwacji formatek do zwolnienia');
      return { success: true, releasedCount: 0, logs };
    }
    
    // Anuluj rezerwacje - błąd DB propagujemy do głównej transakcji
    await client.query(`
      UPDATE production.production_buffer_reservations
      SET status = 'CANCELLED',
          cancelled_at = CURRENT_TIMESTAMP,
          notes = COALESCE(notes, '') || ' [Auto-anulowano - produkt spakowany zarezerwowany]'
      WHERE zlp_item_id = $1 AND status = 'ACTIVE'
    `, [lineId]);
    
    for (const res of reservationsResult.rows) {
      logs.push(`🔓 Zwolniono rezerwację: ${res.product_sku} (${res.quantity_reserved} szt.)`);
    }
    
    if (!isClient) await client.query('COMMIT');
    console.log(`\n📦 [AUTO-ZWOLNIENIE FORMATEK] Linia #${lineId}:`);
    logs.forEach(log => console.log(`  ${log}`));
    
    return { success: true, releasedCount, logs };
  } catch (error) {
    if (!isClient) await client.query('ROLLBACK');
    console.error('Błąd zwalniania rezerwacji formatek:', error);
    // Rzucamy wyjątek by główna transakcja też się rollback
    throw error;
  } finally {
    if (!isClient) client.release();
  }
}
