import { pool } from '../postgres';
import { processImage, generateImageVersionFilenames } from '../image-processor';
import { getFileStorageAdapter } from '../file-storage-adapter';
import path from 'path';

const BATCH_SIZE = 10; // Process 10 images at a time to avoid memory issues

async function generateThumbnailsForExistingImages() {
  console.log('Starting thumbnail generation for existing product images...');
  console.log('Note: This will download images from SFTP, process them, and upload thumbnails back.\n');
  
  const client = await pool.connect();

  try {
    // Get count of images that need thumbnails (only modern SFTP URLs)
    const countResult = await client.query(`
      SELECT COUNT(*) as count
      FROM catalog.product_images
      WHERE url IS NOT NULL
        AND url LIKE 'https://files.alpsys.pl/%'
        AND (thumbnail_url IS NULL OR medium_url IS NULL)
    `);

    const totalCount = parseInt(countResult.rows[0].count);
    console.log(`Found ${totalCount} images that need thumbnails`);

    if (totalCount === 0) {
      console.log('No images need thumbnail generation. All done!');
      return;
    }

    let processedCount = 0;
    let errorCount = 0;
    let skippedCount = 0;
    let offset = 0;

    const storageAdapter = await getFileStorageAdapter();
    console.log(`Using storage adapter: ${storageAdapter.constructor.name}\n`);

    while (offset < totalCount) {
      // Get a batch of images (only modern SFTP URLs)
      const result = await client.query(`
        SELECT 
          pi.id, 
          pi.product_id, 
          pi.url, 
          pi.thumbnail_url, 
          pi.medium_url,
          p.sku
        FROM catalog.product_images pi
        JOIN catalog.products p ON p.id = pi.product_id
        WHERE pi.url IS NOT NULL
          AND pi.url LIKE 'https://files.alpsys.pl/%'
          AND (pi.thumbnail_url IS NULL OR pi.medium_url IS NULL)
        ORDER BY pi.id
        LIMIT $1 OFFSET $2
      `, [BATCH_SIZE, offset]);

      const batch = result.rows;
      console.log(`\n📦 Processing batch ${Math.floor(offset / BATCH_SIZE) + 1} (${batch.length} images)...`);

      for (const image of batch) {
        try {
          console.log(`\n  🔄 Processing image ${image.id} for product ${image.sku}...`);
          console.log(`     URL: ${image.url}`);

          // Get the file buffer from SFTP
          let fileBuffer: Buffer;
          try {
            fileBuffer = await storageAdapter.getBuffer(image.url);
            console.log(`     ✓ Downloaded from SFTP (${(fileBuffer.length / 1024).toFixed(1)} KB)`);
          } catch (error) {
            console.error(`     ❌ File not found or cannot download: ${image.url}`);
            console.error(`     Error: ${error instanceof Error ? error.message : error}`);
            
            // Skip files that don't exist on SFTP
            if (error instanceof Error && error.message.includes('No such file')) {
              skippedCount++;
              console.log(`     ⏭️  Skipping (file doesn't exist on SFTP)`);
            } else {
              errorCount++;
            }
            continue;
          }

          // Get original filename
          const originalFileName = path.basename(image.url);
          const fileExtension = path.extname(originalFileName);
          const baseFileName = path.basename(originalFileName, fileExtension);

          // Generate thumbnails using image processor
          console.log(`     🖼️  Generating thumbnails...`);
          const processed = await processImage(fileBuffer, {
            thumbnailSize: 80,
            mediumSize: 400,
            quality: 85,
            format: 'webp'
          });

          // Generate filenames
          const filenames = generateImageVersionFilenames(baseFileName, processed.format);

          // Determine base upload directory based on original URL
          let baseDir = 'products/images';
          if (image.url.includes('/product-matrix-colors/')) {
            baseDir = 'product-matrix-colors';
          } else if (image.url.includes('/accessories/')) {
            baseDir = 'accessories';
          }

          // Upload thumbnails to separate subdirectories
          let thumbnailUrl: string;
          let mediumUrl: string;

          try {
            console.log(`     ⬆️  Uploading thumbnails to SFTP...`);
            [thumbnailUrl, mediumUrl] = await Promise.all([
              storageAdapter.upload({
                filename: filenames.thumbnail,
                buffer: processed.buffers.thumbnail,
                mimetype: 'image/webp',
                subfolder: `${baseDir}/thumbnails`
              }),
              storageAdapter.upload({
                filename: filenames.medium,
                buffer: processed.buffers.medium,
                mimetype: 'image/webp',
                subfolder: `${baseDir}/medium`
              })
            ]);
            console.log(`     ✓ Uploaded to SFTP`);
            console.log(`       - Thumbnails: ${baseDir}/thumbnails/${filenames.thumbnail}`);
            console.log(`       - Medium: ${baseDir}/medium/${filenames.medium}`);
          } catch (uploadError) {
            console.error(`     ❌ Failed to upload thumbnails:`, uploadError);
            errorCount++;
            continue;
          }

          // Update database - only update if uploads succeeded
          try {
            await client.query(`
              UPDATE catalog.product_images
              SET 
                thumbnail_url = $1,
                medium_url = $2
              WHERE id = $3
            `, [thumbnailUrl, mediumUrl, image.id]);

            console.log(`     ✅ Success!`);
            console.log(`        Thumbnail: ${filenames.thumbnail} (~${(processed.buffers.thumbnail.length / 1024).toFixed(1)} KB)`);
            console.log(`        Medium: ${filenames.medium} (~${(processed.buffers.medium.length / 1024).toFixed(1)} KB)`);
            processedCount++;
          } catch (dbError) {
            console.error(`     ❌ Database update failed:`, dbError);
            errorCount++;
          }
        } catch (error) {
          console.error(`  ❌ Error processing image ${image.id}:`, error);
          errorCount++;
        }
      }

      offset += BATCH_SIZE;
      
      // Show progress
      const progress = Math.min(offset, totalCount);
      const percentage = ((progress / totalCount) * 100).toFixed(1);
      console.log(`\n📊 Progress: ${progress}/${totalCount} (${percentage}%) | Success: ${processedCount} | Errors: ${errorCount} | Skipped: ${skippedCount}`);
    }

    console.log('\n' + '='.repeat(60));
    console.log('FINAL SUMMARY');
    console.log('='.repeat(60));
    console.log(`Total images needing thumbnails: ${totalCount}`);
    console.log(`✅ Successfully processed: ${processedCount}`);
    console.log(`⏭️  Skipped (file not found): ${skippedCount}`);
    console.log(`❌ Errors: ${errorCount}`);
    console.log('='.repeat(60));
  } catch (error) {
    console.error('\n💥 Fatal error:', error);
    process.exit(1);
  } finally {
    client.release();
  }
}

// Run the migration
generateThumbnailsForExistingImages()
  .then(() => {
    console.log('\n✅ Migration completed successfully');
    process.exit(0);
  })
  .catch((error) => {
    console.error('\n❌ Migration failed:', error);
    process.exit(1);
  });
