import { Router } from "express";
import { pool } from "../../postgres";
import {
  getProductionNormGroups,
  getProductionNormGroupById,
  createProductionNormGroup,
  updateProductionNormGroup,
  deleteProductionNormGroup,
  getProductionNorms,
  getProductionNormById,
  createProductionNorm,
  updateProductionNorm,
  deleteProductionNorm,
  getMaterialConsumptionNorms,
  getMaterialConsumptionNormById,
  createMaterialConsumptionNorm,
  updateMaterialConsumptionNorm,
  deleteMaterialConsumptionNorm,
} from "../../services/production/norms";
import { 
  insertProductionNormGroupSchema,
  insertProductionNormSchema,
  insertProductionMaterialConsumptionNormSchema,
} from "@shared/schema";

export const normsRouter = Router();

// ===== Norm Groups Endpoints =====

// GET /api/production/norms/groups - Get all norm groups
normsRouter.get("/groups", async (req, res) => {
  try {
    const groups = await getProductionNormGroups(pool);
    res.json(groups);
  } catch (error: any) {
    console.error("Error fetching norm groups:", error);
    res.status(500).json({ error: "Failed to fetch norm groups" });
  }
});

// GET /api/production/norms/groups/:id - Get single norm group
normsRouter.get("/groups/:id", async (req, res) => {
  try {
    const id = parseInt(req.params.id);
    const group = await getProductionNormGroupById(pool, id);
    
    if (!group) {
      return res.status(404).json({ error: "Norm group not found" });
    }
    
    res.json(group);
  } catch (error: any) {
    console.error("Error fetching norm group:", error);
    res.status(500).json({ error: "Failed to fetch norm group" });
  }
});

// POST /api/production/norms/groups - Create norm group
normsRouter.post("/groups", async (req, res) => {
  try {
    const parsed = insertProductionNormGroupSchema.safeParse(req.body);
    
    if (!parsed.success) {
      return res.status(400).json({ 
        error: "Invalid request data", 
        details: parsed.error.errors 
      });
    }
    
    const group = await createProductionNormGroup(pool, parsed.data);
    res.status(201).json(group);
  } catch (error: any) {
    console.error("Error creating norm group:", error);
    res.status(500).json({ error: "Failed to create norm group" });
  }
});

// PATCH /api/production/norms/groups/:id - Update norm group
normsRouter.patch("/groups/:id", async (req, res) => {
  try {
    const id = parseInt(req.params.id);
    
    // Validate partial update with Zod
    const partialSchema = insertProductionNormGroupSchema.partial();
    const parsed = partialSchema.safeParse(req.body);
    
    if (!parsed.success) {
      return res.status(400).json({ 
        error: "Invalid request data", 
        details: parsed.error.errors 
      });
    }
    
    // Reject empty updates
    if (Object.keys(parsed.data).length === 0) {
      return res.status(400).json({ error: "No fields to update" });
    }
    
    const group = await updateProductionNormGroup(pool, id, parsed.data);
    res.json(group);
  } catch (error: any) {
    console.error("Error updating norm group:", error);
    
    if (error.message.includes("not found")) {
      return res.status(404).json({ error: error.message });
    }
    
    res.status(500).json({ error: "Failed to update norm group" });
  }
});

// DELETE /api/production/norms/groups/:id - Delete norm group
normsRouter.delete("/groups/:id", async (req, res) => {
  try {
    const id = parseInt(req.params.id);
    await deleteProductionNormGroup(pool, id);
    res.status(204).send();
  } catch (error: any) {
    console.error("Error deleting norm group:", error);
    res.status(500).json({ error: "Failed to delete norm group" });
  }
});

// ===== Norms Endpoints =====

// GET /api/production/norms - Get all norms (optionally filtered by group)
normsRouter.get("/", async (req, res) => {
  try {
    const groupId = req.query.groupId ? parseInt(req.query.groupId as string) : undefined;
    const norms = await getProductionNorms(pool, groupId);
    res.json(norms);
  } catch (error: any) {
    console.error("Error fetching norms:", error);
    res.status(500).json({ error: "Failed to fetch norms" });
  }
});

// GET /api/production/norms/:id - Get single norm
normsRouter.get("/:id", async (req, res) => {
  try {
    const id = parseInt(req.params.id);
    const norm = await getProductionNormById(pool, id);
    
    if (!norm) {
      return res.status(404).json({ error: "Norm not found" });
    }
    
    res.json(norm);
  } catch (error: any) {
    console.error("Error fetching norm:", error);
    res.status(500).json({ error: "Failed to fetch norm" });
  }
});

// POST /api/production/norms - Create norm
normsRouter.post("/", async (req, res) => {
  try {
    const parsed = insertProductionNormSchema.safeParse(req.body);
    
    if (!parsed.success) {
      return res.status(400).json({ 
        error: "Invalid request data", 
        details: parsed.error.errors 
      });
    }
    
    const norm = await createProductionNorm(pool, parsed.data);
    res.status(201).json(norm);
  } catch (error: any) {
    console.error("Error creating norm:", error);
    res.status(500).json({ error: "Failed to create norm" });
  }
});

// PATCH /api/production/norms/:id - Update norm
normsRouter.patch("/:id", async (req, res) => {
  try {
    const id = parseInt(req.params.id);
    
    // Validate partial update with Zod
    const partialSchema = insertProductionNormSchema.partial();
    const parsed = partialSchema.safeParse(req.body);
    
    if (!parsed.success) {
      return res.status(400).json({ 
        error: "Invalid request data", 
        details: parsed.error.errors 
      });
    }
    
    // Reject empty updates
    if (Object.keys(parsed.data).length === 0) {
      return res.status(400).json({ error: "No fields to update" });
    }
    
    const norm = await updateProductionNorm(pool, id, parsed.data);
    res.json(norm);
  } catch (error: any) {
    console.error("Error updating norm:", error);
    
    if (error.message.includes("not found")) {
      return res.status(404).json({ error: error.message });
    }
    
    res.status(500).json({ error: "Failed to update norm" });
  }
});

// DELETE /api/production/norms/:id - Delete norm
normsRouter.delete("/:id", async (req, res) => {
  try {
    const id = parseInt(req.params.id);
    await deleteProductionNorm(pool, id);
    res.status(204).send();
  } catch (error: any) {
    console.error("Error deleting norm:", error);
    res.status(500).json({ error: "Failed to delete norm" });
  }
});

// ===== Material Consumption Norms Endpoints =====

// GET /api/production/norms/materials - Get all material consumption norms
normsRouter.get("/materials", async (req, res) => {
  try {
    const norms = await getMaterialConsumptionNorms(pool);
    res.json(norms);
  } catch (error: any) {
    console.error("Error fetching material consumption norms:", error);
    res.status(500).json({ error: "Failed to fetch material consumption norms" });
  }
});

// GET /api/production/norms/materials/:id - Get single material consumption norm
normsRouter.get("/materials/:id", async (req, res) => {
  try {
    const id = parseInt(req.params.id);
    const norm = await getMaterialConsumptionNormById(pool, id);
    
    if (!norm) {
      return res.status(404).json({ error: "Material consumption norm not found" });
    }
    
    res.json(norm);
  } catch (error: any) {
    console.error("Error fetching material consumption norm:", error);
    res.status(500).json({ error: "Failed to fetch material consumption norm" });
  }
});

// POST /api/production/norms/materials - Create material consumption norm
normsRouter.post("/materials", async (req, res) => {
  try {
    const parsed = insertProductionMaterialConsumptionNormSchema.safeParse(req.body);
    
    if (!parsed.success) {
      return res.status(400).json({ 
        error: "Invalid request data", 
        details: parsed.error.errors 
      });
    }
    
    const norm = await createMaterialConsumptionNorm(pool, parsed.data);
    res.status(201).json(norm);
  } catch (error: any) {
    console.error("Error creating material consumption norm:", error);
    res.status(500).json({ error: "Failed to create material consumption norm" });
  }
});

// PATCH /api/production/norms/materials/:id - Update material consumption norm
normsRouter.patch("/materials/:id", async (req, res) => {
  try {
    const id = parseInt(req.params.id);
    
    // Validate partial update with Zod
    const partialSchema = insertProductionMaterialConsumptionNormSchema.partial();
    const parsed = partialSchema.safeParse(req.body);
    
    if (!parsed.success) {
      return res.status(400).json({ 
        error: "Invalid request data", 
        details: parsed.error.errors 
      });
    }
    
    // Reject empty updates
    if (Object.keys(parsed.data).length === 0) {
      return res.status(400).json({ error: "No fields to update" });
    }
    
    const norm = await updateMaterialConsumptionNorm(pool, id, parsed.data);
    res.json(norm);
  } catch (error: any) {
    console.error("Error updating material consumption norm:", error);
    
    if (error.message.includes("not found")) {
      return res.status(404).json({ error: error.message });
    }
    
    res.status(500).json({ error: "Failed to update material consumption norm" });
  }
});

// DELETE /api/production/norms/materials/:id - Delete material consumption norm
normsRouter.delete("/materials/:id", async (req, res) => {
  try {
    const id = parseInt(req.params.id);
    await deleteMaterialConsumptionNorm(pool, id);
    res.status(204).send();
  } catch (error: any) {
    console.error("Error deleting material consumption norm:", error);
    res.status(500).json({ error: "Failed to delete material consumption norm" });
  }
});
