import type { Express } from "express";
import { pool } from "../../postgres";
import { renderFormatkaVisualization } from "../../formatka-visualizer";
import { requirePermission } from "../../auth";
import { 
  calculateMaterialBreakdown,
  saveComponentMaterialBreakdown,
  getComponentMaterialBreakdown 
} from "../../services/bom/material-breakdown";

interface ComponentForGeneration {
  id: number;
  generatedName: string;
  color: string | null;
  calculatedLength: string | null;
  calculatedWidth: string | null;
  thickness: string | null;
  edgingPattern: string | null;
  edgingMaterial: string | null;
}

async function getColorHexFromDictionary(colorCode: string): Promise<string | null> {
  if (!colorCode) return null;
  
  try {
    const result = await pool.query(`
      SELECT color FROM product_creator.dictionaries 
      WHERE dictionary_type = 'color' AND code = $1
    `, [colorCode]);
    
    if (result.rows.length > 0) {
      return result.rows[0].color;
    }
    
    console.warn(`⚠️  Color code "${colorCode}" not found in dictionaries`);
    return null;
  } catch (error) {
    console.error(`❌ Error fetching color hex for "${colorCode}":`, error);
    return null;
  }
}

function extractEdgingColorCode(edgingMaterial: string | null): string | null {
  if (!edgingMaterial) return null;
  
  const parts = edgingMaterial.split('_');
  if (parts.length >= 2) {
    return parts[parts.length - 1];
  }
  
  return edgingMaterial;
}

function parseEdgingPattern(pattern: string | null): { edge1: boolean; edge2: boolean; edge3: boolean; edge4: boolean } {
  if (!pattern || pattern.length !== 4) {
    return { edge1: false, edge2: false, edge3: false, edge4: false };
  }
  
  return {
    edge1: pattern[0] === 'T',
    edge2: pattern[1] === 'T',
    edge3: pattern[2] === 'T',
    edge4: pattern[3] === 'T',
  };
}

export function registerBomRoutes(app: Express) {
  // POST /api/production/boms/:id/formatkas - Generate formatka visualizations for all components
  app.post("/api/production/boms/:id/formatkas", requirePermission('manage_production'), async (req, res) => {
    try {
      const bomId = parseInt(req.params.id);
      if (isNaN(bomId)) {
        return res.status(400).json({ message: "Invalid BOM ID" });
      }

      // Fetch all components for this BOM
      const componentsResult = await pool.query<ComponentForGeneration>(`
        SELECT 
          id, 
          generated_name,
          color,
          calculated_length,
          calculated_width,
          thickness,
          edging_pattern,
          edging_material
        FROM bom.product_components
        WHERE product_bom_id = (
          SELECT id FROM bom.product_boms WHERE id = $1
        )
        ORDER BY position_in_bom
      `, [bomId]);

      if (componentsResult.rows.length === 0) {
        return res.status(404).json({ message: "No components found for this BOM" });
      }

      const results = [];
      let successCount = 0;
      let skippedCount = 0;
      let errorCount = 0;

      console.log(`\n🎨 Starting formatka generation for ${componentsResult.rows.length} components in BOM ${bomId}\n`);

      for (const component of componentsResult.rows) {
        try {
          // Skip if missing critical dimensions
          if (!component.calculatedLength || !component.calculatedWidth) {
            console.log(`⏭️  Skipping ${component.generatedName} - missing dimensions`);
            skippedCount++;
            results.push({
              componentId: component.id,
              name: component.generatedName,
              status: 'skipped',
              reason: 'Missing dimensions'
            });
            continue;
          }

          // Get color hex from dictionary
          const colorHex = component.color 
            ? await getColorHexFromDictionary(component.color)
            : null;

          // Get edging color hex
          const edgingColorCode = extractEdgingColorCode(component.edgingMaterial);
          const edgingColorHex = edgingColorCode 
            ? await getColorHexFromDictionary(edgingColorCode)
            : null;

          // Parse edging pattern
          const edgingFlags = parseEdgingPattern(component.edgingPattern);

          // Generate formatka
          const visualizationUrl = await renderFormatkaVisualization({
            name: component.generatedName,
            length: parseFloat(component.calculatedLength),
            width: parseFloat(component.calculatedWidth),
            thickness: component.thickness ? parseFloat(component.thickness) : 18,
            color: component.color || 'UNKNOWN',
            colorHex: colorHex || undefined,
            edgingColorHex: edgingColorHex || undefined,
            edgingPattern: component.edgingPattern || undefined,
            componentType: 'board',
            ...edgingFlags,
          });

          // Update visualization_url in database
          await pool.query(`
            UPDATE bom.product_components
            SET visualization_url = $1
            WHERE id = $2
          `, [visualizationUrl, component.id]);

          console.log(`✅ Generated: ${component.generatedName} -> ${visualizationUrl}`);
          successCount++;

          results.push({
            componentId: component.id,
            name: component.generatedName,
            status: 'success',
            visualizationUrl,
          });

        } catch (error) {
          console.error(`❌ Error generating formatka for ${component.generatedName}:`, error);
          errorCount++;
          results.push({
            componentId: component.id,
            name: component.generatedName,
            status: 'error',
            error: error instanceof Error ? error.message : 'Unknown error',
          });
        }
      }

      console.log(`\n📊 Formatka generation summary for BOM ${bomId}:`);
      console.log(`   ✅ Success: ${successCount}`);
      console.log(`   ⏭️  Skipped: ${skippedCount}`);
      console.log(`   ❌ Errors: ${errorCount}\n`);

      res.json({
        bomId,
        totalComponents: componentsResult.rows.length,
        successCount,
        skippedCount,
        errorCount,
        results,
      });

    } catch (error) {
      console.error("Error generating formatkas:", error);
      res.status(500).json({ 
        message: "Failed to generate formatkas",
        error: error instanceof Error ? error.message : 'Unknown error'
      });
    }
  });

  // Material Breakdown Endpoints
  
  /**
   * POST /api/production/boms/material-breakdown/calculate
   * Oblicza breakdown dla podanych wymiarów (bez zapisywania)
   */
  app.post("/api/production/boms/material-breakdown/calculate", requirePermission('view_production'), async (req, res) => {
    try {
      const { width, height, thickness, edgingPattern } = req.body;
      
      if (!width || !height) {
        return res.status(400).json({ error: 'Width and height are required' });
      }
      
      const breakdown = await calculateMaterialBreakdown(pool, {
        componentId: 0, // temporary
        width: parseFloat(width),
        height: parseFloat(height),
        thickness: parseFloat(thickness || '18'),
        edgingPattern,
      });
      
      res.json(breakdown);
    } catch (error) {
      console.error('Error calculating material breakdown:', error);
      res.status(500).json({ error: 'Failed to calculate material breakdown' });
    }
  });

  /**
   * POST /api/production/boms/components/:id/material-breakdown/save
   * Oblicza i zapisuje breakdown dla komponentu
   */
  app.post("/api/production/boms/components/:id/material-breakdown/save", requirePermission('manage_production'), async (req, res) => {
    try {
      const componentId = parseInt(req.params.id);
      
      if (isNaN(componentId)) {
        return res.status(400).json({ error: 'Invalid component ID' });
      }
      
      const breakdownId = await saveComponentMaterialBreakdown(pool, componentId);
      const breakdown = await getComponentMaterialBreakdown(pool, componentId);
      
      res.json({ 
        message: 'Material breakdown saved successfully',
        breakdownId,
        breakdown 
      });
    } catch (error) {
      console.error('Error saving component material breakdown:', error);
      res.status(500).json({ 
        error: 'Failed to save component material breakdown',
        details: error instanceof Error ? error.message : 'Unknown error'
      });
    }
  });

  /**
   * GET /api/production/boms/components/:id/material-breakdown
   * Pobiera breakdown dla komponentu
   */
  app.get("/api/production/boms/components/:id/material-breakdown", requirePermission('view_production'), async (req, res) => {
    try {
      const componentId = parseInt(req.params.id);
      
      if (isNaN(componentId)) {
        return res.status(400).json({ error: 'Invalid component ID' });
      }
      
      const breakdown = await getComponentMaterialBreakdown(pool, componentId);
      
      if (!breakdown) {
        return res.status(404).json({ error: 'Material breakdown not found for this component' });
      }
      
      res.json(breakdown);
    } catch (error) {
      console.error('Error fetching component material breakdown:', error);
      res.status(500).json({ error: 'Failed to fetch component material breakdown' });
    }
  });

  /**
   * POST /api/production/boms/components/:id/material-breakdown/recalculate
   * Przelicza breakdown dla komponentu (używaj gdy zmieniły się wymiary lub ceny)
   */
  app.post("/api/production/boms/components/:id/material-breakdown/recalculate", requirePermission('manage_production'), async (req, res) => {
    try {
      const componentId = parseInt(req.params.id);
      
      if (isNaN(componentId)) {
        return res.status(400).json({ error: 'Invalid component ID' });
      }
      
      // Recalculate and save
      const breakdownId = await saveComponentMaterialBreakdown(pool, componentId);
      const breakdown = await getComponentMaterialBreakdown(pool, componentId);
      
      res.json({ 
        message: 'Material breakdown recalculated successfully',
        breakdownId,
        breakdown 
      });
    } catch (error) {
      console.error('Error recalculating component material breakdown:', error);
      res.status(500).json({ 
        error: 'Failed to recalculate component material breakdown',
        details: error instanceof Error ? error.message : 'Unknown error'
      });
    }
  });
}
