import { Pool } from 'pg';
import { triggerOrderWebhooks } from './webhooks.js';
import { mapOrderStatus } from './status-mapper.js';
import { getAllegroOfferDetails } from './allegro-api.js';
import { getShoperProductDetails } from './shoper-api.js';
import path from 'path';
import { promises as fs } from 'fs';

export const pool = new Pool({
  connectionString: process.env.DATABASE_URL,
  max: 10, // Maksymalna liczba połączeń w pool - zalecane przez Replit
  connectionTimeoutMillis: 60000, // 60 sekund - Replit PostgreSQL może wymagać >10s dla pierwszego TLS handshake
  idleTimeoutMillis: 30000, // 30 sekund - timeout dla bezczynnych połączeń
  query_timeout: 30000, // 30 sekund - dla długich zapytań (order matching, change tracking)
  keepAlive: true, // Utrzymuj połączenia przy życiu
});

// Helper function for retrying pool connections with exponential backoff
async function connectWithRetry(maxAttempts = 2, baseDelayMs = 3000) {
  for (let attempt = 1; attempt <= maxAttempts; attempt++) {
    try {
      const client = await pool.connect();
      return client;
    } catch (error) {
      if (attempt === maxAttempts) {
        throw error;
      }
      const delay = baseDelayMs * attempt; // Linear backoff: 3s, 6s
      console.log(`⚠️  Connection attempt ${attempt}/${maxAttempts} failed, retrying in ${delay}ms...`);
      await new Promise(resolve => setTimeout(resolve, delay));
    }
  }
  throw new Error('Failed to connect after maximum attempts');
}

export async function saveOrderToPostgres(allegroOrder: any, shipments: any[] = [], customerReturns: any[] = [], accessToken?: string): Promise<{ isNew: boolean }> {
  const client = await pool.connect();
  
  try {
    await client.query('BEGIN');

    const buyer = allegroOrder.buyer || {};
    const delivery = allegroOrder.delivery || {};
    const payment = allegroOrder.payment || {};
    const lineItems = allegroOrder.lineItems || [];

    const existsResult = await client.query(
      'SELECT id FROM allegro.orders WHERE order_id = $1',
      [allegroOrder.id]
    );
    const isNew = existsResult.rows.length === 0;
    const shouldFetchImages = true; // TYMCZASOWO: Pobieraj zdjęcia dla wszystkich zamówień

    const invoice = allegroOrder.invoice || {};
    const invoiceAddress = invoice.address || {};

    // Pobierz zwroty płatności z API /payments/refunds
    let refundAmount = 0;
    let refundDate: string | null = null;
    const refundReconciliation: any[] = [];
    
    if (accessToken) {
      const { fetchPaymentRefunds } = await import('./allegro-api.js');
      const refunds = await fetchPaymentRefunds(allegroOrder.id, accessToken);
      
      for (const refund of refunds) {
        const amount = parseFloat(refund.totalValue?.amount || '0');
        refundAmount += amount;
        refundReconciliation.push(refund);
        
        // Zapisz datę ostatniego zwrotu
        if (refund.createdAt) {
          if (!refundDate || new Date(refund.createdAt) > new Date(refundDate)) {
            refundDate = refund.createdAt;
          }
        }
      }
    }

    // Sprawdź czy są zwroty produktów (type: QUANTITY) w refund_reconciliation
    const hasReturnsInReconciliation = refundReconciliation.some(refund => 
      refund.lineItems && refund.lineItems.some((item: any) => item.type === 'QUANTITY')
    );
    const hasReturns = hasReturnsInReconciliation;

    // Mapuj status Allegro na polski
    const allegroMappedStatus = await mapOrderStatus(allegroOrder.status, 'ALLEGRO');

    await client.query(`
      INSERT INTO allegro.orders (
        order_id, seller_id, seller_login, order_date, seller_status,
        marketplace, buyer_id, buyer_login, buyer_email, buyer_company,
        buyer_name, buyer_phone, buyer_address, buyer_zip, buyer_city,
        buyer_country_code, payment_id, payment_status, payment_provider,
        allegro_pay, payment_amount, payment_currency, payment_type,
        invoice_required, delivery_method, delivery_amount, delivery_currency,
        total_to_pay_amount, total_to_pay_currency, buyer_notes, smart,
        tracking_numbers, payment_last_date, shipments, 
        billing_address,
        refund_amount, refund_date, refund_reconciliation, has_returns,
        imported_at
      ) VALUES (
        $1, $2, $3, $4, $5, $6, $7, $8, $9, $10,
        $11, $12, $13, $14, $15, $16, $17, $18, $19,
        $20, $21, $22, $23, $24, $25, $26, $27, $28, $29, $30, $31, $32, $33, $34,
        $35, $36, $37, $38, $39, NOW()
      )
      ON CONFLICT (order_id) DO UPDATE SET
        seller_status = EXCLUDED.seller_status,
        payment_status = EXCLUDED.payment_status,
        payment_amount = EXCLUDED.payment_amount,
        payment_type = EXCLUDED.payment_type,
        invoice_required = EXCLUDED.invoice_required,
        payment_last_date = EXCLUDED.payment_last_date,
        tracking_numbers = EXCLUDED.tracking_numbers,
        shipments = EXCLUDED.shipments,
        buyer_notes = EXCLUDED.buyer_notes,
        delivery_amount = EXCLUDED.delivery_amount,
        buyer_address = EXCLUDED.buyer_address,
        buyer_zip = EXCLUDED.buyer_zip,
        buyer_city = EXCLUDED.buyer_city,
        billing_address = EXCLUDED.billing_address,
        refund_amount = EXCLUDED.refund_amount,
        refund_date = EXCLUDED.refund_date,
        refund_reconciliation = EXCLUDED.refund_reconciliation,
        has_returns = EXCLUDED.has_returns,
        imported_at = EXCLUDED.imported_at
    `, [
      allegroOrder.id,
      allegroOrder.seller?.id,
      allegroOrder.seller?.login,
      allegroOrder.boughtAt || allegroOrder.updatedAt,
      allegroMappedStatus,
      'allegro-pl',
      buyer.id,
      buyer.login,
      buyer.email,
      buyer.companyName,
      buyer.firstName && buyer.lastName ? `${buyer.firstName} ${buyer.lastName}` : buyer.login,
      buyer.phoneNumber,
      delivery.address?.street,
      delivery.address?.zipCode,
      delivery.address?.city,
      delivery.address?.countryCode,
      payment.id,
      // Dokładne obliczanie statusu płatności
      (() => {
        const paidAmount = parseFloat(payment.paidAmount?.amount || '0');
        const totalToPay = parseFloat(allegroOrder.summary?.totalToPay?.amount || '0');
        const hasFinishedAt = !!(payment.finishedAt);
        
        if (paidAmount >= totalToPay - 0.01 && totalToPay > 0) {
          return 'PAID';
        } else if (paidAmount > 0) {
          return 'PARTIAL';
        } else if (hasFinishedAt && paidAmount === 0) {
          return 'PAID';
        } else {
          return 'UNPAID';
        }
      })(),
      payment.provider,
      payment.type === 'ONLINE' ? true : false,
      payment.paidAmount?.amount,
      payment.paidAmount?.currency,
      payment.type || null,
      invoice.required === true,
      delivery.method?.name,
      delivery.cost?.amount,
      delivery.cost?.currency,
      allegroOrder.summary?.totalToPay?.amount,
      allegroOrder.summary?.totalToPay?.currency,
      allegroOrder.messageToSeller,
      delivery.smart === true,
      delivery.shipments && delivery.shipments.length > 0 
        ? delivery.shipments.map((s: any) => s.carrierId).join('|') 
        : null,
      payment.finishedAt,
      JSON.stringify(shipments),
      invoiceAddress.street || null,
      refundAmount || 0,
      refundDate || null,
      JSON.stringify(refundReconciliation),
      hasReturns
    ]);

    // Najpierw wstaw wszystkie order_items
    for (const item of lineItems) {
      // Pobierz szczegóły oferty z API aby uzyskać URL zdjęcia (tylko dla nowych zamówień)
      let imageUrl = null;
      const offerId = item.offer?.id;
      
      if (offerId && accessToken && shouldFetchImages) {
        try {
          const offerDetails = await getAllegroOfferDetails(offerId, accessToken);
          if (offerDetails?.images && offerDetails.images.length > 0) {
            imageUrl = offerDetails.images[0].url;
          }
        } catch (error) {
          console.error(`⚠️ Failed to fetch image for offer ${offerId}:`, error);
        }
      }
      
      // Use external.id if available, otherwise use offer.id as fallback
      const externalId = item.offer?.external?.id || item.offer?.id;
      let productId = null;
      
      if (externalId) {
        const productResult = await client.query(`
          INSERT INTO allegro.products (
            product_id, external_id, name, description, image_url, updated_at
          ) VALUES (
            $1, $2, $3, $4, $5, NOW()
          )
          ON CONFLICT (external_id) DO UPDATE SET
            name = EXCLUDED.name,
            description = EXCLUDED.description,
            image_url = COALESCE(EXCLUDED.image_url, allegro.products.image_url),
            updated_at = NOW()
          RETURNING id
        `, [
          item.offer?.id,
          externalId,
          item.offer?.name,
          item.offer?.description || null,
          imageUrl
        ]);
        
        productId = productResult.rows[0]?.id;
      }
      
      await client.query(`
        INSERT INTO allegro.order_items (
          order_id, line_item_id, offer_id, offer_external_id, name,
          quantity, price, currency, tax_rate, tax_subject,
          sending_country, returns_quantity, image_url, product_id, imported_at
        ) VALUES (
          $1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, NOW()
        )
        ON CONFLICT (line_item_id) DO UPDATE SET
          price = EXCLUDED.price,
          quantity = EXCLUDED.quantity,
          returns_quantity = EXCLUDED.returns_quantity,
          image_url = EXCLUDED.image_url,
          product_id = EXCLUDED.product_id,
          imported_at = EXCLUDED.imported_at
      `, [
        allegroOrder.id,
        item.id,
        item.offer?.id,
        externalId,
        item.offer?.name,
        item.quantity,
        item.price?.amount,
        item.price?.currency,
        item.reconciliation?.tax?.percentage,
        item.reconciliation?.tax?.subjectType,
        'PL',
        0,
        imageUrl,
        productId
      ]);
    }

    // Aktualizuj returns_quantity na podstawie refund_reconciliation (bardziej stabilne dane)
    // ID w refund_reconciliation.lineItems to checkoutFormLineItemId (line_item_id)
    if (refundReconciliation && Array.isArray(refundReconciliation)) {
      for (const refund of refundReconciliation) {
        const refundLineItems = refund.lineItems || [];
        for (const refundLineItem of refundLineItems) {
          if (refundLineItem.id && refundLineItem.quantity) {
            // ID w refund_reconciliation to line_item_id (checkoutFormLineItemId)
            await client.query(`
              UPDATE allegro.order_items
              SET returns_quantity = $1
              WHERE order_id = $2
                AND line_item_id = $3
            `, [refundLineItem.quantity, allegroOrder.id, refundLineItem.id]);
          }
        }
      }
    }

    await client.query('COMMIT');
    console.log(`✅ Saved order ${allegroOrder.id} to PostgreSQL (${isNew ? 'NEW' : 'UPDATED'})`);
    
    return { isNew };
  } catch (error) {
    await client.query('ROLLBACK');
    console.error(`❌ Error saving order to PostgreSQL:`, error);
    throw error;
  } finally {
    client.release();
  }
}

export async function saveShoperOrderToPostgres(shoperOrder: any, shipments: any[] = []): Promise<{ isNew: boolean }> {
  const client = await pool.connect();
  
  try {
    await client.query('BEGIN');

    const orderId = shoperOrder.order_id || shoperOrder.id;
    const billingAddress = shoperOrder.billing_address || {};
    const shippingAddress = shoperOrder.shipping_address || {};
    const products = shoperOrder.products_data || [];

    const existsResult = await client.query(
      'SELECT id FROM shoper.orders WHERE shoper_order_id = $1',
      [orderId]
    );
    const isNew = existsResult.rows.length === 0;
    const shouldFetchImages = true; // TYMCZASOWO: Pobieraj zdjęcia dla wszystkich zamówień

    const buyerLogin = billingAddress.firstname && billingAddress.lastname 
      ? `${billingAddress.firstname} ${billingAddress.lastname}` 
      : shoperOrder.user_login || 'Unknown';

    // Shoper API zwraca daty w czasie lokalnym Polski (bez strefy czasowej)
    // Dodaj strefę czasową aby PostgreSQL poprawnie zinterpretował datę
    const convertShoperDate = (dateStr: string | null) => {
      if (!dateStr || dateStr === '0000-00-00 00:00:00') return null;
      // Dodaj strefę czasową Europe/Warsaw - PostgreSQL sam przekonwertuje na UTC
      return dateStr + ' Europe/Warsaw';
    };

    const orderDateUTC = convertShoperDate(shoperOrder.date || shoperOrder.confirm_date);
    const paidDateUTC = convertShoperDate(shoperOrder.paid_date);

    await client.query(`
      INSERT INTO shoper.orders (
        id, shoper_order_id, buyer_login, buyer_email, total_amount, currency,
        payment_status, fulfillment_status, items_count, order_date, 
        payment_date, shipments, raw_data, created_at, updated_at
      ) VALUES (
        gen_random_uuid(), $1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, NOW(), NOW()
      )
      ON CONFLICT (shoper_order_id) DO UPDATE SET
        payment_status = EXCLUDED.payment_status,
        fulfillment_status = EXCLUDED.fulfillment_status,
        total_amount = EXCLUDED.total_amount,
        payment_date = EXCLUDED.payment_date,
        shipments = EXCLUDED.shipments,
        raw_data = EXCLUDED.raw_data,
        updated_at = NOW()
    `, [
      orderId,
      buyerLogin,
      billingAddress.email || shoperOrder.email,
      shoperOrder.sum,
      shoperOrder.currency || 'PLN',
      shoperOrder.paid_date ? 'PAID' : 'PENDING',
      shoperOrder.status_id || 'NEW',
      products.length.toString(),
      orderDateUTC,
      paidDateUTC,
      JSON.stringify(shipments),
      JSON.stringify(shoperOrder)
    ]);

    for (const product of products) {
      // Pobierz szczegóły produktu z API aby uzyskać URL zdjęcia
      let imageUrl = product.main_image || null;
      const productId = product.product_id || product.id;
      
      if (productId && !imageUrl && shouldFetchImages) {
        try {
          const productDetails = await getShoperProductDetails(productId);
          console.log(`🔍 Shoper product ${productId} details:`, JSON.stringify(productDetails, null, 2));
          
          if (productDetails?.main_image) {
            const mainImage = productDetails.main_image;
            if (mainImage.unic_name && mainImage.extension) {
              imageUrl = `https://alpmeb.pl/userdata/gfx/${mainImage.unic_name}.${mainImage.extension}`;
              console.log(`✅ Built image URL from main_image: ${imageUrl}`);
            } else {
              console.log(`⚠️ Missing unic_name or extension in main_image for product ${productId}`);
            }
          } else {
            console.log(`⚠️ No main_image found in product ${productId} details`);
          }
        } catch (error) {
          console.error(`⚠️ Failed to fetch image for Shoper product ${productId}:`, error);
        }
      }
      
      // Determine payment type: if paid, it's ONLINE; if unpaid and method includes POBRANIE, it's COD
      const isPaid = shoperOrder.paid_date || parseFloat(shoperOrder.paid || '0') > 0;
      const paymentType = (!isPaid && shoperOrder.payment_method_name?.toUpperCase().includes('POBRANIE'))
        ? 'CASH_ON_DELIVERY' 
        : 'ONLINE';
      
      await client.query(`
        INSERT INTO shoper.order_items (
          id, order_id, shoper_product_id, product_name, quantity,
          unit_price, total_price, image_url, payment_type, invoice_required,
          raw_data, created_at
        ) VALUES (
          gen_random_uuid(), 
          (SELECT id FROM shoper.orders WHERE shoper_order_id = $1),
          $2, $3, $4, $5, $6, $7, $8, $9, $10, NOW()
        )
        ON CONFLICT (order_id, shoper_product_id) DO UPDATE SET
          product_name = EXCLUDED.product_name,
          quantity = EXCLUDED.quantity,
          unit_price = EXCLUDED.unit_price,
          total_price = EXCLUDED.total_price,
          image_url = EXCLUDED.image_url,
          payment_type = EXCLUDED.payment_type,
          invoice_required = EXCLUDED.invoice_required,
          raw_data = EXCLUDED.raw_data
      `, [
        orderId,
        product.product_id || product.id,
        product.name || product.translation?.name,
        product.quantity || 1,
        product.price || 0,
        (product.price || 0) * (product.quantity || 1),
        imageUrl,
        paymentType,
        shoperOrder.invoice === '1' || shoperOrder.invoice === true,
        JSON.stringify(product)
      ]);
    }

    await client.query('COMMIT');
    console.log(`✅ Saved Shoper order ${orderId} to shoper.orders (${isNew ? 'NEW' : 'UPDATED'})`);
    
    return { isNew };
  } catch (error) {
    await client.query('ROLLBACK');
    console.error(`❌ Error saving Shoper order to shoper.orders:`, error);
    throw error;
  } finally {
    client.release();
  }
}

async function buildOrderFromRow(orderRow: any, items: any[]) {
  const rawPayload = orderRow.raw_payload || {};
  const billingAddress = orderRow.billing_address || {};
  const deliveryAddress = orderRow.delivery_address || {};
  
  // Convert local image URLs to SFTP URLs if needed
  const { getFileStorageAdapter } = await import('./file-storage-adapter.js');
  const adapter = await getFileStorageAdapter();
  
  const convertImageUrl = (url: string | null): string | null => {
    if (!url) return null;
    // If URL is already full URL (starts with http), return as-is
    if (url.startsWith('http')) return url;
    // Extract filename and convert to SFTP URL
    const filename = url.split('/').pop();
    if (!filename) return null;
    return adapter.getUrl(`products/images/${filename}`);
  };
  
  return {
    id: orderRow.id,
    allegroOrderId: orderRow.source === 'ALLEGRO' ? orderRow.source_order_id : null,
    shoperOrderId: orderRow.source === 'SHOPER' ? orderRow.source_order_id : null,
    orderNumber: orderRow.order_number || orderRow.id, // order_number = sekwencyjny numer bez luk
    odooOrderId: orderRow.odoo_order_id,
    source: orderRow.source,
    hasReturns: orderRow.has_returns || false,
    boughtAt: orderRow.order_date_with_tz || orderRow.order_date, // Preferuj datę ze strefą czasową
    updatedAt: orderRow.updated_at,
    status: orderRow.status,
    marketplace: rawPayload.marketplace || (orderRow.source === 'ALLEGRO' ? 'allegro-pl' : null),
    buyer: {
      id: rawPayload.buyer_id,
      login: orderRow.buyer_login,
      email: orderRow.buyer_email,
      companyName: orderRow.buyer_company,
      firstName: orderRow.buyer_first_name,
      lastName: orderRow.buyer_last_name,
      phoneNumber: orderRow.buyer_phone
    },
    payment: {
      id: orderRow.payment_id,
      type: orderRow.payment_type || null,
      provider: orderRow.payment_provider,
      status: orderRow.payment_status, // Status płatności (PAID, PENDING, etc.)
      paidAmount: {
        amount: orderRow.payment_amount,
        currency: orderRow.payment_currency
      },
      finishedAt: orderRow.payment_date || rawPayload.payment_last_date,
      refundAmount: orderRow.refund_amount || 0,
      refundDate: orderRow.refund_date,
      refundReconciliation: orderRow.refund_reconciliation || []
    },
    invoice: {
      required: orderRow.invoice_required === true,
      address: billingAddress
    },
    billingAddress: {
      companyName: billingAddress.companyName || orderRow.buyer_company,
      firstName: billingAddress.firstName || orderRow.buyer_first_name,
      lastName: billingAddress.lastName || orderRow.buyer_last_name,
      street: billingAddress.street || orderRow.buyer_address,
      zipCode: billingAddress.zipCode || orderRow.buyer_zip,
      city: billingAddress.city || orderRow.buyer_city,
      countryCode: billingAddress.countryCode || orderRow.buyer_country_code,
      phoneNumber: billingAddress.phoneNumber || orderRow.buyer_phone,
      email: billingAddress.email || orderRow.buyer_email,
      taxId: orderRow.tax_id
    },
    delivery: {
      method: {
        name: orderRow.delivery_method
      },
      address: deliveryAddress.street ? {
        firstName: deliveryAddress.firstName,
        lastName: deliveryAddress.lastName,
        companyName: deliveryAddress.companyName,
        street: deliveryAddress.street,
        zipCode: deliveryAddress.zipCode,
        city: deliveryAddress.city,
        countryCode: deliveryAddress.countryCode,
        phoneNumber: deliveryAddress.phoneNumber
      } : {
        street: orderRow.buyer_address,
        zipCode: orderRow.buyer_zip,
        city: orderRow.buyer_city,
        countryCode: orderRow.buyer_country_code
      },
      cost: {
        amount: orderRow.delivery_amount,
        currency: orderRow.delivery_currency
      },
      smart: rawPayload.smart,
      shipments: typeof orderRow.shipments === 'string' 
        ? (orderRow.shipments ? JSON.parse(orderRow.shipments) : [])
        : (orderRow.shipments || [])
    },
    summary: {
      totalToPay: {
        amount: orderRow.total_to_pay_amount,
        currency: orderRow.total_to_pay_currency
      }
    },
    messageToSeller: orderRow.buyer_notes,
    lineItems: items.map((item: any) => ({
      id: item.id,
      offer: {
        id: item.offer_id,
        name: item.product_name || item.name,
        external: {
          id: item.offer_external_id
        }
      },
      quantity: item.quantity,
      returnsQuantity: item.returns_quantity || 0,
      price: {
        amount: item.price,
        currency: item.currency
      },
      reconciliation: {
        quantity: item.quantity,
        tax: {
          percentage: item.tax_rate,
          subjectType: item.tax_subject
        }
      },
      imageUrl: convertImageUrl(item.product_image_url || item.image_url),
      description: item.product_description
    })),
    rawPayload: rawPayload
  };
}

export async function getOrdersFromPostgres(
  sortBy: string = 'order_date',
  sortOrder: 'ASC' | 'DESC' = 'DESC',
  limit: number = 50,
  offset: number = 0,
  search?: string,
  sourceFilter?: string,
  dateRangeType?: string,
  customDateFrom?: string,
  customDateTo?: string,
  paymentFilter?: string,
  fulfillmentFilter?: string,
  invoiceFilter?: string,
  paymentTypeFilter?: string
) {
  const startTime = Date.now();
  const client = await pool.connect();
  
  try {
    // Whitelist dozwolonych kolumn do sortowania
    const allowedSortColumns = [
      'id', // id = klucz główny (nie zmieniać)
      'order_number', // order_number = sekwencyjny numer zamówienia
      'order_date',
      'total_to_pay_amount',
      'buyer_login',
      'buyer_email',
      'status',
      'payment_status',
      'source',
      'created_at'
    ];
    
    const sortColumn = allowedSortColumns.includes(sortBy) ? sortBy : 'order_date';
    const sortDirection = sortOrder === 'ASC' ? 'ASC' : 'DESC';
    
    // Buduj WHERE clause dla wyszukiwania i filtrów
    const conditions: string[] = [];
    let queryParams: any[] = [];
    let paramIndex = 1;
    
    if (search && search.trim()) {
      const searchTerm = `%${search.trim().toLowerCase()}%`;
      conditions.push(`(
        CAST(order_number AS TEXT) LIKE $${paramIndex} OR
        LOWER(buyer_first_name) LIKE $${paramIndex} OR
        LOWER(buyer_last_name) LIKE $${paramIndex} OR
        LOWER(CONCAT(buyer_first_name, ' ', buyer_last_name)) LIKE $${paramIndex} OR
        LOWER(buyer_email) LIKE $${paramIndex}
      )`);
      queryParams.push(searchTerm);
      paramIndex++;
    }
    
    // Filtr źródła
    if (sourceFilter && sourceFilter !== 'all') {
      conditions.push(`source = $${paramIndex}`);
      queryParams.push(sourceFilter);
      paramIndex++;
    }
    
    // Filtr zakresu dat
    if (dateRangeType && dateRangeType !== 'all') {
      const now = new Date();
      let startDate: Date | null = null;
      let endDate: Date | null = null;

      if (dateRangeType === 'today') {
        startDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 0, 0, 0);
        endDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 23, 59, 59);
      } else if (dateRangeType === 'yesterday') {
        const yesterday = new Date(now);
        yesterday.setDate(yesterday.getDate() - 1);
        startDate = new Date(yesterday.getFullYear(), yesterday.getMonth(), yesterday.getDate(), 0, 0, 0);
        endDate = new Date(yesterday.getFullYear(), yesterday.getMonth(), yesterday.getDate(), 23, 59, 59);
      } else if (dateRangeType === '7days') {
        startDate = new Date(now);
        startDate.setDate(startDate.getDate() - 6);
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 23, 59, 59);
      } else if (dateRangeType === '30days') {
        startDate = new Date(now);
        startDate.setDate(startDate.getDate() - 29);
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 23, 59, 59);
      } else if (dateRangeType === 'custom' && customDateFrom && customDateTo) {
        startDate = new Date(customDateFrom);
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date(customDateTo);
        endDate.setHours(23, 59, 59, 999);
      }

      if (startDate && endDate) {
        conditions.push(`order_date >= $${paramIndex}`);
        queryParams.push(startDate.toISOString());
        paramIndex++;
        conditions.push(`order_date <= $${paramIndex}`);
        queryParams.push(endDate.toISOString());
        paramIndex++;
      }
    }

    // Filtr płatności
    if (paymentFilter && paymentFilter !== 'all' && paymentFilter !== 'Wszystkie') {
      conditions.push(`payment_status = $${paramIndex}`);
      queryParams.push(paymentFilter);
      paramIndex++;
    }

    // Filtr realizacji
    if (fulfillmentFilter && fulfillmentFilter !== 'all' && fulfillmentFilter !== 'Wszystkie') {
      conditions.push(`status = $${paramIndex}`);
      queryParams.push(fulfillmentFilter);
      paramIndex++;
    }

    // Filtr faktury
    if (invoiceFilter && invoiceFilter !== 'all' && invoiceFilter !== 'Wszystkie') {
      const hasInvoice = invoiceFilter === 'Tak';
      conditions.push(`has_invoice = $${paramIndex}`);
      queryParams.push(hasInvoice);
      paramIndex++;
    }

    // Filtr typu płatności
    if (paymentTypeFilter && paymentTypeFilter !== 'all' && paymentTypeFilter !== 'Wszystkie') {
      conditions.push(`payment_type = $${paramIndex}`);
      queryParams.push(paymentTypeFilter);
      paramIndex++;
    }
    
    const whereClause = conditions.length > 0 ? `WHERE ${conditions.join(' AND ')}` : '';
    queryParams.push(limit, offset);
    
    // Pobierz zamówienia z paginacją i wyszukiwaniem
    const t1 = Date.now();
    
    // Sortowanie po order_date - wszystkie daty są teraz w UTC, sortuj bezpośrednio
    const orderByClause = `${sortColumn} ${sortDirection}`;
    
    const ordersResult = await client.query(`
      SELECT 
        *,
        -- Wszystkie daty są teraz w UTC w bazie, konwertuj na Europe/Warsaw z automatycznym DST
        to_char(order_date AT TIME ZONE 'Europe/Warsaw', 'YYYY-MM-DD"T"HH24:MI:SS.MS') || 
          (CASE 
            WHEN EXTRACT(MONTH FROM order_date) >= 4 AND EXTRACT(MONTH FROM order_date) <= 9 
            THEN '+02:00'
            WHEN EXTRACT(MONTH FROM order_date) = 3 AND EXTRACT(DAY FROM order_date) >= 25 
            THEN '+02:00'
            WHEN EXTRACT(MONTH FROM order_date) = 10 AND EXTRACT(DAY FROM order_date) < 25 
            THEN '+02:00'
            ELSE '+01:00'
          END) as order_date_with_tz
      FROM commerce.orders 
      ${whereClause}
      ORDER BY ${orderByClause}
      LIMIT $${paramIndex} OFFSET $${paramIndex + 1}
    `, queryParams);
    console.log(`📊 Orders query: ${Date.now() - t1}ms, rows: ${ordersResult.rows.length}, limit: ${limit}, offset: ${offset}, search: ${search || 'none'}`);
    
    // DEBUG: Pokaż pierwsze 3 zamówienia z ich datami
    if (ordersResult.rows.length > 0 && sortColumn === 'order_date') {
      console.log('📊 DEBUG - First 3 orders (sorted):', ordersResult.rows.slice(0, 3).map(r => ({
        id: r.id,
        order_number: r.order_number || `ID:${r.id}`,
        source: r.source,
        order_date_raw: r.order_date,
        order_date_display: r.order_date_with_tz
      })));
    }
    
    if (ordersResult.rows.length === 0) {
      return [];
    }
    
    // Pobierz wszystkie produkty dla wszystkich zamówień w JEDNYM zapytaniu
    const orderIds = ordersResult.rows.map(row => row.id);
    const t2 = Date.now();
    const itemsResult = await client.query(`
      SELECT 
        oi.*,
        COALESCE(mp.name, oi.name) as product_name,
        mp.image_url as product_image_url
      FROM commerce.order_items oi
      JOIN commerce.orders o ON oi.order_id = o.id
      LEFT JOIN commerce.marketplace_products mp 
        ON oi.offer_external_id = mp.offer_external_id 
        AND mp.source = o.source
      WHERE oi.order_id = ANY($1)
      ORDER BY oi.order_id, oi.id
    `, [orderIds]);
    console.log(`📊 Items query: ${Date.now() - t2}ms, rows: ${itemsResult.rows.length}`);
    
    // Grupuj produkty po order_id
    const t3 = Date.now();
    const itemsByOrderId = new Map<string, any[]>();
    for (const item of itemsResult.rows) {
      const orderId = item.order_id;
      if (!itemsByOrderId.has(orderId)) {
        itemsByOrderId.set(orderId, []);
      }
      itemsByOrderId.get(orderId)!.push(item);
    }
    console.log(`📊 Group items: ${Date.now() - t3}ms`);
    
    // Zbuduj zamówienia z odpowiednimi produktami
    const t4 = Date.now();
    const orders = await Promise.all(
      ordersResult.rows.map(async (orderRow) => {
        const items = itemsByOrderId.get(orderRow.id) || [];
        return await buildOrderFromRow(orderRow, items);
      })
    );
    console.log(`📊 Build orders: ${Date.now() - t4}ms`);
    console.log(`📊 TOTAL getOrdersFromPostgres: ${Date.now() - startTime}ms`);
    
    return orders;
  } catch (error) {
    console.error('❌ Error fetching orders from PostgreSQL:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function getOrdersCount(
  search?: string,
  sourceFilter?: string,
  dateRangeType?: string,
  customDateFrom?: string,
  customDateTo?: string,
  paymentFilter?: string,
  fulfillmentFilter?: string,
  invoiceFilter?: string,
  paymentTypeFilter?: string
) {
  const client = await pool.connect();
  
  try {
    const conditions: string[] = [];
    let queryParams: any[] = [];
    let paramIndex = 1;
    
    if (search && search.trim()) {
      const searchTerm = `%${search.trim().toLowerCase()}%`;
      conditions.push(`(
        CAST(order_number AS TEXT) LIKE $${paramIndex} OR
        LOWER(buyer_first_name) LIKE $${paramIndex} OR
        LOWER(buyer_last_name) LIKE $${paramIndex} OR
        LOWER(CONCAT(buyer_first_name, ' ', buyer_last_name)) LIKE $${paramIndex} OR
        LOWER(buyer_email) LIKE $${paramIndex}
      )`);
      queryParams.push(searchTerm);
      paramIndex++;
    }
    
    // Filtr źródła
    if (sourceFilter && sourceFilter !== 'all') {
      conditions.push(`source = $${paramIndex}`);
      queryParams.push(sourceFilter);
      paramIndex++;
    }
    
    // Filtr zakresu dat
    if (dateRangeType && dateRangeType !== 'all') {
      const now = new Date();
      let startDate: Date | null = null;
      let endDate: Date | null = null;

      if (dateRangeType === 'today') {
        startDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 0, 0, 0);
        endDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 23, 59, 59);
      } else if (dateRangeType === 'yesterday') {
        const yesterday = new Date(now);
        yesterday.setDate(yesterday.getDate() - 1);
        startDate = new Date(yesterday.getFullYear(), yesterday.getMonth(), yesterday.getDate(), 0, 0, 0);
        endDate = new Date(yesterday.getFullYear(), yesterday.getMonth(), yesterday.getDate(), 23, 59, 59);
      } else if (dateRangeType === '7days') {
        startDate = new Date(now);
        startDate.setDate(startDate.getDate() - 6);
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 23, 59, 59);
      } else if (dateRangeType === '30days') {
        startDate = new Date(now);
        startDate.setDate(startDate.getDate() - 29);
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 23, 59, 59);
      } else if (dateRangeType === 'custom' && customDateFrom && customDateTo) {
        startDate = new Date(customDateFrom);
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date(customDateTo);
        endDate.setHours(23, 59, 59, 999);
      }

      if (startDate && endDate) {
        conditions.push(`order_date >= $${paramIndex}`);
        queryParams.push(startDate.toISOString());
        paramIndex++;
        conditions.push(`order_date <= $${paramIndex}`);
        queryParams.push(endDate.toISOString());
        paramIndex++;
      }
    }

    // Filtr płatności
    if (paymentFilter && paymentFilter !== 'all' && paymentFilter !== 'Wszystkie') {
      conditions.push(`payment_status = $${paramIndex}`);
      queryParams.push(paymentFilter);
      paramIndex++;
    }

    // Filtr realizacji
    if (fulfillmentFilter && fulfillmentFilter !== 'all' && fulfillmentFilter !== 'Wszystkie') {
      conditions.push(`status = $${paramIndex}`);
      queryParams.push(fulfillmentFilter);
      paramIndex++;
    }

    // Filtr faktury
    if (invoiceFilter && invoiceFilter !== 'all' && invoiceFilter !== 'Wszystkie') {
      const hasInvoice = invoiceFilter === 'Tak';
      conditions.push(`has_invoice = $${paramIndex}`);
      queryParams.push(hasInvoice);
      paramIndex++;
    }

    // Filtr typu płatności
    if (paymentTypeFilter && paymentTypeFilter !== 'all' && paymentTypeFilter !== 'Wszystkie') {
      conditions.push(`payment_type = $${paramIndex}`);
      queryParams.push(paymentTypeFilter);
      paramIndex++;
    }
    
    const whereClause = conditions.length > 0 ? `WHERE ${conditions.join(' AND ')}` : '';
    
    const result = await client.query(
      `SELECT COUNT(*) as count FROM commerce.orders ${whereClause}`,
      queryParams
    );
    return parseInt(result.rows[0].count, 10);
  } catch (error) {
    console.error('❌ Error counting orders:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function getOrderByNumberFromPostgres(orderNumber: string) {
  const client = await pool.connect();
  
  try {
    const orderResult = await client.query(`
      SELECT 
        *,
        -- Wszystkie daty są teraz w UTC w bazie, konwertuj na Europe/Warsaw z automatycznym DST
        to_char(order_date AT TIME ZONE 'Europe/Warsaw', 'YYYY-MM-DD"T"HH24:MI:SS.MS') || 
          (CASE 
            WHEN EXTRACT(MONTH FROM order_date) >= 4 AND EXTRACT(MONTH FROM order_date) <= 9 
            THEN '+02:00'
            WHEN EXTRACT(MONTH FROM order_date) = 3 AND EXTRACT(DAY FROM order_date) >= 25 
            THEN '+02:00'
            WHEN EXTRACT(MONTH FROM order_date) = 10 AND EXTRACT(DAY FROM order_date) < 25 
            THEN '+02:00'
            ELSE '+01:00'
          END) as order_date_with_tz
      FROM commerce.orders 
      WHERE order_number = $1
    `, [orderNumber]);
    
    if (orderResult.rows.length === 0) {
      return null;
    }
    
    const orderRow = orderResult.rows[0];
    
    const itemsResult = await client.query(`
      SELECT 
        oi.*,
        COALESCE(mp.name, oi.name) as product_name,
        mp.image_url as product_image_url
      FROM commerce.order_items oi
      LEFT JOIN commerce.marketplace_products mp 
        ON oi.offer_external_id = mp.offer_external_id 
        AND mp.source = $2
      WHERE oi.order_id = $1
    `, [orderRow.id, orderRow.source]);
    
    return await buildOrderFromRow(orderRow, itemsResult.rows);
  } catch (error) {
    console.error('❌ Error fetching order by number from PostgreSQL:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function getOrderFromPostgres(orderId: number) {
  const client = await pool.connect();
  
  try {
    const orderResult = await client.query(`
      SELECT 
        *,
        -- Wszystkie daty są teraz w UTC w bazie, konwertuj na Europe/Warsaw z automatycznym DST
        to_char(order_date AT TIME ZONE 'Europe/Warsaw', 'YYYY-MM-DD"T"HH24:MI:SS.MS') || 
          (CASE 
            WHEN EXTRACT(MONTH FROM order_date) >= 4 AND EXTRACT(MONTH FROM order_date) <= 9 
            THEN '+02:00'
            WHEN EXTRACT(MONTH FROM order_date) = 3 AND EXTRACT(DAY FROM order_date) >= 25 
            THEN '+02:00'
            WHEN EXTRACT(MONTH FROM order_date) = 10 AND EXTRACT(DAY FROM order_date) < 25 
            THEN '+02:00'
            ELSE '+01:00'
          END) as order_date_with_tz
      FROM commerce.orders 
      WHERE id = $1
    `, [orderId]);
    
    if (orderResult.rows.length === 0) {
      return null;
    }
    
    const orderRow = orderResult.rows[0];
    
    const itemsResult = await client.query(`
      SELECT 
        oi.*,
        COALESCE(mp.name, oi.name) as product_name,
        mp.image_url as product_image_url
      FROM commerce.order_items oi
      LEFT JOIN commerce.marketplace_products mp 
        ON oi.offer_external_id = mp.offer_external_id 
        AND mp.source = $2
      WHERE oi.order_id = $1
    `, [orderRow.id, orderRow.source]);
    
    return await buildOrderFromRow(orderRow, itemsResult.rows);
  } catch (error) {
    console.error('❌ Error fetching order from PostgreSQL:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function getUniqueProductsFromPostgres() {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      SELECT product_id as offer_id, external_id as offer_external_id, name
      FROM catalog.products
      WHERE external_id IS NOT NULL
        AND name IS NOT NULL
        AND name != ''
      ORDER BY name
    `);
    
    return result.rows;
  } catch (error) {
    console.error('❌ Error fetching unique products from PostgreSQL:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function getUniqueProductsWithImagesFromPostgres() {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      SELECT DISTINCT ON (external_id)
        product_id as offer_id, 
        external_id as offer_external_id, 
        name,
        image_url
      FROM catalog.products
      WHERE external_id IS NOT NULL
      ORDER BY external_id, updated_at DESC
    `);
    
    console.log(`📦 Found ${result.rows.length} unique products`);
    const withImages = result.rows.filter(p => p.image_url).length;
    console.log(`🖼️ ${withImages} products have image URLs`);
    
    return result.rows;
  } catch (error) {
    console.error('❌ Error fetching unique products with images from PostgreSQL:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function updateProductImageUrl(externalId: string, imageUrl: string) {
  const client = await pool.connect();
  
  try {
    await client.query(`
      UPDATE catalog.products
      SET image_url = $1, updated_at = NOW()
      WHERE external_id = $2
    `, [imageUrl, externalId]);
    
    await client.query(`
      UPDATE commerce.order_items
      SET image_url = $1
      WHERE offer_external_id = $2
    `, [imageUrl, externalId]);
  } catch (error) {
    console.error(`❌ Error updating image URL for product ${externalId}:`, error);
    throw error;
  } finally {
    client.release();
  }
}

export async function getAllProducts() {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      SELECT * FROM catalog.products
      ORDER BY updated_at DESC
    `);
    
    return result.rows;
  } catch (error) {
    console.error('❌ Error fetching products from PostgreSQL:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function testConnection() {
  try {
    const client = await connectWithRetry();
    const result = await client.query('SELECT NOW()');
    console.log('✅ PostgreSQL connection successful:', result.rows[0]);
    client.release();
    return true;
  } catch (error) {
    console.error('❌ PostgreSQL connection failed:', error);
    return false;
  }
}

export async function createSyncLog(startedAt: Date) {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      INSERT INTO allegro.sync_logs (sync_started_at, status)
      VALUES ($1, 'RUNNING')
      RETURNING id
    `, [startedAt]);
    
    return result.rows[0].id;
  } catch (error) {
    console.error('❌ Error creating sync log:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function completeSyncLog(
  logId: number, 
  status: 'SUCCESS' | 'ERROR',
  ordersFetched: number,
  ordersNew: number,
  ordersUpdated: number,
  errorMessage?: string
) {
  const client = await pool.connect();
  
  try {
    const completedAt = new Date();
    const startResult = await client.query(
      'SELECT sync_started_at FROM allegro.sync_logs WHERE id = $1',
      [logId]
    );
    
    const syncDuration = startResult.rows[0] 
      ? completedAt.getTime() - new Date(startResult.rows[0].sync_started_at).getTime()
      : 0;
    
    await client.query(`
      UPDATE allegro.sync_logs
      SET sync_completed_at = $1,
          status = $2,
          orders_fetched = $3,
          orders_new = $4,
          orders_updated = $5,
          error_message = $6,
          sync_duration_ms = $7
      WHERE id = $8
    `, [completedAt, status, ordersFetched, ordersNew, ordersUpdated, errorMessage, syncDuration, logId]);
  } catch (error) {
    console.error('❌ Error completing sync log:', error);
  } finally {
    client.release();
  }
}

export async function getSyncLogs(limit: number = 50) {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      SELECT * FROM allegro.sync_logs
      ORDER BY sync_started_at DESC
      LIMIT $1
    `, [limit]);
    
    return result.rows;
  } catch (error) {
    console.error('❌ Error fetching sync logs:', error);
    throw error;
  } finally {
    client.release();
  }
}

// Konwertuj czas UTC na lokalny (Europe/Warsaw, UTC+1/+2)
function convertUTCToLocal(utcDateString: string): string {
  const utcDate = new Date(utcDateString);
  // Konwertuj na lokalny czas Polski (Europe/Warsaw)
  const localDate = new Date(utcDate.toLocaleString('en-US', { timeZone: 'Europe/Warsaw' }));
  
  // Format: YYYY-MM-DD HH:MM:SS.mmm
  const year = localDate.getFullYear();
  const month = String(localDate.getMonth() + 1).padStart(2, '0');
  const day = String(localDate.getDate()).padStart(2, '0');
  const hours = String(localDate.getHours()).padStart(2, '0');
  const minutes = String(localDate.getMinutes()).padStart(2, '0');
  const seconds = String(localDate.getSeconds()).padStart(2, '0');
  const ms = String(localDate.getMilliseconds()).padStart(3, '0');
  
  return `${year}-${month}-${day} ${hours}:${minutes}:${seconds}.${ms}`;
}

export async function saveOrderToCommerce(
  source: 'ALLEGRO' | 'SHOPER',
  sourceOrderId: string,
  orderData: any,
  shipments: any[] = [],
  customerReturns: any[] = [],
  accessToken?: string
): Promise<{ isNew: boolean }> {
  const client = await pool.connect();
  
  try {
    // Wymuś odświeżenie snapshotu - usuń stary cache połączenia
    await client.query('DISCARD ALL');
    await client.query('BEGIN');

    const existsResult = await client.query(
      'SELECT id FROM commerce.orders WHERE source_order_id = $1 AND source = $2',
      [sourceOrderId, source]
    );
    const isNew = existsResult.rows.length === 0;

    let buyer, payment, delivery, lineItems, orderDate, billingAddress;

    if (source === 'ALLEGRO') {
      buyer = orderData.buyer || {};
      
      // Oblicz payment status dla Allegro z dokładnym porównaniem kwot
      const allegroPayment = orderData.payment || {};
      const paidAmount = parseFloat(allegroPayment.paidAmount?.amount || '0');
      const totalToPay = parseFloat(orderData.summary?.totalToPay?.amount || '0');
      const hasFinishedAt = !!(allegroPayment.finishedAt);
      
      let paymentStatus = 'UNPAID';
      if (paidAmount >= totalToPay - 0.01 && totalToPay > 0) {
        // Pełna płatność (z tolerancją na zaokrąglenia)
        paymentStatus = 'PAID';
      } else if (paidAmount > 0) {
        // Częściowa płatność
        paymentStatus = 'PARTIAL';
      } else if (hasFinishedAt && paidAmount === 0) {
        // Płatność zakończona ale kwota 0 (może być błąd API lub zwrot)
        paymentStatus = 'PAID';
      }
      
      console.log(`🔍 DEBUG - Allegro payment raw data for order ${orderData.id}:`);
      console.log(`   payment.paidAmount.amount = "${allegroPayment.paidAmount?.amount}" → paidAmount = ${paidAmount}`);
      console.log(`   summary.totalToPay.amount = "${orderData.summary?.totalToPay?.amount}" → totalToPay = ${totalToPay}`);
      console.log(`   payment.finishedAt = "${allegroPayment.finishedAt}" → hasFinishedAt = ${hasFinishedAt}`);
      console.log(`   Calculated paymentStatus = "${paymentStatus}"`);
      
      payment = {
        ...allegroPayment,
        status: paymentStatus  // Dodaj obliczony status
      };
      
      delivery = orderData.delivery || {};
      
      // 🛒 POPRAWKA: Jeśli lineItems jest puste w API, pobierz z allegro.order_items
      if (!orderData.lineItems || orderData.lineItems.length === 0) {
        const itemsResult = await client.query(`
          SELECT 
            order_id as id,
            offer_id,
            name,
            quantity,
            price,
            offer_external_id
          FROM allegro.order_items
          WHERE order_id = $1
        `, [sourceOrderId]);
        
        if (itemsResult.rows.length > 0) {
          // Konwertuj do formatu lineItems z API Allegro
          lineItems = itemsResult.rows.map((item: any) => ({
            id: item.id,
            offer: {
              id: item.offer_id,
              name: item.name,
              external: { id: item.offer_external_id }
            },
            quantity: item.quantity,
            price: {
              amount: item.price?.toString(),
              currency: 'PLN'
            }
          }));
          console.log(`🛒 Uzupełniono ${lineItems.length} produktów z allegro.order_items dla zamówienia ${sourceOrderId}`);
        } else {
          lineItems = [];
        }
      } else {
        lineItems = orderData.lineItems;
      }
      
      // 📅 POPRAWKA: Jeśli boughtAt jest NULL, użyj daty z allegro.orders zamiast updatedAt
      if (orderData.boughtAt) {
        orderDate = orderData.boughtAt;
      } else {
        // Pobierz poprawną datę z allegro.orders
        const allegroOrderDate = await client.query(`
          SELECT order_date FROM allegro.orders WHERE order_id = $1
        `, [sourceOrderId]);
        
        if (allegroOrderDate.rows.length > 0 && allegroOrderDate.rows[0].order_date) {
          orderDate = allegroOrderDate.rows[0].order_date;
          console.log(`📅 Użyto order_date z allegro.orders: ${orderDate}`);
        } else {
          // Ostateczny fallback - updatedAt
          orderDate = orderData.updatedAt || null;
          console.log(`⚠️ Fallback do updatedAt: ${orderDate}`);
        }
      }
      
      billingAddress = orderData.invoice || {}; // Allegro: invoice jako billing address
    } else if (source === 'SHOPER') {
      billingAddress = orderData.billing_address || {};
      const deliveryAddress = orderData.delivery_address || {};
      buyer = {
        login: billingAddress.firstname && billingAddress.lastname 
          ? `${billingAddress.firstname} ${billingAddress.lastname}` 
          : orderData.user_login || 'Unknown',
        email: billingAddress.email || orderData.email,
        firstName: billingAddress.firstname,
        lastName: billingAddress.lastname,
        companyName: billingAddress.company,
        phoneNumber: billingAddress.phone || deliveryAddress.phone,
      };
      
      const paidAmount = parseFloat(orderData.paid || '0');
      const totalAmount = parseFloat(orderData.sum || '0');
      const hasPaidDate = !!(orderData.paid_date && orderData.paid_date !== '0000-00-00 00:00:00');
      
      console.log(`🔍 DEBUG - Shoper payment raw data for order ${orderData.order_id}:`);
      console.log(`   orderData.paid = "${orderData.paid}" → paidAmount = ${paidAmount}`);
      console.log(`   orderData.sum = "${orderData.sum}" → totalAmount = ${totalAmount}`);
      console.log(`   orderData.paid_date = "${orderData.paid_date}" → hasPaidDate = ${hasPaidDate}`);
      console.log(`   orderData.is_paid = ${orderData.is_paid}`);
      
      let paymentStatus = 'PENDING';
      if (hasPaidDate || orderData.is_paid || paidAmount >= totalAmount - 0.01) {
        paymentStatus = 'PAID';
      } else if (paidAmount > 0) {
        paymentStatus = 'PARTIAL';
      } else {
        paymentStatus = 'UNPAID';
      }
      
      console.log(`   Calculated paymentStatus = "${paymentStatus}"`);
      
      payment = {
        id: orderData.payment_id,
        status: paymentStatus,
        // If payment is PAID or PARTIAL, it's ONLINE. Only UNPAID with POBRANIE method is COD
        type: (paymentStatus === 'UNPAID' && orderData.payment_method_name?.toUpperCase().includes('POBRANIE')) 
          ? 'CASH_ON_DELIVERY' 
          : 'ONLINE',
        provider: orderData.payment_method_name,
        paidAmount: {
          amount: paidAmount,
          currency: orderData.currency || 'PLN'
        }
      };
      
      // Mapuj adres dostawy z poprawnych pól Shoper API
      const deliveryAddressObj: any = {};
      if (deliveryAddress.street1) {
        deliveryAddressObj.street = deliveryAddress.street1;
      }
      if (deliveryAddress.postcode) {
        deliveryAddressObj.zipCode = deliveryAddress.postcode;
      }
      if (deliveryAddress.city) {
        deliveryAddressObj.city = deliveryAddress.city;
      }
      if (deliveryAddress.country_code) {
        deliveryAddressObj.countryCode = deliveryAddress.country_code;
      }
      if (deliveryAddress.firstname) {
        deliveryAddressObj.firstName = deliveryAddress.firstname;
      }
      if (deliveryAddress.lastname) {
        deliveryAddressObj.lastName = deliveryAddress.lastname;
      }
      if (deliveryAddress.company) {
        deliveryAddressObj.companyName = deliveryAddress.company;
      }
      if (deliveryAddress.phone) {
        deliveryAddressObj.phoneNumber = deliveryAddress.phone;
      }
      
      delivery = {
        method: { name: orderData.delivery_method_name },
        address: Object.keys(deliveryAddressObj).length > 0 ? deliveryAddressObj : undefined,
        cost: {
          amount: orderData.delivery_price,
          currency: orderData.currency || 'PLN'
        }
      } as any;
      
      // Dodaj delivery_date jeśli istnieje
      if (orderData.delivery_date) {
        (delivery as any).time = {
          from: orderData.delivery_date,
          to: orderData.delivery_date
        };
      }
      
      lineItems = orderData.products_data || [];
      // Shoper zwraca datę w czasie lokalnym Polski (Europe/Warsaw)
      // Przekaż jako string a PostgreSQL skonwertuje używając timezone
      const shoperDateStr = orderData.date || orderData.confirm_date;
      if (shoperDateStr) {
        // Dodaj marker timezone aby PostgreSQL wiedział że to Europe/Warsaw time
        orderDate = shoperDateStr; // Zostanie skonwertowane w INSERT query
      } else {
        orderDate = null;
      }
    }

    // Przetwórz shipments z Shoper do jednolitego formatu
    if (source === 'SHOPER' && shipments && shipments.length > 0) {
      const processedShipments: any[] = [];
      const carrierName = delivery.method?.name || 'Unknown';
      
      for (const parcel of shipments) {
        // shipping_code zawiera numery przesyłek oddzielone przecinkami
        const trackingNumbers = (parcel.shipping_code || '').split(',').filter((n: string) => n.trim());
        
        // Jeśli są numery przesyłek, utwórz wpis dla każdego
        if (trackingNumbers.length > 0) {
          for (const trackingNumber of trackingNumbers) {
            processedShipments.push({
              waybill: trackingNumber.trim(),
              carrierId: carrierName,
              createdAt: parcel.send_date && parcel.send_date !== '0000-00-00 00:00:00' ? parcel.send_date : parcel.order_date,
              sent: parcel.sent === '1' || parcel.sent === 1
            });
          }
        } else if (carrierName && carrierName !== 'Unknown') {
          // Jeśli nie ma numerów, ale jest metoda dostawy, pokaż to
          processedShipments.push({
            waybill: null,
            carrierId: carrierName,
            createdAt: parcel.send_date && parcel.send_date !== '0000-00-00 00:00:00' ? parcel.send_date : parcel.order_date,
            sent: parcel.sent === '1' || parcel.sent === 1
          });
        }
      }
      
      shipments = processedShipments;
    }

    // Pobierz zwroty płatności z API /payments/refunds (tylko dla ALLEGRO)
    let refundAmount = 0;
    let refundDate: string | null = null;
    const refundReconciliation: any[] = [];
    
    if (source === 'ALLEGRO' && accessToken) {
      const { fetchPaymentRefunds } = await import('./allegro-api.js');
      const refunds = await fetchPaymentRefunds(sourceOrderId, accessToken);
      
      for (const refund of refunds) {
        const amount = parseFloat(refund.totalValue?.amount || '0');
        refundAmount += amount;
        refundReconciliation.push(refund);
        
        if (refund.createdAt) {
          if (!refundDate || new Date(refund.createdAt) > new Date(refundDate)) {
            refundDate = refund.createdAt;
          }
        }
      }
    }

    // Sprawdź czy są zwroty produktów (type: QUANTITY) w refund_reconciliation
    const hasReturnsInReconciliation = refundReconciliation.some(refund => 
      refund.lineItems && refund.lineItems.some((item: any) => item.type === 'QUANTITY')
    );
    const hasReturns = hasReturnsInReconciliation;

    // Wzbogać surowe dane API o informacje o zwrotach i refundach
    const enrichedRawPayload = {
      ...orderData,
      payment: {
        ...payment,
        refundAmount: refundAmount || 0,
        refundDate: refundDate || null,
        refundReconciliation: refundReconciliation
      },
      customerReturns: customerReturns,
      shipments: shipments
    };

    // 🔍 Detect changes before UPDATE
    let hasActualChanges = isNew; // Nowe zamówienie zawsze ma zmiany
    
    if (!isNew) {
      // Wymusz odświeżenie snapshotu przed SELECT
      await client.query('SAVEPOINT before_select');
      
      // Ustaw lock_timeout na 5 sekund (zamiast NOWAIT)
      await client.query('SET LOCAL lock_timeout = 5000');
      
      const previousState = await client.query(`
        SELECT 
          id, status, payment_status, payment_amount,
          tracking_numbers, refund_amount, has_returns, buyer_address, buyer_city
        FROM commerce.orders 
        WHERE source_order_id = $1 AND source = $2
        FOR UPDATE
      `, [sourceOrderId, source]);
      
      await client.query('RELEASE SAVEPOINT before_select');

      console.log(`🔍 DEBUG - SELECT previousState result for ${source}-${sourceOrderId}:`, JSON.stringify(previousState.rows, null, 2));

      if (previousState.rows.length > 0) {
        const prev = previousState.rows[0];
        const orderId = prev.id;
        const orderNumber = prev.id?.toString(); // id = numer zamówienia

        // Nowy stan
        // Dla Allegro: używamy głównego statusu zamówienia (orderData.status), nie fulfillment.status
        const newStatus = source === 'ALLEGRO' ? orderData.status : (orderData.status_id || 'new');
        const newPaymentStatus = payment.status || 'PENDING';
        const newPaymentAmount = payment.paidAmount?.amount ?? orderData.sum;
        const newRefundAmount = refundAmount || 0;
        const newHasReturns = hasReturns;
        const newTrackingNumbers = orderData.delivery?.trackingNumber || (orderData.packages ? JSON.stringify(orderData.packages) : null);
        const newBuyerAddress = delivery.address ? 
          `${delivery.address.street || ''}, ${delivery.address.zipCode || ''} ${delivery.address.city || ''}` : null;

        // Funkcja porównująca wartości numeryczne
        const compareNumeric = (oldVal: any, newVal: any): boolean => {
          const oldNum = parseFloat(oldVal?.toString() || '0');
          const newNum = parseFloat(newVal?.toString() || '0');
          return Math.abs(oldNum - newNum) > 0.001; // Tolerancja na błędy zaokrągleń
        };

        // DEBUG: Log wartości przed porównaniem
        console.log(`🔍 DEBUG - Porównanie wartości dla #${orderNumber}:`);
        console.log(`   payment_status: prev="${prev.payment_status}" new="${newPaymentStatus}"`);
        console.log(`   payment_amount: prev="${prev.payment_amount}" (${typeof prev.payment_amount}) new="${newPaymentAmount}" (${typeof newPaymentAmount})`);
        console.log(`   buyer_address: prev="${prev.buyer_address}" new="${newBuyerAddress}"`);

        // Porównaj pola
        const changes: Array<{field: string; oldValue: string; newValue: string}> = [];

        if (prev.status !== newStatus) {
          changes.push({ field: 'status', oldValue: prev.status || 'null', newValue: newStatus || 'null' });
        }
        if (prev.payment_status !== newPaymentStatus) {
          changes.push({ field: 'payment_status', oldValue: prev.payment_status || 'null', newValue: newPaymentStatus });
        }
        if (compareNumeric(prev.payment_amount, newPaymentAmount)) {
          console.log(`   ⚠️ compareNumeric zwrócił TRUE: ${prev.payment_amount} vs ${newPaymentAmount}`);
          changes.push({ 
            field: 'payment_amount', 
            oldValue: prev.payment_amount?.toString() || '0', 
            newValue: newPaymentAmount?.toString() || '0' 
          });
        } else {
          console.log(`   ✅ compareNumeric zwrócił FALSE: ${prev.payment_amount} === ${newPaymentAmount}`);
        }
        if (compareNumeric(prev.refund_amount, newRefundAmount)) {
          changes.push({ 
            field: 'refund_amount', 
            oldValue: prev.refund_amount?.toString() || '0', 
            newValue: newRefundAmount?.toString() || '0' 
          });
        }
        if (prev.has_returns !== newHasReturns) {
          changes.push({ field: 'has_returns', oldValue: prev.has_returns ? 'true' : 'false', newValue: newHasReturns ? 'true' : 'false' });
        }
        if (prev.tracking_numbers !== newTrackingNumbers) {
          changes.push({ field: 'tracking_numbers', oldValue: prev.tracking_numbers || 'null', newValue: newTrackingNumbers || 'null' });
        }
        if (prev.buyer_address !== newBuyerAddress) {
          changes.push({ field: 'buyer_address', oldValue: prev.buyer_address || 'null', newValue: newBuyerAddress || 'null' });
        }

        // Oznacz czy są faktyczne zmiany
        hasActualChanges = changes.length > 0;

        // Loguj zmiany
        if (changes.length > 0) {
          console.log(`🔄 Wykryto ${changes.length} zmian w zamówieniu #${orderNumber}:`);
          for (const change of changes) {
            console.log(`   📝 ${change.field}: "${change.oldValue}" → "${change.newValue}"`);
            
            // Zapisz do bazy
            await logOrderChange(
              orderId,
              parseInt(orderNumber || '0'),
              source,
              'UPDATED',
              change.field,
              change.oldValue,
              change.newValue
            );
          }
        } else {
          console.log(`   ✅ Brak zmian w zamówieniu #${orderNumber} - pomijam aktualizację`);
          // Jeśli brak zmian, zwróć bez aktualizacji bazy
          await client.query('COMMIT');
          return { isNew: false };
        }
      }
    }

    // Dla zamówień za pobraniem (COD): payment_amount = NULL jeśli UNPAID (dla poprawnego wyświetlania na froncie)
    const paymentType = payment.type || (source === 'SHOPER' && orderData.cod ? 'CASH_ON_DELIVERY' : 'ONLINE');
    const totalToPayAmount = orderData.summary?.totalToPay?.amount ?? orderData.sum;
    const paymentAmountParam = paymentType === 'CASH_ON_DELIVERY' 
      ? (payment.status === 'PAID' ? totalToPayAmount : null) // NULL dla UNPAID COD
      : (payment.paidAmount?.amount ?? orderData.sum);
    
    console.log(`🔍 DEBUG - ${isNew ? 'INSERT' : 'UPDATE'} parameter (payment_amount) for ${source}-${sourceOrderId}:`);
    console.log(`   payment.type = ${paymentType}`);
    console.log(`   payment.paidAmount = ${JSON.stringify(payment.paidAmount)}`);
    console.log(`   payment.paidAmount?.amount = ${payment.paidAmount?.amount}`);
    console.log(`   orderData.sum = ${orderData.sum}`);
    console.log(`   totalToPayAmount = ${totalToPayAmount}`);
    console.log(`   Final value = ${paymentAmountParam}`);

    // Przygotuj parametry wspólne dla INSERT i UPDATE
    // Dla Allegro: używamy głównego statusu zamówienia (orderData.status), nie fulfillment.status
    const rawStatus = source === 'ALLEGRO' ? orderData.status : (orderData.status_id?.toString() || '1');
    const mappedStatus = await mapOrderStatus(rawStatus, source);
    
    const orderParams = [
      source,
      sourceOrderId,
      orderDate,
      mappedStatus,
      buyer.login,
      buyer.email,
      buyer.companyName,
      buyer.firstName,
      buyer.lastName,
      buyer.phoneNumber,
      delivery.address ? 
        `${delivery.address.street || ''}, ${delivery.address.zipCode || ''} ${delivery.address.city || ''}` : null,
      delivery.address?.zipCode,
      delivery.address?.city,
      delivery.address?.countryCode || 'PL',
      payment.id,
      payment.status || 'PENDING',
      paymentType,
      payment.provider,
      paymentAmountParam,
      payment.paidAmount?.currency || orderData.currency || 'PLN',
      delivery.method?.name || orderData.delivery_method_name,
      delivery.cost?.amount ?? orderData.delivery_price,
      delivery.cost?.currency || orderData.currency || 'PLN',
      delivery.address ? JSON.stringify(delivery.address) : null,
      billingAddress ? JSON.stringify(billingAddress) : null,
      billingAddress?.zipCode || null,
      billingAddress?.city || null,
      billingAddress?.countryCode || null,
      billingAddress?.company?.taxId || null,
      orderData.summary?.totalToPay?.amount ?? orderData.sum,
      orderData.summary?.totalToPay?.currency || orderData.currency || 'PLN',
      orderData.messageToSeller || orderData.notes,
      orderData.invoice?.required || orderData.want_invoice || false,
      orderData.delivery?.trackingNumber || (orderData.packages ? JSON.stringify(orderData.packages) : null),
      JSON.stringify(shipments),
      refundAmount || 0,
      refundDate || null,
      JSON.stringify(refundReconciliation),
      hasReturns,
      enrichedRawPayload
    ];

    // 🔢 NADAJ NUMER ZAMÓWIENIA (PRZED INSERT) - dla nowych zamówień lub istniejących bez numeru
    let nextOrderNumber: string | null = null;
    console.log(`🔍 [ORDER NUMBERING DEBUG] ${source}-${sourceOrderId}: isNew=${isNew}`);
    if (isNew) {
      // Pobierz i zablokuj sekwencję (SELECT FOR UPDATE) NA POCZĄTKU transakcji
      const { getNextOrderNumber } = await import('./order-numbering.js');
      // Przekazujemy id=0 bo jeszcze nie mamy commerce_order_id, będzie zaktualizowane później w audit
      nextOrderNumber = await getNextOrderNumber(client, 0, source, sourceOrderId);
      console.log(`🔢 [PRE-INSERT] Zarezerwowano numer #${nextOrderNumber} dla ${source}-${sourceOrderId}`);
    } else {
      // Sprawdź czy istniejące zamówienie ma numer
      const checkNumberResult = await client.query(
        'SELECT order_number FROM commerce.orders WHERE source_order_id = $1 AND source = $2',
        [sourceOrderId, source]
      );
      if (checkNumberResult.rows.length > 0 && !checkNumberResult.rows[0].order_number) {
        // Zamówienie istnieje ale nie ma numeru - nadaj numer
        const { getNextOrderNumber } = await import('./order-numbering.js');
        nextOrderNumber = await getNextOrderNumber(client, 0, source, sourceOrderId);
        console.log(`🔢 [FIX-MISSING] Nadano brakujący numer #${nextOrderNumber} dla istniejącego zamówienia ${source}-${sourceOrderId}`);
      } else {
        console.log(`⚠️  [PRE-INSERT] Pomijam nadawanie numeru - zamówienie już istnieje w commerce.orders z numerem`);
      }
    }

    let orderResult;
    
    if (isNew) {
      // Nowe zamówienie - INSERT z predefiniowanym numerem
      console.log(`✨ Tworzę nowe zamówienie ${source}-${sourceOrderId} z numerem #${nextOrderNumber}`);
      console.log(`🔍 [INSERT DEBUG] nextOrderNumber = ${nextOrderNumber}, type = ${typeof nextOrderNumber}`);
      const insertParams = [...orderParams, nextOrderNumber]; // Dodaj order_number jako $41
      console.log(`🔍 [INSERT DEBUG] insertParams[40] (order_number) = ${insertParams[40]}`);
      orderResult = await client.query(`
        INSERT INTO commerce.orders (
          source, source_order_id, order_date, status,
          buyer_login, buyer_email, buyer_company, buyer_first_name, buyer_last_name,
          buyer_phone, buyer_address, buyer_zip, buyer_city, buyer_country_code,
          payment_id, payment_status, payment_type, payment_provider,
          payment_amount, payment_currency,
          delivery_method, delivery_amount, delivery_currency,
          delivery_address, billing_address, billing_zip, billing_city, billing_country_code, tax_id,
          total_to_pay_amount, total_to_pay_currency,
          buyer_notes, invoice_required, tracking_numbers, shipments,
          refund_amount, refund_date, refund_reconciliation, has_returns,
          raw_payload, order_number, created_at, updated_at
        ) VALUES (
          $1, $2, 
          CASE WHEN $1 = 'SHOPER' THEN timezone('Europe/Warsaw', $3::timestamp) ELSE $3::timestamptz END,
          $4, $5, $6, $7, $8, $9, $10,
          $11, $12, $13, $14, $15, $16, $17, $18, $19, $20,
          $21, $22, $23, $24, $25, $26, $27, $28, $29, $30, $31, $32, $33, $34, $35, $36, $37, $38, $39, $40, $41, NOW(), NOW()
        )
        ON CONFLICT (source, source_order_id) DO UPDATE SET
          order_date = EXCLUDED.order_date,
          status = EXCLUDED.status,
          buyer_login = EXCLUDED.buyer_login,
          buyer_email = EXCLUDED.buyer_email,
          buyer_company = EXCLUDED.buyer_company,
          buyer_first_name = EXCLUDED.buyer_first_name,
          buyer_last_name = EXCLUDED.buyer_last_name,
          buyer_phone = EXCLUDED.buyer_phone,
          buyer_address = EXCLUDED.buyer_address,
          buyer_zip = EXCLUDED.buyer_zip,
          buyer_city = EXCLUDED.buyer_city,
          buyer_country_code = EXCLUDED.buyer_country_code,
          payment_id = EXCLUDED.payment_id,
          payment_status = EXCLUDED.payment_status,
          payment_type = EXCLUDED.payment_type,
          payment_provider = EXCLUDED.payment_provider,
          payment_amount = EXCLUDED.payment_amount,
          payment_currency = EXCLUDED.payment_currency,
          delivery_method = EXCLUDED.delivery_method,
          delivery_amount = EXCLUDED.delivery_amount,
          delivery_currency = EXCLUDED.delivery_currency,
          delivery_address = EXCLUDED.delivery_address,
          billing_address = EXCLUDED.billing_address,
          billing_zip = EXCLUDED.billing_zip,
          billing_city = EXCLUDED.billing_city,
          billing_country_code = EXCLUDED.billing_country_code,
          tax_id = EXCLUDED.tax_id,
          total_to_pay_amount = EXCLUDED.total_to_pay_amount,
          total_to_pay_currency = EXCLUDED.total_to_pay_currency,
          buyer_notes = EXCLUDED.buyer_notes,
          invoice_required = EXCLUDED.invoice_required,
          tracking_numbers = EXCLUDED.tracking_numbers,
          shipments = EXCLUDED.shipments,
          refund_amount = EXCLUDED.refund_amount,
          refund_date = EXCLUDED.refund_date,
          refund_reconciliation = EXCLUDED.refund_reconciliation,
          has_returns = EXCLUDED.has_returns,
          raw_payload = EXCLUDED.raw_payload,
          order_number = COALESCE(commerce.orders.order_number, EXCLUDED.order_number), -- Zachowaj istniejący numer, jeśli NULL użyj nowego
          updated_at = NOW()
        RETURNING id, payment_amount AS returned_payment_amount
      `, insertParams);
    } else {
      // Istniejące zamówienie - UPDATE
      console.log(`🔄 Aktualizuję istniejące zamówienie ${source}-${sourceOrderId}`);
      
      // 🔢 Sprawdź czy zamówienie ma numer - jeśli nie, nadaj go teraz
      // TYLKO jeśli nie został już nadany wcześniej (w linii 1653)
      if (!nextOrderNumber) {
        // Użyj SELECT FOR UPDATE żeby zablokować wiersz i zapobiec race condition
        const checkNumberResult = await client.query(`
          SELECT order_number FROM commerce.orders
          WHERE source = $1 AND source_order_id = $2
          FOR UPDATE
        `, [source, sourceOrderId]);
        
        if (checkNumberResult.rows.length > 0 && !checkNumberResult.rows[0].order_number) {
          console.log(`⚠️  Zamówienie ${source}-${sourceOrderId} istnieje bez numeru - nadam numer teraz`);
          const { getNextOrderNumber } = await import('./order-numbering.js');
          // Przekazujemy id=0 bo zaktualizujemy później w audit
          nextOrderNumber = await getNextOrderNumber(client, 0, source, sourceOrderId);
          console.log(`🔢 [UPDATE-FIX] Zarezerwowano numer #${nextOrderNumber} dla ${source}-${sourceOrderId}`);
        }
      } else {
        console.log(`✅ Numer ${nextOrderNumber} został już nadany wcześniej - pomijam dublowanie`);
      }
      
      // Jeśli nadaliśmy numer, dodaj go do parametrów UPDATE
      const updateParams = nextOrderNumber ? [...orderParams, nextOrderNumber] : orderParams;
      const orderNumberClause = nextOrderNumber ? `, order_number = COALESCE(order_number, $41)` : '';
      
      orderResult = await client.query(`
        UPDATE commerce.orders SET
          order_date = CASE WHEN $1 = 'SHOPER' THEN timezone('Europe/Warsaw', $3::timestamp) ELSE $3::timestamptz END,
          status = $4,
          buyer_login = $5,
          buyer_email = $6,
          buyer_company = $7,
          buyer_first_name = $8,
          buyer_last_name = $9,
          buyer_phone = $10,
          buyer_address = $11,
          buyer_zip = $12,
          buyer_city = $13,
          buyer_country_code = $14,
          payment_id = $15,
          payment_status = $16,
          payment_type = $17,
          payment_provider = $18,
          payment_amount = $19,
          payment_currency = $20,
          delivery_method = $21,
          delivery_amount = $22,
          delivery_currency = $23,
          delivery_address = $24,
          billing_address = $25,
          billing_zip = $26,
          billing_city = $27,
          billing_country_code = $28,
          tax_id = $29,
          total_to_pay_amount = $30,
          total_to_pay_currency = $31,
          buyer_notes = $32,
          invoice_required = $33,
          tracking_numbers = $34,
          shipments = $35,
          refund_amount = $36,
          refund_date = $37,
          refund_reconciliation = $38,
          has_returns = $39,
          raw_payload = $40,
          updated_at = NOW()${orderNumberClause}
        WHERE source = $1 AND source_order_id = $2
        RETURNING id, payment_amount AS returned_payment_amount, order_number
      `, updateParams);
    }

    const commerceOrderId = orderResult.rows[0].id;
    console.log(`🔍 DEBUG - RETURNING after INSERT/UPDATE for ${source}-${sourceOrderId}:`);
    console.log(`   returned_payment_amount = ${orderResult.rows[0].returned_payment_amount}`);

    // 🔢 Sprawdź czy był faktyczny INSERT czy ON CONFLICT lub UPDATE
    if (nextOrderNumber) {
      if (isNew) {
        // Sprawdź czy zamówienie faktycznie zostało wstawione (rowCount > 0)
        // ON CONFLICT DO UPDATE również zwraca rowCount=1, więc musimy sprawdzić czy order_number się zmienił
        const actuallyInserted = await client.query(`
          SELECT order_number FROM commerce.orders WHERE id = $1
        `, [commerceOrderId]);
        
        const finalOrderNumber = actuallyInserted.rows[0]?.order_number;
        
        if (finalOrderNumber === nextOrderNumber) {
          // Faktyczny INSERT - zaktualizuj audit
          await client.query(`
            UPDATE commerce.order_number_audit 
            SET order_id = $1
            WHERE order_number = $2 AND order_id = 0
          `, [commerceOrderId, nextOrderNumber]);
          console.log(`🔢 [POST-INSERT] Zaktualizowano audit dla #${nextOrderNumber} z order_id=${commerceOrderId}`);
        } else {
          // ON CONFLICT - numer nie został użyty, usuń audit i wycofaj sekwencję
          await client.query(`
            DELETE FROM commerce.order_number_audit 
            WHERE order_number = $1 AND order_id = 0
          `, [nextOrderNumber]);
          
          await client.query(`
            UPDATE commerce.order_sequences 
            SET current_value = current_value - 1
            WHERE sequence_name = 'main_order_sequence'
          `);
          
          console.log(`🔢 [ROLLBACK] Wycofano numer #${nextOrderNumber} (ON CONFLICT), zamówienie ma numer #${finalOrderNumber}`);
        }
      } else {
        // UPDATE z nadaniem numeru - sprawdź czy faktycznie został użyty
        const finalOrderNumber = orderResult.rows[0]?.order_number;
        
        if (finalOrderNumber === nextOrderNumber) {
          // Numer faktycznie został użyty - zaktualizuj audit
          await client.query(`
            UPDATE commerce.order_number_audit 
            SET order_id = $1
            WHERE order_number = $2 AND order_id = 0
          `, [commerceOrderId, nextOrderNumber]);
          console.log(`🔢 [POST-UPDATE] Zaktualizowano audit dla #${nextOrderNumber} z order_id=${commerceOrderId}`);
        } else {
          // Numer nie został użyty (race condition) - wycofaj audit i sekwencję
          await client.query(`
            DELETE FROM commerce.order_number_audit 
            WHERE order_number = $1 AND order_id = 0
          `, [nextOrderNumber]);
          
          await client.query(`
            UPDATE commerce.order_sequences 
            SET current_value = current_value - 1
            WHERE sequence_name = 'main_order_sequence'
          `);
          
          console.log(`🔢 [ROLLBACK-UPDATE] Wycofano numer #${nextOrderNumber} (race condition), zamówienie ma numer #${finalOrderNumber}`);
        }
      }
    }

    // Najpierw wstaw wszystkie order_items bez returns_quantity
    for (const item of lineItems) {
      let catalogProductId = null; // DEPRECATED - no longer used, marketplace products go to commerce.marketplace_products
      let platformProductId = null; // ID z platformy (VARCHAR)
      let externalId, itemName, quantity, unitPrice, totalPrice, imageUrl;

      if (source === 'ALLEGRO') {
        // Use external.id if available, otherwise use offer.id as fallback
        externalId = item.offer?.external?.id || item.offer?.id;
        itemName = item.offer?.name;
        quantity = item.quantity;
        unitPrice = item.price?.amount;
        totalPrice = parseFloat(unitPrice) * quantity;
        imageUrl = item.imageUrl || item.offer?.images?.[0]?.url;
        platformProductId = item.offer?.id; // Allegro offer ID
      } else if (source === 'SHOPER') {
        externalId = item.code || item.product_id?.toString();
        itemName = item.name || item.translation?.name;
        quantity = item.quantity;
        unitPrice = item.price || item.price_brutto;
        totalPrice = parseFloat(unitPrice) * quantity;
        // Prefer local image (from enrichShoperOrderWithImages) over platform URL
        imageUrl = item.image_url || item.main_image?.url;
        platformProductId = item.product_id?.toString(); // Shoper product ID
      }

      // DISABLED: Old code that created products in catalog.products
      // Now marketplace products are aggregated in commerce.marketplace_products via /api/marketplace-products/sync
      // catalogProductId remains NULL for all new orders

      await client.query(`
        INSERT INTO commerce.order_items (
          order_id, product_id, offer_external_id, name,
          quantity, unit_price, price, image_url,
          returns_quantity, raw_data
        ) VALUES (
          $1, $2, $3, $4, $5, $6, $7, $8, $9, $10
        )
        ON CONFLICT (order_id, offer_external_id) DO UPDATE SET
          quantity = EXCLUDED.quantity,
          unit_price = EXCLUDED.unit_price,
          price = EXCLUDED.price,
          image_url = EXCLUDED.image_url,
          raw_data = EXCLUDED.raw_data
      `, [
        commerceOrderId,
        platformProductId,
        externalId,
        itemName,
        quantity,
        unitPrice,
        totalPrice,
        imageUrl,
        0,
        JSON.stringify(item)
      ]);
      
      // Insert or update marketplace product
      if (externalId && itemName) {
        await client.query(`
          INSERT INTO commerce.marketplace_products (
            offer_external_id, source, name, image_url, sku,
            last_sold_at, times_sold, total_quantity_sold, avg_unit_price
          ) VALUES (
            $1, $2, $3, $4, $5, NOW(), 1, $6, $7
          )
          ON CONFLICT (offer_external_id, source) DO UPDATE SET
            name = COALESCE(EXCLUDED.name, commerce.marketplace_products.name),
            image_url = COALESCE(EXCLUDED.image_url, commerce.marketplace_products.image_url),
            sku = COALESCE(EXCLUDED.sku, commerce.marketplace_products.sku),
            last_sold_at = NOW(),
            times_sold = commerce.marketplace_products.times_sold + 1,
            total_quantity_sold = commerce.marketplace_products.total_quantity_sold + $8::numeric,
            avg_unit_price = (
              (commerce.marketplace_products.avg_unit_price * commerce.marketplace_products.total_quantity_sold + $9::numeric * $10::numeric) / 
              NULLIF(commerce.marketplace_products.total_quantity_sold + $11::numeric, 0)
            ),
            updated_at = NOW()
        `, [
          externalId,
          source,
          itemName,
          imageUrl,
          externalId, // SKU = externalId for now
          quantity,
          unitPrice,
          quantity, // for total_quantity_sold update
          unitPrice, // for avg calculation
          quantity, // for avg calculation
          quantity  // for avg calculation
        ]);
      }
    }

    // Aktualizuj returns_quantity na podstawie refund_reconciliation (bardziej stabilne dane)
    // ID w refund_reconciliation.lineItems to checkoutFormLineItemId (line_item_id z Allegro)
    if (source === 'ALLEGRO' && refundReconciliation && Array.isArray(refundReconciliation)) {
      for (const refund of refundReconciliation) {
        const refundLineItems = refund.lineItems || [];
        for (const refundLineItem of refundLineItems) {
          if (refundLineItem.id && refundLineItem.quantity) {
            // ID w refund_reconciliation to line_item_id (checkoutFormLineItemId)
            // Musimy znaleźć offer_external_id poprzez allegro.order_items
            await client.query(`
              UPDATE commerce.order_items
              SET returns_quantity = $1
              WHERE order_id = $2
                AND offer_external_id IN (
                  SELECT external_id 
                  FROM allegro.order_items 
                  WHERE line_item_id = $3 
                    AND order_id = $4
                )
            `, [refundLineItem.quantity, commerceOrderId, refundLineItem.id, sourceOrderId]);
          }
        }
      }
    }

    await client.query('COMMIT');
    
    console.log(`✅ Order ${sourceOrderId} saved to commerce.orders (${source})`);
    
    return { isNew };
  } catch (error) {
    await client.query('ROLLBACK');
    console.error(`❌ Error saving order ${sourceOrderId} to commerce:`, error);
    throw error;
  } finally {
    client.release();
  }
}

export async function logOrderChange(
  orderId: string,
  orderNumber: number,
  source: string,
  changeType: 'NEW' | 'UPDATED',
  fieldChanged?: string,
  oldValue?: string,
  newValue?: string
) {
  // Uruchom asynchronicznie, nie blokuj głównego procesu
  setImmediate(async () => {
    const client = await pool.connect();
    
    try {
      // Ustaw timeout na 3 sekundy dla zapytań
      await client.query('SET statement_timeout = 3000');
      
      // Sprawdź czy taka sama zmiana już istnieje (ostatnie 5 minut)
      if (fieldChanged && newValue !== undefined) {
        const existingChange = await client.query(`
          SELECT id FROM public.order_changes
          WHERE order_id = $1
            AND field_changed = $2
            AND new_value = $3
            AND detected_at >= NOW() - INTERVAL '5 minutes'
          LIMIT 1
        `, [String(orderNumber), fieldChanged, newValue]);
        
        if (existingChange.rows.length > 0) {
          console.log(`⏭️  Pominięto duplikat zmiany: #${orderNumber} - ${fieldChanged}`);
          return; // Nie zapisuj duplikatu
        }
      }
      
      const orderNumString = String(orderNumber);
      await client.query(`
        INSERT INTO public.order_changes (
          order_id, order_number, source, change_type, 
          field_changed, old_value, new_value, detected_at
        ) VALUES (
          $1, $2, $3, $4, $5, $6, $7, NOW()
        )
      `, [orderNumString, orderNumString, source, changeType, fieldChanged, oldValue, newValue]);
      
      console.log(`✅ Zapisano zmianę: #${orderNumber} - ${fieldChanged || changeType}`);
    } catch (error) {
      console.error(`❌ Error logging order change #${orderNumber}:`, error);
    } finally {
      client.release();
    }
  });
}

export async function getRecentOrderChanges(limitMinutes: number = 60) {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      SELECT 
        oc.*,
        o.order_number,
        o.source,
        o.total_to_pay_amount,
        o.total_to_pay_currency,
        o.status,
        o.buyer_login,
        o.order_date
      FROM public.order_changes oc
      LEFT JOIN commerce.orders o ON oc.order_id::integer = o.id
      WHERE oc.detected_at >= NOW() - INTERVAL '${limitMinutes} minutes'
      ORDER BY oc.detected_at DESC
      LIMIT 100
    `);
    
    return result.rows;
  } catch (error) {
    console.error('❌ Error fetching recent order changes:', error);
    return [];
  } finally {
    client.release();
  }
}

export async function getRecentlyUpdatedOrders(timeWindowMinutes: number = 10) {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      SELECT 
        o.*,
        CASE 
          WHEN o.created_at >= NOW() - INTERVAL '${timeWindowMinutes} minutes' THEN 'NEW'
          ELSE 'UPDATED'
        END as change_type
      FROM commerce.orders o
      WHERE o.updated_at >= NOW() - INTERVAL '${timeWindowMinutes} minutes'
         OR o.created_at >= NOW() - INTERVAL '${timeWindowMinutes} minutes'
      ORDER BY 
        CASE 
          WHEN o.updated_at > o.created_at THEN o.updated_at 
          ELSE o.created_at 
        END DESC
      LIMIT 100
    `);
    
    return result.rows;
  } catch (error) {
    console.error('❌ Error fetching recently updated orders:', error);
    return [];
  } finally {
    client.release();
  }
}

export async function getSyncSettingsFromPostgres() {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      SELECT * FROM sync_settings
      ORDER BY created_at DESC
      LIMIT 1
    `);
    
    if (result.rows.length === 0) {
      return null;
    }
    
    const row = result.rows[0];
    return {
      id: row.id,
      autoRefreshEnabled: row.auto_refresh_enabled,
      refreshIntervalMinutes: row.refresh_interval_minutes,
      fullSyncEnabled: row.full_sync_enabled,
      fullSyncWindowHours: row.full_sync_window_hours,
      lastSyncAt: row.last_sync_at,
      shoperAutoRefreshEnabled: row.shoper_auto_refresh_enabled,
      shoperRefreshIntervalMinutes: row.shoper_refresh_interval_minutes,
      shoperFullSyncEnabled: row.shoper_full_sync_enabled,
      shoperFullSyncWindowHours: row.shoper_full_sync_window_hours,
      lastShoperSyncAt: row.last_shoper_sync_at,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  } catch (error) {
    console.error('❌ Error fetching sync settings:', error);
    return null;
  } finally {
    client.release();
  }
}

export async function createOrUpdateSyncSettingsInPostgres(settings: any) {
  const client = await pool.connect();
  
  try {
    // Sprawdź czy istnieją ustawienia
    const existing = await client.query(`
      SELECT id FROM sync_settings LIMIT 1
    `);
    
    if (existing.rows.length === 0) {
      // Utwórz nowe
      const result = await client.query(`
        INSERT INTO sync_settings (
          auto_refresh_enabled,
          refresh_interval_minutes,
          last_sync_at,
          shoper_auto_refresh_enabled,
          shoper_refresh_interval_minutes,
          last_shoper_sync_at,
          created_at,
          updated_at
        ) VALUES ($1, $2, $3, $4, $5, $6, NOW(), NOW())
        RETURNING *
      `, [
        settings.autoRefreshEnabled ?? true,
        settings.refreshIntervalMinutes ?? '3',
        settings.lastSyncAt ?? null,
        settings.shoperAutoRefreshEnabled ?? true,
        settings.shoperRefreshIntervalMinutes ?? '5',
        settings.lastShoperSyncAt ?? null,
      ]);
      
      const row = result.rows[0];
      return {
        id: row.id,
        autoRefreshEnabled: row.auto_refresh_enabled,
        refreshIntervalMinutes: row.refresh_interval_minutes,
        lastSyncAt: row.last_sync_at,
        shoperAutoRefreshEnabled: row.shoper_auto_refresh_enabled,
        shoperRefreshIntervalMinutes: row.shoper_refresh_interval_minutes,
        lastShoperSyncAt: row.last_shoper_sync_at,
        createdAt: row.created_at,
        updatedAt: row.updated_at,
      };
    } else {
      // Zaktualizuj istniejące
      const result = await client.query(`
        UPDATE sync_settings
        SET 
          auto_refresh_enabled = COALESCE($1, auto_refresh_enabled),
          refresh_interval_minutes = COALESCE($2, refresh_interval_minutes),
          last_sync_at = COALESCE($3, last_sync_at),
          shoper_auto_refresh_enabled = COALESCE($4, shoper_auto_refresh_enabled),
          shoper_refresh_interval_minutes = COALESCE($5, shoper_refresh_interval_minutes),
          last_shoper_sync_at = COALESCE($6, last_shoper_sync_at),
          updated_at = NOW()
        WHERE id = $7
        RETURNING *
      `, [
        settings.autoRefreshEnabled,
        settings.refreshIntervalMinutes,
        settings.lastSyncAt,
        settings.shoperAutoRefreshEnabled,
        settings.shoperRefreshIntervalMinutes,
        settings.lastShoperSyncAt,
        existing.rows[0].id,
      ]);
      
      const row = result.rows[0];
      return {
        id: row.id,
        autoRefreshEnabled: row.auto_refresh_enabled,
        refreshIntervalMinutes: row.refresh_interval_minutes,
        lastSyncAt: row.last_sync_at,
        shoperAutoRefreshEnabled: row.shoper_auto_refresh_enabled,
        shoperRefreshIntervalMinutes: row.shoper_refresh_interval_minutes,
        lastShoperSyncAt: row.last_shoper_sync_at,
        createdAt: row.created_at,
        updatedAt: row.updated_at,
      };
    }
  } catch (error) {
    console.error('❌ Error creating/updating sync settings:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function getOrderStatistics() {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      SELECT 
        COUNT(*) as total_orders,
        COUNT(CASE 
          WHEN COALESCE(payment_amount, 0)::numeric > 0 THEN 1 
        END) as paid_orders,
        COUNT(CASE 
          WHEN COALESCE(payment_amount, 0)::numeric = 0 THEN 1 
        END) as unpaid_orders,
        COALESCE(SUM(COALESCE(total_to_pay_amount, 0)::numeric), 0) as total_revenue
      FROM commerce.orders
    `);
    
    const stats = result.rows[0];
    return {
      totalOrders: parseInt(stats.total_orders) || 0,
      paidOrders: parseInt(stats.paid_orders) || 0,
      unpaidOrders: parseInt(stats.unpaid_orders) || 0,
      totalRevenue: parseFloat(stats.total_revenue) || 0,
    };
  } catch (error) {
    console.error('❌ Error fetching order statistics:', error);
    return {
      totalOrders: 0,
      paidOrders: 0,
      unpaidOrders: 0,
      totalRevenue: 0,
    };
  } finally {
    client.release();
  }
}

export async function getOrdersChartData(days: number = 30) {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      WITH daily_stats AS (
        SELECT 
          DATE(order_date) as date,
          COUNT(*) as orders,
          COUNT(CASE WHEN COALESCE(payment_amount, 0) > 0 THEN 1 END) as paid_orders,
          COALESCE(SUM(COALESCE(total_to_pay_amount, 0)), 0) as revenue
        FROM commerce.orders
        WHERE order_date >= NOW() - INTERVAL '${days} days'
        GROUP BY DATE(order_date)
      )
      SELECT 
        TO_CHAR(date, 'YYYY-MM-DD') as date,
        orders,
        paid_orders,
        revenue
      FROM daily_stats
      ORDER BY date ASC
    `);
    
    return result.rows.map(row => ({
      date: row.date,
      orders: parseInt(row.orders) || 0,
      paidOrders: parseInt(row.paid_orders) || 0,
      revenue: parseFloat(row.revenue) || 0,
    }));
  } catch (error) {
    console.error('❌ Error fetching chart data:', error);
    return [];
  } finally {
    client.release();
  }
}

export async function getOrdersPeriodSummary(days: number = 30) {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      WITH period_stats AS (
        SELECT 
          -- Przychód całkowity za okres
          COALESCE(SUM(COALESCE(total_to_pay_amount, 0)), 0) as revenue_total,
          -- Zapłacono (suma payment_amount)
          COALESCE(SUM(COALESCE(payment_amount, 0)), 0) as paid_amount,
          -- Przychód z Allegro
          COALESCE(SUM(CASE WHEN source = 'ALLEGRO' THEN COALESCE(total_to_pay_amount, 0) ELSE 0 END), 0) as revenue_allegro,
          -- Przychód ze Shoper
          COALESCE(SUM(CASE WHEN source = 'SHOPER' THEN COALESCE(total_to_pay_amount, 0) ELSE 0 END), 0) as revenue_shoper
        FROM commerce.orders
        WHERE order_date >= NOW() - INTERVAL '${days} days'
      ),
      today_stats AS (
        SELECT 
          COALESCE(SUM(COALESCE(total_to_pay_amount, 0)), 0) as revenue_today
        FROM commerce.orders
        WHERE DATE(order_date) = CURRENT_DATE
      )
      SELECT 
        p.revenue_total,
        p.paid_amount,
        p.revenue_allegro,
        p.revenue_shoper,
        t.revenue_today
      FROM period_stats p, today_stats t
    `);
    
    const stats = result.rows[0] || {};
    return {
      revenueTotal: parseFloat(stats.revenue_total) || 0,
      paidAmount: parseFloat(stats.paid_amount) || 0,
      revenueAllegro: parseFloat(stats.revenue_allegro) || 0,
      revenueShoper: parseFloat(stats.revenue_shoper) || 0,
      revenueToday: parseFloat(stats.revenue_today) || 0,
    };
  } catch (error) {
    console.error('❌ Error fetching period summary:', error);
    return {
      revenueTotal: 0,
      paidAmount: 0,
      revenueAllegro: 0,
      revenueShoper: 0,
      revenueToday: 0,
    };
  } finally {
    client.release();
  }
}

export async function getTodayDetailedStats() {
  const client = await pool.connect();
  
  try {
    const result = await client.query(`
      WITH today_stats AS (
        SELECT 
          -- Dzisiejszy przychód
          COALESCE(SUM(COALESCE(total_to_pay_amount, 0)), 0) as revenue_today,
          -- Zapłacono dzisiaj (suma payment_amount)
          COALESCE(SUM(COALESCE(payment_amount, 0)), 0) as paid_today,
          -- Pobrania (COD) - liczba i kwota
          COUNT(CASE WHEN payment_type = 'CASH_ON_DELIVERY' THEN 1 END) as cod_count,
          COALESCE(SUM(CASE WHEN payment_type = 'CASH_ON_DELIVERY' THEN COALESCE(total_to_pay_amount, 0) ELSE 0 END), 0) as cod_amount,
          -- Niezapłacono (poza pobraniami)
          COALESCE(SUM(CASE WHEN payment_type != 'CASH_ON_DELIVERY' AND COALESCE(payment_amount, 0) = 0 THEN COALESCE(total_to_pay_amount, 0) ELSE 0 END), 0) as unpaid_non_cod,
          -- Zapłacono Allegro
          COALESCE(SUM(CASE WHEN source = 'ALLEGRO' AND COALESCE(payment_amount, 0) > 0 THEN COALESCE(payment_amount, 0) ELSE 0 END), 0) as paid_allegro,
          -- Zapłacono Shoper
          COALESCE(SUM(CASE WHEN source = 'SHOPER' AND COALESCE(payment_amount, 0) > 0 THEN COALESCE(payment_amount, 0) ELSE 0 END), 0) as paid_shoper,
          -- Liczba zamówień ogółem
          COUNT(*) as orders_count,
          -- Liczba zamówień Allegro
          COUNT(CASE WHEN source = 'ALLEGRO' THEN 1 END) as orders_allegro,
          -- Liczba zamówień Shoper
          COUNT(CASE WHEN source = 'SHOPER' THEN 1 END) as orders_shoper
        FROM commerce.orders
        WHERE DATE(order_date) = CURRENT_DATE
      ),
      yesterday_stats AS (
        SELECT 
          COALESCE(SUM(COALESCE(total_to_pay_amount, 0)), 0) as revenue_yesterday
        FROM commerce.orders
        WHERE DATE(order_date) = CURRENT_DATE - INTERVAL '1 day'
      )
      SELECT 
        t.*,
        y.revenue_yesterday,
        TO_CHAR(CURRENT_DATE, 'Day') as day_name,
        TO_CHAR(CURRENT_DATE, 'DD.MM.YYYY') as date_formatted,
        CASE 
          WHEN y.revenue_yesterday > 0 THEN 
            ROUND(((t.revenue_today - y.revenue_yesterday) / y.revenue_yesterday * 100)::numeric, 1)
          ELSE 0
        END as revenue_change_percent
      FROM today_stats t, yesterday_stats y
    `);
    
    const stats = result.rows[0] || {};
    return {
      revenueToday: parseFloat(stats.revenue_today) || 0,
      paidToday: parseFloat(stats.paid_today) || 0,
      codCount: parseInt(stats.cod_count) || 0,
      codAmount: parseFloat(stats.cod_amount) || 0,
      unpaidNonCod: parseFloat(stats.unpaid_non_cod) || 0,
      paidAllegro: parseFloat(stats.paid_allegro) || 0,
      paidShoper: parseFloat(stats.paid_shoper) || 0,
      ordersCount: parseInt(stats.orders_count) || 0,
      ordersAllegro: parseInt(stats.orders_allegro) || 0,
      ordersShoper: parseInt(stats.orders_shoper) || 0,
      revenueYesterday: parseFloat(stats.revenue_yesterday) || 0,
      dayName: (stats.day_name || '').trim(),
      dateFormatted: stats.date_formatted || '',
      revenueChangePercent: parseFloat(stats.revenue_change_percent) || 0,
    };
  } catch (error) {
    console.error('❌ Error fetching today detailed stats:', error);
    return {
      revenueToday: 0,
      paidToday: 0,
      codCount: 0,
      codAmount: 0,
      unpaidNonCod: 0,
      paidAllegro: 0,
      paidShoper: 0,
      ordersCount: 0,
      ordersAllegro: 0,
      ordersShoper: 0,
      revenueYesterday: 0,
      dayName: '',
      dateFormatted: '',
      revenueChangePercent: 0,
    };
  } finally {
    client.release();
  }
}

export async function getAllegroConnectionFromPostgres(): Promise<any | null> {
  const client = await connectWithRetry();
  try {
    const result = await client.query(`
      SELECT id, client_id, client_secret, access_token, refresh_token, token_expires_at, is_active, created_at, updated_at
      FROM allegro_connections
      ORDER BY created_at DESC
      LIMIT 1
    `);
    
    if (result.rows.length === 0) {
      return null;
    }
    
    const row = result.rows[0];
    return {
      id: row.id,
      clientId: row.client_id,
      clientSecret: row.client_secret,
      accessToken: row.access_token,
      refreshToken: row.refresh_token,
      tokenExpiresAt: row.token_expires_at,
      isActive: row.is_active,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  } catch (error) {
    console.error('❌ Error fetching Allegro connection from PostgreSQL:', error);
    return null;
  } finally {
    client.release();
  }
}

export async function saveAllegroConnectionToPostgres(connection: {
  clientId: string;
  clientSecret: string;
  accessToken?: string;
  refreshToken?: string;
  tokenExpiresAt?: Date;
  isActive?: boolean;
}): Promise<void> {
  const client = await pool.connect();
  try {
    // Check if connection exists
    const existingResult = await client.query(`
      SELECT id FROM allegro_connections ORDER BY created_at DESC LIMIT 1
    `);
    
    if (existingResult.rows.length > 0) {
      // Update existing connection
      await client.query(`
        UPDATE allegro_connections
        SET client_id = $1, client_secret = $2, access_token = $3, refresh_token = $4, 
            token_expires_at = $5, is_active = $6, updated_at = NOW()
        WHERE id = $7
      `, [
        connection.clientId,
        connection.clientSecret,
        connection.accessToken || null,
        connection.refreshToken || null,
        connection.tokenExpiresAt || null,
        connection.isActive !== undefined ? connection.isActive : true,
        existingResult.rows[0].id,
      ]);
      console.log('✅ Updated Allegro connection in PostgreSQL');
    } else {
      // Insert new connection
      await client.query(`
        INSERT INTO allegro_connections (
          client_id, client_secret, access_token, refresh_token, token_expires_at, is_active, created_at, updated_at
        ) VALUES ($1, $2, $3, $4, $5, $6, NOW(), NOW())
      `, [
        connection.clientId,
        connection.clientSecret,
        connection.accessToken || null,
        connection.refreshToken || null,
        connection.tokenExpiresAt || null,
        connection.isActive !== undefined ? connection.isActive : true,
      ]);
      console.log('✅ Inserted Allegro connection to PostgreSQL');
    }
  } catch (error) {
    console.error('❌ Error saving Allegro connection to PostgreSQL:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function updateAllegroConnectionTokens(
  accessToken: string,
  refreshToken: string,
  expiresAt: Date
): Promise<void> {
  const client = await pool.connect();
  try {
    await client.query(`
      UPDATE allegro_connections
      SET access_token = $1, refresh_token = $2, token_expires_at = $3, is_active = true, updated_at = NOW()
      WHERE id = (SELECT id FROM allegro_connections ORDER BY created_at DESC LIMIT 1)
    `, [accessToken, refreshToken, expiresAt]);
    
    console.log('✅ Updated Allegro connection tokens in PostgreSQL');
  } catch (error) {
    console.error('❌ Error updating Allegro connection tokens in PostgreSQL:', error);
    throw error;
  } finally {
    client.release();
  }
}

// renumberOrdersByDate() USUNIĘTA - id jest teraz SERIAL autoincrement od 1, nie wymaga renumeracji

// ========================================
// Auth Functions
// ========================================

export async function getUserByUsername(username: string) {
  const result = await pool.query(
    'SELECT * FROM users WHERE username = $1',
    [username]
  );
  return result.rows[0];
}

export async function getUserById(id: number) {
  const result = await pool.query(
    'SELECT * FROM users WHERE id = $1',
    [id]
  );
  return result.rows[0];
}

export async function getUserByEmail(email: string) {
  const result = await pool.query(
    'SELECT * FROM users WHERE email = $1',
    [email]
  );
  return result.rows[0];
}

export async function createUser(userData: { username: string; email: string; password: string; firstName?: string; lastName?: string }) {
  const result = await pool.query(
    `INSERT INTO users (username, email, password, first_name, last_name, is_active, created_at, updated_at)
     VALUES ($1, $2, $3, $4, $5, true, NOW(), NOW())
     RETURNING *`,
    [userData.username, userData.email, userData.password, userData.firstName || null, userData.lastName || null]
  );
  return result.rows[0];
}

export async function createPasswordResetToken(userId: number, token: string, expiresAt: Date) {
  const result = await pool.query(
    `INSERT INTO password_reset_tokens (user_id, token, expires_at, used, created_at)
     VALUES ($1, $2, $3, false, NOW())
     RETURNING *`,
    [userId, token, expiresAt]
  );
  return result.rows[0];
}

export async function getPasswordResetToken(token: string) {
  const result = await pool.query(
    `SELECT * FROM password_reset_tokens 
     WHERE token = $1 AND used = false AND expires_at > NOW()`,
    [token]
  );
  return result.rows[0];
}

export async function markTokenAsUsed(token: string) {
  await pool.query(
    'UPDATE password_reset_tokens SET used = true WHERE token = $1',
    [token]
  );
}

export async function updateUserPassword(userId: number, newPassword: string) {
  await pool.query(
    'UPDATE users SET password = $1, updated_at = NOW() WHERE id = $2',
    [newPassword, userId]
  );
}

export async function deleteExpiredTokens() {
  await pool.query(
    'DELETE FROM password_reset_tokens WHERE expires_at < NOW()'
  );
}

export async function getAllUsers() {
  const result = await pool.query(
    `SELECT id, username, email, first_name, last_name, role, permissions, is_active, created_at, updated_at 
     FROM users 
     ORDER BY created_at DESC`
  );
  return result.rows;
}

export async function updateUserRole(userId: number, role: string) {
  const result = await pool.query(
    `UPDATE users 
     SET role = $1, updated_at = NOW() 
     WHERE id = $2
     RETURNING id, username, email, first_name, last_name, role, permissions, is_active, created_at, updated_at`,
    [role, userId]
  );
  return result.rows[0];
}

export async function updateUserPermissions(userId: number, permissions: string[]) {
  const result = await pool.query(
    `UPDATE users 
     SET permissions = $1, updated_at = NOW() 
     WHERE id = $2
     RETURNING id, username, email, first_name, last_name, role, permissions, is_active, created_at, updated_at`,
    [JSON.stringify(permissions), userId]
  );
  return result.rows[0];
}

export async function updateUserStatus(userId: number, isActive: boolean) {
  const result = await pool.query(
    `UPDATE users 
     SET is_active = $1, updated_at = NOW() 
     WHERE id = $2
     RETURNING id, username, email, first_name, last_name, role, permissions, is_active, created_at, updated_at`,
    [isActive, userId]
  );
  return result.rows[0];
}

export async function deleteUser(userId: number) {
  await pool.query('DELETE FROM users WHERE id = $1', [userId]);
}

/**
 * Renumeruje wszystkie zamówienia chronologicznie według order_date
 * Usuwa luki w numeracji - zamówienia będą miały ID: 1, 2, 3, 4...
 */
export async function renumberOrdersByDate() {
  console.log('🔄 Rozpoczynam renumerację zamówień według order_date...');
  
  try {
    // Użyj WITH CTE do renumeracji wszystkich powiązanych tabel w jednym zapytaniu
    const result = await pool.query(`
      WITH numbered AS (
        SELECT 
          id as old_id,
          ROW_NUMBER() OVER (ORDER BY order_date ASC, id ASC)::integer as new_id
        FROM commerce.orders
      ),
      update_order_items AS (
        UPDATE commerce.order_items oi
        SET order_id = n.new_id
        FROM numbered n
        WHERE oi.order_id = n.old_id
        RETURNING 1
      ),
      update_queue AS (
        UPDATE odoo_sync_queue q
        SET order_id = n.new_id
        FROM numbered n
        WHERE q.order_id = n.old_id
        RETURNING 1
      ),
      update_changes AS (
        UPDATE commerce.order_changes c
        SET order_id = n.new_id
        FROM numbered n
        WHERE c.order_id = n.old_id
        RETURNING 1
      ),
      update_orders AS (
        UPDATE commerce.orders o
        SET id = n.new_id
        FROM numbered n
        WHERE o.id = n.old_id
        RETURNING o.id
      )
      SELECT MAX(id) as max_id FROM update_orders
    `);
    
    const maxId = result.rows[0]?.max_id || 0;
    
    // Zresetuj sequence
    await pool.query(`SELECT setval('commerce.orders_id_seq', $1)`, [maxId]);
    
    console.log(`✅ Renumeracja zakończona! Zamówienia: 1-${maxId}, sequence ustawiony na ${maxId}`);
    return maxId;
  } catch (error) {
    console.error('❌ Błąd podczas renumeracji zamówień:', error);
    throw error;
  }
}

// Funkcja naprawy zamówień bez order_number
export async function fixOrdersWithoutNumbers(): Promise<number> {
  const client = await pool.connect();
  
  try {
    // Znajdź zamówienia bez order_number
    const ordersWithoutNumber = await client.query(`
      SELECT id, source, source_order_id, order_date
      FROM commerce.orders
      WHERE order_number IS NULL
      ORDER BY order_date ASC
    `);
    
    if (ordersWithoutNumber.rows.length === 0) {
      return 0; // Brak zamówień do naprawy
    }
    
    console.log(`🔧 Znaleziono ${ordersWithoutNumber.rows.length} zamówień bez order_number - naprawiam...`);
    
    let fixed = 0;
    
    for (const order of ordersWithoutNumber.rows) {
      await client.query('BEGIN');
      
      try {
        // Pobierz następny numer z sekwencji
        const { getNextOrderNumber } = await import('./order-numbering.js');
        const nextNumber = await getNextOrderNumber(client, order.id, order.source, order.source_order_id);
        
        // Przypisz numer zamówienia
        await client.query(`
          UPDATE commerce.orders
          SET order_number = $1
          WHERE id = $2
        `, [nextNumber, order.id]);
        
        console.log(`✅ Naprawiono zamówienie ID:${order.id} → #${nextNumber}`);
        fixed++;
        
        await client.query('COMMIT');
      } catch (error) {
        await client.query('ROLLBACK');
        console.error(`❌ Błąd podczas naprawy zamówienia ID:${order.id}:`, error);
      }
    }
    
    console.log(`✅ Naprawiono ${fixed}/${ordersWithoutNumber.rows.length} zamówień`);
    return fixed;
  } catch (error) {
    console.error('❌ Błąd w fixOrdersWithoutNumbers():', error);
    throw error;
  } finally {
    client.release();
  }
}

// ==================== TEMPLATE CATEGORIES ====================

export async function getAllTemplateCategories() {
  const client = await pool.connect();
  try {
    const result = await client.query(`
      SELECT id, name, description, sort_order, created_at, updated_at
      FROM description_template_categories
      ORDER BY sort_order ASC, name ASC
    `);
    
    return result.rows.map(row => ({
      id: row.id,
      name: row.name,
      description: row.description,
      sortOrder: row.sort_order,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    }));
  } catch (error) {
    console.error('❌ Error fetching template categories:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function getTemplateCategoryById(id: number) {
  const client = await pool.connect();
  try {
    const result = await client.query(`
      SELECT id, name, description, sort_order, created_at, updated_at
      FROM description_template_categories
      WHERE id = $1
    `, [id]);
    
    if (result.rows.length === 0) {
      return null;
    }
    
    const row = result.rows[0];
    return {
      id: row.id,
      name: row.name,
      description: row.description,
      sortOrder: row.sort_order,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  } catch (error) {
    console.error('❌ Error fetching template category:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function createTemplateCategory(data: {
  name: string;
  description?: string;
  sortOrder?: number;
}) {
  const client = await pool.connect();
  try {
    const result = await client.query(`
      INSERT INTO description_template_categories (name, description, sort_order, created_at, updated_at)
      VALUES ($1, $2, $3, NOW(), NOW())
      RETURNING id, name, description, sort_order, created_at, updated_at
    `, [data.name, data.description || null, data.sortOrder || 0]);
    
    const row = result.rows[0];
    return {
      id: row.id,
      name: row.name,
      description: row.description,
      sortOrder: row.sort_order,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  } catch (error) {
    console.error('❌ Error creating template category:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function updateTemplateCategory(id: number, data: {
  name?: string;
  description?: string;
  sortOrder?: number;
}) {
  const client = await pool.connect();
  try {
    const result = await client.query(`
      UPDATE description_template_categories
      SET name = COALESCE($1, name),
          description = COALESCE($2, description),
          sort_order = COALESCE($3, sort_order),
          updated_at = NOW()
      WHERE id = $4
      RETURNING id, name, description, sort_order, created_at, updated_at
    `, [data.name, data.description, data.sortOrder, id]);
    
    if (result.rows.length === 0) {
      return null;
    }
    
    const row = result.rows[0];
    return {
      id: row.id,
      name: row.name,
      description: row.description,
      sortOrder: row.sort_order,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  } catch (error) {
    console.error('❌ Error updating template category:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function deleteTemplateCategory(id: number) {
  const client = await pool.connect();
  try {
    const result = await client.query(`
      DELETE FROM description_template_categories WHERE id = $1 RETURNING id
    `, [id]);
    
    return result.rows.length > 0;
  } catch (error) {
    console.error('❌ Error deleting template category:', error);
    throw error;
  } finally {
    client.release();
  }
}

// ==================== DESCRIPTION TEMPLATES ====================

export async function getAllDescriptionTemplates(categoryId?: number) {
  const client = await pool.connect();
  try {
    let query = `
      SELECT t.id, t.name, t.category_id, t.template_type, t.content_doc, 
             t.html_content, t.variables, t.created_by, t.is_global, t.is_active,
             t.created_at, t.updated_at,
             c.name as category_name,
             u.username as created_by_username
      FROM catalog.description_templates t
      LEFT JOIN description_template_categories c ON t.category_id = c.id
      LEFT JOIN users u ON t.created_by = u.id
    `;
    
    const params: any[] = [];
    if (categoryId !== undefined) {
      query += ` WHERE t.category_id = $1`;
      params.push(categoryId);
    }
    
    query += ` ORDER BY t.is_global DESC, t.name ASC`;
    
    const result = await client.query(query, params);
    
    return result.rows.map(row => ({
      id: row.id,
      name: row.name,
      categoryId: row.category_id,
      categoryName: row.category_name,
      templateType: row.template_type,
      contentDoc: row.content_doc, // Already JSON from JSONB
      htmlContent: row.html_content,
      variables: row.variables, // Already JSON from JSONB
      createdBy: row.created_by,
      createdByUsername: row.created_by_username,
      isGlobal: row.is_global,
      isActive: row.is_active,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    }));
  } catch (error) {
    console.error('❌ Error fetching description templates:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function getDescriptionTemplateById(id: number) {
  const client = await pool.connect();
  try {
    const result = await client.query(`
      SELECT t.id, t.name, t.category_id, t.template_type, t.content_doc, 
             t.html_content, t.variables, t.created_by, t.is_global, t.is_active,
             t.created_at, t.updated_at,
             c.name as category_name,
             u.username as created_by_username
      FROM catalog.description_templates t
      LEFT JOIN description_template_categories c ON t.category_id = c.id
      LEFT JOIN users u ON t.created_by = u.id
      WHERE t.id = $1
    `, [id]);
    
    if (result.rows.length === 0) {
      return null;
    }
    
    const row = result.rows[0];
    return {
      id: row.id,
      name: row.name,
      categoryId: row.category_id,
      categoryName: row.category_name,
      templateType: row.template_type,
      contentDoc: row.content_doc,
      htmlContent: row.html_content,
      variables: row.variables,
      createdBy: row.created_by,
      createdByUsername: row.created_by_username,
      isGlobal: row.is_global,
      isActive: row.is_active,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  } catch (error) {
    console.error('❌ Error fetching description template:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function createDescriptionTemplate(data: {
  name: string;
  categoryId?: number;
  templateType: string;
  contentDoc?: any;
  htmlContent: string;
  variables?: any;
  createdBy?: number;
  isGlobal?: boolean;
  isActive?: boolean;
}) {
  const client = await pool.connect();
  try {
    const result = await client.query(`
      INSERT INTO catalog.description_templates (
        name, category_id, template_type, content_doc, html_content, 
        variables, created_by, is_global, is_active, created_at, updated_at
      )
      VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, NOW(), NOW())
      RETURNING id, name, category_id, template_type, content_doc, html_content, 
                variables, created_by, is_global, is_active, created_at, updated_at
    `, [
      data.name,
      data.categoryId || null,
      data.templateType,
      data.contentDoc ? JSON.stringify(data.contentDoc) : null,
      data.htmlContent,
      data.variables ? JSON.stringify(data.variables) : null,
      data.createdBy || null,
      data.isGlobal || false,
      data.isActive !== undefined ? data.isActive : true,
    ]);
    
    const row = result.rows[0];
    return {
      id: row.id,
      name: row.name,
      categoryId: row.category_id,
      templateType: row.template_type,
      contentDoc: row.content_doc,
      htmlContent: row.html_content,
      variables: row.variables,
      createdBy: row.created_by,
      isGlobal: row.is_global,
      isActive: row.is_active,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  } catch (error) {
    console.error('❌ Error creating description template:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function updateDescriptionTemplate(id: number, data: {
  name?: string;
  categoryId?: number;
  templateType?: string;
  contentDoc?: any;
  htmlContent?: string;
  variables?: any;
  isGlobal?: boolean;
  isActive?: boolean;
}) {
  const client = await pool.connect();
  try {
    const result = await client.query(`
      UPDATE catalog.description_templates
      SET name = COALESCE($1, name),
          category_id = COALESCE($2, category_id),
          template_type = COALESCE($3, template_type),
          content_doc = COALESCE($4, content_doc),
          html_content = COALESCE($5, html_content),
          variables = COALESCE($6, variables),
          is_global = COALESCE($7, is_global),
          is_active = COALESCE($8, is_active),
          updated_at = NOW()
      WHERE id = $9
      RETURNING id, name, category_id, template_type, content_doc, html_content,
                variables, created_by, is_global, is_active, created_at, updated_at
    `, [
      data.name,
      data.categoryId,
      data.templateType,
      data.contentDoc ? JSON.stringify(data.contentDoc) : null,
      data.htmlContent,
      data.variables ? JSON.stringify(data.variables) : null,
      data.isGlobal,
      data.isActive,
      id,
    ]);
    
    if (result.rows.length === 0) {
      return null;
    }
    
    const row = result.rows[0];
    return {
      id: row.id,
      name: row.name,
      categoryId: row.category_id,
      templateType: row.template_type,
      contentDoc: row.content_doc,
      htmlContent: row.html_content,
      variables: row.variables,
      createdBy: row.created_by,
      isGlobal: row.is_global,
      isActive: row.is_active,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  } catch (error) {
    console.error('❌ Error updating description template:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function deleteDescriptionTemplate(id: number) {
  const client = await pool.connect();
  try {
    const result = await client.query(`
      DELETE FROM catalog.description_templates WHERE id = $1 RETURNING id
    `, [id]);
    
    return result.rows.length > 0;
  } catch (error) {
    console.error('❌ Error deleting description template:', error);
    throw error;
  } finally {
    client.release();
  }
}

// ==================== PRODUCT-TEMPLATE ASSIGNMENTS ====================

export async function getProductTemplates(productId: number) {
  const client = await pool.connect();
  try {
    const result = await client.query(`
      SELECT pt.id, pt.product_id, pt.template_id, pt.render_config, pt.created_at,
             t.name as template_name, t.template_type, t.category_id, c.name as category_name
      FROM product_description_templates pt
      JOIN catalog.description_templates t ON pt.template_id = t.id
      LEFT JOIN description_template_categories c ON t.category_id = c.id
      WHERE pt.product_id = $1
      ORDER BY c.sort_order ASC, t.name ASC
    `, [productId]);
    
    return result.rows.map(row => ({
      id: row.id,
      productId: row.product_id,
      templateId: row.template_id,
      templateName: row.template_name,
      templateType: row.template_type,
      categoryId: row.category_id,
      categoryName: row.category_name,
      renderConfig: row.render_config, // Already JSON from JSONB
      createdAt: row.created_at,
    }));
  } catch (error) {
    console.error('❌ Error fetching product templates:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function assignTemplateToProduct(productId: number, templateId: number, renderConfig?: any) {
  const client = await pool.connect();
  try {
    const result = await client.query(`
      INSERT INTO product_description_templates (product_id, template_id, render_config, created_at)
      VALUES ($1, $2, $3, NOW())
      RETURNING id, product_id, template_id, render_config, created_at
    `, [productId, templateId, renderConfig ? JSON.stringify(renderConfig) : null]);
    
    const row = result.rows[0];
    return {
      id: row.id,
      productId: row.product_id,
      templateId: row.template_id,
      renderConfig: row.render_config,
      createdAt: row.created_at,
    };
  } catch (error) {
    console.error('❌ Error assigning template to product:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function updateProductTemplateConfig(id: number, renderConfig: any) {
  const client = await pool.connect();
  try {
    const result = await client.query(`
      UPDATE product_description_templates
      SET render_config = $1
      WHERE id = $2
      RETURNING id, product_id, template_id, render_config, created_at
    `, [JSON.stringify(renderConfig), id]);
    
    if (result.rows.length === 0) {
      return null;
    }
    
    const row = result.rows[0];
    return {
      id: row.id,
      productId: row.product_id,
      templateId: row.template_id,
      renderConfig: row.render_config,
      createdAt: row.created_at,
    };
  } catch (error) {
    console.error('❌ Error updating product template config:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function removeTemplateFromProduct(id: number) {
  const client = await pool.connect();
  try {
    const result = await client.query(`
      DELETE FROM product_description_templates WHERE id = $1 RETURNING id
    `, [id]);
    
    return result.rows.length > 0;
  } catch (error) {
    console.error('❌ Error removing template from product:', error);
    throw error;
  } finally {
    client.release();
  }
}

// ==================== AI GENERATION REQUESTS ====================

export async function createAiGenerationRequest(data: {
  productId: number;
  templateId?: number;
  prompt?: string;
  createdBy?: number;
}) {
  const client = await pool.connect();
  try {
    const result = await client.query(`
      INSERT INTO catalog.ai_generation_requests (
        product_id, template_id, status, prompt, created_by, created_at, updated_at
      )
      VALUES ($1, $2, 'pending', $3, $4, NOW(), NOW())
      RETURNING id, product_id, template_id, status, prompt, response, 
                cost_metadata, created_by, created_at, updated_at
    `, [data.productId, data.templateId || null, data.prompt || null, data.createdBy || null]);
    
    const row = result.rows[0];
    return {
      id: row.id,
      productId: row.product_id,
      templateId: row.template_id,
      status: row.status,
      prompt: row.prompt,
      response: row.response,
      costMetadata: row.cost_metadata,
      createdBy: row.created_by,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  } catch (error) {
    console.error('❌ Error creating AI generation request:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function updateAiGenerationRequest(id: number, data: {
  status?: string;
  response?: string;
  costMetadata?: any;
}) {
  const client = await pool.connect();
  try {
    const result = await client.query(`
      UPDATE catalog.ai_generation_requests
      SET status = COALESCE($1, status),
          response = COALESCE($2, response),
          cost_metadata = COALESCE($3, cost_metadata),
          updated_at = NOW()
      WHERE id = $4
      RETURNING id, product_id, template_id, status, prompt, response,
                cost_metadata, created_by, created_at, updated_at
    `, [
      data.status,
      data.response,
      data.costMetadata ? JSON.stringify(data.costMetadata) : null,
      id,
    ]);
    
    if (result.rows.length === 0) {
      return null;
    }
    
    const row = result.rows[0];
    return {
      id: row.id,
      productId: row.product_id,
      templateId: row.template_id,
      status: row.status,
      prompt: row.prompt,
      response: row.response,
      costMetadata: row.cost_metadata,
      createdBy: row.created_by,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  } catch (error) {
    console.error('❌ Error updating AI generation request:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function getAiGenerationHistory(productId: number, limit: number = 10) {
  const client = await pool.connect();
  try {
    const result = await client.query(`
      SELECT r.id, r.product_id, r.template_id, r.status, r.prompt, r.response,
             r.cost_metadata, r.created_by, r.created_at, r.updated_at,
             t.name as template_name,
             u.username as created_by_username
      FROM catalog.ai_generation_requests r
      LEFT JOIN catalog.description_templates t ON r.template_id = t.id
      LEFT JOIN users u ON r.created_by = u.id
      WHERE r.product_id = $1
      ORDER BY r.created_at DESC
      LIMIT $2
    `, [productId, limit]);
    
    return result.rows.map(row => ({
      id: row.id,
      productId: row.product_id,
      templateId: row.template_id,
      templateName: row.template_name,
      status: row.status,
      prompt: row.prompt,
      response: row.response,
      costMetadata: row.cost_metadata,
      createdBy: row.created_by,
      createdByUsername: row.created_by_username,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    }));
  } catch (error) {
    console.error('❌ Error fetching AI generation history:', error);
    throw error;
  } finally {
    client.release();
  }
}

// Allegro Fees Management
export async function saveAllegroFeesToDatabase(entries: any[]): Promise<number> {
  const client = await pool.connect();
  try {
    await client.query('BEGIN');

    let savedCount = 0;

    for (const entry of entries) {
      const typeId = entry.type?.id || 'UNKNOWN';
      const typeName = entry.type?.name || 'Unknown';
      const amount = Math.abs(parseFloat(entry.value?.amount || '0'));
      const currency = entry.value?.currency || 'PLN';
      const occurredAt = entry.occurredAt;
      const entryId = entry.id;
      const orderId = entry.order?.id || null;
      const offerId = entry.offer?.id || null;

      if (amount === 0) continue;

      // Określ kategorię opłaty
      let category = 'other';
      if (['SUC'].includes(typeId)) {
        category = 'commission';
      } else if (['LIS'].includes(typeId)) {
        category = 'listing';
      } else if (['NSP', 'ADS', 'PRO', 'FHB', 'FSF'].includes(typeId)) {
        category = 'ads';
      } else if (['HB1', 'DEL', 'DPD', 'SHP', 'DTR'].includes(typeId)) {
        category = 'delivery';
      } else if (['REF', 'RET'].includes(typeId)) {
        category = 'refund';
      }

      const result = await client.query(`
        INSERT INTO allegro_fees (
          entry_id, type_id, type_name, amount, currency, 
          occurred_at, order_id, offer_id, raw_data, category
        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10)
        ON CONFLICT (entry_id) DO UPDATE SET
          amount = EXCLUDED.amount,
          occurred_at = EXCLUDED.occurred_at,
          order_id = EXCLUDED.order_id,
          offer_id = EXCLUDED.offer_id,
          raw_data = EXCLUDED.raw_data,
          updated_at = NOW()
        RETURNING id
      `, [
        entryId,
        typeId,
        typeName,
        amount,
        currency,
        occurredAt,
        orderId,
        offerId,
        JSON.stringify(entry),
        category
      ]);

      if (result.rows.length > 0) {
        savedCount++;
      }
    }

    await client.query('COMMIT');
    console.log(`💾 Zapisano ${savedCount} opłat Allegro do bazy danych`);
    return savedCount;
  } catch (error) {
    await client.query('ROLLBACK');
    console.error('❌ Error saving Allegro fees to database:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function updateAllegroFeeSummary(date: Date): Promise<void> {
  const client = await pool.connect();
  try {
    const startOfDay = new Date(date.getFullYear(), date.getMonth(), date.getDate(), 0, 0, 0);
    const endOfDay = new Date(date.getFullYear(), date.getMonth(), date.getDate(), 23, 59, 59);

    const result = await client.query(`
      SELECT 
        category,
        COUNT(*) as count,
        SUM(amount) as total
      FROM allegro_fees
      WHERE occurred_at >= $1 AND occurred_at <= $2
      GROUP BY category
    `, [startOfDay, endOfDay]);

    const summary = {
      commission: { total: 0, count: 0 },
      listing: { total: 0, count: 0 },
      ads: { total: 0, count: 0 },
      delivery: { total: 0, count: 0 },
      refund: { total: 0, count: 0 },
      other: { total: 0, count: 0 }
    };

    for (const row of result.rows) {
      if (summary[row.category as keyof typeof summary]) {
        summary[row.category as keyof typeof summary].total = parseFloat(row.total || '0');
        summary[row.category as keyof typeof summary].count = parseInt(row.count || '0', 10);
      }
    }

    const grandTotal = Object.values(summary).reduce((acc, item) => acc + item.total, 0);

    await client.query(`
      INSERT INTO allegro_fee_summaries (
        summary_date,
        commission_total, commission_count,
        listing_total, listing_count,
        ads_total, ads_count,
        delivery_total, delivery_count,
        refunds_total, refunds_count,
        other_total, other_count,
        grand_total
      ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14)
      ON CONFLICT (summary_date) DO UPDATE SET
        commission_total = EXCLUDED.commission_total,
        commission_count = EXCLUDED.commission_count,
        listing_total = EXCLUDED.listing_total,
        listing_count = EXCLUDED.listing_count,
        ads_total = EXCLUDED.ads_total,
        ads_count = EXCLUDED.ads_count,
        delivery_total = EXCLUDED.delivery_total,
        delivery_count = EXCLUDED.delivery_count,
        refunds_total = EXCLUDED.refunds_total,
        refunds_count = EXCLUDED.refunds_count,
        other_total = EXCLUDED.other_total,
        other_count = EXCLUDED.other_count,
        grand_total = EXCLUDED.grand_total,
        updated_at = NOW()
    `, [
      startOfDay,
      summary.commission.total, summary.commission.count,
      summary.listing.total, summary.listing.count,
      summary.ads.total, summary.ads.count,
      summary.delivery.total, summary.delivery.count,
      summary.refund.total, summary.refund.count,
      summary.other.total, summary.other.count,
      grandTotal
    ]);

    console.log(`📊 Zaktualizowano podsumowanie opłat dla ${startOfDay.toLocaleDateString('pl-PL')}`);
  } catch (error) {
    console.error('❌ Error updating fee summary:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function getAllegroFeesHistory(
  startDate?: Date,
  endDate?: Date,
  category?: string,
  limit: number = 100,
  offset: number = 0,
  sortBy: string = 'occurred_at',
  sortOrder: string = 'DESC'
) {
  const client = await pool.connect();
  try {
    // Whitelist columns for sorting to prevent SQL injection
    const validSortColumns = ['occurred_at', 'category', 'fee_type', 'fee_name', 'amount'];
    const validSortBy = validSortColumns.includes(sortBy) ? sortBy : 'occurred_at';
    const validSortOrder = sortOrder.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';
    
    let query = `
      SELECT * FROM allegro_fees
      WHERE 1=1
    `;
    const params: any[] = [];
    let paramIndex = 1;

    if (startDate) {
      query += ` AND occurred_at >= $${paramIndex}`;
      params.push(startDate);
      paramIndex++;
    }

    if (endDate) {
      query += ` AND occurred_at <= $${paramIndex}`;
      params.push(endDate);
      paramIndex++;
    }

    if (category) {
      query += ` AND category = $${paramIndex}`;
      params.push(category);
      paramIndex++;
    }

    query += ` ORDER BY ${validSortBy} ${validSortOrder} LIMIT $${paramIndex} OFFSET $${paramIndex + 1}`;
    params.push(limit, offset);

    const result = await client.query(query, params);
    
    // Count total and sum amounts
    let statsQuery = `
      SELECT 
        COUNT(*) as total,
        COALESCE(SUM(amount), 0) as total_amount
      FROM allegro_fees 
      WHERE 1=1
    `;
    const statsParams: any[] = [];
    let statsParamIndex = 1;
    
    if (startDate) {
      statsQuery += ` AND occurred_at >= $${statsParamIndex}`;
      statsParams.push(startDate);
      statsParamIndex++;
    }
    
    if (endDate) {
      statsQuery += ` AND occurred_at <= $${statsParamIndex}`;
      statsParams.push(endDate);
      statsParamIndex++;
    }
    
    if (category) {
      statsQuery += ` AND category = $${statsParamIndex}`;
      statsParams.push(category);
    }
    
    const statsResult = await client.query(statsQuery, statsParams);
    const total = parseInt(statsResult.rows[0].total, 10);
    const totalAmount = parseFloat(statsResult.rows[0].total_amount);

    return {
      fees: result.rows,
      total,
      totalAmount,
      limit,
      offset
    };
  } catch (error) {
    console.error('❌ Error fetching fees history:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function getAllegroFeeSummaries(days: number = 30) {
  const client = await pool.connect();
  try {
    const startDate = new Date();
    startDate.setDate(startDate.getDate() - days);

    const result = await client.query(`
      SELECT * FROM allegro_fee_summaries
      WHERE summary_date >= $1
      ORDER BY summary_date DESC
    `, [startDate]);

    return result.rows;
  } catch (error) {
    console.error('❌ Error fetching fee summaries:', error);
    throw error;
  } finally {
    client.release();
  }
}

export async function getAllegroFeesGrouped(startDate?: Date, endDate?: Date) {
  const client = await pool.connect();
  try {
    let query = `
      SELECT 
        category,
        type_id,
        type_name,
        COUNT(*) as count,
        SUM(amount) as total,
        AVG(amount) as average,
        MIN(amount) as min,
        MAX(amount) as max
      FROM allegro_fees
      WHERE 1=1
    `;
    const params: any[] = [];
    let paramIndex = 1;

    if (startDate) {
      query += ` AND occurred_at >= $${paramIndex}`;
      params.push(startDate);
      paramIndex++;
    }

    if (endDate) {
      query += ` AND occurred_at <= $${paramIndex}`;
      params.push(endDate);
      paramIndex++;
    }

    query += ` GROUP BY category, type_id, type_name ORDER BY category, total DESC`;

    const result = await client.query(query, params);
    return result.rows;
  } catch (error) {
    console.error('❌ Error fetching grouped fees:', error);
    throw error;
  } finally {
    client.release();
  }
}
