import dotenv from "dotenv";
dotenv.config();

import express, { type Request, Response, NextFunction } from "express";
import { registerRoutes, autoConnectAllegro } from "./routes";
import { setupVite, serveStatic, log } from "./vite";
import { testConnection, pool } from "./postgres";
import { seedDefaultWorkCenters } from "./services/production/work-centers";
import { seedDefaultCarriers } from "./services/production/carriers";
import { seedDefaultRoutings } from "./services/production/routings";
import { seedDefaultPlanNameTemplates } from "./services/production/plan-name-templates";
import { startProcessor } from "./job-processor";

const app = express();
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: false, limit: '10mb' }));

// Serve uploaded files
app.use('/uploads', express.static('uploads'));

app.use((req, res, next) => {
  const start = Date.now();
  const path = req.path;
  let capturedJsonResponse: Record<string, any> | undefined = undefined;

  const originalResJson = res.json;
  res.json = function (bodyJson, ...args) {
    capturedJsonResponse = bodyJson;
    return originalResJson.apply(res, [bodyJson, ...args]);
  };

  res.on("finish", () => {
    const duration = Date.now() - start;
    if (path.startsWith("/api")) {
      let logLine = `${req.method} ${path} ${res.statusCode} in ${duration}ms`;
      if (capturedJsonResponse) {
        logLine += ` :: ${JSON.stringify(capturedJsonResponse)}`;
      }

      if (logLine.length > 80) {
        logLine = logLine.slice(0, 79) + "…";
      }

      log(logLine);
    }
  });

  next();
});

(async () => {
  // Baza danych włączona - ustaw SKIP_DB_CONNECTION=true aby wyłączyć
  const skipDatabase = process.env.SKIP_DB_CONNECTION === 'true';
  
  let dbSuccess = false;
  if (!skipDatabase) {
    // Sekwencyjne połączenia - najpierw testConnection, potem autoConnect
    dbSuccess = await testConnection();
    if (dbSuccess) {
      log('PostgreSQL connection ready');
      
      // Seed default work centers
      try {
        const seedResult = await seedDefaultWorkCenters(pool);
        if (seedResult.inserted > 0) {
          log(`✅ Seeded ${seedResult.inserted} default work centers`);
        }
        if (seedResult.skipped > 0) {
          log(`ℹ️  Skipped ${seedResult.skipped} existing work centers`);
        }
      } catch (error) {
        log(`⚠️  Failed to seed work centers: ${error instanceof Error ? error.message : 'Unknown error'}`);
      }

      // Seed default carriers (pallets, carts)
      try {
        const carrierSeedResult = await seedDefaultCarriers(pool);
        if (carrierSeedResult.groupsInserted > 0) {
          log(`✅ Seeded ${carrierSeedResult.groupsInserted} carrier groups`);
        }
        if (carrierSeedResult.carriersInserted > 0) {
          log(`✅ Seeded ${carrierSeedResult.carriersInserted} carriers (pallets & carts)`);
        }
        const totalCarriers = carrierSeedResult.carriersInserted;
        const skippedCarriers = 18 - totalCarriers; // 10 euro + 5 standard + 3 carts
        if (skippedCarriers > 0) {
          log(`ℹ️  Skipped ${skippedCarriers} existing carriers`);
        }
      } catch (error) {
        log(`⚠️  Failed to seed carriers: ${error instanceof Error ? error.message : 'Unknown error'}`);
      }

      // Seed default routings (manufacturing paths)
      try {
        const routingSeedResult = await seedDefaultRoutings(pool);
        if (routingSeedResult.routingsInserted > 0) {
          log(`✅ Seeded ${routingSeedResult.routingsInserted} default routing(s)`);
        }
        if (routingSeedResult.operationsInserted > 0) {
          log(`✅ Seeded ${routingSeedResult.operationsInserted} routing operations`);
        }
        if (routingSeedResult.routingsInserted === 0 && routingSeedResult.operationsInserted === 0) {
          log(`ℹ️  Skipped existing routings and operations`);
        }
      } catch (error) {
        log(`⚠️  Failed to seed routings: ${error instanceof Error ? error.message : 'Unknown error'}`);
      }

      // Seed default plan name templates
      try {
        const templateSeedResult = await seedDefaultPlanNameTemplates(pool);
        if (templateSeedResult.inserted > 0) {
          log(`✅ Seeded ${templateSeedResult.inserted} plan name template(s)`);
        }
        if (templateSeedResult.skipped > 0) {
          log(`ℹ️  Skipped ${templateSeedResult.skipped} existing plan name templates`);
        }
      } catch (error) {
        log(`⚠️  Failed to seed plan name templates: ${error instanceof Error ? error.message : 'Unknown error'}`);
      }

      // Start background job processor
      try {
        await startProcessor();
        log('✅ Background job processor started');
      } catch (error) {
        log(`⚠️  Failed to start job processor: ${error instanceof Error ? error.message : 'Unknown error'}`);
      }
    } else {
      log('PostgreSQL connection failed - orders will sync to memory only');
    }

    // Tylko jeśli baza działa, próbuj auto-connect do Allegro
    if (dbSuccess) {
      const connected = await autoConnectAllegro();
      if (connected) {
        log('✅ Allegro auto-connected successfully');
      } else {
        log('ℹ️ Allegro auto-connect skipped (manual OAuth may be required)');
      }
    }
  } else {
    log('⚠️  Database connection DISABLED (SKIP_DB_CONNECTION is set) - app running without database');
  }
  
  const server = await registerRoutes(app, skipDatabase);

  app.use((err: any, _req: Request, res: Response, _next: NextFunction) => {
    const status = err.status || err.statusCode || 500;
    const message = err.message || "Internal Server Error";

    res.status(status).json({ message });
    throw err;
  });

  // importantly only setup vite in development and after
  // setting up all the other routes so the catch-all route
  // doesn't interfere with the other routes
  if (app.get("env") === "development") {
    await setupVite(app, server);
  } else {
    serveStatic(app);
  }

  // ALWAYS serve the app on the port specified in the environment variable PORT
  // Other ports are firewalled. Default to 5000 if not specified.
  // this serves both the API and the client.
  // It is the only port that is not firewalled.
  const port = parseInt(process.env.PORT || '5000', 10);
  server.listen({
    port,
    host: "0.0.0.0",
    reusePort: true,
  }, () => {
    log(`serving on port ${port}`);
  });
})();
