import { pool } from "./postgres";
import { logError } from "./error-logger";

/**
 * Auto-Healing System - Automatyczne naprawianie błędów synchronizacji
 * 
 * Funkcjonalność:
 * 1. Retry failed Odoo sync queue items z exponential backoff
 * 2. Detect missing orders (order matching)
 * 3. Auto-fetch missing orders from platforms (Allegro, Shoper)
 * 4. Update sync health stats
 * 5. Auto-reconnect integrations if possible
 */

interface RetryStats {
  resetCount: number;
  errors: string[];
}

/**
 * Exponential backoff delays based on attempt number
 * attempt 1: 1 min
 * attempt 2: 5 min
 * attempt 3: 15 min
 * attempt 4: 30 min
 * attempt 5+: 60 min (1 hour)
 */
function getRetryDelayMinutes(attempts: number): number {
  switch (attempts) {
    case 1:
      return 1;
    case 2:
      return 5;
    case 3:
      return 15;
    case 4:
      return 30;
    default:
      return 60; // 1 hour for 5+ attempts
  }
}

/**
 * Auto-retry failed Odoo sync queue items
 * Używa exponential backoff - im więcej prób, tym dłużej czeka
 */
export async function autoRetryFailedSyncs(): Promise<RetryStats> {
  const stats: RetryStats = {
    resetCount: 0,
    errors: [],
  };

  try {
    console.log("🔄 [AUTO-HEAL] Checking failed Odoo syncs for retry...");

    // Znajdź failed items które mogą być retry
    const result = await pool.query(`
      SELECT 
        id,
        order_number,
        attempts,
        last_error,
        updated_at,
        EXTRACT(EPOCH FROM (NOW() - updated_at))/60 as minutes_since_last_attempt
      FROM odoo_sync_queue
      WHERE status = 'failed'
      ORDER BY updated_at ASC
    `);

    for (const item of result.rows) {
      const requiredWaitMinutes = getRetryDelayMinutes(item.attempts);
      const minutesSinceLastAttempt = item.minutes_since_last_attempt;

      if (minutesSinceLastAttempt >= requiredWaitMinutes) {
        // Reset do pending - system automatycznie spróbuje ponownie
        await pool.query(`
          UPDATE odoo_sync_queue
          SET 
            status = 'pending',
            updated_at = NOW()
          WHERE id = $1
        `, [item.id]);

        stats.resetCount++;
        
        console.log(`✅ [AUTO-HEAL] Reset order #${item.order_number} to pending (attempt ${item.attempts + 1}, waited ${Math.round(minutesSinceLastAttempt)}min)`);
      }
    }

    // Update sync health stats
    if (stats.resetCount > 0) {
      await pool.query(`
        UPDATE sync_health_stats
        SET 
          auto_heal_count = auto_heal_count + $1,
          last_auto_heal_at = NOW(),
          updated_at = NOW()
        WHERE sync_type = 'odoo_queue'
      `, [stats.resetCount]);
      
      console.log(`✅ [AUTO-HEAL] Reset ${stats.resetCount} failed orders to retry`);
    } else {
      console.log(`ℹ️  [AUTO-HEAL] No failed orders ready for retry yet`);
    }

  } catch (error: any) {
    const errorMessage = `Auto-retry failed syncs error: ${error.message}`;
    stats.errors.push(errorMessage);
    
    await logError({
      type: "sync",
      message: errorMessage,
      error,
      context: { component: "auto-healing", action: "retry-failed-syncs" },
    });
    
    console.error("❌ [AUTO-HEAL] Error during auto-retry:", error);
  }

  return stats;
}

/**
 * Detect and add missing orders to sync queue
 * Sprawdza czy wszystkie zamówienia z ostatnich 24h są w Odoo
 */
export async function autoDetectMissingOrders(): Promise<RetryStats> {
  const stats: RetryStats = {
    resetCount: 0,
    errors: [],
  };

  try {
    console.log("🔍 [AUTO-HEAL] Checking for missing orders in Odoo...");

    // Znajdź zamówienia z ostatnich 24h które nie są w kolejce Odoo
    const result = await pool.query(`
      SELECT 
        o.id,
        o.order_number,
        o.source,
        o.odoo_order_id
      FROM commerce.orders o
      WHERE 
        o.created_at > NOW() - INTERVAL '24 hours'
        AND o.odoo_order_id IS NULL
        AND o.order_number IS NOT NULL
        AND o.order_number != ''
        AND NOT EXISTS (
          SELECT 1 FROM odoo_sync_queue q 
          WHERE q.order_number = o.order_number
        )
      ORDER BY o.created_at DESC
      LIMIT 50
    `);

    for (const order of result.rows) {
      // Dodaj do kolejki
      await pool.query(`
        INSERT INTO odoo_sync_queue (
          order_number,
          source,
          operation, 
          status, 
          scheduled_at
        )
        VALUES ($1, $2, 'create', 'pending', NOW())
        ON CONFLICT (order_number) DO NOTHING
      `, [order.order_number, order.source]);

      stats.resetCount++;
      console.log(`✅ [AUTO-HEAL] Added missing order #${order.order_number} to Odoo queue`);
    }

    if (stats.resetCount > 0) {
      console.log(`✅ [AUTO-HEAL] Added ${stats.resetCount} missing orders to Odoo sync queue`);
      
      await pool.query(`
        UPDATE sync_health_stats
        SET 
          auto_heal_count = auto_heal_count + $1,
          last_auto_heal_at = NOW(),
          updated_at = NOW()
        WHERE sync_type = 'odoo_queue'
      `, [stats.resetCount]);
    } else {
      console.log(`ℹ️  [AUTO-HEAL] No missing orders detected`);
    }

  } catch (error: any) {
    const errorMessage = `Auto-detect missing orders error: ${error.message}`;
    stats.errors.push(errorMessage);
    
    await logError({
      type: "sync",
      message: errorMessage,
      error,
      context: { component: "auto-healing", action: "detect-missing-orders" },
    });
    
    console.error("❌ [AUTO-HEAL] Error during missing orders detection:", error);
  }

  return stats;
}

/**
 * Auto-fetch missing orders from Allegro
 * Pobiera zamówienia z Allegro które nie są w OMS
 */
export async function autoFetchMissingOrdersFromAllegro(): Promise<RetryStats> {
  const stats: RetryStats = {
    resetCount: 0,
    errors: [],
  };

  try {
    console.log("🔍 [AUTO-HEAL] Checking for missing Allegro orders in OMS...");

    // Pobierz connection Allegro
    const connectionResult = await pool.query(`
      SELECT id, access_token, is_active, token_expires_at, refresh_token
      FROM allegro_connections
      WHERE is_active = true
      LIMIT 1
    `);

    if (connectionResult.rows.length === 0 || !connectionResult.rows[0].access_token) {
      console.log("⚠️  [AUTO-HEAL] No active Allegro connection - skipping");
      return stats;
    }

    const connection = connectionResult.rows[0];
    let accessToken = connection.access_token;

    // Sprawdź czy token nie wygasł
    if (connection.token_expires_at && new Date(connection.token_expires_at) < new Date()) {
      console.log("⚠️  [AUTO-HEAL] Allegro token expired - skipping auto-fetch");
      return stats;
    }

    // Pobierz zamówienia z ostatnich 24h z Allegro
    const startDate = new Date();
    startDate.setHours(startDate.getHours() - 24);
    const endDate = new Date();

    const axios = (await import('axios')).default;
    const ALLEGRO_API_URL = "https://api.allegro.pl";

    const response = await axios.get(
      `${ALLEGRO_API_URL}/order/checkout-forms`,
      {
        headers: {
          Authorization: `Bearer ${accessToken}`,
          Accept: "application/vnd.allegro.public.v1+json",
        },
        params: {
          limit: 50,
          offset: 0,
          'updatedAt.gte': startDate.toISOString(),
          'updatedAt.lte': endDate.toISOString(),
        },
      }
    );

    const allegroOrders = response.data.checkoutForms || [];
    console.log(`📦 [AUTO-HEAL] Found ${allegroOrders.length} Allegro orders from last 24h`);

    // Sprawdź które zamówienia nie są w OMS
    for (const allegroOrder of allegroOrders) {
      const sourceOrderId = allegroOrder.id;

      // Sprawdź czy zamówienie już jest w OMS
      const omsCheck = await pool.query(`
        SELECT id FROM commerce.orders 
        WHERE source = 'ALLEGRO' AND source_order_id = $1
      `, [sourceOrderId]);

      if (omsCheck.rows.length === 0) {
        // Zamówienie nie ma w OMS - pobierz szczegóły i zapisz
        try {
          console.log(`📥 [AUTO-HEAL] Fetching missing Allegro order: ${sourceOrderId}`);

          // Pobierz pełne szczegóły zamówienia
          const orderResponse = await axios.get(
            `${ALLEGRO_API_URL}/order/checkout-forms/${sourceOrderId}`,
            {
              headers: {
                Authorization: `Bearer ${accessToken}`,
                Accept: "application/vnd.allegro.public.v1+json",
              },
            }
          );

          const fullOrder = orderResponse.data;

          // Pobierz shipments
          let shipments: any[] = [];
          try {
            const shipmentsResponse = await axios.get(
              `${ALLEGRO_API_URL}/order/checkout-forms/${sourceOrderId}/shipments`,
              {
                headers: {
                  Authorization: `Bearer ${accessToken}`,
                  Accept: "application/vnd.allegro.public.v1+json",
                },
              }
            );
            shipments = shipmentsResponse.data.shipments || [];
          } catch (error) {
            // Shipments mogą nie istnieć
            shipments = [];
          }

          // Pobierz customer returns (puste, bo to nowe zamówienie)
          const customerReturns: any[] = [];

          // Pobierz refunds
          const { fetchPaymentRefunds } = await import('./allegro-api.js');
          const refunds = await fetchPaymentRefunds(sourceOrderId, accessToken);
          
          if (refunds && refunds.length > 0 && fullOrder.payment) {
            fullOrder.payment.refundReconciliation = refunds;
            fullOrder.payment.refundAmount = refunds[0]?.totalValue?.amount || 0;
            fullOrder.payment.refundDate = refunds[0]?.createdAt || null;
          }

          // Zapisz do OMS
          const { saveOrderToPostgres, saveOrderToCommerce } = await import('./postgres.js');
          await saveOrderToPostgres(fullOrder, shipments, customerReturns, accessToken);
          await saveOrderToCommerce('ALLEGRO', sourceOrderId, fullOrder, shipments, customerReturns, accessToken);

          stats.resetCount++;
          console.log(`✅ [AUTO-HEAL] Fetched and saved Allegro order: ${sourceOrderId}`);

        } catch (error: any) {
          const errorMsg = `Failed to fetch Allegro order ${sourceOrderId}: ${error.message}`;
          stats.errors.push(errorMsg);
          console.error(`❌ [AUTO-HEAL] ${errorMsg}`);
        }
      }
    }

    if (stats.resetCount > 0) {
      console.log(`✅ [AUTO-HEAL] Auto-fetched ${stats.resetCount} missing Allegro orders`);
    } else {
      console.log(`ℹ️  [AUTO-HEAL] No missing Allegro orders to fetch`);
    }

  } catch (error: any) {
    const errorMessage = `Auto-fetch Allegro orders error: ${error.message}`;
    stats.errors.push(errorMessage);
    
    await logError({
      type: "sync",
      message: errorMessage,
      error,
      context: { component: "auto-healing", action: "auto-fetch-allegro" },
    });
    
    console.error("❌ [AUTO-HEAL] Error during Allegro auto-fetch:", error);
  }

  return stats;
}

/**
 * Auto-fetch missing orders from Shoper
 * Pobiera zamówienia z Shoper które nie są w OMS
 */
export async function autoFetchMissingOrdersFromShoper(): Promise<RetryStats> {
  const stats: RetryStats = {
    resetCount: 0,
    errors: [],
  };

  try {
    console.log("🔍 [AUTO-HEAL] Checking for missing Shoper orders in OMS...");

    // Sprawdź czy Shoper jest skonfigurowany
    const shopUrl = process.env.SHOPER_SHOP_URL;
    const username = process.env.SHOPER_USERNAME;
    const password = process.env.SHOPER_PASSWORD;

    if (!shopUrl || !username || !password) {
      console.log("⚠️  [AUTO-HEAL] Shoper not configured - skipping");
      return stats;
    }

    // Pobierz zamówienia z ostatnich 24h z Shoper
    const { getShoperOrders, getShoperOrderProducts, getShoperParcels, getShoperPayments, getShoperDeliveries } = await import('./shoper-api.js');
    
    const startDate = new Date();
    startDate.setHours(startDate.getHours() - 24);
    const endDate = new Date();

    const filters = {
      updated_at: {
        ">=": startDate.toISOString().replace('T', ' ').replace('Z', ''),
        "<=": endDate.toISOString().replace('T', ' ').replace('Z', '')
      }
    };

    const shoperOrders = await getShoperOrders(50, filters);
    console.log(`📦 [AUTO-HEAL] Found ${shoperOrders.length} Shoper orders from last 24h`);

    // Pobierz payment i delivery mappings raz
    const payments = await getShoperPayments();
    const deliveries = await getShoperDeliveries();
    const paymentMap = new Map(payments.map((p: any) => [p.payment_id, p.translations?.pl_PL?.title || p.name]));
    const deliveryMap = new Map(deliveries.map((d: any) => [d.delivery_id, d.name]));

    // Sprawdź które zamówienia nie są w OMS
    for (const shoperOrder of shoperOrders) {
      const sourceOrderId = shoperOrder.order_id?.toString() || shoperOrder.id?.toString();

      if (!sourceOrderId) {
        continue;
      }

      // Sprawdź czy zamówienie już jest w OMS
      const omsCheck = await pool.query(`
        SELECT id FROM commerce.orders 
        WHERE source = 'SHOPER' AND source_order_id = $1
      `, [sourceOrderId]);

      if (omsCheck.rows.length === 0) {
        // Zamówienie nie ma w OMS - pobierz szczegóły i zapisz
        try {
          console.log(`📥 [AUTO-HEAL] Fetching missing Shoper order: ${sourceOrderId}`);

          // Dodaj payment i delivery names
          if (shoperOrder.payment_id && paymentMap.has(shoperOrder.payment_id.toString())) {
            shoperOrder.payment_method_name = paymentMap.get(shoperOrder.payment_id.toString());
          }
          if (shoperOrder.shipping_id && deliveryMap.has(shoperOrder.shipping_id.toString())) {
            shoperOrder.delivery_method_name = deliveryMap.get(shoperOrder.shipping_id.toString());
          }

          // Pobierz paczki
          let parcels: any[] = [];
          try {
            parcels = await getShoperParcels(sourceOrderId);
          } catch (error) {
            parcels = [];
          }

          // Zapisz do OMS
          const { saveShoperOrderToPostgres, saveOrderToCommerce } = await import('./postgres.js');
          await saveShoperOrderToPostgres(shoperOrder, parcels);
          await saveOrderToCommerce('SHOPER', sourceOrderId, shoperOrder, parcels, []);

          stats.resetCount++;
          console.log(`✅ [AUTO-HEAL] Fetched and saved Shoper order: ${sourceOrderId}`);

        } catch (error: any) {
          const errorMsg = `Failed to fetch Shoper order ${sourceOrderId}: ${error.message}`;
          stats.errors.push(errorMsg);
          console.error(`❌ [AUTO-HEAL] ${errorMsg}`);
        }
      }
    }

    if (stats.resetCount > 0) {
      console.log(`✅ [AUTO-HEAL] Auto-fetched ${stats.resetCount} missing Shoper orders`);
    } else {
      console.log(`ℹ️  [AUTO-HEAL] No missing Shoper orders to fetch`);
    }

  } catch (error: any) {
    const errorMessage = `Auto-fetch Shoper orders error: ${error.message}`;
    stats.errors.push(errorMessage);
    
    await logError({
      type: "sync",
      message: errorMessage,
      error,
      context: { component: "auto-healing", action: "auto-fetch-shoper" },
    });
    
    console.error("❌ [AUTO-HEAL] Error during Shoper auto-fetch:", error);
  }

  return stats;
}

/**
 * Update sync health statistics
 * Oblicza aktualny stan synchronizacji i zapisuje do bazy
 */
export async function updateSyncHealthStats(): Promise<void> {
  try {
    console.log("📊 [AUTO-HEAL] Updating sync health stats...");

    // === ALLEGRO HEALTH ===
    const allegroHealth = await pool.query(`
      SELECT 
        COUNT(*) as total_24h,
        COUNT(*) FILTER (WHERE created_at > NOW() - INTERVAL '1 hour') as recent_count
      FROM commerce.orders
      WHERE source = 'ALLEGRO' AND created_at > NOW() - INTERVAL '24 hours'
    `);

    const allegroConnection = await pool.query(`
      SELECT is_active, connection_error, last_error_at
      FROM allegro_connections
      LIMIT 1
    `);

    const allegroStatus = allegroConnection.rows[0];
    const allegroHealthStatus = allegroStatus?.is_active 
      ? 'healthy' 
      : (allegroStatus?.connection_error ? 'critical' : 'unknown');

    await pool.query(`
      INSERT INTO sync_health_stats (
        sync_type, 
        last_sync_at, 
        total_synced_24h,
        health_status,
        last_error,
        updated_at
      ) VALUES (
        'allegro',
        NOW(),
        $1,
        $2,
        $3,
        NOW()
      )
      ON CONFLICT (sync_type) 
      DO UPDATE SET
        last_sync_at = NOW(),
        total_synced_24h = $1,
        health_status = $2,
        last_error = $3,
        updated_at = NOW()
    `, [
      allegroHealth.rows[0].total_24h,
      allegroHealthStatus,
      allegroStatus?.connection_error || null
    ]);

    // === SHOPER HEALTH ===
    const shoperHealth = await pool.query(`
      SELECT 
        COUNT(*) as total_24h,
        COUNT(*) FILTER (WHERE created_at > NOW() - INTERVAL '1 hour') as recent_count
      FROM commerce.orders
      WHERE source = 'SHOPER' AND created_at > NOW() - INTERVAL '24 hours'
    `);

    await pool.query(`
      INSERT INTO sync_health_stats (
        sync_type, 
        last_sync_at, 
        total_synced_24h,
        health_status,
        updated_at
      ) VALUES (
        'shoper',
        NOW(),
        $1,
        'healthy',
        NOW()
      )
      ON CONFLICT (sync_type) 
      DO UPDATE SET
        last_sync_at = NOW(),
        total_synced_24h = $1,
        health_status = 'healthy',
        updated_at = NOW()
    `, [shoperHealth.rows[0].total_24h]);

    // === ODOO QUEUE HEALTH ===
    const queueStats = await pool.query(`
      SELECT 
        COUNT(*) FILTER (WHERE status = 'pending') as pending_count,
        COUNT(*) FILTER (WHERE status = 'failed') as failed_count,
        COUNT(*) FILTER (WHERE status = 'completed' AND updated_at > NOW() - INTERVAL '24 hours') as completed_24h,
        COUNT(*) FILTER (WHERE status = 'failed' AND updated_at > NOW() - INTERVAL '1 hour') as recent_failures
      FROM odoo_sync_queue
    `);

    const qStats = queueStats.rows[0];
    const odooHealthStatus = 
      qStats.failed_count > 10 ? 'critical' :
      qStats.failed_count > 3 ? 'degraded' :
      'healthy';

    await pool.query(`
      INSERT INTO sync_health_stats (
        sync_type, 
        last_sync_at,
        total_synced_24h,
        queue_pending,
        queue_failed,
        health_status,
        consecutive_failures,
        updated_at
      ) VALUES (
        'odoo_queue',
        NOW(),
        $1,
        $2,
        $3,
        $4,
        $5,
        NOW()
      )
      ON CONFLICT (sync_type) 
      DO UPDATE SET
        last_sync_at = NOW(),
        total_synced_24h = $1,
        queue_pending = $2,
        queue_failed = $3,
        health_status = $4,
        consecutive_failures = $5,
        updated_at = NOW()
    `, [
      qStats.completed_24h,
      qStats.pending_count,
      qStats.failed_count,
      odooHealthStatus,
      qStats.recent_failures
    ]);

    console.log(`✅ [AUTO-HEAL] Health stats updated - Allegro: ${allegroHealthStatus}, Odoo: ${odooHealthStatus}`);

  } catch (error: any) {
    await logError({
      type: "sync",
      message: `Update sync health stats error: ${error.message}`,
      error,
      context: { component: "auto-healing", action: "update-health-stats" },
    });
    
    console.error("❌ [AUTO-HEAL] Error updating health stats:", error);
  }
}

/**
 * Run all auto-healing tasks
 * Wywołuje wszystkie funkcje auto-healing w odpowiedniej kolejności
 */
export async function runAutoHealing(): Promise<void> {
  console.log("🚀 [AUTO-HEAL] Starting auto-healing cycle...");
  
  const startTime = Date.now();
  
  // 1. Update health stats first
  await updateSyncHealthStats();
  
  // 2. Auto-fetch missing orders from platforms
  const allegroFetchStats = await autoFetchMissingOrdersFromAllegro();
  const shoperFetchStats = await autoFetchMissingOrdersFromShoper();
  
  // 3. Detect missing orders in Odoo (for orders already in OMS)
  const missingStats = await autoDetectMissingOrders();
  
  // 4. Retry failed syncs
  const retryStats = await autoRetryFailedSyncs();
  
  const duration = Date.now() - startTime;
  
  console.log(`✅ [AUTO-HEAL] Cycle completed in ${duration}ms - Allegro: ${allegroFetchStats.resetCount}, Shoper: ${shoperFetchStats.resetCount}, Missing: ${missingStats.resetCount}, Retry: ${retryStats.resetCount}`);
}
