import { pool } from '../server/postgres';
import { renderFormatkaVisualization } from '../server/formatka-visualizer';

interface ComponentForGeneration {
  id: number;
  generatedName: string;
  color: string | null;
  calculatedLength: string | null;
  calculatedWidth: string | null;
  thickness: string | null;
  edgingPattern: string | null;
  edgingMaterial: string | null;
}

async function getColorHexFromDictionary(colorCode: string): Promise<string | null> {
  if (!colorCode) return null;
  
  try {
    const result = await pool.query(`
      SELECT color FROM product_creator.dictionaries 
      WHERE dictionary_type = 'color' AND code = $1
    `, [colorCode]);
    
    if (result.rows.length > 0) {
      return result.rows[0].color;
    }
    
    console.warn(`⚠️  Color code "${colorCode}" not found in dictionaries`);
    return null;
  } catch (error) {
    console.error(`❌ Error fetching color hex for "${colorCode}":`, error);
    return null;
  }
}

function extractEdgingColorCode(edgingMaterial: string | null): string | null {
  if (!edgingMaterial) return null;
  
  const parts = edgingMaterial.split('_');
  if (parts.length >= 2) {
    return parts[parts.length - 1];
  }
  
  return edgingMaterial;
}

function parseEdgingPattern(pattern: string | null): { edge1: boolean; edge2: boolean; edge3: boolean; edge4: boolean } {
  if (!pattern || pattern.length !== 4) {
    return { edge1: false, edge2: false, edge3: false, edge4: false };
  }
  
  return {
    edge1: pattern[0] === 'T',
    edge2: pattern[1] === 'T',
    edge3: pattern[2] === 'T',
    edge4: pattern[3] === 'T',
  };
}

async function generateFormatkasForBom(bomId: number) {
  console.log(`\n🎨 Generating formatkas for BOM ${bomId}...\n`);

  // Fetch all components
  const componentsResult = await pool.query<ComponentForGeneration>(`
    SELECT 
      id, 
      generated_name as "generatedName",
      color,
      calculated_length as "calculatedLength",
      calculated_width as "calculatedWidth",
      thickness,
      edging_pattern as "edgingPattern",
      edging_material as "edgingMaterial"
    FROM bom.product_components
    WHERE product_bom_id = (
      SELECT id FROM bom.product_boms WHERE id = $1
    )
    ORDER BY position_in_bom
  `, [bomId]);

  let successCount = 0;
  let skippedCount = 0;
  let errorCount = 0;

  for (const component of componentsResult.rows) {
    try {
      // Skip if missing dimensions
      if (!component.calculatedLength || !component.calculatedWidth) {
        console.log(`⏭️  Skipping ${component.generatedName} - missing dimensions`);
        skippedCount++;
        continue;
      }

      // Get colors
      const colorHex = component.color 
        ? await getColorHexFromDictionary(component.color)
        : null;

      const edgingColorCode = extractEdgingColorCode(component.edgingMaterial);
      const edgingColorHex = edgingColorCode 
        ? await getColorHexFromDictionary(edgingColorCode)
        : null;

      const edgingFlags = parseEdgingPattern(component.edgingPattern);

      // Generate formatka
      const visualizationUrl = await renderFormatkaVisualization({
        name: component.generatedName,
        length: parseFloat(component.calculatedLength),
        width: parseFloat(component.calculatedWidth),
        thickness: component.thickness ? parseFloat(component.thickness) : 18,
        color: component.color || 'UNKNOWN',
        colorHex: colorHex || undefined,
        edgingColorHex: edgingColorHex || undefined,
        edgingPattern: component.edgingPattern || undefined,
        componentType: 'board',
        ...edgingFlags,
      });

      // Update database
      await pool.query(`
        UPDATE bom.product_components
        SET visualization_url = $1
        WHERE id = $2
      `, [visualizationUrl, component.id]);

      console.log(`✅ Generated: ${component.generatedName}`);
      successCount++;

    } catch (error) {
      console.error(`❌ Error generating formatka for ${component.generatedName}:`, error);
      errorCount++;
    }
  }

  console.log(`\n📊 Summary for BOM ${bomId}:`);
  console.log(`   ✅ Success: ${successCount}`);
  console.log(`   ⏭️  Skipped: ${skippedCount}`);
  console.log(`   ❌ Errors: ${errorCount}\n`);
}

async function main() {
  console.log('🚀 Starting formatka generation for all test BOMs...\n');

  // Generate for all 3 test BOMs
  const bomIds = [197, 186, 114]; // Products: 329, 283, 278
  
  for (const bomId of bomIds) {
    await generateFormatkasForBom(bomId);
  }

  console.log('✅ Formatka generation completed!');
  process.exit(0);
}

main().catch((error) => {
  console.error('Fatal error:', error);
  process.exit(1);
});
