import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { useToast } from "@/hooks/use-toast";
import { Plus, Pencil, Trash2, Image as ImageIcon, ExternalLink, MoreVertical, ArrowRight } from "lucide-react";
import { Link } from "wouter";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
  DropdownMenuSeparator,
} from "@/components/ui/dropdown-menu";
import { CreateGroupDialog } from "@/components/create-group-dialog";

type MaterialGroup = {
  id: number;
  name: string;
  code: string;
  category: string | null;
  description: string | null;
  displayOrder: number;
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
};

type Material = {
  id: number;
  groupId: number | null;
  name: string;
  internalCode: string;
  supplierCode: string | null;
  description: string | null;
  specifications: any;
  unitOfMeasure: string;
  quantity?: number;
  price?: number;
  gallery: string[];
  primaryImageUrl: string | null;
  displayOrder: number;
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
  groupName?: string;
  groupCode?: string;
};

export default function OkuciaPage() {
  const { toast } = useToast();
  const [selectedGroupId, setSelectedGroupId] = useState<number | null>(null);
  const [isGroupDialogOpen, setIsGroupDialogOpen] = useState(false);
  const [isMaterialDialogOpen, setIsMaterialDialogOpen] = useState(false);
  const [editingGroup, setEditingGroup] = useState<MaterialGroup | null>(null);
  const [editingMaterial, setEditingMaterial] = useState<Material | null>(null);
  const [isAddGroupDialogOpen, setIsAddGroupDialogOpen] = useState(false);
  
  // Form state for groups
  const [groupForm, setGroupForm] = useState({
    name: "",
    code: "",
    description: "",
    displayOrder: 0,
  });
  
  // Form state for materials
  const [materialForm, setMaterialForm] = useState({
    groupId: null as number | null,
    name: "",
    internalCode: "",
    supplierCode: "",
    description: "",
    unitOfMeasure: "szt",
    displayOrder: 0,
  });

  // Fetch groups (filtered by category 'okucia')
  const { data: groups = [], isLoading: groupsLoading } = useQuery<MaterialGroup[]>({
    queryKey: ["/api/warehouse/material-groups"],
  });

  // Filter groups by 'okucia' category
  const okuciaGroups = groups.filter(g => g.category === 'okucia');

  // Fetch materials
  const { data: materials = [], isLoading: materialsLoading } = useQuery<Material[]>({
    queryKey: ["/api/warehouse/materials", { category: 'okucia' }],
  });

  // Fetch category data for CreateGroupDialog
  const { data: categoryData } = useQuery<{ id: number; code: string; name: string }>({
    queryKey: [`/api/warehouse/categories/by-code/okucia`],
  });

  // Filter materials by selected group
  const filteredMaterials = selectedGroupId
    ? materials.filter((m) => m.groupId === selectedGroupId)
    : materials;

  // Create/Update Group Mutation
  const groupMutation = useMutation({
    mutationFn: async (data: Partial<MaterialGroup>) => {
      if (editingGroup) {
        return apiRequest("PATCH", `/api/warehouse/material-groups/${editingGroup.id}`, data);
      } else {
        return apiRequest("POST", "/api/warehouse/material-groups", { ...data, category: 'okucia' });
      }
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/material-groups"] });
      setIsGroupDialogOpen(false);
      setEditingGroup(null);
      resetGroupForm();
      toast({
        title: "Sukces",
        description: editingGroup ? "Grupa zaktualizowana" : "Grupa utworzona",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zapisać grupy",
        variant: "destructive",
      });
    },
  });

  // Delete Group Mutation
  const deleteGroupMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("DELETE", `/api/warehouse/material-groups/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/material-groups"] });
      toast({ title: "Sukces", description: "Grupa usunięta" });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się usunąć grupy",
        variant: "destructive",
      });
    },
  });

  // Create/Update Material Mutation
  const materialMutation = useMutation({
    mutationFn: async (data: Partial<Material>) => {
      if (editingMaterial) {
        return apiRequest("PATCH", `/api/warehouse/materials/${editingMaterial.id}`, data);
      } else {
        return apiRequest("POST", "/api/warehouse/materials", data);
      }
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/materials", { category: 'okucia' }] });
      setIsMaterialDialogOpen(false);
      setEditingMaterial(null);
      resetMaterialForm();
      toast({
        title: "Sukces",
        description: editingMaterial ? "Materiał zaktualizowany" : "Materiał utworzony",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zapisać materiału",
        variant: "destructive",
      });
    },
  });

  // Delete Material Mutation
  const deleteMaterialMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("DELETE", `/api/warehouse/materials/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/materials", { category: 'okucia' }] });
      toast({ title: "Sukces", description: "Materiał usunięty" });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się usunąć materiału",
        variant: "destructive",
      });
    },
  });

  const resetGroupForm = () => {
    setGroupForm({
      name: "",
      code: "",
      description: "",
      displayOrder: 0,
    });
  };

  const resetMaterialForm = () => {
    setMaterialForm({
      groupId: null,
      name: "",
      internalCode: "",
      supplierCode: "",
      description: "",
      unitOfMeasure: "szt",
      displayOrder: 0,
    });
  };

  const openGroupDialog = (group?: MaterialGroup) => {
    if (group) {
      setEditingGroup(group);
      setGroupForm({
        name: group.name,
        code: group.code,
        description: group.description || "",
        displayOrder: group.displayOrder,
      });
    } else {
      setEditingGroup(null);
      resetGroupForm();
    }
    setIsGroupDialogOpen(true);
  };

  const openMaterialDialog = (material?: Material) => {
    if (material) {
      setEditingMaterial(material);
      setMaterialForm({
        groupId: material.groupId,
        name: material.name,
        internalCode: material.internalCode,
        supplierCode: material.supplierCode || "",
        description: material.description || "",
        unitOfMeasure: material.unitOfMeasure,
        displayOrder: material.displayOrder,
      });
    } else {
      setEditingMaterial(null);
      resetMaterialForm();
    }
    setIsMaterialDialogOpen(true);
  };

  const handleGroupSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    groupMutation.mutate(groupForm);
  };

  const handleMaterialSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    materialMutation.mutate(materialForm);
  };

  if (groupsLoading || materialsLoading) {
    return <div className="p-6">Ładowanie...</div>;
  }

  return (
    <div className="p-6 space-y-6">
      <div className="flex items-center justify-between">
        <h1 className="text-3xl font-bold">Magazyn - Okucia</h1>
        <div className="flex gap-2">
          <Button
            data-testid="button-add-group"
            onClick={() => openGroupDialog()}
          >
            <Plus className="w-4 h-4 mr-2" />
            Dodaj grupę
          </Button>
          <Button
            data-testid="button-add-material"
            onClick={() => openMaterialDialog()}
            disabled={okuciaGroups.length === 0}
          >
            <Plus className="w-4 h-4 mr-2" />
            Dodaj materiał
          </Button>
        </div>
      </div>

      <div>
          <Card>
            <CardHeader>
              <CardTitle>
                {selectedGroupId
                  ? `Materiały - ${okuciaGroups.find(g => g.id === selectedGroupId)?.name}`
                  : "Wszystkie materiały"}
              </CardTitle>
            </CardHeader>
            <CardContent>
              {filteredMaterials.length === 0 ? (
                <div className="text-center text-muted-foreground py-8">
                  Brak materiałów w tej grupie
                </div>
              ) : (
                <div className="space-y-4">
                  {filteredMaterials.map((material) => (
                    <Link key={material.id} href={`/warehouse/okucia/${material.id}`}>
                      <Card className="hover-elevate cursor-pointer" data-testid={`card-material-${material.id}`}>
                        <CardContent className="p-4">
                          <div className="flex gap-4 items-start">
                            {material.primaryImageUrl ? (
                              <div className="w-24 h-24 bg-muted rounded flex-shrink-0 flex items-center justify-center overflow-hidden">
                                <img
                                  src={material.primaryImageUrl}
                                  alt={material.name}
                                  className="w-full h-full object-cover"
                                />
                              </div>
                            ) : (
                              <div className="w-24 h-24 bg-muted rounded flex-shrink-0 flex items-center justify-center">
                                <ImageIcon className="w-8 h-8 text-muted-foreground" />
                              </div>
                            )}
                            <div className="flex-1 flex items-center gap-2">
                              <div className="flex-1">
                                <h3 className="font-semibold text-lg" data-testid={`text-material-name-${material.id}`}>
                                  {material.name}
                                </h3>
                              <p className="text-sm text-muted-foreground">
                                Kod wewnętrzny: {material.internalCode}
                              </p>
                              {material.supplierCode && (
                                <p className="text-sm text-muted-foreground">
                                  Kod dostawcy: {material.supplierCode}
                                </p>
                              )}
                              {material.groupName && (
                                <p className="text-sm text-muted-foreground">
                                  Grupa: {material.groupName}
                                </p>
                              )}
                              <div className="flex gap-4 mt-2">
                                <p className="text-sm font-medium" data-testid={`text-material-quantity-${material.id}`}>
                                  Ilość: <span className="text-primary">{material.quantity || 0} {material.unitOfMeasure}</span>
                                </p>
                                {material.price && (
                                  <p className="text-sm text-muted-foreground" data-testid={`text-material-price-${material.id}`}>
                                    Cena: {Number(material.price).toFixed(2)} zł
                                  </p>
                                )}
                              </div>
                              {material.description && (
                                <p className="text-sm mt-2">{material.description}</p>
                              )}
                              </div>
                              <div className="flex items-center justify-center flex-shrink-0">
                                <ArrowRight className="h-5 w-5" />
                              </div>
                            </div>
                            <div onClick={(e) => e.stopPropagation()} className="flex-shrink-0">
                              <DropdownMenu>
                                <DropdownMenuTrigger asChild onClick={(e) => e.stopPropagation()}>
                                  <Button
                                    variant="ghost"
                                    size="icon"
                                    data-testid={`button-menu-material-${material.id}`}
                                  >
                                    <MoreVertical className="w-4 h-4" />
                                  </Button>
                                </DropdownMenuTrigger>
                                <DropdownMenuContent align="end">
                                  <DropdownMenuItem onClick={() => openMaterialDialog(material)}>
                                    <Pencil className="w-4 h-4 mr-2" />
                                    Edytuj
                                  </DropdownMenuItem>
                                </DropdownMenuContent>
                              </DropdownMenu>
                            </div>
                          </div>
                        </CardContent>
                      </Card>
                    </Link>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>
        </div>

      {/* Group Dialog */}
      <Dialog open={isGroupDialogOpen} onOpenChange={setIsGroupDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>
              {editingGroup ? "Edytuj grupę" : "Dodaj grupę"}
            </DialogTitle>
            <DialogDescription>
              Uzupełnij dane grupy okuć
            </DialogDescription>
          </DialogHeader>
          <form onSubmit={handleGroupSubmit}>
            <div className="space-y-4 py-4">
              <div>
                <Label htmlFor="group-name">Nazwa *</Label>
                <Input
                  id="group-name"
                  data-testid="input-group-name"
                  value={groupForm.name}
                  onChange={(e) => setGroupForm({ ...groupForm, name: e.target.value })}
                  required
                />
              </div>
              <div>
                <Label htmlFor="group-code">Kod *</Label>
                <Input
                  id="group-code"
                  data-testid="input-group-code"
                  value={groupForm.code}
                  onChange={(e) => setGroupForm({ ...groupForm, code: e.target.value })}
                  required
                />
              </div>
              <div>
                <Label htmlFor="group-description">Opis</Label>
                <Textarea
                  id="group-description"
                  data-testid="input-group-description"
                  value={groupForm.description}
                  onChange={(e) => setGroupForm({ ...groupForm, description: e.target.value })}
                />
              </div>
              <div>
                <Label htmlFor="group-display-order">Kolejność wyświetlania</Label>
                <Input
                  id="group-display-order"
                  data-testid="input-group-display-order"
                  type="number"
                  value={groupForm.displayOrder}
                  onChange={(e) => setGroupForm({ ...groupForm, displayOrder: parseInt(e.target.value) || 0 })}
                />
              </div>
            </div>
            <DialogFooter>
              <Button
                type="button"
                variant="outline"
                onClick={() => setIsGroupDialogOpen(false)}
                data-testid="button-cancel-group"
              >
                Anuluj
              </Button>
              <Button type="submit" data-testid="button-save-group">
                Zapisz
              </Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>

      {/* Material Dialog */}
      <Dialog open={isMaterialDialogOpen} onOpenChange={setIsMaterialDialogOpen}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>
              {editingMaterial ? "Edytuj materiał" : "Dodaj materiał"}
            </DialogTitle>
            <DialogDescription>
              Uzupełnij dane materiału
            </DialogDescription>
          </DialogHeader>
          <form onSubmit={handleMaterialSubmit}>
            <div className="space-y-4 py-4">
              <div>
                <Label htmlFor="material-group">Grupa *</Label>
                <Select
                  value={materialForm.groupId?.toString() || ""}
                  onValueChange={(value) => setMaterialForm({ ...materialForm, groupId: parseInt(value) })}
                  required
                >
                  <SelectTrigger data-testid="select-material-group">
                    <SelectValue placeholder="Wybierz grupę" />
                  </SelectTrigger>
                  <SelectContent>
                    {okuciaGroups.map((group) => (
                      <SelectItem key={group.id} value={group.id.toString()}>
                        {group.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label htmlFor="material-name">Nazwa *</Label>
                <Input
                  id="material-name"
                  data-testid="input-material-name"
                  value={materialForm.name}
                  onChange={(e) => setMaterialForm({ ...materialForm, name: e.target.value })}
                  required
                />
              </div>
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="material-internal-code">Kod wewnętrzny *</Label>
                  <Input
                    id="material-internal-code"
                    data-testid="input-material-internal-code"
                    value={materialForm.internalCode}
                    onChange={(e) => setMaterialForm({ ...materialForm, internalCode: e.target.value })}
                    required
                  />
                </div>
                <div>
                  <Label htmlFor="material-supplier-code">Kod dostawcy</Label>
                  <Input
                    id="material-supplier-code"
                    data-testid="input-material-supplier-code"
                    value={materialForm.supplierCode}
                    onChange={(e) => setMaterialForm({ ...materialForm, supplierCode: e.target.value })}
                  />
                </div>
              </div>
              <div>
                <Label htmlFor="material-description">Opis</Label>
                <Textarea
                  id="material-description"
                  data-testid="input-material-description"
                  value={materialForm.description}
                  onChange={(e) => setMaterialForm({ ...materialForm, description: e.target.value })}
                />
              </div>
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="material-unit">Jednostka miary</Label>
                  <Input
                    id="material-unit"
                    data-testid="input-material-unit"
                    value={materialForm.unitOfMeasure}
                    onChange={(e) => setMaterialForm({ ...materialForm, unitOfMeasure: e.target.value })}
                  />
                </div>
                <div>
                  <Label htmlFor="material-display-order">Kolejność wyświetlania</Label>
                  <Input
                    id="material-display-order"
                    data-testid="input-material-display-order"
                    type="number"
                    value={materialForm.displayOrder}
                    onChange={(e) => setMaterialForm({ ...materialForm, displayOrder: parseInt(e.target.value) || 0 })}
                  />
                </div>
              </div>
            </div>
            <DialogFooter>
              <Button
                type="button"
                variant="outline"
                onClick={() => setIsMaterialDialogOpen(false)}
                data-testid="button-cancel-material"
              >
                Anuluj
              </Button>
              <Button type="submit" data-testid="button-save-material">
                Zapisz
              </Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>

      {/* Create Group Dialog */}
      {categoryData && (
        <CreateGroupDialog
          open={isAddGroupDialogOpen}
          onOpenChange={setIsAddGroupDialogOpen}
          categoryId={categoryData.id}
          categoryCode={categoryData.code}
        />
      )}
    </div>
  );
}
