import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { useToast } from "@/hooks/use-toast";
import { Plus, Edit, Trash2, Loader2, FolderPlus, ArrowLeft } from "lucide-react";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogFooter } from "@/components/ui/dialog";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Link, useRoute } from "wouter";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";

interface MaterialGroup {
  id: number;
  name: string;
  code: string;
  category: string;
  description: string | null;
  color: string | null;
  displayOrder: number;
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
}

interface WarehouseCategory {
  id: number;
  name: string;
  code: string;
}

const CATEGORY_LABELS: Record<string, string> = {
  okucia: "Okucia",
  plyty: "Płyty meblowe",
  obrzeza: "Obrzeża",
  sruby: "Śruby",
  pianki: "Pianki",
  formatki: "Formatki",
  tkaniny: "Tkaniny",
  opakowania: "Opakowania",
  "produkty-spakowane": "Produkty spakowane"
};

export default function CategoryGroupsPage() {
  const { toast } = useToast();
  const [, params] = useRoute("/warehouse/:category/groups");
  const categoryCode = params?.category || "";
  
  // Dialog states
  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false);
  const [isEditDialogOpen, setIsEditDialogOpen] = useState(false);
  const [editingGroup, setEditingGroup] = useState<MaterialGroup | null>(null);
  
  // Form state
  const [formData, setFormData] = useState({
    name: "",
    code: "",
    description: "",
    displayOrder: "0",
    color: ""
  });
  
  // Fetch category info
  const { data: category } = useQuery<WarehouseCategory>({
    queryKey: [`/api/warehouse/categories/by-code/${categoryCode}`],
    enabled: !!categoryCode,
  });
  
  // Fetch material groups
  const { data: allGroups = [], isLoading } = useQuery<MaterialGroup[]>({
    queryKey: ['/api/warehouse/material-groups'],
    staleTime: 0,
    refetchOnMount: true,
  });
  
  // Filter groups by current category
  const groups = allGroups
    .filter(g => g.category === categoryCode && g.isActive)
    .sort((a, b) => {
      if (a.displayOrder !== b.displayOrder) {
        return a.displayOrder - b.displayOrder;
      }
      return a.name.localeCompare(b.name);
    });
  
  // Debug logging
  useEffect(() => {
    console.log('[CATEGORY DEBUG]', {
      categoryCode,
      allGroupsCount: allGroups.length,
      filteredGroupsCount: groups.length,
      sampleGroups: allGroups.slice(0, 5).map(g => ({ name: g.name, category: g.category, isActive: g.isActive }))
    });
  }, [categoryCode, allGroups.length, groups.length]);
  
  // Create mutation
  const createMutation = useMutation({
    mutationFn: async (data: any) => {
      return await apiRequest('POST', '/api/warehouse/material-groups', {
        ...data,
        category: categoryCode,
      });
    },
    onSuccess: () => {
      toast({
        title: "Grupa utworzona",
        description: "Grupa została pomyślnie utworzona."
      });
      queryClient.invalidateQueries({ queryKey: ['/api/warehouse/material-groups'] });
      setIsAddDialogOpen(false);
      resetForm();
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się utworzyć grupy.",
        variant: "destructive"
      });
    }
  });
  
  // Update mutation
  const updateMutation = useMutation({
    mutationFn: async ({ id, data }: { id: number; data: any }) => {
      return await apiRequest('PATCH', `/api/warehouse/material-groups/${id}`, data);
    },
    onSuccess: () => {
      toast({
        title: "Grupa zaktualizowana",
        description: "Grupa została pomyślnie zaktualizowana."
      });
      queryClient.invalidateQueries({ queryKey: ['/api/warehouse/material-groups'] });
      setIsEditDialogOpen(false);
      setEditingGroup(null);
      resetForm();
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zaktualizować grupy.",
        variant: "destructive"
      });
    }
  });
  
  // Delete mutation
  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      return await apiRequest('DELETE', `/api/warehouse/material-groups/${id}`);
    },
    onSuccess: () => {
      toast({
        title: "Grupa usunięta",
        description: "Grupa została pomyślnie usunięta."
      });
      queryClient.invalidateQueries({ queryKey: ['/api/warehouse/material-groups'] });
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się usunąć grupy.",
        variant: "destructive"
      });
    }
  });
  
  const resetForm = () => {
    setFormData({
      name: "",
      code: "",
      description: "",
      displayOrder: "0",
      color: ""
    });
  };
  
  const handleSubmit = () => {
    if (!formData.name || !formData.code) {
      toast({
        title: "Błąd walidacji",
        description: "Nazwa i kod grupy są wymagane.",
        variant: "destructive"
      });
      return;
    }
    
    const submitData = {
      name: formData.name,
      code: formData.code,
      description: formData.description || null,
      displayOrder: parseInt(formData.displayOrder) || 0,
      color: formData.color || null
    };
    
    if (editingGroup) {
      updateMutation.mutate({ id: editingGroup.id, data: submitData });
    } else {
      createMutation.mutate(submitData);
    }
  };
  
  const handleEdit = (group: MaterialGroup) => {
    setEditingGroup(group);
    setFormData({
      name: group.name,
      code: group.code,
      description: group.description || "",
      displayOrder: group.displayOrder.toString(),
      color: group.color || ""
    });
    setIsEditDialogOpen(true);
  };
  
  const handleDelete = (id: number) => {
    if (confirm("Czy na pewno chcesz usunąć tę grupę?")) {
      deleteMutation.mutate(id);
    }
  };
  
  const handleAddNew = () => {
    resetForm();
    setIsAddDialogOpen(true);
  };
  
  const categoryLabel = CATEGORY_LABELS[categoryCode] || categoryCode;
  
  return (
    <div className="p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-4">
          <Link href={`/warehouse/${categoryCode}`}>
            <Button variant="outline" size="sm" data-testid="button-back">
              <ArrowLeft className="w-4 h-4 mr-2" />
              Powrót
            </Button>
          </Link>
          <h1 className="text-3xl font-bold">Grupy - {categoryLabel}</h1>
        </div>
        <Button onClick={handleAddNew} data-testid="button-add-group">
          <Plus className="w-4 h-4 mr-2" />
          Dodaj grupę
        </Button>
      </div>

      {isLoading ? (
        <div className="flex items-center justify-center py-8">
          <Loader2 className="w-6 h-6 animate-spin" />
        </div>
      ) : groups.length === 0 ? (
        <Card>
          <CardContent className="py-8">
            <div className="text-center text-muted-foreground">
              <FolderPlus className="w-12 h-12 mx-auto mb-4 opacity-50" />
              <p>Brak grup</p>
              <p className="text-sm mt-2">Dodaj pierwszą grupę dla {categoryLabel}</p>
            </div>
          </CardContent>
        </Card>
      ) : (
        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
          {groups.map((group) => (
            <Card key={group.id} data-testid={`card-group-${group.id}`}>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-lg font-medium">
                  {group.name}
                </CardTitle>
                <div className="flex items-center gap-2">
                  <Button
                    variant="ghost"
                    size="icon"
                    onClick={() => handleEdit(group)}
                    data-testid={`button-edit-${group.id}`}
                  >
                    <Edit className="w-4 h-4" />
                  </Button>
                  <Button
                    variant="ghost"
                    size="icon"
                    onClick={() => handleDelete(group.id)}
                    data-testid={`button-delete-${group.id}`}
                  >
                    <Trash2 className="w-4 h-4" />
                  </Button>
                </div>
              </CardHeader>
              <CardContent>
                <div className="space-y-2 text-sm">
                  <div>
                    <span className="text-muted-foreground">Kod:</span>{" "}
                    <span className="font-medium">{group.code}</span>
                  </div>
                  {group.description && (
                    <div>
                      <span className="text-muted-foreground">Opis:</span>{" "}
                      <span>{group.description}</span>
                    </div>
                  )}
                  <div>
                    <span className="text-muted-foreground">Kolejność:</span>{" "}
                    <span>{group.displayOrder}</span>
                  </div>
                  <Link href={`/warehouse/${categoryCode}/group/${group.code}`}>
                    <Button variant="outline" size="sm" className="w-full mt-2" data-testid={`button-view-materials-${group.id}`}>
                      Zobacz materiały
                    </Button>
                  </Link>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>
      )}

      {/* Add/Edit Dialog */}
      <Dialog open={isAddDialogOpen || isEditDialogOpen} onOpenChange={(open) => {
        if (!open) {
          setIsAddDialogOpen(false);
          setIsEditDialogOpen(false);
          setEditingGroup(null);
          resetForm();
        }
      }}>
        <DialogContent data-testid="dialog-group-form">
          <DialogHeader>
            <DialogTitle>
              {editingGroup ? "Edytuj grupę" : "Dodaj nową grupę"}
            </DialogTitle>
            <DialogDescription>
              {editingGroup ? `Zaktualizuj szczegóły grupy - ${categoryLabel}` : `Utwórz nową grupę dla ${categoryLabel}`}
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4">
            <div>
              <Label htmlFor="name">Nazwa grupy *</Label>
              <Input
                id="name"
                value={formData.name}
                onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                placeholder="np. Duże"
                data-testid="input-name"
              />
            </div>
            
            <div>
              <Label htmlFor="code">Kod grupy *</Label>
              <Input
                id="code"
                value={formData.code}
                onChange={(e) => setFormData({ ...formData, code: e.target.value })}
                placeholder="np. 50"
                data-testid="input-code"
              />
            </div>
            
            <div>
              <Label htmlFor="description">Opis</Label>
              <Textarea
                id="description"
                value={formData.description}
                onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                placeholder="Opcjonalny opis grupy"
                data-testid="textarea-description"
              />
            </div>
            
            <div>
              <Label htmlFor="displayOrder">Kolejność wyświetlania</Label>
              <Input
                id="displayOrder"
                type="number"
                value={formData.displayOrder}
                onChange={(e) => setFormData({ ...formData, displayOrder: e.target.value })}
                data-testid="input-display-order"
              />
            </div>
            
            <div>
              <Label htmlFor="color">Kolor grupy</Label>
              <div className="flex items-center gap-2">
                <Input
                  id="color"
                  type="text"
                  value={formData.color}
                  onChange={(e) => setFormData({ ...formData, color: e.target.value })}
                  placeholder="#FF5733"
                  className="flex-1"
                  data-testid="input-color"
                />
                <input
                  type="color"
                  value={formData.color || "#6366f1"}
                  onChange={(e) => setFormData({ ...formData, color: e.target.value })}
                  className="w-10 h-9 rounded border cursor-pointer"
                  data-testid="input-color-picker"
                />
                {formData.color && (
                  <div 
                    className="w-9 h-9 rounded border" 
                    style={{ backgroundColor: formData.color }}
                    title={formData.color}
                  />
                )}
              </div>
              <p className="text-xs text-muted-foreground mt-1">Format: #RRGGBB (np. #FF5733)</p>
            </div>
          </div>
          
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => {
                setIsAddDialogOpen(false);
                setIsEditDialogOpen(false);
                setEditingGroup(null);
                resetForm();
              }}
              data-testid="button-cancel"
            >
              Anuluj
            </Button>
            <Button
              onClick={handleSubmit}
              disabled={createMutation.isPending || updateMutation.isPending}
              data-testid="button-submit"
            >
              {(createMutation.isPending || updateMutation.isPending) && (
                <Loader2 className="w-4 h-4 mr-2 animate-spin" />
              )}
              {editingGroup ? "Zapisz zmiany" : "Dodaj grupę"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
