import { useQuery, useMutation } from "@tanstack/react-query";
import { WarehouseLayout } from "@/features/warehouse/warehouse-layout";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { FileCheck, Plus, Trash2, Archive, CheckCircle, Download } from "lucide-react";
import { Link, useLocation } from "wouter";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { useState } from "react";

interface InventoryCount {
  id: number;
  name: string;
  status: string;
  notes: string | null;
  createdBy: string;
  createdAt: string;
  finalizedBy: string | null;
  finalizedAt: string | null;
  itemCount: string;
  countedItemCount: string;
}

export default function WarehouseInventoryCountsPage() {
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [archiveDialogOpen, setArchiveDialogOpen] = useState(false);
  const [finalizeDialogOpen, setFinalizeDialogOpen] = useState(false);
  const [selectedCount, setSelectedCount] = useState<InventoryCount | null>(null);
  
  const { data: inventoryCounts = [], isLoading } = useQuery<InventoryCount[]>({
    queryKey: ["/api/warehouse/inventory-counts"],
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      return await apiRequest('DELETE', `/api/warehouse/inventory-counts/${id}`);
    },
    onSuccess: () => {
      toast({
        title: "Sukces",
        description: "Spis inwentaryzacyjny został usunięty",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/inventory-counts"] });
      setDeleteDialogOpen(false);
      setSelectedCount(null);
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się usunąć spisu",
        variant: "destructive",
      });
    },
  });

  const archiveMutation = useMutation({
    mutationFn: async (id: number) => {
      return await apiRequest('PATCH', `/api/warehouse/inventory-counts/${id}`, {
        status: 'archived'
      });
    },
    onSuccess: () => {
      toast({
        title: "Sukces",
        description: "Spis inwentaryzacyjny został zarchiwizowany",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/inventory-counts"] });
      setArchiveDialogOpen(false);
      setSelectedCount(null);
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zarchiwizować spisu",
        variant: "destructive",
      });
    },
  });

  const finalizeMutation = useMutation({
    mutationFn: async (id: number) => {
      return await apiRequest('POST', `/api/warehouse/inventory-counts/${id}/finalize`, {});
    },
    onSuccess: () => {
      toast({
        title: "Sukces",
        description: "Spis inwentaryzacyjny zakończony - magazyn przeliczony",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/inventory-counts"] });
      setFinalizeDialogOpen(false);
      setSelectedCount(null);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zakończyć spisu",
        variant: "destructive",
      });
    },
  });

  const draftCounts = inventoryCounts.filter(c => c.status === 'draft');
  const finalizedCounts = inventoryCounts.filter(c => c.status === 'finalized');
  const archivedCounts = inventoryCounts.filter(c => c.status === 'archived');

  const handleDelete = (count: InventoryCount) => {
    setSelectedCount(count);
    setDeleteDialogOpen(true);
  };

  const handleArchive = (count: InventoryCount) => {
    setSelectedCount(count);
    setArchiveDialogOpen(true);
  };

  const handleFinalize = (count: InventoryCount) => {
    setSelectedCount(count);
    setFinalizeDialogOpen(true);
  };

  const getStatusBadge = (status: string) => {
    switch (status) {
      case 'draft':
        return <Badge variant="secondary">Roboczy</Badge>;
      case 'finalized':
        return <Badge variant="default">Zamknięty</Badge>;
      case 'archived':
        return <Badge variant="outline">Zarchiwizowany</Badge>;
      default:
        return <Badge>{status}</Badge>;
    }
  };

  if (isLoading) {
    return (
      <WarehouseLayout category="all" hideSubmenu={true}>
        <div className="p-6">Ładowanie...</div>
      </WarehouseLayout>
    );
  }

  const renderTable = (counts: InventoryCount[], title: string) => {
    if (counts.length === 0) return null;

    return (
      <div className="space-y-3">
        <h2 className="text-lg font-semibold">{title} ({counts.length})</h2>
        <div className="border rounded-md">
          <Table>
            <TableHeader>
              <TableRow className="bg-muted/50">
                <TableHead className="w-[8%]">ID</TableHead>
                <TableHead className="w-[32%]">Nazwa</TableHead>
                <TableHead className="w-[12%]">Status</TableHead>
                <TableHead className="w-[10%]">Pozycji</TableHead>
                <TableHead className="w-[15%]">Utworzono przez</TableHead>
                <TableHead className="w-[13%]">Data</TableHead>
                <TableHead className="w-[10%] text-right">Akcje</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {counts.map((count) => {
                const itemCount = parseInt(count.itemCount);
                const countedItemCount = parseInt(count.countedItemCount);
                const canDelete = itemCount === 0;
                const canFinalize = count.status === 'draft' && countedItemCount === itemCount && itemCount > 0;
                const uncountedItems = itemCount - countedItemCount;
                
                return (
                  <TableRow key={count.id} className="hover-elevate" data-testid={`row-count-${count.id}`}>
                    <TableCell className="text-sm text-muted-foreground font-mono">
                      #{count.id}
                    </TableCell>
                    <TableCell>
                      <Link href={`/warehouse/inventory-counts/${count.id}`}>
                        <span className="font-medium text-primary hover:underline cursor-pointer flex items-center gap-2">
                          <FileCheck className="h-4 w-4" />
                          {count.name}
                        </span>
                      </Link>
                    </TableCell>
                    <TableCell>
                      {getStatusBadge(count.status)}
                    </TableCell>
                    <TableCell>
                      {count.itemCount}
                      {count.status === 'draft' && uncountedItems > 0 && (
                        <span className="text-xs text-muted-foreground ml-1">
                          ({countedItemCount} spisanych)
                        </span>
                      )}
                    </TableCell>
                    <TableCell className="text-sm">
                      {count.status === 'finalized' ? count.finalizedBy : count.createdBy}
                    </TableCell>
                    <TableCell className="text-sm text-muted-foreground">
                      {count.status === 'finalized' && count.finalizedAt
                        ? new Date(count.finalizedAt).toLocaleDateString("pl-PL")
                        : new Date(count.createdAt).toLocaleDateString("pl-PL")}
                    </TableCell>
                    <TableCell className="text-right">
                      <div className="flex items-center justify-end gap-2">
                        {count.status === 'draft' && (
                          <Button
                            variant="default"
                            size="sm"
                            onClick={(e) => {
                              e.preventDefault();
                              handleFinalize(count);
                            }}
                            disabled={!canFinalize}
                            title={canFinalize ? "Zakończ inwentaryzację" : `Pozostało ${uncountedItems} pozycji do spisania`}
                            data-testid={`button-finalize-${count.id}`}
                          >
                            <CheckCircle className="h-4 w-4 mr-1" />
                            Zakończ
                          </Button>
                        )}
                        {count.status === 'finalized' && (
                          <Button
                            variant="outline"
                            size="sm"
                            onClick={(e) => {
                              e.preventDefault();
                              window.location.href = `/api/warehouse/inventory-counts/${count.id}/export/nexo`;
                            }}
                            title="Eksport do Subiekt Nexo (XML)"
                            data-testid={`button-export-nexo-${count.id}`}
                          >
                            <Download className="h-4 w-4" />
                          </Button>
                        )}
                        {canDelete ? (
                          <Button
                            variant="ghost"
                            size="sm"
                            onClick={(e) => {
                              e.preventDefault();
                              handleDelete(count);
                            }}
                            data-testid={`button-delete-${count.id}`}
                          >
                            <Trash2 className="h-4 w-4 text-destructive" />
                          </Button>
                        ) : (
                          count.status !== 'archived' && (
                            <Button
                              variant="ghost"
                              size="sm"
                              onClick={(e) => {
                                e.preventDefault();
                                handleArchive(count);
                              }}
                              data-testid={`button-archive-${count.id}`}
                            >
                              <Archive className="h-4 w-4 text-muted-foreground" />
                            </Button>
                          )
                        )}
                      </div>
                    </TableCell>
                  </TableRow>
                );
              })}
            </TableBody>
          </Table>
        </div>
      </div>
    );
  };

  return (
    <WarehouseLayout category="all" hideSubmenu={true}>
      <div className="space-y-6 p-6">
        {/* ERP Header */}
        <div className="flex items-center justify-between border-b pb-4">
          <div>
            <h1 className="text-2xl font-bold">Spisy inwentaryzacyjne</h1>
            <p className="text-sm text-muted-foreground mt-1">
              Zarządzaj spisami inwentaryzacyjnymi magazynu
            </p>
          </div>
          <Button onClick={() => setLocation("/warehouse/inventory-counts/new")} data-testid="button-create-new-count">
            <Plus className="h-4 w-4 mr-2" />
            Nowy spis
          </Button>
        </div>

        {/* Tables */}
        {renderTable(draftCounts, "Robocze")}
        {renderTable(finalizedCounts, "Zamknięte")}
        {renderTable(archivedCounts, "Zarchiwizowane")}

        {/* Empty State */}
        {inventoryCounts.length === 0 && (
          <div className="text-center py-12 border rounded-md bg-muted/20">
            <FileCheck className="h-12 w-12 mx-auto text-muted-foreground mb-4" />
            <p className="text-lg text-muted-foreground">Brak spisów inwentaryzacyjnych</p>
            <p className="text-sm text-muted-foreground mt-2">
              Kliknij "Nowy spis" aby rozpocząć
            </p>
          </div>
        )}
      </div>

      {/* Delete Confirmation Dialog */}
      <AlertDialog open={deleteDialogOpen} onOpenChange={setDeleteDialogOpen}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Czy na pewno chcesz usunąć?</AlertDialogTitle>
            <AlertDialogDescription>
              Spis "{selectedCount?.name}" zostanie trwale usunięty. Ta operacja jest nieodwracalna.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel data-testid="button-cancel-delete">Anuluj</AlertDialogCancel>
            <AlertDialogAction
              onClick={() => selectedCount && deleteMutation.mutate(selectedCount.id)}
              className="bg-destructive hover:bg-destructive/90"
              data-testid="button-confirm-delete"
            >
              Usuń
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>

      {/* Archive Confirmation Dialog */}
      <AlertDialog open={archiveDialogOpen} onOpenChange={setArchiveDialogOpen}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Archiwizować spis?</AlertDialogTitle>
            <AlertDialogDescription>
              Spis "{selectedCount?.name}" zostanie przeniesiony do archiwum. Będzie można go nadal przeglądać w sekcji "Zarchiwizowane".
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel data-testid="button-cancel-archive">Anuluj</AlertDialogCancel>
            <AlertDialogAction
              onClick={() => selectedCount && archiveMutation.mutate(selectedCount.id)}
              data-testid="button-confirm-archive"
            >
              Archiwizuj
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>

      {/* Finalize Confirmation Dialog */}
      <AlertDialog open={finalizeDialogOpen} onOpenChange={setFinalizeDialogOpen}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Zakończyć inwentaryzację?</AlertDialogTitle>
            <AlertDialogDescription>
              Spis "{selectedCount?.name}" zostanie zamknięty, a stany magazynowe zostaną przeliczone na podstawie różnic.
              Po zakończeniu nie będzie możliwości edycji pozycji.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel data-testid="button-cancel-finalize">Anuluj</AlertDialogCancel>
            <AlertDialogAction
              onClick={() => selectedCount && finalizeMutation.mutate(selectedCount.id)}
              data-testid="button-confirm-finalize"
            >
              Przelicz magazyn
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </WarehouseLayout>
  );
}
