import { useState, useEffect, useCallback } from "react";
import { useRoute, Link, useLocation } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { WarehouseLayout } from "@/features/warehouse/warehouse-layout";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { useToast } from "@/hooks/use-toast";
import { Search, X, ChevronLeft, ChevronRight, ChevronsLeft, ChevronsRight, Loader2, Download, Trash2, Plus, FolderPlus, Upload, Edit, Pencil, Image as ImageIcon, Star, Copy, MoreVertical, ExternalLink, Building2, Truck, ArrowRight, FileCheck, Package, MapPin, Archive, ArchiveRestore, Check, ChevronDown, ChevronUp, PackageCheck, PackageX, Settings2, CheckSquare, GripVertical } from "lucide-react";
import { Badge } from "@/components/ui/badge";
import { Checkbox } from "@/components/ui/checkbox";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger, DialogFooter } from "@/components/ui/dialog";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover";
import { CSVImportDialog } from "@/components/csv-import-dialog";
import { CSVImportWithMapping } from "@/components/csv-import-with-mapping";
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger, DropdownMenuSeparator, DropdownMenuCheckboxItem, DropdownMenuLabel } from "@/components/ui/dropdown-menu";
import { useDropzone } from "react-dropzone";
import {
  DndContext,
  closestCenter,
  KeyboardSensor,
  PointerSensor,
  useSensor,
  useSensors,
  DragEndEvent,
} from "@dnd-kit/core";
import {
  arrayMove,
  SortableContext,
  sortableKeyboardCoordinates,
  useSortable,
  verticalListSortingStrategy,
} from "@dnd-kit/sortable";
import { CSS } from "@dnd-kit/utilities";
import { GenerationLogsDialog } from "@/components/generation-logs-dialog";
import { DictionaryComboboxWithAdd } from "@/components/dictionary-combobox-with-add";
import { Combobox } from "@/components/ui/combobox";

// Tri-state filter types: undefined = not selected, 'include' = contains, 'exclude' = does not contain
type FilterState = 'include' | 'exclude';
type FilterStates = Record<string, FilterState>;

interface Material {
  id: number;
  groupId: number | null;
  name: string;
  internalCode: string;
  supplierCode: string | null;
  description: string | null;
  specifications: any;
  unitOfMeasure: string;
  quantity?: number;
  quantityReserved?: number;
  price?: string | null;
  gallery: string[];
  primaryImageUrl: string | null;
  displayOrder: number;
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
  groupName?: string;
  groupCode?: string;
  groupCategory?: string;
  locationId?: number;
  carrierId?: number;
  locationName?: string;
  carrierName?: string;
  
  // Pola specyficzne dla formatek
  parentId?: number | null;
  length?: number | null;
  width?: number | null;
  materialThickness?: number | null;
  cz1?: string | null;
  cz2?: string | null;
  furnitureType?: string | null;
  color?: string | null;
  plateType?: string | null;
  edgingMaterial?: string | null;
  edge1?: boolean;
  edge2?: boolean;
  edge3?: boolean;
  edge4?: boolean;
  edge5?: boolean;
  status?: string | null;
  source?: string | null;
  referenceCode?: string | null;
  
  // Pola z produktu katalogowego (dla produkty-spakowane)
  catalogProductId?: number | null;
  catalogProductSku?: string | null;
  catalogLength?: number | null;
  catalogWidth?: number | null;
  catalogColor?: string | null;
  catalogProductType?: string | null;
  catalogDoors?: string | null;
  catalogLegs?: string | null;
}

interface DictionaryItem {
  id: number;
  dictionaryType: string;
  code: string;
  readableName: string;
  color?: string | null;
  isActive: boolean;
}

function getTextColorForBackground(hexColor: string | null | undefined): string {
  if (!hexColor || !hexColor.startsWith('#')) return 'currentColor';
  const r = parseInt(hexColor.slice(1, 3), 16);
  const g = parseInt(hexColor.slice(3, 5), 16);
  const b = parseInt(hexColor.slice(5, 7), 16);
  const luminance = (0.299 * r + 0.587 * g + 0.114 * b) / 255;
  return luminance > 0.5 ? '#000000' : '#FFFFFF';
}

function needsVisibleBorder(hexColor: string | null | undefined): boolean {
  if (!hexColor || !hexColor.startsWith('#')) return false;
  const r = parseInt(hexColor.slice(1, 3), 16);
  const g = parseInt(hexColor.slice(3, 5), 16);
  const b = parseInt(hexColor.slice(5, 7), 16);
  const brightness = (r * 299 + g * 587 + b * 114) / 1000;
  return brightness > 230;
}

interface FilterCounts {
  colors: Record<string, number>;
  lengths: Record<string, number>;
  widths: Record<string, number>;
  stock: {
    inStock: number;
    outOfStock: number;
  };
}

interface SearchResponse {
  materials: Material[];
  pagination: {
    page: number;
    limit: number;
    total: number;
    totalPages: number;
    offset: number; // Globalny offset dla numeracji
  };
  filterCounts?: FilterCounts;
}

const CATEGORY_LABELS: Record<string, string> = {
  okucia: "Okucia",
  plyty_meblowe: "Płyty meblowe",
  sruby: "Śruby",
  tkaniny: "Tkaniny",
  pianki: "Pianki",
};

const STORAGE_KEY_PREFIX = 'warehouse-filters-';

// Helper function to generate slug from name
function generateSlug(name: string): string {
  // Map Polish characters to ASCII equivalents
  const polishMap: Record<string, string> = {
    'ą': 'a', 'ć': 'c', 'ę': 'e', 'ł': 'l', 'ń': 'n',
    'ó': 'o', 'ś': 's', 'ź': 'z', 'ż': 'z',
    'Ą': 'A', 'Ć': 'C', 'Ę': 'E', 'Ł': 'L', 'Ń': 'N',
    'Ó': 'O', 'Ś': 'S', 'Ź': 'Z', 'Ż': 'Z'
  };
  
  return name
    .split('')
    .map(char => polishMap[char] || char)
    .join('')
    .toLowerCase()
    .replace(/[^a-z0-9]+/g, '_')
    .replace(/^_+|_+$/g, '')
    .replace(/_+/g, '_');
}

// Bulk Formatka Edit Dialog Component
interface BulkFormatkaEditDialogProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  selectedCount: number;
  onSubmit: (updates: Record<string, any>) => void;
  isPending: boolean;
  cz1Dictionaries: any[];
  cz2Dictionaries: any[];
  colorDictionaries: any[];
  furnitureTypeDictionaries: any[];
  boardOptions: { value: string; label: string }[];
  edgingOptions: { value: string; label: string }[];
  carriers: any[];
  locations: any[];
  groups: any[];
}

function BulkFormatkaEditDialog({
  open,
  onOpenChange,
  selectedCount,
  onSubmit,
  isPending,
  cz1Dictionaries,
  cz2Dictionaries,
  colorDictionaries,
  furnitureTypeDictionaries,
  boardOptions,
  edgingOptions,
  carriers,
  locations,
  groups,
}: BulkFormatkaEditDialogProps) {
  const [enabledFields, setEnabledFields] = useState<Record<string, boolean>>({});
  const [fieldValues, setFieldValues] = useState<Record<string, any>>({});

  const toggleField = (field: string) => {
    setEnabledFields(prev => ({ ...prev, [field]: !prev[field] }));
  };

  const setFieldValue = (field: string, value: any) => {
    setFieldValues(prev => ({ ...prev, [field]: value }));
  };

  const handleSubmit = () => {
    const updates: Record<string, any> = {};
    for (const [field, enabled] of Object.entries(enabledFields)) {
      if (enabled && fieldValues[field] !== undefined) {
        updates[field] = fieldValues[field];
      }
    }
    if (Object.keys(updates).length === 0) return;
    onSubmit(updates);
  };

  const resetForm = () => {
    setEnabledFields({});
    setFieldValues({});
  };

  useEffect(() => {
    if (!open) resetForm();
  }, [open]);

  const enabledCount = Object.values(enabledFields).filter(Boolean).length;

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-2xl max-h-[85vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <Pencil className="w-5 h-5" />
            Edycja grupowa formatek
          </DialogTitle>
          <DialogDescription>
            Edytujesz {selectedCount} zaznaczonych formatek. Włącz pola, które chcesz zmienić.
          </DialogDescription>
        </DialogHeader>

        <div className="space-y-4 py-4">
          {/* CZ1 */}
          <div className="flex items-start gap-3 p-3 rounded-lg border bg-card">
            <Checkbox 
              checked={enabledFields.cz1 || false}
              onCheckedChange={() => toggleField('cz1')}
              data-testid="checkbox-bulk-cz1"
            />
            <div className="flex-1 space-y-2">
              <Label className={!enabledFields.cz1 ? 'text-muted-foreground' : ''}>CZ1</Label>
              <Combobox
                options={cz1Dictionaries.map((d: any) => ({ value: d.code, label: `${d.code} - ${d.name}` }))}
                value={fieldValues.cz1 || ''}
                onChange={(v: string) => setFieldValue('cz1', v)}
                placeholder="Wybierz CZ1..."
                searchPlaceholder="Szukaj CZ1..."
                emptyText="Nie znaleziono"
                disabled={!enabledFields.cz1}
              />
            </div>
          </div>

          {/* CZ2 */}
          <div className="flex items-start gap-3 p-3 rounded-lg border bg-card">
            <Checkbox 
              checked={enabledFields.cz2 || false}
              onCheckedChange={() => toggleField('cz2')}
              data-testid="checkbox-bulk-cz2"
            />
            <div className="flex-1 space-y-2">
              <Label className={!enabledFields.cz2 ? 'text-muted-foreground' : ''}>CZ2</Label>
              <Combobox
                options={cz2Dictionaries.map((d: any) => ({ value: d.code, label: `${d.code} - ${d.name}` }))}
                value={fieldValues.cz2 || ''}
                onChange={(v: string) => setFieldValue('cz2', v)}
                placeholder="Wybierz CZ2..."
                searchPlaceholder="Szukaj CZ2..."
                emptyText="Nie znaleziono"
                disabled={!enabledFields.cz2}
              />
            </div>
          </div>

          {/* Color */}
          <div className="flex items-start gap-3 p-3 rounded-lg border bg-card">
            <Checkbox 
              checked={enabledFields.color || false}
              onCheckedChange={() => toggleField('color')}
              data-testid="checkbox-bulk-color"
            />
            <div className="flex-1 space-y-2">
              <Label className={!enabledFields.color ? 'text-muted-foreground' : ''}>Kolor</Label>
              <Combobox
                options={colorDictionaries.map((d: any) => ({ value: d.code, label: `${d.code} - ${d.name}` }))}
                value={fieldValues.color || ''}
                onChange={(v: string) => setFieldValue('color', v)}
                placeholder="Wybierz kolor..."
                searchPlaceholder="Szukaj koloru..."
                emptyText="Nie znaleziono"
                disabled={!enabledFields.color}
              />
            </div>
          </div>

          {/* Board Code */}
          <div className="flex items-start gap-3 p-3 rounded-lg border bg-card">
            <Checkbox 
              checked={enabledFields.board_code || false}
              onCheckedChange={() => toggleField('board_code')}
              data-testid="checkbox-bulk-board"
            />
            <div className="flex-1 space-y-2">
              <Label className={!enabledFields.board_code ? 'text-muted-foreground' : ''}>Płyta</Label>
              <Combobox
                options={boardOptions}
                value={fieldValues.board_code || ''}
                onChange={(v: string) => setFieldValue('board_code', v)}
                placeholder="Wybierz płytę..."
                searchPlaceholder="Szukaj płyty..."
                emptyText="Nie znaleziono płyty"
                disabled={!enabledFields.board_code}
              />
            </div>
          </div>

          {/* Edging Code */}
          <div className="flex items-start gap-3 p-3 rounded-lg border bg-card">
            <Checkbox 
              checked={enabledFields.edging_code || false}
              onCheckedChange={() => toggleField('edging_code')}
              data-testid="checkbox-bulk-edging"
            />
            <div className="flex-1 space-y-2">
              <Label className={!enabledFields.edging_code ? 'text-muted-foreground' : ''}>Obrzeże</Label>
              <Combobox
                options={edgingOptions}
                value={fieldValues.edging_code || ''}
                onChange={(v: string) => setFieldValue('edging_code', v)}
                placeholder="Wybierz obrzeże..."
                searchPlaceholder="Szukaj obrzeża..."
                emptyText="Nie znaleziono obrzeża"
                disabled={!enabledFields.edging_code}
              />
            </div>
          </div>

          {/* Furniture Type */}
          <div className="flex items-start gap-3 p-3 rounded-lg border bg-card">
            <Checkbox 
              checked={enabledFields.furniture_type || false}
              onCheckedChange={() => toggleField('furniture_type')}
              data-testid="checkbox-bulk-furniture-type"
            />
            <div className="flex-1 space-y-2">
              <Label className={!enabledFields.furniture_type ? 'text-muted-foreground' : ''}>Typ mebla</Label>
              <Combobox
                options={furnitureTypeDictionaries.map((d: any) => ({ value: d.code, label: `${d.code} - ${d.name}` }))}
                value={fieldValues.furniture_type || ''}
                onChange={(v: string) => setFieldValue('furniture_type', v)}
                placeholder="Wybierz typ mebla..."
                searchPlaceholder="Szukaj typu..."
                emptyText="Nie znaleziono"
                disabled={!enabledFields.furniture_type}
              />
            </div>
          </div>

          {/* Thickness */}
          <div className="flex items-start gap-3 p-3 rounded-lg border bg-card">
            <Checkbox 
              checked={enabledFields.thickness || false}
              onCheckedChange={() => toggleField('thickness')}
              data-testid="checkbox-bulk-thickness"
            />
            <div className="flex-1 space-y-2">
              <Label className={!enabledFields.thickness ? 'text-muted-foreground' : ''}>Grubość (mm)</Label>
              <Input
                type="number"
                disabled={!enabledFields.thickness}
                value={fieldValues.thickness || ''}
                onChange={(e) => setFieldValue('thickness', e.target.value ? parseFloat(e.target.value) : null)}
                placeholder="np. 18"
                data-testid="input-bulk-thickness"
              />
            </div>
          </div>

          {/* Status */}
          <div className="flex items-start gap-3 p-3 rounded-lg border bg-card">
            <Checkbox 
              checked={enabledFields.status || false}
              onCheckedChange={() => toggleField('status')}
              data-testid="checkbox-bulk-status"
            />
            <div className="flex-1 space-y-2">
              <Label className={!enabledFields.status ? 'text-muted-foreground' : ''}>Status</Label>
              <Combobox
                options={[
                  { value: 'stock', label: 'W magazynie' },
                  { value: 'ordered', label: 'Zamówione' },
                  { value: 'reserved', label: 'Zarezerwowane' },
                  { value: 'out_of_stock', label: 'Brak' },
                ]}
                value={fieldValues.status || ''}
                onChange={(v: string) => setFieldValue('status', v)}
                placeholder="Wybierz status..."
                searchPlaceholder="Szukaj statusu..."
                emptyText="Nie znaleziono"
                disabled={!enabledFields.status}
              />
            </div>
          </div>

          {/* Carrier */}
          <div className="flex items-start gap-3 p-3 rounded-lg border bg-card">
            <Checkbox 
              checked={enabledFields.carrier_id || false}
              onCheckedChange={() => toggleField('carrier_id')}
              data-testid="checkbox-bulk-carrier"
            />
            <div className="flex-1 space-y-2">
              <Label className={!enabledFields.carrier_id ? 'text-muted-foreground' : ''}>Nośnik</Label>
              <Combobox
                options={[
                  { value: 'null', label: 'Brak nośnika' },
                  ...carriers.map((c: any) => ({ value: c.id.toString(), label: c.name }))
                ]}
                value={fieldValues.carrier_id?.toString() || 'null'}
                onChange={(v: string) => setFieldValue('carrier_id', v === 'null' ? null : parseInt(v))}
                placeholder="Wybierz nośnik..."
                searchPlaceholder="Szukaj nośnika..."
                emptyText="Nie znaleziono"
                disabled={!enabledFields.carrier_id}
              />
            </div>
          </div>

          {/* Location */}
          <div className="flex items-start gap-3 p-3 rounded-lg border bg-card">
            <Checkbox 
              checked={enabledFields.location_id || false}
              onCheckedChange={() => toggleField('location_id')}
              data-testid="checkbox-bulk-location"
            />
            <div className="flex-1 space-y-2">
              <Label className={!enabledFields.location_id ? 'text-muted-foreground' : ''}>Lokalizacja</Label>
              <Combobox
                options={[
                  { value: 'null', label: 'Brak lokalizacji' },
                  ...locations.map((l: any) => ({ value: l.id.toString(), label: l.name }))
                ]}
                value={fieldValues.location_id?.toString() || 'null'}
                onChange={(v: string) => setFieldValue('location_id', v === 'null' ? null : parseInt(v))}
                placeholder="Wybierz lokalizację..."
                searchPlaceholder="Szukaj lokalizacji..."
                emptyText="Nie znaleziono"
                disabled={!enabledFields.location_id}
              />
            </div>
          </div>

          {/* Group */}
          <div className="flex items-start gap-3 p-3 rounded-lg border bg-card">
            <Checkbox 
              checked={enabledFields.group_id || false}
              onCheckedChange={() => toggleField('group_id')}
              data-testid="checkbox-bulk-group"
            />
            <div className="flex-1 space-y-2">
              <Label className={!enabledFields.group_id ? 'text-muted-foreground' : ''}>Grupa</Label>
              <Combobox
                options={[
                  { value: 'null', label: 'Bez grupy' },
                  ...groups.map((g: any) => ({ value: g.id.toString(), label: g.name }))
                ]}
                value={fieldValues.group_id?.toString() || 'null'}
                onChange={(v: string) => setFieldValue('group_id', v === 'null' ? null : parseInt(v))}
                placeholder="Wybierz grupę..."
                searchPlaceholder="Szukaj grupy..."
                emptyText="Nie znaleziono"
                disabled={!enabledFields.group_id}
              />
            </div>
          </div>

          {/* Is Drilled */}
          <div className="flex items-start gap-3 p-3 rounded-lg border bg-card">
            <Checkbox 
              checked={enabledFields.is_drilled || false}
              onCheckedChange={() => toggleField('is_drilled')}
              data-testid="checkbox-bulk-is-drilled"
            />
            <div className="flex-1 space-y-2">
              <Label className={!enabledFields.is_drilled ? 'text-muted-foreground' : ''}>Wiercona</Label>
              <Combobox
                options={[
                  { value: 'true', label: 'Tak' },
                  { value: 'false', label: 'Nie' },
                ]}
                value={fieldValues.is_drilled?.toString() || ''}
                onChange={(v: string) => setFieldValue('is_drilled', v === 'true')}
                placeholder="Wybierz..."
                searchPlaceholder="Szukaj..."
                emptyText="Nie znaleziono"
                disabled={!enabledFields.is_drilled}
              />
            </div>
          </div>

          {/* Is Complete */}
          <div className="flex items-start gap-3 p-3 rounded-lg border bg-card">
            <Checkbox 
              checked={enabledFields.is_complete || false}
              onCheckedChange={() => toggleField('is_complete')}
              data-testid="checkbox-bulk-is-complete"
            />
            <div className="flex-1 space-y-2">
              <Label className={!enabledFields.is_complete ? 'text-muted-foreground' : ''}>Komplet</Label>
              <Combobox
                options={[
                  { value: 'true', label: 'Tak' },
                  { value: 'false', label: 'Nie' },
                ]}
                value={fieldValues.is_complete?.toString() || ''}
                onChange={(v: string) => setFieldValue('is_complete', v === 'true')}
                placeholder="Wybierz..."
                searchPlaceholder="Szukaj..."
                emptyText="Nie znaleziono"
                disabled={!enabledFields.is_complete}
              />
            </div>
          </div>
        </div>

        <DialogFooter className="flex items-center justify-between gap-2">
          <div className="text-sm text-muted-foreground">
            Wybrano {enabledCount} pól do edycji
          </div>
          <div className="flex gap-2">
            <Button
              variant="outline"
              onClick={() => onOpenChange(false)}
              data-testid="button-cancel-bulk-edit"
            >
              Anuluj
            </Button>
            <Button
              onClick={handleSubmit}
              disabled={isPending || enabledCount === 0}
              data-testid="button-submit-bulk-edit"
            >
              {isPending && <Loader2 className="w-4 h-4 mr-2 animate-spin" />}
              Zapisz zmiany
            </Button>
          </div>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}

// Sortable column item component for drag & drop reordering
function SortableColumnItem({ 
  col, 
  columnVisibility, 
  setColumnVisibility, 
  getColumnWidth, 
  setColumnWidths, 
  columnWidths,
  defaultColumnWidths 
}: { 
  col: { key: string; label: string; default: boolean };
  columnVisibility: Record<string, boolean>;
  setColumnVisibility: (v: Record<string, boolean>) => void;
  getColumnWidth: (key: string) => number;
  setColumnWidths: (v: Record<string, number>) => void;
  columnWidths: Record<string, number>;
  defaultColumnWidths: Record<string, number>;
}) {
  const {
    attributes,
    listeners,
    setNodeRef,
    transform,
    transition,
    isDragging,
  } = useSortable({ id: col.key });

  const style = {
    transform: CSS.Transform.toString(transform),
    transition,
    opacity: isDragging ? 0.5 : 1,
  };

  return (
    <div 
      ref={setNodeRef} 
      style={style}
      className="flex items-center gap-2 px-2 py-1 hover:bg-accent rounded-sm"
    >
      <div 
        {...attributes} 
        {...listeners}
        className="cursor-grab active:cursor-grabbing text-muted-foreground hover:text-foreground"
      >
        <GripVertical className="h-4 w-4" />
      </div>
      <Checkbox
        checked={columnVisibility[col.key] !== false}
        onCheckedChange={(checked) => setColumnVisibility({...columnVisibility, [col.key]: !!checked})}
        data-testid={`checkbox-column-${col.key}`}
      />
      <span className="flex-1 text-sm">{col.label}</span>
      <input
        type="number"
        value={getColumnWidth(col.key)}
        onChange={(e) => {
          const newWidth = parseInt(e.target.value) || defaultColumnWidths[col.key];
          setColumnWidths({...columnWidths, [col.key]: Math.max(30, Math.min(500, newWidth))});
        }}
        className="w-14 h-6 px-1 text-xs border rounded bg-background text-center"
        min={30}
        max={500}
        title="Szerokość kolumny (px)"
        data-testid={`input-column-width-${col.key}`}
      />
    </div>
  );
}

export default function WarehouseCategoryPage() {
  const { toast } = useToast();
  const [, setLocation] = useLocation();
  const [, params] = useRoute("/warehouse/:category/group/:groupCode");
  const [, paramsNoGroup] = useRoute("/warehouse/:category");
  const category = params?.category || paramsNoGroup?.category || 'okucia';
  const groupCode = params?.groupCode || null;

  // Load filters from localStorage
  const loadFilters = () => {
    try {
      const saved = localStorage.getItem(STORAGE_KEY_PREFIX + category);
      if (saved) {
        return JSON.parse(saved);
      }
    } catch (error) {
      console.error('Error loading filters from localStorage:', error);
    }
    return {};
  };

  const savedFilters = loadFilters();

  const [searchQuery, setSearchQuery] = useState(savedFilters.searchQuery || "");
  const [page, setPage] = useState(savedFilters.page || 1);
  const [limit, setLimit] = useState(savedFilters.limit || 25);
  const [sortBy, setSortBy] = useState(savedFilters.sortBy || 'name');
  const [sortOrder, setSortOrder] = useState<'ASC' | 'DESC'>(savedFilters.sortOrder || 'ASC');
  const [selectedMaterials, setSelectedMaterials] = useState<number[]>([]);
  const [isSelectingAllPages, setIsSelectingAllPages] = useState(false);
  
  // Archive filter state (for produkty-spakowane)
  const [showArchived, setShowArchived] = useState(false);
  
  // Formatki filter states (with localStorage persistence)
  const [formatkiFiltersExpanded, setFormatkiFiltersExpanded] = useState<boolean>(
    savedFilters.formatkiFiltersExpanded !== undefined ? savedFilters.formatkiFiltersExpanded : true
  );
  const [colorFilterStates, setColorFilterStates] = useState<FilterStates>(savedFilters.colorFilterStates || {});
  const [lengthFilterStates, setLengthFilterStates] = useState<FilterStates>(savedFilters.lengthFilterStates || {});
  const [widthFilterStates, setWidthFilterStates] = useState<FilterStates>(savedFilters.widthFilterStates || {});
  const [stockFilter, setStockFilter] = useState<'all' | 'inStock' | 'outOfStock'>(savedFilters.stockFilter || 'all');
  
  // Extended formatki filter states
  const [cz1FilterStates, setCz1FilterStates] = useState<FilterStates>(savedFilters.cz1FilterStates || {});
  const [cz2FilterStates, setCz2FilterStates] = useState<FilterStates>(savedFilters.cz2FilterStates || {});
  const [selectedFurnitureTypes, setSelectedFurnitureTypes] = useState<string[]>(savedFilters.selectedFurnitureTypes || []);
  const [selectedSources, setSelectedSources] = useState<string[]>(savedFilters.selectedSources || []);
  const [locationFilterStates, setLocationFilterStates] = useState<FilterStates>(savedFilters.locationFilterStates || {});
  const [carrierFilterStates, setCarrierFilterStates] = useState<FilterStates>(savedFilters.carrierFilterStates || {});
  
  // Filter group search states (for filtering within groups)
  const [filterSearch, setFilterSearch] = useState<Record<string, string>>({
    colors: '', lengths: '', widths: '', cz1: '', cz2: '',
    furnitureTypes: '', locations: '', carriers: ''
  });
  
  // Filter settings dropdown open state
  const [filterSettingsOpen, setFilterSettingsOpen] = useState(false);
  
  // Column visibility settings dropdown state
  const [columnSettingsOpen, setColumnSettingsOpen] = useState(false);
  
  // Formatki column definitions
  const formatkiColumns = [
    { key: 'rowNumber', label: '#', default: true },
    { key: 'internalCode', label: 'Kod wew.', default: true },
    { key: 'symbol', label: 'Symbol', default: true },
    { key: 'image', label: 'Foto', default: true },
    { key: 'name', label: 'Nazwa', default: true },
    { key: 'quantity', label: 'Stan', default: true },
    { key: 'reserved', label: 'Zarezerwow.', default: true },
    { key: 'available', label: 'Dostępne', default: true },
    { key: 'source', label: 'Rodz', default: true },
    { key: 'length', label: 'Długość', default: true },
    { key: 'width', label: 'Szerokość', default: true },
    { key: 'color', label: 'Kolor', default: true },
    { key: 'thickness', label: 'Grubość', default: true },
    { key: 'cz1', label: 'CZ1', default: true },
    { key: 'cz2', label: 'CZ2', default: true },
    { key: 'furnitureType', label: 'Typ mebla', default: true },
    { key: 'plyta', label: 'Płyta', default: false },
    { key: 'obrzeze', label: 'Obrzeże', default: false },
    { key: 'complete', label: 'Kompletne', default: false },
    { key: 'status', label: 'Status', default: false },
    { key: 'sourceRef', label: 'Źródło', default: false },
    { key: 'reference', label: 'Referencja', default: false },
    { key: 'cost', label: 'Koszt mat.', default: true },
    { key: 'location', label: 'Lokalizacja', default: true },
    { key: 'carrier', label: 'Nośnik', default: true },
    { key: 'duplicates', label: 'Duplikaty', default: true },
  ];
  
  const defaultColumnOrder = formatkiColumns.map(col => col.key);
  
  // Column visibility state (default all visible based on column defaults)
  const defaultColumnVisibility = formatkiColumns.reduce((acc, col) => {
    acc[col.key] = col.default;
    return acc;
  }, {} as Record<string, boolean>);
  
  const [columnVisibility, setColumnVisibility] = useState<Record<string, boolean>>(() => {
    // First try server settings, then localStorage
    return savedFilters.columnVisibility || defaultColumnVisibility;
  });
  
  // Helper to check if column is visible
  const isColumnVisible = (key: string) => columnVisibility[key] !== false;
  
  // Column widths state (default widths from column definitions)
  const defaultColumnWidths: Record<string, number> = {
    rowNumber: 40, internalCode: 60, symbol: 100, image: 40, name: 200, quantity: 70, reserved: 70, available: 70,
    source: 60, length: 70, width: 70, color: 80, thickness: 70, cz1: 80, cz2: 80,
    furnitureType: 100, plyta: 100, obrzeze: 100, complete: 100, status: 80,
    sourceRef: 100, reference: 100, cost: 90, location: 120, carrier: 120, duplicates: 100
  };
  
  const [columnWidths, setColumnWidths] = useState<Record<string, number>>(() => {
    return savedFilters.columnWidths || defaultColumnWidths;
  });
  
  // Column order for drag & drop reordering
  const [columnOrder, setColumnOrder] = useState<string[]>(() => {
    const saved = savedFilters.columnOrder || [];
    if (saved.length === 0) return defaultColumnOrder;
    // Add any new columns that might not be in saved columnOrder yet
    const allKeys = formatkiColumns.map(col => col.key);
    const missingKeys = allKeys.filter(key => !saved.includes(key));
    return [...saved, ...missingKeys];
  });
  
  // DnD sensors for column reordering
  const sensors = useSensors(
    useSensor(PointerSensor, { activationConstraint: { distance: 5 } }),
    useSensor(KeyboardSensor, { coordinateGetter: sortableKeyboardCoordinates })
  );
  
  // Handle drag end for column reordering
  const handleColumnDragEnd = (event: DragEndEvent) => {
    const { active, over } = event;
    if (over && active.id !== over.id) {
      setColumnOrder((items) => {
        const oldIndex = items.indexOf(active.id as string);
        const newIndex = items.indexOf(over.id as string);
        return arrayMove(items, oldIndex, newIndex);
      });
    }
  };
  
  // Get ordered columns based on columnOrder
  const getOrderedColumns = () => {
    const orderedCols = columnOrder
      .map(key => formatkiColumns.find(col => col.key === key))
      .filter((col): col is typeof formatkiColumns[0] => col !== undefined);
    // Add any new columns that might not be in columnOrder yet
    const missingCols = formatkiColumns.filter(col => !columnOrder.includes(col.key));
    return [...orderedCols, ...missingCols];
  };
  
  // Render column header for formatki table based on column key
  const renderFormatkiHeader = (colKey: string) => {
    if (!isColumnVisible(colKey)) return null;
    
    const headerLabels: Record<string, string> = {
      rowNumber: '#',
      internalCode: 'Kod wew.',
      symbol: 'Symbol',
      image: '',
      name: 'Nazwa',
      quantity: 'Stan',
      reserved: 'Zarezerwow.',
      available: 'Dostępne',
      source: 'Rodz',
      length: 'Długość',
      width: 'Szerokość',
      color: 'Kolor',
      thickness: 'Grubość',
      cz1: 'CZ1',
      cz2: 'CZ2',
      furnitureType: 'Typ mebla',
      plyta: 'Płyta',
      obrzeze: 'Obrzeże',
      complete: 'Kompletne',
      status: 'Status',
      sourceRef: 'Źródło',
      reference: 'Referencja',
      cost: 'Koszt mat.',
      location: 'Lokalizacja',
      carrier: 'Nośnik',
      duplicates: 'Duplikaty',
    };
    
    const sortableColumns: Record<string, string> = {
      name: 'name',
      color: 'colorCode',
      length: 'length',
      width: 'width',
      thickness: 'thickness',
      quantity: 'quantity',
      reserved: 'reservedQuantity',
      available: 'availableQuantity',
      plyta: 'boardCode',
      obrzeze: 'edgingCode',
      symbol: 'originalSymbol',
      cost: 'price',
      internalCode: 'internalCode',
    };
    
    const sortKey = sortableColumns[colKey];
    const isSortable = !!sortKey;
    const isCurrentSort = sortBy === sortKey;
    
    const handleSort = () => {
      if (!isSortable) return;
      if (isCurrentSort) {
        setSortOrder(sortOrder === 'ASC' ? 'DESC' : 'ASC');
      } else {
        setSortBy(sortKey);
        setSortOrder('ASC');
      }
    };
    
    return (
      <div 
        key={colKey}
        className={`shrink-0 relative flex items-center gap-1 ${isSortable ? 'cursor-pointer hover:text-primary select-none' : ''}`}
        style={{width: getColumnWidth(colKey)}}
        onClick={isSortable ? handleSort : undefined}
      >
        <span className="truncate">{headerLabels[colKey] || colKey}</span>
        {isSortable && isCurrentSort && (
          <span className="text-primary text-xs">{sortOrder === 'ASC' ? '↑' : '↓'}</span>
        )}
        <ResizeHandle columnKey={colKey} />
      </div>
    );
  };
  
  // Render cell for formatki table row based on column key
  const renderFormatkiCell = (colKey: string, material: any, rowIndex: number, isZeroQuantity: boolean) => {
    if (!isColumnVisible(colKey)) return null;
    
    const width = getColumnWidth(colKey);
    
    switch (colKey) {
      case 'rowNumber':
        return (
          <div key={colKey} className="shrink-0 text-center text-xs text-muted-foreground font-mono" style={{width}}>
            {rowIndex + 1}
          </div>
        );
      case 'internalCode':
        return (
          <div key={colKey} className="shrink-0 text-center text-xs text-muted-foreground font-mono" style={{width}}>
            {material.id}
          </div>
        );
      case 'symbol':
        return (
          <div key={colKey} className="shrink-0 text-xs text-muted-foreground truncate" style={{width}}>
            {material.originalSymbol || '-'}
          </div>
        );
      case 'image':
        const imageUrl = material.primaryImageUrl || (material.gallery && material.gallery.length > 0 ? material.gallery[0] : null);
        return (
          <div key={colKey} className="shrink-0 flex items-center justify-center" style={{width}}>
            {imageUrl ? (
              <img src={imageUrl} alt={material.name} className="w-8 h-8 object-cover border rounded" />
            ) : (
              <div className="w-8 h-8 bg-muted border flex items-center justify-center rounded">
                <Upload className="w-4 h-4 text-muted-foreground" />
              </div>
            )}
          </div>
        );
      case 'name':
        return (
          <div key={colKey} className="shrink-0 flex items-center gap-2 min-w-0" style={{width}}>
            <span className="font-medium text-sm truncate">{material.name}</span>
          </div>
        );
      case 'quantity':
        return (
          <div key={colKey} className="shrink-0 flex items-center justify-center" style={{width}}>
            <Badge 
              variant="outline" 
              className={`text-xs font-medium min-w-[45px] justify-center ${
                isZeroQuantity ? 'bg-gray-200 dark:bg-gray-700 text-gray-500 dark:text-gray-400 border-gray-300 dark:border-gray-600' : ''
              }`}
            >
              {material.quantity || 0}
            </Badge>
          </div>
        );
      case 'reserved':
        return (
          <div key={colKey} className="shrink-0 flex items-center justify-center" style={{width}}>
            <Badge 
              variant={isZeroQuantity ? "outline" : (material.quantityReserved && material.quantityReserved > 0 ? "destructive" : "outline")} 
              className={`text-xs font-medium min-w-[45px] justify-center ${
                isZeroQuantity ? 'bg-gray-200 dark:bg-gray-700 text-gray-500 dark:text-gray-400 border-gray-300 dark:border-gray-600' : ''
              }`}
            >
              {material.quantityReserved || 0}
            </Badge>
          </div>
        );
      case 'available':
        return (
          <div key={colKey} className="shrink-0 flex items-center justify-center" style={{width}}>
            <Badge 
              variant={isZeroQuantity ? "outline" : "default"} 
              className={`text-xs font-medium min-w-[45px] justify-center ${
                isZeroQuantity 
                  ? 'bg-gray-200 dark:bg-gray-700 text-gray-500 dark:text-gray-400 border-gray-300 dark:border-gray-600' 
                  : 'bg-green-600 hover:bg-green-700'
              }`}
            >
              {(material.quantity || 0) - (material.quantityReserved || 0)}
            </Badge>
          </div>
        );
      case 'source':
        return (
          <div key={colKey} className="shrink-0 text-xs truncate" style={{width}}>
            {material.specifications?.source || '-'}
          </div>
        );
      case 'length':
        return (
          <div key={colKey} className="shrink-0 text-xs" style={{width}}>
            {material.specifications?.length || '-'}
          </div>
        );
      case 'width':
        return (
          <div key={colKey} className="shrink-0 text-xs" style={{width}}>
            {material.specifications?.width || '-'}
          </div>
        );
      case 'color':
        const colorVal = material.specifications?.colorCode;
        const colorDict = colorDictionaries.find((d: any) => d.code === colorVal);
        return (
          <div key={colKey} className="shrink-0" style={{width}}>
            {colorVal ? (
              <div 
                className="h-6 text-xs font-medium flex items-center justify-center border truncate px-1"
                style={{
                  backgroundColor: colorDict?.color || '#e5e7eb',
                  color: getTextColorForBackground(colorDict?.color),
                  borderColor: needsVisibleBorder(colorDict?.color) ? '#d1d5db' : 'transparent'
                }}
                title={colorDict?.readableName || colorVal}
              >
                {colorVal}
              </div>
            ) : <span className="text-xs text-muted-foreground">-</span>}
          </div>
        );
      case 'thickness':
        return (
          <div key={colKey} className="shrink-0 text-xs" style={{width}}>
            {material.specifications?.thickness || '-'}
          </div>
        );
      case 'cz1':
        return (
          <div key={colKey} className="shrink-0 text-xs truncate" style={{width}}>
            {material.specifications?.cz1 || '-'}
          </div>
        );
      case 'cz2':
        return (
          <div key={colKey} className="shrink-0 text-xs truncate" style={{width}}>
            {material.specifications?.cz2 || '-'}
          </div>
        );
      case 'furnitureType':
        return (
          <div key={colKey} className="shrink-0 text-xs truncate" style={{width}}>
            {material.specifications?.furnitureType || '-'}
          </div>
        );
      case 'plyta':
        const plytaCode = material.specifications?.boardCode;
        const colorCode = material.specifications?.color;
        // Try to find by boardCode first, then by color pattern (18_COLOR)
        let plytaItem = plytyMaterials.find((p: any) => p.internalCode === plytaCode);
        if (!plytaItem && colorCode) {
          plytaItem = plytyMaterials.find((p: any) => p.internalCode === `18_${colorCode}`);
        }
        return (
          <div key={colKey} className="shrink-0 text-xs truncate" style={{width}}>
            {plytaItem?.name || '-'}
          </div>
        );
      case 'obrzeze':
        const obrzezeCode = material.specifications?.edgingCode;
        const colorForObrzeze = material.specifications?.color;
        // Try to find by edgingCode first, then by color name
        let obrzezeItem = obrzezaMaterials.find((o: any) => o.internalCode === obrzezeCode);
        if (!obrzezeItem && colorForObrzeze) {
          obrzezeItem = obrzezaMaterials.find((o: any) => o.internalCode === colorForObrzeze || o.name === colorForObrzeze);
        }
        return (
          <div key={colKey} className="shrink-0 text-xs truncate" style={{width}}>
            {obrzezeItem?.name || '-'}
          </div>
        );
      case 'complete':
        return (
          <div key={colKey} className="shrink-0 flex items-center justify-center" style={{width}}>
            {material.specifications?.isComplete ? (
              <Check className="h-4 w-4 text-green-600" />
            ) : (
              <X className="h-4 w-4 text-muted-foreground" />
            )}
          </div>
        );
      case 'status':
        return (
          <div key={colKey} className="shrink-0 text-xs truncate" style={{width}}>
            {material.specifications?.status || '-'}
          </div>
        );
      case 'sourceRef':
        return (
          <div key={colKey} className="shrink-0 text-xs truncate" style={{width}}>
            {material.specifications?.sourceRef || '-'}
          </div>
        );
      case 'reference':
        return (
          <div key={colKey} className="shrink-0 text-xs truncate" style={{width}}>
            {material.reference || '-'}
          </div>
        );
      case 'cost':
        const cost = material.materialCost;
        return (
          <div key={colKey} className="shrink-0 text-xs" style={{width}}>
            {cost ? `${cost.toFixed(2)} zł` : '-'}
          </div>
        );
      case 'location':
        const loc = material.locationId;
        const locItem = locations.find((l: any) => l.id === loc);
        return (
          <div key={colKey} className="shrink-0 text-xs truncate" style={{width}}>
            {locItem?.code || '-'}
          </div>
        );
      case 'carrier':
        const carr = material.carrierId;
        const carrItem = carriers.find((c: any) => c.id === carr);
        return (
          <div key={colKey} className="shrink-0 text-xs truncate" style={{width}}>
            {carrItem?.code || '-'}
          </div>
        );
      case 'duplicates':
        const dupCount = material.duplicateCount || 0;
        const dupIds: number[] = material.duplicateIds || [];
        if (dupCount <= 1) {
          return (
            <div key={colKey} className="shrink-0 text-xs text-muted-foreground" style={{width}}>
              -
            </div>
          );
        }
        const otherDupIds = dupIds.filter((id: number) => id !== material.id);
        return (
          <div key={colKey} className="shrink-0 text-xs" style={{width}}>
            <Popover>
              <PopoverTrigger asChild>
                <Badge 
                  variant="outline" 
                  className="cursor-pointer bg-amber-500/20 text-amber-700 dark:text-amber-400 border-amber-500/50 hover:bg-amber-500/30"
                >
                  {dupCount}x
                </Badge>
              </PopoverTrigger>
              <PopoverContent className="w-auto p-2" align="start">
                <div className="text-xs space-y-1">
                  <div className="font-medium text-muted-foreground mb-2">Duplikaty ({dupCount}):</div>
                  {otherDupIds.map((dupId: number) => (
                    <a 
                      key={dupId}
                      href={`/warehouse/formatki/${dupId}`}
                      className="block text-primary hover:underline"
                      onClick={(e) => {
                        e.preventDefault();
                        window.location.href = `/warehouse/formatki/${dupId}`;
                      }}
                    >
                      ID: {dupId}
                    </a>
                  ))}
                </div>
              </PopoverContent>
            </Popover>
          </div>
        );
      default:
        return null;
    }
  };
  
  // Resizing state
  const [resizingColumn, setResizingColumn] = useState<string | null>(null);
  const [resizeStartX, setResizeStartX] = useState(0);
  const [resizeStartWidth, setResizeStartWidth] = useState(0);
  
  // Get column width helper
  const getColumnWidth = (key: string) => columnWidths[key] || defaultColumnWidths[key] || 100;
  
  // Column resize handlers
  const handleResizeStart = (e: React.MouseEvent, columnKey: string) => {
    e.preventDefault();
    e.stopPropagation();
    setResizingColumn(columnKey);
    setResizeStartX(e.clientX);
    setResizeStartWidth(getColumnWidth(columnKey));
  };
  
  // Global mouse move/up handlers for resize
  useEffect(() => {
    if (!resizingColumn) return;
    
    const handleMouseMove = (e: MouseEvent) => {
      const delta = e.clientX - resizeStartX;
      const newWidth = Math.max(30, Math.min(500, resizeStartWidth + delta));
      setColumnWidths(prev => ({...prev, [resizingColumn]: newWidth}));
    };
    
    const handleMouseUp = () => {
      setResizingColumn(null);
    };
    
    document.addEventListener('mousemove', handleMouseMove);
    document.addEventListener('mouseup', handleMouseUp);
    
    return () => {
      document.removeEventListener('mousemove', handleMouseMove);
      document.removeEventListener('mouseup', handleMouseUp);
    };
  }, [resizingColumn, resizeStartX, resizeStartWidth]);
  
  // Resize handle component
  const ResizeHandle = ({ columnKey }: { columnKey: string }) => (
    <div
      className="absolute right-0 top-0 bottom-0 w-1 cursor-col-resize hover:bg-primary/50 active:bg-primary"
      onMouseDown={(e) => handleResizeStart(e, columnKey)}
      title="Przeciągnij aby zmienić szerokość"
    />
  );
  
  // Filter groups visibility (for user preferences)
  const [filterGroupsVisible, setFilterGroupsVisible] = useState<Record<string, boolean>>(() => {
    const defaultVisible: Record<string, boolean> = {
      stock: true,
      colors: true,
      lengths: true,
      widths: true,
      cz1: false,
      cz2: false,
      furnitureTypes: false,
      sources: false,
      locations: false,
      carriers: false,
    };
    return savedFilters.filterGroupsVisible || defaultVisible;
  });
  
  // User settings loading state
  const [userSettingsLoaded, setUserSettingsLoaded] = useState(false);


  // Fetch user settings from API (server-side persistence)
  const userSettingsPageSlug = `warehouse-${category}`;
  const { data: serverSettings, isSuccess: serverSettingsLoaded } = useQuery<{
    filters?: Record<string, any>;
    columnVisibility?: Record<string, boolean>;
    columnOrder?: string[];
    columnWidths?: Record<string, number>;
  }>({
    queryKey: ['/api/user-settings', userSettingsPageSlug],
  });
  
  // Mutation to save user settings to API
  const saveUserSettings = useMutation({
    mutationFn: async (settings: {
      filters?: Record<string, any>;
      columnVisibility?: Record<string, boolean>;
      columnOrder?: string[];
      columnWidths?: Record<string, number>;
    }) => {
      return apiRequest('PUT', `/api/user-settings/${userSettingsPageSlug}`, settings);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/user-settings', userSettingsPageSlug] });
    },
  });
  
  // Apply server settings when loaded (priority over localStorage)
  useEffect(() => {
    if (serverSettingsLoaded && !userSettingsLoaded) {
      // Apply filters from server if they exist
      if (serverSettings?.filters) {
        const f = serverSettings.filters;
        if (f.colorFilterStates) setColorFilterStates(f.colorFilterStates || {});
        if (f.lengthFilterStates) setLengthFilterStates(f.lengthFilterStates);
        if (f.widthFilterStates) setWidthFilterStates(f.widthFilterStates);
        if (f.stockFilter) setStockFilter(f.stockFilter);
        if (f.cz1FilterStates) setCz1FilterStates(f.cz1FilterStates);
        if (f.cz2FilterStates) setCz2FilterStates(f.cz2FilterStates);
        if (f.selectedFurnitureTypes) setSelectedFurnitureTypes(f.selectedFurnitureTypes);
        if (f.selectedSources) setSelectedSources(f.selectedSources);
        if (f.locationFilterStates) setLocationFilterStates(f.locationFilterStates);
        if (f.carrierFilterStates) setCarrierFilterStates(f.carrierFilterStates || {});
        if (f.filterGroupsVisible) setFilterGroupsVisible(f.filterGroupsVisible);
        if (f.formatkiFiltersExpanded !== undefined) setFormatkiFiltersExpanded(f.formatkiFiltersExpanded);
      }
      // Apply column visibility from server if exists
      if (serverSettings?.columnVisibility) {
        setColumnVisibility(serverSettings.columnVisibility);
      }
      // Apply column widths from server if exists
      if (serverSettings?.columnWidths) {
        setColumnWidths(serverSettings.columnWidths);
      }
      if (serverSettings?.columnOrder && serverSettings.columnOrder.length > 0) {
        // Add any new columns that might not be in saved columnOrder yet
        const savedOrder = serverSettings.columnOrder;
        const allKeys = formatkiColumns.map(col => col.key);
        const missingKeys = allKeys.filter(key => !savedOrder.includes(key));
        setColumnOrder([...savedOrder, ...missingKeys]);
      }
      // Always mark as loaded once query settles (even if no filters returned)
      setUserSettingsLoaded(true);
    }
  }, [serverSettingsLoaded, serverSettings, userSettingsLoaded]);
  
  // Reset userSettingsLoaded when category changes
  useEffect(() => {
    setUserSettingsLoaded(false);
  }, [category]);
  
  // Dialog states
  const [isAddGroupDialogOpen, setIsAddGroupDialogOpen] = useState(false);
  const [isEditGroupDialogOpen, setIsEditGroupDialogOpen] = useState(false);
  const [isAddMaterialDialogOpen, setIsAddMaterialDialogOpen] = useState(false);
  const [isEditMaterialDialogOpen, setIsEditMaterialDialogOpen] = useState(false);
  const [isBulkImportCatalogDialogOpen, setIsBulkImportCatalogDialogOpen] = useState(false);
  
  // Form states for new group
  const [newGroupName, setNewGroupName] = useState("");
  const [newGroupCode, setNewGroupCode] = useState("");
  const [newGroupDescription, setNewGroupDescription] = useState("");
  const [newGroupColor, setNewGroupColor] = useState("");
  
  // Form states for edit group
  const [editingGroup, setEditingGroup] = useState<any | null>(null);
  const [editGroupName, setEditGroupName] = useState("");
  const [editGroupCode, setEditGroupCode] = useState("");
  const [editGroupDescription, setEditGroupDescription] = useState("");
  const [editGroupDisplayOrder, setEditGroupDisplayOrder] = useState("0");
  const [editGroupColor, setEditGroupColor] = useState("");
  
  // Form states for new material
  const [newMaterialName, setNewMaterialName] = useState("");
  const [newMaterialInternalCode, setNewMaterialInternalCode] = useState("");
  const [newMaterialSupplierCode, setNewMaterialSupplierCode] = useState("");
  const [newMaterialDescription, setNewMaterialDescription] = useState("");
  const [newMaterialUnitOfMeasure, setNewMaterialUnitOfMeasure] = useState("szt");
  const [newMaterialPrice, setNewMaterialPrice] = useState("");
  const [newMaterialGroupId, setNewMaterialGroupId] = useState<number | null>(null);
  const [isInternalCodeManuallyEdited, setIsInternalCodeManuallyEdited] = useState(false);
  
  // Form states for new formatka (stock panel)
  const [newFormatkaCz1, setNewFormatkaCz1] = useState("");
  const [newFormatkaCz2, setNewFormatkaCz2] = useState("");
  const [newFormatkaLength, setNewFormatkaLength] = useState("");
  const [newFormatkaWidth, setNewFormatkaWidth] = useState("");
  const [newFormatkaThickness, setNewFormatkaThickness] = useState("18");
  const [newFormatkaBoardCode, setNewFormatkaBoardCode] = useState("");
  const [newFormatkaEdgingCode, setNewFormatkaEdgingCode] = useState("");
  const [newFormatkaColorCode, setNewFormatkaColorCode] = useState("");
  const [newFormatkaEdge1, setNewFormatkaEdge1] = useState(false);
  const [newFormatkaEdge2, setNewFormatkaEdge2] = useState(false);
  const [newFormatkaEdge3, setNewFormatkaEdge3] = useState(false);
  const [newFormatkaEdge4, setNewFormatkaEdge4] = useState(false);
  
  // Form states for edit material
  const [editingMaterial, setEditingMaterial] = useState<Material | null>(null);
  const [editMaterialName, setEditMaterialName] = useState("");
  const [editMaterialInternalCode, setEditMaterialInternalCode] = useState("");
  const [editMaterialSupplierCode, setEditMaterialSupplierCode] = useState("");
  const [editMaterialDescription, setEditMaterialDescription] = useState("");
  const [editMaterialUnitOfMeasure, setEditMaterialUnitOfMeasure] = useState("szt");
  const [editMaterialPrice, setEditMaterialPrice] = useState("");
  const [editMaterialGroupId, setEditMaterialGroupId] = useState<number | null>(null);
  const [editMaterialGallery, setEditMaterialGallery] = useState<string[]>([]);
  const [isEditInternalCodeManuallyEdited, setIsEditInternalCodeManuallyEdited] = useState(true);
  const [isDragging, setIsDragging] = useState(false);
  const [uploadHoverMaterials, setUploadHoverMaterials] = useState<Record<number, boolean>>({});
  const [uploadingMaterials, setUploadingMaterials] = useState<Record<number, boolean>>({});
  
  // Duplicate material states
  const [duplicateMaterial, setDuplicateMaterial] = useState<Material | null>(null);
  const [duplicateTargetGroupId, setDuplicateTargetGroupId] = useState<number | null>(null);
  const [showDuplicateDialog, setShowDuplicateDialog] = useState(false);
  
  // Bulk edit states
  const [showBulkEditPriceDialog, setShowBulkEditPriceDialog] = useState(false);
  const [showBulkEditUnitDialog, setShowBulkEditUnitDialog] = useState(false);
  const [bulkEditPrice, setBulkEditPrice] = useState("");
  const [bulkEditUnit, setBulkEditUnit] = useState("szt");
  
  // Bulk operations states
  const [showBulkInventoryDialog, setShowBulkInventoryDialog] = useState(false);
  const [showBulkCarrierDialog, setShowBulkCarrierDialog] = useState(false);
  const [showBulkGroupDialog, setShowBulkGroupDialog] = useState(false);
  const [showBulkLocationDialog, setShowBulkLocationDialog] = useState(false);
  const [showBulkFormatkaEditDialog, setShowBulkFormatkaEditDialog] = useState(false);
  
  // Bulk operations selected values for comboboxes
  const [bulkCarrierId, setBulkCarrierId] = useState<string>("");
  const [bulkLocationId, setBulkLocationId] = useState<string>("");
  const [bulkGroupId, setBulkGroupId] = useState<string>("");
  
  // Bulk import states
  const [bulkImportFiles, setBulkImportFiles] = useState<File[]>([]);
  const [bulkImportSessionId, setBulkImportSessionId] = useState<string>('');
  const [showBulkImportLogs, setShowBulkImportLogs] = useState(false);
  
  // Bulk import from catalog states
  const [selectedCatalogProducts, setSelectedCatalogProducts] = useState<number[]>([]);
  const [catalogProductQuantities, setCatalogProductQuantities] = useState<Record<number, number>>({});
  const [catalogProductSearch, setCatalogProductSearch] = useState("");
  const [inventoryCountName, setInventoryCountName] = useState("");
  const [catalogGroupId, setCatalogGroupId] = useState<number | null>(null);
  
  // Auto-generate internal code from name
  useEffect(() => {
    if (!isInternalCodeManuallyEdited && newMaterialName.trim()) {
      const generatedCode = generateSlug(newMaterialName);
      setNewMaterialInternalCode(generatedCode);
    }
  }, [newMaterialName, isInternalCodeManuallyEdited]);

  // Save filters to localStorage
  useEffect(() => {
    const filters = {
      searchQuery,
      page,
      limit,
      sortBy,
      sortOrder,
      // Formatki specific filters
      formatkiFiltersExpanded,
      colorFilterStates,
      lengthFilterStates,
      widthFilterStates,
      stockFilter,
      // Extended formatki filters
      cz1FilterStates,
      cz2FilterStates,
      selectedFurnitureTypes,
      selectedSources,
      locationFilterStates,
      carrierFilterStates,
      filterGroupsVisible,
      // Column visibility
      columnVisibility,
      // Column widths
      columnWidths,
      // Column order
      columnOrder,
    };
    localStorage.setItem(STORAGE_KEY_PREFIX + category, JSON.stringify(filters));
  }, [searchQuery, page, limit, sortBy, sortOrder, category, formatkiFiltersExpanded, colorFilterStates, lengthFilterStates, widthFilterStates, stockFilter, cz1FilterStates, cz2FilterStates, selectedFurnitureTypes, selectedSources, locationFilterStates, carrierFilterStates, filterGroupsVisible, columnVisibility, columnWidths, columnOrder]);

  // Save filter settings to API (with debounce to prevent excessive requests)
  useEffect(() => {
    if (!userSettingsLoaded) return; // Wait for initial load before saving
    
    const timeoutId = setTimeout(() => {
      const filterSettings = {
        colorFilterStates,
        lengthFilterStates,
        widthFilterStates,
        stockFilter,
        cz1FilterStates,
        cz2FilterStates,
        selectedFurnitureTypes,
        selectedSources,
        locationFilterStates,
        carrierFilterStates,
        filterGroupsVisible,
        formatkiFiltersExpanded,
      };
      
      saveUserSettings.mutate({ 
        filters: filterSettings, 
        columnVisibility, 
        columnWidths,
        columnOrder,
      });
    }, 1000); // 1 second debounce
    
    return () => clearTimeout(timeoutId);
  }, [colorFilterStates, lengthFilterStates, widthFilterStates, stockFilter, cz1FilterStates, cz2FilterStates, selectedFurnitureTypes, selectedSources, locationFilterStates, carrierFilterStates, filterGroupsVisible, formatkiFiltersExpanded, userSettingsLoaded, columnVisibility, columnWidths, columnOrder]);


  // Reset page when search or groupCode changes
  useEffect(() => {
    setPage(1);
    setSelectedMaterials([]);
  }, [searchQuery, groupCode]);

  // Build query parameters
  const queryParams: Record<string, string> = {
    category,
    page: page.toString(),
    limit: limit.toString(),
    sortBy,
    sortOrder,
  };

  if (groupCode) {
    queryParams.groupCode = groupCode;
  }

  if (searchQuery) {
    queryParams.search = searchQuery;
  }
  
  // Add archived filter for produkty-spakowane
  if (category === 'produkty-spakowane') {
    queryParams.archived = showArchived.toString();
  }
  
  // Add formatki specific filters
  if (category === 'formatki') {
    const includeColors = Object.entries(colorFilterStates).filter(([_, state]) => state === 'include').map(([code]) => code);
    const excludeColors = Object.entries(colorFilterStates).filter(([_, state]) => state === 'exclude').map(([code]) => code);
    if (includeColors.length > 0) {
      queryParams.colors = includeColors.join(',');
    }
    if (excludeColors.length > 0) {
      queryParams.excludeColors = excludeColors.join(',');
    }
    const includeLengths = Object.entries(lengthFilterStates).filter(([_, state]) => state === 'include').map(([code]) => code);
    const excludeLengths = Object.entries(lengthFilterStates).filter(([_, state]) => state === 'exclude').map(([code]) => code);
    if (includeLengths.length > 0) {
      queryParams.lengths = includeLengths.join(',');
    }
    if (excludeLengths.length > 0) {
      queryParams.excludeLengths = excludeLengths.join(',');
    }
    const includeWidths = Object.entries(widthFilterStates).filter(([_, state]) => state === 'include').map(([code]) => code);
    const excludeWidths = Object.entries(widthFilterStates).filter(([_, state]) => state === 'exclude').map(([code]) => code);
    if (includeWidths.length > 0) {
      queryParams.widths = includeWidths.join(',');
    }
    if (excludeWidths.length > 0) {
      queryParams.excludeWidths = excludeWidths.join(',');
    }
    if (stockFilter !== 'all') {
      queryParams.stockFilter = stockFilter;
    }
    // Extended formatki filters
    const includeCz1 = Object.entries(cz1FilterStates).filter(([_, state]) => state === 'include').map(([code]) => code);
    const excludeCz1 = Object.entries(cz1FilterStates).filter(([_, state]) => state === 'exclude').map(([code]) => code);
    if (includeCz1.length > 0) {
      queryParams.cz1Codes = includeCz1.join(',');
    }
    if (excludeCz1.length > 0) {
      queryParams.excludeCz1Codes = excludeCz1.join(',');
    }
    const includeCz2 = Object.entries(cz2FilterStates).filter(([_, state]) => state === 'include').map(([code]) => code);
    const excludeCz2 = Object.entries(cz2FilterStates).filter(([_, state]) => state === 'exclude').map(([code]) => code);
    if (includeCz2.length > 0) {
      queryParams.cz2Codes = includeCz2.join(',');
    }
    if (excludeCz2.length > 0) {
      queryParams.excludeCz2Codes = excludeCz2.join(',');
    }
    if (selectedFurnitureTypes.length > 0) {
      queryParams.furnitureTypes = selectedFurnitureTypes.join(',');
    }
    if (selectedSources.length > 0) {
      queryParams.sources = selectedSources.join(',');
    }
    const includeLocations = Object.entries(locationFilterStates).filter(([_, state]) => state === 'include').map(([id]) => id);
    const excludeLocations = Object.entries(locationFilterStates).filter(([_, state]) => state === 'exclude').map(([id]) => id);
    if (includeLocations.length > 0) {
      queryParams.locationIds = includeLocations.join(',');
    }
    if (excludeLocations.length > 0) {
      queryParams.excludeLocationIds = excludeLocations.join(',');
    }
    const includeCarriers = Object.entries(carrierFilterStates).filter(([_, state]) => state === 'include').map(([id]) => id);
    const excludeCarriers = Object.entries(carrierFilterStates).filter(([_, state]) => state === 'exclude').map(([id]) => id);
    if (includeCarriers.length > 0) {
      queryParams.carrierIds = includeCarriers.join(',');
    }
    if (excludeCarriers.length > 0) {
      queryParams.excludeCarrierIds = excludeCarriers.join(',');
    }
  }

  const queryString = new URLSearchParams(queryParams).toString();

  const { data, isLoading } = useQuery<SearchResponse>({
    queryKey: [`/api/warehouse/materials/search?${queryString}`],
  });

  const materials = data?.materials || [];
  const pagination = data?.pagination;
  const filterCounts = data?.filterCounts;

  // Load material groups for the current category
  const { data: allGroups = [] } = useQuery<any[]>({
    queryKey: ["/api/warehouse/material-groups"],
  });

  const categoryGroups = allGroups
    .filter(g => g.category === category && g.isActive)
    .sort((a, b) => {
      if (a.displayOrder !== b.displayOrder) {
        return a.displayOrder - b.displayOrder;
      }
      return a.name.localeCompare(b.name);
    });

  // Set default group for formatki category (Płyta meblowa 18mm)
  useEffect(() => {
    if (category === 'formatki' && !newMaterialGroupId && categoryGroups.length > 0) {
      const defaultGroup = categoryGroups.find(g => g.code === 'plyta_meblowa_18_mm');
      if (defaultGroup) {
        setNewMaterialGroupId(defaultGroup.id);
      }
    }
  }, [category, categoryGroups, newMaterialGroupId]);

  // Load units of measure from dictionaries
  const { data: units = [] } = useQuery<any[]>({
    queryKey: ["/api/dictionaries?type=unit"],
  });

  // Load dictionaries for formatki category
  const { data: cz1Dictionaries = [] } = useQuery<any[]>({
    queryKey: ["/api/dictionaries?type=component_cz1"],
    enabled: category === 'formatki',
  });

  const { data: cz2Dictionaries = [] } = useQuery<any[]>({
    queryKey: ["/api/dictionaries?type=component_cz2"],
    enabled: category === 'formatki',
  });

  const { data: furnitureTypeDictionaries = [] } = useQuery<any[]>({
    queryKey: ["/api/dictionaries?type=furniture_type"],
    enabled: category === 'formatki',
  });

  const { data: lengthDictionaries = [] } = useQuery<any[]>({
    queryKey: ["/api/dictionaries?type=dimension_length"],
    enabled: category === 'formatki',
  });

  const { data: widthDictionaries = [] } = useQuery<any[]>({
    queryKey: ["/api/dictionaries?type=dimension_width"],
    enabled: category === 'formatki',
  });

  const { data: colorDictionaries = [] } = useQuery<any[]>({
    queryKey: ["/api/dictionaries?type=color"],
    enabled: category === 'formatki' || category === 'produkty-spakowane',
  });

  // Load dictionaries for produkty-spakowane category
  const { data: productTypeDictionaries = [] } = useQuery<DictionaryItem[]>({
    queryKey: ["/api/dictionaries?type=product_type"],
    enabled: category === 'produkty-spakowane',
  });

  const { data: doorDictionaries = [] } = useQuery<DictionaryItem[]>({
    queryKey: ["/api/dictionaries?type=door"],
    enabled: category === 'produkty-spakowane',
  });

  const { data: legDictionaries = [] } = useQuery<DictionaryItem[]>({
    queryKey: ["/api/dictionaries?type=leg"],
    enabled: category === 'produkty-spakowane',
  });

  // Load production carriers for bulk operations
  const { data: carriers = [] } = useQuery<any[]>({
    queryKey: ["/api/production/carriers"],
  });

  // Load production locations for bulk operations
  const { data: locations = [] } = useQuery<any[]>({
    queryKey: ["/api/production/locations"],
  });

  // Load plyty (boards) for formatka form
  const { data: plytyMaterials = [] } = useQuery<any[]>({
    queryKey: ["/api/warehouse/plyty"],
    enabled: category === 'formatki',
  });

  // Load obrzeza (edgings) for formatka form
  const { data: obrzezaMaterials = [] } = useQuery<any[]>({
    queryKey: ["/api/warehouse/obrzeza"],
    enabled: category === 'formatki',
  });

  // Parse search query into AND groups (separated by semicolons)
  // Each AND group may contain OR terms (separated by commas)
  const getSearchGroups = () => {
    if (!searchQuery) return [];
    const groups: string[] = [];
    searchQuery.split(';').forEach((group: string) => {
      const trimmed = group.trim();
      if (trimmed) groups.push(trimmed);
    });
    return groups;
  };

  const searchGroups = getSearchGroups();

  // Remove a specific AND group
  const removeSearchGroup = (groupToRemove: string) => {
    const updatedGroups = searchGroups.filter(g => g !== groupToRemove);
    setSearchQuery(updatedGroups.join('; '));
  };

  const clearAllFilters = () => {
    setSearchQuery("");
  };

  // Bulk delete mutation
  const bulkDeleteMutation = useMutation({
    mutationFn: async () => {
      if (category === 'formatki') {
        return apiRequest("POST", "/api/warehouse/stock-panels/bulk-delete", {
          panelIds: selectedMaterials,
        });
      } else if (category === 'produkty-spakowane') {
        return apiRequest("POST", "/api/warehouse/packed-products/bulk-delete", {
          ids: selectedMaterials,
        });
      } else {
        return apiRequest("POST", "/api/warehouse/materials/bulk-delete", {
          materialIds: selectedMaterials,
        });
      }
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      toast({
        title: "Materiały usunięte",
        description: `Pomyślnie usunięto ${selectedMaterials.length} materiałów.`,
      });
      setSelectedMaterials([]);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd usuwania",
        description: error.message || "Nie udało się usunąć materiałów.",
        variant: "destructive",
      });
    },
  });

  // Bulk archive mutation (for produkty-spakowane)
  const bulkArchiveMutation = useMutation({
    mutationFn: async () => {
      // Use different endpoints based on category
      if (category === 'produkty-spakowane') {
        return apiRequest("POST", "/api/warehouse/packed-products/bulk-archive", {
          ids: selectedMaterials,
        });
      } else if (category === 'formatki') {
        return apiRequest("POST", "/api/warehouse/stock-panels/bulk-archive", {
          ids: selectedMaterials,
        });
      } else {
        return apiRequest("POST", "/api/warehouse/materials/bulk-archive", {
          materialIds: selectedMaterials,
        });
      }
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      toast({
        title: "Produkty zarchiwizowane",
        description: `Pomyślnie zarchiwizowano ${selectedMaterials.length} produktów.`,
      });
      setSelectedMaterials([]);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd archiwizacji",
        description: error.message || "Nie udało się zarchiwizować produktów.",
        variant: "destructive",
      });
    },
  });

  // Bulk restore mutation (for all categories)
  const bulkRestoreMutation = useMutation({
    mutationFn: async () => {
      // Use different endpoints based on category
      if (category === 'produkty-spakowane') {
        return apiRequest("POST", "/api/warehouse/packed-products/bulk-restore", {
          ids: selectedMaterials,
        });
      } else if (category === 'formatki') {
        return apiRequest("POST", "/api/warehouse/stock-panels/bulk-restore", {
          ids: selectedMaterials,
        });
      } else {
        return apiRequest("POST", "/api/warehouse/materials/bulk-restore", {
          materialIds: selectedMaterials,
        });
      }
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      toast({
        title: "Produkty przywrócone",
        description: `Pomyślnie przywrócono ${selectedMaterials.length} produktów z archiwum.`,
      });
      setSelectedMaterials([]);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd przywracania",
        description: error.message || "Nie udało się przywrócić produktów.",
        variant: "destructive",
      });
    },
  });

  // Bulk update carrier mutation
  const bulkUpdateCarrierMutation = useMutation({
    mutationFn: async (carrierId: number | null) => {
      // Use different endpoint for formatki (stock_panels) vs other materials
      const endpoint = category === 'formatki' 
        ? "/api/warehouse/stock-panels/bulk-update"
        : "/api/warehouse/materials/bulk-update";
      const idField = category === 'formatki' ? 'panelIds' : 'materialIds';
      
      return apiRequest("POST", endpoint, {
        [idField]: selectedMaterials,
        updates: {
          carrier_id: carrierId,
        },
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      toast({
        title: "Nośnik zaktualizowany",
        description: `Pomyślnie zaktualizowano nośnik dla ${selectedMaterials.length} materiałów.`,
      });
      setSelectedMaterials([]);
      setShowBulkCarrierDialog(false);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd aktualizacji",
        description: error.message || "Nie udało się zaktualizować nośnika.",
        variant: "destructive",
      });
    },
  });

  // Bulk update location mutation
  const bulkUpdateLocationMutation = useMutation({
    mutationFn: async (locationId: number | null) => {
      // Use different endpoint for formatki (stock_panels) vs other materials
      const endpoint = category === 'formatki' 
        ? "/api/warehouse/stock-panels/bulk-update"
        : "/api/warehouse/materials/bulk-update";
      const idField = category === 'formatki' ? 'panelIds' : 'materialIds';
      
      return apiRequest("POST", endpoint, {
        [idField]: selectedMaterials,
        updates: {
          location_id: locationId,
        },
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      toast({
        title: "Lokalizacja zaktualizowana",
        description: `Pomyślnie zaktualizowano lokalizację dla ${selectedMaterials.length} materiałów.`,
      });
      setSelectedMaterials([]);
      setShowBulkLocationDialog(false);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd aktualizacji",
        description: error.message || "Nie udało się zaktualizować lokalizacji.",
        variant: "destructive",
      });
    },
  });

  // Bulk formatka edit mutation
  const bulkFormatkaEditMutation = useMutation({
    mutationFn: async (updates: Record<string, any>) => {
      return apiRequest("POST", "/api/warehouse/stock-panels/bulk-update", {
        panelIds: selectedMaterials,
        updates,
        skipLockedPanels: true,
      });
    },
    onSuccess: (data: any) => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && (
            key.includes('/api/warehouse/stock-panels') ||
            key.includes('/api/warehouse/materials/search')
          );
        }
      });
      const message = data.skippedCount > 0 
        ? `Zaktualizowano ${data.updatedCount} formatek. Pominięto ${data.skippedCount} (używane w BOM/produkcji).`
        : `Pomyślnie zaktualizowano ${data.updatedCount} formatek.`;
      toast({
        title: "Formatki zaktualizowane",
        description: message,
      });
      setSelectedMaterials([]);
      setShowBulkFormatkaEditDialog(false);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd aktualizacji",
        description: error.message || "Nie udało się zaktualizować formatek.",
        variant: "destructive",
      });
    },
  });

  // Bulk update group mutation
  const bulkUpdateGroupMutation = useMutation({
    mutationFn: async (groupId: number | null) => {
      return apiRequest("POST", "/api/warehouse/materials/bulk-update", {
        materialIds: selectedMaterials,
        updates: {
          group_id: groupId,
        },
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      toast({
        title: "Grupa zaktualizowana",
        description: `Pomyślnie zaktualizowano grupę dla ${selectedMaterials.length} materiałów.`,
      });
      setSelectedMaterials([]);
      setShowBulkGroupDialog(false);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd aktualizacji",
        description: error.message || "Nie udało się zaktualizować grupy.",
        variant: "destructive",
      });
    },
  });

  // Create inventory count mutation
  const createInventoryCountMutation = useMutation({
    mutationFn: async (data: { name: string; notes?: string }) => {
      // For formatki category, use panelIds instead of materialIds
      const payload: any = {
        name: data.name,
        notes: data.notes,
      };
      
      if (category === 'formatki') {
        payload.panelIds = selectedMaterials;
      } else if (category === 'opakowania') {
        payload.packagingMaterialIds = selectedMaterials;
      } else if (category === 'produkty-spakowane') {
        payload.packedProductIds = selectedMaterials;
      } else {
        payload.materialIds = selectedMaterials;
      }
      
      const res = await apiRequest("POST", "/api/warehouse/inventory-counts", payload);
      return await res.json();
    },
    onSuccess: (response) => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/inventory-counts"] });
      toast({
        title: "Spis utworzony",
        description: `Utworzono spis inwentaryzacyjny z ${selectedMaterials.length} materiałami.`,
      });
      setSelectedMaterials([]);
      setShowBulkInventoryDialog(false);
      // Navigate to the inventory count detail page
      if (response.inventoryCount?.id) {
        setLocation(`/warehouse/inventory-counts/${response.inventoryCount.id}`);
      }
    },
    onError: (error: any) => {
      toast({
        title: "Błąd tworzenia spisu",
        description: error.message || "Nie udało się utworzyć spisu inwentaryzacyjnego.",
        variant: "destructive",
      });
    },
  });

  // Add material group mutation
  const addGroupMutation = useMutation({
    mutationFn: async (data: { name: string; code: string; category: string; description?: string; color?: string }) => {
      const response = await apiRequest("POST", "/api/warehouse/material-groups", data);
      return await response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/material-groups"] });
      toast({
        title: "Grupa utworzona",
        description: "Grupa materiałów została pomyślnie utworzona.",
      });
      setIsAddGroupDialogOpen(false);
      setNewGroupName("");
      setNewGroupCode("");
      setNewGroupDescription("");
      setNewGroupColor("");
    },
    onError: (error: any) => {
      toast({
        title: "Błąd tworzenia grupy",
        description: error.message || "Nie udało się utworzyć grupy materiałów.",
        variant: "destructive",
      });
    },
  });

  // Edit material group mutation
  const editGroupMutation = useMutation({
    mutationFn: async (data: { id: number; name: string; code: string; description?: string; displayOrder: number; color?: string }) => {
      const response = await apiRequest("PATCH", `/api/warehouse/material-groups/${data.id}`, data);
      return await response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/material-groups"] });
      toast({
        title: "Grupa zaktualizowana",
        description: "Grupa materiałów została pomyślnie zaktualizowana.",
      });
      setIsEditGroupDialogOpen(false);
      setEditingGroup(null);
      setEditGroupName("");
      setEditGroupCode("");
      setEditGroupDescription("");
      setEditGroupDisplayOrder("0");
      setEditGroupColor("");
    },
    onError: (error: any) => {
      toast({
        title: "Błąd aktualizacji grupy",
        description: error.message || "Nie udało się zaktualizować grupy materiałów.",
        variant: "destructive",
      });
    },
  });

  // Query to fetch catalog products
  const { data: catalogProducts = [], isLoading: isCatalogProductsLoading } = useQuery({
    queryKey: ['/api/catalog/products', catalogProductSearch],
    queryFn: async () => {
      const params = new URLSearchParams();
      if (catalogProductSearch) {
        params.set('search', catalogProductSearch);
      }
      params.set('limit', '100');
      params.set('sortBy', 'title');
      params.set('sortOrder', 'ASC');
      
      const response = await fetch(`/api/catalog/products?${params.toString()}`);
      if (!response.ok) throw new Error('Failed to fetch catalog products');
      const data = await response.json();
      return data.products || [];
    },
    enabled: isBulkImportCatalogDialogOpen,
  });

  // Bulk import from catalog mutation
  const bulkImportFromCatalogMutation = useMutation({
    mutationFn: async (data: { 
      name: string;
      products: { catalogProductId: number; countedQuantity?: number }[];
      groupId?: number | null;
      notes?: string;
    }) => {
      const response = await apiRequest("POST", "/api/warehouse/bulk-import-from-catalog", data);
      return await response.json();
    },
    onSuccess: (response) => {
      queryClient.invalidateQueries({ queryKey: [`/api/warehouse/materials/search?${queryString}`] });
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/inventory-counts"] });
      toast({
        title: "Import zakończony",
        description: response.message || `Zaimportowano ${response.items?.length || 0} produktów.`,
      });
      setIsBulkImportCatalogDialogOpen(false);
      setSelectedCatalogProducts([]);
      setCatalogProductQuantities({});
      setInventoryCountName("");
      setCatalogGroupId(null);
      setCatalogProductSearch("");
      
      // Navigate to the inventory count detail page
      if (response.inventoryCount?.id) {
        setLocation(`/warehouse/inventory-counts/${response.inventoryCount.id}`);
      }
    },
    onError: (error: any) => {
      toast({
        title: "Błąd importu",
        description: error.message || "Nie udało się zaimportować produktów z katalogu.",
        variant: "destructive",
      });
    },
  });

  // Add material mutation
  const addMaterialMutation = useMutation({
    mutationFn: async (data: { 
      name: string; 
      internalCode: string;
      supplierCode?: string;
      description?: string;
      unitOfMeasure: string;
      price?: number;
      groupId?: number;
    }) => {
      const response = await apiRequest("POST", "/api/warehouse/materials", data);
      return await response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/warehouse/materials/search?${queryString}`] });
      toast({
        title: "Materiał utworzony",
        description: "Materiał został pomyślnie utworzony.",
      });
      setIsAddMaterialDialogOpen(false);
      setNewMaterialName("");
      setNewMaterialInternalCode("");
      setNewMaterialSupplierCode("");
      setNewMaterialDescription("");
      setNewMaterialUnitOfMeasure("szt");
      setNewMaterialPrice("");
      setNewMaterialGroupId(null);
      setIsInternalCodeManuallyEdited(false);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd tworzenia materiału",
        description: error.message || "Nie udało się utworzyć materiału.",
        variant: "destructive",
      });
    },
  });

  // Add stock panel (formatka) mutation
  const addStockPanelMutation = useMutation({
    mutationFn: async (data: { 
      generatedName: string;
      length: number;
      width: number;
      thickness: number;
      cz1?: string;
      cz2?: string;
      furnitureType?: string;
      boardCode?: string;
      edgingCode?: string;
      colorCode?: string;
      edge1?: boolean;
      edge2?: boolean;
      edge3?: boolean;
      edge4?: boolean;
      quantity?: number;
      groupId?: number;
    }) => {
      const response = await apiRequest("POST", "/api/warehouse/stock-panels", data);
      return await response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      toast({
        title: "Formatka utworzona",
        description: "Formatka została pomyślnie utworzona.",
      });
      setIsAddMaterialDialogOpen(false);
      // Clear formatka fields
      setNewFormatkaCz1("");
      setNewFormatkaCz2("");
      setNewFormatkaLength("");
      setNewFormatkaWidth("");
      setNewFormatkaThickness("18");
      setNewFormatkaBoardCode("");
      setNewFormatkaEdgingCode("");
      setNewFormatkaColorCode("");
      setNewFormatkaEdge1(false);
      setNewFormatkaEdge2(false);
      setNewFormatkaEdge3(false);
      setNewFormatkaEdge4(false);
      setNewMaterialGroupId(null);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd tworzenia formatki",
        description: error.message || "Nie udało się utworzyć formatki.",
        variant: "destructive",
      });
    },
  });

  // Edit material mutation
  const editMaterialMutation = useMutation({
    mutationFn: async (data: { 
      id: number;
      groupId?: number | null;
      name?: string; 
      internalCode?: string;
      supplierCode?: string;
      description?: string;
      unitOfMeasure?: string;
      price?: number | null;
      gallery?: string[];
    }) => {
      const response = await apiRequest("PATCH", `/api/warehouse/materials/${data.id}`, data);
      return await response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      toast({
        title: "Materiał zaktualizowany",
        description: "Materiał został pomyślnie zaktualizowany.",
      });
      setIsEditMaterialDialogOpen(false);
      setEditingMaterial(null);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd aktualizacji materiału",
        description: error.message || "Nie udało się zaktualizować materiału.",
        variant: "destructive",
      });
    },
  });

  // Duplicate material mutation
  const duplicateMaterialMutation = useMutation({
    mutationFn: async (data: { id: number; targetGroupId: number }) => {
      const response = await apiRequest("POST", `/api/warehouse/materials/${data.id}/duplicate`, {
        targetGroupId: data.targetGroupId,
      });
      return await response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      toast({
        title: "Materiał zduplikowany",
        description: "Materiał został pomyślnie zduplikowany.",
      });
      setShowDuplicateDialog(false);
      setDuplicateMaterial(null);
      setDuplicateTargetGroupId(null);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd duplikacji materiału",
        description: error.message || "Nie udało się zduplikować materiału.",
        variant: "destructive",
      });
    },
  });

  // Upload material image mutation
  const uploadMaterialImageMutation = useMutation({
    mutationFn: async ({ materialId, file }: { materialId: number; file: File }) => {
      const formData = new FormData();
      formData.append('image', file);
      
      const response = await fetch(`/api/warehouse/materials/${materialId}/images`, {
        method: 'POST',
        body: formData,
      });
      
      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || 'Upload failed');
      }
      
      return await response.json();
    },
    onMutate: ({ materialId }) => {
      setUploadingMaterials(prev => ({ ...prev, [materialId]: true }));
    },
    onSuccess: (data, { materialId }) => {
      if (editingMaterial && data.material) {
        setEditMaterialGallery(data.material.gallery || []);
      }
      // Invalidate cache to refresh material list
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      toast({
        title: "Zdjęcie dodane",
        description: "Zdjęcie zostało pomyślnie dodane.",
      });
      setUploadingMaterials(prev => ({ ...prev, [materialId]: false }));
    },
    onError: (error: any, { materialId }) => {
      toast({
        title: "Błąd uploadu",
        description: error.message || "Nie udało się dodać zdjęcia.",
        variant: "destructive",
      });
      setUploadingMaterials(prev => ({ ...prev, [materialId]: false }));
    },
  });

  // Delete material image mutation
  const deleteMaterialImageMutation = useMutation({
    mutationFn: async ({ materialId, imageUrl }: { materialId: number; imageUrl: string }) => {
      const response = await apiRequest("DELETE", `/api/warehouse/materials/${materialId}/images`, { imageUrl });
      return await response.json();
    },
    onSuccess: (data) => {
      if (editingMaterial && data.material) {
        setEditMaterialGallery(data.material.gallery || []);
      }
      // Invalidate cache to refresh material list
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      toast({
        title: "Zdjęcie usunięte",
        description: "Zdjęcie zostało pomyślnie usunięte.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd usuwania",
        description: error.message || "Nie udało się usunąć zdjęcia.",
        variant: "destructive",
      });
    },
  });

  // Set primary material image mutation
  const setPrimaryImageMutation = useMutation({
    mutationFn: async ({ materialId, imageUrl }: { materialId: number; imageUrl: string }) => {
      const response = await apiRequest("PATCH", `/api/warehouse/materials/${materialId}/primary-image`, { imageUrl });
      return await response.json();
    },
    onSuccess: (data) => {
      // Update local dialog state with new primary image and gallery
      if (editingMaterial && data.material) {
        setEditingMaterial({
          ...editingMaterial,
          primaryImageUrl: data.material.primaryImageUrl,
          gallery: data.material.gallery,
        });
        // Also update gallery state for consistency
        setEditMaterialGallery(data.material.gallery || []);
      }
      // Invalidate cache to refresh material list
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      toast({
        title: "Główne zdjęcie ustawione",
        description: "Główne zdjęcie zostało pomyślnie ustawione.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd ustawiania zdjęcia",
        description: error.message || "Nie udało się ustawić głównego zdjęcia.",
        variant: "destructive",
      });
    },
  });

  // Export mutation
  const exportMutation = useMutation({
    mutationFn: async () => {
      const response = await fetch("/api/warehouse/materials/export", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          materialIds: selectedMaterials,
          format: "csv",
        }),
      });

      if (!response.ok) {
        throw new Error("Export failed");
      }

      const blob = await response.blob();
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      a.download = `materialy-${category}-${new Date().toISOString().split('T')[0]}.csv`;
      document.body.appendChild(a);
      a.click();
      window.URL.revokeObjectURL(url);
      document.body.removeChild(a);
    },
    onSuccess: () => {
      toast({
        title: "Eksport zakończony",
        description: `Wyeksportowano ${selectedMaterials.length} materiałów.`,
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd eksportu",
        description: error.message || "Nie udało się wyeksportować materiałów.",
        variant: "destructive",
      });
    },
  });

  // Bulk update price mutation
  const bulkUpdatePriceMutation = useMutation({
    mutationFn: async (price: string) => {
      const response = await apiRequest("POST", "/api/warehouse/materials/bulk-update", {
        materialIds: selectedMaterials,
        updates: { price }
      });
      return await response.json();
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      toast({
        title: "Ceny zaktualizowane",
        description: `Zaktualizowano cenę dla ${data.updatedCount} materiałów.`,
      });
      setSelectedMaterials([]);
      setShowBulkEditPriceDialog(false);
      setBulkEditPrice("");
    },
    onError: (error: any) => {
      toast({
        title: "Błąd aktualizacji",
        description: error.message || "Nie udało się zaktualizować cen.",
        variant: "destructive",
      });
    },
  });

  // Bulk update unit mutation
  const bulkUpdateUnitMutation = useMutation({
    mutationFn: async (unit: string) => {
      const response = await apiRequest("POST", "/api/warehouse/materials/bulk-update", {
        materialIds: selectedMaterials,
        updates: { unit_of_measure: unit }
      });
      return await response.json();
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      toast({
        title: "Jednostki zaktualizowane",
        description: `Zaktualizowano jednostkę miary dla ${data.updatedCount} materiałów.`,
      });
      setSelectedMaterials([]);
      setShowBulkEditUnitDialog(false);
      setBulkEditUnit("szt");
    },
    onError: (error: any) => {
      toast({
        title: "Błąd aktualizacji",
        description: error.message || "Nie udało się zaktualizować jednostek.",
        variant: "destructive",
      });
    },
  });

  // Bulk import mutation
  const bulkImportMutation = useMutation({
    mutationFn: async (sessionId: string) => {
      const formData = new FormData();
      bulkImportFiles.forEach((file) => {
        formData.append('images', file);
      });
      
      // Use currently selected group from URL
      if (groupCode) {
        const selectedGroup = categoryGroups.find(g => g.code === groupCode);
        if (selectedGroup) {
          formData.append('groupId', selectedGroup.id.toString());
        }
      }
      
      formData.append('category', category);
      formData.append('sessionId', sessionId);

      const response = await fetch('/api/warehouse/materials/bulk-import', {
        method: 'POST',
        body: formData,
        credentials: 'include',
      });

      if (!response.ok) {
        const errorText = await response.text();
        throw new Error(errorText || response.statusText);
      }

      return await response.json();
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      setBulkImportFiles([]);
      toast({
        title: "Sukces",
        description: `Zaimportowano ${data.created} materiałów${data.errors > 0 ? `, ${data.errors} błędów` : ''}`,
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zaimportować materiałów",
        variant: "destructive",
      });
    },
  });

  // Drag & drop for bulk import
  const onDropBulk = useCallback((acceptedFiles: File[]) => {
    console.log('📦 Dropped files:', acceptedFiles.length, acceptedFiles.map(f => f.name));
    setBulkImportFiles(prev => [...prev, ...acceptedFiles]);
  }, []);

  const { getRootProps, getInputProps, isDragActive } = useDropzone({
    onDrop: onDropBulk,
    accept: {
      'image/*': ['.png', '.jpg', '.jpeg', '.gif', '.webp']
    },
    multiple: true
  });

  const handleBulkImport = () => {
    console.log('🚀 Bulk import clicked, files:', bulkImportFiles.length);
    console.log('📍 Current category:', category, 'groupCode:', groupCode);
    if (bulkImportFiles.length === 0) {
      toast({
        title: "Błąd",
        description: "Wybierz przynajmniej jedno zdjęcie",
        variant: "destructive",
      });
      return;
    }
    
    // Generate session ID and show progress dialog
    const sessionId = `bulk-import-${Date.now()}-${Math.random().toString(36).substring(7)}`;
    setBulkImportSessionId(sessionId);
    setShowBulkImportLogs(true);
    bulkImportMutation.mutate(sessionId);
  };

  const removeBulkFile = (index: number) => {
    setBulkImportFiles(prev => prev.filter((_, i) => i !== index));
  };

  const handleBulkDelete = () => {
    if (selectedMaterials.length === 0) return;
    if (confirm(`Czy na pewno chcesz usunąć ${selectedMaterials.length} materiałów?`)) {
      bulkDeleteMutation.mutate();
    }
  };

  const handleExport = () => {
    if (selectedMaterials.length === 0) {
      toast({
        title: "Brak zaznaczonych materiałów",
        description: "Zaznacz materiały do eksportu.",
        variant: "destructive",
      });
      return;
    }
    exportMutation.mutate();
  };

  // Select all on current page
  const selectAllOnPage = () => {
    setSelectedMaterials(materials.map(m => m.id));
  };

  // Clear all selections
  const clearSelection = () => {
    setSelectedMaterials([]);
    setIsSelectingAllPages(false);
  };

  // Toggle select all on current page
  const toggleSelectAll = () => {
    if (selectedMaterials.length > 0) {
      clearSelection();
    } else {
      selectAllOnPage();
    }
  };

  const toggleSelectMaterial = (id: number) => {
    setSelectedMaterials(prev =>
      prev.includes(id) ? prev.filter(i => i !== id) : [...prev, id]
    );
  };

  // Function to select all materials from ALL pages
  const selectAllFromAllPages = async () => {
    if (isSelectingAllPages) return;
    
    setIsSelectingAllPages(true);
    try {
      // Build query string with same filters but without pagination limit
      const params = new URLSearchParams();
      params.set('category', category);
      params.set('limit', '10000'); // Get all records
      params.set('page', '1');
      if (searchQuery) params.set('search', searchQuery);
      if (sortBy) params.set('sortBy', sortBy);
      if (sortOrder) params.set('sortOrder', sortOrder);
      if (stockFilter) params.set('stockFilter', stockFilter);
      const includeColorsForSelectAll = Object.entries(colorFilterStates).filter(([_, state]) => state === 'include').map(([code]) => code);
      if (includeColorsForSelectAll.length > 0) params.set('colors', includeColorsForSelectAll.join(','));
      if (Object.keys(lengthFilterStates).length > 0) params.set('lengths', Object.keys(lengthFilterStates).join(','));
      if (Object.keys(widthFilterStates).length > 0) params.set('widths', Object.keys(widthFilterStates).join(','));
      if (Object.keys(cz1FilterStates).length > 0) params.set('cz1', Object.keys(cz1FilterStates).join(','));
      if (Object.keys(cz2FilterStates).length > 0) params.set('cz2', Object.keys(cz2FilterStates).join(','));
      if (selectedFurnitureTypes.length > 0) params.set('furnitureTypes', selectedFurnitureTypes.join(','));
      if (selectedSources.length > 0) params.set('sources', selectedSources.join(','));
      if (Object.keys(locationFilterStates).length > 0) params.set('locations', Object.keys(locationFilterStates).join(','));
      const includeCarriersForSelectAll = Object.entries(carrierFilterStates).filter(([_, state]) => state === 'include').map(([id]) => id);
      if (includeCarriersForSelectAll.length > 0) params.set('carriers', includeCarriersForSelectAll.join(','));
      
      const response = await fetch(`/api/warehouse/materials/search?${params.toString()}`);
      const data = await response.json();
      
      if (data.materials && Array.isArray(data.materials)) {
        setSelectedMaterials(data.materials.map((m: any) => m.id));
        toast({
          title: "Zaznaczono wszystkie",
          description: `Zaznaczono ${data.materials.length} rekordów ze wszystkich stron`,
        });
      }
    } catch (error) {
      console.error('Error selecting all from all pages:', error);
      toast({
        title: "Błąd",
        description: "Nie udało się pobrać wszystkich rekordów",
        variant: "destructive",
      });
    } finally {
      setIsSelectingAllPages(false);
    }
  };

  const handleAddGroup = () => {
    if (!newGroupName.trim() || !newGroupCode.trim()) {
      toast({
        title: "Błąd walidacji",
        description: "Nazwa i kod grupy są wymagane.",
        variant: "destructive",
      });
      return;
    }
    
    addGroupMutation.mutate({
      name: newGroupName,
      code: newGroupCode,
      category: category,
      description: newGroupDescription || undefined,
      color: newGroupColor || undefined,
    });
  };

  const handleEditGroup = (group: any) => {
    setEditingGroup(group);
    setEditGroupName(group.name);
    setEditGroupCode(group.code);
    setEditGroupDescription(group.description || "");
    setEditGroupDisplayOrder(group.displayOrder?.toString() || "0");
    setEditGroupColor(group.color || "");
    setIsEditGroupDialogOpen(true);
  };

  const handleUpdateGroup = () => {
    if (!editGroupName.trim() || !editGroupCode.trim()) {
      toast({
        title: "Błąd walidacji",
        description: "Nazwa i kod grupy są wymagane.",
        variant: "destructive",
      });
      return;
    }
    
    if (!editingGroup) return;
    
    editGroupMutation.mutate({
      id: editingGroup.id,
      name: editGroupName,
      code: editGroupCode,
      description: editGroupDescription || undefined,
      displayOrder: parseInt(editGroupDisplayOrder) || 0,
      color: editGroupColor || undefined,
    });
  };

  const handleAddMaterial = () => {
    // Special handling for formatki category
    if (category === 'formatki') {
      const length = parseFloat(newFormatkaLength);
      const width = parseFloat(newFormatkaWidth);
      const thickness = parseFloat(newFormatkaThickness);
      
      if (isNaN(length) || length <= 0) {
        toast({
          title: "Błąd walidacji",
          description: "Długość musi być liczbą większą od zera.",
          variant: "destructive",
        });
        return;
      }
      
      if (isNaN(width) || width <= 0) {
        toast({
          title: "Błąd walidacji",
          description: "Szerokość musi być liczbą większą od zera.",
          variant: "destructive",
        });
        return;
      }
      
      if (isNaN(thickness) || thickness <= 0) {
        toast({
          title: "Błąd walidacji",
          description: "Grubość musi być liczbą większą od zera.",
          variant: "destructive",
        });
        return;
      }
      
      // Build generated name: CZ1-CZ2-długość×szerokość-KOLOR
      let generatedName = '';
      if (newFormatkaCz1) {
        generatedName += newFormatkaCz1;
      }
      if (newFormatkaCz2) {
        generatedName += (generatedName ? '-' : '') + newFormatkaCz2;
      }
      generatedName += (generatedName ? '-' : '') + `${length}×${width}`;
      if (newFormatkaColorCode) {
        generatedName += '-' + newFormatkaColorCode;
      }
      
      addStockPanelMutation.mutate({
        generatedName,
        length,
        width,
        thickness,
        cz1: newFormatkaCz1 || undefined,
        cz2: newFormatkaCz2 || undefined,
        boardCode: newFormatkaBoardCode || undefined,
        edgingCode: newFormatkaEdgingCode || undefined,
        colorCode: newFormatkaColorCode || undefined,
        edge1: newFormatkaEdge1,
        edge2: newFormatkaEdge2,
        edge3: newFormatkaEdge3,
        edge4: newFormatkaEdge4,
        quantity: 1,
        groupId: newMaterialGroupId || undefined,
      });
    } else {
      // Standard material creation
      if (!newMaterialName.trim()) {
        toast({
          title: "Błąd walidacji",
          description: "Nazwa jest wymagana.",
          variant: "destructive",
        });
        return;
      }
      
      if (!newMaterialInternalCode.trim()) {
        toast({
          title: "Błąd walidacji",
          description: "Kod wewnętrzny jest wymagany.",
          variant: "destructive",
        });
        return;
      }
      
      addMaterialMutation.mutate({
        name: newMaterialName,
        internalCode: newMaterialInternalCode,
        supplierCode: newMaterialSupplierCode || undefined,
        description: newMaterialDescription || undefined,
        unitOfMeasure: newMaterialUnitOfMeasure,
        price: newMaterialPrice ? parseFloat(newMaterialPrice) : undefined,
        groupId: newMaterialGroupId || undefined,
      });
    }
  };

  const handleEditMaterial = (material: Material) => {
    setEditingMaterial(material);
    setEditMaterialName(material.name);
    setEditMaterialInternalCode(material.internalCode);
    setEditMaterialSupplierCode(material.supplierCode || "");
    setEditMaterialDescription(material.description || "");
    setEditMaterialUnitOfMeasure(material.unitOfMeasure);
    setEditMaterialPrice(material.price || "");
    setEditMaterialGroupId(material.groupId);
    setEditMaterialGallery(material.gallery || []);
    setIsEditInternalCodeManuallyEdited(true);
    setIsEditMaterialDialogOpen(true);
  };

  const handleUpdateMaterial = () => {
    if (!editMaterialName.trim() || !editMaterialInternalCode.trim()) {
      toast({
        title: "Błąd walidacji",
        description: "Nazwa i kod wewnętrzny są wymagane.",
        variant: "destructive",
      });
      return;
    }
    
    if (!editingMaterial) return;
    
    editMaterialMutation.mutate({
      id: editingMaterial.id,
      name: editMaterialName,
      internalCode: editMaterialInternalCode,
      supplierCode: editMaterialSupplierCode || undefined,
      description: editMaterialDescription || undefined,
      unitOfMeasure: editMaterialUnitOfMeasure,
      price: editMaterialPrice ? parseFloat(editMaterialPrice) : null,
      groupId: editMaterialGroupId || null,
      gallery: editMaterialGallery,
    });
  };

  const handleDuplicateSameGroup = (material: Material) => {
    if (!material.groupId) {
      toast({
        title: "Błąd duplikacji",
        description: "Materiał musi należeć do grupy.",
        variant: "destructive",
      });
      return;
    }
    
    duplicateMaterialMutation.mutate({
      id: material.id,
      targetGroupId: material.groupId,
    });
  };

  const handleDuplicateToGroup = (material: Material) => {
    setDuplicateMaterial(material);
    setDuplicateTargetGroupId(material.groupId);
    setShowDuplicateDialog(true);
  };

  const handleConfirmDuplicate = () => {
    if (!duplicateMaterial || !duplicateTargetGroupId) {
      toast({
        title: "Błąd walidacji",
        description: "Wybierz grupę docelową.",
        variant: "destructive",
      });
      return;
    }
    
    duplicateMaterialMutation.mutate({
      id: duplicateMaterial.id,
      targetGroupId: duplicateTargetGroupId,
    });
  };

  const handleImageUpload = (files: FileList | null, material?: Material) => {
    const targetMaterial = material || editingMaterial;
    if (!files || files.length === 0 || !targetMaterial) return;
    
    Array.from(files).forEach(file => {
      uploadMaterialImageMutation.mutate({
        materialId: targetMaterial.id,
        file,
      });
    });
  };

  const handleImageDelete = (imageUrl: string) => {
    if (!editingMaterial) return;
    
    deleteMaterialImageMutation.mutate({
      materialId: editingMaterial.id,
      imageUrl,
    });
  };

  const handleSetPrimaryImage = (imageUrl: string) => {
    if (!editingMaterial) return;
    
    setPrimaryImageMutation.mutate({
      materialId: editingMaterial.id,
      imageUrl,
    });
  };

  // Set default group when dialog opens and reset manual edit flag
  useEffect(() => {
    if (isAddMaterialDialogOpen) {
      setIsInternalCodeManuallyEdited(false);
      
      if (groupCode && categoryGroups.length > 0) {
        const currentGroup = categoryGroups.find(g => g.code === groupCode);
        if (currentGroup) {
          setNewMaterialGroupId(currentGroup.id);
        }
      }
    }
  }, [isAddMaterialDialogOpen, groupCode, categoryGroups]);

  return (
    <WarehouseLayout 
      category={category} 
      onEditGroup={handleEditGroup}
      onAddGroup={() => setIsAddGroupDialogOpen(true)}
    >
      <div className="p-6 space-y-6">
        {/* Header */}
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
          <h1 className="text-2xl sm:text-3xl font-bold">
            {CATEGORY_LABELS[category] || category}
            {groupCode && ` - ${materials[0]?.groupName || groupCode}`}
          </h1>
          <div className="flex flex-wrap gap-2">
            {/* Action buttons */}
            {!groupCode && (
              <Button
                variant="outline"
                size="sm"
                onClick={() => setIsAddGroupDialogOpen(true)}
                data-testid="button-add-group"
                className="border-blue-500 text-blue-600 hover:bg-blue-50 dark:border-blue-400 dark:text-blue-400 dark:hover:bg-blue-950"
              >
                <FolderPlus className="w-4 h-4 mr-2" />
                <span className="hidden sm:inline">Dodaj grupę</span>
                <span className="sm:hidden">Grupa</span>
              </Button>
            )}
            {category === 'produkty-spakowane' && !groupCode && (
              <Button
                variant="outline"
                size="sm"
                onClick={() => setIsBulkImportCatalogDialogOpen(true)}
                data-testid="button-import-from-catalog"
                className="border-purple-500 text-purple-600 hover:bg-purple-50 dark:border-purple-400 dark:text-purple-400 dark:hover:bg-purple-950"
              >
                <Package className="w-4 h-4 mr-2" />
                <span className="hidden sm:inline">Dodaj produkty z katalogu</span>
                <span className="sm:hidden">Z katalogu</span>
              </Button>
            )}
            <Button
              variant="default"
              size="sm"
              onClick={() => setIsAddMaterialDialogOpen(true)}
              data-testid="button-add-material"
              className="bg-green-600 hover:bg-green-700 dark:bg-green-700 dark:hover:bg-green-800"
            >
              <Plus className="w-4 h-4 mr-2" />
              <span className="hidden sm:inline">{groupCode ? `Dodaj materiał do grupy` : 'Dodaj materiał'}</span>
              <span className="sm:hidden">Materiał</span>
            </Button>
            <CSVImportWithMapping
              endpoint={
                category === 'formatki' 
                  ? "/api/warehouse/stock-panels/import-with-mapping"
                  : category === 'produkty-spakowane'
                  ? "/api/warehouse/packed-products/import-with-mapping"
                  : "/api/warehouse/materials/import-with-mapping"
              }
              fields={
                category === 'plyty' ? [
                  { key: 'name', label: 'Nazwa', description: 'Nazwa płyty (opcjonalna przy aktualizacji)' },
                  { key: 'internal_code', label: 'Kod wewnętrzny', description: 'Kod wewnętrzny (opcjonalny przy aktualizacji po Symbolu Nexo)' },
                  { key: 'thickness', label: 'Grubość', description: 'Grubość w mm' },
                  { key: 'color', label: 'Kolor', description: 'Kolor płyty' },
                  { key: 'material_type', label: 'Typ materiału', description: 'np. MDF, HDF' },
                  { key: 'supplier_code', label: 'Symbol (Nexo)', description: 'Symbol z programu Subiekt Nexo' },
                  { key: 'description', label: 'Opis', description: 'Opis materiału' },
                  { key: 'group_code', label: 'Kod grupy', description: 'Kod grupy materiałowej' },
                ] : category === 'obrzeza' ? [
                  { key: 'name', label: 'Nazwa', description: 'Nazwa obrzeża (opcjonalna przy aktualizacji)' },
                  { key: 'internal_code', label: 'Kod wewnętrzny', description: 'Kod wewnętrzny (opcjonalny przy aktualizacji po Symbolu Nexo)' },
                  { key: 'width', label: 'Szerokość', description: 'Szerokość w mm' },
                  { key: 'thickness', label: 'Grubość', description: 'Grubość w mm' },
                  { key: 'color', label: 'Kolor', description: 'Kolor obrzeża' },
                  { key: 'supplier_code', label: 'Symbol (Nexo)', description: 'Symbol z programu Subiekt Nexo' },
                  { key: 'price', label: 'Cena', description: 'Cena jednostkowa' },
                  { key: 'description', label: 'Opis', description: 'Opis materiału' },
                  { key: 'group_code', label: 'Kod grupy', description: 'Kod grupy materiałowej' },
                ] : category === 'formatki' ? [
                  { key: 'name', label: 'Nazwa', required: true, description: 'Nazwa formatki' },
                  { key: 'internal_code', label: 'Symbol (oryginalny kod z Nexo)', required: true, description: 'Unikalny symbol z Subiekt Nexo - po tym polu następuje matchowanie przy reimporcie' },
                  { key: 'length', label: 'Długość', description: 'Długość w mm' },
                  { key: 'width', label: 'Szerokość', description: 'Szerokość w mm' },
                  { key: 'thickness', label: 'Grubość', description: 'Grubość płyty (domyślnie 18)' },
                  { key: 'color', label: 'Kolor', description: 'Kolor formatki' },
                  { key: 'board_code', label: 'Kod płyty', description: 'Kod materiału płyty (np. 18_BIALY)' },
                  { key: 'edging_code', label: 'Kod obrzeża', description: 'Kod materiału obrzeża (np. 0.8_BIALY)' },
                  { key: 'quantity', label: 'Ilość', description: 'Stan magazynowy' },
                  { key: 'price', label: 'Cena', description: 'Cena jednostkowa' },
                  { key: 'group_code', label: 'Kod grupy', description: 'Kod grupy materiałowej' },
                ] : category === 'produkty-spakowane' ? [
                  { key: 'product_name', label: 'Nazwa produktu', required: true, description: 'Nazwa produktu' },
                  { key: 'product_sku', label: 'SKU', required: true, description: 'Unikalny SKU produktu' },
                  { key: 'product_type', label: 'Typ', description: 'Typ produktu (product/set)' },
                  { key: 'quantity', label: 'Ilość', description: 'Stan magazynowy' },
                  { key: 'notes', label: 'Notatki', description: 'Dodatkowe notatki' },
                  { key: 'group_code', label: 'Kod grupy', description: 'Kod grupy materiałowej' },
                ] : [
                  { key: 'name', label: 'Nazwa', description: 'Nazwa materiału (opcjonalna przy aktualizacji)' },
                  { key: 'internal_code', label: 'Kod wewnętrzny', description: 'Kod wewnętrzny (opcjonalny przy aktualizacji po Symbolu Nexo)' },
                  { key: 'supplier_code', label: 'Symbol (Nexo)', description: 'Symbol z programu Subiekt Nexo' },
                  { key: 'description', label: 'Opis', description: 'Opis materiału' },
                  { key: 'unit_of_measure', label: 'Jednostka', description: 'np. szt, m, kg' },
                  { key: 'price', label: 'Cena', description: 'Cena jednostkowa' },
                  { key: 'group_code', label: 'Kod grupy', description: 'Kod grupy materiałowej' },
                ]
              }
              templateCsv={
                category === 'plyty' 
                  ? "name,internal_code,thickness,color,material_type,supplier_code,group_code\n18_BIALY,PLY-18-B,18,Biały,MDF,,hdf_3mm"
                  : category === 'obrzeza'
                  ? "name,internal_code,width,thickness,color,supplier_code,group_code\nBIALY,0.8_BIALY,22,0.8,Biały,,08"
                  : category === 'formatki'
                  ? "name,internal_code,length,width,thickness,color,board_code,edging_code,quantity,price,group_code\n600x400-BIALY,NEXO-00012345,600,400,18,BIALY,18_BIALY,0.8_BIALY,10,25.50,NEXO"
                  : category === 'produkty-spakowane'
                  ? "product_name,product_sku,product_type,quantity,notes,group_code\nPufa VB-40-BIALY,VB-40-001,product,5,,gotowe"
                  : "name,internal_code,supplier_code,description,unit_of_measure,price,group_code\nZawias 35mm,ZAW-35-STD,SUP-123,Zawias standardowy 35mm,szt,5.99,zawiasy"
              }
              templateFilename={`szablon_${category}.csv`}
              invalidateQueryKey={['/api/warehouse/materials/search']}
              title={`Import ${category === 'plyty' ? 'płyt' : category === 'obrzeza' ? 'obrzeży' : category === 'formatki' ? 'formatek' : 'materiałów'} z CSV/Excel`}
              description="Wczytaj plik CSV lub Excel (XLSX) i zmapuj kolumny do pól w bazie danych"
              category={category}
            >
              <Button
                variant="outline"
                size="sm"
                data-testid="button-import-csv"
              >
                <Upload className="w-4 h-4 mr-2" />
                <span className="hidden sm:inline">Import CSV/Excel</span>
                <span className="sm:hidden">Import</span>
              </Button>
            </CSVImportWithMapping>
          </div>
        </div>

        {/* Filters and Bulk Import - side by side */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          {/* Filters */}
          <Card>
            <CardHeader className="py-2 px-4">
              <div className="flex items-center justify-between">
                <CardTitle className="text-sm">Filtry</CardTitle>
                {category === 'formatki' && (<div className="flex gap-1">
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => setFormatkiFiltersExpanded(!formatkiFiltersExpanded)}
                    className="h-6 w-6 p-0"
                    data-testid="button-toggle-filters"
                  >
                    {formatkiFiltersExpanded ? (
                      <ChevronUp className="h-4 w-4" />
                    ) : (
                      <ChevronDown className="h-4 w-4" />
                    )}
                  </Button>
                  <DropdownMenu open={filterSettingsOpen} onOpenChange={setFilterSettingsOpen}>
                    <DropdownMenuTrigger asChild>
                      <Button
                        variant="ghost"
                        size="sm"
                        className="h-6 w-6 p-0"
                        data-testid="button-filter-settings"
                      >
                        <Settings2 className="h-4 w-4" />
                      </Button>
                    </DropdownMenuTrigger>
                    <DropdownMenuContent align="end" className="w-48">
                      <div className="flex items-center justify-between px-2 py-1">
                        <DropdownMenuLabel className="text-xs p-0">Widoczne grupy filtrów</DropdownMenuLabel>
                        <Button
                          variant="ghost"
                          size="sm"
                          className="h-5 w-5 p-0"
                          onClick={() => setFilterSettingsOpen(false)}
                          data-testid="button-close-filter-settings"
                        >
                          <X className="h-3 w-3" />
                        </Button>
                      </div>
                      <div className="flex gap-1 px-2 py-1">
                        <Button
                          variant="ghost"
                          size="sm"
                          className="h-5 text-[10px] px-2 flex-1"
                          onClick={() => setFilterGroupsVisible({
                            stock: true, colors: true, lengths: true, widths: true,
                            cz1: true, cz2: true, furnitureTypes: true, sources: true,
                            locations: true, carriers: true
                          })}
                          data-testid="button-select-all-filters"
                        >
                          Wszystkie
                        </Button>
                        <Button
                          variant="ghost"
                          size="sm"
                          className="h-5 text-[10px] px-2 flex-1"
                          onClick={() => setFilterGroupsVisible({
                            stock: false, colors: false, lengths: false, widths: false,
                            cz1: false, cz2: false, furnitureTypes: false, sources: false,
                            locations: false, carriers: false
                          })}
                          data-testid="button-deselect-all-filters"
                        >
                          Żadne
                        </Button>
                      </div>
                      <DropdownMenuSeparator />
                      <DropdownMenuCheckboxItem
                        onSelect={(e) => e.preventDefault()}
                        checked={filterGroupsVisible.stock !== false}
                        onCheckedChange={(checked) => setFilterGroupsVisible({...filterGroupsVisible, stock: checked})}
                        data-testid="checkbox-filter-stock"
                      >
                        Stan magazynowy
                      </DropdownMenuCheckboxItem>
                      <DropdownMenuCheckboxItem
                        onSelect={(e) => e.preventDefault()}
                        checked={filterGroupsVisible.colors !== false}
                        onCheckedChange={(checked) => setFilterGroupsVisible({...filterGroupsVisible, colors: checked})}
                        data-testid="checkbox-filter-colors"
                      >
                        Kolory
                      </DropdownMenuCheckboxItem>
                      <DropdownMenuCheckboxItem
                        onSelect={(e) => e.preventDefault()}
                        checked={filterGroupsVisible.lengths !== false}
                        onCheckedChange={(checked) => setFilterGroupsVisible({...filterGroupsVisible, lengths: checked})}
                        data-testid="checkbox-filter-lengths"
                      >
                        Długości
                      </DropdownMenuCheckboxItem>
                      <DropdownMenuCheckboxItem
                        onSelect={(e) => e.preventDefault()}
                        checked={filterGroupsVisible.widths !== false}
                        onCheckedChange={(checked) => setFilterGroupsVisible({...filterGroupsVisible, widths: checked})}
                        data-testid="checkbox-filter-widths"
                      >
                        Szerokości
                      </DropdownMenuCheckboxItem>
                      <DropdownMenuCheckboxItem
                        onSelect={(e) => e.preventDefault()}
                        checked={filterGroupsVisible.cz1 === true}
                        onCheckedChange={(checked) => setFilterGroupsVisible({...filterGroupsVisible, cz1: checked})}
                        data-testid="checkbox-filter-cz1"
                      >
                        CZ1
                      </DropdownMenuCheckboxItem>
                      <DropdownMenuCheckboxItem
                        onSelect={(e) => e.preventDefault()}
                        checked={filterGroupsVisible.cz2 === true}
                        onCheckedChange={(checked) => setFilterGroupsVisible({...filterGroupsVisible, cz2: checked})}
                        data-testid="checkbox-filter-cz2"
                      >
                        CZ2
                      </DropdownMenuCheckboxItem>
                      <DropdownMenuCheckboxItem
                        onSelect={(e) => e.preventDefault()}
                        checked={filterGroupsVisible.furnitureTypes === true}
                        onCheckedChange={(checked) => setFilterGroupsVisible({...filterGroupsVisible, furnitureTypes: checked})}
                        data-testid="checkbox-filter-furniture"
                      >
                        Typ mebla
                      </DropdownMenuCheckboxItem>
                      <DropdownMenuCheckboxItem
                        onSelect={(e) => e.preventDefault()}
                        checked={filterGroupsVisible.sources === true}
                        onCheckedChange={(checked) => setFilterGroupsVisible({...filterGroupsVisible, sources: checked})}
                        data-testid="checkbox-filter-sources"
                      >
                        Źródło
                      </DropdownMenuCheckboxItem>
                      <DropdownMenuCheckboxItem
                        onSelect={(e) => e.preventDefault()}
                        checked={filterGroupsVisible.locations === true}
                        onCheckedChange={(checked) => setFilterGroupsVisible({...filterGroupsVisible, locations: checked})}
                        data-testid="checkbox-filter-locations"
                      >
                        Lokalizacja
                      </DropdownMenuCheckboxItem>
                      <DropdownMenuCheckboxItem
                        onSelect={(e) => e.preventDefault()}
                        checked={filterGroupsVisible.carriers === true}
                        onCheckedChange={(checked) => setFilterGroupsVisible({...filterGroupsVisible, carriers: checked})}
                        data-testid="checkbox-filter-carriers"
                      >
                        Nośnik
                      </DropdownMenuCheckboxItem>
                    </DropdownMenuContent>
                  </DropdownMenu>
                </div>)}
              </div>
            </CardHeader>
            <CardContent className="space-y-3 py-2 px-4">
              {/* Archive toggle for produkty-spakowane */}
              {category === 'produkty-spakowane' && (
                <div className="flex gap-2">
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => { setShowArchived(false); setSelectedMaterials([]); }}
                    className={`h-7 px-3 ${
                      !showArchived 
                        ? 'bg-zinc-400 text-zinc-800' 
                        : 'bg-zinc-700/50 text-zinc-400 hover:bg-zinc-600/50'
                    }`}
                    data-testid="button-show-active"
                  >
                    Aktywne
                  </Button>
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => { setShowArchived(true); setSelectedMaterials([]); }}
                    className={`h-7 px-3 ${
                      showArchived 
                        ? 'bg-zinc-400 text-zinc-800' 
                        : 'bg-zinc-700/50 text-zinc-400 hover:bg-zinc-600/50'
                    }`}
                    data-testid="button-show-archived"
                  >
                    <Archive className="w-4 h-4 mr-2" />
                    Archiwum
                  </Button>
                </div>
              )}
              
              {/* Search Input */}
              <div className="relative">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                <Input
                  placeholder="Szukaj... (użyj ; dla AND, , dla OR)"
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="pl-9 h-8 text-sm"
                  data-testid="input-search"
                />
              </div>

              {/* Formatki specific filters - collapsible */}
              {category === 'formatki' && formatkiFiltersExpanded && (
                <div className="space-y-2">
                  {/* Stock filter buttons */}
                  {filterGroupsVisible.stock !== false && (
                  <div className="flex items-center gap-1">
                    <span className="text-xs text-muted-foreground mr-1">Stan:</span>
                    <button
                      onClick={() => setStockFilter('all')}
                      className={`h-5 px-2 text-[10px] font-medium inline-flex items-center justify-center ${
                        stockFilter === 'all' 
                          ? 'bg-zinc-400 text-zinc-800' 
                          : 'bg-zinc-700/50 text-zinc-400 hover:bg-zinc-600/50'
                      }`}
                      data-testid="button-stock-all"
                    >
                      Wszystkie
                    </button>
                    <button
                      onClick={() => setStockFilter('inStock')}
                      className={`h-5 px-2 text-[10px] font-medium inline-flex items-center justify-center gap-1 ${
                        stockFilter === 'inStock' 
                          ? 'bg-zinc-400 text-zinc-800' 
                          : 'bg-zinc-700/50 text-zinc-400 hover:bg-zinc-600/50'
                      }`}
                      data-testid="button-stock-in"
                    >
                      <PackageCheck className="h-3 w-3" />
                      &gt;0
                      {filterCounts?.stock?.inStock !== undefined && (
                        <span className="text-[9px] opacity-70">({filterCounts.stock.inStock})</span>
                      )}
                    </button>
                    <button
                      onClick={() => setStockFilter('outOfStock')}
                      className={`h-5 px-2 text-[10px] font-medium inline-flex items-center justify-center gap-1 ${
                        stockFilter === 'outOfStock' 
                          ? 'bg-zinc-400 text-zinc-800' 
                          : 'bg-zinc-700/50 text-zinc-400 hover:bg-zinc-600/50'
                      }`}
                      data-testid="button-stock-out"
                    >
                      <PackageX className="h-3 w-3" />
                      =0
                      {filterCounts?.stock?.outOfStock !== undefined && (
                        <span className="text-[9px] opacity-70">({filterCounts.stock.outOfStock})</span>
                      )}
                    </button>
                  </div>
                  )}

                  {/* Colors row */}
                  {filterGroupsVisible.colors !== false && colorDictionaries.length > 0 && (
                    <div className="flex flex-wrap items-center gap-1">
                      <span className="text-xs text-muted-foreground mr-1">Kolor:</span>
                      <input
                        type="text"
                        placeholder="Szukaj..."
                        value={filterSearch.colors}
                        onChange={(e) => setFilterSearch({...filterSearch, colors: e.target.value})}
                        className="h-5 w-16 px-1 text-[10px] bg-zinc-800 border border-zinc-600 rounded text-zinc-300 placeholder:text-zinc-500"
                        data-testid="input-filter-search-colors"
                      />
                      {colorDictionaries
                        .filter((c: any) => c.isActive)
                        .filter((c: any) => !filterSearch.colors || c.code.toLowerCase().includes(filterSearch.colors.toLowerCase()) || (c.readableName && c.readableName.toLowerCase().includes(filterSearch.colors.toLowerCase())))
                        .map((color: any) => {
                        const filterState = colorFilterStates[color.code];
                        const isInclude = filterState === 'include';
                        const isExclude = filterState === 'exclude';
                        const bgColor = isInclude ? '#3b82f6' : isExclude ? '#ef4444' : (color.color || '#cccccc');
                        const textColor = (isInclude || isExclude) ? '#ffffff' : getTextColorForBackground(color.color || '#cccccc');
                        return (
                          <button
                            key={color.code}
                            onClick={() => {
                              setColorFilterStates(prev => {
                                const current = prev[color.code];
                                const newState = { ...prev };
                                if (!current) {
                                  newState[color.code] = 'include';
                                } else if (current === 'include') {
                                  newState[color.code] = 'exclude';
                                } else {
                                  delete newState[color.code];
                                }
                                return newState;
                              });
                            }}
                            className={`h-5 text-[10px] font-medium inline-flex items-center justify-center ${
                              filterCounts?.colors?.[color.code] === 0 ? 'opacity-30' : ''
                            } ${!filterState ? 'hover:opacity-80' : ''}`}
                            style={{ 
                              backgroundColor: bgColor, 
                              color: textColor,
                              width: '85px',
                              textDecoration: isExclude ? 'line-through' : 'none'
                            }}
                            title={`${color.readableName} (${filterCounts?.colors?.[color.code] || 0}) - Kliknij: zawiera → wyklucza → brak`}
                            data-testid={`button-color-${color.code}`}
                          >
                            {color.code.length > 6 ? color.code.slice(0, 6) : color.code}
                            {filterCounts?.colors?.[color.code] !== undefined && (
                              <span className="ml-0.5 text-[8px] opacity-80">({filterCounts.colors[color.code]})</span>
                            )}
                          </button>
                        );
                      })}
                    </div>
                  )}

                  {/* Lengths row */}
                  {filterGroupsVisible.lengths !== false && lengthDictionaries.length > 0 && (
                    <div className="flex flex-wrap items-center gap-1">
                      <span className="text-xs text-muted-foreground mr-1">Dł:</span>
                      {lengthDictionaries.filter((l: any) => l.isActive).sort((a: any, b: any) => {
                        const numA = parseInt(a.code) || 0;
                        const numB = parseInt(b.code) || 0;
                        return numA - numB;
                      }).map((length: any) => {
                        const filterState = lengthFilterStates[length.code];
                        const isInclude = filterState === 'include';
                        const isExclude = filterState === 'exclude';
                        const count = filterCounts?.lengths?.[length.code] || 0;
                        return (
                          <button
                            key={length.code}
                            onClick={() => {
                              setLengthFilterStates(prev => {
                                const current = prev[length.code];
                                const newState = { ...prev };
                                if (!current) {
                                  newState[length.code] = 'include';
                                } else if (current === 'include') {
                                  newState[length.code] = 'exclude';
                                } else {
                                  delete newState[length.code];
                                }
                                return newState;
                              });
                            }}
                            className={`h-5 text-[10px] font-medium inline-flex items-center justify-center ${
                              isInclude ? 'bg-blue-500 text-white' : isExclude ? 'bg-red-500 text-white line-through' : 'bg-zinc-700/50 text-zinc-400 hover:bg-zinc-600/50'
                            } ${count === 0 ? 'opacity-30' : ''}`}
                            style={{ width: '52px' }}
                            data-testid={`button-length-${length.code}`}
                            title={`${length.code} (${count}) - Kliknij: zawiera → wyklucza → brak`}
                          >
                            {length.code}
                            <span className="ml-0.5 text-[8px] opacity-70">({count})</span>
                          </button>
                        );
                      })}
                    </div>
                  )}

                  {/* Widths row */}
                  {filterGroupsVisible.widths !== false && widthDictionaries.length > 0 && (
                    <div className="flex flex-wrap items-center gap-1">
                      <span className="text-xs text-muted-foreground mr-1">Szer:</span>
                      {widthDictionaries.filter((w: any) => w.isActive).sort((a: any, b: any) => {
                        const numA = parseInt(a.code) || 0;
                        const numB = parseInt(b.code) || 0;
                        return numA - numB;
                      }).map((width: any) => {
                        const filterState = widthFilterStates[width.code];
                        const isInclude = filterState === 'include';
                        const isExclude = filterState === 'exclude';
                        const count = filterCounts?.widths?.[width.code] || 0;
                        return (
                          <button
                            key={width.code}
                            onClick={() => {
                              setWidthFilterStates(prev => {
                                const current = prev[width.code];
                                const newState = { ...prev };
                                if (!current) {
                                  newState[width.code] = 'include';
                                } else if (current === 'include') {
                                  newState[width.code] = 'exclude';
                                } else {
                                  delete newState[width.code];
                                }
                                return newState;
                              });
                            }}
                            className={`h-5 text-[10px] font-medium inline-flex items-center justify-center ${
                              isInclude ? 'bg-blue-500 text-white' : isExclude ? 'bg-red-500 text-white line-through' : 'bg-zinc-700/50 text-zinc-400 hover:bg-zinc-600/50'
                            } ${count === 0 ? 'opacity-30' : ''}`}
                            style={{ width: '46px' }}
                            data-testid={`button-width-${width.code}`}
                            title={`${width.code} (${count}) - Kliknij: zawiera → wyklucza → brak`}
                          >
                            {width.code}
                            <span className="ml-0.5 text-[8px] opacity-70">({count})</span>
                          </button>
                        );
                      })}
                    </div>
                  )}

                  {/* CZ1 row */}
                  {filterGroupsVisible.cz1 === true && cz1Dictionaries.length > 0 && (
                    <div className="flex flex-wrap items-center gap-1">
                      <span className="text-xs text-muted-foreground mr-1">CZ1:</span>
                      <input
                        type="text"
                        placeholder="Szukaj..."
                        value={filterSearch.cz1}
                        onChange={(e) => setFilterSearch({...filterSearch, cz1: e.target.value})}
                        className="h-5 w-16 px-1 text-[10px] bg-zinc-800 border border-zinc-600 rounded text-zinc-300 placeholder:text-zinc-500"
                        data-testid="input-filter-search-cz1"
                      />
                      {cz1Dictionaries
                        .filter((c: any) => c.isActive)
                        .filter((c: any) => !filterSearch.cz1 || c.code.toLowerCase().includes(filterSearch.cz1.toLowerCase()))
                        .map((cz1: any) => {
                        const filterState = cz1FilterStates[cz1.code];
                        const isInclude = filterState === 'include';
                        const isExclude = filterState === 'exclude';
                        return (
                          <button
                            key={cz1.code}
                            onClick={() => {
                              setCz1FilterStates(prev => {
                                const current = prev[cz1.code];
                                const newState = { ...prev };
                                if (!current) {
                                  newState[cz1.code] = 'include';
                                } else if (current === 'include') {
                                  newState[cz1.code] = 'exclude';
                                } else {
                                  delete newState[cz1.code];
                                }
                                return newState;
                              });
                            }}
                            className={`h-5 px-2 text-[10px] font-medium inline-flex items-center justify-center ${
                              isInclude ? 'bg-blue-500 text-white' : isExclude ? 'bg-red-500 text-white line-through' : 'bg-zinc-700/50 text-zinc-400 hover:bg-zinc-600/50'
                            }`}
                            data-testid={`button-cz1-${cz1.code}`}
                            title={`${cz1.code} - Kliknij: zawiera → wyklucza → brak`}
                          >
                            {cz1.code}
                          </button>
                        );
                      })}
                    </div>
                  )}

                  {/* CZ2 row */}
                  {filterGroupsVisible.cz2 === true && cz2Dictionaries.length > 0 && (
                    <div className="flex flex-wrap items-center gap-1">
                      <span className="text-xs text-muted-foreground mr-1">CZ2:</span>
                      <input
                        type="text"
                        placeholder="Szukaj..."
                        value={filterSearch.cz2}
                        onChange={(e) => setFilterSearch({...filterSearch, cz2: e.target.value})}
                        className="h-5 w-16 px-1 text-[10px] bg-zinc-800 border border-zinc-600 rounded text-zinc-300 placeholder:text-zinc-500"
                        data-testid="input-filter-search-cz2"
                      />
                      {cz2Dictionaries
                        .filter((c: any) => c.isActive)
                        .filter((c: any) => !filterSearch.cz2 || c.code.toLowerCase().includes(filterSearch.cz2.toLowerCase()))
                        .map((cz2: any) => {
                        const filterState = cz2FilterStates[cz2.code];
                        const isInclude = filterState === 'include';
                        const isExclude = filterState === 'exclude';
                        return (
                          <button
                            key={cz2.code}
                            onClick={() => {
                              setCz2FilterStates(prev => {
                                const current = prev[cz2.code];
                                const newState = { ...prev };
                                if (!current) {
                                  newState[cz2.code] = 'include';
                                } else if (current === 'include') {
                                  newState[cz2.code] = 'exclude';
                                } else {
                                  delete newState[cz2.code];
                                }
                                return newState;
                              });
                            }}
                            className={`h-5 px-2 text-[10px] font-medium inline-flex items-center justify-center ${
                              isInclude ? 'bg-blue-500 text-white' : isExclude ? 'bg-red-500 text-white line-through' : 'bg-zinc-700/50 text-zinc-400 hover:bg-zinc-600/50'
                            }`}
                            data-testid={`button-cz2-${cz2.code}`}
                            title={`${cz2.code} - Kliknij: zawiera → wyklucza → brak`}
                          >
                            {cz2.code}
                          </button>
                        );
                      })}
                    </div>
                  )}

                  {/* Furniture Type row */}
                  {filterGroupsVisible.furnitureTypes === true && furnitureTypeDictionaries.length > 0 && (
                    <div className="flex flex-wrap items-center gap-1">
                      <span className="text-xs text-muted-foreground mr-1">Typ:</span>
                      {furnitureTypeDictionaries.filter((f: any) => f.isActive).map((ft: any) => {
                        const isSelected = selectedFurnitureTypes.includes(ft.code);
                        return (
                          <button
                            key={ft.code}
                            onClick={() => {
                              if (isSelected) {
                                setSelectedFurnitureTypes(selectedFurnitureTypes.filter(f => f !== ft.code));
                              } else {
                                setSelectedFurnitureTypes([...selectedFurnitureTypes, ft.code]);
                              }
                            }}
                            className={`h-5 px-2 text-[10px] font-medium inline-flex items-center justify-center ${
                              isSelected 
                                ? 'bg-zinc-400 text-zinc-800' 
                                : 'bg-zinc-700/50 text-zinc-400 hover:bg-zinc-600/50'
                            }`}
                            data-testid={`button-furnitureType-${ft.code}`}
                          >
                            {ft.readableName || ft.code}
                          </button>
                        );
                      })}
                    </div>
                  )}

                  {/* Locations row */}
                  {filterGroupsVisible.locations === true && locations.length > 0 && (
                    <div className="flex flex-wrap items-center gap-1">
                      <span className="text-xs text-muted-foreground mr-1">Lok:</span>
                      <input
                        type="text"
                        placeholder="Szukaj..."
                        value={filterSearch.locations}
                        onChange={(e) => setFilterSearch({...filterSearch, locations: e.target.value})}
                        className="h-5 w-16 px-1 text-[10px] bg-zinc-800 border border-zinc-600 rounded text-zinc-300 placeholder:text-zinc-500"
                        data-testid="input-filter-search-locations"
                      />
                      {locations
                        .filter((loc: any) => !filterSearch.locations || (loc.name || loc.code || '').toLowerCase().includes(filterSearch.locations.toLowerCase()))
                        .map((loc: any) => {
                        const locId = loc.id.toString();
                        const filterState = locationFilterStates[locId];
                        const isInclude = filterState === 'include';
                        const isExclude = filterState === 'exclude';
                        return (
                          <button
                            key={loc.id}
                            onClick={() => {
                              setLocationFilterStates(prev => {
                                const current = prev[locId];
                                const newState = { ...prev };
                                if (!current) {
                                  newState[locId] = 'include';
                                } else if (current === 'include') {
                                  newState[locId] = 'exclude';
                                } else {
                                  delete newState[locId];
                                }
                                return newState;
                              });
                            }}
                            className={`h-5 px-2 text-[10px] font-medium inline-flex items-center justify-center ${
                              isInclude ? 'bg-blue-500 text-white' : isExclude ? 'bg-red-500 text-white line-through' : 'bg-zinc-700/50 text-zinc-400 hover:bg-zinc-600/50'
                            }`}
                            data-testid={`button-location-${loc.id}`}
                            title={`${loc.name || loc.code} - Kliknij: zawiera → wyklucza → brak`}
                          >
                            {loc.name || loc.code}
                          </button>
                        );
                      })}
                    </div>
                  )}

                  {/* Carriers row */}
                  {filterGroupsVisible.carriers === true && carriers.length > 0 && (
                    <div className="flex flex-wrap items-center gap-1">
                      <span className="text-xs text-muted-foreground mr-1">Nośnik:</span>
                      <input
                        type="text"
                        placeholder="Szukaj..."
                        value={filterSearch.carriers}
                        onChange={(e) => setFilterSearch({...filterSearch, carriers: e.target.value})}
                        className="h-5 w-16 px-1 text-[10px] bg-zinc-800 border border-zinc-600 rounded text-zinc-300 placeholder:text-zinc-500"
                        data-testid="input-filter-search-carriers"
                      />
                      {carriers
                        .filter((carrier: any) => !filterSearch.carriers || (carrier.name || carrier.code || '').toLowerCase().includes(filterSearch.carriers.toLowerCase()))
                        .map((carrier: any) => {
                        const carrierId = carrier.id.toString();
                        const filterState = carrierFilterStates[carrierId];
                        const isInclude = filterState === 'include';
                        const isExclude = filterState === 'exclude';
                        return (
                          <button
                            key={carrier.id}
                            onClick={() => {
                              setCarrierFilterStates(prev => {
                                const current = prev[carrierId];
                                const newState = { ...prev };
                                if (!current) {
                                  newState[carrierId] = 'include';
                                } else if (current === 'include') {
                                  newState[carrierId] = 'exclude';
                                } else {
                                  delete newState[carrierId];
                                }
                                return newState;
                              });
                            }}
                            className={`h-5 px-2 text-[10px] font-medium inline-flex items-center justify-center ${
                              isInclude 
                                ? 'bg-blue-500 text-white' 
                                : isExclude
                                ? 'bg-red-500 text-white line-through'
                                : 'bg-zinc-700/50 text-zinc-400 hover:bg-zinc-600/50'
                            }`}
                            title="Kliknij: zawiera → wyklucza → brak"
                            data-testid={`button-carrier-${carrier.id}`}
                          >
                            {carrier.name || carrier.code}
                          </button>
                        );
                      })}
                    </div>
                  )}
                </div>
              )}

              {/* Active Filters Summary */}
              {(searchGroups.length > 0 || Object.keys(colorFilterStates).length > 0 || Object.keys(lengthFilterStates).length > 0 || Object.keys(widthFilterStates).length > 0 || stockFilter !== 'all' || Object.keys(cz1FilterStates).length > 0 || Object.keys(cz2FilterStates).length > 0 || selectedFurnitureTypes.length > 0 || selectedSources.length > 0 || Object.keys(locationFilterStates).length > 0 || Object.keys(carrierFilterStates).length > 0) && (
                <div className="flex flex-wrap items-center gap-1 pt-1 border-t">
                  {searchGroups.map((group, index) => (
                    <Badge
                      key={`search-${index}`}
                      variant="secondary"
                      className="gap-1 h-5 text-[10px] px-1.5"
                      data-testid={`badge-search-${index}`}
                    >
                      {group}
                      <X
                        className="h-2.5 w-2.5 cursor-pointer hover:text-destructive"
                        onClick={() => removeSearchGroup(group)}
                      />
                    </Badge>
                  ))}
                  {stockFilter !== 'all' && (
                    <Badge
                      variant="secondary"
                      className="gap-1 h-5 text-[10px] px-1.5"
                      data-testid="badge-stock-filter"
                    >
                      {stockFilter === 'inStock' ? '>0' : '=0'}
                      <X
                        className="h-2.5 w-2.5 cursor-pointer hover:text-destructive"
                        onClick={() => setStockFilter('all')}
                      />
                    </Badge>
                  )}
                  {Object.entries(colorFilterStates).map(([colorCode, state]) => {
                    const colorDict = colorDictionaries.find((c: any) => c.code === colorCode);
                    const isExclude = state === 'exclude';
                    return (
                      <Badge
                        key={`color-${colorCode}`}
                        variant="secondary"
                        className={`gap-1 h-5 text-[10px] px-1.5 ${isExclude ? 'line-through' : ''}`}
                        style={{ 
                          backgroundColor: isExclude ? '#ef4444' : (state === 'include' ? '#3b82f6' : (colorDict?.color || undefined)),
                          color: '#ffffff'
                        }}
                        data-testid={`badge-color-${colorCode}`}
                      >
                        {isExclude ? '- ' : '+ '}{colorCode}
                        <X
                          className="h-2.5 w-2.5 cursor-pointer hover:opacity-70"
                          onClick={() => setColorFilterStates(prev => {
                            const newState = { ...prev };
                            delete newState[colorCode];
                            return newState;
                          })}
                        />
                      </Badge>
                    );
                  })}
                  {Object.keys(lengthFilterStates).map((lengthCode) => (
                    <Badge
                      key={`length-${lengthCode}`}
                      variant="secondary"
                      className="gap-1 h-5 text-[10px] px-1.5"
                      data-testid={`badge-length-${lengthCode}`}
                    >
                      Dł: {lengthCode}
                      <X
                        className="h-2.5 w-2.5 cursor-pointer hover:text-destructive"
                        onClick={() => setLengthFilterStates(prev => { const next = {...prev}; delete next[lengthCode]; return next; })}
                      />
                    </Badge>
                  ))}
                  {Object.keys(widthFilterStates).map((widthCode) => (
                    <Badge
                      key={`width-${widthCode}`}
                      variant="secondary"
                      className="gap-1 h-5 text-[10px] px-1.5"
                      data-testid={`badge-width-${widthCode}`}
                    >
                      Szer: {widthCode}
                      <X
                        className="h-2.5 w-2.5 cursor-pointer hover:text-destructive"
                        onClick={() => setWidthFilterStates(prev => { const next = {...prev}; delete next[widthCode]; return next; })}
                      />
                    </Badge>
                  ))}
                  {Object.keys(cz1FilterStates).map((cz1Code) => (
                    <Badge
                      key={`cz1-${cz1Code}`}
                      variant="secondary"
                      className="gap-1 h-5 text-[10px] px-1.5"
                      data-testid={`badge-cz1-${cz1Code}`}
                    >
                      CZ1: {cz1Code}
                      <X
                        className="h-2.5 w-2.5 cursor-pointer hover:text-destructive"
                        onClick={() => setCz1FilterStates(prev => { const next = {...prev}; delete next[cz1Code]; return next; })}
                      />
                    </Badge>
                  ))}
                  {Object.keys(cz2FilterStates).map((cz2Code) => (
                    <Badge
                      key={`cz2-${cz2Code}`}
                      variant="secondary"
                      className="gap-1 h-5 text-[10px] px-1.5"
                      data-testid={`badge-cz2-${cz2Code}`}
                    >
                      CZ2: {cz2Code}
                      <X
                        className="h-2.5 w-2.5 cursor-pointer hover:text-destructive"
                        onClick={() => setCz2FilterStates(prev => { const next = {...prev}; delete next[cz2Code]; return next; })}
                      />
                    </Badge>
                  ))}
                  {selectedFurnitureTypes.map((ftCode) => (
                    <Badge
                      key={`ft-${ftCode}`}
                      variant="secondary"
                      className="gap-1 h-5 text-[10px] px-1.5"
                      data-testid={`badge-furnitureType-${ftCode}`}
                    >
                      Typ: {ftCode}
                      <X
                        className="h-2.5 w-2.5 cursor-pointer hover:text-destructive"
                        onClick={() => setSelectedFurnitureTypes(selectedFurnitureTypes.filter(f => f !== ftCode))}
                      />
                    </Badge>
                  ))}
                  {selectedSources.map((sourceCode) => (
                    <Badge
                      key={`source-${sourceCode}`}
                      variant="secondary"
                      className="gap-1 h-5 text-[10px] px-1.5"
                      data-testid={`badge-source-${sourceCode}`}
                    >
                      Źr: {sourceCode}
                      <X
                        className="h-2.5 w-2.5 cursor-pointer hover:text-destructive"
                        onClick={() => setSelectedSources(selectedSources.filter(s => s !== sourceCode))}
                      />
                    </Badge>
                  ))}
                  {Object.keys(locationFilterStates).map((locId) => {
                    const loc = locations.find((l: any) => l.id.toString() === locId);
                    return (
                      <Badge
                        key={`loc-${locId}`}
                        variant="secondary"
                        className="gap-1 h-5 text-[10px] px-1.5"
                        data-testid={`badge-location-${locId}`}
                      >
                        Lok: {loc?.name || locId}
                        <X
                          className="h-2.5 w-2.5 cursor-pointer hover:text-destructive"
                          onClick={() => setLocationFilterStates(prev => { const next = {...prev}; delete next[locId]; return next; })}
                        />
                      </Badge>
                    );
                  })}
                  {Object.entries(carrierFilterStates).map(([carrierId, state]) => {
                    const carrier = carriers.find((c: any) => c.id.toString() === carrierId);
                    const isExclude = state === 'exclude';
                    return (
                      <Badge
                        key={`carrier-${carrierId}`}
                        variant="secondary"
                        className={`gap-1 h-5 text-[10px] px-1.5 ${isExclude ? 'line-through' : ''}`}
                        style={{ 
                          backgroundColor: isExclude ? '#ef4444' : '#3b82f6',
                          color: '#ffffff'
                        }}
                        data-testid={`badge-carrier-${carrierId}`}
                      >
                        {isExclude ? '- ' : '+ '}Nośnik: {carrier?.name || carrierId}
                        <X
                          className="h-2.5 w-2.5 cursor-pointer hover:opacity-70"
                          onClick={() => setCarrierFilterStates(prev => {
                            const newState = { ...prev };
                            delete newState[carrierId];
                            return newState;
                          })}
                        />
                      </Badge>
                    );
                  })}
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => {
                      setSearchQuery("");
                      setColorFilterStates({});
                      setLengthFilterStates({});
                      setWidthFilterStates({});
                      setStockFilter('all');
                      setCz1FilterStates({});
                      setCz2FilterStates({});
                      setSelectedFurnitureTypes([]);
                      setSelectedSources([]);
                      setLocationFilterStates({});
                      setCarrierFilterStates({});
                    }}
                    className="h-5 px-1.5 text-[10px] bg-purple-600 hover:bg-purple-700 text-white"
                    data-testid="button-clear-all-filters"
                  >
                    <X className="h-3 w-3 mr-0.5" />
                    Wyczyść
                  </Button>
                </div>
              )}
            </CardContent>
          </Card>

          {/* Bulk Import */}
          <Card>
            <CardHeader>
              <CardTitle>Importuj zdjęcia</CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              <div
                {...getRootProps()}
                className={`border-2 border-dashed p-3 text-center cursor-pointer hover-elevate rounded-md ${
                  isDragActive ? 'border-primary bg-primary/10' : 'border-border'
                }`}
                data-testid="dropzone-bulk-import"
              >
                <input {...getInputProps()} />
                <Upload className="h-6 w-6 mx-auto mb-1 text-muted-foreground" />
                {isDragActive ? (
                  <p className="text-xs">Upuść zdjęcia tutaj...</p>
                ) : (
                  <p className="text-xs text-muted-foreground">
                    Przeciągnij zdjęcia lub kliknij
                  </p>
                )}
              </div>

              {bulkImportFiles.length > 0 && (
                <div className="space-y-2">
                  <div className="flex items-center justify-between">
                    <span className="text-xs font-medium">Plików: {bulkImportFiles.length}</span>
                    <Button
                      size="sm"
                      variant="ghost"
                      onClick={() => setBulkImportFiles([])}
                      data-testid="button-clear-bulk-files"
                      className="h-6 text-xs"
                    >
                      <X className="h-3 w-3 mr-1" />
                      Wyczyść
                    </Button>
                  </div>
                  <div className="flex gap-2 flex-wrap">
                    <Button
                      size="sm"
                      onClick={handleBulkImport}
                      disabled={bulkImportMutation.isPending}
                      data-testid="button-bulk-import"
                    >
                      {bulkImportMutation.isPending ? 'Importowanie...' : `Importuj ${bulkImportFiles.length}`}
                    </Button>
                    <Button
                      size="sm"
                      variant="outline"
                      onClick={() => document.getElementById('bulk-file-input')?.click()}
                      disabled={bulkImportMutation.isPending}
                      data-testid="button-add-more-files"
                    >
                      <Plus className="h-3 w-3 mr-1" />
                      Dodaj więcej
                    </Button>
                    <input
                      id="bulk-file-input"
                      type="file"
                      multiple
                      accept="image/*"
                      className="hidden"
                      onChange={(e) => {
                        const files = Array.from(e.target.files || []);
                        setBulkImportFiles(prev => [...prev, ...files]);
                        e.target.value = '';
                      }}
                    />
                  </div>
                </div>
              )}
            </CardContent>
          </Card>
        </div>

        {/* Pagination Top */}
        {pagination && (
          <div className="flex items-center justify-between">
            <div className="text-sm text-muted-foreground">
              Pokazano {((page - 1) * limit) + 1} - {Math.min(page * limit, pagination.total)} z {pagination.total}
            </div>
            <div className="flex items-center gap-2">
              <Button
                variant="outline"
                size="sm"
                onClick={() => setPage(1)}
                disabled={page === 1}
                data-testid="button-page-first"
              >
                <ChevronsLeft className="w-4 h-4" />
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => setPage((p: number) => Math.max(1, p - 1))}
                disabled={page === 1}
                data-testid="button-page-prev"
              >
                <ChevronLeft className="w-4 h-4" />
              </Button>
              <span className="text-sm">
                Strona {page} z {pagination.totalPages}
              </span>
              <Button
                variant="outline"
                size="sm"
                onClick={() => setPage((p: number) => Math.min(pagination.totalPages, p + 1))}
                disabled={page === pagination.totalPages}
                data-testid="button-page-next"
              >
                <ChevronRight className="w-4 h-4" />
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => setPage(pagination.totalPages)}
                disabled={page === pagination.totalPages}
                data-testid="button-page-last"
              >
                <ChevronsRight className="w-4 h-4" />
              </Button>
              <Select value={limit.toString()} onValueChange={(v) => { setLimit(parseInt(v)); setPage(1); }}>
                <SelectTrigger className="w-24" data-testid="select-page-size">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="10">10</SelectItem>
                  <SelectItem value="25">25</SelectItem>
                  <SelectItem value="50">50</SelectItem>
                  <SelectItem value="100">100</SelectItem>
                  <SelectItem value="250">250</SelectItem>
                  <SelectItem value="500">500</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
        )}

        {/* Materials List */}
        <Card>
          <CardContent className="p-4">
            {isLoading ? (
              <div className="flex items-center justify-center py-8">
                <Loader2 className="h-8 w-8 animate-spin text-muted-foreground" />
              </div>
            ) : materials.length === 0 ? (
              <div className="text-center text-muted-foreground py-8">
                Brak materiałów
              </div>
            ) : (
              <div className="space-y-2">
                {/* Compact Selection Panel - vertical 60px */}
                <div className="flex items-center justify-between gap-2 pb-2 border-b">
                  <div className="flex flex-col border rounded bg-muted/30 w-[200px] h-[60px] overflow-hidden text-[11px]">
                    {/* Row 1: Selected count */}
                    <div className="h-[20px] flex items-center justify-center border-b bg-primary/10 text-foreground font-medium shrink-0 whitespace-nowrap">
                      Zaznaczono: {selectedMaterials.length}
                    </div>
                    {/* Row 2: Page / All buttons side by side */}
                    <div className="h-[20px] flex shrink-0">
                      <button
                        onClick={selectAllOnPage}
                        className="flex-1 flex items-center justify-center text-[10px] border-r hover:bg-muted/50 transition-colors whitespace-nowrap"
                        data-testid="badge-select-page"
                      >
                        Strona ({materials.length})
                      </button>
                      {pagination && pagination.totalPages > 1 ? (
                        <button
                          onClick={!isSelectingAllPages ? selectAllFromAllPages : undefined}
                          className={`flex-1 flex items-center justify-center text-[10px] hover:bg-muted/50 transition-colors whitespace-nowrap ${isSelectingAllPages ? 'opacity-50' : ''}`}
                          data-testid="badge-select-all-pages"
                        >
                          {isSelectingAllPages ? (
                            <Loader2 className="h-2.5 w-2.5 animate-spin" />
                          ) : (
                            <>Wszystkie ({pagination.total})</>
                          )}
                        </button>
                      ) : (
                        <div className="flex-1 flex items-center justify-center text-[10px] text-muted-foreground whitespace-nowrap">
                          —
                        </div>
                      )}
                    </div>
                    {/* Row 3: Clear selection - full width */}
                    <button
                      onClick={selectedMaterials.length > 0 ? clearSelection : undefined}
                      className={`h-[20px] flex items-center justify-center gap-1 text-[10px] border-t transition-colors shrink-0 whitespace-nowrap ${selectedMaterials.length > 0 ? 'text-destructive hover:bg-destructive/10 cursor-pointer' : 'text-muted-foreground/40 cursor-default'}`}
                      data-testid="badge-clear-selection"
                      disabled={selectedMaterials.length === 0}
                    >
                      <X className="h-2.5 w-2.5" />
                      Odznacz
                    </button>
                  </div>
                  {selectedMaterials.length > 0 && (
                    <div className="flex gap-2 flex-wrap">
                      <Button
                        onClick={() => setShowBulkInventoryDialog(true)}
                        data-testid="button-create-inventory-count"
                        size="sm"
                      >
                        <FileCheck className="h-4 w-4 mr-2" />
                        Utwórz spis ({selectedMaterials.length})
                      </Button>
                      <Button
                        onClick={() => setShowBulkCarrierDialog(true)}
                        data-testid="button-change-carrier"
                        size="sm"
                        variant="outline"
                      >
                        <Package className="h-4 w-4 mr-2" />
                        Zmień nośnik
                      </Button>
                      <Button
                        onClick={() => setShowBulkLocationDialog(true)}
                        data-testid="button-change-location"
                        size="sm"
                        variant="outline"
                      >
                        <MapPin className="h-4 w-4 mr-2" />
                        Zmień lokalizację
                      </Button>
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => setShowBulkEditPriceDialog(true)}
                        data-testid="button-bulk-edit-price"
                      >
                        <Edit className="w-4 h-4 mr-2" />
                        Zmień cenę
                      </Button>
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => setShowBulkEditUnitDialog(true)}
                        data-testid="button-bulk-edit-unit"
                      >
                        <Edit className="w-4 h-4 mr-2" />
                        Zmień jednostkę
                      </Button>
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => setShowBulkGroupDialog(true)}
                        data-testid="button-bulk-change-group"
                      >
                        <FolderPlus className="w-4 h-4 mr-2" />
                        Zmień grupę
                      </Button>
                      {category === 'formatki' && (
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => setShowBulkFormatkaEditDialog(true)}
                          data-testid="button-bulk-edit-formatki"
                        >
                          <Pencil className="w-4 h-4 mr-2" />
                          Edytuj pola
                        </Button>
                      )}
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={handleExport}
                        disabled={exportMutation.isPending}
                        data-testid="button-export-selected"
                      >
                        {exportMutation.isPending ? (
                          <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                        ) : (
                          <Download className="w-4 h-4 mr-2" />
                        )}
                        Eksportuj
                      </Button>
                      {/* Archive button for all material categories - only when not showing archived */}
                      {!showArchived && (
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => bulkArchiveMutation.mutate()}
                          disabled={bulkArchiveMutation.isPending}
                          data-testid="button-archive-selected"
                        >
                          {bulkArchiveMutation.isPending ? (
                            <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                          ) : (
                            <Archive className="w-4 h-4 mr-2" />
                          )}
                          Archiwizuj
                        </Button>
                      )}
                      {/* Restore button - only when showing archived */}
                      {showArchived && (
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => bulkRestoreMutation.mutate()}
                          disabled={bulkRestoreMutation.isPending}
                          data-testid="button-restore-selected"
                        >
                          {bulkRestoreMutation.isPending ? (
                            <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                          ) : (
                            <ArchiveRestore className="w-4 h-4 mr-2" />
                          )}
                          Przywróć
                        </Button>
                      )}
                      <Button
                        variant="destructive"
                        size="sm"
                        onClick={handleBulkDelete}
                        disabled={bulkDeleteMutation.isPending}
                        data-testid="button-delete-selected"
                      >
                        {bulkDeleteMutation.isPending ? (
                          <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                        ) : (
                          <Trash2 className="w-4 h-4 mr-2" />
                        )}
                        Usuń
                      </Button>
                    </div>
                  )}
                  {/* Column visibility button for formatki */}
                  {category === 'formatki' && (
                    <DropdownMenu open={columnSettingsOpen} onOpenChange={setColumnSettingsOpen}>
                      <DropdownMenuTrigger asChild>
                        <Button
                          variant="outline"
                          size="sm"
                          className="ml-auto"
                          data-testid="button-column-settings"
                        >
                          <Settings2 className="h-4 w-4 mr-2" />
                          Kolumny
                        </Button>
                      </DropdownMenuTrigger>
                      <DropdownMenuContent align="end" className="w-56 max-h-[400px] overflow-y-auto">
                        <div className="flex items-center justify-between px-2 py-1">
                          <DropdownMenuLabel className="text-xs p-0">Widoczne kolumny</DropdownMenuLabel>
                          <Button
                            variant="ghost"
                            size="sm"
                            className="h-5 w-5 p-0"
                            onClick={() => setColumnSettingsOpen(false)}
                            data-testid="button-close-column-settings"
                          >
                            <X className="h-3 w-3" />
                          </Button>
                        </div>
                        <div className="flex gap-1 px-2 py-1">
                          <Button
                            variant="ghost"
                            size="sm"
                            className="h-5 text-[10px] px-2 flex-1"
                            onClick={() => setColumnVisibility(formatkiColumns.reduce((acc, col) => {
                              acc[col.key] = true;
                              return acc;
                            }, {} as Record<string, boolean>))}
                            data-testid="button-select-all-columns"
                          >
                            Wszystkie
                          </Button>
                          <Button
                            variant="ghost"
                            size="sm"
                            className="h-5 text-[10px] px-2 flex-1"
                            onClick={() => {
                              setColumnVisibility(defaultColumnVisibility);
                              setColumnWidths(defaultColumnWidths);
                            }}
                            data-testid="button-reset-columns"
                          >
                            Domyślne
                          </Button>
                        </div>
                        <DropdownMenuSeparator />
                        <DndContext
                          sensors={sensors}
                          collisionDetection={closestCenter}
                          onDragEnd={handleColumnDragEnd}
                        >
                          <SortableContext items={columnOrder} strategy={verticalListSortingStrategy}>
                            {getOrderedColumns().map((col) => (
                              <SortableColumnItem 
                                key={col.key}
                                col={col}
                                columnVisibility={columnVisibility}
                                setColumnVisibility={setColumnVisibility}
                                getColumnWidth={getColumnWidth}
                                setColumnWidths={setColumnWidths}
                                columnWidths={columnWidths}
                                defaultColumnWidths={defaultColumnWidths}
                              />
                            ))}
                          </SortableContext>
                        </DndContext>
                      </DropdownMenuContent>
                    </DropdownMenu>
                  )}
                </div>

                {/* Table Container with horizontal scroll */}
                <div className="overflow-x-auto">
                  {/* Table Header */}
                  <div className="flex items-center gap-2 h-[36px] px-3 bg-muted/50 border-b sticky top-0 z-10 text-xs font-medium">
                    <div className="w-[40px] shrink-0 flex items-center justify-center">
                      <Checkbox
                        checked={selectedMaterials.length === materials.length && materials.length > 0}
                        onCheckedChange={(checked) => {
                          if (checked) {
                            setSelectedMaterials(materials.map(m => m.id));
                          } else {
                            setSelectedMaterials([]);
                          }
                        }}
                        data-testid="checkbox-select-all"
                      />
                    </div>
                    {/* Formatki: dynamic column headers based on columnOrder */}
                    {category === 'formatki' && getOrderedColumns().map(col => renderFormatkiHeader(col.key))}
                    
                    {/* Non-formatki categories: static headers */}
                    {category !== 'formatki' && (
                      <>
                        {/* Image column for non-formatki */}
                        <div className="shrink-0" style={{width: 40}}></div>
                        
                        {/* Produkty-spakowane: Stan mag. PRZED parametrami */}
                        {category === 'produkty-spakowane' && (
                          <div className="w-[140px] shrink-0">Stan mag.</div>
                        )}
                        
                        {/* Non produkty-spakowane: Nazwa przed stanem mag. */}
                        {category !== 'produkty-spakowane' && (
                          <div className="shrink-0" style={{width: 200}}>Nazwa</div>
                        )}
                        
                        {/* Non produkty-spakowane i non-formatki: Stan mag. PO nazwie */}
                        {category !== 'produkty-spakowane' && (
                          <div className="w-[140px] shrink-0">Stan mag.</div>
                        )}
                      </>
                    )}
                    
                    {/* Produkty-spakowane specific columns: sortable headers */}
                    {category === 'produkty-spakowane' && (
                      <>
                        <div 
                          className="w-[70px] shrink-0 cursor-pointer hover:text-foreground flex items-center gap-1"
                          onClick={() => {
                            if (sortBy === 'catalogProductType') {
                              setSortOrder(sortOrder === 'ASC' ? 'DESC' : 'ASC');
                            } else {
                              setSortBy('catalogProductType');
                              setSortOrder('ASC');
                            }
                          }}
                        >
                          Typ
                          {sortBy === 'catalogProductType' && (
                            <span className="text-primary">{sortOrder === 'ASC' ? '↑' : '↓'}</span>
                          )}
                        </div>
                        <div 
                          className="w-[50px] shrink-0 cursor-pointer hover:text-foreground flex items-center gap-1"
                          onClick={() => {
                            if (sortBy === 'catalogLength') {
                              setSortOrder(sortOrder === 'ASC' ? 'DESC' : 'ASC');
                            } else {
                              setSortBy('catalogLength');
                              setSortOrder('ASC');
                            }
                          }}
                        >
                          Dł.
                          {sortBy === 'catalogLength' && (
                            <span className="text-primary">{sortOrder === 'ASC' ? '↑' : '↓'}</span>
                          )}
                        </div>
                        <div 
                          className="w-[50px] shrink-0 cursor-pointer hover:text-foreground flex items-center gap-1"
                          onClick={() => {
                            if (sortBy === 'catalogWidth') {
                              setSortOrder(sortOrder === 'ASC' ? 'DESC' : 'ASC');
                            } else {
                              setSortBy('catalogWidth');
                              setSortOrder('ASC');
                            }
                          }}
                        >
                          Szer.
                          {sortBy === 'catalogWidth' && (
                            <span className="text-primary">{sortOrder === 'ASC' ? '↑' : '↓'}</span>
                          )}
                        </div>
                        <div 
                          className="w-[80px] shrink-0 cursor-pointer hover:text-foreground flex items-center gap-1"
                          onClick={() => {
                            if (sortBy === 'catalogColor') {
                              setSortOrder(sortOrder === 'ASC' ? 'DESC' : 'ASC');
                            } else {
                              setSortBy('catalogColor');
                              setSortOrder('ASC');
                            }
                          }}
                        >
                          Kolor
                          {sortBy === 'catalogColor' && (
                            <span className="text-primary">{sortOrder === 'ASC' ? '↑' : '↓'}</span>
                          )}
                        </div>
                        <div 
                          className="w-[50px] shrink-0 cursor-pointer hover:text-foreground flex items-center gap-1"
                          onClick={() => {
                            if (sortBy === 'catalogLegs') {
                              setSortOrder(sortOrder === 'ASC' ? 'DESC' : 'ASC');
                            } else {
                              setSortBy('catalogLegs');
                              setSortOrder('ASC');
                            }
                          }}
                        >
                          Nogi
                          {sortBy === 'catalogLegs' && (
                            <span className="text-primary">{sortOrder === 'ASC' ? '↑' : '↓'}</span>
                          )}
                        </div>
                        <div 
                          className="w-[50px] shrink-0 cursor-pointer hover:text-foreground flex items-center gap-1"
                          onClick={() => {
                            if (sortBy === 'catalogDoors') {
                              setSortOrder(sortOrder === 'ASC' ? 'DESC' : 'ASC');
                            } else {
                              setSortBy('catalogDoors');
                              setSortOrder('ASC');
                            }
                          }}
                        >
                          Drzwi
                          {sortBy === 'catalogDoors' && (
                            <span className="text-primary">{sortOrder === 'ASC' ? '↑' : '↓'}</span>
                          )}
                        </div>
                        <div className="w-[120px] shrink-0">SKU</div>
                        <div className="w-[200px] shrink-0">Nazwa</div>
                      </>
                    )}
                    
                    {/* Formatki columns are rendered dynamically above */}
                    
                    {/* Default columns for non-formatki and non-produkty-spakowane categories */}
                    {category !== 'formatki' && category !== 'produkty-spakowane' && (
                      <>
                        <div className="w-[100px] shrink-0">Kod wew.</div>
                        <div className="w-[100px] shrink-0">Kod dost.</div>
                        <div className="w-[120px] shrink-0">Grupa</div>
                        <div className="w-[100px] shrink-0">J.m.</div>
                        <div className="w-[80px] shrink-0">Cena</div>
                      </>
                    )}
                    {/* Location and carrier for non-formatki categories */}
                    {category !== 'formatki' && (
                      <>
                        <div className="shrink-0" style={{width: 120}}>Lokalizacja</div>
                        <div className="shrink-0" style={{width: 120}}>Nośnik</div>
                      </>
                    )}
                    <div className="w-[50px] shrink-0"></div>
                  </div>
                
                  {/* Table Rows */}
                  <div className="space-y-1">
                    {materials.map((material, rowIndex) => {
                      const imageUrl = material.primaryImageUrl || (material.gallery && material.gallery.length > 0 ? material.gallery[0] : null);
                      const isZeroQuantity = (material.quantity || 0) === 0;
                      
                      return (
                        <div
                          key={material.id}
                          className={`flex items-center gap-2 h-[40px] px-3 border ${
                            isZeroQuantity 
                              ? 'bg-gray-100 dark:bg-gray-800/50 text-gray-400 dark:text-gray-500 opacity-60' 
                              : 'bg-muted/20'
                          }`}
                          data-testid={`material-${material.id}`}
                        >
                          {/* Checkbox */}
                          <div className="w-[40px] shrink-0 flex items-center justify-center" onClick={(e) => e.stopPropagation()}>
                            <Checkbox
                              checked={selectedMaterials.includes(material.id)}
                              onCheckedChange={() => toggleSelectMaterial(material.id)}
                              data-testid={`checkbox-material-${material.id}`}
                            />
                          </div>

                          {/* Clickable Row Content */}
                          <Link
                            href={`/warehouse/${category}/${material.id}`}
                            className="flex items-center gap-2 flex-1 min-w-0"
                            data-testid={`link-material-detail-${material.id}`}
                          >
                          {/* Formatki: dynamic column cells based on columnOrder */}
                          {category === 'formatki' && getOrderedColumns().map(col => renderFormatkiCell(col.key, material, rowIndex, isZeroQuantity))}
                          
                          {/* Non-formatki: Image thumbnail - no rounded corners for produkty-spakowane */}
                          {category !== 'formatki' && (
                          <div className="shrink-0 flex items-center justify-center" style={{width: 40}}>
                            {imageUrl ? (
                              <img 
                                src={imageUrl} 
                                alt={material.name}
                                className={`w-8 h-8 object-cover border ${category === 'produkty-spakowane' ? '' : 'rounded'}`}
                              />
                            ) : (
                              <div className={`w-8 h-8 bg-muted border flex items-center justify-center ${category === 'produkty-spakowane' ? '' : 'rounded'}`}>
                                <Upload className="w-4 h-4 text-muted-foreground" />
                              </div>
                            )}
                          </div>
                          )}

                          {/* Produkty-spakowane: Quantity Badge BEFORE parameters */}
                          {category === 'produkty-spakowane' && (
                            <div className="w-[140px] shrink-0 flex items-center gap-1">
                              {material.quantityReserved && material.quantityReserved > 0 ? (
                                <>
                                  <Badge variant="destructive" className="text-xs font-medium min-w-[45px] justify-center" title="Zarezerwowane">
                                    {material.quantityReserved}
                                  </Badge>
                                  <Badge variant="default" className="text-xs font-medium min-w-[45px] justify-center bg-green-600 hover:bg-green-700" title="Dostępne">
                                    {(material.quantity || 0) - material.quantityReserved}
                                  </Badge>
                                  <Badge variant="outline" className="text-xs font-medium min-w-[45px] justify-center" title="Całkowite">
                                    {material.quantity || 0}
                                  </Badge>
                                </>
                              ) : (
                                <Badge variant="default" className="font-semibold text-sm min-w-[60px] justify-center">
                                  {material.quantity || 0}
                                </Badge>
                              )}
                            </div>
                          )}

                          {/* Name - for non produkty-spakowane and non-formatki */}
                          {category !== 'produkty-spakowane' && category !== 'formatki' && (
                            <div className="shrink-0 flex items-center gap-2 min-w-0" style={{width: 200}}>
                              <span className="font-medium text-sm truncate">{material.name}</span>
                            </div>
                          )}
                          
                          {/* Non produkty-spakowane i non-formatki: Quantity Badge AFTER name */}
                          {category !== 'produkty-spakowane' && category !== 'formatki' && (
                            <div className="w-[140px] shrink-0 flex items-center gap-1">
                              {material.quantityReserved && material.quantityReserved > 0 ? (
                                <>
                                  <Badge variant="destructive" className="text-xs font-medium min-w-[45px] justify-center" title="Zarezerwowane">
                                    {material.quantityReserved}
                                  </Badge>
                                  <Badge variant="default" className="text-xs font-medium min-w-[45px] justify-center bg-green-600 hover:bg-green-700" title="Dostępne">
                                    {(material.quantity || 0) - material.quantityReserved}
                                  </Badge>
                                  <Badge variant="outline" className="text-xs font-medium min-w-[45px] justify-center" title="Całkowite">
                                    {material.quantity || 0}
                                  </Badge>
                                </>
                              ) : (
                                <Badge variant="default" className="font-semibold text-sm min-w-[60px] justify-center">
                                  {material.quantity || 0}
                                </Badge>
                              )}
                            </div>
                          )}
                          
                          {/* Produkty-spakowane specific columns - separate columns matching headers */}
                          {category === 'produkty-spakowane' && (() => {
                            const productTypeDict = productTypeDictionaries.find(d => d.code === material.catalogProductType);
                            const colorDict = colorDictionaries.find((d: any) => d.code === material.catalogColor);
                            const doorDict = doorDictionaries.find(d => d.code === material.catalogDoors);
                            const legDict = legDictionaries.find(d => d.code === material.catalogLegs);
                            
                            const baseTileClass = "h-6 text-xs font-medium flex items-center justify-center border";
                            
                            return (
                              <>
                                {/* Product Type column - w-[70px] */}
                                <div className="w-[70px] shrink-0 flex items-center">
                                  {material.catalogProductType ? (
                                    <div
                                      className={`${baseTileClass} w-full`}
                                      style={{
                                        backgroundColor: productTypeDict?.color || '#e5e7eb',
                                        color: getTextColorForBackground(productTypeDict?.color),
                                        borderColor: needsVisibleBorder(productTypeDict?.color) ? '#d1d5db' : 'transparent'
                                      }}
                                      title={productTypeDict?.readableName || material.catalogProductType}
                                    >
                                      {material.catalogProductType}
                                    </div>
                                  ) : <span className="text-xs text-muted-foreground">-</span>}
                                </div>
                                
                                {/* Length column - w-[50px] */}
                                <div className="w-[50px] shrink-0 flex items-center">
                                  {material.catalogLength ? (
                                    <div className={`${baseTileClass} w-full bg-muted`} title="Długość">
                                      {material.catalogLength}
                                    </div>
                                  ) : <span className="text-xs text-muted-foreground">-</span>}
                                </div>
                                
                                {/* Width column - w-[50px] */}
                                <div className="w-[50px] shrink-0 flex items-center">
                                  {material.catalogWidth ? (
                                    <div className={`${baseTileClass} w-full bg-muted`} title="Szerokość">
                                      {material.catalogWidth}
                                    </div>
                                  ) : <span className="text-xs text-muted-foreground">-</span>}
                                </div>
                                
                                {/* Color column - w-[80px] */}
                                <div className="w-[80px] shrink-0 flex items-center">
                                  {material.catalogColor ? (
                                    <div
                                      className={`${baseTileClass} w-full`}
                                      style={{
                                        backgroundColor: colorDict?.color || '#e5e7eb',
                                        color: getTextColorForBackground(colorDict?.color),
                                        borderColor: needsVisibleBorder(colorDict?.color) ? '#d1d5db' : 'transparent'
                                      }}
                                      title={colorDict?.readableName || material.catalogColor}
                                    >
                                      {material.catalogColor}
                                    </div>
                                  ) : <span className="text-xs text-muted-foreground">-</span>}
                                </div>
                                
                                {/* Legs column - w-[50px] */}
                                <div className="w-[50px] shrink-0 flex items-center">
                                  {material.catalogLegs ? (
                                    <div
                                      className={`${baseTileClass} w-full`}
                                      style={{
                                        backgroundColor: legDict?.color || '#e5e7eb',
                                        color: getTextColorForBackground(legDict?.color),
                                        borderColor: needsVisibleBorder(legDict?.color) ? '#d1d5db' : 'transparent'
                                      }}
                                      title={legDict?.readableName || material.catalogLegs}
                                    >
                                      {material.catalogLegs}
                                    </div>
                                  ) : <span className="text-xs text-muted-foreground">-</span>}
                                </div>
                                
                                {/* Doors column - w-[50px] */}
                                <div className="w-[50px] shrink-0 flex items-center">
                                  {material.catalogDoors ? (
                                    <div
                                      className={`${baseTileClass} w-full`}
                                      style={{
                                        backgroundColor: doorDict?.color || '#e5e7eb',
                                        color: getTextColorForBackground(doorDict?.color),
                                        borderColor: needsVisibleBorder(doorDict?.color) ? '#d1d5db' : 'transparent'
                                      }}
                                      title={doorDict?.readableName || material.catalogDoors}
                                    >
                                      {material.catalogDoors}
                                    </div>
                                  ) : <span className="text-xs text-muted-foreground">-</span>}
                                </div>
                                
                                {/* SKU column - w-[120px] linked to catalog product */}
                                <div className="w-[120px] shrink-0 flex items-center">
                                  {material.catalogProductSku && material.catalogProductId ? (
                                    <a
                                      href={`/catalog-products/${material.catalogProductId}`}
                                      onClick={(e) => {
                                        e.stopPropagation();
                                      }}
                                      className="text-xs text-primary hover:underline truncate"
                                      title={material.catalogProductSku}
                                    >
                                      {material.catalogProductSku}
                                    </a>
                                  ) : <span className="text-xs text-muted-foreground">-</span>}
                                </div>
                              </>
                            );
                          })()}
                          
                          {/* Name - for produkty-spakowane (AFTER parameters) */}
                          {category === 'produkty-spakowane' && (
                            <div className="w-[200px] shrink-0 flex items-center gap-2 min-w-0">
                              <span className="font-medium text-sm truncate">{material.name}</span>
                            </div>
                          )}
                          
                          {/* Default columns for non-formatki and non-produkty-spakowane categories */}
                          {category !== 'formatki' && category !== 'produkty-spakowane' && (
                            <>
                              {/* Internal Code */}
                              <div className="w-[100px] shrink-0 text-xs text-muted-foreground truncate">{material.internalCode}</div>
                              
                              {/* Supplier Code */}
                              <div className="w-[100px] shrink-0 text-xs text-muted-foreground truncate">{material.supplierCode || '-'}</div>
                              
                              {/* Group */}
                              <div className="w-[120px] shrink-0 text-xs truncate">{material.groupName || '-'}</div>
                              
                              {/* Unit of Measure */}
                              <div className="w-[100px] shrink-0 text-xs text-muted-foreground">{material.unitOfMeasure}</div>
                              
                              {/* Price */}
                              <div className="w-[80px] shrink-0 text-xs">
                                {material.price ? `${parseFloat(material.price).toFixed(2)} zł` : '-'}
                              </div>
                            </>
                          )}
                          
                          {/* Location - for non-formatki categories only (formatki use dynamic columns) */}
                          {category !== 'formatki' && (
                          <div className="shrink-0 text-xs text-muted-foreground truncate" style={{width: 120}}>
                            {material.locationName || '-'}
                          </div>
                          )}
                          
                          {/* Carrier - for non-formatki categories only (formatki use dynamic columns) */}
                          {category !== 'formatki' && (
                          <div className="shrink-0 text-xs text-muted-foreground truncate" style={{width: 120}}>
                            {material.carrierName || '-'}
                          </div>
                          )}

                          {/* Icon indicator */}
                          <div className="w-[50px] shrink-0 flex items-center justify-center">
                            <ArrowRight className="h-4 w-4" />
                          </div>
                        </Link>
                        </div>
                      );
                    })}
                  </div>
                </div>
              </div>
            )}
          </CardContent>
        </Card>

        {/* Pagination Bottom */}
        {pagination && (
          <div className="flex items-center justify-between">
            <div className="text-sm text-muted-foreground">
              Pokazano {((page - 1) * limit) + 1} - {Math.min(page * limit, pagination.total)} z {pagination.total}
            </div>
            <div className="flex items-center gap-2">
              <Button
                variant="outline"
                size="sm"
                onClick={() => setPage(1)}
                disabled={page === 1}
                data-testid="button-page-first-bottom"
              >
                <ChevronsLeft className="w-4 h-4" />
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => setPage((p: number) => Math.max(1, p - 1))}
                disabled={page === 1}
                data-testid="button-page-prev-bottom"
              >
                <ChevronLeft className="w-4 h-4" />
              </Button>
              <span className="text-sm">
                Strona {page} z {pagination.totalPages}
              </span>
              <Button
                variant="outline"
                size="sm"
                onClick={() => setPage((p: number) => Math.min(pagination.totalPages, p + 1))}
                disabled={page === pagination.totalPages}
                data-testid="button-page-next-bottom"
              >
                <ChevronRight className="w-4 h-4" />
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => setPage(pagination.totalPages)}
                disabled={page === pagination.totalPages}
                data-testid="button-page-last-bottom"
              >
                <ChevronsRight className="w-4 h-4" />
              </Button>
              <Select value={limit.toString()} onValueChange={(v) => { setLimit(parseInt(v)); setPage(1); }}>
                <SelectTrigger className="w-24" data-testid="select-page-size-bottom">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="10">10</SelectItem>
                  <SelectItem value="25">25</SelectItem>
                  <SelectItem value="50">50</SelectItem>
                  <SelectItem value="100">100</SelectItem>
                  <SelectItem value="250">250</SelectItem>
                  <SelectItem value="500">500</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
        )}
      </div>

      {/* Add Group Dialog */}
      <Dialog open={isAddGroupDialogOpen} onOpenChange={setIsAddGroupDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Dodaj grupę materiałów</DialogTitle>
            <DialogDescription>
              Utwórz nową grupę materiałów dla kategorii {CATEGORY_LABELS[category]}
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label htmlFor="group-name">Nazwa grupy *</Label>
              <Input
                id="group-name"
                value={newGroupName}
                onChange={(e) => {
                  const name = e.target.value;
                  setNewGroupName(name);
                  // Auto-generate code from name
                  setNewGroupCode(generateSlug(name));
                }}
                placeholder="np. Zawiasy standardowe"
                data-testid="input-group-name"
              />
            </div>
            <div>
              <Label htmlFor="group-code">Kod grupy *</Label>
              <Input
                id="group-code"
                value={newGroupCode}
                onChange={(e) => setNewGroupCode(e.target.value)}
                placeholder="np. zawiasy_std"
                data-testid="input-group-code"
              />
            </div>
            <div>
              <Label htmlFor="group-description">Opis (opcjonalnie)</Label>
              <Textarea
                id="group-description"
                value={newGroupDescription}
                onChange={(e) => setNewGroupDescription(e.target.value)}
                placeholder="Opis grupy materiałów"
                data-testid="input-group-description"
              />
            </div>
            <div>
              <Label htmlFor="group-color">Kolor grupy (opcjonalnie)</Label>
              <div className="flex items-center gap-2">
                <Input
                  id="group-color"
                  type="color"
                  value={newGroupColor || "#6366f1"}
                  onChange={(e) => setNewGroupColor(e.target.value)}
                  className="w-12 h-9 p-1 cursor-pointer"
                  data-testid="input-group-color"
                />
                <Input
                  value={newGroupColor}
                  onChange={(e) => {
                    const val = e.target.value;
                    if (val === '' || /^#[0-9A-Fa-f]{0,6}$/.test(val)) {
                      setNewGroupColor(val);
                    }
                  }}
                  placeholder="#FF5733"
                  className="flex-1"
                  maxLength={7}
                  data-testid="input-group-color-hex"
                />
                {newGroupColor && (
                  <Button
                    type="button"
                    variant="ghost"
                    size="icon"
                    onClick={() => setNewGroupColor("")}
                    data-testid="button-clear-group-color"
                  >
                    <X className="w-4 h-4" />
                  </Button>
                )}
              </div>
              <p className="text-xs text-muted-foreground mt-1">
                Kolor używany do wyróżnienia grupy w interfejsie
              </p>
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => setIsAddGroupDialogOpen(false)}
              data-testid="button-cancel-group"
            >
              Anuluj
            </Button>
            <Button
              onClick={handleAddGroup}
              disabled={addGroupMutation.isPending}
              data-testid="button-submit-group"
            >
              {addGroupMutation.isPending && <Loader2 className="w-4 h-4 mr-2 animate-spin" />}
              Utwórz grupę
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Edit Group Dialog */}
      <Dialog open={isEditGroupDialogOpen} onOpenChange={setIsEditGroupDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Edytuj grupę materiałów</DialogTitle>
            <DialogDescription>
              Edytuj grupę materiałów: {editingGroup?.name}
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label htmlFor="edit-group-name">Nazwa grupy *</Label>
              <Input
                id="edit-group-name"
                value={editGroupName}
                onChange={(e) => setEditGroupName(e.target.value)}
                placeholder="np. Zawiasy standardowe"
                data-testid="input-edit-group-name"
              />
            </div>
            <div>
              <Label htmlFor="edit-group-code">Kod grupy *</Label>
              <Input
                id="edit-group-code"
                value={editGroupCode}
                onChange={(e) => setEditGroupCode(e.target.value)}
                placeholder="np. zawiasy_std"
                data-testid="input-edit-group-code"
              />
            </div>
            <div>
              <Label htmlFor="edit-group-display-order">Kolejność wyświetlania</Label>
              <Input
                id="edit-group-display-order"
                type="number"
                value={editGroupDisplayOrder}
                onChange={(e) => setEditGroupDisplayOrder(e.target.value)}
                placeholder="0"
                data-testid="input-edit-group-display-order"
              />
              <p className="text-xs text-muted-foreground mt-1">
                Niższy numer = wyżej na liście
              </p>
            </div>
            <div>
              <Label htmlFor="edit-group-description">Opis (opcjonalnie)</Label>
              <Textarea
                id="edit-group-description"
                value={editGroupDescription}
                onChange={(e) => setEditGroupDescription(e.target.value)}
                placeholder="Opis grupy materiałów"
                data-testid="input-edit-group-description"
              />
            </div>
            <div>
              <Label htmlFor="edit-group-color">Kolor grupy (opcjonalnie)</Label>
              <div className="flex items-center gap-2">
                <Input
                  id="edit-group-color"
                  type="color"
                  value={editGroupColor || "#6366f1"}
                  onChange={(e) => setEditGroupColor(e.target.value)}
                  className="w-12 h-9 p-1 cursor-pointer"
                  data-testid="input-edit-group-color"
                />
                <Input
                  value={editGroupColor}
                  onChange={(e) => {
                    const val = e.target.value;
                    if (val === '' || /^#[0-9A-Fa-f]{0,6}$/.test(val)) {
                      setEditGroupColor(val);
                    }
                  }}
                  placeholder="#FF5733"
                  className="flex-1"
                  maxLength={7}
                  data-testid="input-edit-group-color-hex"
                />
                {editGroupColor && (
                  <Button
                    type="button"
                    variant="ghost"
                    size="icon"
                    onClick={() => setEditGroupColor("")}
                    data-testid="button-clear-edit-group-color"
                  >
                    <X className="w-4 h-4" />
                  </Button>
                )}
              </div>
              <p className="text-xs text-muted-foreground mt-1">
                Kolor używany do wyróżnienia grupy w interfejsie
              </p>
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => setIsEditGroupDialogOpen(false)}
              data-testid="button-cancel-edit-group"
            >
              Anuluj
            </Button>
            <Button
              onClick={handleUpdateGroup}
              disabled={editGroupMutation.isPending}
              data-testid="button-update-group"
            >
              {editGroupMutation.isPending && <Loader2 className="w-4 h-4 mr-2 animate-spin" />}
              Zapisz zmiany
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Add Material Dialog */}
      <Dialog open={isAddMaterialDialogOpen} onOpenChange={setIsAddMaterialDialogOpen}>
        <DialogContent className="max-w-[95vw] sm:max-w-2xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>{category === 'formatki' ? 'Dodaj formatkę' : 'Dodaj materiał'}</DialogTitle>
            <DialogDescription>
              {groupCode 
                ? `Dodajesz ${category === 'formatki' ? 'formatkę' : 'materiał'} do grupy: ${categoryGroups.find(g => g.code === groupCode)?.name || groupCode}`
                : `Utwórz ${category === 'formatki' ? 'nową formatkę' : 'nowy materiał'} w kategorii ` + (CATEGORY_LABELS[category] || category)
              }
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            {category === 'formatki' ? (
              <>
                {/* Formatka-specific fields */}
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="formatka-cz1">CZ1 (opcjonalnie)</Label>
                    <DictionaryComboboxWithAdd
                      items={cz1Dictionaries}
                      value={newFormatkaCz1}
                      onChange={setNewFormatkaCz1}
                      dictionaryType="component_cz1"
                      placeholder="Wybierz CZ1..."
                      searchPlaceholder="Szukaj CZ1..."
                      emptyText="Nie znaleziono CZ1."
                      testId="combobox-formatka-cz1"
                      valueField="code"
                      displayField="readableName"
                    />
                  </div>
                  <div>
                    <Label htmlFor="formatka-cz2">CZ2 (opcjonalnie)</Label>
                    <DictionaryComboboxWithAdd
                      items={cz2Dictionaries}
                      value={newFormatkaCz2}
                      onChange={setNewFormatkaCz2}
                      dictionaryType="component_cz2"
                      placeholder="Wybierz CZ2..."
                      searchPlaceholder="Szukaj CZ2..."
                      emptyText="Nie znaleziono CZ2."
                      testId="combobox-formatka-cz2"
                      valueField="code"
                      displayField="readableName"
                    />
                  </div>
                </div>
                <div className="grid grid-cols-3 gap-4">
                  <div>
                    <Label htmlFor="formatka-length">Długość (mm) *</Label>
                    <DictionaryComboboxWithAdd
                      items={lengthDictionaries}
                      value={newFormatkaLength}
                      onChange={setNewFormatkaLength}
                      dictionaryType="dimension_length"
                      placeholder="Wybierz długość..."
                      searchPlaceholder="Szukaj długości..."
                      emptyText="Nie znaleziono długości."
                      testId="combobox-formatka-length"
                      valueField="code"
                      displayField="readableName"
                    />
                  </div>
                  <div>
                    <Label htmlFor="formatka-width">Szerokość (mm) *</Label>
                    <DictionaryComboboxWithAdd
                      items={widthDictionaries}
                      value={newFormatkaWidth}
                      onChange={setNewFormatkaWidth}
                      dictionaryType="dimension_width"
                      placeholder="Wybierz szerokość..."
                      searchPlaceholder="Szukaj szerokości..."
                      emptyText="Nie znaleziono szerokości."
                      testId="combobox-formatka-width"
                      valueField="code"
                      displayField="readableName"
                    />
                  </div>
                  <div>
                    <Label htmlFor="formatka-thickness">Grubość (mm) *</Label>
                    <Input
                      id="formatka-thickness"
                      type="number"
                      step="1"
                      min="1"
                      value={newFormatkaThickness}
                      onChange={(e) => setNewFormatkaThickness(e.target.value)}
                      placeholder="18"
                      data-testid="input-formatka-thickness"
                    />
                  </div>
                </div>
                <div className="grid grid-cols-3 gap-4">
                  <div>
                    <Label htmlFor="formatka-board-code">Płyta (opcjonalnie)</Label>
                    <Combobox
                      options={plytyMaterials.map((plyta: any) => ({
                        value: plyta.internalCode || plyta.name,
                        label: `${plyta.name}${plyta.thickness ? ` (${plyta.thickness}mm)` : ''}`
                      }))}
                      value={newFormatkaBoardCode}
                      onChange={setNewFormatkaBoardCode}
                      placeholder="Wybierz płytę..."
                      searchPlaceholder="Szukaj płyty..."
                      emptyText="Nie znaleziono płyty."
                      className="w-full"
                    />
                  </div>
                  <div>
                    <Label htmlFor="formatka-edging-code">Obrzeże (opcjonalnie)</Label>
                    <Combobox
                      options={obrzezaMaterials.map((obrzeze: any) => ({
                        value: obrzeze.internalCode || obrzeze.name,
                        label: `${obrzeze.name}${obrzeze.width ? ` (${obrzeze.width}mm)` : ''}`
                      }))}
                      value={newFormatkaEdgingCode}
                      onChange={setNewFormatkaEdgingCode}
                      placeholder="Wybierz obrzeże..."
                      searchPlaceholder="Szukaj obrzeża..."
                      emptyText="Nie znaleziono obrzeża."
                      className="w-full"
                    />
                  </div>
                  <div>
                    <Label htmlFor="formatka-color-code">Kod koloru (opcjonalnie)</Label>
                    <DictionaryComboboxWithAdd
                      items={colorDictionaries}
                      value={newFormatkaColorCode}
                      onChange={setNewFormatkaColorCode}
                      dictionaryType="color"
                      placeholder="Wybierz kolor..."
                      searchPlaceholder="Szukaj koloru..."
                      emptyText="Nie znaleziono koloru."
                      testId="combobox-formatka-color-code"
                      valueField="code"
                      displayField="name"
                    />
                  </div>
                </div>
                <div>
                  <div className="flex items-center gap-2 mb-2">
                    <Label>Krawędzie oklejone</Label>
                    <Button
                      type="button"
                      variant="ghost"
                      size="icon"
                      className="h-6 w-6"
                      onClick={() => {
                        const allChecked = newFormatkaEdge1 && newFormatkaEdge2 && newFormatkaEdge3 && newFormatkaEdge4;
                        setNewFormatkaEdge1(!allChecked);
                        setNewFormatkaEdge2(!allChecked);
                        setNewFormatkaEdge3(!allChecked);
                        setNewFormatkaEdge4(!allChecked);
                      }}
                      title={newFormatkaEdge1 && newFormatkaEdge2 && newFormatkaEdge3 && newFormatkaEdge4 ? "Odznacz wszystkie" : "Zaznacz wszystkie"}
                      data-testid="button-toggle-all-edges"
                    >
                      {newFormatkaEdge1 && newFormatkaEdge2 && newFormatkaEdge3 && newFormatkaEdge4 ? (
                        <X className="h-4 w-4" />
                      ) : (
                        <Check className="h-4 w-4" />
                      )}
                    </Button>
                  </div>
                  <div className="grid grid-cols-4 gap-2">
                    <div className="flex items-center space-x-2">
                      <Checkbox
                        id="formatka-edge1"
                        checked={newFormatkaEdge1}
                        onCheckedChange={(checked) => setNewFormatkaEdge1(!!checked)}
                        data-testid="checkbox-formatka-edge1"
                      />
                      <Label htmlFor="formatka-edge1" className="cursor-pointer">Krawędź 1</Label>
                    </div>
                    <div className="flex items-center space-x-2">
                      <Checkbox
                        id="formatka-edge2"
                        checked={newFormatkaEdge2}
                        onCheckedChange={(checked) => setNewFormatkaEdge2(!!checked)}
                        data-testid="checkbox-formatka-edge2"
                      />
                      <Label htmlFor="formatka-edge2" className="cursor-pointer">Krawędź 2</Label>
                    </div>
                    <div className="flex items-center space-x-2">
                      <Checkbox
                        id="formatka-edge3"
                        checked={newFormatkaEdge3}
                        onCheckedChange={(checked) => setNewFormatkaEdge3(!!checked)}
                        data-testid="checkbox-formatka-edge3"
                      />
                      <Label htmlFor="formatka-edge3" className="cursor-pointer">Krawędź 3</Label>
                    </div>
                    <div className="flex items-center space-x-2">
                      <Checkbox
                        id="formatka-edge4"
                        checked={newFormatkaEdge4}
                        onCheckedChange={(checked) => setNewFormatkaEdge4(!!checked)}
                        data-testid="checkbox-formatka-edge4"
                      />
                      <Label htmlFor="formatka-edge4" className="cursor-pointer">Krawędź 4</Label>
                    </div>
                  </div>
                </div>
              </>
            ) : (
              <>
                {/* Standard material fields */}
                <div>
                  <Label htmlFor="material-name">Nazwa materiału *</Label>
                  <Input
                    id="material-name"
                    value={newMaterialName}
                    onChange={(e) => setNewMaterialName(e.target.value)}
                    placeholder="np. Zawias 35mm"
                    data-testid="input-material-name"
                  />
                </div>
                <div>
                  <Label htmlFor="material-internal-code">Kod wewnętrzny *</Label>
                  <Input
                    id="material-internal-code"
                    value={newMaterialInternalCode}
                    onChange={(e) => {
                      setNewMaterialInternalCode(e.target.value);
                      setIsInternalCodeManuallyEdited(true);
                    }}
                    placeholder="np. ZAW-35-STD"
                    data-testid="input-material-internal-code"
                  />
                  <p className="text-xs text-muted-foreground mt-1">
                    Kod generowany automatycznie z nazwy. Możesz go edytować.
                  </p>
                </div>
              </>
            )}
            <div>
              <Label htmlFor="material-group">
                Grupa materiałów {groupCode ? '' : '(opcjonalnie)'}
              </Label>
              <Select 
                value={newMaterialGroupId?.toString() || "none"} 
                onValueChange={(value) => setNewMaterialGroupId(value === "none" ? null : parseInt(value))}
                disabled={!!groupCode}
              >
                <SelectTrigger id="material-group" data-testid="select-material-group">
                  <SelectValue placeholder="Bez grupy" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="none">Bez grupy</SelectItem>
                  {categoryGroups.map((group) => (
                    <SelectItem key={group.id} value={group.id.toString()}>
                      {group.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              {groupCode && (
                <p className="text-xs text-muted-foreground mt-1">
                  Grupa jest automatycznie wybrana, ponieważ dodajesz {category === 'formatki' ? 'formatkę' : 'materiał'} z poziomu grupy
                </p>
              )}
            </div>
            {category !== 'formatki' && (
              <>
                <div>
                  <Label htmlFor="material-supplier-code">Kod dostawcy (opcjonalnie)</Label>
                  <Input
                    id="material-supplier-code"
                    value={newMaterialSupplierCode}
                    onChange={(e) => setNewMaterialSupplierCode(e.target.value)}
                    placeholder="np. SUP-12345"
                    data-testid="input-material-supplier-code"
                  />
                </div>
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="material-unit">Jednostka miary</Label>
                    <Select value={newMaterialUnitOfMeasure} onValueChange={setNewMaterialUnitOfMeasure}>
                      <SelectTrigger id="material-unit" data-testid="select-material-unit">
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        {units.map((unit: any) => (
                          <SelectItem key={unit.id} value={unit.code}>
                            {unit.name}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>
                  <div>
                    <Label htmlFor="material-price">Cena (opcjonalnie)</Label>
                    <Input
                      id="material-price"
                      type="number"
                      step="0.01"
                      min="0"
                      value={newMaterialPrice}
                      onChange={(e) => setNewMaterialPrice(e.target.value)}
                      placeholder="0.00"
                      data-testid="input-material-price"
                    />
                  </div>
                </div>
                <div>
                  <Label htmlFor="material-description">Opis (opcjonalnie)</Label>
                  <Textarea
                    id="material-description"
                    value={newMaterialDescription}
                    onChange={(e) => setNewMaterialDescription(e.target.value)}
                    placeholder="Opis materiału"
                    data-testid="input-material-description"
                  />
                </div>
              </>
            )}
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => setIsAddMaterialDialogOpen(false)}
              data-testid="button-cancel-material"
            >
              Anuluj
            </Button>
            <Button
              onClick={handleAddMaterial}
              disabled={category === 'formatki' ? addStockPanelMutation.isPending : addMaterialMutation.isPending}
              data-testid="button-submit-material"
            >
              {(category === 'formatki' ? addStockPanelMutation.isPending : addMaterialMutation.isPending) && <Loader2 className="w-4 h-4 mr-2 animate-spin" />}
              {category === 'formatki' ? 'Utwórz formatkę' : 'Utwórz materiał'}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Edit Material Dialog */}
      <Dialog open={isEditMaterialDialogOpen} onOpenChange={setIsEditMaterialDialogOpen}>
        <DialogContent className="max-w-[95vw] sm:max-w-2xl max-h-[90vh] overflow-y-auto p-4">
          <DialogHeader className="pb-2">
            <DialogTitle className="text-base">Edytuj materiał</DialogTitle>
            <DialogDescription className="text-xs">
              Edytuj materiał: {editingMaterial?.name}
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-3">
            <div className="grid grid-cols-1 md:grid-cols-3 gap-3">
              <div>
                <Label htmlFor="edit-material-name" className="text-xs">Nazwa materiału *</Label>
                <Input
                  id="edit-material-name"
                  value={editMaterialName}
                  onChange={(e) => setEditMaterialName(e.target.value)}
                  placeholder="np. Zawias 35mm"
                  className="h-8 text-sm"
                  data-testid="input-edit-material-name"
                />
              </div>
              <div>
                <Label htmlFor="edit-material-internal-code" className="text-xs">Kod wewnętrzny *</Label>
                <Input
                  id="edit-material-internal-code"
                  value={editMaterialInternalCode}
                  onChange={(e) => setEditMaterialInternalCode(e.target.value)}
                  placeholder="np. ZAW-35-STD"
                  className="h-8 text-sm"
                  data-testid="input-edit-material-internal-code"
                />
              </div>
              <div>
                <Label htmlFor="edit-material-supplier-code" className="text-xs">Kod dostawcy</Label>
                <Input
                  id="edit-material-supplier-code"
                  value={editMaterialSupplierCode}
                  onChange={(e) => setEditMaterialSupplierCode(e.target.value)}
                  placeholder="np. SUP-12345"
                  className="h-8 text-sm"
                  data-testid="input-edit-material-supplier-code"
                />
              </div>
            </div>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-3">
              <div>
                <Label htmlFor="edit-material-group" className="text-xs">Grupa materiałów</Label>
                <Select 
                  value={editMaterialGroupId?.toString() || "none"} 
                  onValueChange={(value) => setEditMaterialGroupId(value === "none" ? null : parseInt(value))}
                >
                  <SelectTrigger id="edit-material-group" className="h-8 text-sm" data-testid="select-edit-material-group">
                    <SelectValue placeholder="Bez grupy" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="none">Bez grupy</SelectItem>
                    {categoryGroups.map((group) => (
                      <SelectItem key={group.id} value={group.id.toString()}>
                        {group.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label htmlFor="edit-material-unit" className="text-xs">Jednostka miary</Label>
                <Select value={editMaterialUnitOfMeasure} onValueChange={setEditMaterialUnitOfMeasure}>
                  <SelectTrigger id="edit-material-unit" className="h-8 text-sm" data-testid="select-edit-material-unit">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    {units.map((unit: any) => (
                      <SelectItem key={unit.id} value={unit.code}>
                        {unit.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label htmlFor="edit-material-price" className="text-xs">Cena</Label>
                <Input
                  id="edit-material-price"
                  type="number"
                  step="0.01"
                  min="0"
                  value={editMaterialPrice}
                  onChange={(e) => setEditMaterialPrice(e.target.value)}
                  placeholder="0.00"
                  className="h-8 text-sm"
                  data-testid="input-edit-material-price"
                />
              </div>
            </div>
            <div>
              <Label htmlFor="edit-material-description" className="text-xs">Opis (opcjonalnie)</Label>
              <Textarea
                id="edit-material-description"
                value={editMaterialDescription}
                onChange={(e) => setEditMaterialDescription(e.target.value)}
                placeholder="Opis materiału"
                className="text-sm min-h-[60px]"
                data-testid="input-edit-material-description"
              />
            </div>

            {/* Image Gallery Section */}
            <div className="border-t pt-3">
              <Label className="text-xs font-semibold mb-2 block">Galeria zdjęć</Label>
              
              {/* Drag & Drop Upload Area */}
              <div 
                className={`border-2 border-dashed rounded-lg p-3 text-center hover-elevate cursor-pointer mb-3 transition-colors ${
                  isDragging ? 'border-primary bg-primary/10' : 'border-border'
                }`}
                onDragEnter={(e) => {
                  e.preventDefault();
                  e.stopPropagation();
                  setIsDragging(true);
                }}
                onDragLeave={(e) => {
                  e.preventDefault();
                  e.stopPropagation();
                  setIsDragging(false);
                }}
                onDragOver={(e) => {
                  e.preventDefault();
                  e.stopPropagation();
                }}
                onDrop={(e) => {
                  e.preventDefault();
                  e.stopPropagation();
                  setIsDragging(false);
                  handleImageUpload(e.dataTransfer.files);
                }}
                onClick={() => {
                  const input = document.createElement('input');
                  input.type = 'file';
                  input.accept = 'image/*';
                  input.multiple = true;
                  input.onchange = (e) => {
                    const target = e.target as HTMLInputElement;
                    handleImageUpload(target.files);
                  };
                  input.click();
                }}
                data-testid="dropzone-material-images"
              >
                <Upload className={`w-5 h-5 mx-auto mb-1 ${isDragging ? 'text-primary' : 'text-muted-foreground'}`} />
                <p className={`text-xs ${isDragging ? 'text-primary font-medium' : 'text-muted-foreground'}`}>
                  {isDragging ? 'Upuść zdjęcia tutaj' : 'Przeciągnij i upuść zdjęcia lub kliknij'}
                </p>
              </div>

              {/* Gallery Grid */}
              {editMaterialGallery.length > 0 && (
                <div className="grid grid-cols-4 gap-2">
                  {editMaterialGallery.map((imageUrl, index) => {
                    const isPrimary = editingMaterial?.primaryImageUrl === imageUrl;
                    return (
                      <div 
                        key={index} 
                        className={`relative group aspect-square rounded-md overflow-hidden border ${
                          isPrimary ? 'border-primary border-2' : 'border-border'
                        }`}
                      >
                        {isPrimary && (
                          <div className="absolute top-1 right-1 z-10">
                            <Badge variant="default" className="gap-0.5 h-5 px-1.5 text-[10px]">
                              <Star className="w-2.5 h-2.5 fill-current" />
                              Główne
                            </Badge>
                          </div>
                        )}
                        <img 
                          src={imageUrl} 
                          alt={`Material image ${index + 1}`} 
                          className="w-full h-full object-cover"
                        />
                        <div className="absolute inset-0 bg-black/60 opacity-0 group-hover:opacity-100 transition-opacity flex items-center justify-center gap-1">
                          {!isPrimary && (
                            <Button
                              variant="secondary"
                              size="icon"
                              className="h-7 w-7"
                              onClick={() => handleSetPrimaryImage(imageUrl)}
                              disabled={setPrimaryImageMutation.isPending}
                              data-testid={`button-set-primary-${index}`}
                              title="Ustaw jako główne"
                            >
                              <Star className="w-3 h-3" />
                            </Button>
                          )}
                          <Button
                            variant="destructive"
                            size="icon"
                            className="h-7 w-7"
                            onClick={() => handleImageDelete(imageUrl)}
                            disabled={deleteMaterialImageMutation.isPending}
                            data-testid={`button-delete-image-${index}`}
                            title="Usuń zdjęcie"
                          >
                            <Trash2 className="w-3 h-3" />
                          </Button>
                        </div>
                      </div>
                    );
                  })}
                </div>
              )}

              {uploadMaterialImageMutation.isPending && (
                <div className="flex items-center justify-center py-3">
                  <Loader2 className="w-4 h-4 animate-spin text-muted-foreground" />
                  <span className="ml-2 text-xs text-muted-foreground">Uploading...</span>
                </div>
              )}
            </div>
          </div>
          <DialogFooter className="gap-2 pt-3">
            <Button
              variant="outline"
              size="sm"
              onClick={() => setIsEditMaterialDialogOpen(false)}
              data-testid="button-cancel-edit-material"
            >
              Anuluj
            </Button>
            <Button
              size="sm"
              onClick={handleUpdateMaterial}
              disabled={editMaterialMutation.isPending}
              data-testid="button-update-material"
            >
              {editMaterialMutation.isPending && <Loader2 className="w-3 h-3 mr-1 animate-spin" />}
              Zapisz zmiany
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Duplicate Material Dialog */}
      <Dialog open={showDuplicateDialog} onOpenChange={setShowDuplicateDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Duplikuj materiał</DialogTitle>
            <DialogDescription>
              Wybierz grupę docelową dla kopii materiału: {duplicateMaterial?.name}
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label htmlFor="duplicate-target-group">Grupa docelowa *</Label>
              <Select 
                value={duplicateTargetGroupId?.toString() || ""} 
                onValueChange={(value) => setDuplicateTargetGroupId(parseInt(value))}
              >
                <SelectTrigger id="duplicate-target-group" data-testid="select-duplicate-target-group">
                  <SelectValue placeholder="Wybierz grupę" />
                </SelectTrigger>
                <SelectContent>
                  {categoryGroups.map((group) => (
                    <SelectItem key={group.id} value={group.id.toString()}>
                      {group.name}
                      {group.id === duplicateMaterial?.groupId && " (aktualna)"}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => {
                setShowDuplicateDialog(false);
                setDuplicateMaterial(null);
                setDuplicateTargetGroupId(null);
              }}
              data-testid="button-cancel-duplicate"
            >
              Anuluj
            </Button>
            <Button
              onClick={handleConfirmDuplicate}
              disabled={duplicateMaterialMutation.isPending}
              data-testid="button-confirm-duplicate"
            >
              {duplicateMaterialMutation.isPending && <Loader2 className="w-4 h-4 mr-2 animate-spin" />}
              Duplikuj
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Bulk Edit Price Dialog */}
      <Dialog open={showBulkEditPriceDialog} onOpenChange={setShowBulkEditPriceDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Zmień cenę materiałów</DialogTitle>
            <DialogDescription>
              Ustaw nową cenę dla {selectedMaterials.length} zaznaczonych materiałów
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label htmlFor="bulk-price">Nowa cena (zł) *</Label>
              <Input
                id="bulk-price"
                type="number"
                step="0.01"
                min="0"
                value={bulkEditPrice}
                onChange={(e) => setBulkEditPrice(e.target.value)}
                placeholder="np. 12.50"
                data-testid="input-bulk-price"
              />
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => {
                setShowBulkEditPriceDialog(false);
                setBulkEditPrice("");
              }}
              data-testid="button-cancel-bulk-price"
            >
              Anuluj
            </Button>
            <Button
              onClick={() => bulkUpdatePriceMutation.mutate(bulkEditPrice)}
              disabled={bulkUpdatePriceMutation.isPending || !bulkEditPrice}
              data-testid="button-confirm-bulk-price"
            >
              {bulkUpdatePriceMutation.isPending && <Loader2 className="w-4 h-4 mr-2 animate-spin" />}
              Zmień cenę
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Bulk Edit Unit Dialog */}
      <Dialog open={showBulkEditUnitDialog} onOpenChange={setShowBulkEditUnitDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Zmień jednostkę miary</DialogTitle>
            <DialogDescription>
              Ustaw nową jednostkę miary dla {selectedMaterials.length} zaznaczonych materiałów
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label htmlFor="bulk-unit">Nowa jednostka miary *</Label>
              <Select value={bulkEditUnit} onValueChange={setBulkEditUnit}>
                <SelectTrigger id="bulk-unit" data-testid="select-bulk-unit">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  {units.map((unit: any) => (
                    <SelectItem key={unit.id} value={unit.code}>
                      {unit.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => {
                setShowBulkEditUnitDialog(false);
                setBulkEditUnit("szt");
              }}
              data-testid="button-cancel-bulk-unit"
            >
              Anuluj
            </Button>
            <Button
              onClick={() => bulkUpdateUnitMutation.mutate(bulkEditUnit)}
              disabled={bulkUpdateUnitMutation.isPending}
              data-testid="button-confirm-bulk-unit"
            >
              {bulkUpdateUnitMutation.isPending && <Loader2 className="w-4 h-4 mr-2 animate-spin" />}
              Zmień jednostkę
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Bulk import progress dialog */}
      <GenerationLogsDialog
        open={showBulkImportLogs}
        onOpenChange={setShowBulkImportLogs}
        sessionId={bulkImportSessionId}
        title="Import materiałów"
        description="Podgląd na żywo procesu importu materiałów"
      />


      {/* Bulk Inventory Count Dialog */}
      <Dialog open={showBulkInventoryDialog} onOpenChange={setShowBulkInventoryDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Utwórz spis inwentaryzacyjny</DialogTitle>
            <DialogDescription>
              Utwórz spis inwentaryzacyjny dla {selectedMaterials.length} zaznaczonych materiałów
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label htmlFor="inventory-name">Nazwa spisu *</Label>
              <Input
                id="inventory-name"
                placeholder="Np. Spis miesięczny - grudzień 2025"
                data-testid="input-inventory-name"
                defaultValue={(() => {
                  // Generate default inventory name: "Spis inwentaryzacyjny DD-MM-YYYY - group1-group2"
                  const today = new Date();
                  const day = String(today.getDate()).padStart(2, '0');
                  const month = String(today.getMonth() + 1).padStart(2, '0');
                  const year = today.getFullYear();
                  const dateStr = `${day}-${month}-${year}`;
                  
                  // Extract unique group names from selected materials
                  const selectedMaterialsData = materials.filter(m => selectedMaterials.includes(m.id));
                  const uniqueGroups = Array.from(new Set(selectedMaterialsData.map(m => m.groupName).filter((g): g is string => Boolean(g))));
                  const groupsStr = uniqueGroups.join('-');
                  
                  return `Spis inwentaryzacyjny ${dateStr}${groupsStr ? ' - ' + groupsStr : ''}`;
                })()}
              />
            </div>
            <div>
              <Label htmlFor="inventory-notes">Uwagi (opcjonalnie)</Label>
              <Textarea
                id="inventory-notes"
                placeholder="Dodatkowe uwagi do spisu"
                data-testid="input-inventory-notes"
              />
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => setShowBulkInventoryDialog(false)}
              data-testid="button-cancel-inventory"
            >
              Anuluj
            </Button>
            <Button
              onClick={() => {
                const nameInput = document.getElementById('inventory-name') as HTMLInputElement;
                const notesInput = document.getElementById('inventory-notes') as HTMLTextAreaElement;
                if (!nameInput.value.trim()) {
                  toast({
                    title: "Błąd",
                    description: "Podaj nazwę spisu",
                    variant: "destructive",
                  });
                  return;
                }
                createInventoryCountMutation.mutate({
                  name: nameInput.value,
                  notes: notesInput.value || undefined,
                });
              }}
              disabled={createInventoryCountMutation.isPending}
              data-testid="button-confirm-inventory"
            >
              {createInventoryCountMutation.isPending && <Loader2 className="w-4 h-4 mr-2 animate-spin" />}
              Utwórz spis
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Bulk Carrier Dialog */}
      <Dialog open={showBulkCarrierDialog} onOpenChange={setShowBulkCarrierDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Zmień nośnik</DialogTitle>
            <DialogDescription>
              Ustaw nośnik dla {selectedMaterials.length} zaznaczonych materiałów
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label htmlFor="bulk-carrier">Nośnik</Label>
              <Combobox
                options={[
                  { value: "null", label: "Brak nośnika" },
                  ...carriers.map((carrier: any) => ({
                    value: carrier.id.toString(),
                    label: carrier.name
                  }))
                ]}
                value={bulkCarrierId}
                onChange={setBulkCarrierId}
                placeholder="Wybierz nośnik"
                searchPlaceholder="Szukaj nośnika..."
                emptyText="Nie znaleziono nośnika"
                className="w-full"
              />
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => {
                setShowBulkCarrierDialog(false);
                setBulkCarrierId("");
              }}
              data-testid="button-cancel-carrier"
            >
              Anuluj
            </Button>
            <Button
              onClick={() => {
                let carrierId: number | null = null;
                if (bulkCarrierId && bulkCarrierId !== "null") {
                  const parsed = parseInt(bulkCarrierId);
                  if (!isNaN(parsed)) {
                    carrierId = parsed;
                  }
                }
                bulkUpdateCarrierMutation.mutate(carrierId);
                setBulkCarrierId("");
              }}
              disabled={!bulkCarrierId || bulkUpdateCarrierMutation.isPending}
              data-testid="button-save-carrier"
            >
              {bulkUpdateCarrierMutation.isPending ? "Zapisywanie..." : "Zapisz"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Bulk Location Dialog */}
      <Dialog open={showBulkLocationDialog} onOpenChange={setShowBulkLocationDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Zmień lokalizację</DialogTitle>
            <DialogDescription>
              Ustaw lokalizację dla {selectedMaterials.length} zaznaczonych materiałów
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label htmlFor="bulk-location">Lokalizacja</Label>
              <Combobox
                options={[
                  { value: "null", label: "Brak lokalizacji" },
                  ...locations.map((location: any) => ({
                    value: location.id.toString(),
                    label: location.name
                  }))
                ]}
                value={bulkLocationId}
                onChange={setBulkLocationId}
                placeholder="Wybierz lokalizację"
                searchPlaceholder="Szukaj lokalizacji..."
                emptyText="Nie znaleziono lokalizacji"
                className="w-full"
              />
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => {
                setShowBulkLocationDialog(false);
                setBulkLocationId("");
              }}
              data-testid="button-cancel-location"
            >
              Anuluj
            </Button>
            <Button
              onClick={() => {
                let locationId: number | null = null;
                if (bulkLocationId && bulkLocationId !== "null") {
                  const parsed = parseInt(bulkLocationId);
                  if (!isNaN(parsed)) {
                    locationId = parsed;
                  }
                }
                bulkUpdateLocationMutation.mutate(locationId);
                setBulkLocationId("");
              }}
              disabled={!bulkLocationId || bulkUpdateLocationMutation.isPending}
              data-testid="button-save-location"
            >
              {bulkUpdateLocationMutation.isPending ? "Zapisywanie..." : "Zapisz"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Bulk Group Change Dialog */}
      <Dialog open={showBulkGroupDialog} onOpenChange={setShowBulkGroupDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Zmień grupę</DialogTitle>
            <DialogDescription>
              Ustaw grupę dla {selectedMaterials.length} zaznaczonych materiałów
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label htmlFor="bulk-group">Grupa</Label>
              <Combobox
                options={[
                  { value: "null", label: "Bez grupy" },
                  ...categoryGroups.map((group: any) => ({
                    value: group.id.toString(),
                    label: group.name
                  }))
                ]}
                value={bulkGroupId}
                onChange={setBulkGroupId}
                placeholder="Wybierz grupę"
                searchPlaceholder="Szukaj grupy..."
                emptyText="Nie znaleziono grupy"
                className="w-full"
              />
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => {
                setShowBulkGroupDialog(false);
                setBulkGroupId("");
              }}
              data-testid="button-cancel-group"
            >
              Anuluj
            </Button>
            <Button
              onClick={() => {
                let groupId: number | null = null;
                if (bulkGroupId && bulkGroupId !== "null") {
                  const parsed = parseInt(bulkGroupId);
                  if (!isNaN(parsed)) {
                    groupId = parsed;
                  }
                }
                bulkUpdateGroupMutation.mutate(groupId);
                setBulkGroupId("");
              }}
              disabled={!bulkGroupId || bulkUpdateGroupMutation.isPending}
              data-testid="button-save-group"
            >
              {bulkUpdateGroupMutation.isPending ? "Zapisywanie..." : "Zapisz"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Bulk Formatka Edit Dialog */}
      {category === 'formatki' && (
        <BulkFormatkaEditDialog
          open={showBulkFormatkaEditDialog}
          onOpenChange={setShowBulkFormatkaEditDialog}
          selectedCount={selectedMaterials.length}
          onSubmit={(updates) => bulkFormatkaEditMutation.mutate(updates)}
          isPending={bulkFormatkaEditMutation.isPending}
          cz1Dictionaries={cz1Dictionaries}
          cz2Dictionaries={cz2Dictionaries}
          colorDictionaries={colorDictionaries}
          furnitureTypeDictionaries={furnitureTypeDictionaries}
          boardOptions={plytyMaterials
            .filter((b: any) => b.internalCode && b.internalCode.trim() !== '')
            .map((b: any) => ({ 
              value: b.internalCode, 
              label: `${b.name}${b.thickness ? ` (${b.thickness}mm)` : ''}`
            }))}
          edgingOptions={obrzezaMaterials
            .filter((e: any) => e.internalCode && e.internalCode.trim() !== '')
            .map((e: any) => ({ 
              value: e.internalCode, 
              label: `${e.name}${e.width ? ` (${e.width}mm)` : ''}`
            }))}
          carriers={carriers}
          locations={locations}
          groups={categoryGroups}
        />
      )}

      {/* Bulk Import from Catalog Dialog */}
      <Dialog open={isBulkImportCatalogDialogOpen} onOpenChange={setIsBulkImportCatalogDialogOpen}>
        <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>Dodaj produkty z katalogu</DialogTitle>
            <DialogDescription>
              Wybierz produkty z katalogu, które chcesz dodać do magazynu produktów spakowanych. 
              Zostanie utworzony spis inwentaryzacyjny, a produkty zostaną automatycznie dodane do magazynu.
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4">
            {/* Inventory Count Name */}
            <div>
              <Label htmlFor="inventory-count-name">Nazwa spisu inwentaryzacyjnego</Label>
              <Input
                id="inventory-count-name"
                placeholder="np. Import produktów katalogowych 2025-01"
                value={inventoryCountName}
                onChange={(e) => setInventoryCountName(e.target.value)}
                data-testid="input-inventory-count-name"
              />
            </div>

            {/* Group Selection */}
            <div>
              <Label htmlFor="catalog-group">Grupa (opcjonalnie)</Label>
              <Select 
                value={catalogGroupId?.toString() || "null"}
                onValueChange={(value) => {
                  setCatalogGroupId(value === "null" ? null : parseInt(value));
                }}
              >
                <SelectTrigger id="catalog-group" data-testid="select-catalog-group">
                  <SelectValue placeholder="Wybierz grupę" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="null">Bez grupy</SelectItem>
                  {categoryGroups.map((group: any) => (
                    <SelectItem key={group.id} value={group.id.toString()}>
                      {group.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            {/* Search Products */}
            <div>
              <Label htmlFor="catalog-search">Szukaj produktów</Label>
              <div className="relative">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                <Input
                  id="catalog-search"
                  placeholder="Wpisz SKU lub nazwę produktu..."
                  value={catalogProductSearch}
                  onChange={(e) => setCatalogProductSearch(e.target.value)}
                  className="pl-9"
                  data-testid="input-catalog-search"
                />
              </div>
            </div>

            {/* Product List */}
            <div className="border rounded-md">
              <div className="bg-muted px-4 py-2 flex items-center justify-between">
                <span className="text-sm font-medium">
                  Produkty ({selectedCatalogProducts.length} zaznaczonych)
                </span>
                {selectedCatalogProducts.length > 0 && (
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => setSelectedCatalogProducts([])}
                    data-testid="button-clear-selection"
                  >
                    Wyczyść zaznaczenie
                  </Button>
                )}
              </div>
              
              <div className="max-h-96 overflow-y-auto">
                {isCatalogProductsLoading ? (
                  <div className="flex items-center justify-center p-8">
                    <Loader2 className="w-6 h-6 animate-spin text-muted-foreground" />
                  </div>
                ) : catalogProducts.length === 0 ? (
                  <div className="text-center p-8 text-muted-foreground">
                    {catalogProductSearch ? 'Brak produktów spełniających kryteria' : 'Wpisz frazę aby wyszukać produkty'}
                  </div>
                ) : (
                  <div className="divide-y">
                    {catalogProducts.map((product: any) => (
                      <div 
                        key={product.id} 
                        className="p-3 hover:bg-muted/50 cursor-pointer flex items-start gap-3"
                        onClick={() => {
                          setSelectedCatalogProducts(prev => 
                            prev.includes(product.id)
                              ? prev.filter(id => id !== product.id)
                              : [...prev, product.id]
                          );
                        }}
                        data-testid={`product-item-${product.id}`}
                      >
                        <Checkbox
                          checked={selectedCatalogProducts.includes(product.id)}
                          onCheckedChange={(checked) => {
                            setSelectedCatalogProducts(prev => 
                              checked
                                ? [...prev, product.id]
                                : prev.filter(id => id !== product.id)
                            );
                          }}
                          data-testid={`checkbox-product-${product.id}`}
                        />
                        <div className="flex-1 min-w-0">
                          <div className="font-medium">{product.title || product.sku}</div>
                          <div className="text-sm text-muted-foreground">SKU: {product.sku}</div>
                          {product.length && product.width && product.height && (
                            <div className="text-xs text-muted-foreground">
                              Wymiary: {product.length} × {product.width} × {product.height} cm
                            </div>
                          )}
                        </div>
                        {selectedCatalogProducts.includes(product.id) && (
                          <div className="w-24" onClick={(e) => e.stopPropagation()}>
                            <Input
                              type="number"
                              placeholder="Ilość"
                              value={catalogProductQuantities[product.id] || ''}
                              onChange={(e) => {
                                const value = parseInt(e.target.value) || 0;
                                setCatalogProductQuantities(prev => ({
                                  ...prev,
                                  [product.id]: value
                                }));
                              }}
                              min="0"
                              step="1"
                              data-testid={`input-quantity-${product.id}`}
                            />
                          </div>
                        )}
                      </div>
                    ))}
                  </div>
                )}
              </div>
            </div>
          </div>

          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => {
                setIsBulkImportCatalogDialogOpen(false);
                setSelectedCatalogProducts([]);
                setCatalogProductQuantities({});
                setInventoryCountName("");
                setCatalogGroupId(null);
                setCatalogProductSearch("");
              }}
              data-testid="button-cancel-import"
            >
              Anuluj
            </Button>
            <Button
              onClick={() => {
                if (!inventoryCountName.trim()) {
                  toast({
                    title: "Brak nazwy spisu",
                    description: "Podaj nazwę spisu inwentaryzacyjnego",
                    variant: "destructive",
                  });
                  return;
                }
                
                if (selectedCatalogProducts.length === 0) {
                  toast({
                    title: "Brak produktów",
                    description: "Wybierz przynajmniej jeden produkt",
                    variant: "destructive",
                  });
                  return;
                }

                bulkImportFromCatalogMutation.mutate({
                  name: inventoryCountName,
                  products: selectedCatalogProducts.map(id => ({ 
                    catalogProductId: id,
                    countedQuantity: catalogProductQuantities[id] || 0
                  })),
                  groupId: catalogGroupId,
                });
              }}
              disabled={bulkImportFromCatalogMutation.isPending || selectedCatalogProducts.length === 0 || !inventoryCountName.trim()}
              data-testid="button-submit-import"
            >
              {bulkImportFromCatalogMutation.isPending && <Loader2 className="w-4 h-4 mr-2 animate-spin" />}
              Importuj {selectedCatalogProducts.length > 0 && `(${selectedCatalogProducts.length})`}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </WarehouseLayout>
  );
}
