import { useState, useCallback } from "react";
import { Link } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { WarehouseLayout } from "@/features/warehouse/warehouse-layout";
import { Card, CardContent } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { useToast } from "@/hooks/use-toast";
import { Search, ArchiveRestore, Loader2, Package, Box, Layers, Wrench, Archive, Trash2 } from "lucide-react";
import { Badge } from "@/components/ui/badge";
import { Checkbox } from "@/components/ui/checkbox";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import dayjs from "dayjs";

interface ArchivedMaterial {
  id: number;
  name: string;
  internalCode: string;
  supplierCode: string | null;
  quantity: number;
  unit: string;
  price: string | null;
  archivedAt: string;
  groupId: number | null;
  groupName: string | null;
  category: string;
  materialCategory: string;
}

interface ArchiveResponse {
  materials: ArchivedMaterial[];
  totalCount: number;
  categoryCounts: Record<string, number>;
}

const categoryConfig: Record<string, { label: string; icon: any; color: string; urlPath: string }> = {
  plyty: { label: "Płyty", icon: Layers, color: "bg-amber-500/20 text-amber-700 dark:text-amber-400", urlPath: "plyty" },
  obrzeza: { label: "Obrzeża", icon: Box, color: "bg-blue-500/20 text-blue-700 dark:text-blue-400", urlPath: "obrzeza" },
  okucia: { label: "Okucia", icon: Wrench, color: "bg-purple-500/20 text-purple-700 dark:text-purple-400", urlPath: "okucia" },
  formatki: { label: "Formatki", icon: Package, color: "bg-green-500/20 text-green-700 dark:text-green-400", urlPath: "formatki" },
  "produkty-spakowane": { label: "Produkty spakowane", icon: Package, color: "bg-indigo-500/20 text-indigo-700 dark:text-indigo-400", urlPath: "produkty-spakowane" },
  opakowania: { label: "Opakowania", icon: Box, color: "bg-orange-500/20 text-orange-700 dark:text-orange-400", urlPath: "opakowania" },
  other: { label: "Inne", icon: Archive, color: "bg-gray-500/20 text-gray-700 dark:text-gray-400", urlPath: "other" },
};

export default function WarehouseArchivePage() {
  const { toast } = useToast();
  const [searchQuery, setSearchQuery] = useState("");
  const [selectedCategory, setSelectedCategory] = useState("all");
  const [selectedItems, setSelectedItems] = useState<Set<number>>(new Set());
  const [showRestoreDialog, setShowRestoreDialog] = useState(false);
  const [itemToRestore, setItemToRestore] = useState<ArchivedMaterial | null>(null);

  const { data: archiveData, isLoading, refetch } = useQuery<ArchiveResponse>({
    queryKey: ['/api/warehouse/archive', selectedCategory, searchQuery],
    queryFn: async () => {
      const params = new URLSearchParams();
      if (selectedCategory !== 'all') params.append('category', selectedCategory);
      if (searchQuery) params.append('search', searchQuery);
      const response = await fetch(`/api/warehouse/archive?${params}`);
      if (!response.ok) throw new Error('Failed to fetch archive');
      return response.json();
    },
  });

  const restoreMaterialMutation = useMutation({
    mutationFn: async (material: ArchivedMaterial) => {
      const endpoint = material.materialCategory === 'formatki'
        ? `/api/warehouse/stock-panels/${material.id}/restore`
        : material.materialCategory === 'produkty-spakowane'
        ? `/api/warehouse/packed-products/${material.id}/restore`
        : `/api/warehouse/materials/${material.id}/restore`;
      
      return apiRequest('PATCH', endpoint);
    },
    onSuccess: () => {
      toast({ title: "Materiał przywrócony", description: "Materiał został przywrócony z archiwum." });
      queryClient.invalidateQueries({ queryKey: ['/api/warehouse/archive'] });
      queryClient.invalidateQueries({ queryKey: ['/api/warehouse/materials/search'] });
      setItemToRestore(null);
      setShowRestoreDialog(false);
    },
    onError: (error: any) => {
      toast({ title: "Błąd", description: error.message || "Nie udało się przywrócić materiału.", variant: "destructive" });
    },
  });

  const bulkRestoreMutation = useMutation({
    mutationFn: async () => {
      const materials = archiveData?.materials.filter(m => selectedItems.has(m.id)) || [];
      
      const materialIds = materials.filter(m => m.materialCategory !== 'formatki' && m.materialCategory !== 'produkty-spakowane').map(m => m.id);
      const stockPanelIds = materials.filter(m => m.materialCategory === 'formatki').map(m => m.id);
      const packedProductIds = materials.filter(m => m.materialCategory === 'produkty-spakowane').map(m => m.id);
      
      const promises = [];
      if (materialIds.length > 0) {
        promises.push(apiRequest('POST', '/api/warehouse/materials/bulk-restore', { materialIds }));
      }
      if (stockPanelIds.length > 0) {
        promises.push(apiRequest('POST', '/api/warehouse/stock-panels/bulk-restore', { ids: stockPanelIds }));
      }
      if (packedProductIds.length > 0) {
        promises.push(apiRequest('POST', '/api/warehouse/packed-products/bulk-restore', { ids: packedProductIds }));
      }
      
      return Promise.all(promises);
    },
    onSuccess: () => {
      toast({ title: "Materiały przywrócone", description: `Przywrócono ${selectedItems.size} materiałów z archiwum.` });
      queryClient.invalidateQueries({ queryKey: ['/api/warehouse/archive'] });
      queryClient.invalidateQueries({ queryKey: ['/api/warehouse/materials/search'] });
      setSelectedItems(new Set());
    },
    onError: (error: any) => {
      toast({ title: "Błąd", description: error.message || "Nie udało się przywrócić materiałów.", variant: "destructive" });
    },
  });

  const handleSelectAll = useCallback((checked: boolean) => {
    if (checked && archiveData?.materials) {
      setSelectedItems(new Set(archiveData.materials.map(m => m.id)));
    } else {
      setSelectedItems(new Set());
    }
  }, [archiveData?.materials]);

  const handleSelectItem = useCallback((id: number, checked: boolean) => {
    setSelectedItems(prev => {
      const next = new Set(prev);
      if (checked) {
        next.add(id);
      } else {
        next.delete(id);
      }
      return next;
    });
  }, []);

  const getMaterialLink = (material: ArchivedMaterial) => {
    if (material.materialCategory === 'formatki') {
      return `/warehouse/formatki/${material.id}`;
    }
    if (material.materialCategory === 'produkty-spakowane') {
      return `/warehouse/produkty-spakowane/${material.id}`;
    }
    const urlPath = categoryConfig[material.materialCategory]?.urlPath || material.materialCategory;
    return `/warehouse/${urlPath}/${material.id}`;
  };

  const filteredMaterials = archiveData?.materials || [];
  const allSelected = filteredMaterials.length > 0 && selectedItems.size === filteredMaterials.length;
  const someSelected = selectedItems.size > 0 && selectedItems.size < filteredMaterials.length;

  return (
    <WarehouseLayout category="archiwum">
      <div className="flex flex-col gap-4 p-4">
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-2">
            <Archive className="h-6 w-6 text-muted-foreground" />
            <h1 className="text-2xl font-bold">Archiwum materiałów</h1>
            {archiveData && (
              <Badge variant="secondary" className="ml-2">
                {archiveData.totalCount} pozycji
              </Badge>
            )}
          </div>
          
          {selectedItems.size > 0 && (
            <Button 
              onClick={() => bulkRestoreMutation.mutate()}
              disabled={bulkRestoreMutation.isPending}
              data-testid="button-bulk-restore"
            >
              {bulkRestoreMutation.isPending ? (
                <Loader2 className="h-4 w-4 mr-2 animate-spin" />
              ) : (
                <ArchiveRestore className="h-4 w-4 mr-2" />
              )}
              Przywróć zaznaczone ({selectedItems.size})
            </Button>
          )}
        </div>

        <Card>
          <CardContent className="p-4">
            <div className="flex flex-col gap-4">
              <div className="flex items-center gap-4">
                <div className="relative flex-1 max-w-md">
                  <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                  <Input
                    placeholder="Szukaj w archiwum..."
                    value={searchQuery}
                    onChange={(e) => setSearchQuery(e.target.value)}
                    className="pl-9"
                    data-testid="input-archive-search"
                  />
                </div>
              </div>

              <Tabs value={selectedCategory} onValueChange={setSelectedCategory}>
                <TabsList className="flex-wrap h-auto gap-1">
                  <TabsTrigger value="all" className="gap-1" data-testid="tab-all">
                    Wszystkie
                    {archiveData?.totalCount ? (
                      <Badge variant="secondary" className="ml-1 h-5 px-1.5 text-xs">
                        {archiveData.totalCount}
                      </Badge>
                    ) : null}
                  </TabsTrigger>
                  {Object.entries(categoryConfig).map(([key, config]) => {
                    const count = archiveData?.categoryCounts[key] || 0;
                    if (count === 0) return null;
                    const Icon = config.icon;
                    return (
                      <TabsTrigger key={key} value={key} className="gap-1" data-testid={`tab-${key}`}>
                        <Icon className="h-3.5 w-3.5" />
                        {config.label}
                        <Badge variant="secondary" className="ml-1 h-5 px-1.5 text-xs">
                          {count}
                        </Badge>
                      </TabsTrigger>
                    );
                  })}
                </TabsList>
              </Tabs>

              {isLoading ? (
                <div className="flex items-center justify-center py-12">
                  <Loader2 className="h-8 w-8 animate-spin text-muted-foreground" />
                </div>
              ) : filteredMaterials.length === 0 ? (
                <div className="flex flex-col items-center justify-center py-12 text-muted-foreground">
                  <Archive className="h-12 w-12 mb-4 opacity-50" />
                  <p className="text-lg font-medium">Archiwum jest puste</p>
                  <p className="text-sm">Zarchiwizowane materiały pojawią się tutaj</p>
                </div>
              ) : (
                <div className="border rounded-lg overflow-hidden">
                  <Table>
                    <TableHeader>
                      <TableRow className="bg-muted/50">
                        <TableHead className="w-10">
                          <Checkbox
                            checked={allSelected}
                            onCheckedChange={handleSelectAll}
                            aria-label="Zaznacz wszystkie"
                            data-testid="checkbox-select-all"
                          />
                        </TableHead>
                        <TableHead>Nazwa</TableHead>
                        <TableHead>Kod wewnętrzny</TableHead>
                        <TableHead>Kategoria</TableHead>
                        <TableHead>Grupa</TableHead>
                        <TableHead className="text-right">Ilość</TableHead>
                        <TableHead>Data archiwizacji</TableHead>
                        <TableHead className="w-24">Akcje</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {filteredMaterials.map((material) => {
                        const config = categoryConfig[material.materialCategory] || categoryConfig.other;
                        const Icon = config.icon;
                        return (
                          <TableRow key={`${material.materialCategory}-${material.id}`} data-testid={`row-archive-${material.id}`}>
                            <TableCell>
                              <Checkbox
                                checked={selectedItems.has(material.id)}
                                onCheckedChange={(checked) => handleSelectItem(material.id, checked as boolean)}
                                aria-label={`Zaznacz ${material.name}`}
                                data-testid={`checkbox-item-${material.id}`}
                              />
                            </TableCell>
                            <TableCell>
                              <Link href={getMaterialLink(material)} className="hover:underline font-medium">
                                {material.name}
                              </Link>
                            </TableCell>
                            <TableCell className="font-mono text-sm text-muted-foreground">
                              {material.internalCode}
                            </TableCell>
                            <TableCell>
                              <Badge variant="outline" className={config.color}>
                                <Icon className="h-3 w-3 mr-1" />
                                {config.label}
                              </Badge>
                            </TableCell>
                            <TableCell>
                              {material.groupName ? (
                                <span className="text-sm">{material.groupName}</span>
                              ) : (
                                <span className="text-sm text-muted-foreground">—</span>
                              )}
                            </TableCell>
                            <TableCell className="text-right font-medium">
                              {material.quantity} {material.unit}
                            </TableCell>
                            <TableCell className="text-sm text-muted-foreground">
                              {material.archivedAt ? dayjs(material.archivedAt).format('DD.MM.YYYY HH:mm') : '—'}
                            </TableCell>
                            <TableCell>
                              <Button
                                variant="ghost"
                                size="sm"
                                onClick={() => {
                                  setItemToRestore(material);
                                  setShowRestoreDialog(true);
                                }}
                                data-testid={`button-restore-${material.id}`}
                              >
                                <ArchiveRestore className="h-4 w-4" />
                              </Button>
                            </TableCell>
                          </TableRow>
                        );
                      })}
                    </TableBody>
                  </Table>
                </div>
              )}
            </div>
          </CardContent>
        </Card>
      </div>

      <AlertDialog open={showRestoreDialog} onOpenChange={setShowRestoreDialog}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Przywróć materiał</AlertDialogTitle>
            <AlertDialogDescription>
              Czy na pewno chcesz przywrócić materiał "{itemToRestore?.name}" z archiwum?
              Materiał zostanie ponownie widoczny na liście materiałów.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel data-testid="button-cancel-restore">Anuluj</AlertDialogCancel>
            <AlertDialogAction
              onClick={() => itemToRestore && restoreMaterialMutation.mutate(itemToRestore)}
              disabled={restoreMaterialMutation.isPending}
              data-testid="button-confirm-restore"
            >
              {restoreMaterialMutation.isPending ? (
                <Loader2 className="h-4 w-4 mr-2 animate-spin" />
              ) : (
                <ArchiveRestore className="h-4 w-4 mr-2" />
              )}
              Przywróć
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </WarehouseLayout>
  );
}
