import { useState } from "react";
import { useLocation } from "wouter";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Label } from "@/components/ui/label";
import { Package, Loader2, ArrowLeft, Check } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";

type ResetStep = "request" | "reset" | "success";

export default function ResetPasswordPage() {
  const [, navigate] = useLocation();
  const { toast } = useToast();
  const [step, setStep] = useState<ResetStep>("request");
  const [isLoading, setIsLoading] = useState(false);

  // Request step
  const [email, setEmail] = useState("");
  const [resetToken, setResetToken] = useState("");

  // Reset step
  const [token, setToken] = useState("");
  const [newPassword, setNewPassword] = useState("");
  const [confirmPassword, setConfirmPassword] = useState("");

  const handleRequestReset = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsLoading(true);

    try {
      const res = await apiRequest("POST", "/api/password-reset/request", { email });
      const data = await res.json();
      
      // In development, the token is returned in response
      if (data.resetToken) {
        setResetToken(data.resetToken);
      }

      toast({
        title: "Email wysłany",
        description: data.message,
      });

      setStep("reset");
    } catch (error: any) {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się wysłać żądania resetowania hasła",
        variant: "destructive",
      });
    } finally {
      setIsLoading(false);
    }
  };

  const handleResetPassword = async (e: React.FormEvent) => {
    e.preventDefault();

    if (newPassword !== confirmPassword) {
      toast({
        title: "Błąd",
        description: "Hasła nie są identyczne",
        variant: "destructive",
      });
      return;
    }

    if (newPassword.length < 6) {
      toast({
        title: "Błąd",
        description: "Hasło musi mieć minimum 6 znaków",
        variant: "destructive",
      });
      return;
    }

    setIsLoading(true);

    try {
      const res = await apiRequest("POST", "/api/password-reset/reset", {
        token: token || resetToken,
        newPassword,
      });
      const data = await res.json();

      toast({
        title: "Sukces",
        description: data.message,
      });

      setStep("success");
    } catch (error: any) {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zresetować hasła",
        variant: "destructive",
      });
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-background to-muted p-4">
      <div className="w-full max-w-md">
        <Button
          variant="ghost"
          onClick={() => navigate("/auth")}
          className="mb-4"
          data-testid="button-back-to-login"
        >
          <ArrowLeft className="h-4 w-4 mr-2" />
          Powrót do logowania
        </Button>

        <Card>
          <CardHeader>
            <div className="flex items-center gap-3 mb-2">
              <Package className="h-8 w-8 text-primary" />
              <CardTitle>Resetowanie hasła</CardTitle>
            </div>
            <CardDescription>
              {step === "request" && "Podaj adres email, aby otrzymać link do resetowania hasła"}
              {step === "reset" && "Ustaw nowe hasło dla swojego konta"}
              {step === "success" && "Hasło zostało pomyślnie zresetowane"}
            </CardDescription>
          </CardHeader>
          <CardContent>
            {step === "request" && (
              <form onSubmit={handleRequestReset} className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="email">Adres email</Label>
                  <Input
                    id="email"
                    type="email"
                    placeholder="twoj@email.com"
                    value={email}
                    onChange={(e) => setEmail(e.target.value)}
                    required
                    data-testid="input-reset-email"
                  />
                </div>
                <Button
                  type="submit"
                  className="w-full"
                  disabled={isLoading}
                  data-testid="button-request-reset"
                >
                  {isLoading ? (
                    <>
                      <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                      Wysyłanie...
                    </>
                  ) : (
                    "Wyślij link resetujący"
                  )}
                </Button>
              </form>
            )}

            {step === "reset" && (
              <form onSubmit={handleResetPassword} className="space-y-4">
                {resetToken && (
                  <div className="p-3 bg-muted rounded-md">
                    <p className="text-sm text-muted-foreground mb-1">
                      Token resetowania (development only):
                    </p>
                    <code className="text-xs break-all">{resetToken}</code>
                  </div>
                )}
                
                <div className="space-y-2">
                  <Label htmlFor="token">Token resetowania</Label>
                  <Input
                    id="token"
                    type="text"
                    placeholder="Wklej token z emaila"
                    value={token}
                    onChange={(e) => setToken(e.target.value)}
                    required={!resetToken}
                    data-testid="input-reset-token"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="new-password">Nowe hasło</Label>
                  <Input
                    id="new-password"
                    type="password"
                    placeholder="••••••••"
                    value={newPassword}
                    onChange={(e) => setNewPassword(e.target.value)}
                    required
                    minLength={6}
                    data-testid="input-new-password"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="confirm-password">Potwierdź hasło</Label>
                  <Input
                    id="confirm-password"
                    type="password"
                    placeholder="••••••••"
                    value={confirmPassword}
                    onChange={(e) => setConfirmPassword(e.target.value)}
                    required
                    minLength={6}
                    data-testid="input-confirm-password"
                  />
                </div>
                <Button
                  type="submit"
                  className="w-full"
                  disabled={isLoading}
                  data-testid="button-reset-password"
                >
                  {isLoading ? (
                    <>
                      <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                      Resetowanie...
                    </>
                  ) : (
                    "Zresetuj hasło"
                  )}
                </Button>
              </form>
            )}

            {step === "success" && (
              <div className="text-center space-y-4">
                <div className="mx-auto w-12 h-12 rounded-full bg-green-500/20 flex items-center justify-center">
                  <Check className="h-6 w-6 text-green-500" />
                </div>
                <p className="text-muted-foreground">
                  Twoje hasło zostało pomyślnie zresetowane. Możesz teraz zalogować się używając nowego hasła.
                </p>
                <Button
                  onClick={() => navigate("/auth")}
                  className="w-full"
                  data-testid="button-go-to-login"
                >
                  Przejdź do logowania
                </Button>
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </div>
  );
}
