import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { Link, useLocation } from "wouter";
import { Plus, Edit, Trash2, Loader2, Play, CheckCircle, XCircle, Calendar, Package } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { queryClient } from "@/lib/queryClient";
import type { ProductionOrder } from "@shared/schema";
import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { ProductionOrderForm, orderFormSchema, type OrderFormValues } from "@/features/production-orders/production-order-form";

const statusColors: Record<string, string> = {
  draft: "bg-gray-500",
  confirmed: "bg-blue-500",
  planned: "bg-indigo-500",
  in_progress: "bg-yellow-500",
  paused: "bg-orange-500",
  done: "bg-green-500",
  cancelled: "bg-red-500",
};

const statusLabels: Record<string, string> = {
  draft: "Projekt",
  confirmed: "Potwierdzone",
  planned: "Zaplanowane",
  in_progress: "W realizacji",
  paused: "Wstrzymane",
  done: "Zakończone",
  cancelled: "Anulowane",
};

const priorityColors: Record<string, string> = {
  low: "bg-gray-400",
  normal: "bg-blue-400",
  high: "bg-orange-400",
  urgent: "bg-red-500",
};

const priorityLabels: Record<string, string> = {
  low: "Niski",
  normal: "Normalny",
  high: "Wysoki",
  urgent: "Pilny",
};

const workflowStageColors: Record<string, string> = {
  magazine: "bg-gray-500",
  cutting: "bg-blue-500",
  edging: "bg-indigo-500",
  drilling: "bg-purple-500",
  upholstering: "bg-pink-500",
  picking: "bg-yellow-500",
  packing: "bg-orange-500",
  strapping: "bg-amber-500",
  ready: "bg-green-500",
  shipped: "bg-emerald-600",
};

const workflowStageLabels: Record<string, string> = {
  magazine: "Magazyn",
  cutting: "Cięcie",
  edging: "Oklejanie",
  drilling: "Wiercenie",
  upholstering: "Tapicerowanie",
  picking: "Kompletowanie",
  packing: "Pakowanie",
  strapping: "Bandowanie",
  ready: "Gotowe",
  shipped: "Wysłane",
};

const colorMap: Record<string, { hex: string; text: string }> = {
  WOTAN: { hex: "#972f0c", text: "#ffffff" },
  BIALY: { hex: "#ffffff", text: "#000000" },
  SUROWA: { hex: "#d9c9b0", text: "#000000" },
  CZARNY: { hex: "#000000", text: "#ffffff" },
  SONOMA: { hex: "#a67c52", text: "#ffffff" },
  ANTRACYT: { hex: "#202027", text: "#ffffff" },
  ARTISAN: { hex: "#d4b896", text: "#000000" },
  ESTANA: { hex: "#753a0a", text: "#ffffff" },
  KASZMIR: { hex: "#e8dfd2", text: "#000000" },
  LANCELOT: { hex: "#d2691e", text: "#ffffff" },
  OLEJOWANY: { hex: "#b89968", text: "#000000" },
  SCANDI: { hex: "#e5ddd3", text: "#000000" },
};

function extractColorFromOrderNumber(orderNumber: string): string | null {
  const parts = orderNumber.split('-');
  if (parts.length >= 3) {
    const colorCode = parts[parts.length - 1];
    return colorMap[colorCode] ? colorCode : null;
  }
  return null;
}

export default function ProductionOrdersPage() {
  const { toast } = useToast();
  const [location, setLocation] = useLocation();
  const [selectedOrderId, setSelectedOrderId] = useState<number | null>(null);
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingOrder, setEditingOrder] = useState<ProductionOrder | null>(null);
  
  // Parse URL params
  const urlParams = new URLSearchParams(window.location.search);
  
  // Filters state (initialized from URL)
  const [statusFilter, setStatusFilter] = useState<string>(urlParams.get("status") || "");
  const [workflowStageFilter, setWorkflowStageFilter] = useState<string>(urlParams.get("workflowStage") || "");
  const [priorityFilter, setPriorityFilter] = useState<string>(urlParams.get("priority") || "");
  const [searchFilter, setSearchFilter] = useState<string>(urlParams.get("search") || "");

  // Sync filters to URL
  useEffect(() => {
    const params = new URLSearchParams();
    if (statusFilter) params.set("status", statusFilter);
    if (workflowStageFilter) params.set("workflowStage", workflowStageFilter);
    if (priorityFilter) params.set("priority", priorityFilter);
    if (searchFilter) params.set("search", searchFilter);
    
    const newSearch = params.toString();
    const currentSearch = window.location.search.slice(1);
    
    if (newSearch !== currentSearch) {
      setLocation(`/production/orders${newSearch ? `?${newSearch}` : ""}`, { replace: true });
    }
  }, [statusFilter, workflowStageFilter, priorityFilter, searchFilter, setLocation]);

  // Build query params
  const queryParams = new URLSearchParams();
  if (statusFilter) queryParams.append("status", statusFilter);
  if (workflowStageFilter) queryParams.append("workflowStage", workflowStageFilter);
  if (priorityFilter) queryParams.append("priority", priorityFilter);
  if (searchFilter) queryParams.append("search", searchFilter);
  const queryString = queryParams.toString();

  // Fetch orders with filters
  const { data: orders = [], isLoading } = useQuery<ProductionOrder[]>({
    queryKey: ["/api/production/orders", queryString],
    queryFn: async () => {
      const url = queryString ? `/api/production/orders?${queryString}` : "/api/production/orders";
      const response = await fetch(url, { credentials: "include" });
      if (!response.ok) throw new Error("Failed to fetch orders");
      return response.json();
    },
  });

  // Form
  const form = useForm<OrderFormValues>({
    resolver: zodResolver(orderFormSchema),
    defaultValues: {
      orderNumber: "",
      productId: 0,
      quantityPlanned: 1,
      unitOfMeasure: "szt",
      status: "draft",
      priority: "normal",
    },
  });

  // Create/Update mutation
  const saveMutation = useMutation({
    mutationFn: async (data: OrderFormValues) => {
      if (editingOrder) {
        const response = await fetch(`/api/production/orders/${editingOrder.id}`, {
          method: "PATCH",
          headers: { "Content-Type": "application/json" },
          credentials: "include",
          body: JSON.stringify(data),
        });
        if (!response.ok) {
          const error = await response.json().catch(() => ({ message: "Failed to update order" }));
          throw new Error(error.message);
        }
        return await response.json();
      } else {
        const response = await fetch("/api/production/orders", {
          method: "POST",
          headers: { "Content-Type": "application/json" },
          credentials: "include",
          body: JSON.stringify(data),
        });
        if (!response.ok) {
          const error = await response.json().catch(() => ({ message: "Failed to create order" }));
          throw new Error(error.message);
        }
        return await response.json();
      }
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/orders"] });
      toast({
        title: "Sukces",
        description: editingOrder ? "Zlecenie zaktualizowane" : "Zlecenie utworzone",
      });
      setIsDialogOpen(false);
      setEditingOrder(null);
      form.reset();
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się zapisać zlecenia",
      });
    },
  });

  // Delete mutation
  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      const response = await fetch(`/api/production/orders/${id}`, {
        method: "DELETE",
        credentials: "include",
      });
      if (!response.ok) {
        const error = await response.json().catch(() => ({ message: "Failed to delete order" }));
        throw new Error(error.message);
      }
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/orders"] });
      toast({
        title: "Sukces",
        description: "Zlecenie produkcyjne zostało usunięte",
      });
      setSelectedOrderId(null);
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się usunąć zlecenia",
      });
    },
  });

  // Status transition mutations
  const startMutation = useMutation({
    mutationFn: async (id: number) => {
      const response = await fetch(`/api/production/orders/${id}/start`, {
        method: "POST",
        credentials: "include",
      });
      if (!response.ok) {
        const error = await response.json().catch(() => ({ message: "Failed to start order" }));
        throw new Error(error.message);
      }
      return await response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/orders"] });
      toast({
        title: "Sukces",
        description: "Zlecenie rozpoczęte",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się rozpocząć zlecenia",
      });
    },
  });

  const completeMutation = useMutation({
    mutationFn: async (id: number) => {
      const response = await fetch(`/api/production/orders/${id}/complete`, {
        method: "POST",
        credentials: "include",
      });
      if (!response.ok) {
        const error = await response.json().catch(() => ({ message: "Failed to complete order" }));
        throw new Error(error.message);
      }
      return await response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/orders"] });
      toast({
        title: "Sukces",
        description: "Zlecenie zakończone",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się zakończyć zlecenia",
      });
    },
  });

  const cancelMutation = useMutation({
    mutationFn: async (id: number) => {
      const response = await fetch(`/api/production/orders/${id}/cancel`, {
        method: "POST",
        credentials: "include",
      });
      if (!response.ok) {
        const error = await response.json().catch(() => ({ message: "Failed to cancel order" }));
        throw new Error(error.message);
      }
      return await response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/orders"] });
      toast({
        title: "Sukces",
        description: "Zlecenie anulowane",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się anulować zlecenia",
      });
    },
  });

  const selectedOrder = orders.find(o => o.id === selectedOrderId);

  const formatDate = (date: Date | null | undefined) => {
    if (!date) return "-";
    return format(new Date(date), "dd.MM.yyyy", { locale: pl });
  };

  const openCreateDialog = () => {
    setEditingOrder(null);
    form.reset({
      orderNumber: "",
      productId: 0,
      quantityPlanned: 1,
      unitOfMeasure: "szt",
      status: "draft",
      priority: "normal",
    });
    setIsDialogOpen(true);
  };

  const openEditDialog = (order: ProductionOrder) => {
    setEditingOrder(order);
    form.reset({
      orderNumber: order.orderNumber,
      productId: order.productId,
      bomId: order.bomId ?? undefined,
      routingId: order.routingId ?? undefined,
      status: order.status as any,
      priority: order.priority as any,
      quantityPlanned: Number(order.quantityPlanned),
      unitOfMeasure: order.unitOfMeasure,
      plannedStartDate: order.plannedStartDate ? format(new Date(order.plannedStartDate), "yyyy-MM-dd") : undefined,
      plannedEndDate: order.plannedEndDate ? format(new Date(order.plannedEndDate), "yyyy-MM-dd") : undefined,
      responsibleUserId: order.responsibleUserId ?? undefined,
      sourceOrderNumber: order.sourceOrderNumber ?? undefined,
      locationId: order.locationId ?? undefined,
      notes: order.notes ?? undefined,
    });
    setIsDialogOpen(true);
  };

  const handleSubmit = (data: OrderFormValues) => {
    saveMutation.mutate(data);
  };

  return (
    <div className="flex flex-col h-full">
      <div className="border-b p-4">
        <div className="flex items-center justify-between mb-4">
          <div>
            <h1 className="text-2xl font-bold" data-testid="text-title">Zlecenia Produkcyjne</h1>
            <p className="text-sm text-muted-foreground">Zarządzaj zleceniami produkcyjnymi (ZLP)</p>
          </div>
          <Button
            asChild
            data-testid="button-create-order"
          >
            <Link href="/production/orders/new">
              <Plus className="h-4 w-4 mr-2" />
              Nowe Zlecenie
            </Link>
          </Button>
        </div>

        {/* Filters */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-3">
          <div>
            <label className="text-sm font-medium mb-1 block">Status</label>
            <select
              value={statusFilter}
              onChange={(e) => setStatusFilter(e.target.value)}
              className="w-full h-9 px-3 rounded-md border bg-background text-sm"
              data-testid="select-status-filter"
            >
              <option value="">Wszystkie</option>
              {Object.entries(statusLabels).map(([value, label]) => (
                <option key={value} value={value}>{label}</option>
              ))}
            </select>
          </div>

          <div>
            <label className="text-sm font-medium mb-1 block">Etap Workflow</label>
            <select
              value={workflowStageFilter}
              onChange={(e) => setWorkflowStageFilter(e.target.value)}
              className="w-full h-9 px-3 rounded-md border bg-background text-sm"
              data-testid="select-workflow-filter"
            >
              <option value="">Wszystkie</option>
              {Object.entries(workflowStageLabels).map(([value, label]) => (
                <option key={value} value={value}>{label}</option>
              ))}
            </select>
          </div>

          <div>
            <label className="text-sm font-medium mb-1 block">Priorytet</label>
            <select
              value={priorityFilter}
              onChange={(e) => setPriorityFilter(e.target.value)}
              className="w-full h-9 px-3 rounded-md border bg-background text-sm"
              data-testid="select-priority-filter"
            >
              <option value="">Wszystkie</option>
              {Object.entries(priorityLabels).map(([value, label]) => (
                <option key={value} value={value}>{label}</option>
              ))}
            </select>
          </div>

          <div>
            <label className="text-sm font-medium mb-1 block">Szukaj</label>
            <input
              type="text"
              value={searchFilter}
              onChange={(e) => setSearchFilter(e.target.value)}
              placeholder="Numer zlecenia..."
              className="w-full h-9 px-3 rounded-md border bg-background text-sm"
              data-testid="input-search"
            />
          </div>
        </div>
      </div>

      <div className="flex-1 overflow-auto p-4">
        {isLoading ? (
          <div className="flex items-center justify-center h-full">
            <Loader2 className="h-8 w-8 animate-spin text-muted-foreground" />
          </div>
        ) : (
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
            {/* Orders List */}
            <Card>
              <CardHeader>
                <CardTitle>Lista Zleceń ({orders.length})</CardTitle>
              </CardHeader>
              <CardContent>
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Numer</TableHead>
                      <TableHead>Status</TableHead>
                      <TableHead>Etap</TableHead>
                      <TableHead>Priorytet</TableHead>
                      <TableHead>Ilość</TableHead>
                      <TableHead>Plan Start</TableHead>
                      <TableHead>Akcje</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {orders.length === 0 ? (
                      <TableRow>
                        <TableCell colSpan={7} className="text-center text-muted-foreground">
                          Brak zleceń produkcyjnych
                        </TableCell>
                      </TableRow>
                    ) : (
                      orders.map((order) => (
                        <TableRow
                          key={order.id}
                          data-testid={`row-order-${order.id}`}
                          className={selectedOrderId === order.id ? "bg-accent" : "hover-elevate cursor-pointer"}
                          onClick={() => setSelectedOrderId(order.id)}
                        >
                          <TableCell>
                            <Link href={`/production/orders/${order.id}`}>
                              <div className="flex items-center gap-2">
                                <span className="font-mono text-primary hover:underline cursor-pointer">
                                  {order.orderNumber}
                                </span>
                                {extractColorFromOrderNumber(order.orderNumber) && (
                                  <Badge 
                                    className="h-5 px-2 text-xs font-medium"
                                    style={{
                                      backgroundColor: colorMap[extractColorFromOrderNumber(order.orderNumber)!].hex,
                                      color: colorMap[extractColorFromOrderNumber(order.orderNumber)!].text,
                                      border: `1px solid ${colorMap[extractColorFromOrderNumber(order.orderNumber)!].hex === '#ffffff' ? '#e5e7eb' : 'transparent'}`
                                    }}
                                  >
                                    {extractColorFromOrderNumber(order.orderNumber)}
                                  </Badge>
                                )}
                              </div>
                            </Link>
                          </TableCell>
                          <TableCell>
                            <Badge className={statusColors[order.status]}>
                              {statusLabels[order.status]}
                            </Badge>
                          </TableCell>
                          <TableCell>
                            {order.workflowStage ? (
                              <Badge className={workflowStageColors[order.workflowStage]}>
                                {workflowStageLabels[order.workflowStage]}
                              </Badge>
                            ) : (
                              <span className="text-muted-foreground text-sm">-</span>
                            )}
                          </TableCell>
                          <TableCell>
                            <Badge variant="outline" className={priorityColors[order.priority]}>
                              {priorityLabels[order.priority]}
                            </Badge>
                          </TableCell>
                          <TableCell>
                            {order.quantityPlanned} {order.unitOfMeasure}
                          </TableCell>
                          <TableCell>{formatDate(order.plannedStartDate)}</TableCell>
                          <TableCell>
                            <div className="flex items-center gap-1">
                              <Button
                                size="icon"
                                variant="ghost"
                                onClick={(e) => {
                                  e.stopPropagation();
                                  openEditDialog(order);
                                }}
                                data-testid={`button-edit-order-${order.id}`}
                              >
                                <Edit className="h-4 w-4" />
                              </Button>
                              <Button
                                size="icon"
                                variant="ghost"
                                onClick={(e) => {
                                  e.stopPropagation();
                                  deleteMutation.mutate(order.id);
                                }}
                                data-testid={`button-delete-order-${order.id}`}
                                disabled={deleteMutation.isPending}
                              >
                                <Trash2 className="h-4 w-4" />
                              </Button>
                            </div>
                          </TableCell>
                        </TableRow>
                      ))
                    )}
                  </TableBody>
                </Table>
              </CardContent>
            </Card>

            {/* Order Details */}
            {selectedOrder ? (
              <Card>
                <CardHeader>
                  <div className="flex items-center justify-between">
                    <CardTitle>Szczegóły Zlecenia</CardTitle>
                    <div className="flex items-center gap-2">
                      {selectedOrder.status === 'draft' || selectedOrder.status === 'confirmed' || selectedOrder.status === 'planned' ? (
                        <Button
                          size="sm"
                          onClick={() => startMutation.mutate(selectedOrder.id)}
                          disabled={startMutation.isPending}
                          data-testid="button-start-order"
                        >
                          <Play className="h-4 w-4 mr-2" />
                          Rozpocznij
                        </Button>
                      ) : null}
                      {selectedOrder.status === 'in_progress' ? (
                        <Button
                          size="sm"
                          onClick={() => completeMutation.mutate(selectedOrder.id)}
                          disabled={completeMutation.isPending}
                          data-testid="button-complete-order"
                        >
                          <CheckCircle className="h-4 w-4 mr-2" />
                          Zakończ
                        </Button>
                      ) : null}
                      {selectedOrder.status !== 'done' && selectedOrder.status !== 'cancelled' ? (
                        <Button
                          size="sm"
                          variant="destructive"
                          onClick={() => cancelMutation.mutate(selectedOrder.id)}
                          disabled={cancelMutation.isPending}
                          data-testid="button-cancel-order"
                        >
                          <XCircle className="h-4 w-4 mr-2" />
                          Anuluj
                        </Button>
                      ) : null}
                    </div>
                  </div>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <div>
                      <p className="text-sm text-muted-foreground">Numer Zlecenia</p>
                      <p className="font-mono font-semibold">{selectedOrder.orderNumber}</p>
                    </div>

                    <div className="grid grid-cols-3 gap-4">
                      <div>
                        <p className="text-sm text-muted-foreground">Status</p>
                        <Badge className={statusColors[selectedOrder.status]}>
                          {statusLabels[selectedOrder.status]}
                        </Badge>
                      </div>
                      <div>
                        <p className="text-sm text-muted-foreground">Etap Workflow</p>
                        {selectedOrder.workflowStage ? (
                          <Badge className={workflowStageColors[selectedOrder.workflowStage]}>
                            {workflowStageLabels[selectedOrder.workflowStage]}
                          </Badge>
                        ) : (
                          <span className="text-sm text-muted-foreground">-</span>
                        )}
                      </div>
                      <div>
                        <p className="text-sm text-muted-foreground">Priorytet</p>
                        <Badge variant="outline" className={priorityColors[selectedOrder.priority]}>
                          {priorityLabels[selectedOrder.priority]}
                        </Badge>
                      </div>
                    </div>

                    <div className="grid grid-cols-2 gap-4">
                      <div>
                        <p className="text-sm text-muted-foreground">Ilość Planowana</p>
                        <p className="font-semibold">{selectedOrder.quantityPlanned} {selectedOrder.unitOfMeasure}</p>
                      </div>
                      <div>
                        <p className="text-sm text-muted-foreground">Ilość Wyprodukowana</p>
                        <p className="font-semibold">{selectedOrder.quantityProduced ?? 0} {selectedOrder.unitOfMeasure}</p>
                      </div>
                    </div>

                    <div className="grid grid-cols-2 gap-4">
                      <div>
                        <p className="text-sm text-muted-foreground">Plan Start</p>
                        <p>{formatDate(selectedOrder.plannedStartDate)}</p>
                      </div>
                      <div>
                        <p className="text-sm text-muted-foreground">Plan Koniec</p>
                        <p>{formatDate(selectedOrder.plannedEndDate)}</p>
                      </div>
                    </div>

                    {(selectedOrder.actualStartDate || selectedOrder.actualEndDate) && (
                      <div className="grid grid-cols-2 gap-4">
                        <div>
                          <p className="text-sm text-muted-foreground">Rzeczywisty Start</p>
                          <p>{formatDate(selectedOrder.actualStartDate)}</p>
                        </div>
                        <div>
                          <p className="text-sm text-muted-foreground">Rzeczywisty Koniec</p>
                          <p>{formatDate(selectedOrder.actualEndDate)}</p>
                        </div>
                      </div>
                    )}

                    {selectedOrder.notes && (
                      <div>
                        <p className="text-sm text-muted-foreground">Uwagi</p>
                        <p className="text-sm">{selectedOrder.notes}</p>
                      </div>
                    )}

                    <div className="pt-4">
                      <Button
                        variant="outline"
                        className="w-full"
                        asChild
                      >
                        <Link href={`/production/orders/${selectedOrder.id}`}>
                          <Package className="h-4 w-4 mr-2" />
                          Zobacz pełne szczegóły
                        </Link>
                      </Button>
                    </div>
                  </div>
                </CardContent>
              </Card>
            ) : (
              <Card>
                <CardContent className="flex items-center justify-center h-full min-h-[400px]">
                  <p className="text-muted-foreground">Wybierz zlecenie aby zobaczyć szczegóły</p>
                </CardContent>
              </Card>
            )}
          </div>
        )}
      </div>

      {/* Create/Edit Dialog */}
      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>{editingOrder ? "Edytuj Zlecenie" : "Nowe Zlecenie Produkcyjne"}</DialogTitle>
            <DialogDescription>
              {editingOrder ? "Zaktualizuj dane zlecenia produkcyjnego" : "Utwórz nowe zlecenie produkcyjne"}
            </DialogDescription>
          </DialogHeader>

          <ProductionOrderForm
            form={form}
            onSubmit={handleSubmit}
            onCancel={() => {
              setIsDialogOpen(false);
              setEditingOrder(null);
              form.reset();
            }}
            isLoading={saveMutation.isPending}
            mode={editingOrder ? 'edit' : 'create'}
          />
        </DialogContent>
      </Dialog>
    </div>
  );
}
