import { useState, useRef, useEffect, useMemo } from "react";
import { useRoute, Link } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Plus, Edit2, Trash2, Loader2, Upload, FileDown, X, Image as ImageIcon, Search, ChevronDown, ChevronUp } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import { Form, FormControl, FormDescription, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Switch } from "@/components/ui/switch";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useToast } from "@/hooks/use-toast";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { apiRequest, queryClient } from "@/lib/queryClient";
import type { ProductCreatorDictionary, DictionaryType } from "@shared/schema";

const dictionaryFormSchema = z.object({
  dictionaryType: z.string().min(1, "Typ słownika jest wymagany"),
  code: z.string().min(1, "Kod jest wymagany"),
  name: z.string().min(1, "Nazwa jest wymagana"),
  readableName: z.string().optional(),
  description: z.string().optional(),
  color: z.string().regex(/^#[0-9A-Fa-f]{6}$/, "Nieprawidłowy format koloru HEX").optional().or(z.literal("")),
  category: z.string().optional(),
  sortOrder: z.coerce.number().default(0),
  isActive: z.boolean().default(true),
});

type DictionaryFormData = z.infer<typeof dictionaryFormSchema>;

const DICTIONARY_LABELS: Record<DictionaryType, { label: string; description: string }> = {
  product_type: { label: "Rodzaj produktu", description: "SUPRA, TRES, VB, BINI" },
  element_type: { label: "Rodzaj elementu", description: "BOCZKI, BOK-L, BOK-P, POLKA, DRZWI" },
  component_cz1: { label: "Komponenty cz1", description: "BOK-L, BOK-P, POLKA, DRZWI, WD, WG, HDF, SIEDZISKO" },
  component_cz2: { label: "Komponenty cz2", description: "VB, VB-D2, VB-D3, ALTUS, FORM, SUPRA, TRES, NEX, BONES" },
  unit: { label: "Jednostka miary", description: "szt, m, m², m³, kg, l, mb, t, opak" },
  color: { label: "Kolor", description: "BIAŁY, KASZMIR, WOTAN" },
  material: { label: "Materiał", description: "płyta meblowa, szkło, metal" },
  product_group: { label: "Grupa produktów", description: "A1, A2, A3, A4" },
  door: { label: "Drzwi", description: "D1, D2, D3, D4" },
  leg: { label: "Nóżki", description: "N1, N2, N3, N4" },
  fabric: { label: "Tkaniny", description: "ES1, ES2" },
  dimension_length: { label: "Wymiar - długość", description: "50-100 cm" },
  dimension_width: { label: "Wymiar - szerokość", description: "30-36 cm" },
  dimension_height: { label: "Wymiar - wysokość", description: "45/82/205 cm" },
  scrap_reason_product: { label: "Powody złomowania produktów", description: "Uszkodzenie w transporcie, błąd produkcji" },
  scrap_reason_cutting: { label: "Powody złomowania formatek", description: "Błąd cięcia, uszkodzenie, wada płyty" },
};

const STORAGE_KEY_PREFIX = 'dictionary-filters-';

const generateCodeFromName = (name: string): string => {
  const polishCharsMap: Record<string, string> = {
    'ą': 'a', 'ć': 'c', 'ę': 'e', 'ł': 'l', 'ń': 'n',
    'ó': 'o', 'ś': 's', 'ź': 'z', 'ż': 'z',
    'Ą': 'a', 'Ć': 'c', 'Ę': 'e', 'Ł': 'l', 'Ń': 'n',
    'Ó': 'o', 'Ś': 's', 'Ź': 'z', 'Ż': 'z'
  };
  
  return name
    .split('')
    .map(char => polishCharsMap[char] || char)
    .join('')
    .toLowerCase()
    .replace(/\s+/g, '_')
    .replace(/[^a-z0-9_]/g, '')
    .replace(/_+/g, '_')
    .replace(/^_|_$/g, '');
};

export default function DictionaryTypePage() {
  const [, params] = useRoute("/dictionaries/:type");
  const dictionaryType = params?.type as DictionaryType;
  const { toast } = useToast();
  
  // Load filters from localStorage
  const loadFilters = () => {
    if (!dictionaryType) return {};
    try {
      const saved = localStorage.getItem(STORAGE_KEY_PREFIX + dictionaryType);
      if (saved) {
        return JSON.parse(saved);
      }
    } catch (error) {
      console.error('Error loading filters from localStorage:', error);
    }
    return {};
  };

  const savedFilters = loadFilters();
  
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingEntry, setEditingEntry] = useState<ProductCreatorDictionary | null>(null);
  const [isImportDialogOpen, setIsImportDialogOpen] = useState(false);
  const [csvData, setCsvData] = useState("");
  const [uploadingImages, setUploadingImages] = useState(false);
  const [isDraggingFiles, setIsDraggingFiles] = useState(false);
  const fileInputRef = useRef<HTMLInputElement>(null);
  const [codeManuallyEdited, setCodeManuallyEdited] = useState(false);
  const [codeError, setCodeError] = useState<string | null>(null);
  
  // Filter states
  const [searchQuery, setSearchQuery] = useState(savedFilters.searchQuery || "");
  const [filterCategory, setFilterCategory] = useState<string>(savedFilters.filterCategory || "");
  const [filterIsActive, setFilterIsActive] = useState<string>(savedFilters.filterIsActive || "all");
  const [filtersExpanded, setFiltersExpanded] = useState(false);
  
  // Reload filters when dictionary type changes
  useEffect(() => {
    if (!dictionaryType) return;
    
    try {
      const saved = localStorage.getItem(STORAGE_KEY_PREFIX + dictionaryType);
      if (saved) {
        const filters = JSON.parse(saved);
        setSearchQuery(filters.searchQuery || "");
        setFilterCategory(filters.filterCategory || "");
        setFilterIsActive(filters.filterIsActive || "all");
      } else {
        // Reset to defaults if no saved filters for this type
        setSearchQuery("");
        setFilterCategory("");
        setFilterIsActive("all");
      }
      // Collapse filters panel when switching types
      setFiltersExpanded(false);
    } catch (error) {
      console.error('Error loading filters from localStorage:', error);
    }
  }, [dictionaryType]);
  
  // Save filters to localStorage whenever they change
  useEffect(() => {
    if (!dictionaryType) return;
    const filters = {
      searchQuery,
      filterCategory,
      filterIsActive,
    };
    try {
      localStorage.setItem(STORAGE_KEY_PREFIX + dictionaryType, JSON.stringify(filters));
    } catch (error) {
      console.error('Error saving filters to localStorage:', error);
    }
  }, [dictionaryType, searchQuery, filterCategory, filterIsActive]);

  // Fetch dictionaries for this type
  const { data: dictionaries = [], isLoading } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries", dictionaryType],
    queryFn: async () => {
      if (!dictionaryType) return [];
      const response = await fetch(`/api/dictionaries?type=${dictionaryType}`, {
        credentials: 'include'
      });
      if (!response.ok) throw new Error('Failed to fetch dictionaries');
      return response.json();
    },
    enabled: !!dictionaryType,
  });
  
  // Split search query into phrases (by semicolon)
  const getSearchPhrases = () => {
    const phrases: string[] = [];
    searchQuery.split(/[;]/).forEach((phrase: string) => {
      const trimmed = phrase.trim();
      if (trimmed) phrases.push(trimmed);
    });
    return phrases;
  };

  const searchPhrases = getSearchPhrases();

  // Remove a specific search phrase
  const removeSearchPhrase = (phraseToRemove: string) => {
    const updatedPhrases = searchPhrases.filter(p => p !== phraseToRemove);
    setSearchQuery(updatedPhrases.join('; '));
  };
  
  // Filter dictionaries
  const filteredDictionaries = useMemo(() => {
    return dictionaries.filter(entry => {
      // Search filter - supports AND logic with semicolon separator
      if (searchQuery) {
        const terms = searchQuery.split(';').map((t: string) => t.trim()).filter(Boolean);
        const matchesAllTerms = terms.every((term: string) => {
          const lowerTerm = term.toLowerCase();
          const matchesCode = entry.code.toLowerCase().includes(lowerTerm);
          const matchesName = entry.name.toLowerCase().includes(lowerTerm);
          const matchesShort = entry.shortName?.toLowerCase().includes(lowerTerm);
          const matchesReadable = entry.readableName?.toLowerCase().includes(lowerTerm);
          const matchesCategory = entry.category?.toLowerCase().includes(lowerTerm);
          return matchesCode || matchesName || matchesShort || matchesReadable || matchesCategory;
        });
        if (!matchesAllTerms) {
          return false;
        }
      }
      
      // Category filter
      if (filterCategory && entry.category !== filterCategory) {
        return false;
      }
      
      // Active status filter
      if (filterIsActive === "active" && !entry.isActive) {
        return false;
      }
      if (filterIsActive === "inactive" && entry.isActive) {
        return false;
      }
      
      return true;
    });
  }, [dictionaries, searchQuery, filterCategory, filterIsActive]);
  
  // Get unique categories for filter
  const uniqueCategories = useMemo(() => {
    const categories = new Set<string>();
    dictionaries.forEach(entry => {
      if (entry.category) categories.add(entry.category);
    });
    return Array.from(categories).sort();
  }, [dictionaries]);
  
  // Count active filters
  const activeFiltersCount = useMemo(() => {
    let count = 0;
    count += searchPhrases.length;
    if (filterCategory) count++;
    if (filterIsActive !== "all") count++;
    return count;
  }, [searchPhrases, filterCategory, filterIsActive]);
  
  // Clear all filters
  const clearAllFilters = () => {
    setSearchQuery("");
    setFilterCategory("");
    setFilterIsActive("all");
  };

  // Form
  const form = useForm<DictionaryFormData>({
    resolver: zodResolver(dictionaryFormSchema),
    defaultValues: {
      dictionaryType: dictionaryType || "",
      code: "",
      name: "",
      readableName: "",
      description: "",
      color: "",
      category: "",
      sortOrder: 0,
      isActive: true,
    },
  });

  // Create mutation
  const createMutation = useMutation({
    mutationFn: async (data: DictionaryFormData) => {
      const response = await apiRequest("POST", "/api/dictionaries", {
        dictionaryType: data.dictionaryType,
        code: data.code.toLowerCase(),
        name: data.name.toLowerCase(),
        readableName: data.readableName || null,
        description: data.description || null,
        color: data.color || null,
        category: data.category || null,
        sortOrder: data.sortOrder,
        isActive: data.isActive,
      });
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/dictionaries"] });
      toast({ title: "Pozycja słownika utworzona pomyślnie" });
      setIsDialogOpen(false);
      form.reset({ dictionaryType });
    },
    onError: (error: Error) => {
      toast({ 
        title: "Błąd tworzenia pozycji", 
        description: error.message,
        variant: "destructive" 
      });
    },
  });

  // Update mutation
  const updateMutation = useMutation({
    mutationFn: async ({ id, data }: { id: number; data: Partial<DictionaryFormData> }) => {
      const response = await apiRequest("PATCH", `/api/dictionaries/${id}`, {
        ...data,
        code: data.code ? data.code.toLowerCase() : undefined,
        name: data.name ? data.name.toLowerCase() : undefined,
      });
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/dictionaries"] });
      toast({ title: "Pozycja zaktualizowana" });
      setIsDialogOpen(false);
      setEditingEntry(null);
      form.reset({ dictionaryType });
    },
    onError: (error: Error) => {
      toast({ 
        title: "Błąd aktualizacji pozycji", 
        description: error.message,
        variant: "destructive" 
      });
    },
  });

  // Delete mutation
  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      return await apiRequest("DELETE", `/api/dictionaries/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/dictionaries"] });
      toast({ title: "Pozycja usunięta" });
    },
    onError: () => {
      toast({ title: "Błąd usuwania pozycji", variant: "destructive" });
    },
  });

  // Images query
  const { data: dictionaryImages = [] } = useQuery<any[]>({
    queryKey: editingEntry ? [`/api/dictionaries/${editingEntry.id}/images`] : [],
    enabled: !!editingEntry,
  });

  // Delete image mutation
  const deleteImageMutation = useMutation({
    mutationFn: async ({ dictionaryId, imageId }: { dictionaryId: number; imageId: number }) => {
      return await apiRequest("DELETE", `/api/dictionaries/${dictionaryId}/images/${imageId}`);
    },
    onSuccess: () => {
      if (editingEntry) {
        queryClient.invalidateQueries({ queryKey: [`/api/dictionaries/${editingEntry.id}/images`] });
      }
      toast({ title: "Obraz usunięty" });
    },
    onError: () => {
      toast({ title: "Błąd usuwania obrazu", variant: "destructive" });
    },
  });

  // CSV Import mutation
  const importCsvMutation = useMutation({
    mutationFn: async (data: { type: string; csvData: string }) => {
      const response = await apiRequest("POST", "/api/dictionaries/import", data);
      return response.json();
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({ queryKey: ["/api/dictionaries"] });
      toast({ 
        title: "Import zakończony", 
        description: `Zaimportowano: ${data.imported}, Pominięto: ${data.skipped}, Błędy: ${data.errors}`
      });
      setIsImportDialogOpen(false);
      setCsvData("");
    },
    onError: (error: Error) => {
      toast({ title: "Błąd importu", description: error.message, variant: "destructive" });
    },
  });

  // Export CSV
  const handleExportCsv = () => {
    const csv = [
      ["Kod", "Nazwa", "Nazwa czytelna", "Opis", "Kolor HEX", "Kategoria", "Kolejność", "Aktywny"].join(","),
      ...dictionaries.map(d => [
        d.code,
        d.name,
        d.readableName || "",
        d.description || "",
        d.color || "",
        d.category || "",
        d.sortOrder,
        d.isActive ? "TAK" : "NIE"
      ].join(","))
    ].join("\n");
    
    const blob = new Blob([csv], { type: "text/csv;charset=utf-8;" });
    const link = document.createElement("a");
    link.href = URL.createObjectURL(blob);
    link.download = `slownik_${dictionaryType}_${Date.now()}.csv`;
    link.click();
  };

  // Handle file upload for images
  const handleFileUpload = async (files: FileList | null, dictionaryId: number) => {
    if (!files || files.length === 0) return;

    setUploadingImages(true);
    const formData = new FormData();
    
    Array.from(files).forEach(file => {
      formData.append("images", file);
    });

    try {
      const response = await fetch(`/api/dictionaries/${dictionaryId}/images`, {
        method: "POST",
        body: formData,
        credentials: "include",
      });

      if (!response.ok) throw new Error("Upload failed");
      
      queryClient.invalidateQueries({ queryKey: [`/api/dictionaries/${dictionaryId}/images`] });
      toast({ title: "Obrazy przesłane pomyślnie" });
    } catch (error) {
      toast({ title: "Błąd przesyłania obrazów", variant: "destructive" });
    } finally {
      setUploadingImages(false);
    }
  };

  const handleOpenDialog = (entry?: ProductCreatorDictionary) => {
    if (entry) {
      setEditingEntry(entry);
      setCodeManuallyEdited(true);
      form.reset({
        dictionaryType: entry.dictionaryType,
        code: entry.code,
        name: entry.name,
        readableName: entry.readableName || "",
        description: entry.description || "",
        color: entry.color || "",
        category: entry.category || "",
        sortOrder: entry.sortOrder ?? 0,
        isActive: entry.isActive ?? true,
      });
    } else {
      setEditingEntry(null);
      setCodeManuallyEdited(false);
      form.reset({ dictionaryType });
    }
    setCodeError(null);
    setIsDialogOpen(true);
  };
  
  const checkCodeUniqueness = (code: string): boolean => {
    if (!code) return true;
    const existingCodes = dictionaries
      .filter(d => !editingEntry || d.id !== editingEntry.id)
      .map(d => d.code.toLowerCase());
    return !existingCodes.includes(code.toLowerCase());
  };
  
  const handleReadableNameChange = (value: string) => {
    form.setValue('readableName', value);
    if (!codeManuallyEdited && !editingEntry) {
      const generatedCode = generateCodeFromName(value);
      form.setValue('code', generatedCode);
      form.setValue('name', generatedCode);
      if (generatedCode && !checkCodeUniqueness(generatedCode)) {
        setCodeError('Ten kod już istnieje');
      } else {
        setCodeError(null);
      }
    }
  };
  
  const handleCodeChange = (value: string) => {
    setCodeManuallyEdited(true);
    form.setValue('code', value);
    form.setValue('name', value);
    if (value && !checkCodeUniqueness(value)) {
      setCodeError('Ten kod już istnieje');
    } else {
      setCodeError(null);
    }
  };

  const handleCloseDialog = (open?: boolean) => {
    if (open === false || open === undefined) {
      setIsDialogOpen(false);
      setEditingEntry(null);
      setCodeManuallyEdited(false);
      setCodeError(null);
      form.reset({ dictionaryType });
    }
  };

  const handleSubmit = (data: DictionaryFormData) => {
    if (codeError) {
      toast({
        title: "Błąd walidacji",
        description: "Kod musi być unikalny",
        variant: "destructive"
      });
      return;
    }
    if (editingEntry) {
      updateMutation.mutate({ id: editingEntry.id, data });
    } else {
      createMutation.mutate(data);
    }
  };

  if (!dictionaryType) {
    return (
      <div className="p-8">
        <Card>
          <CardHeader>
            <CardTitle>Wybierz typ słownika</CardTitle>
            <CardDescription>Wybierz typ słownika z menu po lewej stronie</CardDescription>
          </CardHeader>
        </Card>
      </div>
    );
  }

  const typeInfo = DICTIONARY_LABELS[dictionaryType];

  return (
    <div className="p-2 sm:p-4 md:p-8 space-y-4 md:space-y-6">
      <div className="flex flex-col gap-2">
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-3">
          <div className="min-w-0 flex-1">
            <div className="flex flex-wrap items-center gap-2">
              <h1 className="text-2xl md:text-3xl font-bold" data-testid="text-dictionary-title">{typeInfo.label}</h1>
              <Badge variant="outline" className="h-6 text-xs">
                {filteredDictionaries.length} {filteredDictionaries.length === 1 ? 'pozycja' : 'pozycji'}
              </Badge>
              {/* Active Filters Badges */}
              {activeFiltersCount > 0 && (
                <>
                  {/* Search phrases badges - each phrase as separate badge */}
                  {searchPhrases.map((phrase, index) => (
                    <Badge 
                      key={`search-${index}`}
                      variant="default" 
                      className="gap-1.5 pr-1 h-6"
                      data-testid={`badge-search-phrase-${index}`}
                    >
                      <Search className="h-3 w-3" />
                      {phrase}
                      <Button
                        variant="ghost"
                        size="sm"
                        className="h-4 w-4 p-0 hover:bg-primary-foreground/20"
                        onClick={() => removeSearchPhrase(phrase)}
                        data-testid={`button-remove-search-${index}`}
                      >
                        <X className="h-3 w-3" />
                      </Button>
                    </Badge>
                  ))}
                  {filterCategory && (
                    <Badge variant="default" className="gap-1.5 pr-1 h-6">
                      Kategoria: {filterCategory}
                      <Button
                        variant="ghost"
                        size="sm"
                        className="h-4 w-4 p-0 hover:bg-primary-foreground/20"
                        onClick={() => setFilterCategory("")}
                      >
                        <X className="h-3 w-3" />
                      </Button>
                    </Badge>
                  )}
                  {filterIsActive !== "all" && (
                    <Badge variant="default" className="gap-1.5 pr-1 h-6">
                      Status: {filterIsActive === "active" ? "Aktywne" : "Nieaktywne"}
                      <Button
                        variant="ghost"
                        size="sm"
                        className="h-4 w-4 p-0 hover:bg-primary-foreground/20"
                        onClick={() => setFilterIsActive("all")}
                      >
                        <X className="h-3 w-3" />
                      </Button>
                    </Badge>
                  )}
                </>
              )}
            </div>
            <p className="text-muted-foreground mt-1 text-sm">{typeInfo.description}</p>
          </div>
          <div className="flex gap-2 shrink-0">
            <Button 
              variant="outline" 
              size="sm"
              onClick={() => setIsImportDialogOpen(true)}
              data-testid="button-import-csv"
            >
              <Upload className="h-4 w-4 md:mr-2" />
              <span className="hidden md:inline">Import</span>
            </Button>
            <Button 
              variant="outline" 
              size="sm"
              onClick={handleExportCsv}
              disabled={dictionaries.length === 0}
              data-testid="button-export-csv"
            >
              <FileDown className="h-4 w-4 md:mr-2" />
              <span className="hidden md:inline">Export</span>
            </Button>
            <Button 
              size="sm"
              onClick={() => handleOpenDialog()} 
              data-testid="button-add-entry"
            >
              <Plus className="h-4 w-4 md:mr-2" />
              <span className="hidden md:inline">Dodaj</span>
            </Button>
          </div>
        </div>
      </div>

      {/* Search Bar */}
      <Card className="bg-primary/5 border-primary/20">
        <CardContent className="py-3 md:py-4">
          <div className="flex items-center gap-2">
            <Search className="h-4 w-4 text-primary shrink-0" />
            {searchQuery && (
              <Button
                variant="ghost"
                size="icon"
                className="h-6 w-6 hover:bg-primary/10 shrink-0"
                onClick={() => setSearchQuery("")}
                data-testid="button-clear-search"
              >
                <X className="h-3.5 w-3.5 text-muted-foreground" />
              </Button>
            )}
            <Input
              placeholder="Szukaj (użyj ; aby połączyć filtry np. sup;30)"
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              className="h-9 md:h-10 text-xs md:text-sm bg-background border-primary/30 focus-visible:ring-primary px-3"
              data-testid="input-search-dictionary"
            />
          </div>
        </CardContent>
      </Card>

      {/* Collapsible Filters Panel */}
      <Card>
        <CardHeader className="cursor-pointer py-1 px-3" onClick={() => setFiltersExpanded(!filtersExpanded)}>
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-1.5">
              <CardTitle className="text-xs font-medium">Filtry</CardTitle>
              {activeFiltersCount > 0 && (
                <Badge variant="default" className="rounded-full h-3.5 min-w-3.5 px-1 text-[9px]">{activeFiltersCount}</Badge>
              )}
            </div>
            <div className="flex items-center gap-1.5">
              {activeFiltersCount > 0 && (
                <Button
                  variant="ghost"
                  size="sm"
                  onClick={(e) => {
                    e.stopPropagation();
                    clearAllFilters();
                  }}
                  className="h-5 text-[9px] px-1.5"
                  data-testid="button-clear-all-filters"
                >
                  Wyczyść
                </Button>
              )}
              {filtersExpanded ? <ChevronUp className="h-3 w-3" /> : <ChevronDown className="h-3 w-3" />}
            </div>
          </div>
        </CardHeader>
        {filtersExpanded && (
          <CardContent className="pt-0 pb-2 grid grid-cols-1 md:grid-cols-2 gap-2">
            <div className="space-y-0.5">
              <Label className="text-[10px]">Kategoria</Label>
              <Select value={filterCategory || "all"} onValueChange={(val) => setFilterCategory(val === "all" ? "" : val)}>
                <SelectTrigger data-testid="select-filter-category" className="h-7 text-xs">
                  <SelectValue placeholder="Wszystkie" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Wszystkie</SelectItem>
                  {uniqueCategories.map((category) => (
                    <SelectItem key={category} value={category}>
                      {category}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-0.5">
              <Label className="text-[10px]">Status</Label>
              <Select value={filterIsActive} onValueChange={setFilterIsActive}>
                <SelectTrigger data-testid="select-filter-active" className="h-7 text-xs">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Wszystkie</SelectItem>
                  <SelectItem value="active">Aktywne</SelectItem>
                  <SelectItem value="inactive">Nieaktywne</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </CardContent>
        )}
      </Card>

      {isLoading ? (
        <div className="flex justify-center p-8">
          <Loader2 className="h-8 w-8 animate-spin" />
        </div>
      ) : (
        <Card>
          <CardContent className="p-0">
            <div className="overflow-x-auto">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Kod</TableHead>
                    <TableHead>Nazwa</TableHead>
                    <TableHead className="hidden lg:table-cell">Nazwa czytelna</TableHead>
                    <TableHead className="hidden md:table-cell">Kolor</TableHead>
                    <TableHead className="hidden xl:table-cell">Kategoria</TableHead>
                    <TableHead className="hidden xl:table-cell">Kolejność</TableHead>
                    <TableHead className="hidden sm:table-cell">Status</TableHead>
                    <TableHead className="text-right">Akcje</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {filteredDictionaries.length === 0 ? (
                    <TableRow>
                      <TableCell colSpan={8} className="text-center text-muted-foreground">
                        {searchQuery || filterCategory || filterIsActive !== "all" 
                          ? "Brak pozycji spełniających kryteria filtrowania" 
                          : "Brak pozycji w tym słowniku"}
                      </TableCell>
                    </TableRow>
                  ) : (
                    filteredDictionaries.map((entry) => (
                      <TableRow 
                        key={entry.id} 
                        data-testid={`row-dictionary-${entry.id}`}
                        className="cursor-pointer hover-elevate"
                        onClick={() => handleOpenDialog(entry)}
                      >
                        <TableCell className="font-mono text-sm">{entry.code}</TableCell>
                        <TableCell className="font-medium">
                          <div className="flex flex-col">
                            <span>{entry.name}</span>
                            {entry.readableName && (
                              <span className="text-xs text-muted-foreground lg:hidden">{entry.readableName}</span>
                            )}
                          </div>
                        </TableCell>
                        <TableCell className="hidden lg:table-cell">{entry.readableName || "-"}</TableCell>
                        <TableCell className="hidden md:table-cell">
                          {entry.color ? (
                            <div className="flex items-center gap-2">
                              <div 
                                className="w-6 h-6 rounded border" 
                                style={{ backgroundColor: entry.color }}
                                title={entry.color}
                              />
                              <span className="text-sm font-mono hidden lg:inline">{entry.color}</span>
                            </div>
                          ) : (
                            <span className="text-muted-foreground">-</span>
                          )}
                        </TableCell>
                        <TableCell className="hidden xl:table-cell">{entry.category || "-"}</TableCell>
                        <TableCell className="hidden xl:table-cell">{entry.sortOrder}</TableCell>
                        <TableCell className="hidden sm:table-cell">
                          <Badge variant={entry.isActive ? "default" : "secondary"} className="text-xs">
                            {entry.isActive ? "Aktywny" : "Nieaktywny"}
                          </Badge>
                        </TableCell>
                        <TableCell className="text-right" onClick={(e) => e.stopPropagation()}>
                          <div className="flex justify-end gap-1">
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={() => handleOpenDialog(entry)}
                              data-testid={`button-edit-${entry.id}`}
                            >
                              <Edit2 className="h-4 w-4" />
                            </Button>
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={() => {
                                if (confirm("Czy na pewno chcesz usunąć tę pozycję?")) {
                                  deleteMutation.mutate(entry.id);
                                }
                              }}
                              data-testid={`button-delete-${entry.id}`}
                            >
                              <Trash2 className="h-4 w-4" />
                            </Button>
                          </div>
                        </TableCell>
                      </TableRow>
                    ))
                  )}
                </TableBody>
              </Table>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Add/Edit Dialog */}
      <Dialog open={isDialogOpen} onOpenChange={(open) => { if (!open) handleCloseDialog(); }}>
        <DialogContent 
          className="max-w-[95vw] sm:max-w-2xl max-h-[90vh] overflow-y-auto"
          onEscapeKeyDown={(e) => e.preventDefault()}
          onPointerDownOutside={(e) => e.preventDefault()}
        >
          <DialogHeader>
            <DialogTitle>
              {editingEntry ? "Edytuj pozycję" : "Dodaj pozycję"}
            </DialogTitle>
            <DialogDescription>
              {editingEntry ? "Edytuj pola poniżej" : "Wypełnij pola poniżej"}
            </DialogDescription>
          </DialogHeader>

          <Form {...form}>
            <form onSubmit={form.handleSubmit(handleSubmit)} className="space-y-4">
              <FormField
                control={form.control}
                name="readableName"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Nazwa czytelna *</FormLabel>
                    <FormControl>
                      <Input 
                        {...field} 
                        placeholder="np. Uszkodzenie w transporcie" 
                        data-testid="input-readable-name"
                        onChange={(e) => handleReadableNameChange(e.target.value)}
                      />
                    </FormControl>
                    <FormDescription>
                      Kod zostanie wygenerowany automatycznie z tej nazwy
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="code"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Kod *</FormLabel>
                      <FormControl>
                        <Input 
                          {...field} 
                          placeholder="np. uszkodzenie_transport" 
                          data-testid="input-code"
                          onChange={(e) => handleCodeChange(e.target.value)}
                        />
                      </FormControl>
                      {codeError && (
                        <p className="text-sm text-destructive">{codeError}</p>
                      )}
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="name"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Nazwa (techniczna)</FormLabel>
                      <FormControl>
                        <Input {...field} placeholder="np. uszkodzenie_transport" data-testid="input-name" disabled className="bg-muted" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={form.control}
                name="description"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Opis</FormLabel>
                    <FormControl>
                      <Textarea {...field} rows={3} data-testid="input-description" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="color"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Kolor (opcjonalny)</FormLabel>
                    <FormControl>
                      <div className="flex gap-2">
                        <Input 
                          {...field} 
                          placeholder="#6366F1" 
                          data-testid="input-color"
                          className="flex-1"
                        />
                        <div className="flex items-center gap-2">
                          <input
                            type="color"
                            value={field.value || "#6366F1"}
                            onChange={(e) => field.onChange(e.target.value)}
                            className="h-10 w-16 rounded border cursor-pointer"
                            data-testid="input-color-picker"
                          />
                          {field.value && (
                            <div 
                              className="h-10 w-10 rounded border flex items-center justify-center"
                              style={{ backgroundColor: field.value }}
                              title={field.value}
                            />
                          )}
                        </div>
                      </div>
                    </FormControl>
                    <FormDescription>
                      Kolor tła dla etykiet (np. #6366F1)
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="category"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Kategoria</FormLabel>
                      <FormControl>
                        <Input {...field} data-testid="input-category" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="sortOrder"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Kolejność</FormLabel>
                      <FormControl>
                        <Input type="number" {...field} data-testid="input-sort-order" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={form.control}
                name="isActive"
                render={({ field }) => (
                  <FormItem className="flex items-center justify-between rounded-lg border p-4">
                    <div className="space-y-0.5">
                      <FormLabel className="text-base">Aktywny</FormLabel>
                      <FormDescription>
                        Czy ta pozycja jest aktywna i widoczna
                      </FormDescription>
                    </div>
                    <FormControl>
                      <Switch
                        checked={field.value}
                        onCheckedChange={field.onChange}
                        data-testid="switch-active"
                      />
                    </FormControl>
                  </FormItem>
                )}
              />

              {/* Image upload section for editing */}
              {editingEntry && (
                <div className="space-y-4 pt-4 border-t">
                  <div className="flex items-center justify-between">
                    <h3 className="font-semibold">Obrazy</h3>
                    <Button
                      type="button"
                      variant="outline"
                      size="sm"
                      onClick={() => fileInputRef.current?.click()}
                      disabled={uploadingImages}
                      data-testid="button-upload-images"
                    >
                      {uploadingImages ? (
                        <Loader2 className="h-4 w-4 animate-spin mr-2" />
                      ) : (
                        <ImageIcon className="h-4 w-4 mr-2" />
                      )}
                      Dodaj obrazy
                    </Button>
                    <input
                      ref={fileInputRef}
                      type="file"
                      multiple
                      accept="image/*"
                      className="hidden"
                      onChange={(e) => handleFileUpload(e.target.files, editingEntry.id)}
                    />
                  </div>

                  <div className="grid grid-cols-4 gap-4">
                    {dictionaryImages.map((img: any) => (
                      <div key={img.id} className="relative group">
                        <img
                          src={img.url}
                          alt={img.alt || ""}
                          className="w-full h-24 object-cover rounded border"
                        />
                        <Button
                          type="button"
                          variant="destructive"
                          size="icon"
                          className="absolute top-1 right-1 h-6 w-6 opacity-0 group-hover:opacity-100 transition-opacity"
                          onClick={() => deleteImageMutation.mutate({ 
                            dictionaryId: editingEntry.id, 
                            imageId: img.id 
                          })}
                          data-testid={`button-delete-image-${img.id}`}
                        >
                          <X className="h-3 w-3" />
                        </Button>
                      </div>
                    ))}
                  </div>
                </div>
              )}

              <div className="flex justify-end gap-2 pt-4">
                <Button type="button" variant="outline" onClick={() => handleCloseDialog()} data-testid="button-cancel">
                  Anuluj
                </Button>
                <Button 
                  type="submit" 
                  disabled={createMutation.isPending || updateMutation.isPending}
                  data-testid="button-submit"
                >
                  {(createMutation.isPending || updateMutation.isPending) && (
                    <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                  )}
                  {editingEntry ? "Zapisz" : "Dodaj"}
                </Button>
              </div>
            </form>
          </Form>
        </DialogContent>
      </Dialog>

      {/* CSV Import Dialog */}
      <Dialog open={isImportDialogOpen} onOpenChange={setIsImportDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Import CSV</DialogTitle>
            <DialogDescription>
              Wklej dane CSV w formacie: Kod,Nazwa,Nazwa czytelna,Opis,Kolor HEX,Kategoria,Kolejność,Aktywny
            </DialogDescription>
          </DialogHeader>
          
          <Textarea
            value={csvData}
            onChange={(e) => setCsvData(e.target.value)}
            rows={10}
            placeholder="SUPRA,Supra,System Supra,Opis,,Systemy,1,TAK"
            data-testid="textarea-csv"
          />

          <div className="flex justify-end gap-2">
            <Button 
              variant="outline" 
              onClick={() => setIsImportDialogOpen(false)}
              data-testid="button-cancel-import"
            >
              Anuluj
            </Button>
            <Button
              onClick={() => importCsvMutation.mutate({ type: dictionaryType, csvData })}
              disabled={!csvData || importCsvMutation.isPending}
              data-testid="button-submit-import"
            >
              {importCsvMutation.isPending && <Loader2 className="h-4 w-4 mr-2 animate-spin" />}
              Importuj
            </Button>
          </div>
        </DialogContent>
      </Dialog>
    </div>
  );
}
