import { UseFormReturn } from "react-hook-form";
import { UseMutationResult } from "@tanstack/react-query";
import { Loader2 } from "lucide-react";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Checkbox } from "@/components/ui/checkbox";
import { Button } from "@/components/ui/button";
import type { ProductionLocation } from "@shared/schema";
import { z } from "zod";
import { insertProductionWorkCenterSchema } from "@shared/schema";

export const workCenterFormSchema = insertProductionWorkCenterSchema.extend({
  code: z.string().min(1, "Kod jest wymagany"),
  name: z.string().min(1, "Nazwa jest wymagana"),
  locationId: z.coerce.number().optional().nullable(),
  throughputPerHour: z.coerce.number().optional().nullable(),
  operatingCostPerHour: z.coerce.number().optional().nullable(),
  oeeTarget: z.coerce.number().min(0).max(100).optional().nullable(),
  currentOee: z.coerce.number().min(0).max(100).optional().nullable(),
});

export type WorkCenterFormData = z.infer<typeof workCenterFormSchema>;

interface ProductionWorkCenterFormProps {
  form: UseFormReturn<WorkCenterFormData>;
  onSubmit: (data: WorkCenterFormData) => void;
  mutation: UseMutationResult<any, any, any, any>;
  locations: ProductionLocation[];
  mode: "create" | "edit";
  onCancel?: () => void;
}

export function ProductionWorkCenterForm({
  form,
  onSubmit,
  mutation,
  locations,
  mode,
  onCancel,
}: ProductionWorkCenterFormProps) {
  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <FormField
            control={form.control}
            name="code"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Kod</FormLabel>
                <FormControl>
                  <Input 
                    {...field} 
                    placeholder="np. WC-001"
                    data-testid="input-workcenter-code"
                  />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
          
          <FormField
            control={form.control}
            name="name"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Nazwa</FormLabel>
                <FormControl>
                  <Input 
                    {...field} 
                    placeholder="Nazwa gniazda roboczego"
                    data-testid="input-workcenter-name"
                  />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
        </div>

        <FormField
          control={form.control}
          name="description"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Opis</FormLabel>
              <FormControl>
                <Textarea 
                  {...field} 
                  value={field.value ?? ""}
                  placeholder="Opis gniazda roboczego..."
                  data-testid="input-workcenter-description"
                />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />

        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <FormField
            control={form.control}
            name="locationId"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Lokalizacja</FormLabel>
                <Select 
                  onValueChange={(value) => field.onChange(value === "none" ? null : parseInt(value))} 
                  value={field.value?.toString() ?? "none"}
                >
                  <FormControl>
                    <SelectTrigger data-testid="select-workcenter-location">
                      <SelectValue placeholder="Wybierz lokalizację" />
                    </SelectTrigger>
                  </FormControl>
                  <SelectContent>
                    <SelectItem value="none">Brak</SelectItem>
                    {locations.map((loc) => (
                      <SelectItem key={loc.id} value={loc.id.toString()}>
                        {loc.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
                <FormMessage />
              </FormItem>
            )}
          />

          <FormField
            control={form.control}
            name="status"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Status</FormLabel>
                <Select onValueChange={field.onChange} value={field.value}>
                  <FormControl>
                    <SelectTrigger data-testid="select-workcenter-status">
                      <SelectValue placeholder="Wybierz status" />
                    </SelectTrigger>
                  </FormControl>
                  <SelectContent>
                    <SelectItem value="available">Dostępne</SelectItem>
                    <SelectItem value="busy">Zajęte</SelectItem>
                    <SelectItem value="maintenance">Konserwacja</SelectItem>
                    <SelectItem value="offline">Wyłączone</SelectItem>
                  </SelectContent>
                </Select>
                <FormMessage />
              </FormItem>
            )}
          />
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <FormField
            control={form.control}
            name="throughputPerHour"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Przepustowość/godzinę</FormLabel>
                <FormControl>
                  <Input 
                    {...field} 
                    type="number"
                    step="0.01"
                    value={field.value ?? ""}
                    onChange={(e) => field.onChange(e.target.value ? parseFloat(e.target.value) : null)}
                    placeholder="0"
                    data-testid="input-workcenter-throughput"
                  />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />

          <FormField
            control={form.control}
            name="throughputUnit"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Jednostka przepustowości</FormLabel>
                <FormControl>
                  <Input 
                    {...field} 
                    value={field.value ?? ""}
                    placeholder="np. szt, kg, m²"
                    data-testid="input-workcenter-unit"
                  />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
        </div>

        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <FormField
            control={form.control}
            name="operatingCostPerHour"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Koszt operacyjny (PLN/h)</FormLabel>
                <FormControl>
                  <Input 
                    {...field} 
                    type="number"
                    step="0.01"
                    value={field.value ?? ""}
                    onChange={(e) => field.onChange(e.target.value ? parseFloat(e.target.value) : null)}
                    placeholder="0"
                    data-testid="input-workcenter-cost"
                  />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />

          <FormField
            control={form.control}
            name="oeeTarget"
            render={({ field }) => (
              <FormItem>
                <FormLabel>OEE Target (%)</FormLabel>
                <FormControl>
                  <Input 
                    {...field} 
                    type="number"
                    min="0"
                    max="100"
                    step="0.1"
                    value={field.value ?? ""}
                    onChange={(e) => field.onChange(e.target.value ? parseFloat(e.target.value) : null)}
                    placeholder="0"
                    data-testid="input-workcenter-oee-target"
                  />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />

          <FormField
            control={form.control}
            name="currentOee"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Bieżące OEE (%)</FormLabel>
                <FormControl>
                  <Input 
                    {...field} 
                    type="number"
                    min="0"
                    max="100"
                    step="0.1"
                    value={field.value ?? ""}
                    onChange={(e) => field.onChange(e.target.value ? parseFloat(e.target.value) : null)}
                    placeholder="0"
                    data-testid="input-workcenter-oee-current"
                  />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
        </div>

        <FormField
          control={form.control}
          name="notes"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Notatki</FormLabel>
              <FormControl>
                <Textarea 
                  {...field} 
                  value={field.value ?? ""}
                  placeholder="Dodatkowe notatki..."
                  data-testid="input-workcenter-notes"
                />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />

        <FormField
          control={form.control}
          name="isActive"
          render={({ field }) => (
            <FormItem className="flex flex-row items-start space-x-3 space-y-0">
              <FormControl>
                <Checkbox
                  checked={field.value ?? true}
                  onCheckedChange={field.onChange}
                  data-testid="checkbox-workcenter-active"
                />
              </FormControl>
              <div className="space-y-1 leading-none">
                <FormLabel>Aktywne</FormLabel>
              </div>
            </FormItem>
          )}
        />

        <div className="flex gap-2 justify-end pt-4">
          {onCancel && (
            <Button
              type="button"
              variant="outline"
              onClick={onCancel}
              disabled={mutation.isPending}
              data-testid="button-cancel"
            >
              Anuluj
            </Button>
          )}
          <Button 
            type="submit" 
            disabled={mutation.isPending}
            data-testid="button-submit-workcenter"
          >
            {mutation.isPending && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
            {mode === "create" ? "Utwórz gniazdo" : "Zapisz zmiany"}
          </Button>
        </div>
      </form>
    </Form>
  );
}
