import { useMemo, useState } from "react";
import { Package, ChevronDown, ChevronRight, FileText, Layers } from "lucide-react";
import { Badge } from "@/components/ui/badge";
import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { 
  DraggableProductItem, 
  type MarketplaceOrderItem,
  type SetComponent
} from "./DraggableProductItem";
import { useLocation } from "wouter";
import { useToast } from "@/hooks/use-toast";
import { Button } from "@/components/ui/button";

interface DictionaryItem {
  id: number;
  code: string;
  name: string | null;
  shortName: string | null;
  color: string | null;
  isActive: boolean;
}

interface MarketplaceOrder {
  order_id: number;
  order_number: string;
  marketplace: string;
  buyer_first_name: string;
  buyer_last_name: string;
  buyer_email: string;
  order_date: string;
  payment_status: string;
  total_to_pay_amount: string;
  currency: string;
  items: MarketplaceOrderItem[];
}

interface CompactOrdersListProps {
  orders: MarketplaceOrder[];
  colors?: DictionaryItem[];
  productTypes?: DictionaryItem[];
  onAddItem?: (item: MarketplaceOrderItem, orderNumber: string, buyerName: string, marketplace: string, setId?: number, componentId?: number) => void;
  isAddingItem?: boolean;
  recentlyAddedItems?: Set<number>;
  selectedItemIds?: Set<string>;
  onToggleItemSelection?: (itemId: string) => void;
  expandedSets?: Set<number>;
  setExpandedSets?: React.Dispatch<React.SetStateAction<Set<number>>>;
  onHighlightInPlan?: (productId: number, orderNumber: string) => void;
}

// Helper: Create deterministic order tone map (0 or 1) for alternating backgrounds
function getOrderToneMap(orders: MarketplaceOrder[]): Map<number, 0 | 1> {
  const uniqueOrderIds = Array.from(new Set(orders.map(o => o.order_id))).sort((a, b) => a - b);
  const toneMap = new Map<number, 0 | 1>();
  
  uniqueOrderIds.forEach((orderId, index) => {
    toneMap.set(orderId, (index % 2) as 0 | 1);
  });
  
  return toneMap;
}

export function CompactOrdersList({ 
  orders, 
  colors, 
  productTypes = [], 
  onAddItem, 
  isAddingItem, 
  recentlyAddedItems = new Set(), 
  selectedItemIds = new Set(), 
  onToggleItemSelection,
  expandedSets: externalExpandedSets,
  setExpandedSets: externalSetExpandedSets,
  onHighlightInPlan
}: CompactOrdersListProps) {
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  
  // Use internal state only if external state is not provided
  const [internalExpandedSets, setInternalExpandedSets] = useState<Set<number>>(() => {
    const allSetItemIds = new Set<number>();
    orders.forEach(order => {
      order.items.forEach(item => {
        if (item.set_components && item.set_components.length > 0) {
          allSetItemIds.add(item.item_id);
        }
      });
    });
    return allSetItemIds;
  });
  
  // Use external state if provided, otherwise use internal
  const expandedSets = externalExpandedSets ?? internalExpandedSets;
  const setExpandedSets = externalSetExpandedSets ?? setInternalExpandedSets;
  
  const handleCopyOrderNumber = (orderNumber: string) => {
    navigator.clipboard.writeText(orderNumber);
    toast({
      description: `Skopiowano numer zamówienia: ${orderNumber}`,
      duration: 2000,
    });
  };
  
  const handleGoToOrder = (orderNumber: string) => {
    setLocation(`/order/${orderNumber}`);
  };
  
  const toggleSetExpansion = (itemId: number) => {
    setExpandedSets(prev => {
      const newSet = new Set(prev);
      if (newSet.has(itemId)) {
        newSet.delete(itemId);
      } else {
        newSet.add(itemId);
      }
      return newSet;
    });
  };
  
  // Create order tone mapping for alternating backgrounds
  const orderToneMap = useMemo(() => getOrderToneMap(orders), [orders]);
  
  // Group orders by order number (each order as a group)
  const groupedOrders = useMemo(() => {
    return orders.map(order => ({
      order,
      buyerName: `${order.buyer_first_name} ${order.buyer_last_name}`,
      tone: orderToneMap.get(order.order_id) ?? 0
    }));
  }, [orders, orderToneMap]);

  return (
    <div className="space-y-2" data-testid="compact-orders-list">
      {groupedOrders.map(({ order, buyerName, tone }) => (
        <div 
          key={order.order_id}
          className="border border-border"
        >
          {/* Order Header - 40px - light background for order info */}
          <div
            className="flex items-center gap-2 h-[40px] px-3 bg-background"
            data-testid={`order-group-${order.order_number}`}
          >
            <Badge 
              className="h-5 px-2 text-[10px] bg-black text-white border-transparent cursor-pointer hover:bg-gray-800 transition-colors"
              onClick={() => handleCopyOrderNumber(order.order_number)}
              title="Kliknij aby skopiować numer zamówienia"
            >
              #{order.order_number}
            </Badge>
            <button
              className="w-3 h-3 flex items-center justify-center text-primary hover:text-primary/80 transition-colors"
              title="Przejdź do zamówienia"
              onClick={() => handleGoToOrder(order.order_number)}
            >
              <FileText className="w-3 h-3" />
            </button>
            <span className="text-xs text-muted-foreground">
              {buyerName}
            </span>
            <span className="text-xs text-muted-foreground">
              • {format(new Date(order.order_date), "dd.MM.yyyy HH:mm", { locale: pl })}
            </span>
            <span className="text-xs text-muted-foreground">
              • {order.marketplace}
            </span>
            <span className="text-xs text-muted-foreground ml-auto">
              {order.items.length} {order.items.length === 1 ? 'produkt' : 'produktów'}
            </span>
          </div>

          {/* Order Items - dark background container */}
          <div className="bg-muted/20">
          {order.items.map((item, index) => {
            const hasSetComponents = item.set_components && item.set_components.length > 0;
            const isExpanded = expandedSets.has(item.item_id);
            
            return (
              <div key={item.item_id}>
                {/* Parent Item (Set) */}
                <div className="relative flex items-stretch">
                  {hasSetComponents && (
                    <button
                      onClick={() => toggleSetExpansion(item.item_id)}
                      className="flex-shrink-0 w-6 flex items-center justify-center hover:bg-muted/50 transition-colors border-r border-border/50"
                      title={isExpanded ? "Zwiń komponenty" : "Rozwiń komponenty"}
                      data-testid={`toggle-set-${item.item_id}`}
                    >
                      {isExpanded ? <ChevronDown className="w-3 h-3" /> : <ChevronRight className="w-3 h-3" />}
                    </button>
                  )}
                  {!hasSetComponents && <div className="flex-shrink-0 w-6" />}
                  <div className="flex-1">
                  <DraggableProductItem
                    item={item}
                    orderNumber={order.order_number}
                    buyerName={buyerName}
                    marketplace={order.marketplace}
                    colors={colors}
                    productTypes={productTypes}
                    onAddClick={hasSetComponents ? undefined : onAddItem}
                    isAdding={isAddingItem}
                    isRecentlyAdded={recentlyAddedItems.has(item.item_id)}
                    rowHeight="40px"
                    isSelected={selectedItemIds?.has(`item-${item.item_id}`) ?? false}
                    onToggleSelect={onToggleItemSelection ? () => onToggleItemSelection(`item-${item.item_id}`) : undefined}
                    showOrderInfo={false}
                    orderTone={tone}
                    onAddSetComponents={hasSetComponents && item.set_components?.some(c => !c.is_in_any_plan) ? () => {
                      // Add all components from set that aren't already in any plan
                      const componentsToAdd = item.set_components?.filter(c => !c.is_in_any_plan) || [];
                      componentsToAdd.forEach(component => {
                        if (onAddItem) {
                          const componentItem: typeof item = {
                            ...item,
                            item_id: component.component_id,
                            name: component.component_title,
                            product_sku: component.component_sku,
                            catalog_product_id: component.component_id,
                            quantity: component.quantity,
                            image_url: component.primary_image_url,
                          };
                          onAddItem(componentItem, order.order_number, buyerName, order.marketplace, item.catalog_set_id ?? undefined, component.component_id);
                        }
                      });
                    } : undefined}
                    allSetComponentsInPlan={Boolean(hasSetComponents && item.set_components?.every(c => c.is_in_any_plan))}
                    isSetParent={Boolean(hasSetComponents)}
                    onHighlightInPlan={onHighlightInPlan}
                  />
                  </div>
                </div>
                
                {/* Set Components (expandable) - larger indent to align with parent image right edge */}
                {hasSetComponents && isExpanded && (
                  <div className="border-l-2 border-primary/30 ml-[88px]">
                    {item.set_components!.map((component) => {
                      const allComponentsInPlan = item.set_components!.every(c => c.is_in_any_plan);
                      
                      return (
                        <div key={`${item.catalog_set_id}-${component.component_id}-${order.order_number}`} className="relative flex items-stretch min-h-[40px]">
                          {/* Component status indicator */}
                          <div className="w-1 flex-shrink-0 h-full">
                            {component.is_in_any_plan && (
                              <div className="w-full h-full bg-green-500/30" title={`W planie: ${component.in_plan_number}`} />
                            )}
                          </div>
                          
                          {/* Component item */}
                          <div className="flex-1">
                            <DraggableProductItem
                              item={{
                                ...item,
                                item_id: component.component_id,
                                name: component.component_title,
                                product_sku: component.component_sku,
                                product_length: component.component_length,
                                product_width: component.component_width,
                                product_height: component.component_height,
                                product_color: component.component_color,
                                product_type: component.component_product_type,
                                product_doors: component.component_doors,
                                product_legs: component.component_legs,
                                image_url: component.primary_image_url,
                                catalog_product_id: component.component_id,
                                catalog_set_id: item.catalog_set_id,
                                bom_component_count: null,
                                is_in_plan: component.is_in_any_plan,
                                in_plan_number: component.in_plan_number,
                                in_plan_id: component.in_plan_id,
                                in_current_plan: component.is_in_current_plan,
                                quantity: component.quantity,
                                set_components: null,
                              }}
                              orderNumber={order.order_number}
                              buyerName={buyerName}
                              marketplace={order.marketplace}
                              colors={colors}
                              productTypes={productTypes}
                              onAddClick={onAddItem}
                              setId={item.catalog_set_id ?? undefined}
                              componentId={component.component_id}
                              isAdding={isAddingItem}
                              isRecentlyAdded={false}
                              rowHeight="40px"
                              isSelected={selectedItemIds?.has(`component-${order.order_number}-${component.component_id}`) ?? false}
                              onToggleSelect={onToggleItemSelection ? () => onToggleItemSelection(`component-${order.order_number}-${component.component_id}`) : undefined}
                              showOrderInfo={false}
                              isSetChild={true}
                              onHighlightInPlan={onHighlightInPlan}
                            />
                          </div>
                          
                          {/* Component metadata badge */}
                          <div className="flex items-center px-2 gap-1 h-[40px]">
                            <Badge variant="outline" className="h-5 text-[9px] px-1.5 whitespace-nowrap">
                              <Layers className="w-2.5 h-2.5 mr-0.5" />
                              {component.quantity}x
                            </Badge>
                            {component.is_in_any_plan && (
                              <Badge variant="outline" className="h-5 text-[9px] px-1.5 bg-green-500/10 border-green-500/30 whitespace-nowrap">
                                {component.in_plan_number}
                              </Badge>
                            )}
                          </div>
                        </div>
                      );
                    })}
                    
                    {/* Summary footer */}
                    <div className="px-3 py-1 bg-muted/20 text-xs text-muted-foreground flex items-center justify-between">
                      <span>
                        Komponentów: {item.set_components!.length}
                      </span>
                      {item.set_components!.every(c => c.is_in_any_plan) && (
                        <Badge variant="outline" className="h-4 text-[9px] bg-green-500/10 border-green-500/30">
                          ✓ Wszystkie w planach
                        </Badge>
                      )}
                    </div>
                  </div>
                )}
              </div>
            );
          })}
          </div>
        </div>
      ))}
      
      {groupedOrders.length === 0 && (
        <div className="flex flex-col items-center justify-center p-8 text-center text-muted-foreground">
          <Package className="h-12 w-12 mb-4 opacity-50" />
          <p>Brak zamówień spełniających kryteria</p>
          <p className="text-sm">Zmień filtry aby zobaczyć więcej zamówień</p>
        </div>
      )}
    </div>
  );
}
