import { useState, useEffect } from "react";
import { Link } from "wouter";
import { X, ChevronRight, ShoppingCart, User, Calendar, DollarSign, Package, Layers, ChevronDown } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { useQuery } from "@tanstack/react-query";
import { Skeleton } from "@/components/ui/skeleton";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";

const SIDEBAR_STORAGE_KEY = 'catalogProductRightSidebarCollapsed';
const ORDERS_SECTION_STORAGE_KEY = 'catalogProductOrdersSectionCollapsed';
const SETS_SECTION_STORAGE_KEY = 'catalogProductSetsSectionCollapsed';

interface RightSidebarProps {
  productId: number | null;
  defaultCollapsed?: boolean;
}

interface OrderItem {
  order_id: string;
  buyer_login: string;
  buyer_email: string | null;
  buyer_full_name: string | null;
  order_date: string;
  marketplace: string;
  payment_status: string;
  total_amount: string;
  currency: string;
  quantity: number;
  item_price: string;
  item_name: string;
  offer_id: string | null;
  item_image_url: string | null;
}

interface ProductSet {
  id: number;
  title: string;
  sku: string;
  quantity: number;
  setMatrixId: number | null;
  componentType: string;
}

export function CatalogProductRightSidebar({ productId, defaultCollapsed = false }: RightSidebarProps) {
  const [isCollapsed, setIsCollapsed] = useState(() => {
    if (typeof window === 'undefined') return defaultCollapsed;
    const saved = localStorage.getItem(SIDEBAR_STORAGE_KEY);
    return saved !== null ? saved === 'true' : defaultCollapsed;
  });

  const [isOrdersSectionCollapsed, setIsOrdersSectionCollapsed] = useState(() => {
    if (typeof window === 'undefined') return false;
    const saved = localStorage.getItem(ORDERS_SECTION_STORAGE_KEY);
    return saved === 'true';
  });

  const [isSetsSectionCollapsed, setIsSetsSectionCollapsed] = useState(() => {
    if (typeof window === 'undefined') return false;
    const saved = localStorage.getItem(SETS_SECTION_STORAGE_KEY);
    return saved === 'true';
  });

  useEffect(() => {
    if (typeof window !== 'undefined') {
      localStorage.setItem(SIDEBAR_STORAGE_KEY, String(isCollapsed));
    }
  }, [isCollapsed]);

  useEffect(() => {
    if (typeof window !== 'undefined') {
      localStorage.setItem(ORDERS_SECTION_STORAGE_KEY, String(isOrdersSectionCollapsed));
    }
  }, [isOrdersSectionCollapsed]);

  useEffect(() => {
    if (typeof window !== 'undefined') {
      localStorage.setItem(SETS_SECTION_STORAGE_KEY, String(isSetsSectionCollapsed));
    }
  }, [isSetsSectionCollapsed]);

  const { data: orders, isLoading } = useQuery<OrderItem[]>({
    queryKey: [`/api/catalog-products/${productId}/ordered-by`],
    enabled: !!productId && !isCollapsed,
  });

  const { data: productSets = [], isLoading: isSetsLoading } = useQuery<ProductSet[]>({
    queryKey: [`/api/catalog-products/${productId}/sets`],
    enabled: !!productId && !isCollapsed,
  });

  if (isCollapsed) {
    return (
      <div className="hidden md:flex w-12 border-l bg-card flex-shrink-0 flex-col items-center py-4">
        <Button
          variant="ghost"
          size="icon"
          onClick={() => setIsCollapsed(false)}
          data-testid="button-expand-catalog-right-sidebar"
          className="h-8 w-8"
        >
          <ChevronRight className="h-4 w-4 rotate-180" />
        </Button>
      </div>
    );
  }

  if (!productId) {
    return (
      <div className="hidden md:flex w-80 border-l bg-card flex-shrink-0 overflow-y-auto flex-col">
        <div className="border-b px-4 py-4 flex items-center justify-between gap-2 flex-shrink-0">
          <h2 className="text-lg font-semibold flex-1">Zamówienia</h2>
          <Button
            variant="ghost"
            size="icon"
            className="h-8 w-8 flex-shrink-0"
            onClick={() => setIsCollapsed(true)}
            data-testid="button-collapse-catalog-right-sidebar"
          >
            <X className="h-4 w-4" />
          </Button>
        </div>
        <div className="flex-1 flex items-center justify-center p-4">
          <p className="text-sm text-muted-foreground text-center">
            Wybierz produkt aby zobaczyć zamówienia
          </p>
        </div>
      </div>
    );
  }

  return (
    <div className="hidden md:flex w-80 border-l bg-card flex-shrink-0 overflow-y-auto flex-col">
      {/* Header */}
      <div className="border-b px-4 py-4 flex items-center justify-between gap-2 flex-shrink-0">
        <h2 className="text-lg font-semibold flex-1">Zamówienia</h2>
        <Button
          variant="ghost"
          size="icon"
          className="h-8 w-8 flex-shrink-0"
          onClick={() => setIsCollapsed(true)}
          data-testid="button-collapse-catalog-right-sidebar"
        >
          <X className="h-4 w-4" />
        </Button>
      </div>

      {/* Content */}
      <div className="flex-1 overflow-y-auto">
        {isLoading ? (
          <div className="p-4 space-y-3">
            {[...Array(3)].map((_, i) => (
              <div key={i} className="space-y-2">
                <Skeleton className="h-4 w-3/4" />
                <Skeleton className="h-3 w-1/2" />
                <Skeleton className="h-3 w-2/3" />
              </div>
            ))}
          </div>
        ) : orders && orders.length > 0 ? (
          <>
            {/* Summary Card - only PAID orders */}
            <div className="m-4 p-4 rounded-lg bg-green-500/10 border-2 border-green-500/30">
              <div className="space-y-3">
                <div className="flex items-center justify-between">
                  <span className="text-sm font-medium text-muted-foreground">Zapłacono łącznie</span>
                  <div className="text-right">
                    <div className="text-2xl font-bold text-green-600 dark:text-green-400">
                      {orders.filter(o => o.payment_status === 'PAID').reduce((sum, order) => sum + order.quantity, 0)} szt.
                    </div>
                    <div className="text-sm font-semibold text-green-600 dark:text-green-400">
                      {orders.filter(o => o.payment_status === 'PAID').reduce((sum, order) => sum + (parseFloat(order.item_price) * order.quantity), 0).toFixed(2)} PLN
                    </div>
                  </div>
                </div>
                <div className="flex items-center justify-between pt-2 border-t border-green-500/20">
                  <span className="text-sm font-medium text-muted-foreground">Ostatnia sprzedaż</span>
                  <div className="text-right">
                    <div className="text-sm font-semibold text-foreground">
                      {new Date(Math.max(...orders.map(o => new Date(o.order_date).getTime()))).toLocaleDateString('pl-PL', {
                        day: '2-digit',
                        month: '2-digit',
                        year: 'numeric'
                      })}
                    </div>
                    <div className="text-xs text-muted-foreground">
                      {new Date(Math.max(...orders.map(o => new Date(o.order_date).getTime()))).toLocaleTimeString('pl-PL', {
                        hour: '2-digit',
                        minute: '2-digit'
                      })}
                    </div>
                  </div>
                </div>
              </div>
            </div>

            <div className="border-b last:border-b-0">
              <button
                onClick={() => setIsOrdersSectionCollapsed(!isOrdersSectionCollapsed)}
                className="w-full px-4 py-3 flex items-center gap-2 bg-primary/5 hover-elevate active-elevate-2 transition-colors"
                data-testid="button-toggle-orders-section"
              >
                <ShoppingCart className="h-5 w-5 text-primary" />
                <h3 className="font-semibold text-base flex-1 text-left text-primary">Historia zakupów</h3>
                <Badge variant="secondary" className="text-xs font-bold">
                  {orders.length}
                </Badge>
                <ChevronDown 
                  className={`h-4 w-4 transition-transform ${isOrdersSectionCollapsed ? '-rotate-90' : ''}`} 
                />
              </button>
              {!isOrdersSectionCollapsed && (
                <div className="p-2 space-y-2">
                {orders.map((order, index) => (
                <Link key={`${order.order_id}-${index}`} href={`/order/${order.order_id}`}>
                  <div
                    className="flex flex-col gap-2 px-3 py-3 rounded-md text-sm transition-colors hover-elevate active-elevate-2 border border-border/50"
                    data-testid={`order-item-${order.order_id}`}
                  >
                    {/* Buyer info */}
                    <div className="flex items-start gap-2">
                      <User className="h-3.5 w-3.5 mt-0.5 flex-shrink-0 text-muted-foreground" />
                      <div className="flex-1 min-w-0">
                        <div className="font-medium truncate">
                          {order.buyer_full_name || order.buyer_login}
                        </div>
                        {order.buyer_email && (
                          <div className="text-xs text-muted-foreground truncate">
                            {order.buyer_email}
                          </div>
                        )}
                      </div>
                    </div>

                    {/* Order date */}
                    <div className="flex items-center gap-2">
                      <Calendar className="h-3.5 w-3.5 flex-shrink-0 text-muted-foreground" />
                      <span className="text-xs text-muted-foreground">
                        {new Date(order.order_date).toLocaleDateString('pl-PL', {
                          year: 'numeric',
                          month: 'short',
                          day: 'numeric',
                          hour: '2-digit',
                          minute: '2-digit'
                        })}
                      </span>
                    </div>

                    {/* Marketplace and payment status */}
                    <div className="flex items-center gap-2 flex-wrap">
                      <Badge variant="outline" className="text-xs">
                        {order.marketplace === 'allegro-pl' ? 'Allegro' : order.marketplace}
                      </Badge>
                      <Badge 
                        variant={
                          order.payment_status === 'PAID' ? 'default' : 
                          order.payment_status === 'PARTIAL' ? 'secondary' : 
                          'destructive'
                        } 
                        className="text-xs"
                      >
                        {order.payment_status}
                      </Badge>
                    </div>

                    {/* Quantity and price */}
                    <div className="flex items-center gap-2 pt-1 border-t border-border/30">
                      <DollarSign className="h-3.5 w-3.5 flex-shrink-0 text-muted-foreground" />
                      <div className="flex-1 flex items-center justify-between gap-2">
                        <span className="text-xs text-muted-foreground">
                          Ilość: <span className="font-medium text-foreground">{order.quantity}</span>
                        </span>
                        <span className="text-xs font-medium">
                          {parseFloat(order.item_price).toFixed(2)} {order.currency}
                        </span>
                      </div>
                    </div>

                    {/* Total order amount */}
                    {order.total_amount && (
                      <div className="text-xs text-muted-foreground">
                        Zamówienie ogółem: <span className="font-medium text-foreground">
                          {parseFloat(order.total_amount).toFixed(2)} {order.currency}
                        </span>
                      </div>
                    )}
                  </div>
                </Link>
              ))}
                </div>
              )}
            </div>
          </>
        ) : (
          <div className="p-4">
            <p className="text-sm text-muted-foreground text-center">
              Brak zamówień dla tego produktu
            </p>
          </div>
        )}

        {/* Sets Section */}
        {isSetsLoading ? (
          <div className="p-4 space-y-3 border-t">
            <Skeleton className="h-5 w-1/2" />
            {[...Array(3)].map((_, i) => (
              <Skeleton key={i} className="h-10 w-full" />
            ))}
          </div>
        ) : productSets && productSets.length > 0 ? (
          <div className="border-t">
            <button
              onClick={() => setIsSetsSectionCollapsed(!isSetsSectionCollapsed)}
              className="w-full px-4 py-3 flex items-center gap-2 bg-primary/5 hover-elevate active-elevate-2 transition-colors"
              data-testid="button-toggle-sets-section"
            >
              <Layers className="h-5 w-5 text-primary" />
              <h3 className="font-semibold text-base flex-1 text-left text-primary">Występowanie w zestawach</h3>
              <Badge variant="secondary" className="text-xs font-bold">
                {productSets.length}
              </Badge>
              <ChevronDown 
                className={`h-4 w-4 transition-transform ${isSetsSectionCollapsed ? '-rotate-90' : ''}`} 
              />
            </button>
            {!isSetsSectionCollapsed && (
              <div className="p-2 space-y-2">
              {productSets.slice(0, 10).map((set) => (
                <Link key={set.id} href={`/catalog-sets/${set.id}`}>
                  <div
                    className="flex flex-col gap-2 px-3 py-3 rounded-md text-sm transition-colors hover-elevate active-elevate-2 border border-border/50"
                    data-testid={`set-item-${set.id}`}
                  >
                    <div className="flex items-start gap-2">
                      <Package className="h-3.5 w-3.5 mt-0.5 flex-shrink-0 text-muted-foreground" />
                      <div className="flex-1 min-w-0">
                        <div className="font-medium line-clamp-2">
                          {set.title}
                        </div>
                        <div className="text-xs text-muted-foreground mt-1">
                          SKU: {set.sku}
                        </div>
                      </div>
                    </div>
                    <div className="flex items-center gap-2 pt-1 border-t border-border/30">
                      <span className="text-xs text-muted-foreground">
                        Ilość: <span className="font-medium text-foreground">{set.quantity}</span>
                      </span>
                      {set.componentType && (
                        <Badge variant="outline" className="text-xs">
                          {set.componentType}
                        </Badge>
                      )}
                    </div>
                  </div>
                </Link>
              ))}
              
              {productSets.length > 10 && (
                <Dialog>
                  <DialogTrigger asChild>
                    <Button 
                      variant="outline" 
                      className="w-full text-sm"
                      data-testid="button-show-all-sets"
                    >
                      Pokaż wszystkie ({productSets.length})
                    </Button>
                  </DialogTrigger>
                  <DialogContent className="max-w-2xl max-h-[80vh] overflow-y-auto">
                    <DialogHeader>
                      <DialogTitle>Występowanie w zestawach</DialogTitle>
                      <DialogDescription>
                        Wszystkie zestawy zawierające ten produkt ({productSets.length})
                      </DialogDescription>
                    </DialogHeader>
                    <div className="space-y-2 mt-4">
                      {productSets.map((set, index) => (
                        <Link key={set.id} href={`/catalog-sets/${set.id}`}>
                          <div
                            className="flex items-start gap-3 px-4 py-3 rounded-md border hover-elevate active-elevate-2"
                            data-testid={`modal-set-item-${set.id}`}
                          >
                            <div className="flex-shrink-0 w-8 h-8 rounded-full bg-primary/10 flex items-center justify-center">
                              <span className="text-sm font-medium text-primary">{index + 1}</span>
                            </div>
                            <div className="flex-1 min-w-0">
                              <div className="font-medium">{set.title}</div>
                              <div className="flex items-center gap-2 mt-1 flex-wrap">
                                <span className="text-xs text-muted-foreground">
                                  SKU: {set.sku}
                                </span>
                                <span className="text-xs text-muted-foreground">•</span>
                                <span className="text-xs text-muted-foreground">
                                  Ilość: {set.quantity}
                                </span>
                                {set.componentType && (
                                  <>
                                    <span className="text-xs text-muted-foreground">•</span>
                                    <Badge variant="outline" className="text-xs">
                                      {set.componentType}
                                    </Badge>
                                  </>
                                )}
                              </div>
                            </div>
                          </div>
                        </Link>
                      ))}
                    </div>
                  </DialogContent>
                </Dialog>
              )}
              </div>
            )}
          </div>
        ) : null}
      </div>
    </div>
  );
}
