import { useState } from "react";
import { Check, ChevronsUpDown, Plus } from "lucide-react";
import { cn } from "@/lib/utils";
import { Button } from "@/components/ui/button";
import {
  Command,
  CommandEmpty,
  CommandGroup,
  CommandInput,
  CommandItem,
  CommandList,
  CommandSeparator,
} from "@/components/ui/command";
import {
  Popover,
  PopoverContent,
  PopoverTrigger,
} from "@/components/ui/popover";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { useToast } from "@/hooks/use-toast";
import { useMutation } from "@tanstack/react-query";
import { queryClient } from "@/lib/queryClient";

interface DictionaryItem {
  id: number;
  code: string;
  name: string;
  readableName: string | null;
  isActive: boolean;
}

interface DictionaryComboboxWithAddProps {
  items: DictionaryItem[] | undefined;
  value: string;
  onChange: (value: string) => void;
  dictionaryType: string;
  placeholder?: string;
  searchPlaceholder?: string;
  emptyText?: string;
  testId?: string;
  valueField?: 'id' | 'name' | 'code';
  displayField?: 'name' | 'readableName';
  backgroundColor?: string;
  disabled?: boolean;
}

export function DictionaryComboboxWithAdd({
  items,
  value,
  onChange,
  dictionaryType,
  placeholder = "Wybierz...",
  searchPlaceholder = "Szukaj...",
  emptyText = "Nie znaleziono.",
  testId,
  valueField = 'code',
  displayField = 'readableName',
  backgroundColor,
  disabled = false,
}: DictionaryComboboxWithAddProps) {
  const { toast } = useToast();
  const [open, setOpen] = useState(false);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [newCode, setNewCode] = useState("");
  const [newName, setNewName] = useState("");
  const [newReadableName, setNewReadableName] = useState("");

  const getDisplayName = (item: DictionaryItem) => {
    if (displayField === 'name') {
      return item.name;
    }
    return item.readableName || item.name;
  };

  const activeItems = (items?.filter(item => item.isActive) || [])
    .sort((a, b) => {
      const nameA = getDisplayName(a).toLowerCase();
      const nameB = getDisplayName(b).toLowerCase();
      return nameA.localeCompare(nameB);
    });
  
  const selectedItem = activeItems.find(item => {
    const itemValue = valueField === 'id' ? item.id.toString() : item[valueField];
    return itemValue === value;
  });

  const createMutation = useMutation({
    mutationFn: async (data: { dictionaryType: string; code: string; name: string; readableName: string }) => {
      const response = await fetch("/api/dictionaries", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          ...data,
          code: data.code.toUpperCase(),
          isActive: true,
        }),
      });
      if (!response.ok) {
        const errorText = await response.text();
        throw new Error(errorText || "Nie udało się dodać elementu");
      }
      return response.json();
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({ queryKey: ["/api/dictionaries"] });
      toast({
        title: "Sukces",
        description: "Nowa grupa została dodana",
      });
      // Select the newly created item
      onChange(data.code);
      setDialogOpen(false);
      setNewCode("");
      setNewName("");
      setNewReadableName("");
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const handleAddNew = () => {
    setOpen(false);
    setDialogOpen(true);
  };

  const handleSaveNew = () => {
    console.log("🔍 handleSaveNew called with:", { newCode, newName, newReadableName });
    
    if (!newCode.trim() || !newName.trim()) {
      console.log("❌ Validation failed - kod or nazwa empty");
      toast({
        title: "Błąd",
        description: "Kod i nazwa są wymagane",
        variant: "destructive",
      });
      return;
    }

    const mutationData = {
      dictionaryType,
      code: newCode.trim(),
      name: newName.trim(),
      readableName: newReadableName.trim() || newName.trim(),
    };
    
    console.log("✅ Validation passed, calling mutation with:", mutationData);
    createMutation.mutate(mutationData);
  };

  return (
    <>
      <Popover open={disabled ? false : open} onOpenChange={disabled ? undefined : setOpen}>
        <PopoverTrigger asChild>
          <Button
            variant="outline"
            role="combobox"
            aria-expanded={open}
            className="w-full justify-between"
            data-testid={testId}
            disabled={disabled}
            style={backgroundColor ? { 
              backgroundColor, 
              color: 'white',
              borderColor: backgroundColor
            } : undefined}
          >
            {selectedItem ? getDisplayName(selectedItem) : placeholder}
            <ChevronsUpDown className="ml-2 h-4 w-4 shrink-0" style={backgroundColor ? { opacity: 0.8 } : { opacity: 0.5 }} />
          </Button>
        </PopoverTrigger>
        <PopoverContent className="w-[--radix-popover-trigger-width] p-0" align="start">
          <Command>
            <div className="flex items-center border-b px-3">
              <CommandInput 
                placeholder={searchPlaceholder} 
                className="flex-1 border-0 focus:ring-0"
              />
              <Button
                type="button"
                size="icon"
                variant="ghost"
                onClick={handleAddNew}
                className="h-8 w-8 shrink-0"
                data-testid="button-add-new"
              >
                <Plus className="h-4 w-4" />
              </Button>
            </div>
            <CommandList>
              <CommandEmpty>{emptyText}</CommandEmpty>
              <CommandGroup>
                {activeItems.map((item) => {
                  const itemValue = valueField === 'id' ? item.id.toString() : item[valueField];
                  const displayName = getDisplayName(item);
                  return (
                    <CommandItem
                      key={item.id}
                      value={displayName}
                      onSelect={() => {
                        onChange(itemValue === value ? "" : itemValue);
                        setOpen(false);
                      }}
                      data-testid={`option-${item.code}`}
                    >
                      <Check
                        className={cn(
                          "mr-2 h-4 w-4",
                          value === itemValue ? "opacity-100" : "opacity-0"
                        )}
                      />
                      {displayName}
                    </CommandItem>
                  );
                })}
              </CommandGroup>
            </CommandList>
          </Command>
        </PopoverContent>
      </Popover>

      <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
        <DialogContent className="w-[95vw] max-w-md" data-testid="dialog-add-dictionary">
          <DialogHeader>
            <DialogTitle>Dodaj nową grupę produktów</DialogTitle>
            <DialogDescription>
              Wypełnij poniższe pola aby dodać nową grupę produktów do słownika.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="code">
                Kod <span className="text-destructive">*</span>
              </Label>
              <Input
                id="code"
                value={newCode}
                onChange={(e) => setNewCode(e.target.value)}
                placeholder="np. GR_A"
                data-testid="input-new-code"
                className="uppercase"
              />
              <p className="text-xs text-muted-foreground">
                Unikalny kod grupy (tylko duże litery i cyfry)
              </p>
            </div>
            <div className="space-y-2">
              <Label htmlFor="name">
                Nazwa <span className="text-destructive">*</span>
              </Label>
              <Input
                id="name"
                value={newName}
                onChange={(e) => setNewName(e.target.value)}
                placeholder="np. Grupa A"
                data-testid="input-new-name"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="readableName">Nazwa czytelna</Label>
              <Input
                id="readableName"
                value={newReadableName}
                onChange={(e) => setNewReadableName(e.target.value)}
                placeholder="np. Szafki podstawowe"
                data-testid="input-new-readable-name"
              />
              <p className="text-xs text-muted-foreground">
                Opcjonalna nazwa wyświetlana użytkownikom
              </p>
            </div>
          </div>
          <DialogFooter className="flex-col-reverse sm:flex-row gap-2">
            <Button
              variant="outline"
              onClick={() => setDialogOpen(false)}
              disabled={createMutation.isPending}
              data-testid="button-cancel-add"
              className="w-full sm:w-auto"
            >
              Anuluj
            </Button>
            <Button
              onClick={handleSaveNew}
              disabled={createMutation.isPending}
              data-testid="button-save-new"
              className="w-full sm:w-auto"
            >
              {createMutation.isPending ? "Dodawanie..." : "Dodaj grupę"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </>
  );
}
