import { useState, useRef, useMemo } from "react";
import { useMutation } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger, DialogFooter } from "@/components/ui/dialog";
import { useToast } from "@/hooks/use-toast";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { Upload, Loader2, Download, AlertCircle, CheckCircle2, ArrowRight, X, ChevronDown, FileSpreadsheet } from "lucide-react";
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import { Label } from "@/components/ui/label";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import Papa from "papaparse";
import * as XLSX from "xlsx";

interface FieldDefinition {
  key: string;
  label: string;
  required?: boolean;
  description?: string;
}

interface CSVImportWithMappingProps {
  endpoint: string;
  fields: FieldDefinition[];
  templateCsv?: string;
  templateFilename?: string;
  invalidateQueryKey: string[];
  title?: string;
  description?: string;
  children?: React.ReactNode;
  category?: string;
}

interface ImportResult {
  success: number;
  failed: number;
  errors: Array<{ row: number; error: string; data?: any }>;
}

export function CSVImportWithMapping({
  endpoint,
  fields,
  templateCsv,
  templateFilename = "template.csv",
  invalidateQueryKey,
  title = "Import z CSV/Excel",
  description = "Importuj dane z pliku CSV lub Excel (XLSX) z mapowaniem pól",
  children,
  category,
}: CSVImportWithMappingProps) {
  const [open, setOpen] = useState(false);
  const [step, setStep] = useState<'upload' | 'mapping' | 'preview' | 'result'>('upload');
  const [file, setFile] = useState<File | null>(null);
  const [fileType, setFileType] = useState<'csv' | 'xlsx' | null>(null);
  const [csvHeaders, setCsvHeaders] = useState<string[]>([]);
  const [csvData, setCsvData] = useState<any[]>([]);
  const [mapping, setMapping] = useState<Record<string, string>>({});
  const [result, setResult] = useState<ImportResult | null>(null);
  const fileInputRef = useRef<HTMLInputElement>(null);
  const { toast } = useToast();

  const importMutation = useMutation({
    mutationFn: async (data: { rows: any[]; mapping: Record<string, string>; category?: string }) => {
      const response = await apiRequest("POST", endpoint, data);
      return await response.json();
    },
    onSuccess: (data: ImportResult) => {
      setResult(data);
      setStep('result');
      // Use predicate to match queries that start with the invalidateQueryKey prefix
      queryClient.invalidateQueries({ 
        predicate: (query) => {
          const key = query.queryKey;
          if (!key || !Array.isArray(key) || key.length === 0) return false;
          const firstKey = String(key[0]);
          // Match any query that starts with the invalidation key prefix
          return invalidateQueryKey.some(k => firstKey.startsWith(k));
        }
      });
      
      if (data.failed === 0) {
        toast({
          title: "Import zakończony sukcesem",
          description: `Zaimportowano ${data.success} rekordów`,
        });
      } else {
        toast({
          title: "Import częściowo zakończony",
          description: `Sukces: ${data.success}, Błędy: ${data.failed}`,
          variant: "destructive",
        });
      }
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd importu",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const parseCSV = (text: string) => {
    const parseResult = Papa.parse(text, {
      header: true,
      skipEmptyLines: true,
      transformHeader: (header: string) => header.trim(),
    });
    
    if (parseResult.errors.length > 0) {
      toast({
        title: "Błąd parsowania CSV",
        description: parseResult.errors[0].message,
        variant: "destructive",
      });
      return null;
    }
    
    return {
      headers: parseResult.meta.fields || [],
      data: parseResult.data as any[],
    };
  };

  const parseXLSX = (buffer: ArrayBuffer) => {
    try {
      const workbook = XLSX.read(buffer, { type: 'array' });
      const sheetName = workbook.SheetNames[0];
      const worksheet = workbook.Sheets[sheetName];
      
      const jsonData = XLSX.utils.sheet_to_json(worksheet, { header: 1 }) as any[][];
      
      if (jsonData.length === 0) {
        toast({
          title: "Pusty plik",
          description: "Plik Excel nie zawiera danych",
          variant: "destructive",
        });
        return null;
      }
      
      const headers = jsonData[0].map((h: any) => String(h || '').trim());
      const data = jsonData.slice(1).map(row => {
        const obj: Record<string, any> = {};
        headers.forEach((header, index) => {
          obj[header] = row[index] !== undefined ? String(row[index]) : '';
        });
        return obj;
      }).filter(row => Object.values(row).some(v => v !== ''));
      
      return { headers, data };
    } catch (error) {
      toast({
        title: "Błąd parsowania Excel",
        description: "Nie udało się odczytać pliku Excel",
        variant: "destructive",
      });
      return null;
    }
  };

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const selectedFile = e.target.files?.[0];
    if (selectedFile) {
      const fileName = selectedFile.name.toLowerCase();
      const isCSV = fileName.endsWith('.csv');
      const isXLSX = fileName.endsWith('.xlsx') || fileName.endsWith('.xls');
      
      if (!isCSV && !isXLSX) {
        toast({
          title: "Nieprawidłowy format",
          description: "Proszę wybrać plik CSV lub Excel (XLSX/XLS)",
          variant: "destructive",
        });
        return;
      }
      
      setFileType(isCSV ? 'csv' : 'xlsx');
      
      if (isCSV) {
        const reader = new FileReader();
        reader.onload = (event) => {
          const text = event.target?.result as string;
          const result = parseCSV(text);
          if (result) {
            setCsvHeaders(result.headers);
            setCsvData(result.data);
            setFile(selectedFile);
            autoMapFields(result.headers);
            setStep('mapping');
          }
        };
        reader.readAsText(selectedFile);
      } else {
        const reader = new FileReader();
        reader.onload = (event) => {
          const buffer = event.target?.result as ArrayBuffer;
          const result = parseXLSX(buffer);
          if (result) {
            setCsvHeaders(result.headers);
            setCsvData(result.data);
            setFile(selectedFile);
            autoMapFields(result.headers);
            setStep('mapping');
          }
        };
        reader.readAsArrayBuffer(selectedFile);
      }
    }
  };

  const autoMapFields = (headers: string[]) => {
    const autoMapping: Record<string, string> = {};
    fields.forEach(field => {
      const matchingHeader = headers.find(h => {
        const headerLower = h.toLowerCase().replace(/[_\s-]/g, '');
        const fieldLower = field.key.toLowerCase().replace(/[_\s-]/g, '');
        const labelLower = field.label.toLowerCase().replace(/[_\s-]/g, '');
        
        return headerLower === fieldLower ||
          headerLower === labelLower ||
          headerLower.includes(fieldLower) ||
          fieldLower.includes(headerLower) ||
          headerLower.includes(labelLower) ||
          labelLower.includes(headerLower) ||
          (field.key === 'supplier_code' && (h.toLowerCase().includes('symbol') || h.toLowerCase().includes('kod'))) ||
          (field.key === 'name' && (h.toLowerCase().includes('nazwa') || h.toLowerCase().includes('name'))) ||
          (field.key === 'internal_code' && (h.toLowerCase().includes('wewnętrzny') || h.toLowerCase().includes('internal'))) ||
          (field.key === 'unit' && (h.toLowerCase().includes('jednostka') || h.toLowerCase().includes('jm') || h.toLowerCase() === 'j.m.')) ||
          (field.key === 'price' && (h.toLowerCase().includes('cena') || h.toLowerCase().includes('price'))) ||
          (field.key === 'description' && (h.toLowerCase().includes('opis') || h.toLowerCase().includes('uwagi')));
      });
      if (matchingHeader) {
        autoMapping[field.key] = matchingHeader;
      }
    });
    setMapping(autoMapping);
  };

  const handleMappingChange = (fieldKey: string, csvColumn: string) => {
    setMapping(prev => ({
      ...prev,
      [fieldKey]: csvColumn === '__none__' ? '' : csvColumn
    }));
  };

  const requiredFieldsMapped = useMemo(() => {
    return fields
      .filter(f => f.required)
      .every(f => mapping[f.key] && mapping[f.key] !== '');
  }, [fields, mapping]);

  const mappedPreviewData = useMemo(() => {
    return csvData.slice(0, 5).map(row => {
      const mappedRow: Record<string, any> = {};
      fields.forEach(field => {
        const csvColumn = mapping[field.key];
        mappedRow[field.key] = csvColumn ? row[csvColumn] : '';
      });
      return mappedRow;
    });
  }, [csvData, mapping, fields]);

  const validationResults = useMemo(() => {
    const requiredFields = fields.filter(f => f.required);
    const errors: Array<{ row: number; field: string; message: string }> = [];
    const validRows: any[] = [];
    const invalidRows: number[] = [];

    csvData.forEach((row, index) => {
      const rowNumber = index + 2;
      let rowValid = true;

      requiredFields.forEach(field => {
        const csvColumn = mapping[field.key];
        const value = csvColumn ? row[csvColumn]?.toString().trim() : '';
        
        if (!value) {
          errors.push({
            row: rowNumber,
            field: field.label,
            message: `Brak wartości dla pola "${field.label}"`
          });
          rowValid = false;
        }
      });

      if (rowValid) {
        const mappedRow: Record<string, any> = {};
        fields.forEach(field => {
          const csvColumn = mapping[field.key];
          mappedRow[field.key] = csvColumn ? row[csvColumn] : '';
        });
        validRows.push(mappedRow);
      } else {
        invalidRows.push(rowNumber);
      }
    });

    return { errors, validRows, invalidRows, totalRows: csvData.length };
  }, [csvData, mapping, fields]);

  const validPreviewData = useMemo(() => {
    return validationResults.validRows.slice(0, 5);
  }, [validationResults.validRows]);

  const handleImport = () => {
    if (validationResults.validRows.length === 0) {
      toast({
        title: "Brak poprawnych danych",
        description: "Wszystkie wiersze zawierają błędy. Popraw plik.",
        variant: "destructive",
      });
      return;
    }
    
    importMutation.mutate({ rows: validationResults.validRows, mapping, category });
  };

  const handleDownloadTemplate = () => {
    const template = templateCsv || fields.map(f => f.key).join(',') + '\n' + fields.map(f => f.description || '').join(',');
    const blob = new Blob([template], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    link.href = URL.createObjectURL(blob);
    link.download = templateFilename;
    link.click();
  };

  const handleReset = () => {
    setFile(null);
    setFileType(null);
    setCsvHeaders([]);
    setCsvData([]);
    setMapping({});
    setResult(null);
    setStep('upload');
    if (fileInputRef.current) {
      fileInputRef.current.value = '';
    }
  };

  const handleClose = (newOpen: boolean) => {
    setOpen(newOpen);
    if (!newOpen) {
      handleReset();
    }
  };

  return (
    <Dialog open={open} onOpenChange={handleClose}>
      <DialogTrigger asChild>
        {children || (
          <Button variant="outline" data-testid="button-import-csv-mapping">
            <Upload className="mr-2 h-4 w-4" />
            Import CSV/Excel
          </Button>
        )}
      </DialogTrigger>
      <DialogContent className="max-w-4xl max-h-[90vh] overflow-hidden flex flex-col">
        <DialogHeader>
          <DialogTitle>{title}</DialogTitle>
          <DialogDescription>{description}</DialogDescription>
        </DialogHeader>

        <div className="flex-1 overflow-hidden flex flex-col">
          {/* Step indicator */}
          <div className="flex items-center gap-2 mb-4 text-sm">
            <Badge variant={step === 'upload' ? 'default' : 'secondary'}>1. Wczytaj plik</Badge>
            <ArrowRight className="h-4 w-4 text-muted-foreground" />
            <Badge variant={step === 'mapping' ? 'default' : 'secondary'}>2. Mapowanie pól</Badge>
            <ArrowRight className="h-4 w-4 text-muted-foreground" />
            <Badge variant={step === 'preview' ? 'default' : 'secondary'}>3. Podgląd</Badge>
            <ArrowRight className="h-4 w-4 text-muted-foreground" />
            <Badge variant={step === 'result' ? 'default' : 'secondary'}>4. Wynik</Badge>
          </div>

          {/* Step 1: Upload */}
          {step === 'upload' && (
            <div className="space-y-4">
              <div className="space-y-2">
                <Button
                  type="button"
                  variant="outline"
                  size="sm"
                  onClick={handleDownloadTemplate}
                  data-testid="button-download-template"
                >
                  <Download className="mr-2 h-4 w-4" />
                  Pobierz szablon CSV
                </Button>
                <p className="text-sm text-muted-foreground">
                  Pobierz przykładowy plik CSV z poprawnymi nagłówkami kolumn
                </p>
              </div>

              <div className="space-y-2">
                <input
                  ref={fileInputRef}
                  type="file"
                  accept=".csv,.xlsx,.xls"
                  onChange={handleFileChange}
                  className="hidden"
                  id="csv-file-input-mapping"
                  data-testid="input-csv-file-mapping"
                />
                <Button
                  type="button"
                  variant="outline"
                  onClick={() => fileInputRef.current?.click()}
                  className="w-full h-32 border-dashed"
                  data-testid="button-select-file-mapping"
                >
                  <div className="flex flex-col items-center gap-2">
                    <div className="flex gap-2">
                      <Upload className="h-8 w-8 text-muted-foreground" />
                      <FileSpreadsheet className="h-8 w-8 text-muted-foreground" />
                    </div>
                    <span>{file ? file.name : "Kliknij aby wybrać plik CSV lub Excel"}</span>
                    <span className="text-xs text-muted-foreground">lub przeciągnij i upuść (CSV, XLSX, XLS)</span>
                  </div>
                </Button>
              </div>

              <div className="bg-muted/50 p-3 rounded-lg">
                <h4 className="font-medium mb-2">Wymagane pola:</h4>
                <div className="flex flex-wrap gap-2">
                  {fields.filter(f => f.required).map(field => (
                    <Badge key={field.key} variant="outline">
                      {field.label}
                    </Badge>
                  ))}
                </div>
                <h4 className="font-medium mb-2 mt-3">Opcjonalne pola:</h4>
                <div className="flex flex-wrap gap-2">
                  {fields.filter(f => !f.required).map(field => (
                    <Badge key={field.key} variant="secondary">
                      {field.label}
                    </Badge>
                  ))}
                </div>
              </div>
            </div>
          )}

          {/* Step 2: Mapping */}
          {step === 'mapping' && (
            <div className="space-y-4 flex-1 overflow-hidden flex flex-col">
              <div className="flex items-center justify-between">
                <p className="text-sm text-muted-foreground">
                  Wczytano <strong>{csvData.length}</strong> wierszy z <strong>{csvHeaders.length}</strong> kolumnami
                  {fileType && <Badge variant="outline" className="ml-2">{fileType.toUpperCase()}</Badge>}
                </p>
                <Button variant="ghost" size="sm" onClick={() => setStep('upload')}>
                  <X className="h-4 w-4 mr-1" /> Zmień plik
                </Button>
              </div>

              <ScrollArea className="flex-1">
                <div className="space-y-3 pr-4">
                  {fields.map(field => (
                    <div key={field.key} className="flex items-center gap-4">
                      <div className="w-1/3">
                        <Label className="flex items-center gap-2">
                          {field.label}
                          {field.required && <span className="text-destructive">*</span>}
                        </Label>
                        {field.description && (
                          <p className="text-xs text-muted-foreground">{field.description}</p>
                        )}
                      </div>
                      <ArrowRight className="h-4 w-4 text-muted-foreground flex-shrink-0" />
                      <div className="flex-1">
                        <Select
                          value={mapping[field.key] || '__none__'}
                          onValueChange={(value) => handleMappingChange(field.key, value)}
                        >
                          <SelectTrigger data-testid={`select-mapping-${field.key}`}>
                            <SelectValue placeholder="Wybierz kolumnę..." />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="__none__">-- Pomiń --</SelectItem>
                            {csvHeaders.map(header => (
                              <SelectItem key={header} value={header}>
                                {header}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                      </div>
                      {mapping[field.key] && csvData[0] && (
                        <div className="w-1/4 text-xs text-muted-foreground truncate">
                          np: "{csvData[0][mapping[field.key]]}"
                        </div>
                      )}
                    </div>
                  ))}
                </div>
              </ScrollArea>

              <div className="flex justify-between pt-4 border-t">
                <Button variant="outline" onClick={() => setStep('upload')}>
                  Wstecz
                </Button>
                <Button 
                  onClick={() => setStep('preview')} 
                  disabled={!requiredFieldsMapped}
                  data-testid="button-next-preview"
                >
                  Dalej: Podgląd
                </Button>
              </div>
            </div>
          )}

          {/* Step 3: Preview */}
          {step === 'preview' && (
            <div className="space-y-4 flex-1 overflow-hidden flex flex-col">
              {/* Validation summary */}
              <div className="flex items-center gap-4">
                <Badge variant="default" className="bg-green-600">
                  <CheckCircle2 className="h-3 w-3 mr-1" />
                  Poprawne: {validationResults.validRows.length}
                </Badge>
                {validationResults.invalidRows.length > 0 && (
                  <Badge variant="destructive">
                    <AlertCircle className="h-3 w-3 mr-1" />
                    Błędne: {validationResults.invalidRows.length}
                  </Badge>
                )}
                <span className="text-sm text-muted-foreground">
                  z {validationResults.totalRows} wierszy
                </span>
              </div>

              {/* Validation errors */}
              {validationResults.errors.length > 0 && (
                <Alert variant="destructive">
                  <AlertCircle className="h-4 w-4" />
                  <AlertTitle>Błędy walidacji ({validationResults.errors.length})</AlertTitle>
                  <AlertDescription>
                    <ScrollArea className="max-h-24 mt-2">
                      <div className="space-y-1 text-xs">
                        {validationResults.errors.slice(0, 10).map((err, i) => (
                          <div key={i}>Wiersz {err.row}: {err.message}</div>
                        ))}
                        {validationResults.errors.length > 10 && (
                          <div>...i {validationResults.errors.length - 10} więcej błędów</div>
                        )}
                      </div>
                    </ScrollArea>
                  </AlertDescription>
                </Alert>
              )}

              {validPreviewData.length > 0 ? (
                <>
                  <p className="text-sm text-muted-foreground">
                    Podgląd pierwszych {Math.min(5, validPreviewData.length)} poprawnych wierszy:
                  </p>

                  <ScrollArea className="flex-1 border rounded-lg">
                    <Table>
                      <TableHeader>
                        <TableRow>
                          <TableHead className="w-12">#</TableHead>
                          {fields.filter(f => mapping[f.key]).map(field => (
                            <TableHead key={field.key}>{field.label}</TableHead>
                          ))}
                        </TableRow>
                      </TableHeader>
                      <TableBody>
                        {validPreviewData.map((row, index) => (
                          <TableRow key={index}>
                            <TableCell className="font-mono text-xs">{index + 1}</TableCell>
                            {fields.filter(f => mapping[f.key]).map(field => (
                              <TableCell key={field.key} className="max-w-[200px] truncate">
                                {row[field.key] || <span className="text-muted-foreground">-</span>}
                              </TableCell>
                            ))}
                          </TableRow>
                        ))}
                      </TableBody>
                    </Table>
                  </ScrollArea>
                </>
              ) : (
                <Alert variant="destructive">
                  <AlertCircle className="h-4 w-4" />
                  <AlertTitle>Brak poprawnych wierszy</AlertTitle>
                  <AlertDescription>
                    Wszystkie wiersze zawierają błędy. Wróć do poprzedniego kroku i popraw mapowanie lub popraw plik.
                  </AlertDescription>
                </Alert>
              )}

              <div className="flex justify-between pt-4 border-t">
                <Button variant="outline" onClick={() => setStep('mapping')}>
                  Wstecz
                </Button>
                <Button 
                  onClick={handleImport} 
                  disabled={importMutation.isPending || validationResults.validRows.length === 0}
                  data-testid="button-start-import"
                >
                  {importMutation.isPending && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
                  Importuj {validationResults.validRows.length} wierszy
                </Button>
              </div>
            </div>
          )}

          {/* Step 4: Result */}
          {step === 'result' && result && (
            <div className="space-y-4 flex-1 overflow-hidden flex flex-col">
              {result.failed === 0 ? (
                <Alert className="bg-green-500/10 border-green-500">
                  <CheckCircle2 className="h-4 w-4 text-green-500" />
                  <AlertTitle className="text-green-500">Sukces!</AlertTitle>
                  <AlertDescription>
                    Pomyślnie zaimportowano {result.success} rekordów.
                  </AlertDescription>
                </Alert>
              ) : (
                <Alert variant="destructive">
                  <AlertCircle className="h-4 w-4" />
                  <AlertTitle>Import zakończony z błędami</AlertTitle>
                  <AlertDescription>
                    Sukces: {result.success}, Błędy: {result.failed}
                  </AlertDescription>
                </Alert>
              )}

              {result.errors.length > 0 && (
                <ScrollArea className="flex-1 border rounded-lg">
                  <div className="p-4 space-y-2">
                    <h4 className="font-medium text-destructive">Błędy ({result.errors.length}):</h4>
                    {result.errors.slice(0, 50).map((error, index) => (
                      <div key={index} className="text-sm p-2 bg-destructive/10 rounded">
                        <span className="font-medium">Wiersz {error.row}:</span> {error.error}
                      </div>
                    ))}
                    {result.errors.length > 50 && (
                      <p className="text-sm text-muted-foreground">
                        ...i {result.errors.length - 50} więcej błędów
                      </p>
                    )}
                  </div>
                </ScrollArea>
              )}

              <DialogFooter>
                <Button onClick={() => handleClose(false)}>
                  Zamknij
                </Button>
              </DialogFooter>
            </div>
          )}
        </div>
      </DialogContent>
    </Dialog>
  );
}
