import { useState } from "react";
import { useMutation } from "@tanstack/react-query";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";

interface CreateGroupDialogProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  categoryCode: string;
}

export function CreateGroupDialog({ 
  open, 
  onOpenChange, 
  categoryCode 
}: CreateGroupDialogProps) {
  const { toast } = useToast();
  const [formData, setFormData] = useState({
    name: "",
    code: "",
    description: "",
  });

  const createGroupMutation = useMutation({
    mutationFn: async (data: { name: string; code: string; description: string; category: string }) => {
      return apiRequest("POST", `/api/warehouse/material-groups`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/material-groups"] });
      queryClient.invalidateQueries({ queryKey: ["/api/stats/counts"] });
      toast({ title: "Grupa utworzona pomyślnie" });
      onOpenChange(false);
      resetForm();
    },
    onError: () => {
      toast({ title: "Błąd", description: "Nie udało się utworzyć grupy", variant: "destructive" });
    },
  });

  const resetForm = () => {
    setFormData({ name: "", code: "", description: "" });
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (!formData.name.trim()) {
      toast({ title: "Błąd", description: "Nazwa grupy jest wymagana", variant: "destructive" });
      return;
    }
    if (!formData.code.trim()) {
      toast({ title: "Błąd", description: "Kod grupy jest wymagany", variant: "destructive" });
      return;
    }
    createGroupMutation.mutate({
      ...formData,
      category: categoryCode,
    });
  };

  const handleClose = () => {
    resetForm();
    onOpenChange(false);
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent data-testid="dialog-create-group">
        <DialogHeader>
          <DialogTitle>Dodaj nową grupę</DialogTitle>
        </DialogHeader>
        <form onSubmit={handleSubmit}>
          <div className="space-y-4 py-4">
            <div>
              <Label htmlFor="name">Nazwa grupy *</Label>
              <Input
                id="name"
                value={formData.name}
                onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                placeholder="np. Kartony"
                data-testid="input-group-name"
                autoFocus
              />
            </div>
            <div>
              <Label htmlFor="code">Kod grupy *</Label>
              <Input
                id="code"
                value={formData.code}
                onChange={(e) => setFormData({ ...formData, code: e.target.value })}
                placeholder="np. 50"
                data-testid="input-group-code"
              />
            </div>
            <div>
              <Label htmlFor="description">Opis (opcjonalnie)</Label>
              <Input
                id="description"
                value={formData.description}
                onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                placeholder="Dodatkowy opis grupy"
                data-testid="input-group-description"
              />
            </div>
          </div>
          <DialogFooter>
            <Button
              type="button"
              variant="outline"
              onClick={handleClose}
              data-testid="button-cancel"
            >
              Anuluj
            </Button>
            <Button
              type="submit"
              disabled={createGroupMutation.isPending}
              data-testid="button-create-group"
            >
              {createGroupMutation.isPending ? "Tworzenie..." : "Utwórz grupę"}
            </Button>
          </DialogFooter>
        </form>
      </DialogContent>
    </Dialog>
  );
}
