import { useQuery, useMutation } from "@tanstack/react-query";
import { useParams, useLocation } from "wouter";
import { ArrowLeft, Edit, Trash2, Loader2, Clock } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { useToast } from "@/hooks/use-toast";
import { queryClient } from "@/lib/queryClient";
import type { ProductionRouting, ProductionRoutingOperation, ProductionWorkCenter } from "@shared/schema";

export default function ProductionRoutingDetail() {
  const { id } = useParams();
  const [, navigate] = useLocation();
  const { toast } = useToast();

  const { data: routing, isLoading } = useQuery<ProductionRouting>({
    queryKey: ["/api/production/routings", id],
    queryFn: async () => {
      const res = await fetch(`/api/production/routings/${id}`);
      if (!res.ok) throw new Error("Failed to fetch routing");
      return res.json();
    },
  });

  const { data: operations = [] } = useQuery<ProductionRoutingOperation[]>({
    queryKey: ["/api/production/routings", id, "operations"],
    enabled: !!routing,
  });

  const { data: workCenters = [] } = useQuery<ProductionWorkCenter[]>({
    queryKey: ["/api/production/work-centers"],
  });

  const deleteMutation = useMutation({
    mutationFn: async () => {
      const res = await fetch(`/api/production/routings/${id}`, { method: "DELETE" });
      if (!res.ok) throw new Error("Failed to delete");
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/routings"] });
      toast({ title: "Marszruta została usunięta" });
      navigate("/production/routings");
    },
  });

  const getWorkCenterName = (wcId: number | null) => {
    if (!wcId) return "-";
    return workCenters.find(wc => wc.id === wcId)?.name || "Nieznane";
  };

  if (isLoading) {
    return (
      <div className="container mx-auto p-6">
        <div className="flex justify-center p-8">
          <Loader2 className="h-8 w-8 animate-spin" data-testid="loader-routing-detail" />
        </div>
      </div>
    );
  }

  if (!routing) {
    return (
      <div className="container mx-auto p-6">
        <Card>
          <CardContent className="p-8 text-center text-muted-foreground">
            Nie znaleziono marszruty
          </CardContent>
        </Card>
      </div>
    );
  }

  const totalTime = operations.reduce((sum, op) => sum + (Number(op.estimatedTime) || 0), 0);

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-4">
          <Button
            variant="ghost"
            size="icon"
            onClick={() => navigate("/production/routings")}
            data-testid="button-back"
          >
            <ArrowLeft className="h-4 w-4" />
          </Button>
          <div>
            <h1 className="text-3xl font-bold" data-testid="text-routing-name">
              {routing.name}
            </h1>
            <p className="text-muted-foreground font-mono" data-testid="text-routing-code">
              {routing.code}
            </p>
          </div>
        </div>
        <div className="flex gap-2">
          <Button
            variant="outline"
            onClick={() => navigate(`/production/routings/${id}/edit`)}
            data-testid="button-edit"
          >
            <Edit className="h-4 w-4 mr-2" />
            Edytuj
          </Button>
          <Button
            variant="outline"
            onClick={() => {
              if (confirm("Czy na pewno chcesz usunąć tę marszrutę?")) {
                deleteMutation.mutate();
              }
            }}
            disabled={deleteMutation.isPending}
            data-testid="button-delete"
          >
            <Trash2 className="h-4 w-4 mr-2" />
            Usuń
          </Button>
        </div>
      </div>

      <div className="grid gap-6 md:grid-cols-2">
        <Card>
          <CardHeader>
            <CardTitle>Podstawowe Informacje</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div>
              <p className="text-sm text-muted-foreground">Kod</p>
              <p className="font-mono" data-testid="detail-code">{routing.code}</p>
            </div>
            <div>
              <p className="text-sm text-muted-foreground">Nazwa</p>
              <p data-testid="detail-name">{routing.name}</p>
            </div>
            <div>
              <p className="text-sm text-muted-foreground">Typ produktu</p>
              <p data-testid="detail-type">{routing.productType || "-"}</p>
            </div>
            <div>
              <p className="text-sm text-muted-foreground">Wersja</p>
              <p data-testid="detail-version">{routing.version || 1}</p>
            </div>
            <div>
              <p className="text-sm text-muted-foreground">Status</p>
              <Badge variant={routing.isActive ? "default" : "secondary"} data-testid="detail-active">
                {routing.isActive ? "Aktywna" : "Nieaktywna"}
              </Badge>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Podsumowanie Operacji</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div>
              <p className="text-sm text-muted-foreground">Liczba operacji</p>
              <p className="text-2xl font-bold" data-testid="detail-operations-count">{operations.length}</p>
            </div>
            <div>
              <p className="text-sm text-muted-foreground">Całkowity czas</p>
              <div className="flex items-center gap-2">
                <Clock className="h-4 w-4" />
                <p className="text-2xl font-bold" data-testid="detail-total-time">{totalTime} min</p>
              </div>
            </div>
          </CardContent>
        </Card>

        {routing.description && (
          <Card className="md:col-span-2">
            <CardHeader>
              <CardTitle>Opis</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-muted-foreground" data-testid="detail-description">
                {routing.description}
              </p>
            </CardContent>
          </Card>
        )}

        <Card className="md:col-span-2">
          <CardHeader>
            <CardTitle>Operacje ({operations.length})</CardTitle>
          </CardHeader>
          <CardContent>
            {operations.length === 0 ? (
              <p className="text-center text-muted-foreground p-8">Brak operacji</p>
            ) : (
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="w-16">Sekw.</TableHead>
                    <TableHead>Kod</TableHead>
                    <TableHead>Nazwa</TableHead>
                    <TableHead>Gniazdo robocze</TableHead>
                    <TableHead className="text-right">Czas (min)</TableHead>
                    <TableHead className="text-right">Setup (min)</TableHead>
                    <TableHead className="text-center">Status</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {operations
                    .sort((a, b) => a.sequence - b.sequence)
                    .map((op) => (
                      <TableRow key={op.id} data-testid={`row-operation-${op.id}`}>
                        <TableCell className="font-bold">{op.sequence}</TableCell>
                        <TableCell className="font-mono">{op.code}</TableCell>
                        <TableCell>{op.name}</TableCell>
                        <TableCell>{getWorkCenterName(op.workCenterId)}</TableCell>
                        <TableCell className="text-right">{op.estimatedTime}</TableCell>
                        <TableCell className="text-right">{op.setupTime || "-"}</TableCell>
                        <TableCell className="text-center">
                          <Badge variant={op.isActive ? "default" : "secondary"}>
                            {op.isActive ? "Aktywna" : "Nieaktywna"}
                          </Badge>
                        </TableCell>
                      </TableRow>
                    ))}
                </TableBody>
              </Table>
            )}
          </CardContent>
        </Card>

        {routing.notes && (
          <Card className="md:col-span-2">
            <CardHeader>
              <CardTitle>Notatki</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-muted-foreground whitespace-pre-wrap" data-testid="detail-notes">
                {routing.notes}
              </p>
            </CardContent>
          </Card>
        )}
      </div>
    </div>
  );
}
