import { useState, useEffect, useMemo } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { Scissors, Plus, Play, CheckCircle2, Clock, Upload, FileUp, X, Search, ChevronLeft, ChevronRight, ChevronsLeft, ChevronsRight, Edit } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import { Sheet, SheetContent, SheetDescription, SheetHeader, SheetTitle } from "@/components/ui/sheet";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { queryClient } from "@/lib/queryClient";
import { useDropzone } from "react-dropzone";
import { SavedFiltersManager } from "@/components/saved-filters-manager";

const STORAGE_KEY = "cutting-patterns-filters";

interface CuttingPattern {
  id: number;
  code: string;
  board_length: string;
  board_width: string;
  board_thickness: string;
  kerf: string;
  image_url: string | null;
  opis: string | null;
  status: string;
  notes: string | null;
  is_active: boolean;
  created_at: string;
  updated_at: string;
  items_count: number;
  total_quantity: number;
  planned_quantity: number;
  in_production_quantity: number;
  completed_quantity: number;
}

interface CuttingPatternItem {
  id: number;
  item_name: string;
  length: string;
  width: string;
  thickness: string;
  board_material_id: number | null;
  edging_material_id: number | null;
  quantity_requested: number;
  quantity_completed: number;
  status: string;
  notes: string | null;
  is_active: boolean;
  created_at: string;
  updated_at: string;
}

interface CuttingPatternDetail extends CuttingPattern {
  items: CuttingPatternItem[];
}

const formatDimension = (dim: string | number): string => {
  const num = typeof dim === 'string' ? parseFloat(dim) : dim;
  return Number.isInteger(num) ? num.toString() : num.toFixed(2).replace(/\.?0+$/, '');
};

export default function CuttingPatternsPage() {
  const { toast } = useToast();
  const [, navigate] = useLocation();
  
  // Load filters from localStorage
  const loadFilters = () => {
    try {
      const saved = localStorage.getItem(STORAGE_KEY);
      if (saved) {
        return JSON.parse(saved);
      }
    } catch (error) {
      console.error('Error loading filters from localStorage:', error);
    }
    return {};
  };

  const savedFilters = loadFilters();
  
  // State
  const [selectedPatternId, setSelectedPatternId] = useState<number | null>(null);
  const [isImportDialogOpen, setIsImportDialogOpen] = useState(false);
  const [importName, setImportName] = useState("");
  
  // Filter & pagination state
  const [searchQuery, setSearchQuery] = useState(savedFilters.searchQuery || "");
  const [filterStatus, setFilterStatus] = useState<string>(savedFilters.filterStatus || "all");
  const [page, setPage] = useState(savedFilters.page || 1);
  const [pageSize, setPageSize] = useState(savedFilters.pageSize || 25);

  // Helper functions for SavedFiltersManager
  const getCurrentFilters = () => ({
    searchQuery,
    filterStatus,
    page,
    pageSize,
  });

  const loadFiltersFromSaved = (filters: Record<string, any>) => {
    if (filters.searchQuery !== undefined) setSearchQuery(filters.searchQuery);
    if (filters.filterStatus !== undefined) setFilterStatus(filters.filterStatus);
    if (filters.page !== undefined) setPage(filters.page);
    if (filters.pageSize !== undefined) setPageSize(filters.pageSize);
  };

  // Save filters to localStorage whenever they change
  useEffect(() => {
    const filters = {
      searchQuery,
      filterStatus,
      page,
      pageSize,
    };
    try {
      localStorage.setItem(STORAGE_KEY, JSON.stringify(filters));
    } catch (error) {
      console.error('Error saving filters to localStorage:', error);
    }
  }, [searchQuery, filterStatus, page, pageSize]);

  // Fetch patterns list
  const { data: patterns, isLoading: patternsLoading } = useQuery<CuttingPattern[]>({
    queryKey: ["/api/production/cut-patterns"],
  });

  // Fetch pattern details
  const { data: patternDetail, isLoading: detailLoading } = useQuery<CuttingPatternDetail>({
    queryKey: ["/api/production/cut-patterns", selectedPatternId],
    queryFn: async () => {
      const response = await fetch(`/api/production/cut-patterns/${selectedPatternId}`);
      if (!response.ok) {
        throw new Error("Failed to fetch pattern details");
      }
      return response.json();
    },
    enabled: selectedPatternId !== null,
  });

  // Parse search query into individual search phrases
  const getSearchPhrases = () => {
    if (!searchQuery) return [];
    const phrases: string[] = [];
    searchQuery.split(';').forEach((phrase: string) => {
      const trimmed = phrase.trim();
      if (trimmed) phrases.push(trimmed);
    });
    return phrases;
  };

  const searchPhrases = getSearchPhrases();

  // Remove a specific search phrase
  const removeSearchPhrase = (phraseToRemove: string) => {
    const updatedPhrases = searchPhrases.filter(p => p !== phraseToRemove);
    setSearchQuery(updatedPhrases.join('; '));
    setPage(1);
  };

  // Clear all filters
  const clearAllFilters = () => {
    setSearchQuery("");
    setFilterStatus("all");
    setPage(1);
  };

  // Client-side filtering
  const filteredPatterns = useMemo(() => {
    if (!patterns) return [];
    
    return patterns.filter((pattern) => {
      // Status filter
      if (filterStatus !== "all" && pattern.status !== filterStatus) {
        return false;
      }
      
      // Text search - supports AND (;)
      if (searchQuery) {
        const terms = searchQuery.split(';').map((t: string) => t.trim()).filter(Boolean);
        const matches = terms.every((term: string) => {
          const lowerTerm = term.toLowerCase();
          return (
            pattern.code?.toLowerCase().includes(lowerTerm) ||
            pattern.opis?.toLowerCase().includes(lowerTerm) ||
            pattern.status?.toLowerCase().includes(lowerTerm)
          );
        });
        if (!matches) return false;
      }
      
      return true;
    });
  }, [patterns, searchQuery, filterStatus]);

  // Count active filters
  const activeFiltersCount = [
    ...searchPhrases,
    filterStatus !== "all" ? 1 : 0,
  ].filter(Boolean).length;

  // Pagination
  const totalPages = Math.ceil(filteredPatterns.length / pageSize);
  const paginatedPatterns = useMemo(() => {
    const startIndex = (page - 1) * pageSize;
    return filteredPatterns.slice(startIndex, startIndex + pageSize);
  }, [filteredPatterns, page, pageSize]);

  // Reset to page 1 if current page exceeds total pages
  useEffect(() => {
    if (page > totalPages && totalPages > 0) {
      setPage(1);
    }
  }, [page, totalPages]);

  const handleStartProduction = async (patternId: number) => {
    try {
      await apiRequest("POST", `/api/production/cut-patterns/${patternId}/start`);
      queryClient.invalidateQueries({ queryKey: ["/api/production/cut-patterns"] });
      queryClient.invalidateQueries({ queryKey: ["/api/production/cut-patterns", patternId] });
      toast({
        title: "Produkcja rozpoczęta",
        description: "Status rozkroju został zmieniony na 'w produkcji'",
      });
    } catch (error: any) {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się rozpocząć produkcji",
        variant: "destructive",
      });
    }
  };

  const handleCompleteProduction = async (patternId: number) => {
    try {
      await apiRequest("POST", `/api/production/cut-patterns/${patternId}/complete`);
      queryClient.invalidateQueries({ queryKey: ["/api/production/cut-patterns"] });
      queryClient.invalidateQueries({ queryKey: ["/api/production/cut-patterns", patternId] });
      toast({
        title: "Produkcja zakończona",
        description: "Rozkrój został oznaczony jako zakończony",
      });
      setSelectedPatternId(null);
    } catch (error: any) {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zakończyć produkcji",
        variant: "destructive",
      });
    }
  };

  // CSV import mutation dla nowego rozkroju
  const importNewPatternMutation = useMutation({
    mutationFn: async (file: File) => {
      const formData = new FormData();
      formData.append('file', file);
      formData.append('name', importName);

      const response = await fetch('/api/production/cut-pattern-templates/import-csv', {
        method: 'POST',
        body: formData,
        credentials: 'include'
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.details || error.error || 'Import failed');
      }

      return response.json();
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/cut-patterns"] });
      toast({ description: data.message || "Rozkrój zaimportowany z CSV" });
      setIsImportDialogOpen(false);
      setImportName("");
    },
    onError: (error: any) => {
      toast({ 
        variant: "destructive", 
        description: error.message || "Błąd importu CSV" 
      });
    }
  });

  const handleImportFormatki = async (file: File, patternId: number) => {
    try {
      const formData = new FormData();
      formData.append('file', file);

      const response = await fetch(`/api/production/cut-patterns/${patternId}/import-formatki`, {
        method: 'POST',
        body: formData,
      });

      const data = await response.json();

      if (!response.ok) {
        throw new Error(data.error || data.details || 'Import failed');
      }

      queryClient.invalidateQueries({ queryKey: ["/api/production/cut-patterns"] });
      queryClient.invalidateQueries({ queryKey: ["/api/production/cut-patterns", patternId] });
      
      toast({
        title: "Import zakończony",
        description: data.message || `Zaimportowano ${data.imported} formatek`,
      });
    } catch (error: any) {
      toast({
        title: "Błąd importu",
        description: error.message || "Nie udało się zaimportować formatek",
        variant: "destructive",
      });
    }
  };

  const onDrop = (acceptedFiles: File[], patternId: number) => {
    if (acceptedFiles.length > 0) {
      handleImportFormatki(acceptedFiles[0], patternId);
    }
  };

  const { getRootProps, getInputProps, isDragActive } = useDropzone({
    onDrop: (files) => selectedPatternId && onDrop(files, selectedPatternId),
    accept: {
      'text/csv': ['.csv'],
    },
    multiple: false,
    disabled: !selectedPatternId,
  });

  const { getRootProps: getNewPatternRootProps, getInputProps: getNewPatternInputProps, isDragActive: isNewPatternDragActive } = useDropzone({
    accept: { 'text/csv': ['.csv'] },
    onDrop: (acceptedFiles) => {
      if (acceptedFiles.length > 0 && importName) {
        importNewPatternMutation.mutate(acceptedFiles[0]);
      } else if (!importName) {
        toast({ 
          variant: "destructive", 
          description: "Podaj nazwę rozkroju przed importem" 
        });
      }
    },
    multiple: false,
    disabled: !importName || importNewPatternMutation.isPending
  });

  const getStatusBadge = (status: string) => {
    switch (status) {
      case "planowany":
        return <Badge variant="outline" className="gap-1"><Clock className="h-3 w-3" />Planowany</Badge>;
      case "w_produkcji":
        return <Badge variant="default" className="gap-1 bg-blue-600"><Play className="h-3 w-3" />W produkcji</Badge>;
      case "wykonany":
        return <Badge variant="default" className="gap-1 bg-green-600"><CheckCircle2 className="h-3 w-3" />Wykonany</Badge>;
      default:
        return <Badge variant="secondary">{status}</Badge>;
    }
  };

  return (
    <div className="container mx-auto p-6 max-w-7xl space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div className="space-y-1">
          <h1 className="text-3xl font-bold tracking-tight flex items-center gap-2" data-testid="text-page-title">
            <Scissors className="h-8 w-8 text-orange-600" />
            Rozkroje Formatek
          </h1>
          <p className="text-muted-foreground" data-testid="text-page-description">
            Zarządzaj rozkrojami płyt i produkcją formatek
          </p>
        </div>
        <div className="flex gap-2">
          <Button 
            variant="outline"
            onClick={() => navigate("/cutting-patterns/new")}
            data-testid="button-add-new-pattern"
          >
            <Plus className="h-4 w-4 mr-2" />
            Dodaj rozkrój
          </Button>
          <Button 
            onClick={() => setIsImportDialogOpen(true)}
            data-testid="button-import-new-pattern"
          >
            <Upload className="h-4 w-4 mr-2" />
            Import CSV
          </Button>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid gap-4 md:grid-cols-3">
        <Card data-testid="card-stats-planned">
          <CardHeader className="pb-3">
            <CardTitle className="text-sm font-medium text-muted-foreground">Planowane</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="text-count-planned">
              {patterns?.filter(p => p.status === "planowany").length || 0}
            </div>
          </CardContent>
        </Card>
        <Card data-testid="card-stats-in-production">
          <CardHeader className="pb-3">
            <CardTitle className="text-sm font-medium text-muted-foreground">W produkcji</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-blue-600" data-testid="text-count-in-production">
              {patterns?.filter(p => p.status === "w_produkcji").length || 0}
            </div>
          </CardContent>
        </Card>
        <Card data-testid="card-stats-completed">
          <CardHeader className="pb-3">
            <CardTitle className="text-sm font-medium text-muted-foreground">Zakończone</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600" data-testid="text-count-completed">
              {patterns?.filter(p => p.status === "wykonany").length || 0}
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Patterns Table */}
      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <div>
              <CardTitle>Lista rozkrojów</CardTitle>
              <CardDescription>
                Kliknij na rozkrój aby zobaczyć szczegóły i listę formatek
              </CardDescription>
            </div>
          </div>
          
          {/* Filter Toolbar */}
          <div className="flex flex-col gap-3 mt-4">
            <div className="flex flex-col md:flex-row gap-2">
              <div className="relative flex-1">
                <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                <Input
                  placeholder="Szukaj rozkroju (kod, opis) - użyj ; do AND"
                  value={searchQuery}
                  onChange={(e) => {
                    setSearchQuery(e.target.value);
                    setPage(1);
                  }}
                  className="pl-9"
                  data-testid="input-search-patterns"
                />
              </div>
              <Select value={filterStatus} onValueChange={(value) => {
                setFilterStatus(value);
                setPage(1);
              }}>
                <SelectTrigger className="w-full md:w-48" data-testid="select-filter-status">
                  <SelectValue placeholder="Status" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Wszystkie statusy</SelectItem>
                  <SelectItem value="planowany">Planowany</SelectItem>
                  <SelectItem value="w_produkcji">W produkcji</SelectItem>
                  <SelectItem value="wykonany">Wykonany</SelectItem>
                </SelectContent>
              </Select>
              <SavedFiltersManager
                context="cutting_patterns"
                currentFilters={getCurrentFilters()}
                onFiltersLoad={loadFiltersFromSaved}
              />
            </div>
            
            {/* Active Filters */}
            {activeFiltersCount > 0 && (
              <div className="flex items-center gap-1.5 flex-wrap">
                <span className="text-xs text-muted-foreground">Aktywne filtry:</span>
                {searchPhrases.map((phrase, index) => (
                  <Badge 
                    key={`search-${index}`} 
                    variant="default" 
                    className="gap-1 text-xs"
                    data-testid={`badge-filter-search-${index}`}
                  >
                    {phrase}
                    <X
                      className="h-3 w-3 cursor-pointer hover:text-destructive"
                      onClick={() => removeSearchPhrase(phrase)}
                      data-testid={`button-remove-search-${index}`}
                    />
                  </Badge>
                ))}
                {filterStatus !== "all" && (
                  <Badge 
                    variant="default" 
                    className="gap-1 text-xs"
                    data-testid="badge-filter-status"
                  >
                    Status: {filterStatus === "planowany" ? "Planowany" : filterStatus === "w_produkcji" ? "W produkcji" : "Wykonany"}
                    <X
                      className="h-3 w-3 cursor-pointer hover:text-destructive"
                      onClick={() => {
                        setFilterStatus("all");
                        setPage(1);
                      }}
                      data-testid="button-remove-status"
                    />
                  </Badge>
                )}
                <Button
                  variant="ghost"
                  size="sm"
                  onClick={clearAllFilters}
                  className="h-6 px-2 text-xs"
                  data-testid="button-clear-all-filters"
                >
                  Wyczyść wszystkie
                </Button>
              </div>
            )}
            
            {/* Results Counter */}
            <div className="text-sm text-muted-foreground">
              Wyświetlono {filteredPatterns.length === 0 ? 0 : (page - 1) * pageSize + 1}-{Math.min(page * pageSize, filteredPatterns.length)} z {filteredPatterns.length} rozkrojów
            </div>
          </div>
        </CardHeader>
        <CardContent>
          {patternsLoading ? (
            <div className="space-y-2">
              {[1, 2, 3].map((i) => <Skeleton key={i} className="h-12 w-full" />)}
            </div>
          ) : (
            <>
              {/* Top Pagination */}
              {totalPages > 1 && (
                <div className="flex items-center justify-between mb-4">
                  <div className="flex items-center gap-2">
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => setPage(1)}
                      disabled={page === 1}
                      data-testid="button-first-page-top"
                    >
                      <ChevronsLeft className="h-4 w-4" />
                    </Button>
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => setPage((p: number) => Math.max(1, p - 1))}
                      disabled={page === 1}
                      data-testid="button-prev-page-top"
                    >
                      <ChevronLeft className="h-4 w-4" />
                    </Button>
                    <span className="text-sm text-muted-foreground">
                      Strona {page} z {totalPages}
                    </span>
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => setPage((p: number) => Math.min(totalPages, p + 1))}
                      disabled={page === totalPages}
                      data-testid="button-next-page-top"
                    >
                      <ChevronRight className="h-4 w-4" />
                    </Button>
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => setPage(totalPages)}
                      disabled={page === totalPages}
                      data-testid="button-last-page-top"
                    >
                      <ChevronsRight className="h-4 w-4" />
                    </Button>
                  </div>
                  <Select value={pageSize.toString()} onValueChange={(value) => {
                    setPageSize(Number(value));
                    setPage(1);
                  }}>
                    <SelectTrigger className="w-32" data-testid="select-page-size-top">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="10">10 / strona</SelectItem>
                      <SelectItem value="25">25 / strona</SelectItem>
                      <SelectItem value="50">50 / strona</SelectItem>
                      <SelectItem value="100">100 / strona</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              )}

              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Kod</TableHead>
                    <TableHead>Podgląd</TableHead>
                    <TableHead>Wymiary płyty</TableHead>
                    <TableHead>Opis</TableHead>
                    <TableHead>Status</TableHead>
                    <TableHead className="text-right">Formatek</TableHead>
                    <TableHead className="text-right">Ilość</TableHead>
                    <TableHead className="text-right">Akcje</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {paginatedPatterns && paginatedPatterns.length > 0 ? (
                    paginatedPatterns.map((pattern) => (
                    <TableRow
                      key={pattern.id}
                      className="cursor-pointer hover-elevate"
                      onClick={() => setSelectedPatternId(pattern.id)}
                      data-testid={`row-pattern-${pattern.id}`}
                    >
                      <TableCell className="font-medium">{pattern.code}</TableCell>
                      <TableCell>
                        {pattern.image_url ? (
                          <img 
                            src={pattern.image_url} 
                            alt={`Rozkrój ${pattern.code}`}
                            className="h-12 w-12 object-contain rounded border"
                            data-testid={`img-pattern-${pattern.id}`}
                          />
                        ) : (
                          <div className="h-12 w-12 flex items-center justify-center bg-muted rounded border text-muted-foreground text-xs">
                            Brak
                          </div>
                        )}
                      </TableCell>
                      <TableCell className="text-muted-foreground text-sm">
                        {formatDimension(pattern.board_length)} × {formatDimension(pattern.board_width)} × {formatDimension(pattern.board_thickness)} mm
                      </TableCell>
                      <TableCell className="text-sm text-muted-foreground">
                        {pattern.opis || '-'}
                      </TableCell>
                      <TableCell>{getStatusBadge(pattern.status)}</TableCell>
                      <TableCell className="text-right">{pattern.items_count}</TableCell>
                      <TableCell className="text-right font-medium">{pattern.total_quantity.toLocaleString()}</TableCell>
                      <TableCell className="text-right">
                        <div className="flex gap-2 justify-end">
                          <Button
                            size="sm"
                            variant="ghost"
                            onClick={(e) => {
                              e.stopPropagation();
                              navigate(`/cutting-patterns/${pattern.id}/edit`);
                            }}
                            data-testid={`button-edit-${pattern.id}`}
                          >
                            <Edit className="h-3 w-3 mr-1" />
                            Edytuj
                          </Button>
                          {pattern.status === "planowany" && (
                            <Button
                              size="sm"
                              variant="outline"
                              onClick={(e) => {
                                e.stopPropagation();
                                handleStartProduction(pattern.id);
                              }}
                              data-testid={`button-start-${pattern.id}`}
                            >
                              <Play className="h-3 w-3 mr-1" />
                              Start
                            </Button>
                          )}
                          {pattern.status === "w_produkcji" && (
                            <Button
                              size="sm"
                              variant="outline"
                              onClick={(e) => {
                                e.stopPropagation();
                                handleCompleteProduction(pattern.id);
                              }}
                              data-testid={`button-complete-${pattern.id}`}
                            >
                              <CheckCircle2 className="h-3 w-3 mr-1" />
                              Zakończ
                            </Button>
                          )}
                        </div>
                      </TableCell>
                    </TableRow>
                  ))
                ) : (
                  <TableRow>
                    <TableCell colSpan={8} className="text-center text-muted-foreground h-24">
                      Brak rozkrojów w bazie
                    </TableCell>
                  </TableRow>
                )}
              </TableBody>
            </Table>

            {/* Bottom Pagination */}
            {totalPages > 1 && (
              <div className="flex items-center justify-between mt-4">
                <div className="flex items-center gap-2">
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setPage(1)}
                    disabled={page === 1}
                    data-testid="button-first-page-bottom"
                  >
                    <ChevronsLeft className="h-4 w-4" />
                  </Button>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setPage((p: number) => Math.max(1, p - 1))}
                    disabled={page === 1}
                    data-testid="button-prev-page-bottom"
                  >
                    <ChevronLeft className="h-4 w-4" />
                  </Button>
                  <span className="text-sm text-muted-foreground">
                    Strona {page} z {totalPages}
                  </span>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setPage((p: number) => Math.min(totalPages, p + 1))}
                    disabled={page === totalPages}
                    data-testid="button-next-page-bottom"
                  >
                    <ChevronRight className="h-4 w-4" />
                  </Button>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setPage(totalPages)}
                    disabled={page === totalPages}
                    data-testid="button-last-page-bottom"
                  >
                    <ChevronsRight className="h-4 w-4" />
                  </Button>
                </div>
                <Select value={pageSize.toString()} onValueChange={(value) => {
                  setPageSize(Number(value));
                  setPage(1);
                }}>
                  <SelectTrigger className="w-32" data-testid="select-page-size-bottom">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="10">10 / strona</SelectItem>
                    <SelectItem value="25">25 / strona</SelectItem>
                    <SelectItem value="50">50 / strona</SelectItem>
                    <SelectItem value="100">100 / strona</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            )}
            </>
          )}
        </CardContent>
      </Card>

      {/* Detail Sheet */}
      <Sheet open={selectedPatternId !== null} onOpenChange={() => setSelectedPatternId(null)}>
        <SheetContent className="sm:max-w-2xl overflow-y-auto">
          {detailLoading ? (
            <div className="space-y-4">
              <Skeleton className="h-8 w-48" />
              <Skeleton className="h-32 w-full" />
              <Skeleton className="h-64 w-full" />
            </div>
          ) : patternDetail ? (
            <>
              <SheetHeader>
                <SheetTitle className="flex items-center gap-2">
                  <Scissors className="h-5 w-5 text-orange-600" />
                  Rozkrój {patternDetail.code}
                </SheetTitle>
                <SheetDescription>
                  Szczegóły rozkroju i lista formatek do pocięcia
                </SheetDescription>
              </SheetHeader>

              <div className="mt-6 space-y-6">
                {/* Pattern Image */}
                {patternDetail.image_url && (
                  <div className="space-y-2">
                    <h3 className="font-semibold">Podgląd rozkroju</h3>
                    <img 
                      src={patternDetail.image_url} 
                      alt={`Rozkrój ${patternDetail.code}`}
                      className="w-full max-w-2xl rounded border shadow-sm"
                      data-testid="img-pattern-detail"
                    />
                  </div>
                )}

                {/* Pattern Info */}
                <div className="space-y-3">
                  <h3 className="font-semibold">Informacje o rozkroju</h3>
                  <div className="grid grid-cols-2 gap-3 text-sm">
                    <div>
                      <span className="text-muted-foreground">Status:</span>
                      <div className="mt-1">{getStatusBadge(patternDetail.status)}</div>
                    </div>
                    <div>
                      <span className="text-muted-foreground">Wymiary płyty:</span>
                      <div className="mt-1 font-medium">
                        {formatDimension(patternDetail.board_length)} × {formatDimension(patternDetail.board_width)} × {formatDimension(patternDetail.board_thickness)} mm
                      </div>
                    </div>
                    <div>
                      <span className="text-muted-foreground">Szerokość cięcia:</span>
                      <div className="mt-1 font-medium">{formatDimension(patternDetail.kerf)} mm</div>
                    </div>
                    <div>
                      <span className="text-muted-foreground">Liczba pozycji:</span>
                      <div className="mt-1 font-medium">{patternDetail.items.length}</div>
                    </div>
                  </div>
                  {patternDetail.opis && (
                    <div>
                      <span className="text-muted-foreground text-sm">Opis:</span>
                      <p className="mt-1 text-sm font-medium">{patternDetail.opis}</p>
                    </div>
                  )}
                  {patternDetail.notes && (
                    <div>
                      <span className="text-muted-foreground text-sm">Uwagi:</span>
                      <p className="mt-1 text-sm">{patternDetail.notes}</p>
                    </div>
                  )}
                </div>

                {/* Action Buttons */}
                <div className="flex gap-2">
                  {patternDetail.status === "planowany" && (
                    <Button
                      onClick={() => handleStartProduction(patternDetail.id)}
                      className="flex-1"
                      data-testid="button-detail-start"
                    >
                      <Play className="h-4 w-4 mr-2" />
                      Rozpocznij produkcję
                    </Button>
                  )}
                  {patternDetail.status === "w_produkcji" && (
                    <Button
                      onClick={() => handleCompleteProduction(patternDetail.id)}
                      className="flex-1"
                      data-testid="button-detail-complete"
                    >
                      <CheckCircle2 className="h-4 w-4 mr-2" />
                      Zakończ produkcję
                    </Button>
                  )}
                </div>

                {/* Import CSV */}
                <div className="space-y-3">
                  <h3 className="font-semibold flex items-center gap-2">
                    <Upload className="h-4 w-4" />
                    Import formatek z CSV
                  </h3>
                  <div
                    {...getRootProps()}
                    className={`border-2 border-dashed rounded-lg p-6 text-center cursor-pointer transition-colors ${
                      isDragActive ? 'border-primary bg-primary/5' : 'border-border hover:border-primary/50'
                    }`}
                    data-testid="dropzone-import-csv"
                  >
                    <input {...getInputProps()} />
                    <FileUp className="h-12 w-12 mx-auto mb-3 text-muted-foreground" />
                    {isDragActive ? (
                      <p className="text-sm text-primary font-medium">Upuść plik CSV tutaj...</p>
                    ) : (
                      <>
                        <p className="text-sm font-medium mb-1">Przeciągnij i upuść plik CSV</p>
                        <p className="text-xs text-muted-foreground">lub kliknij aby wybrać plik</p>
                        <p className="text-xs text-muted-foreground mt-2">Format: plik z programu formatyzacji</p>
                      </>
                    )}
                  </div>
                </div>

                {/* Items List */}
                <div className="space-y-3">
                  <h3 className="font-semibold">Formatki do pocięcia ({patternDetail.items.length})</h3>
                  <div className="border rounded-md">
                    <Table>
                      <TableHeader>
                        <TableRow>
                          <TableHead>Nazwa</TableHead>
                          <TableHead>Wymiary</TableHead>
                          <TableHead className="text-right">Ilość</TableHead>
                          <TableHead>Status</TableHead>
                        </TableRow>
                      </TableHeader>
                      <TableBody>
                        {patternDetail.items.map((item) => (
                          <TableRow key={item.id}>
                            <TableCell className="font-medium">{item.item_name}</TableCell>
                            <TableCell className="text-muted-foreground text-sm">
                              {formatDimension(item.length)} × {formatDimension(item.width)} mm
                            </TableCell>
                            <TableCell className="text-right font-medium">
                              {item.quantity_completed > 0 ? (
                                <span className="text-green-600">
                                  {item.quantity_completed}/{item.quantity_requested}
                                </span>
                              ) : (
                                item.quantity_requested
                              )}
                            </TableCell>
                            <TableCell>{getStatusBadge(item.status)}</TableCell>
                          </TableRow>
                        ))}
                      </TableBody>
                    </Table>
                  </div>
                </div>
              </div>
            </>
          ) : null}
        </SheetContent>
      </Sheet>

      {/* Import Dialog */}
      <Dialog open={isImportDialogOpen} onOpenChange={setIsImportDialogOpen}>
        <DialogContent data-testid="dialog-import-new-pattern">
          <DialogHeader>
            <DialogTitle>Importuj nowy rozkrój z CSV</DialogTitle>
            <DialogDescription>
              Zaimportuj rozkrój z pliku CSV (kodowanie Windows-1250)
            </DialogDescription>
          </DialogHeader>

          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="import-name">Nazwa rozkroju *</Label>
              <Input
                id="import-name"
                value={importName}
                onChange={(e) => setImportName(e.target.value)}
                placeholder="np. Rozkrój frontów 18mm"
                data-testid="input-import-name"
              />
              <p className="text-xs text-muted-foreground">
                Kod rozkroju zostanie wygenerowany automatycznie (ROZ-1, ROZ-2, itd.)
              </p>
            </div>

            <div className="space-y-2">
              <Label>Plik CSV *</Label>
              <div
                {...getNewPatternRootProps()}
                className={`border-2 border-dashed rounded-lg p-8 text-center cursor-pointer transition-colors ${
                  isNewPatternDragActive 
                    ? 'border-primary bg-primary/5' 
                    : 'border-border hover:border-primary/50'
                } ${
                  !importName 
                    ? 'opacity-50 cursor-not-allowed' 
                    : ''
                }`}
                data-testid="dropzone-new-pattern"
              >
                <input {...getNewPatternInputProps()} />
                <FileUp className="h-12 w-12 mx-auto mb-3 text-muted-foreground" />
                {isNewPatternDragActive ? (
                  <p className="text-sm text-primary font-medium">Upuść plik CSV tutaj...</p>
                ) : (
                  <>
                    <p className="text-sm font-medium mb-1">
                      {importName 
                        ? "Przeciągnij i upuść plik CSV" 
                        : "Najpierw podaj nazwę rozkroju"
                      }
                    </p>
                    <p className="text-xs text-muted-foreground">lub kliknij aby wybrać plik</p>
                    <p className="text-xs text-muted-foreground mt-2">
                      Format: Windows-1250, kolumny: Formatka, Długość, Szerokość, Grubość, Ilość, Okleinowania
                    </p>
                  </>
                )}
              </div>
            </div>

            {importNewPatternMutation.isPending && (
              <div className="text-sm text-muted-foreground text-center">
                Importowanie...
              </div>
            )}
          </div>
        </DialogContent>
      </Dialog>
    </div>
  );
}
